import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { ContainerImage } from './container-image';
import { ModelData } from './model-data';
/**
 * Interface that defines a Model resource.
 */
export interface IModel extends cdk.IResource, iam.IGrantable, ec2.IConnectable {
    /**
     * Returns the ARN of this model.
     *
     * @attribute
     */
    readonly modelArn: string;
    /**
     * Returns the name of this model.
     *
     * @attribute
     */
    readonly modelName: string;
    /**
     * The IAM role associated with this Model.
     */
    readonly role?: iam.IRole;
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
}
/**
 * Represents a Model resource defined outside this stack.
 */
export interface ModelAttributes {
    /**
     * The ARN of this model.
     */
    readonly modelArn: string;
    /**
     * The IAM execution role associated with this model.
     *
     * @default - When not provided, any role-related operations will no-op.
     */
    readonly role?: iam.IRole;
    /**
     * The security groups for this model, if in a VPC.
     *
     * @default - When not provided, the connections to/from this model cannot be managed.
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
}
declare abstract class ModelBase extends cdk.Resource implements IModel {
    /**
     * Returns the ARN of this model.
     * @attribute
     */
    abstract readonly modelArn: string;
    /**
     * Returns the name of the model.
     * @attribute
     */
    abstract readonly modelName: string;
    /**
     * Execution role for SageMaker Model
     */
    abstract readonly role?: iam.IRole;
    /**
     * The principal this Model is running as
     */
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * An accessor for the Connections object that will fail if this Model does not have a VPC
     * configured.
     */
    get connections(): ec2.Connections;
    /**
     * The actual Connections object for this Model. This may be unset in the event that a VPC has not
     * been configured.
     * @internal
     */
    protected _connections: ec2.Connections | undefined;
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement: iam.PolicyStatement): void;
}
/**
 * Describes the container, as part of model definition.
 */
export interface ContainerDefinition {
    /**
     * The image used to start a container.
     */
    readonly image: ContainerImage;
    /**
     * A map of environment variables to pass into the container.
     *
     * @default - none
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * Hostname of the container within an inference pipeline. For single container models, this field
     * is ignored. When specifying a hostname for one ContainerDefinition in a pipeline, hostnames
     * must be specified for all other ContainerDefinitions in that pipeline.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-sagemaker-model-containerdefinition.html#cfn-sagemaker-model-containerdefinition-containerhostname
     *
     * @default - Amazon SageMaker will automatically assign a unique name based on the position of
     * this ContainerDefinition in an inference pipeline.
     */
    readonly containerHostname?: string;
    /**
     * S3 path to the model artifacts.
     *
     * @default - none
     */
    readonly modelData?: ModelData;
}
/**
 * Construction properties for a SageMaker Model.
 */
export interface ModelProps {
    /**
     * The IAM role that the Amazon SageMaker service assumes.
     *
     * @see https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-roles.html#sagemaker-roles-createmodel-perms
     *
     * @default - a new IAM role will be created with the `AmazonSageMakerFullAccess` policy attached.
     */
    readonly role?: iam.IRole;
    /**
     * Name of the SageMaker Model.
     *
     * @default - AWS CloudFormation generates a unique physical ID and uses that ID for the model's
     * name.
     */
    readonly modelName?: string;
    /**
     * The VPC to deploy model containers to.
     *
     * @default - none
     */
    readonly vpc?: ec2.IVpc;
    /**
     * The VPC subnets to use when deploying model containers.
     *
     * @default - none
     */
    readonly vpcSubnets?: ec2.SubnetSelection;
    /**
     * The security groups to associate to the Model. If no security groups are provided and 'vpc' is
     * configured, one security group will be created automatically.
     *
     * @default - A security group will be automatically created if 'vpc' is supplied
     */
    readonly securityGroups?: ec2.ISecurityGroup[];
    /**
     * Specifies the container definitions for this model, consisting of either a single primary
     * container or an inference pipeline of multiple containers.
     *
     * @default - none
     */
    readonly containers?: ContainerDefinition[];
    /**
     * Whether to allow the SageMaker Model to send all network traffic
     *
     * If set to false, you must individually add traffic rules to allow the
     * SageMaker Model to connect to network targets.
     *
     * Only used if 'vpc' is supplied.
     *
     * @default true
     */
    readonly allowAllOutbound?: boolean;
}
/**
 * Defines a SageMaker Model.
 */
export declare class Model extends ModelBase {
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param modelArn the ARN of the model.
     */
    static fromModelArn(scope: Construct, id: string, modelArn: string): IModel;
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param modelName the name of the model.
     */
    static fromModelName(scope: Construct, id: string, modelName: string): IModel;
    /**
     * Imports a Model defined either outside the CDK or in a different CDK stack.
     * @param scope the Construct scope.
     * @param id the resource id.
     * @param attrs the attributes of the model to import.
     */
    static fromModelAttributes(scope: Construct, id: string, attrs: ModelAttributes): IModel;
    /**
     * Returns the ARN of this model.
     * @attribute
     */
    readonly modelArn: string;
    /**
     * Returns the name of the model.
     * @attribute
     */
    readonly modelName: string;
    /**
     * Execution role for SageMaker Model
     */
    readonly role?: iam.IRole;
    /**
     * The principal this Model is running as
     */
    readonly grantPrincipal: iam.IPrincipal;
    private readonly subnets;
    private readonly containers;
    constructor(scope: Construct, id: string, props?: ModelProps);
    /**
     * Add containers to the model.
     *
     * @param container The container definition to add.
     */
    addContainer(container: ContainerDefinition): void;
    private validateContainers;
    private renderPrimaryContainer;
    private renderContainers;
    private renderContainer;
    private configureNetworking;
    private renderVpcConfig;
    private createSageMakerRole;
}
export {};
