# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dynamodb_session_flask']

package_data = \
{'': ['*']}

install_requires = \
['Flask>=2.0.0,<3.0.0', 'dynamodb-session-web>=0.2.7,<0.3.0']

setup_kwargs = {
    'name': 'dynamodb-session-flask',
    'version': '1.1.0',
    'description': 'A session implementation for Flask using DynamoDB as a backing store and OWASP best practices for session management.',
    'long_description': '# dynamodb-session-flask\n\nAn implementation of a Flask session using DynamoDB as backend storage. This project was built on \n[dynamodb-session-web](https://github.com/JCapriotti/dynamodb-session-web), but with support for the Flask framework.\n\n## Why This Library?\n\nI tried and acquired an appreciation for some other DynamoDB backend implementations for Flask sessions. \nHowever, I needed a few extra things:\n* Absolute and Idle Timeouts\n* Support for using a header (not a cookie) for session ID\n\nIn addition to the [OWASP Session Management](https://cheatsheetseries.owasp.org/cheatsheets/Session_Management_Cheat_Sheet.html) \nbest practices implemented in `dynamodb-session-web`, \nthis project has additional support for these best practices:\n\n* Non-descript session ID name - Defaults to `id` for cookies, and `x-id` for headers. \n  * Side-Comment - isn\'t a non-descript suggestion for a name actually descriptive?\n* Cookie setting defaults:\n  - [X] Secure = True\n  - [X] HttpOnly = True\n  - [X] SameSite = Strict\n  - [ ] Domain and Path - Must set these yourself\n* ID Exchange\n  - [X] Accepted session ID mechanism (i.e. cookie vs header) is enforced. That is, user cannot submit session IDs \n        through a header if cookie is expected.\n\n## Usage\n\nRequires a DynamoDB table named `app_session` (can be changed in settings). \n\nHere\'s an example table creation statement:\n\n```shell\naws dynamodb create-table \\\n    --attribute-definitions \\\n        AttributeName=id,AttributeType=S \\\n    --key-schema "AttributeName=id,KeyType=HASH" \\\n    --provisioned-throughput "ReadCapacityUnits=5,WriteCapacityUnits=5" \\\n    --table-name app_session \n```\n\nSessions are intended to operate just like the default Flask session implementation:\n\n```python\nfrom flask import Flask, session\nfrom dynamodb_session_flask import DynamoDbSession\n\nflask_app = Flask(__name__)\nflask_app.session_interface = DynamoDbSession()\n\n@flask_app.route(\'/save\')\ndef save():\n    session[\'val\'] = \'My Value\'\n    return \'Success\', 200\n\n@flask_app.route(\'/load\')\ndef load():\n    saved_val = session[\'val\']\n    return saved_val, 200\n\n@flask_app.route(\'/end\')\ndef end_session():\n    # This will remove the session from the database and remove the session ID from cookies/headers\n    session.clear()\n    return \'Success\', 200\n```\n\nIf using the extra methods that are provided ([see below](#Session Instance Methods)), \nyou may find it useful to have an extra module-level variable. It helps with IDE code completion.\n\n```python\nfrom typing import cast\n\nfrom flask import Flask, session as flask_session\nfrom dynamodb_session_flask import DynamoDbSessionInstance\n\ndynamodb_session = cast(DynamoDbSessionInstance, flask_session)\n\n\ndef abandon_session():\n    dynamodb_session.abandon()\n```\n\n### Behavior\n\nWorks within the Flask session interface:\n* Presents a dictionary-like interface for getting/setting values.\n* Session is loaded at the start of the request, and saved at the end.\n\nAdditional behaviors:\n* Session is not saved and ID is not returned if the session is new and no data is added.\n\n### Session Instance Methods\n\nWhile this session implementation is backwards compatible with the Flask session functionality/interface, \nthere are some additional methods available that can be used if needed.\n\n### `abandon()`\n<div style="margin-left: 30px;">\nImmediately removes the session from the database. \n</div>\n\n### `create()`\n<div style="margin-left: 30px;">\nCreates a new session, with new ID. Does not remove the old session record.\n</div>\n\n### `save()`\n<div style="margin-left: 30px;">\nManually saves the session.\n\nThis method is not usually needed since Flask will save the session at the end of a request.\nHowever, it is provided for cases where the session must be saved earlier.\n</div>\n\n## Configuration\n\nThere are additional configuration options, and are set like normal Flask configuration:\n\n```python\nflask_app = Flask(__name__)\nflask_app.config.update(\n    SESSION_DYNAMODB_IDLE_TIMEOUT=600\n)\n```\n\nAll configuration is optional, assuming the defaults are okay.\n\n### `SESSION_DYNAMODB_ABSOLUTE_TIMEOUT`\n<div style="margin-left: 30px;">\nAbsolute session timeout (in seconds).\n\nNote: This setting works in conjunction with Flask\'s [`PERMANENT_SESSION_LIFETIME`](https://flask.palletsprojects.com/en/2.0.x/config/#PERMANENT_SESSION_LIFETIME) setting.  The absolute timeout chosen will be whichever is less.\n\nDefault: `43200` (12 hours)\n</div>\n\n### `SESSION_DYNAMODB_ENDPOINT_URL`\n<div style="margin-left: 30px;">\nThe DynamoDB URL.\n\nDefault: `None` (i.e. Boto3 logic)\n</div>\n\n### `SESSION_DYNAMODB_HEADER_NAME`\n<div style="margin-left: 30px;">\nThe name of the header to use for the session ID.\n\nDefault: `x-id`\n</div>\n\n### `SESSION_DYNAMODB_IDLE_TIMEOUT`\n<div style="margin-left: 30px;">\nIdle session timeout (in seconds).\n\nDefault: `7200` (2 hours)\n</div>\n\n### `SESSION_DYNAMODB_SID_BYTE_LENGTH`\n<div style="margin-left: 30px;">\nSession ID length in bytes. \n\nThis does not correlate to the character length of the ID, which will be either:\n\n* 43 - How many characters a 32-byte value uses when Base64 encoded. \n* 71 - The 43 characters from the previous bullet, plus a dot and finally a 27-character HMAC signature. \n\nDefault: `32`\n</div>\n\n### `SESSION_DYNAMODB_SID_KEYS`\n<div style="margin-left: 30px;">\nFor a slightly more secure session ID, the key can be signed using a configurable and rotatable key. \n\nThe signature is generated using [`itsdangerous`](https://itsdangerous.palletsprojects.com/en/2.1.x/) and includes key rotation. If/When rotation is desired, the array is used in order from oldest to newest. Otherwise, one key is all that is needed.\n\nAn empty array means no signature is generated.\n\nDefault: `[]` (no signature)\n</div>\n\n### `SESSION_DYNAMODB_TABLE_NAME`\n<div style="margin-left: 30px;">\nThe name of the DynamoDB table.\n\nDefault: `app_session`\n</div>\n\n### `SESSION_DYNAMODB_OVERRIDE_COOKIE_NAME`\n<div style="margin-left: 30px;">\nWhether or not to override Flask\'s [SESSION_COOKIE_NAME](https://flask.palletsprojects.com/en/2.0.x/config/#SESSION_COOKIE_NAME)\nconfiguration for the session ID. While somewhat trivial, OWASP\'s recommended value is \n`id` and Flask\'s default is `session`. So to avoid using Flask\'s default or modifying it behind the scenes, this setting\nhelps separate this library\'s preferred default from Flask\'s.\n\nSetting this to `True` will set the cookie name to `id`. Otherwise, Flask\'s configuration will be used.\n\nDefault: `True`\n</div>\n\n### `SESSION_DYNAMODB_OVERRIDE_COOKIE_SECURE`\n<div style="margin-left: 30px;">\nWhether or not to override Flask\'s [`SESSION_COOKIE_SECURE`](https://flask.palletsprojects.com/en/2.0.x/config/#SESSION_COOKIE_SECURE)\nfor the cookie\'s Secure attribute. Flask defaults that attribute to `False`, whereas this should ideally be `True` to prevent \nMan-in-the-Middle attacks. \n\nSetting this to `True` will force the Secure attribute to also be `True`. Otherwise, Flask\'s configuration will be used.\n\nNote: You\'ll want to set this to `False` in any environment where TLS is not used (e.g. local development).\n\nDefault: `True`\n</div>\n\n### `SESSION_DYNAMODB_USE_HEADER`\n<div style="margin-left: 30px;">\nWhether or not to check for the session ID via headers. The cookie value is still used by default (if found).\n\nDefault: `False`\n</div>\n\n### `SESSION_COOKIE_SAMESITE`\n<div style="margin-left: 30px;">\nThis is actually a Flask configuration, which defaults to `None`. However, if the value is `None`, then we set it to \n`Strict` by default.\n\nDefault: `Strict` (indirectly changed)\n</div>\n\n\n## Testing\n\nFlask has a [pattern for accessing the session](https://flask.palletsprojects.com/en/2.0.x/testing/#accessing-and-modifying-the-session) when running tests.\nThis mechanism still uses the backend `session_interface` set for the app (i.e. it will still use DynamoDB). \n\nTo help reduce dependencies when simply trying to run unit tests that need a value set in the session, there\'s a \nseparate `session_interface` that can be used.\n\nBelow is a working example, copied from [this project\'s tests](tests/test_testing.py). Improvements could be made depending on test expectations.\n\n```python\nimport pytest\nfrom dynamodb_session_flask.testing import TestSession\nfrom flask import Flask, session\n\n\n@pytest.fixture\ndef app():\n    flask_app = Flask(__name__)\n\n    @flask_app.route(\'/load\')\n    def load():\n        return {\n            \'actual_value\': session.get(\'val\', None),\n        }\n\n    yield flask_app\n\n\n@pytest.fixture()\ndef test_client(app):\n    app.session_interface = TestSession()\n    return app.test_client()\n\n\ndef test_able_to_use_test_session_transaction(test_client):\n    expected_value = \'fake_value\'\n\n    with test_client:\n        with test_client.session_transaction() as test_session:\n            test_session[\'val\'] = expected_value\n\n        response = test_client.get(\'/load\')\n\n        assert response.json[\'actual_value\'] == expected_value\n```\n',
    'author': 'Jason Capriotti',
    'author_email': 'jason.capriotti@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/JCapriotti/dynamodb-session-flask',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
