"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const api = require("aws-cdk-lib/aws-apigateway");
const waf = require("aws-cdk-lib/aws-wafv2");
const defaults = require("@aws-solutions-constructs/core");
require("@aws-cdk/assert/jest");
function deployConstruct(stack, constructProps) {
    const restApi = new api.RestApi(stack, 'test-api', {});
    restApi.root.addMethod('ANY');
    const props = constructProps ?
        { webaclProps: constructProps, existingApiGatewayInterface: restApi }
        : { existingApiGatewayInterface: restApi };
    return new lib_1.WafwebaclToApiGateway(stack, 'test-wafwebacl-apigateway', props);
}
// --------------------------------------------------------------
// Test error handling for existing WAF web ACL and user provided web ACL props
// --------------------------------------------------------------
test('Test error handling for existing WAF web ACL and user provider web ACL props', () => {
    const stack = new cdk.Stack();
    const props = {
        defaultAction: {
            allow: {}
        },
        scope: 'REGIONAL',
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
    };
    const wafAcl = new waf.CfnWebACL(stack, 'test-waf', props);
    const restApi = new api.RestApi(stack, 'empty-api', {});
    expect(() => {
        new lib_1.WafwebaclToApiGateway(stack, 'test-waf-gateway', {
            existingApiGatewayInterface: restApi,
            existingWebaclObj: wafAcl,
            webaclProps: props
        });
    }).toThrowError();
});
// --------------------------------------------------------------
// Test default deployment
// --------------------------------------------------------------
test('Test default deployment', () => {
    const stack = new cdk.Stack();
    const construct = deployConstruct(stack);
    expect(construct.webacl !== null);
    expect(construct.apiGateway !== null);
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        Rules: [
            {
                Name: "AWS-AWSManagedRulesBotControlRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesBotControlRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesBotControlRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesKnownBadInputsRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesKnownBadInputsRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesKnownBadInputsRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 2,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAnonymousIpList",
                OverrideAction: {
                    None: {}
                },
                Priority: 3,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAnonymousIpList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAnonymousIpList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAmazonIpReputationList",
                OverrideAction: {
                    None: {}
                },
                Priority: 4,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAmazonIpReputationList",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAmazonIpReputationList",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesAdminProtectionRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 5,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesAdminProtectionRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesAdminProtectionRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesSQLiRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 6,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesSQLiRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesSQLiRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
// --------------------------------------------------------------
// Test web acl with user provided acl props
// --------------------------------------------------------------
test('Test user provided acl props', () => {
    const stack = new cdk.Stack();
    const webaclProps = {
        defaultAction: {
            allow: {}
        },
        scope: 'REGIONAL',
        visibilityConfig: {
            cloudWatchMetricsEnabled: false,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
        rules: [
            defaults.wrapManagedRuleSet("AWSManagedRulesCommonRuleSet", "AWS", 0),
            defaults.wrapManagedRuleSet("AWSManagedRulesWordPressRuleSet", "AWS", 1),
        ]
    };
    deployConstruct(stack, webaclProps);
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        VisibilityConfig: {
            CloudWatchMetricsEnabled: false,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        },
        Rules: [
            {
                Name: "AWS-AWSManagedRulesCommonRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 0,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesCommonRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesCommonRuleSet",
                    SampledRequestsEnabled: true
                }
            },
            {
                Name: "AWS-AWSManagedRulesWordPressRuleSet",
                OverrideAction: {
                    None: {}
                },
                Priority: 1,
                Statement: {
                    ManagedRuleGroupStatement: {
                        Name: "AWSManagedRulesWordPressRuleSet",
                        VendorName: "AWS"
                    }
                },
                VisibilityConfig: {
                    CloudWatchMetricsEnabled: true,
                    MetricName: "AWSManagedRulesWordPressRuleSet",
                    SampledRequestsEnabled: true
                }
            }
        ]
    });
});
// --------------------------------------------------------------
// Test existing web ACL
// --------------------------------------------------------------
test('Test existing web ACL', () => {
    const stack = new cdk.Stack();
    const webacl = new waf.CfnWebACL(stack, 'test-webacl', {
        defaultAction: {
            allow: {}
        },
        scope: 'REGIONAL',
        visibilityConfig: {
            cloudWatchMetricsEnabled: true,
            metricName: 'webACL',
            sampledRequestsEnabled: true
        },
    });
    const restApi = new api.RestApi(stack, 'test-api', {});
    restApi.root.addMethod('ANY');
    new lib_1.WafwebaclToApiGateway(stack, 'test-wafwebacl-apigateway', {
        existingWebaclObj: webacl,
        existingApiGatewayInterface: restApi
    });
    expect(stack).toHaveResource("AWS::WAFv2::WebACL", {
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: "webACL",
            SampledRequestsEnabled: true
        }
    });
    expect(stack).toCountResources("AWS::WAFv2::WebACL", 1);
});
//# sourceMappingURL=data:application/json;base64,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