import pytest
from freezegun import freeze_time
from typer.testing import CliRunner

from lib.test_lib import assert_lines_match, isolate_frame, path_in_test_data

HEADER = open(path_in_test_data(__file__, "test_header_entry.txt", local=False)).read()

MOLECULAR_SYSTEM_NMRPIPE = "nef_molecular_system"
METADATA_NMRPIPE = "nef_nmr_meta_data"
NMRPIPE_IMPORT_SEQUENCE = ["nmrpipe", "import", "sequence"]

runner = CliRunner()


@pytest.fixture
def using_nmrpipe():
    # register the module under test
    import transcoders.nmrpipe  # noqa: F401


EXPECTED_3AA = """\
save_nef_molecular_system
   _nef_molecular_system.sf_category   nef_molecular_system
   _nef_molecular_system.sf_framecode  nef_molecular_system

   loop_
      _nef_sequence.index
      _nef_sequence.chain_code
      _nef_sequence.sequence_code
      _nef_sequence.residue_name
      _nef_sequence.linking
      _nef_sequence.residue_variant
      _nef_sequence.cis_peptide

     1   A   1   ALA   start    .   .
     2   A   2   ALA   middle   .   .
     3   A   3   ALA   end      .   .

   stop_

save_"""

EXPECTED_3AA10 = """\
save_nef_molecular_system
   _nef_molecular_system.sf_category   nef_molecular_system
   _nef_molecular_system.sf_framecode  nef_molecular_system

   loop_
      _nef_sequence.index
      _nef_sequence.chain_code
      _nef_sequence.sequence_code
      _nef_sequence.residue_name
      _nef_sequence.linking
      _nef_sequence.residue_variant
      _nef_sequence.cis_peptide

     1   A   10   ALA   start    .   .
     2   A   11   ALA   middle   .   .
     3   A   12   ALA   end      .   .

   stop_

save_"""

NMRVIEW_IMPORT_SEQUENCE = ["nmrview", "import", "sequence"]


# noinspection PyUnusedLocal
def test_3aa(typer_app, using_nmrpipe, monkeypatch):

    monkeypatch.setattr("sys.stdin.isatty", lambda: False)

    path = path_in_test_data(__file__, "3aa.tab")
    result = runner.invoke(typer_app, [*NMRPIPE_IMPORT_SEQUENCE, path], input=HEADER)

    assert result.exit_code == 0
    mol_sys_result = isolate_frame(result.stdout, "%s" % MOLECULAR_SYSTEM_NMRPIPE)

    assert_lines_match(EXPECTED_3AA, mol_sys_result)


# noinspection PyUnusedLocal
def test_3aa10(typer_app, using_nmrpipe, monkeypatch):

    monkeypatch.setattr("sys.stdin.isatty", lambda: False)

    path = path_in_test_data(__file__, "3aa.tab")
    result = runner.invoke(
        typer_app, [*NMRPIPE_IMPORT_SEQUENCE, "--start", "10", path], input=HEADER
    )

    assert result.exit_code == 0

    mol_sys_result = isolate_frame(result.stdout, "%s" % MOLECULAR_SYSTEM_NMRPIPE)

    assert_lines_match(EXPECTED_3AA10, mol_sys_result)


#
# HEADER = open(path_in_test_data(__file__,'test_header_entry.txt', local=False)).read()

EXPECTED_HEADER = """\
save_nef_nmr_meta_data
   _nef_nmr_meta_data.sf_category      nef_nmr_meta_data
   _nef_nmr_meta_data.sf_framecode     nef_nmr_meta_data
   _nef_nmr_meta_data.format_name      nmr_exchange_format
   _nef_nmr_meta_data.format_version   1.1
   _nef_nmr_meta_data.program_name     NEFPipelines
   _nef_nmr_meta_data.script_name      util.py
   _nef_nmr_meta_data.program_version  0.0.1
   _nef_nmr_meta_data.creation_date    2012-01-14T12:00:01.123456
   _nef_nmr_meta_data.uuid             NEFPipelines-2012-01-14T12:00:01.123456-1043321819
   _nef_nmr_meta_data.creation_time    2012-01-14T12:00:01.123456

   loop_
      _nef_run_history.run_number
      _nef_run_history.program_name
      _nef_run_history.program_version
      _nef_run_history.script_name

     1   NEFPipelines   0.0.1   header.py

   stop_

save_
"""


# noinspection PyUnusedLocal
@freeze_time("2012-01-14 12:00:01.123456")
def test_pipe_header(typer_app, using_nmrpipe, monkeypatch, fixed_seed):

    monkeypatch.setattr("sys.stdin.isatty", lambda: True)

    path = path_in_test_data(__file__, "3aa.tab")
    path_header = path_in_test_data(__file__, "test_header_entry.txt", local=False)
    result = runner.invoke(
        typer_app, [*NMRPIPE_IMPORT_SEQUENCE, "--pipe", path_header, path]
    )

    assert result.exit_code == 0

    mol_sys_result = isolate_frame(result.stdout, "%s" % MOLECULAR_SYSTEM_NMRPIPE)
    meta_data_result = isolate_frame(result.stdout, "%s" % METADATA_NMRPIPE)

    assert_lines_match(EXPECTED_3AA, mol_sys_result)
    assert_lines_match(EXPECTED_HEADER, meta_data_result)


# noinspection PyUnusedLocal
@freeze_time("2012-01-14 12:00:01.123456")
def test_header(typer_app, using_nmrpipe, monkeypatch, fixed_seed):

    monkeypatch.setattr("sys.stdin.isatty", lambda: False)

    path = path_in_test_data(__file__, "3aa.tab")
    result = runner.invoke(typer_app, [*NMRPIPE_IMPORT_SEQUENCE, path], input=HEADER)

    assert result.exit_code == 0

    mol_sys_result = isolate_frame(result.stdout, "%s" % MOLECULAR_SYSTEM_NMRPIPE)

    assert_lines_match(EXPECTED_3AA, mol_sys_result)


if __name__ == "__main__":
    pytest.main([f"{__file__}", "-vv"])
