"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
function getTestStack() {
    return new core_1.Stack(undefined, 'TestStack', { env: { account: '123456789012', region: 'us-east-1' } });
}
function toCfnTags(tags) {
    return Object.keys(tags).map(key => {
        return { Key: key, Value: tags[key] };
    });
}
function hasTags(expectedTags) {
    return (props) => {
        try {
            const tags = props.Tags;
            const actualTags = tags.filter((tag) => {
                for (const expectedTag of expectedTags) {
                    if (assert_1.isSuperObject(expectedTag, tag)) {
                        return true;
                    }
                    else {
                        continue;
                    }
                }
                // no values in array so expecting empty
                return false;
            });
            return actualTags.length === expectedTags.length;
        }
        catch (e) {
            // tslint:disable:no-console
            console.error('Tags are incorrect');
            console.error('found tags ', props.Tags);
            console.error('expected tags ', expectedTags);
            // tslint:enable:no-console
            throw e;
        }
    };
}
module.exports = {
    'When creating a VPC': {
        'with the default CIDR range': {
            'vpc.vpcId returns a token to the VPC ID'(test) {
                const stack = getTestStack();
                const vpc = new lib_1.Vpc(stack, 'TheVPC');
                test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
                test.done();
            },
            'it uses the correct network range'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                    CidrBlock: lib_1.Vpc.DEFAULT_CIDR_RANGE,
                    EnableDnsHostnames: true,
                    EnableDnsSupport: true,
                    InstanceTenancy: lib_1.DefaultInstanceTenancy.DEFAULT,
                }));
                test.done();
            },
            'the Name tag is defaulted to path'(test) {
                const stack = getTestStack();
                new lib_1.Vpc(stack, 'TheVPC');
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::InternetGateway', hasTags([{ Key: 'Name', Value: 'TheVPC' }])));
                test.done();
            },
        },
        'with all of the properties set, it successfully sets the correct VPC properties'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '192.168.0.0/16',
                enableDnsHostnames: false,
                enableDnsSupport: false,
                defaultInstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16',
                EnableDnsHostnames: false,
                EnableDnsSupport: false,
                InstanceTenancy: lib_1.DefaultInstanceTenancy.DEDICATED,
            }));
            test.done();
        },
        'contains the correct number of subnets'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const zones = stack.availabilityZones.length;
            test.equal(vpc.publicSubnets.length, zones);
            test.equal(vpc.privateSubnets.length, zones);
            test.deepEqual(stack.resolve(vpc.vpcId), { Ref: 'TheVPC92636AB0' });
            test.done();
        },
        'with only isolated subnets, the VPC should not contain an IGW or NAT Gateways'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    }
                ]
            });
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::InternetGateway'));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                MapPublicIpOnLaunch: false
            }));
            test.done();
        },
        'with no private subnets, the VPC should have an IGW but no NAT Gateways'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED,
                        name: 'Isolated',
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            test.done();
        },
        'with private subnets and custom networkAcl.'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', {
                subnetConfiguration: [
                    {
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        subnetType: lib_1.SubnetType.PRIVATE,
                        name: 'private',
                    }
                ]
            });
            const nacl1 = new lib_1.NetworkAcl(stack, 'myNACL1', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PRIVATE },
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSEgress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.EGRESS,
                cidr: lib_1.AclCidr.ipv4('10.0.0.0/16'),
            });
            new lib_1.NetworkAclEntry(stack, 'AllowDNSIngress', {
                networkAcl: nacl1,
                ruleNumber: 100,
                traffic: lib_1.AclTraffic.udpPort(53),
                direction: lib_1.TrafficDirection.INGRESS,
                cidr: lib_1.AclCidr.anyIpv4(),
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAcl', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NetworkAclEntry', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SubnetNetworkAclAssociation', 3));
            test.done();
        },
        'with no subnets defined, the VPC should have an IGW, and a NAT Gateway per AZ'(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {});
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', zones));
            test.done();
        },
        'with subnets and reserved subnets defined, VPC subnet count should not contain reserved subnets '(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        subnetType: lib_1.SubnetType.PUBLIC,
                        name: 'Public',
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            test.done();
        },
        'with reserved subnets, any other subnets should not have cidrBlock from within reserved space'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/16',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'reserved',
                        subnetType: lib_1.SubnetType.PRIVATE,
                        reserved: true,
                    },
                    {
                        cidrMask: 24,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    }
                ],
                maxAzs: 3
            });
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 3; i < 6; i++) {
                assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 6; i < 9; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            test.done();
        },
        'with custom subnets, the VPC should have the right number of subnets, an IGW, and a NAT Gateway per AZ'(test) {
            const stack = getTestStack();
            const zones = stack.availabilityZones.length;
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', zones));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.6.${i * 16}/28`
                }));
            }
            test.done();
        },
        'with custom subents and natGateways = 2 there should be only two NATGW'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', {
                cidr: '10.0.0.0/21',
                natGateways: 2,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'application',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                    {
                        cidrMask: 28,
                        name: 'rds',
                        subnetType: lib_1.SubnetType.ISOLATED,
                    }
                ],
                maxAzs: 3
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::InternetGateway', 1));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 2));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 9));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i}.0/24`
                }));
            }
            for (let i = 0; i < 3; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.6.${i * 16}/28`
                }));
            }
            test.done();
        },
        'with enableDnsHostnames enabled but enableDnsSupport disabled, should throw an Error'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'TheVPC', {
                enableDnsHostnames: true,
                enableDnsSupport: false
            }));
            test.done();
        },
        'with public subnets MapPublicIpOnLaunch is true'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                maxAzs: 1,
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    }
                ],
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 1));
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::NatGateway'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                MapPublicIpOnLaunch: true
            }));
            test.done();
        },
        'maxAZs defaults to 3 if unset'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC');
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 6));
            for (let i = 0; i < 6; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i * 32}.0/19`
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with maxAZs set to 2'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', { maxAzs: 2 });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 4));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 4));
            for (let i = 0; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', {
                    CidrBlock: `10.0.${i * 64}.0/18`
                }));
            }
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway set to 1'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                natGateways: 1,
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Subnet', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Route', 6));
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 1));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NatGatewayId: {},
            }));
            test.done();
        },
        'with natGateway subnets defined'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'egress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'egress'
                },
            });
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::NatGateway', 3));
            for (let i = 1; i < 4; i++) {
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Subnet', hasTags([{
                        Key: 'Name',
                        Value: `VPC/egressSubnet${i}`,
                    }, {
                        Key: 'aws-cdk:subnet-name',
                        Value: 'egress',
                    }])));
            }
            test.done();
        },
        'natGateways = 0 requires there to be no PRIVATE subnets'(test) {
            const stack = getTestStack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'VPC', {
                    natGateways: 0,
                });
            }, /make sure you don't configure any PRIVATE subnets/);
            test.done();
        },
        'with mis-matched nat and subnet configs it throws'(test) {
            const stack = getTestStack();
            test.throws(() => new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    {
                        cidrMask: 24,
                        name: 'ingress',
                        subnetType: lib_1.SubnetType.PUBLIC,
                    },
                    {
                        cidrMask: 24,
                        name: 'private',
                        subnetType: lib_1.SubnetType.PRIVATE,
                    },
                ],
                natGatewaySubnets: {
                    subnetGroupName: 'notthere',
                },
            }));
            test.done();
        },
        'with a vpn gateway'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                vpnGateway: true,
                vpnGatewayAsn: 65000
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGateway', {
                AmazonSideAsn: 65000,
                Type: 'ipsec.1'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCGatewayAttachment', {
                VpcId: {
                    Ref: 'VPCB9E5F0B4'
                },
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027'
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E'
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.ISOLATED
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'with a vpn gateway and route propagation on private and isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'Private' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
                vpnRoutePropagation: [
                    {
                        subnetType: lib_1.SubnetType.PRIVATE
                    },
                    {
                        subnetType: lib_1.SubnetType.ISOLATED
                    }
                ]
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPrivateSubnet1RouteTableBE8A6027'
                    },
                    {
                        Ref: 'VPCPrivateSubnet2RouteTable0A19E10E'
                    },
                    {
                        Ref: 'VPCPrivateSubnet3RouteTable192186F8'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'route propagation defaults to isolated subnets when there are no private subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCIsolatedSubnet1RouteTableEB156210'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet2RouteTable9B4F78DC'
                    },
                    {
                        Ref: 'VPCIsolatedSubnet3RouteTableCB6A1FDA'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'route propagation defaults to public subnets when there are no private/isolated subnets'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                ],
                vpnGateway: true,
            });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNGatewayRoutePropagation', {
                RouteTableIds: [
                    {
                        Ref: 'VPCPublicSubnet1RouteTableFEE4B781'
                    },
                    {
                        Ref: 'VPCPublicSubnet2RouteTable6F1A15F1'
                    },
                    {
                        Ref: 'VPCPublicSubnet3RouteTable98AE0E14'
                    }
                ],
                VpnGatewayId: {
                    Ref: 'VPCVpnGatewayB5ABAE68'
                }
            }));
            test.done();
        },
        'fails when specifying vpnConnections with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnConnections: {
                    VpnConnection: {
                        asn: 65000,
                        ip: '192.0.2.1'
                    }
                }
            }), /`vpnConnections`.+`vpnGateway`.+false/);
            test.done();
        },
        'fails when specifying vpnGatewayAsn with vpnGateway set to false'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
                vpnGateway: false,
                vpnGatewayAsn: 65000,
            }), /`vpnGatewayAsn`.+`vpnGateway`.+false/);
            test.done();
        },
        'Subnets have a defaultChild'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            test.ok(vpc.publicSubnets[0].node.defaultChild instanceof lib_1.CfnSubnet);
            test.done();
        },
        'CIDR cannot be a Token'(test) {
            const stack = new core_1.Stack();
            test.throws(() => {
                new lib_1.Vpc(stack, 'Vpc', {
                    cidr: core_1.Lazy.stringValue({ produce: () => 'abc' })
                });
            }, /property must be a concrete CIDR string/);
            test.done();
        },
        'Default NAT gateway provider'(test) {
            const stack = new core_1.Stack();
            const natGatewayProvider = lib_1.NatProvider.gateway();
            new lib_1.Vpc(stack, 'VpcNetwork', { natGatewayProvider });
            test.ok(natGatewayProvider.configuredGateways.length > 0);
            test.done();
        },
        'Can add an IPv6 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationIpv6CidrBlock: '2001:4860:4860::8888/32',
                NetworkInterfaceId: 'router-1'
            }));
            test.done();
        },
        'Can add an IPv4 route'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'VPC');
            vpc.publicSubnets[0].addRoute('SomeRoute', {
                destinationCidrBlock: '0.0.0.0/0',
                routerId: 'router-1',
                routerType: lib_1.RouterType.NETWORK_INTERFACE
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Route', {
                DestinationCidrBlock: '0.0.0.0/0',
                NetworkInterfaceId: 'router-1'
            }));
            test.done();
        },
    },
    'NAT instances': {
        'Can configure NAT instances instead of NAT gateways'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const natGatewayProvider = lib_1.NatProvider.instance({
                instanceType: new lib_1.InstanceType('q86.mega'),
                machineImage: new lib_1.GenericLinuxImage({
                    'us-east-1': 'ami-1'
                })
            });
            new lib_1.Vpc(stack, 'TheVPC', { natGatewayProvider });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 3));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
                ImageId: 'ami-1',
                InstanceType: 'q86.mega',
                SourceDestCheck: false,
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Route', {
                RouteTableId: { Ref: 'TheVPCPrivateSubnet1RouteTableF6513BC2' },
                DestinationCidrBlock: '0.0.0.0/0',
                InstanceId: { Ref: 'TheVPCPublicSubnet1NatInstanceCC514192' }
            }));
            test.done();
        },
        'natGateways controls amount of NAT instances'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            new lib_1.Vpc(stack, 'TheVPC', {
                natGatewayProvider: lib_1.NatProvider.instance({
                    instanceType: new lib_1.InstanceType('q86.mega'),
                    machineImage: new lib_1.GenericLinuxImage({
                        'us-east-1': 'ami-1'
                    })
                }),
                natGateways: 1
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Instance', 1));
            test.done();
        },
    },
    'Network ACL association': {
        'by default uses default ACL reference'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { 'Fn::GetAtt': ['TheVPCPublicSubnet1Subnet770D4FF2', 'NetworkAclAssociationId'] }
                    }
                }
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
        'if ACL is replaced new ACL reference is returned'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            // WHEN
            new core_1.CfnOutput(stack, 'Output', {
                value: vpc.publicSubnets[0].subnetNetworkAclAssociationId
            });
            new lib_1.NetworkAcl(stack, 'ACL', {
                vpc,
                subnetSelection: { subnetType: lib_1.SubnetType.PUBLIC }
            });
            assert_1.expect(stack).toMatch({
                Outputs: {
                    Output: {
                        Value: { Ref: 'ACLDBD1BB49' }
                    }
                }
            }, assert_1.MatchStyle.SUPERSET);
            test.done();
        },
    },
    'When creating a VPC with a custom CIDR range': {
        'vpc.vpcCidrBlock is the correct network range'(test) {
            const stack = getTestStack();
            new lib_1.Vpc(stack, 'TheVPC', { cidr: '192.168.0.0/16' });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', {
                CidrBlock: '192.168.0.0/16'
            }));
            test.done();
        }
    },
    'When tagging': {
        'VPC propagated tags will be on subnet, IGW, routetables, NATGW'(test) {
            const stack = getTestStack();
            const tags = {
                VpcType: 'Good',
            };
            const noPropTags = {
                BusinessUnit: 'Marketing',
            };
            const allTags = { ...tags, ...noPropTags };
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            // overwrite to set propagate
            vpc.node.applyAspect(new core_1.Tag('BusinessUnit', 'Marketing', { includeResourceTypes: [lib_1.CfnVPC.CFN_RESOURCE_TYPE_NAME] }));
            vpc.node.applyAspect(new core_1.Tag('VpcType', 'Good'));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags(toCfnTags(allTags))));
            const taggables = ['Subnet', 'InternetGateway', 'NatGateway', 'RouteTable'];
            const propTags = toCfnTags(tags);
            const noProp = toCfnTags(noPropTags);
            for (const resource of taggables) {
                assert_1.expect(stack).to(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(propTags)));
                assert_1.expect(stack).notTo(assert_1.haveResource(`AWS::EC2::${resource}`, hasTags(noProp)));
            }
            test.done();
        },
        'Subnet Name will propagate to route tables and NATGW'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            for (const subnet of vpc.publicSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::NatGateway', hasTags([tag])));
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            for (const subnet of vpc.privateSubnets) {
                const tag = { Key: 'Name', Value: subnet.node.path };
                assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::RouteTable', hasTags([tag])));
            }
            test.done();
        },
        'Tags can be added after the Vpc is created with `vpc.tags.setTag(...)`'(test) {
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'TheVPC');
            const tag = { Key: 'Late', Value: 'Adder' };
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            vpc.node.applyAspect(new core_1.Tag(tag.Key, tag.Value));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPC', hasTags([tag])));
            test.done();
        },
    },
    'subnet selection': {
        'selecting default subnets returns the private ones'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select public subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PUBLIC });
            // THEN
            test.deepEqual(subnetIds, vpc.publicSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select isolated subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'Public' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'Isolated' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetType: lib_1.SubnetType.ISOLATED });
            // THEN
            test.deepEqual(subnetIds, vpc.isolatedSubnets.map(s => s.subnetId));
            test.done();
        },
        'can select subnets by name'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetGroupName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'subnetName is an alias for subnetGroupName (backwards compat)'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VPC', {
                subnetConfiguration: [
                    { subnetType: lib_1.SubnetType.PUBLIC, name: 'BlaBla' },
                    { subnetType: lib_1.SubnetType.PRIVATE, name: 'DontTalkToMe' },
                    { subnetType: lib_1.SubnetType.ISOLATED, name: 'DontTalkAtAll' },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnetName: 'DontTalkToMe' });
            // THEN
            test.deepEqual(subnetIds, vpc.privateSubnets.map(s => s.subnetId));
            test.done();
        },
        'selecting default subnets in a VPC with only isolated subnets returns the isolateds'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                isolatedSubnetIds: ['iso-1', 'iso-2', 'iso-3'],
                isolatedSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['iso-1', 'iso-2', 'iso-3']);
            test.done();
        },
        'selecting default subnets in a VPC with only public subnets returns the publics'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            // WHEN
            const subnets = vpc.selectSubnets();
            // THEN
            test.deepEqual(subnets.subnetIds, ['pub-1', 'pub-2', 'pub-3']);
            test.done();
        },
        'selecting subnets by name fails if the name is unknown'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VPC');
            test.throws(() => {
                vpc.selectSubnets({ subnetGroupName: 'Toot' });
            }, /There are no subnet groups with name 'Toot' in this VPC. Available names: Public,Private/);
            test.done();
        },
        'select subnets with az restriction'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
                maxAzs: 1,
                subnetConfiguration: [
                    { name: 'lb', subnetType: lib_1.SubnetType.PUBLIC },
                    { name: 'app', subnetType: lib_1.SubnetType.PRIVATE },
                    { name: 'db', subnetType: lib_1.SubnetType.PRIVATE },
                ]
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ onePerAz: true });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], vpc.privateSubnets[0].subnetId);
            test.done();
        },
        'select explicitly defined subnets'(test) {
            // GIVEN
            const stack = getTestStack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'vpc-1234',
                availabilityZones: ['dummy1a', 'dummy1b', 'dummy1c'],
                publicSubnetIds: ['pub-1', 'pub-2', 'pub-3'],
                publicSubnetRouteTableIds: ['rt-1', 'rt-2', 'rt-3'],
            });
            const subnet = new lib_1.PrivateSubnet(stack, 'Subnet', {
                availabilityZone: vpc.availabilityZones[0],
                cidrBlock: '10.0.0.0/28',
                vpcId: vpc.vpcId
            });
            // WHEN
            const { subnetIds } = vpc.selectSubnets({ subnets: [subnet] });
            // THEN
            test.deepEqual(subnetIds.length, 1);
            test.deepEqual(subnetIds[0], subnet.subnetId);
            test.done();
        },
        'subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.done();
        },
        'Referencing AZ throws error when subnet created from subnetId'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetId(stack, 'subnet1', 'pub-1');
            // THEN
            // tslint:disable-next-line: max-line-length
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'Referencing AZ throws error when subnet created from attributes without az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: '' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            // tslint:disable-next-line: max-line-length
            test.throws(() => subnet.availabilityZone, "You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
            test.done();
        },
        'AZ have value when subnet created from attributes with az'(test) {
            // GIVEN
            const stack = getTestStack();
            // WHEN
            const subnet = lib_1.Subnet.fromSubnetAttributes(stack, 'subnet1', { subnetId: 'pub-1', availabilityZone: 'az-1234' });
            // THEN
            test.deepEqual(subnet.subnetId, 'pub-1');
            test.deepEqual(subnet.availabilityZone, 'az-1234');
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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