"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
// tslint:disable-next-line:max-line-length
const lib_1 = require("../lib");
module.exports = {
    'gateway endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3
                    }
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.s3'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B'
                    },
                ],
                VpcEndpointType: 'Gateway'
            }));
            test.done();
        },
        'routing on private and public subnets'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            new lib_1.Vpc(stack, 'VpcNetwork', {
                gatewayEndpoints: {
                    S3: {
                        service: lib_1.GatewayVpcEndpointAwsService.S3,
                        subnets: [
                            {
                                subnetType: lib_1.SubnetType.PUBLIC
                            },
                            {
                                subnetType: lib_1.SubnetType.PRIVATE
                            }
                        ]
                    }
                }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.s3'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                RouteTableIds: [
                    {
                        Ref: 'VpcNetworkPublicSubnet1RouteTable25CCC53F'
                    },
                    {
                        Ref: 'VpcNetworkPublicSubnet2RouteTableE5F348DF'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet1RouteTableCD085FF1'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2RouteTableE97B328B'
                    },
                ],
                VpcEndpointType: 'Gateway'
            }));
            test.done();
        },
        'add statements to policy'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3
            });
            // WHEN
            endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                principals: [new aws_iam_1.AnyPrincipal()],
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*']
            }));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                's3:GetObject',
                                's3:ListBucket'
                            ],
                            Effect: 'Allow',
                            Principal: '*',
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            test.done();
        },
        'throws when adding a statement without a principal'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            const endpoint = vpc.addGatewayEndpoint('S3', {
                service: lib_1.GatewayVpcEndpointAwsService.S3
            });
            // THEN
            test.throws(() => endpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetObject', 's3:ListBucket'],
                resources: ['*']
            })), /`Principal`/);
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const ep = lib_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack2, 'ImportedEndpoint', 'endpoint-id');
            // THEN
            test.deepEqual(ep.vpcEndpointId, 'endpoint-id');
            test.done();
        },
        'works with an imported vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                privateSubnetRouteTableIds: ['rt1', 'rt2', 'rt3'],
                availabilityZones: ['a', 'b', 'c']
            });
            // THEN
            vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 });
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: { 'Fn::Join': ['', ['com.amazonaws.', { Ref: 'AWS::Region' }, '.s3']] },
                VpcId: 'id',
                RouteTableIds: ['rt1', 'rt2', 'rt3'],
                VpcEndpointType: 'Gateway',
            }));
            test.done();
        },
        'throws with an imported vpc without route tables ids'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromVpcAttributes(stack, 'VPC', {
                vpcId: 'id',
                privateSubnetIds: ['1', '2', '3'],
                availabilityZones: ['a', 'b', 'c']
            });
            test.throws(() => vpc.addGatewayEndpoint('Gateway', { service: lib_1.GatewayVpcEndpointAwsService.S3 }), /route table/);
            test.done();
        }
    },
    'interface endpoint': {
        'add an endpoint to a vpc'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: {
                    'Fn::Join': [
                        '',
                        [
                            'com.amazonaws.',
                            {
                                Ref: 'AWS::Region'
                            },
                            '.ecr.dkr'
                        ]
                    ]
                },
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                },
                PrivateDnsEnabled: true,
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'VpcNetworkEcrDockerSecurityGroup7C91D347',
                            'GroupId'
                        ]
                    }
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcNetworkPrivateSubnet1Subnet07BA143B'
                    },
                    {
                        Ref: 'VpcNetworkPrivateSubnet2Subnet5E4189D6'
                    },
                ],
                VpcEndpointType: 'Interface'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
                GroupDescription: 'VpcNetwork/EcrDocker/SecurityGroup',
                VpcId: {
                    Ref: 'VpcNetworkB258E83A'
                }
            }));
            test.done();
        },
        'import/export'(test) {
            // GIVEN
            const stack2 = new core_1.Stack();
            // WHEN
            const importedEndpoint = lib_1.InterfaceVpcEndpoint.fromInterfaceVpcEndpointAttributes(stack2, 'ImportedEndpoint', {
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack2, 'SG', 'security-group-id')],
                vpcEndpointId: 'vpc-endpoint-id',
                port: 80
            });
            importedEndpoint.connections.allowDefaultPortFromAnyIpv4();
            // THEN
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                GroupId: 'security-group-id'
            }));
            test.deepEqual(importedEndpoint.vpcEndpointId, 'vpc-endpoint-id');
            test.done();
        },
        'with existing security groups'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('EcrDocker', {
                service: lib_1.InterfaceVpcEndpointAwsService.ECR_DOCKER,
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'existing-id')]
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                SecurityGroupIds: ['existing-id'],
            }));
            test.done();
        },
        'with existing security groups for efs'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('Efs', {
                service: lib_1.InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM,
                securityGroups: [lib_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'existing-id')]
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                SecurityGroupIds: ['existing-id'],
            }));
            test.done();
        },
        'security group has ingress by default'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('SecretsManagerEndpoint', {
                service: lib_1.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupIngress: [
                    {
                        CidrIp: { 'Fn::GetAtt': ['VpcNetworkB258E83A', 'CidrBlock'] },
                        FromPort: 443,
                        IpProtocol: 'tcp',
                        ToPort: 443
                    }
                ]
            }));
            test.done();
        },
        'non-AWS service interface endpoint'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: new lib_1.InterfaceVpcEndpointService('com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc', 443)
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc',
                PrivateDnsEnabled: false
            }));
            test.done();
        },
        'marketplace partner service interface endpoint'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const vpc = new lib_1.Vpc(stack, 'VpcNetwork');
            // WHEN
            vpc.addInterfaceEndpoint('YourService', {
                service: { name: 'com.amazonaws.vpce.us-east-1.vpce-svc-mktplacesvcwprdns',
                    port: 443,
                    privateDnsDefault: true }
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPCEndpoint', {
                ServiceName: 'com.amazonaws.vpce.us-east-1.vpce-svc-mktplacesvcwprdns',
                PrivateDnsEnabled: true
            }));
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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