"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
const vpc_1 = require("./vpc");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets || [{ subnetType: vpc_1.SubnetType.PRIVATE }];
        const routeTableIds = util_1.allRouteTableIds(...subnets.map(s => props.vpc.selectSubnets(s)));
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        const subnets = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        const subnetIds = subnets.subnetIds;
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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