"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    get uniqueId() {
        return this.node.uniqueId;
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description
            });
        }
    }
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *   mutable: false
 * });
 * ```
 */
class SecurityGroup extends SecurityGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.anyValue({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.anyValue({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Import an existing security group into this app.
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description
        });
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.allowAllOutbound) {
            this.directEgressRules.push(ALLOW_ALL_RULE);
        }
        else {
            this.directEgressRules.push(MATCH_NO_TRAFFIC);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
        if (i > -1) {
            this.directEgressRules.splice(i, 1);
        }
    }
}
exports.SecurityGroup = SecurityGroup;
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86
};
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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