# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sql_athame']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions']

extras_require = \
{'asyncpg': ['asyncpg']}

setup_kwargs = {
    'name': 'sql-athame',
    'version': '0.3.16',
    'description': 'Python tool for slicing and dicing SQL',
    'long_description': '# sql-athame\n\nPython tool for slicing and dicing SQL.  Its intended target is\nPostgres with _asyncpg_, though it also includes support for rendering\nto a SQLAlchemy `TextClause`.\n\n## Base query builder\n\n### Example\n\n```python\nfrom sql_athame import sql\n\n\ndef get_orders(query):\n    where = []\n\n    if "id" in query:\n        where.append(sql("id = {}", query["id"]))\n    if "eventId" in query:\n        where.append(sql("event_id = {}", query["eventId"]))\n    if "startTime" in query:\n        where.append(sql("start_time = {}", query["startTime"]))\n    if "from" in query:\n        where.append(sql("start_time >= {}", query["from"]))\n    if "until" in query:\n        where.append(sql("start_time < {}", query["until"]))\n\n    return sql("SELECT * FROM orders WHERE {}", sql.all(where))\n\n\n>>> list(get_orders({}))\n[\'SELECT * FROM orders WHERE TRUE\']\n\n>>> list(get_orders({"id": "xyzzy"}))\n[\'SELECT * FROM orders WHERE (id = $1)\', \'xyzzy\']\n\n>>> list(get_orders({"eventId": "plugh", "from": "2019-05-01", "until": "2019-08-26"}))\n[\'SELECT * FROM orders WHERE (event_id = $1) AND (start_time >= $2) AND (start_time < $3)\',\n \'plugh\',\n \'2019-05-01\',\n \'2019-08-26\']\n\n\nsuperquery = sql(\n    """\n    SELECT *\n      FROM ({subquery}) sq\n      JOIN other_table ot ON (ot.id = sq.id)\n      WHERE ot.foo = {foo}\n      LIMIT {limit}\n    """,\n    subquery=get_orders({"id": "xyzzy"}),\n    foo="bork",\n    limit=50,\n)\n\n\n>>> list(superquery)\n[\'SELECT * FROM (SELECT * FROM orders WHERE (id = $1)) sq JOIN other_table ot ON (ot.id = sq.id) WHERE ot.foo = $2 LIMIT $3\',\n \'xyzzy\',\n \'bork\',\n 50]\n```\n\n### API reference\n\n```python\nfrom sql_athame import sql\n```\n\n#### sql(fmt: str, \\*args, \\*\\*kwargs) -> Fragment\n\nCreates a SQL `Fragment` from the `fmt` string.  The `fmt` string\ncontains literal SQL and may contain positional references, marked by\n`{}`, and named references, marked by `{name}`.  Positional references\n_must_ have a matching argument in `*args`.  Named references _may_\nhave a matching argument in `**kwargs`; if a named reference is not\nfullfilled by `**kwargs` it remains as a named _slot_ to be filled\nlater.\n\nIf a referenced argument is a `Fragment`, it is substituted into the\nSQL along with all of its embedded placeholders if any.  Otherwise, it\nis treated as a placeholder value and substituted in place as a\nplaceholder.\n\n#### Fragment.query(self) -> Tuple[str, List[Any]]\n\nRenders a SQL `Fragment` into a query string and list of placeholder\nparameters.\n\n```python\n>>> q = sql("SELECT * FROM tbl WHERE qty > {qty}", qty=10)\n>>> q.query()\n(\'SELECT * FROM tbl WHERE qty > $1\', [10])\n```\n\nIf there are any unfilled _slots_ `ValueError` will be raised.\n\n```python\n>>> q = sql("SELECT * FROM tbl WHERE qty > {qty}")\n>>> q.query()\nValueError: Unfilled slot: \'qty\'\n>>> q.fill(qty=10).query()\n(\'SELECT * FROM tbl WHERE qty > $1\', [10])\n```\n\n#### Fragment.\\_\\_iter\\_\\_(self) -> Iterator[Any]\n\nA `Fragment` is an iterable which will return the query string\nfollowed by the placeholder parameters as returned by\n`Fragment.query(self)`.  This matches the `(query, *args)` argument\npattern of the _asyncpg_ API:\n\n```python\nq = sql("SELECT * FROM tbl WHERE qty > {}", 10)\nawait conn.fetch(*q)\n```\n\n#### sql.list(parts: Iterable[Fragment]) -> Fragment\n#### sql.list(*parts: Fragment) -> Fragment\n\nCreates a SQL `Fragment` joining the fragments in `parts` together\nwith commas.\n\n```python\n>>> cols = [sql("a"), sql("b"), sql("c")]\n>>> list(sql("SELECT {cols} FROM tbl", cols=sql.list(cols)))\n[\'SELECT a, b, c FROM tbl\']\n```\n\n#### sql.all(parts: Iterable[Fragment]) -> Fragment\n#### sql.all(*parts: Fragment) -> Fragment\n\nCreates a SQL `Fragment` joining the fragments in `parts` together\nwith `AND`.  If `parts` is empty, returns `TRUE`.\n\n```python\n>>> where = [sql("a = {}", 42), sql("x <> {}", "foo")]\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.all(where)))\n[\'SELECT * FROM tbl WHERE (a = $1) AND (x <> $2)\', 42, \'foo\']\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.all([])))\n[\'SELECT * FROM tbl WHERE TRUE\']\n```\n\n#### sql.any(parts: Iterable[Fragment]) -> Fragment\n#### sql.any(*parts: Fragment) -> Fragment\n\nCreates a SQL `Fragment` joining the fragments in `parts` together\nwith `OR`.  If `parts` is empty, returns `FALSE`.\n\n```python\n>>> where = [sql("a = {}", 42), sql("x <> {}", "foo")]\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.any(where)))\n[\'SELECT * FROM tbl WHERE (a = $1) OR (x <> $2)\', 42, \'foo\']\n>>> list(sql("SELECT * FROM tbl WHERE {}", sql.any([])))\n[\'SELECT * FROM tbl WHERE FALSE\']\n```\n\n#### Fragment.join(self, parts: Iterable[Fragment]) -> Fragment\n\nCreates a SQL `Fragment` by joining the fragments in `parts` together\nwith `self`.\n\n```python\n>>> clauses = [sql("WHEN {} THEN {}", a, b) for a, b in ((sql("a"), 1), (sql("b"), 2))]\n>>> case = sql("CASE {clauses} END", clauses=sql(" ").join(clauses))\n>>> list(case)\n[\'CASE WHEN a THEN $1 WHEN b THEN $2 END\', 1, 2]\n```\n\n#### sql.literal(text: str) -> Fragment\n\nCreates a SQL `Fragment` with the literal SQL `text`.  No substitution\nof any kind is performed.  **Be very careful of SQL injection.**\n\n#### sql.identifier(name: str, prefix: Optional[str] = None) -> Fragment\n\nCreates a SQL `Fragment` with a quoted identifier name, optionally\nwith a dotted prefix.\n\n```python\n>>> list(sql("SELECT {a} FROM tbl", a=sql.identifier("a", prefix="tbl")))\n[\'SELECT "tbl"."a" FROM tbl\']\n```\n\n#### sql.value(value: Any) -> Fragment\n\nCreates a SQL `Fragment` with a single placeholder to `value`.\nEquivalent to:\n\n```python\nsql("{}", value)\n```\n\n#### sql.escape(value: Any) -> Fragment\n\nCreates a SQL `Fragment` with `value` escaped and embedded into the\nSQL.  Types currently supported are strings, floats, ints, UUIDs,\n`None`, and sequences of the above.\n\n```python\n>>> list(sql("SELECT * FROM tbl WHERE qty = ANY({})", sql.escape([1, 3, 5])))\n[\'SELECT * FROM tbl WHERE qty = ANY(ARRAY[1, 3, 5])\']\n```\n\nCompare to with a placeholder:\n\n```python\n>>> list(sql("SELECT * FROM tbl WHERE qty = ANY({})", [1, 3, 5]))\n[\'SELECT * FROM tbl WHERE qty = ANY($1)\', [1, 3, 5]]\n```\n\n"Burning" an invariant value into the query can potentially help the\nquery optimizer.\n\n#### sql.slot(name: str) -> Fragment\n\nCreates a SQL `Fragment` with a single empty _slot_ named `name`.\nEquivalent to:\n\n```python\nsql("{name}")\n```\n\n#### sql.unnest(data: Iterable[Sequence[Any]], types: Iterable[str]) -> Fragment\n\nCreates a SQL `Fragment` containing an `UNNEST` expression with\nassociated data.\n\nThe data is specified in tuples (in the "several database columns"\nsense, not necessarily the Python sense) in `data`, and the tuple\nPostgres types must be specified in `types`.  The data is transposed\ninto the correct form for `UNNEST` and embedded in placeholders in the\n`Fragment`.\n\n```python\n>>> list(sql("SELECT * FROM {}", sql.unnest([("a", 1), ("b", 2), ("c", 3)], ["text", "integer"])))\n[\'SELECT * FROM UNNEST($1::text[], $2::integer[])\', (\'a\', \'b\', \'c\'), (1, 2, 3)]\n```\n\n#### Fragment.fill(self, \\*\\*kwargs) -> Fragment\n\nCreates a SQL `Fragment` by filling any empty _slots_ in `self` with\n`kwargs`.  Similar to `sql` subtitution, if a value is a `Fragment` it\nis substituted in-place, otherwise it is substituted as a placeholder.\n\n#### Fragment.compile(self) -> Callable[..., Fragment]\n\nCreates a function that when called with `**kwargs` will create a SQL\n`Fragment` equivalent to calling `self.fill(**kwargs)`.  This is\noptimized to do as much work as possible up front and can be\nconsiderably faster if repeated often.\n\n#### Fragment.prepare(self) -> Tuple[str, Callable[..., List[Any]]]\n\nRenders `self` into a SQL query string; returns that string and a\nfunction that when called with `**kwargs` containing the unfilled\nslots of `self` will return a list containing the placeholder values\nfor `self` as filled with `**kwargs`.\n\n```python\n>>> query, query_args = sql("UPDATE tbl SET foo={foo}, bar={bar} WHERE baz < {baz}", baz=10).prepare()\n>>> query\n\'UPDATE tbl SET foo=$1, bar=$2 WHERE baz < $3\'\n>>> query_args(foo=1, bar=2)\n[1, 2, 10]\n>>> query_args(bar=42, foo=3)\n[3, 42, 10]\n```\n\nAs the name implies this is intended to be used in prepared\nstatements:\n\n```python\nquery, query_args = sql("UPDATE tbl SET foo={foo}, bar={bar} WHERE baz < {baz}", baz=10).prepare()\nstmt = await conn.prepare(query)\nawait stmt.execute(*query_args(foo=1, bar=2))\nawait stmt.execute(*query_args(bar=42, foo=3))\n```\n\n#### Fragment.sqlalchemy_text(self) -> sqlalchemy.sql.expression.TextClause\n\nRenders `self` into a SQLAlchemy `TextClause`.  Placeholder values\nwill be bound with `bindparams`.  Unfilled slots will be included as\nunbound parameters with their keys equal to the slot names.  A\nplaceholder value may be a SQLAlchemy `BindParameter`, in which case\nits value and type are used for the parameter created by this (and the\nkey name is ignored); this allow control of the SQLAlchemy type used\nfor the parameter.\n\nRequires SQLAlchemy to be installed, otherwise raises `ImportError`.\n\n```python\n>>> query = sql("SELECT * FROM tbl WHERE column = {}", 42)\n>>> stmt = query.sqlalchemy_text()\n>>> stmt._bindparams\n{\'_arg_0_140685932797232\': BindParameter(\'_arg_0_140685932797232\', 42, type_=Integer())}\n>>> conn.execute(stmt).fetchall()\n\n>>> query = sql("SELECT * FROM tbl WHERE column = {}", bindparam("ignored", 42, type_=Float()))\n>>> stmt = query.sqlalchemy_text()\n>>> stmt._bindparams\n{\'_arg_0_140294062155280\': BindParameter(\'_arg_0_140294062155280\', 42, type_=Float())}\n>>> conn.execute(stmt).fetchall()\n\n>>> query = sql("SELECT * FROM tbl WHERE column = {val}")\n>>> stmt = query.sqlalchemy_text()\n>>> stmt._bindparams\n{\'val\': BindParameter(\'val\', None, type_=NullType())}\n>>> conn.execute(stmt, val=42).fetchall()\n```\n\n## Dataclass helpers\n\n### Example\n\n```python\nfrom dataclasses import dataclass, field\nfrom datetime import date\nfrom uuid import UUID, uuid4\nfrom typing import Optional\n\nfrom sql_athame import ModelBase, model_field_metadata as MD, sql\n\n\n@dataclass\nclass Person(ModelBase, table_name="people", primary_key="id"):\n    id: UUID\n    name: str\n    birthday: date\n    title: Optional[str] = None\n    extra: Optional[dict] = field(default=None, metadata=MD(type="JSONB"))\n\n\n>>> list(Person.create_table_sql())\n[\'CREATE TABLE IF NOT EXISTS "people" (\'\n \'"id" UUID NOT NULL, \'\n \'"name" TEXT NOT NULL, \'\n \'"birthday" DATE NOT NULL, \'\n \'"title" TEXT, \'\n \'"extra" JSONB, \'\n \'PRIMARY KEY ("id"))\']\n\n>>> list(Person.select_sql(where=sql("title = {}", "Director")))\n[\'SELECT "id", "name", "birthday", "title", "extra" FROM "people" WHERE title = $1\',\n \'Director\']\n\n>>> people = [Person(uuid4(), "Bob", date(1974, 4, 3)), Person(uuid4(), "Anne", date(1985, 5, 4), extra={"awesome": "yes"})]\n>>> people\n[Person(id=UUID(\'8ecfe514-8fa3-48e5-8edb-fa810f5ed3f0\'), name=\'Bob\', birthday=datetime.date(1974, 4, 3), title=None, extra=None),\n Person(id=UUID(\'8d9bfadc-0026-4f6d-ae1b-ed89d1077f66\'), name=\'Anne\', birthday=datetime.date(1985, 5, 4), title=None, extra={\'awesome\': \'yes\'})]\n\n>>> list(Person.insert_multiple_sql(people))\n[\'INSERT INTO "people" ("id", "name", "birthday", "title", "extra")\'\n \' SELECT * FROM UNNEST($1::UUID[], $2::TEXT[], $3::DATE[], $4::TEXT[], $5::TEXT[]::JSONB[])\',\n (UUID(\'8ecfe514-8fa3-48e5-8edb-fa810f5ed3f0\'),\n  UUID(\'8d9bfadc-0026-4f6d-ae1b-ed89d1077f66\')),\n (\'Bob\', \'Anne\'),\n (datetime.date(1974, 4, 3), datetime.date(1985, 5, 4)),\n (None, None),\n [None, \'{"awesome": "yes"}\']]\n\n>>> list(Person.upsert_sql(Person.insert_multiple_sql(people)))\n[\'INSERT INTO "people" ("id", "name", "birthday", "title", "extra")\'\n \' SELECT * FROM UNNEST($1::UUID[], $2::TEXT[], $3::DATE[], $4::TEXT[], $5::TEXT[]::JSONB[])\'\n \' ON CONFLICT ("id") DO UPDATE\'\n \' SET "name"=EXCLUDED."name", "birthday"=EXCLUDED."birthday", "title"=EXCLUDED."title", "extra"=EXCLUDED."extra"\',\n (UUID(\'8ecfe514-8fa3-48e5-8edb-fa810f5ed3f0\'),\n  UUID(\'8d9bfadc-0026-4f6d-ae1b-ed89d1077f66\')),\n (\'Bob\', \'Anne\'),\n (datetime.date(1974, 4, 3), datetime.date(1985, 5, 4)),\n (None, None),\n [None, \'{"awesome": "yes"}\']]\n```\n\n### API reference\n\nTODO, for now read the [source](sql_athame/dataclasses.py).\n\n## License\n\nMIT.\n\n---\nCopyright (c) 2019, 2020 Brian Downing\n',
    'author': 'Brian Downing',
    'author_email': 'bdowning@lavos.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/bdowning/sql-athame',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
