# AUTOGENERATED! DO NOT EDIT! File to edit: 02_optimization.ipynb (unless otherwise specified).

__all__ = ['logger', 'solve_optimization_tensorflow', 'solve_optimization_gurobi', 'solve_optimization']

# Cell

import numpy as np
import itertools as it
from warnings import warn
from collections.abc import Iterable
import logging

import mulearn.kernel as kernel

logger = logging.getLogger(__name__)

try:
    import gurobipy as gpy
    gurobi_ok = True
except ModuleNotFoundError:
    logger.warn('gurobi not available')
    gurobi_ok = False

try:
    import tensorflow as tf
    tensorflow_ok = True
    logging.getLogger('tensorflow').setLevel(logging.ERROR)
except ModuleNotFoundError:
    logger.warn('tensorflow not available')
    tensorflow_ok = False

try:
    import tqdm
    tqdm_ok = True
except ModuleNotFoundError:
    logger.warn('tqdm not available')
    tqdm_ok = False

# Cell

def solve_optimization_tensorflow(xs, mus,
                          c=1.0, k=kernel.GaussianKernel(),
                          init='fixed',
                          init_bound=0.1,
                          init_val=0.01,
                          n_iter=100,
                          optimizer=tf.optimizers.Adam(learning_rate=1e-4) \
                                    if tensorflow_ok else None,
                          tracker=tqdm.trange if tqdm_ok else range,
                          penalization=10):
    '''Builds and solves the constrained optimization problem on the basis
    of the fuzzy learning procedure using the TensorFlow API.

    - xs: iterable of objects
    - mus: iterable of membership values for the objects in xs
    - c: constant managing the trade-off in joint radius/error optimization
    - k: kernel function to be used
    - init: initialization method for weights (string)
    - init_bound: absolute value of the extremes of the interval used
      for random initialization of weights (float)
    - init_val: value used for initializing weights to a constant (float)
    - n_iter: number of iteration of the optimization process (int)
    - optimizer: optiimzation algorithm to be used
    - tracker: tool to graphically depict the optimization progress

    Returns: a lists containing the optimal values for the independent
             variables chis of the problem

    Throws:

    - ValueError if optimization fails or if tensorflow is not installed
    '''

    if not tensorflow_ok:
        raise ValueError('tensorflow not available')

    m = len(xs)

    if type(init) == str and init == 'fixed':
        chis = [tf.Variable(init_val, name=f'chi_{i}',
                            trainable=True, dtype=tf.float32)
                for i in range(m)]
    elif type(init) == str and init == 'random':
        chis = [tf.Variable(ch, name=f'chi_{i}',
                            trainable=True, dtype=tf.float32)
                for i, ch in  enumerate(np.random.uniform(-init_bound,
                                                          init_bound, m))]

    elif isinstance(init, Iterable):
        chis = [tf.Variable(ch, name=f'chi_{i}',
                            trainable=True, dtype=tf.float32)
                for i, ch in  enumerate(init)]
    else:
        raise ValueError("init should either be set to 'fixed', "
                         "'random', or to a list of initial values.")

    if type(k) is kernel.PrecomputedKernel:
        gram = k.kernel_computations
    else:
        gram = np.array([[k.compute(x1, x2) for x1 in xs] for x2 in xs])

    def obj():
        kernels = tf.constant(gram, dtype='float32')

        v = tf.tensordot(tf.linalg.matvec(kernels, chis), chis, axes=1)
        v -= tf.tensordot(chis, [k.compute(x_i, x_i) for x_i in xs], axes=1)

        v += penalization * tf.math.maximum(0, 1 - sum(chis))
        v += penalization * tf.math.maximum(0, sum(chis) - 1)

        if c < np.inf:
            for ch, m in zip(chis, mus):
                v += penalization * tf.math.maximum(0, ch - c*m)
                v += penalization * tf.math.maximum(0, c*(1-m) - ch)

        return v

    for i in tracker(n_iter):
        #old_chis = np.array([ch.numpy() for ch in chis])
        optimizer.minimize(obj, var_list=chis)
        #new_chis = np.array([ch.numpy() for ch in chis])

    return [ch.numpy() for ch in chis]

# Cell

def solve_optimization_gurobi(xs,
                              mus,
                              c=1.0,
                              k=kernel.GaussianKernel(),
                              time_limit=10*60,
                              adjustment=0):
    '''Builds and solves the constrained optimization problem on the basis
    of the fuzzy learning procedure using the gurobi API.

    - `xs`: objects in training set (iterable).

    - `mus`: membership values for the objects in xs (iterable).

    - `c`: constant managing the trade-off in joint radius/error optimization
      (float).

    - `k`: kernel function to be used (kernel.Kernel).

    - `time_limit`: time in seconds before stopping the optimization process
      (int).

    - `adjustment`: diagonal adjustment in order to deal with non PSD
      matrices (float).

    Return a lists containing the optimal values for the independent
    variables chis of the problem.

    Throws:

    - ValueError if optimization fails or if gurobi is not installed
    '''

    if not gurobi_ok:
        raise ValueError('gurobi not available')

    m = len(xs)

    model = gpy.Model('mulearn')
    model.setParam('OutputFlag', 0)
    model.setParam('TimeLimit', time_limit)

    for i in range(m):
        if c < np.inf:
            model.addVar(name=f'chi_{i}', lb=-c*(1-mus[i]), ub=c*mus[i],
                         vtype=gpy.GRB.CONTINUOUS)

        else:
            model.addVar(name=f'chi_{i}', vtype=gpy.GRB.CONTINUOUS)

    model.update()

    chis = model.getVars()

    obj = gpy.QuadExpr()

    for i, j in it.product(range(m), range(m)):
        obj.add(chis[i] * chis[j], k.compute(xs[i], xs[j]))

    for i in range(m):
        obj.add(-1 * chis[i] * k.compute(xs[i], xs[i]))

    if adjustment:
        for i in range(m):
            obj.add(adjustment * chis[i] * chis[i])

    model.setObjective(obj, gpy.GRB.MINIMIZE)

    constEqual = gpy.LinExpr()
    constEqual.add(sum(chis), 1.0)

    model.addConstr(constEqual, gpy.GRB.EQUAL, 1)

    model.optimize()


    if model.Status != gpy.GRB.OPTIMAL:
        raise ValueError('optimal solution not found!')

    return [ch.x for ch in chis]

# Cell

def solve_optimization(xs, mus, c=1.0, k=kernel.GaussianKernel(),
                       solve_strategy=solve_optimization_tensorflow,
                       **solve_strategy_args):
    '''Builds and solves the constrained optimization problem on the basis
    of the fuzzy learning procedure.

    - xs: objects in training set (iterable).

    - mus: membership values for the objects in xs (iterable).

    - c: constant managing the trade-off in joint radius/error optimization
      (float).

    - k: kernel function to be used (kernel.Kernel).

    - solve_strategy: algorithm to be used in order to numerically solve the
      optimization problem.

    - solve_strategy_args: optional parameters for the optimization
      algorithm.

    Return a lists containing the optimal values for the independent
    variables chis of the problem

    Throws:

    - ValueError if c is non-positive or if xs and mus have different lengths
    '''
    if c <= 0:
        raise ValueError('c should be positive')


    mus = np.array(mus)


    chis = solve_strategy(xs, mus, c, k, **solve_strategy_args)

    chis_opt = [np.clip(ch, l, u) for ch,l,u in zip(chis, -c*(1-mus), c*mus)]

    return chis_opt