"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitlabRunnerAutoscaling = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const asg = require("@aws-cdk/aws-autoscaling");
const aws_autoscaling_hooktargets_1 = require("@aws-cdk/aws-autoscaling-hooktargets");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const assets = require("@aws-cdk/aws-s3-assets");
const sns = require("@aws-cdk/aws-sns");
const subscriptions = require("@aws-cdk/aws-sns-subscriptions");
const cdk = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
/**
 * (experimental) GitlabRunnerAutoscaling Construct for create Autoscaling Gitlab Runner.
 *
 * @experimental
 */
class GitlabRunnerAutoscaling extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultProps = {
            instanceType: 't3.micro',
            tags: ['gitlab', 'awscdk', 'runner'],
            gitlabUrl: 'https://gitlab.com/',
            gitlabRunnerImage: 'public.ecr.aws/gitlab/gitlab-runner:alpine',
            alarms: [
                {
                    AlarmName: 'GitlabRunnerDiskUsage',
                    MetricName: 'disk_used_percent',
                },
            ],
        };
        const runnerProps = { ...defaultProps, ...props };
        const asset = new assets.Asset(this, 'GitlabRunnerUserDataAsset', {
            path: path.join(__dirname, '../assets/userdata/amazon-cloudwatch-agent.json'),
        });
        const userData = ec2.UserData.forLinux();
        userData.addS3DownloadCommand({
            bucket: asset.bucket,
            bucketKey: asset.s3ObjectKey,
            localFile: '/opt/aws/amazon-cloudwatch-agent/etc/amazon-cloudwatch-agent.json',
        });
        userData.addCommands(...this.createUserData(runnerProps));
        this.instanceRole = (_b = runnerProps.instanceRole) !== null && _b !== void 0 ? _b : new iam.Role(this, 'GitlabRunnerInstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            description: 'For EC2 Instance (Gitlab Runner) Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchAgentServerPolicy'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
            ],
        });
        this.vpc = (_c = runnerProps.vpc) !== null && _c !== void 0 ? _c : new ec2.Vpc(this, 'VPC');
        this.securityGroup = new ec2.SecurityGroup(this, 'GitlabRunnerSecurityGroup', {
            vpc: this.vpc,
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.instanceRole.roleName],
        });
        const lt = new ec2.CfnLaunchTemplate(this, 'GitlabRunnerLaunchTemplate', {
            launchTemplateData: {
                imageId: ec2.MachineImage.latestAmazonLinux({
                    generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                }).getImage(this).imageId,
                instanceType: runnerProps.instanceType,
                instanceMarketOptions: {
                    marketType: runnerProps.spotInstance ? 'spot' : undefined,
                    spotOptions: runnerProps.spotInstance ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                blockDeviceMappings: [
                    {
                        deviceName: '/dev/xvda',
                        ebs: {
                            volumeSize: (_d = runnerProps.ebsSize) !== null && _d !== void 0 ? _d : 60,
                        },
                    },
                ],
                iamInstanceProfile: {
                    arn: instanceProfile.attrArn,
                },
                securityGroupIds: this.securityGroup.connections.securityGroups.map((m) => m.securityGroupId),
            },
        });
        this.autoscalingGroup = new asg.AutoScalingGroup(this, 'GitlabRunnerAutoscalingGroup', {
            instanceType: new ec2.InstanceType(runnerProps.instanceType),
            autoScalingGroupName: `Gitlab Runners (${runnerProps.instanceType})`,
            vpc: this.vpc,
            vpcSubnets: runnerProps.vpcSubnet,
            machineImage: ec2.MachineImage.latestAmazonLinux({
                generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            minCapacity: runnerProps.minCapacity,
            maxCapacity: runnerProps.maxCapacity,
            desiredCapacity: runnerProps.desiredCapacity,
        });
        const cfnAsg = this.autoscalingGroup.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        this.autoscalingGroup.node.tryRemoveChild('LaunchConfig');
        this.topicAlarm = new sns.Topic(this, 'GitlabRunnerAlarm');
        const alarms = JSON.stringify(runnerProps.alarms);
        // Put alarms at launch
        const registerFunction = new lambda.Function(this, 'GitlabRunnerRegisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.register',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
                SNS_TOPIC_ARN: this.topicAlarm.topicArn,
            },
        });
        (_e = registerFunction.role) === null || _e === void 0 ? void 0 : _e.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'cloudwatch:PutMetricAlarm',
            ],
        }));
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookLaunching', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_LAUNCHING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(registerFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        // Add an alarm action to terminate invalid instances
        const alarmAction = new lambda.Function(this, 'GitlabRunnerAlarmAction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_alarm',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            logRetention: logs.RetentionDays.ONE_DAY,
        });
        (_f = alarmAction.role) === null || _f === void 0 ? void 0 : _f.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'autoscaling:SetInstanceHealth',
            ],
        }));
        const alarmSubscription = new subscriptions.LambdaSubscription(alarmAction);
        this.topicAlarm.addSubscription(alarmSubscription);
        // Unregister gitlab runners and remove alarms on instance termination or CFn stack deletion
        const unregisterRole = new iam.Role(this, 'GitlabRunnerUnregisterRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            description: 'For Gitlab Runner Unregistering Function Role',
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        unregisterRole.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'ssm:SendCommand',
                'autoscaling:DescribeAutoScalingGroups',
                'cloudwatch:DeleteAlarms',
            ],
        }));
        const unregisterFunction = new lambda.Function(this, 'GitlabRunnerUnregisterFunction', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.unregister',
            runtime: lambda.Runtime.PYTHON_3_8,
            timeout: cdk.Duration.seconds(60),
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        this.autoscalingGroup.addLifecycleHook('GitlabRunnerLifeCycleHookTerminating', {
            lifecycleTransition: asg.LifecycleTransition.INSTANCE_TERMINATING,
            notificationTarget: new aws_autoscaling_hooktargets_1.FunctionHook(unregisterFunction),
            defaultResult: asg.DefaultResult.CONTINUE,
            heartbeatTimeout: cdk.Duration.seconds(60),
        });
        const unregisterCustomResource = new lambda.Function(this, 'GitlabRunnerUnregisterCustomResource', {
            code: lambda.Code.fromAsset(path.join(__dirname, '../assets/functions')),
            handler: 'autoscaling_events.on_event',
            runtime: lambda.Runtime.PYTHON_3_8,
            role: unregisterRole,
            logRetention: logs.RetentionDays.ONE_DAY,
            environment: {
                ALARMS: alarms,
            },
        });
        const unregisterProvider = new cr.Provider(this, 'GitlabRunnerUnregisterProvider', {
            onEventHandler: unregisterCustomResource,
        });
        const customResource = new cdk.CustomResource(this, 'GitlabRunnerCustomResource', {
            serviceToken: unregisterProvider.serviceToken,
            properties: {
                AutoScalingGroupNames: [this.autoscalingGroup.autoScalingGroupName],
            },
        });
        customResource.node.addDependency(unregisterProvider);
        new cdk.CfnOutput(this, 'GitlabRunnerAutoScalingGroupArn', {
            value: this.autoscalingGroup.autoScalingGroupArn,
        });
    }
    dockerVolumesList(dockerVolume) {
        let tempString = '--docker-volumes "/var/run/docker.sock:/var/run/docker.sock"';
        if (dockerVolume) {
            let tempList = [];
            dockerVolume.forEach(e => {
                tempList.push(`"${e.hostPath}:${e.containerPath}"`);
            });
            tempList.forEach(e => {
                tempString = `${tempString} --docker-volumes ${e}`;
            });
        }
        return tempString;
    }
    /**
     * @returns Array.
     * @experimental
     */
    createUserData(props) {
        var _b;
        return [
            'yum update -y',
            'sleep 15 && amazon-linux-extras install docker && yum install -y amazon-cloudwatch-agent && systemctl start docker && usermod -aG docker ec2-user && chmod 777 /var/run/docker.sock',
            'systemctl restart docker && systemctl enable docker && systemctl start amazon-cloudwatch-agent && systemctl enable amazon-cloudwatch-agent',
            `docker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock \
      --name gitlab-runner-register ${props.gitlabRunnerImage} register --non-interactive --url ${props.gitlabUrl} --registration-token ${props.gitlabToken} \
      --docker-pull-policy if-not-present ${this.dockerVolumesList(props === null || props === void 0 ? void 0 : props.dockerVolumes)} \
      --executor docker --docker-image "alpine:latest" --description "A Runner on EC2 Instance (${props.instanceType})" \
      --tag-list "${(_b = props.tags) === null || _b === void 0 ? void 0 : _b.join(',')}" --docker-privileged`,
            `sleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner ${props.gitlabRunnerImage}`,
        ];
    }
}
exports.GitlabRunnerAutoscaling = GitlabRunnerAutoscaling;
_a = JSII_RTTI_SYMBOL_1;
GitlabRunnerAutoscaling[_a] = { fqn: "cdk-gitlab-runner.GitlabRunnerAutoscaling", version: "1.115.45" };
//# sourceMappingURL=data:application/json;base64,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