"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("aws-cdk-lib");
const events = require("aws-cdk-lib/aws-events");
const kinesis = require("aws-cdk-lib/aws-kinesis");
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Test snapshot match with default parameters
// --------------------------------------------------------------
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            description: 'event rule props',
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventbridgeToKinesisStreams(stack, 'test-eventbridge-kinesis-streams-default-parameters', props);
}
// --------------------------------------------------------------
// Test properties
// --------------------------------------------------------------
test('Test properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    // Assertions
    expect(construct.eventsRule !== null);
    expect(construct.kinesisStream !== null);
    expect(construct.eventsRole !== null);
});
// --------------------------------------------------------------
// Test default AWS managed encryption key property
// --------------------------------------------------------------
test('Test default AWS managed encryption key property', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    // Assertions
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: "KMS",
            KeyId: "alias/aws/kinesis"
        }
    });
});
// --------------------------------------------------------------
// Test existing resources
// --------------------------------------------------------------
test('Test existing resources', () => {
    const stack = new cdk.Stack();
    // create resource
    const existingStream = new kinesis.Stream(stack, 'test-existing-stream', {
        streamName: 'existing-stream',
        shardCount: 5,
        retentionPeriod: cdk.Duration.hours(48),
        encryption: kinesis.StreamEncryption.UNENCRYPTED
    });
    new lib_1.EventbridgeToKinesisStreams(stack, 'test-eventbridge-kinesis-stream-existing-resource', {
        existingStreamObj: existingStream,
        eventRuleProps: {
            description: 'event rule props',
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        Name: 'existing-stream',
        ShardCount: 5,
        RetentionPeriodHours: 48,
    });
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' }
    };
    const construct = new lib_1.EventbridgeToKinesisStreams(stack, 'test-eventbridge-kinesis-streams-default-parameters', props);
    expect(construct.eventsRule !== null);
    expect(construct.kinesisStream !== null);
    expect(construct.eventsRole !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    expect(stack).toHaveResource('AWS::Events::EventBus');
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' },
        existingEventBusInterface: new events.EventBus(stack, `test-existing-eventbus`, { eventBusName: 'test' })
    };
    const app = () => {
        new lib_1.EventbridgeToKinesisStreams(stack, 'test-eventbridge-kinesisstreams', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const props = {
        eventBusProps: {
            eventBusName: `testeventbus`
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new lib_1.EventbridgeToKinesisStreams(stack, 'test-new-eventbridge-with-props-kinsesisstreams', props);
    expect(stack).toHaveResource('AWS::Events::EventBus', {
        Name: `testeventbus`
    });
});
//# sourceMappingURL=data:application/json;base64,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