# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['seqvec']

package_data = \
{'': ['*']}

install_requires = \
['allennlp>=0.9.0,<0.10.0',
 'gevent==1.4.0',
 'h5py>=2.10.0,<3.0.0',
 'numpy>=1.17,<2.0',
 'torch>=1.2,<2.0',
 'tqdm>=4.36,<5.0']

entry_points = \
{'console_scripts': ['seqvec = seqvec.seqvec:main']}

setup_kwargs = {
    'name': 'seqvec',
    'version': '0.2.1',
    'description': "Embeeder tool for 'Modelling the Language of Life - Deep Learning Protein Sequences'",
    'long_description': '# SeqVec\n\nRepository for the paper [Modeling aspects of the language of life through transfer-learning protein sequences](https://bmcbioinformatics.biomedcentral.com/articles/10.1186/s12859-019-3220-8).\nHolds pre-trained SeqVec model for creating embeddings for amino acid sequences. Also, contains checkpoint for fine-tuning.\n\n# Abstract\n**Background**: One common task in Computational Biology is the prediction of aspects of protein function and structure from their amino acid sequence. For 26 years, most state-of-the-art approaches toward this end have been marrying machine learning and evolutionary information. The retrieval of related proteins from ever growing sequence databases is becoming so time-consuming that the analysis of entire proteomes becomes challenging. On top, evolutionary information is less powerful for small families, e.g. for proteins from the Dark Proteome.\n\n**Results**: We introduce a novel way to represent protein sequences as continuous vectors (embeddings) by using the deep bi-directional model ELMo taken from natural language processing (NLP). The model has effectively captured the biophysical properties of protein sequences from unlabeled big data (UniRef50). After training, this knowledge is transferred to single protein sequences by predicting relevant sequence features. We refer to these new embeddings as SeqVec (Sequence-to-Vector) and demonstrate their effectiveness by training simple convolutional neural networks on existing data sets for two completely different prediction tasks. At the per-residue level, we significantly improved secondary structure (for NetSurfP-2.0 data set: Q3=79%±1, Q8=68%±1) and disorder predictions (MCC=0.59±0.03) over methods not using evolutionary information. At the per-protein level, we predicted subcellular localization in ten classes (for DeepLoc data set: Q10=68%±1) and distinguished membrane- bound from water-soluble proteins (Q2= 87%±1). All results built upon the embeddings gained from the new tool SeqVec neither explicitly nor implicitly using evolutionary information. Nevertheless, it improved over some methods using such information. Where the lightning-fast HHblits needed on average about two minutes to generate the evolutionary information for a target protein, SeqVec created the vector representation on average in 0.03 seconds.\n\n**Conclusion**: We have shown that transfer learning can be used to capture biochemical or biophysical properties of protein sequences from large unlabeled sequence databases. The effectiveness of the proposed approach was showcased for different prediction tasks using only single protein sequences. SeqVec embeddings enable predictions that outperform even some methods using evolutionary information. Thus, they prove to condense the underlying principles of protein sequences. This might be the first step towards competitive predictions based only on single protein sequences.\n\n# t-SNE projections of SeqVec\n![2D t-SNE projections](seqvec_tsne.png "2D t-SNE projections of SeqVec")\n*2D t-SNE projections of unsupervised SeqVec embeddings highlight different realities of proteins and their constituent parts, amino acids.* Panels (b) to (d) are based on the same data set (Structural Classification of Proteins – extended (SCOPe) 2.07, redundancy reduced at 40%). For these plots, only subsets of SCOPe containing proteins with the annotation of interest (enzymatic activity (c) and kingdom (d)) may be displayed. **Panel (a)**: the embedding space confirms: the 20 standard amino acids are clustered according to their biochemical and biophysical properties, i.e. hydrophobicity, charge or size. The unique role of Cysteine (C, mostly hydrophobic and polar) is conserved. **Panel (b)**: SeqVec embeddings capture structural information as annotated in the main classes in SCOPe without ever having been explicitly trained on structural features. **Panel (c)**: many small, local clusters share function as given by the main classes in the Enzyme Commission Number (E.C.). **Panel (d)**: similarly, small, local clusters represent different kingdoms of life.\n\n# Model availability\nThe ELMo model trained on UniRef50 (=SeqVec) is available at:\n[SeqVec-model](https://rostlab.org/~deepppi/seqvec.zip)\n\nThe checkpoint for the pre-trained model is available at:\n[SeqVec-checkpoint](https://rostlab.org/~deepppi/seqvec_checkpoint.tar.gz)\n\n# Installation\n\n```\npip install seqvec\n```\n\nWe are working on a python package with more embedders and a commong interface to them; see [bio_embeddings](https://github.com/sacdallago/bio_embeddings)\n\n# Example\n\nIn the [bio_embeddings](https://github.com/sacdallago/bio_embeddings) github repo, you can find examples in the `notebooks` folder.\n\nFor a general example on how to extract embeddings using ELMo, please check the \nofficial allennlp ELMo website: [ELMo-Tutorial](https://github.com/allenai/allennlp/blob/master/tutorials/how_to/elmo.md)\n\nYou can compute embeddings for a fasta file with the `seqvec` command. Add `--protein True` to get an embedding per protein instead of per residue.\n\n```\nseqvec -i sequences.fasta -o embeddings.npz\n```\n\nLoad the embeddings with numpy:\n\n```python\nimport numpy as np\ndata = np.load("embeddings.npz")  # type: Dict[str, np.ndarray]\n```\n\nIf you specify `.npy` as output format (e.g. with `-o embeddings.npy`), the script will save the embeddings as an numpy array and the corresponding identifiers (as extracted from the header line in the fasta file) in a json file besides it. The sorting in the json file corresponds to the indexing in the npy file. The npy file can be loaded via:\n\n```python\nimport json\nimport numpy as np\n\ndata = np.load("embeddings.npy") # shape=(n_proteins,)\nwith open("embeddings.json") as fp:\n    labels = json.load(fp)\n```\n\n**How to integrate the embedder into an existing workflow:**\n\n\nLoad pre-trained model:\n\n```python\nfrom allennlp.commands.elmo import ElmoEmbedder\nfrom pathlib import Path\n\nmodel_dir = Path(\'path/to/pretrained/SeqVec_directory\')\nweights = model_dir / \'weights.hdf5\'\noptions = model_dir / \'options.json\'\nembedder = ElmoEmbedder(options,weights, cuda_device=0) # cuda_device=-1 for CPU\n```\n\nGet embedding for amino acid sequence:\n\n```python\nseq = \'SEQWENCE\' # your amino acid sequence\nembedding = embedder.embed_sentence(list(seq)) # List-of-Lists with shape [3,L,1024]\n```\n\nBatch embed sequences:\n\n```python\nseq1 = \'SEQWENCE\' # your amino acid sequence\nseq2 = \'PROTEIN\'\nseqs = [list(seq1), list(seq2)]\nseqs.sort(key=len) # sorting is crucial for speed\nembedding = embedder.embed_sentences(seqs) # returns: List-of-Lists with shape [3,L,1024]\n```\n\nGet 1024-dimensional embedding for per-residue predictions:\n\n```python\nimport torch\nresidue_embd = torch.tensor(embedding).sum(dim=0) # Tensor with shape [L,1024]\n```\n\nGet 1024-dimensional embedding for per-protein predictions:\n```python\nprotein_embd = torch.tensor(embedding).sum(dim=0).mean(dim=0) # Vector with shape [1024]\n```\n# FAQ\n**Torch version conflict**\nIf you encounter a version conflict while pip-installing seqvec (```ERROR: No matching distribution found for torch<1.3,>=1.2 (from seqvec)```), creating a new conda-environment with Python 3.7 can resolve your issue. \n\n\n\n# Web-service for Predictions based on SeqVec\n[SeqVec predictions - Chris\' Protein properties](https://embed.protein.properties/)\n\n# Bibtex-Reference\n```\n@article{heinzinger2019modeling,\n  title={Modeling aspects of the language of life through transfer-learning protein sequences},\n  author={Heinzinger, Michael and Elnaggar, Ahmed and Wang, Yu and Dallago, Christian and Nechaev, Dmitrii and Matthes, Florian and Rost, Burkhard},\n  journal={BMC bioinformatics},\n  volume={20},\n  number={1},\n  pages={723},\n  year={2019},\n  publisher={Springer}\n}\n```\n',
    'author': 'Michael Heinzinger',
    'author_email': 'mheinzinger@rostlab.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Rostlab/SeqVec',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.1',
}


setup(**setup_kwargs)
