from .......Internal.Core import Core
from .......Internal.CommandsGroup import CommandsGroup
from .......Internal import Conversions


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class MfOrderCls:
	"""MfOrder commands group definition. 2 total commands, 1 Subgroups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("mfOrder", core, parent)

	@property
	def ledState(self):
		"""ledState commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_ledState'):
			from .LedState import LedStateCls
			self._ledState = LedStateCls(self._core, self._cmd_group)
		return self._ledState

	def set(self, maximum_filter_order: float) -> None:
		"""SCPI: CONFigure:CFReduction:MFORder \n
		Snippet: driver.applications.k18AmplifierEt.configure.cfReduction.mfOrder.set(maximum_filter_order = 1.0) \n
		Sets and queries the maximum filter order for crest factor reduction. \n
			:param maximum_filter_order: numeric value
		"""
		param = Conversions.decimal_value_to_str(maximum_filter_order)
		self._core.io.write(f'CONFigure:CFReduction:MFORder {param}')

	def get(self) -> float:
		"""SCPI: CONFigure:CFReduction:MFORder \n
		Snippet: value: float = driver.applications.k18AmplifierEt.configure.cfReduction.mfOrder.get() \n
		Sets and queries the maximum filter order for crest factor reduction. \n
			:return: maximum_filter_order: numeric value"""
		response = self._core.io.query_str(f'CONFigure:CFReduction:MFORder?')
		return Conversions.str_to_float(response)

	def clone(self) -> 'MfOrderCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = MfOrderCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
