"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    /**
     * Original type for shorter term data processing. Data is not replicated and does not persist on server fail.
     */
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    /**
     * Newer type for shorter term data processing. Data is not replicated and does not persist on server fail.
     * Provides better support for spiky workloads.
     */
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    /**
     * Long term storage. Data is replicated and file servers are replaced if they fail.
     */
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 *
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_a = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _a === void 0 ? void 0 : _a.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
        }
        if (![50, 100, 200].includes(perUnitStorageThroughput)) {
            throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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