/**
 * This type provides a flag that can be used to turn off more lax overloads intended for
 * plugin use only to catch type errors in the TypeDoc codebase. The prepublishOnly npm
 * script will be used to switch this flag to false when publishing, then immediately back
 * to true after a successful publish.
 */
declare type InternalOnly = false;
/**
 * Helper type to convert `T` to `F` if strict mode is on.
 *
 * Can be used in overloads to map a parameter type to `never`. For example, the
 * following function will work with any string argument, but to improve the type safety
 * of internal code, we only ever want to pass 'a' or 'b' to it. Plugins on the other
 * hand need to be able to pass any string to it. Overloads similar to this are used
 * in the {@link Options} class.
 *
 * This is also used to prevent TypeDoc code from using deprecated methods which will
 * be removed in a future release.
 *
 * ```ts
 * function over(flag: 'a' | 'b'): string
 * function over(flag: IfStrict<string, never>): string
 * function over(flag: string): string { return flag }
 * ```
 */
export declare type IfInternal<T, F> = InternalOnly extends true ? T : F;
/**
 * Helper type to convert `T` to `never` if strict mode is on.
 *
 * See {@link IfInternal} for the rationale.
 */
export declare type NeverIfInternal<T> = IfInternal<never, T>;
export declare function hasBeenLoadedMultipleTimes(): boolean;
export {};
