import type { Application } from "../application";
import type { Theme } from "./theme";
import type { ProjectReflection } from "../models/reflections/project";
import { RendererComponent } from "./components";
import { ChildableComponent } from "../utils/component";
import { EventHooks } from "../utils";
import type { Theme as ShikiTheme } from "shiki";
import { ReferenceType } from "../models";
import type { JsxElement } from "../utils/jsx.elements";
import type { DefaultThemeRenderContext } from "./themes/default/DefaultThemeRenderContext";
/**
 * Describes the hooks available to inject output in the default theme.
 * If the available hooks don't let you put something where you'd like, please open an issue!
 */
export interface RendererHooks {
    /**
     * Applied immediately after the opening `<head>` tag.
     */
    "head.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the closing `</head>` tag.
     */
    "head.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately after the opening `<body>` tag.
     */
    "body.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the closing `</body>` tag.
     */
    "body.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before the main template.
     */
    "content.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately after the main template.
     */
    "content.end": [DefaultThemeRenderContext];
    /**
     * Applied immediately before calling `context.navigation`.
     */
    "navigation.begin": [DefaultThemeRenderContext];
    /**
     * Applied immediately after calling `context.navigation`.
     */
    "navigation.end": [DefaultThemeRenderContext];
}
/**
 * The renderer processes a {@link ProjectReflection} using a {@link Theme} instance and writes
 * the emitted html documents to a output directory. You can specify which theme should be used
 * using the `--theme <name>` command line argument.
 *
 * {@link Renderer} is a subclass of {@link EventDispatcher} and triggers a series of events while
 * a project is being processed. You can listen to these events to control the flow or manipulate
 * the output.
 *
 *  * {@link Renderer.EVENT_BEGIN}<br>
 *    Triggered before the renderer starts rendering a project. The listener receives
 *    an instance of {@link RendererEvent}. By calling {@link RendererEvent.preventDefault} the entire
 *    render process can be canceled.
 *
 *    * {@link Renderer.EVENT_BEGIN_PAGE}<br>
 *      Triggered before a document will be rendered. The listener receives an instance of
 *      {@link PageEvent}. By calling {@link PageEvent.preventDefault} the generation of the
 *      document can be canceled.
 *
 *    * {@link Renderer.EVENT_END_PAGE}<br>
 *      Triggered after a document has been rendered, just before it is written to disc. The
 *      listener receives an instance of {@link PageEvent}. When calling
 *      {@link PageEvent.preventDefault} the the document will not be saved to disc.
 *
 *  * {@link Renderer.EVENT_END}<br>
 *    Triggered after the renderer has written all documents. The listener receives
 *    an instance of {@link RendererEvent}.
 */
export declare class Renderer extends ChildableComponent<Application, RendererComponent> {
    private themes;
    private unknownSymbolResolvers;
    /** @event */
    static readonly EVENT_BEGIN_PAGE = "beginPage";
    /** @event */
    static readonly EVENT_END_PAGE = "endPage";
    /** @event */
    static readonly EVENT_BEGIN = "beginRender";
    /** @event */
    static readonly EVENT_END = "endRender";
    /**
     * The theme that is used to render the documentation.
     */
    theme?: Theme;
    /**
     * Hooks which will be called when rendering pages.
     * Note:
     * - Hooks added during output will be discarded at the end of rendering.
     * - Hooks added during a page render will be discarded at the end of that page's render.
     *
     * See {@link RendererHooks} for a description of each available hook, and when it will be called.
     */
    hooks: EventHooks<RendererHooks, JsxElement>;
    /** @internal */
    themeName: string;
    /** @internal */
    cleanOutputDir: boolean;
    /** @internal */
    cname: string;
    /** @internal */
    gaID: string;
    /** @internal */
    gaSite: string;
    /** @internal */
    githubPages: boolean;
    /** @internal */
    hideGenerator: boolean;
    /** @internal */
    lightTheme: ShikiTheme;
    /** @internal */
    darkTheme: ShikiTheme;
    /**
     * Define a new theme that can be used to render output.
     * This API will likely be changing in TypeDoc 0.23.
     * (sorry... changing as soon as it's introduced)
     * As it is, it provides reasonable flexibility, but doesn't give users a sufficiently
     * easy way to overwrite parts of a theme.
     * @param name
     * @param theme
     */
    defineTheme(name: string, theme: new (renderer: Renderer) => Theme): void;
    /**
     * Adds a new resolver that the theme can used to try to figure out how to link to a symbol
     * declared by a third-party library which is not included in the documentation.
     * @param packageName the npm package name that this resolver can handle to limit which files it will be tried on.
     *   If the resolver will create links for Node builtin types, it should be set to `@types/node`.
     *   Links for builtin types live in the default lib files under `typescript`.
     * @param resolver a function that will be called to create links for a given symbol name in the registered path.
     *  If the provided name is not contained within the docs, should return `undefined`.
     * @since 0.22.0
     */
    addUnknownSymbolResolver(packageName: string, resolver: (name: string) => string | undefined): void;
    /**
     * Marked as internal for now. Using this requires the internal `getSymbol()` method on ReferenceType.
     * Someday that needs to be fixed so that this can be made public. ReferenceTypes really shouldn't store
     * symbols so that we don't need to keep the program around forever.
     * @internal
     */
    attemptExternalResolution(type: ReferenceType): string | undefined;
    /**
     * Render the given project reflection to the specified output directory.
     *
     * @param project  The project that should be rendered.
     * @param outputDirectory  The path of the directory the documentation should be rendered to.
     */
    render(project: ProjectReflection, outputDirectory: string): Promise<void>;
    /**
     * Render a single page.
     *
     * @param page An event describing the current page.
     * @return TRUE if the page has been saved to disc, otherwise FALSE.
     */
    private renderDocument;
    /**
     * Ensure that a theme has been setup.
     *
     * If a the user has set a theme we try to find and load it. If no theme has
     * been specified we load the default theme.
     *
     * @returns TRUE if a theme has been setup, otherwise FALSE.
     */
    private prepareTheme;
    /**
     * Prepare the output directory. If the directory does not exist, it will be
     * created. If the directory exists, it will be emptied.
     *
     * @param directory  The path to the directory that should be prepared.
     * @returns TRUE if the directory could be prepared, otherwise FALSE.
     */
    private prepareOutputDirectory;
}
import "./plugins";
