"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Template = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("@aws-cdk/core");
const fs = require("fs-extra");
const match_1 = require("./match");
const matcher_1 = require("./matcher");
const conditions_1 = require("./private/conditions");
const cyclic_1 = require("./private/cyclic");
const mappings_1 = require("./private/mappings");
const outputs_1 = require("./private/outputs");
const parameters_1 = require("./private/parameters");
const resources_1 = require("./private/resources");
/**
 * Suite of assertions that can be run on a CDK stack.
 *
 * Typically used, as part of unit tests, to validate that the rendered
 * CloudFormation template has expected resources and properties.
 *
 * @stability stable
 */
class Template {
    constructor(template) {
        this.template = template;
        cyclic_1.checkTemplateForCyclicDependencies(this.template);
    }
    /**
     * Base your assertions on the CloudFormation template synthesized by a CDK `Stack`.
     *
     * @param stack the CDK Stack to run assertions on.
     * @stability stable
     */
    static fromStack(stack) {
        return new Template(toTemplate(stack));
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as an in-memory JSON object.
     *
     * @param template the CloudFormation template formatted as a nested set of records.
     * @stability stable
     */
    static fromJSON(template) {
        return new Template(template);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as a JSON string.
     *
     * @param template the CloudFormation template in.
     * @stability stable
     */
    static fromString(template) {
        return new Template(JSON.parse(template));
    }
    /**
     * The CloudFormation template deserialized into an object.
     *
     * @stability stable
     */
    toJSON() {
        return this.template;
    }
    /**
     * Assert that the given number of resources of the given type exist in the template.
     *
     * @param type the resource type;
     * @param count number of expected instances.
     * @stability stable
     */
    resourceCountIs(type, count) {
        const counted = resources_1.countResources(this.template, type);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that a resource of the given type and properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavour, use other matchers in the `Match` class.
     *
     * @param type the resource type;
     * @param props the 'Properties' section of the resource as should be expected in the template.
     * @stability stable
     */
    hasResourceProperties(type, props) {
        const matchError = resources_1.hasResourceProperties(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a resource of the given type and given definition exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param type the resource type;
     * @param props the entire defintion of the resource as should be expected in the template.
     * @stability stable
     */
    hasResource(type, props) {
        const matchError = resources_1.hasResource(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching resources of a given type and properties in the CloudFormation template.
     *
     * @param type the type to match in the CloudFormation template.
     * @param props by default, matches all resources with the given type.
     * @stability stable
     */
    findResources(type, props = {}) {
        return resources_1.findResources(this.template, type, props);
    }
    /**
     * Assert that a Parameter with the given properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the parameter, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     *
     * @param logicalId the name of the parameter.
     * @param props the parameter as should be expected in the template.
     * @stability stable
     */
    hasParameter(logicalId, props) {
        const matchError = parameters_1.hasParameter(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Parameters that match the given properties in the CloudFormation template.
     *
     * @param logicalId the name of the parameter.
     * @param props by default, matches all Parameters in the template.
     * @stability stable
     */
    findParameters(logicalId, props = {}) {
        return parameters_1.findParameters(this.template, logicalId, props);
    }
    /**
     * Assert that an Output with the given properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param logicalId the name of the output.
     * @param props the output as should be expected in the template.
     * @stability stable
     */
    hasOutput(logicalId, props) {
        const matchError = outputs_1.hasOutput(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Outputs that match the given properties in the CloudFormation template.
     *
     * @param logicalId the name of the output.
     * @param props by default, matches all Outputs in the template.
     * @stability stable
     */
    findOutputs(logicalId, props = {}) {
        return outputs_1.findOutputs(this.template, logicalId, props);
    }
    /**
     * Assert that a Mapping with the given properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param logicalId the name of the mapping.
     * @param props the output as should be expected in the template.
     * @stability stable
     */
    hasMapping(logicalId, props) {
        const matchError = mappings_1.hasMapping(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Mappings that match the given properties in the CloudFormation template.
     *
     * @param logicalId the name of the mapping.
     * @param props by default, matches all Mappings in the template.
     * @stability stable
     */
    findMappings(logicalId, props = {}) {
        return mappings_1.findMappings(this.template, logicalId, props);
    }
    /**
     * Assert that a Condition with the given properties exists in the CloudFormation template.
     *
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     *
     * @param logicalId the name of the mapping.
     * @param props the output as should be expected in the template.
     * @stability stable
     */
    hasCondition(logicalId, props) {
        const matchError = conditions_1.hasCondition(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Conditions that match the given properties in the CloudFormation template.
     *
     * @param logicalId the name of the condition.
     * @param props by default, matches all Conditions in the template.
     * @stability stable
     */
    findConditions(logicalId, props = {}) {
        return conditions_1.findConditions(this.template, logicalId, props);
    }
    /**
     * Assert that the CloudFormation template matches the given value.
     *
     * @param expected the expected CloudFormation template as key-value pairs.
     * @stability stable
     */
    templateMatches(expected) {
        const matcher = matcher_1.Matcher.isMatcher(expected) ? expected : match_1.Match.objectLike(expected);
        const result = matcher.test(this.template);
        if (result.hasFailed()) {
            throw new Error([
                'Template did not match as expected. The following mismatches were found:',
                ...result.toHumanStrings().map(s => `\t${s}`),
            ].join('\n'));
        }
    }
}
exports.Template = Template;
_a = JSII_RTTI_SYMBOL_1;
Template[_a] = { fqn: "@aws-cdk/assertions.Template", version: "1.142.0" };
function toTemplate(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    const assembly = root.synth();
    if (stack.nestedStackParent) {
        // if this is a nested stack (it has a parent), then just read the template as a string
        return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
    }
    return assembly.getStackArtifact(stack.artifactId).template;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVtcGxhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZW1wbGF0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZCQUE2QjtBQUM3Qix3Q0FBNkM7QUFDN0MsK0JBQStCO0FBQy9CLG1DQUFnQztBQUNoQyx1Q0FBb0M7QUFDcEMscURBQW9FO0FBQ3BFLDZDQUFzRTtBQUN0RSxpREFBOEQ7QUFDOUQsK0NBQTJEO0FBQzNELHFEQUFvRTtBQUNwRSxtREFBd0c7Ozs7Ozs7OztBQUl4RyxNQUFhLFFBQVE7SUFtQm5CLFlBQW9CLFFBQWdDO1FBQ2xELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBd0IsQ0FBQztRQUN6QywyQ0FBa0MsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7S0FDbkQ7Ozs7Ozs7SUFuQk0sTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFZO1FBQ2xDLE9BQU8sSUFBSSxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7S0FDeEM7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsUUFBUSxDQUFDLFFBQWlDO1FBQ3RELE9BQU8sSUFBSSxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7S0FDL0I7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQWdCO1FBQ3ZDLE9BQU8sSUFBSSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO0tBQzNDOzs7Ozs7SUFVTSxNQUFNO1FBQ1gsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDO0tBQ3RCOzs7Ozs7OztJQUdNLGVBQWUsQ0FBQyxJQUFZLEVBQUUsS0FBYTtRQUNoRCxNQUFNLE9BQU8sR0FBRywwQkFBYyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDcEQsSUFBSSxPQUFPLEtBQUssS0FBSyxFQUFFO1lBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsWUFBWSxLQUFLLHNCQUFzQixJQUFJLGNBQWMsT0FBTyxFQUFFLENBQUMsQ0FBQztTQUNyRjtLQUNGOzs7Ozs7Ozs7OztJQUdNLHFCQUFxQixDQUFDLElBQVksRUFBRSxLQUFVO1FBQ25ELE1BQU0sVUFBVSxHQUFHLGlDQUFxQixDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3JFLElBQUksVUFBVSxFQUFFO1lBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUM3QjtLQUNGOzs7Ozs7Ozs7OztJQUdNLFdBQVcsQ0FBQyxJQUFZLEVBQUUsS0FBVTtRQUN6QyxNQUFNLFVBQVUsR0FBRyx1QkFBVyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQzNELElBQUksVUFBVSxFQUFFO1lBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUM3QjtLQUNGOzs7Ozs7OztJQUdNLGFBQWEsQ0FBQyxJQUFZLEVBQUUsUUFBYSxFQUFFO1FBQ2hELE9BQU8seUJBQWEsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztLQUNsRDs7Ozs7Ozs7Ozs7SUFHTSxZQUFZLENBQUMsU0FBaUIsRUFBRSxLQUFVO1FBQy9DLE1BQU0sVUFBVSxHQUFHLHlCQUFZLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDakUsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7Ozs7Ozs7O0lBR00sY0FBYyxDQUFDLFNBQWlCLEVBQUUsUUFBYSxFQUFFO1FBQ3RELE9BQU8sMkJBQWMsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUN4RDs7Ozs7Ozs7Ozs7SUFHTSxTQUFTLENBQUMsU0FBaUIsRUFBRSxLQUFVO1FBQzVDLE1BQU0sVUFBVSxHQUFHLG1CQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDOUQsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7Ozs7Ozs7O0lBR00sV0FBVyxDQUFDLFNBQWlCLEVBQUUsUUFBYSxFQUFFO1FBQ25ELE9BQU8scUJBQVcsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUNyRDs7Ozs7Ozs7Ozs7SUFHTSxVQUFVLENBQUMsU0FBaUIsRUFBRSxLQUFVO1FBQzdDLE1BQU0sVUFBVSxHQUFHLHFCQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDL0QsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7Ozs7Ozs7O0lBR00sWUFBWSxDQUFDLFNBQWlCLEVBQUUsUUFBYSxFQUFFO1FBQ3BELE9BQU8sdUJBQVksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUN0RDs7Ozs7Ozs7Ozs7SUFHTSxZQUFZLENBQUMsU0FBaUIsRUFBRSxLQUFVO1FBQy9DLE1BQU0sVUFBVSxHQUFHLHlCQUFZLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDakUsSUFBSSxVQUFVLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzdCO0tBQ0Y7Ozs7Ozs7O0lBR00sY0FBYyxDQUFDLFNBQWlCLEVBQUUsUUFBYSxFQUFFO1FBQ3RELE9BQU8sMkJBQWMsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztLQUN4RDs7Ozs7OztJQUdNLGVBQWUsQ0FBQyxRQUFhO1FBQ2xDLE1BQU0sT0FBTyxHQUFHLGlCQUFPLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLGFBQUssQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDcEYsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFM0MsSUFBSSxNQUFNLENBQUMsU0FBUyxFQUFFLEVBQUU7WUFDdEIsTUFBTSxJQUFJLEtBQUssQ0FBQztnQkFDZCwwRUFBMEU7Z0JBQzFFLEdBQUcsTUFBTSxDQUFDLGNBQWMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7YUFDOUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztTQUNmO0tBQ0Y7O0FBekhILDRCQTBIQzs7O0FBRUQsU0FBUyxVQUFVLENBQUMsS0FBWTtJQUM5QixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztJQUM3QixJQUFJLENBQUMsWUFBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtRQUN4QixNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxDQUFDLENBQUM7S0FDN0U7SUFFRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLEVBQUU7UUFDM0IsdUZBQXVGO1FBQ3ZGLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztLQUN6RztJQUNELE9BQU8sUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxRQUFRLENBQUM7QUFDOUQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBTdGFjaywgU3RhZ2UgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzLWV4dHJhJztcbmltcG9ydCB7IE1hdGNoIH0gZnJvbSAnLi9tYXRjaCc7XG5pbXBvcnQgeyBNYXRjaGVyIH0gZnJvbSAnLi9tYXRjaGVyJztcbmltcG9ydCB7IGZpbmRDb25kaXRpb25zLCBoYXNDb25kaXRpb24gfSBmcm9tICcuL3ByaXZhdGUvY29uZGl0aW9ucyc7XG5pbXBvcnQgeyBjaGVja1RlbXBsYXRlRm9yQ3ljbGljRGVwZW5kZW5jaWVzIH0gZnJvbSAnLi9wcml2YXRlL2N5Y2xpYyc7XG5pbXBvcnQgeyBmaW5kTWFwcGluZ3MsIGhhc01hcHBpbmcgfSBmcm9tICcuL3ByaXZhdGUvbWFwcGluZ3MnO1xuaW1wb3J0IHsgZmluZE91dHB1dHMsIGhhc091dHB1dCB9IGZyb20gJy4vcHJpdmF0ZS9vdXRwdXRzJztcbmltcG9ydCB7IGZpbmRQYXJhbWV0ZXJzLCBoYXNQYXJhbWV0ZXIgfSBmcm9tICcuL3ByaXZhdGUvcGFyYW1ldGVycyc7XG5pbXBvcnQgeyBjb3VudFJlc291cmNlcywgZmluZFJlc291cmNlcywgaGFzUmVzb3VyY2UsIGhhc1Jlc291cmNlUHJvcGVydGllcyB9IGZyb20gJy4vcHJpdmF0ZS9yZXNvdXJjZXMnO1xuaW1wb3J0IHsgVGVtcGxhdGUgYXMgVGVtcGxhdGVUeXBlIH0gZnJvbSAnLi9wcml2YXRlL3RlbXBsYXRlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBUZW1wbGF0ZSB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tU3RhY2soc3RhY2s6IFN0YWNrKTogVGVtcGxhdGUge1xuICAgIHJldHVybiBuZXcgVGVtcGxhdGUodG9UZW1wbGF0ZShzdGFjaykpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBmcm9tSlNPTih0ZW1wbGF0ZTogeyBba2V5OiBzdHJpbmddIDogYW55IH0pOiBUZW1wbGF0ZSB7XG4gICAgcmV0dXJuIG5ldyBUZW1wbGF0ZSh0ZW1wbGF0ZSk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcodGVtcGxhdGU6IHN0cmluZyk6IFRlbXBsYXRlIHtcbiAgICByZXR1cm4gbmV3IFRlbXBsYXRlKEpTT04ucGFyc2UodGVtcGxhdGUpKTtcbiAgfVxuXG4gIHByaXZhdGUgcmVhZG9ubHkgdGVtcGxhdGU6IFRlbXBsYXRlVHlwZTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHRlbXBsYXRlOiB7IFtrZXk6IHN0cmluZ106IGFueSB9KSB7XG4gICAgdGhpcy50ZW1wbGF0ZSA9IHRlbXBsYXRlIGFzIFRlbXBsYXRlVHlwZTtcbiAgICBjaGVja1RlbXBsYXRlRm9yQ3ljbGljRGVwZW5kZW5jaWVzKHRoaXMudGVtcGxhdGUpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHRvSlNPTigpOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICByZXR1cm4gdGhpcy50ZW1wbGF0ZTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVzb3VyY2VDb3VudElzKHR5cGU6IHN0cmluZywgY291bnQ6IG51bWJlcik6IHZvaWQge1xuICAgIGNvbnN0IGNvdW50ZWQgPSBjb3VudFJlc291cmNlcyh0aGlzLnRlbXBsYXRlLCB0eXBlKTtcbiAgICBpZiAoY291bnRlZCAhPT0gY291bnQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgRXhwZWN0ZWQgJHtjb3VudH0gcmVzb3VyY2VzIG9mIHR5cGUgJHt0eXBlfSBidXQgZm91bmQgJHtjb3VudGVkfWApO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGhhc1Jlc291cmNlUHJvcGVydGllcyh0eXBlOiBzdHJpbmcsIHByb3BzOiBhbnkpOiB2b2lkIHtcbiAgICBjb25zdCBtYXRjaEVycm9yID0gaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKHRoaXMudGVtcGxhdGUsIHR5cGUsIHByb3BzKTtcbiAgICBpZiAobWF0Y2hFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1hdGNoRXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgaGFzUmVzb3VyY2UodHlwZTogc3RyaW5nLCBwcm9wczogYW55KTogdm9pZCB7XG4gICAgY29uc3QgbWF0Y2hFcnJvciA9IGhhc1Jlc291cmNlKHRoaXMudGVtcGxhdGUsIHR5cGUsIHByb3BzKTtcbiAgICBpZiAobWF0Y2hFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1hdGNoRXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZmluZFJlc291cmNlcyh0eXBlOiBzdHJpbmcsIHByb3BzOiBhbnkgPSB7fSk6IHsgW2tleTogc3RyaW5nXTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB9IHtcbiAgICByZXR1cm4gZmluZFJlc291cmNlcyh0aGlzLnRlbXBsYXRlLCB0eXBlLCBwcm9wcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBoYXNQYXJhbWV0ZXIobG9naWNhbElkOiBzdHJpbmcsIHByb3BzOiBhbnkpOiB2b2lkIHtcbiAgICBjb25zdCBtYXRjaEVycm9yID0gaGFzUGFyYW1ldGVyKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICAgIGlmIChtYXRjaEVycm9yKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IobWF0Y2hFcnJvcik7XG4gICAgfVxuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGZpbmRQYXJhbWV0ZXJzKGxvZ2ljYWxJZDogc3RyaW5nLCBwcm9wczogYW55ID0ge30pOiB7IFtrZXk6IHN0cmluZ106IHsgW2tleTogc3RyaW5nXTogYW55IH0gfSB7XG4gICAgcmV0dXJuIGZpbmRQYXJhbWV0ZXJzKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgaGFzT3V0cHV0KGxvZ2ljYWxJZDogc3RyaW5nLCBwcm9wczogYW55KTogdm9pZCB7XG4gICAgY29uc3QgbWF0Y2hFcnJvciA9IGhhc091dHB1dCh0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgICBpZiAobWF0Y2hFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1hdGNoRXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBmaW5kT3V0cHV0cyhsb2dpY2FsSWQ6IHN0cmluZywgcHJvcHM6IGFueSA9IHt9KTogeyBba2V5OiBzdHJpbmddOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IH0ge1xuICAgIHJldHVybiBmaW5kT3V0cHV0cyh0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgaGFzTWFwcGluZyhsb2dpY2FsSWQ6IHN0cmluZywgcHJvcHM6IGFueSk6IHZvaWQge1xuICAgIGNvbnN0IG1hdGNoRXJyb3IgPSBoYXNNYXBwaW5nKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICAgIGlmIChtYXRjaEVycm9yKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IobWF0Y2hFcnJvcik7XG4gICAgfVxuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBmaW5kTWFwcGluZ3MobG9naWNhbElkOiBzdHJpbmcsIHByb3BzOiBhbnkgPSB7fSk6IHsgW2tleTogc3RyaW5nXTogeyBba2V5OiBzdHJpbmddOiBhbnkgfSB9IHtcbiAgICByZXR1cm4gZmluZE1hcHBpbmdzKHRoaXMudGVtcGxhdGUsIGxvZ2ljYWxJZCwgcHJvcHMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgaGFzQ29uZGl0aW9uKGxvZ2ljYWxJZDogc3RyaW5nLCBwcm9wczogYW55KTogdm9pZCB7XG4gICAgY29uc3QgbWF0Y2hFcnJvciA9IGhhc0NvbmRpdGlvbih0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgICBpZiAobWF0Y2hFcnJvcikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKG1hdGNoRXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBmaW5kQ29uZGl0aW9ucyhsb2dpY2FsSWQ6IHN0cmluZywgcHJvcHM6IGFueSA9IHt9KTogeyBba2V5OiBzdHJpbmddOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IH0ge1xuICAgIHJldHVybiBmaW5kQ29uZGl0aW9ucyh0aGlzLnRlbXBsYXRlLCBsb2dpY2FsSWQsIHByb3BzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgdGVtcGxhdGVNYXRjaGVzKGV4cGVjdGVkOiBhbnkpOiB2b2lkIHtcbiAgICBjb25zdCBtYXRjaGVyID0gTWF0Y2hlci5pc01hdGNoZXIoZXhwZWN0ZWQpID8gZXhwZWN0ZWQgOiBNYXRjaC5vYmplY3RMaWtlKGV4cGVjdGVkKTtcbiAgICBjb25zdCByZXN1bHQgPSBtYXRjaGVyLnRlc3QodGhpcy50ZW1wbGF0ZSk7XG5cbiAgICBpZiAocmVzdWx0Lmhhc0ZhaWxlZCgpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoW1xuICAgICAgICAnVGVtcGxhdGUgZGlkIG5vdCBtYXRjaCBhcyBleHBlY3RlZC4gVGhlIGZvbGxvd2luZyBtaXNtYXRjaGVzIHdlcmUgZm91bmQ6JyxcbiAgICAgICAgLi4ucmVzdWx0LnRvSHVtYW5TdHJpbmdzKCkubWFwKHMgPT4gYFxcdCR7c31gKSxcbiAgICAgIF0uam9pbignXFxuJykpO1xuICAgIH1cbiAgfVxufVxuXG5mdW5jdGlvbiB0b1RlbXBsYXRlKHN0YWNrOiBTdGFjayk6IGFueSB7XG4gIGNvbnN0IHJvb3QgPSBzdGFjay5ub2RlLnJvb3Q7XG4gIGlmICghU3RhZ2UuaXNTdGFnZShyb290KSkge1xuICAgIHRocm93IG5ldyBFcnJvcigndW5leHBlY3RlZDogYWxsIHN0YWNrcyBtdXN0IGJlIHBhcnQgb2YgYSBTdGFnZSBvciBhbiBBcHAnKTtcbiAgfVxuXG4gIGNvbnN0IGFzc2VtYmx5ID0gcm9vdC5zeW50aCgpO1xuICBpZiAoc3RhY2submVzdGVkU3RhY2tQYXJlbnQpIHtcbiAgICAvLyBpZiB0aGlzIGlzIGEgbmVzdGVkIHN0YWNrIChpdCBoYXMgYSBwYXJlbnQpLCB0aGVuIGp1c3QgcmVhZCB0aGUgdGVtcGxhdGUgYXMgYSBzdHJpbmdcbiAgICByZXR1cm4gSlNPTi5wYXJzZShmcy5yZWFkRmlsZVN5bmMocGF0aC5qb2luKGFzc2VtYmx5LmRpcmVjdG9yeSwgc3RhY2sudGVtcGxhdGVGaWxlKSkudG9TdHJpbmcoJ3V0Zi04JykpO1xuICB9XG4gIHJldHVybiBhc3NlbWJseS5nZXRTdGFja0FydGlmYWN0KHN0YWNrLmFydGlmYWN0SWQpLnRlbXBsYXRlO1xufSJdfQ==