"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkTemplateForCyclicDependencies = void 0;
/**
 * Check a template for cyclic dependencies
 *
 * This will make sure that we don't happily validate templates
 * in unit tests that wouldn't deploy to CloudFormation anyway.
 */
function checkTemplateForCyclicDependencies(template) {
    var _a, _b, _c;
    const logicalIds = new Set(Object.keys((_a = template.Resources) !== null && _a !== void 0 ? _a : {}));
    const dependencies = new Map();
    for (const [logicalId, resource] of Object.entries((_b = template.Resources) !== null && _b !== void 0 ? _b : {})) {
        dependencies.set(logicalId, intersect(findResourceDependencies(resource), logicalIds));
    }
    // We will now progressively remove entries from the map of 'dependencies' that have
    // 0 elements in them. If we can't do that anymore and the map isn't empty, we
    // have a cyclic dependency.
    while (dependencies.size > 0) {
        const free = Array.from(dependencies.entries()).filter(([_, deps]) => deps.size === 0);
        if (free.length === 0) {
            // Oops!
            const cycle = findCycle(dependencies);
            const cycleResources = {};
            for (const logicalId of cycle) {
                cycleResources[logicalId] = (_c = template.Resources) === null || _c === void 0 ? void 0 : _c[logicalId];
            }
            throw new Error(`Template is undeployable, these resources have a dependency cycle: ${cycle.join(' -> ')}:\n\n${JSON.stringify(cycleResources, undefined, 2)}`);
        }
        for (const [logicalId, _] of free) {
            for (const deps of dependencies.values()) {
                deps.delete(logicalId);
            }
            dependencies.delete(logicalId);
        }
    }
}
exports.checkTemplateForCyclicDependencies = checkTemplateForCyclicDependencies;
function findResourceDependencies(res) {
    var _a;
    return new Set([
        ...toArray((_a = res.DependsOn) !== null && _a !== void 0 ? _a : []),
        ...findExpressionDependencies(res.Properties),
    ]);
}
function toArray(x) {
    return Array.isArray(x) ? x : [x];
}
function findExpressionDependencies(obj) {
    const ret = new Set();
    recurse(obj);
    return ret;
    function recurse(x) {
        if (!x) {
            return;
        }
        if (Array.isArray(x)) {
            x.forEach(recurse);
        }
        if (typeof x === 'object') {
            const keys = Object.keys(x);
            if (keys.length === 1 && keys[0] === 'Ref') {
                ret.add(x[keys[0]]);
            }
            else if (keys.length === 1 && keys[0] === 'Fn::GetAtt') {
                ret.add(x[keys[0]][0]);
            }
            else if (keys.length === 1 && keys[0] === 'Fn::Sub') {
                const argument = x[keys[0]];
                const pattern = Array.isArray(argument) ? argument[0] : argument;
                for (const logId of logicalIdsInSubString(pattern)) {
                    ret.add(logId);
                }
                const contextDict = Array.isArray(argument) ? argument[1] : undefined;
                if (contextDict) {
                    Object.values(contextDict).forEach(recurse);
                }
            }
            else {
                Object.values(x).forEach(recurse);
            }
        }
    }
}
/**
 * Return the logical IDs found in a {Fn::Sub} format string
 */
function logicalIdsInSubString(x) {
    return analyzeSubPattern(x).flatMap((fragment) => {
        switch (fragment.type) {
            case 'getatt':
            case 'ref':
                return [fragment.logicalId];
            case 'literal':
                return [];
        }
    });
}
function analyzeSubPattern(pattern) {
    const ret = [];
    let start = 0;
    let ph0 = pattern.indexOf('${', start);
    while (ph0 > -1) {
        if (pattern[ph0 + 2] === '!') {
            // "${!" means "don't actually substitute"
            start = ph0 + 3;
            ph0 = pattern.indexOf('${', start);
            continue;
        }
        const ph1 = pattern.indexOf('}', ph0 + 2);
        if (ph1 === -1) {
            break;
        }
        const placeholder = pattern.substring(ph0 + 2, ph1);
        if (ph0 > start) {
            ret.push({ type: 'literal', content: pattern.substring(start, ph0) });
        }
        if (placeholder.includes('.')) {
            const [logicalId, attr] = placeholder.split('.');
            ret.push({ type: 'getatt', logicalId: logicalId, attr: attr });
        }
        else {
            ret.push({ type: 'ref', logicalId: placeholder });
        }
        start = ph1 + 1;
        ph0 = pattern.indexOf('${', start);
    }
    if (start < pattern.length - 1) {
        ret.push({ type: 'literal', content: pattern.substr(start) });
    }
    return ret;
}
function intersect(xs, ys) {
    return new Set(Array.from(xs).filter(x => ys.has(x)));
}
/**
 * Find cycles in a graph
 *
 * Not the fastest, but effective and should be rare
 */
function findCycle(deps) {
    for (const node of deps.keys()) {
        const cycle = recurse(node, [node]);
        if (cycle) {
            return cycle;
        }
    }
    throw new Error('No cycle found. Assertion failure!');
    function recurse(node, path) {
        var _a;
        for (const dep of (_a = deps.get(node)) !== null && _a !== void 0 ? _a : []) {
            if (dep === path[0]) {
                return [...path, dep];
            }
            const cycle = recurse(dep, [...path, dep]);
            if (cycle) {
                return cycle;
            }
        }
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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