"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Match = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const matcher_1 = require("./matcher");
const absent_1 = require("./private/matchers/absent");
const type_1 = require("./private/type");
/**
 * Partial and special matching during template assertions.
 *
 * @stability stable
 */
class Match {
    /**
     * Use this matcher in the place of a field's value, if the field must not be present.
     *
     * @stability stable
     */
    static absent() {
        return new absent_1.AbsentMatch('absent');
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must be in the same order as would be found.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static arrayWith(pattern) {
        return new ArrayMatch('arrayWith', pattern);
    }
    /**
     * Matches the specified pattern with the array found in the same relative path of the target.
     *
     * The set of elements (or matchers) must match exactly and in order.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static arrayEquals(pattern) {
        return new ArrayMatch('arrayEquals', pattern, { subsequence: false });
    }
    /**
     * Deep exact matching of the specified pattern to the target.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static exact(pattern) {
        return new LiteralMatch('exact', pattern, { partialObjects: false });
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must be present in the target but the target can be a superset.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static objectLike(pattern) {
        return new ObjectMatch('objectLike', pattern);
    }
    /**
     * Matches the specified pattern to an object found in the same relative path of the target.
     *
     * The keys and their values (or matchers) must match exactly with the target.
     *
     * @param pattern the pattern to match.
     * @stability stable
     */
    static objectEquals(pattern) {
        return new ObjectMatch('objectEquals', pattern, { partial: false });
    }
    /**
     * Matches any target which does NOT follow the specified pattern.
     *
     * @param pattern the pattern to NOT match.
     * @stability stable
     */
    static not(pattern) {
        return new NotMatch('not', pattern);
    }
    /**
     * Matches any string-encoded JSON and applies the specified pattern after parsing it.
     *
     * @param pattern the pattern to match after parsing the encoded JSON.
     * @stability stable
     */
    static serializedJson(pattern) {
        return new SerializedJson('serializedJson', pattern);
    }
    /**
     * Matches any non-null value at the target.
     *
     * @stability stable
     */
    static anyValue() {
        return new AnyMatch('anyValue');
    }
    /**
     * Matches targets according to a regular expression.
     *
     * @stability stable
     */
    static stringLikeRegexp(pattern) {
        return new StringLikeRegexpMatch('stringLikeRegexp', pattern);
    }
}
exports.Match = Match;
_a = JSII_RTTI_SYMBOL_1;
Match[_a] = { fqn: "@aws-cdk/assertions.Match", version: "1.142.0" };
/**
 * A Match class that expects the target to match with the pattern exactly.
 * The pattern may be nested with other matchers that are then deletegated to.
 */
class LiteralMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partialObjects = (_b = options.partialObjects) !== null && _b !== void 0 ? _b : false;
        if (matcher_1.Matcher.isMatcher(this.pattern)) {
            throw new Error('LiteralMatch cannot directly contain another matcher. ' +
                'Remove the top-level matcher or nest it more deeply.');
        }
    }
    test(actual) {
        if (Array.isArray(this.pattern)) {
            return new ArrayMatch(this.name, this.pattern, { subsequence: false, partialObjects: this.partialObjects }).test(actual);
        }
        if (typeof this.pattern === 'object') {
            return new ObjectMatch(this.name, this.pattern, { partial: this.partialObjects }).test(actual);
        }
        const result = new matcher_1.MatchResult(actual);
        if (typeof this.pattern !== typeof actual) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected type ${typeof this.pattern} but received ${type_1.getType(actual)}`,
            });
            return result;
        }
        if (actual !== this.pattern) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected ${this.pattern} but received ${actual}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches arrays.
 */
class ArrayMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b, _c;
        super();
        this.name = name;
        this.pattern = pattern;
        this.subsequence = (_b = options.subsequence) !== null && _b !== void 0 ? _b : true;
        this.partialObjects = (_c = options.partialObjects) !== null && _c !== void 0 ? _c : false;
    }
    test(actual) {
        if (!Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type array but received ${type_1.getType(actual)}`,
            });
        }
        if (!this.subsequence && this.pattern.length !== actual.length) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected array of length ${this.pattern.length} but received ${actual.length}`,
            });
        }
        let patternIdx = 0;
        let actualIdx = 0;
        const result = new matcher_1.MatchResult(actual);
        while (patternIdx < this.pattern.length && actualIdx < actual.length) {
            const patternElement = this.pattern[patternIdx];
            const matcher = matcher_1.Matcher.isMatcher(patternElement)
                ? patternElement
                : new LiteralMatch(this.name, patternElement, { partialObjects: this.partialObjects });
            const matcherName = matcher.name;
            if (this.subsequence && (matcherName == 'absent' || matcherName == 'anyValue')) {
                // array subsequence matcher is not compatible with anyValue() or absent() matcher. They don't make sense to be used together.
                throw new Error(`The Matcher ${matcherName}() cannot be nested within arrayWith()`);
            }
            const innerResult = matcher.test(actual[actualIdx]);
            if (!this.subsequence || !innerResult.hasFailed()) {
                result.compose(`[${actualIdx}]`, innerResult);
                patternIdx++;
                actualIdx++;
            }
            else {
                actualIdx++;
            }
        }
        for (; patternIdx < this.pattern.length; patternIdx++) {
            const pattern = this.pattern[patternIdx];
            const element = (matcher_1.Matcher.isMatcher(pattern) || typeof pattern === 'object') ? ' ' : ` [${pattern}] `;
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Missing element${element}at pattern index ${patternIdx}`,
            });
        }
        return result;
    }
}
/**
 * Match class that matches objects.
 */
class ObjectMatch extends matcher_1.Matcher {
    constructor(name, pattern, options = {}) {
        var _b;
        super();
        this.name = name;
        this.pattern = pattern;
        this.partial = (_b = options.partial) !== null && _b !== void 0 ? _b : true;
    }
    test(actual) {
        if (typeof actual !== 'object' || Array.isArray(actual)) {
            return new matcher_1.MatchResult(actual).recordFailure({
                matcher: this,
                path: [],
                message: `Expected type object but received ${type_1.getType(actual)}`,
            });
        }
        const result = new matcher_1.MatchResult(actual);
        if (!this.partial) {
            for (const a of Object.keys(actual)) {
                if (!(a in this.pattern)) {
                    result.recordFailure({
                        matcher: this,
                        path: [`/${a}`],
                        message: 'Unexpected key',
                    });
                }
            }
        }
        for (const [patternKey, patternVal] of Object.entries(this.pattern)) {
            if (!(patternKey in actual) && !(patternVal instanceof absent_1.AbsentMatch)) {
                result.recordFailure({
                    matcher: this,
                    path: [`/${patternKey}`],
                    message: 'Missing key',
                });
                continue;
            }
            const matcher = matcher_1.Matcher.isMatcher(patternVal) ?
                patternVal :
                new LiteralMatch(this.name, patternVal, { partialObjects: this.partial });
            const inner = matcher.test(actual[patternKey]);
            result.compose(`/${patternKey}`, inner);
        }
        return result;
    }
}
class SerializedJson extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    ;
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (type_1.getType(actual) !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected JSON as a string but found ${type_1.getType(actual)}`,
            });
            return result;
        }
        let parsed;
        try {
            parsed = JSON.parse(actual);
        }
        catch (err) {
            if (err instanceof SyntaxError) {
                result.recordFailure({
                    matcher: this,
                    path: [],
                    message: `Invalid JSON string: ${actual}`,
                });
                return result;
            }
            else {
                throw err;
            }
        }
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(parsed);
        result.compose(`(${this.name})`, innerResult);
        return result;
    }
}
class NotMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const matcher = matcher_1.Matcher.isMatcher(this.pattern) ? this.pattern : new LiteralMatch(this.name, this.pattern);
        const innerResult = matcher.test(actual);
        const result = new matcher_1.MatchResult(actual);
        if (innerResult.failCount === 0) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Found unexpected match: ${JSON.stringify(actual, undefined, 2)}`,
            });
        }
        return result;
    }
}
class AnyMatch extends matcher_1.Matcher {
    constructor(name) {
        super();
        this.name = name;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        if (actual == null) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: 'Expected a value but found none',
            });
        }
        return result;
    }
}
class StringLikeRegexpMatch extends matcher_1.Matcher {
    constructor(name, pattern) {
        super();
        this.name = name;
        this.pattern = pattern;
    }
    test(actual) {
        const result = new matcher_1.MatchResult(actual);
        const regex = new RegExp(this.pattern, 'gm');
        if (typeof actual !== 'string') {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `Expected a string, but got '${typeof actual}'`,
            });
        }
        if (!regex.test(actual)) {
            result.recordFailure({
                matcher: this,
                path: [],
                message: `String '${actual}' did not match pattern '${this.pattern}'`,
            });
        }
        return result;
    }
}
//# sourceMappingURL=data:application/json;base64,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