"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const messages_1 = require("./private/messages");
/**
 * Suite of assertions that can be run on a CDK Stack.
 *
 * Focused on asserting annotations.
 *
 * @stability stable
 */
class Annotations {
    constructor(messages) {
        this._messages = convertArrayToMessagesType(messages);
    }
    /**
     * Base your assertions on the messages returned by a synthesized CDK `Stack`.
     *
     * @param stack the CDK Stack to run assertions on.
     * @stability stable
     */
    static fromStack(stack) {
        return new Annotations(toMessages(stack));
    }
    /**
     * Assert that an error with the given message exists in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the error.
     * @param message the error message as should be expected.
     * @stability stable
     */
    hasError(constructPath, message) {
        const matchError = messages_1.hasMessage(this._messages, constructPath, constructMessage('error', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching errors of a given construct path and message.
     *
     * @param constructPath the construct path to the error.
     * @param message the error message as should be expected.
     * @stability stable
     */
    findError(constructPath, message) {
        return convertMessagesTypeToArray(messages_1.findMessage(this._messages, constructPath, constructMessage('error', message)));
    }
    /**
     * Assert that an warning with the given message exists in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the warning.
     * @param message the warning message as should be expected.
     * @stability stable
     */
    hasWarning(constructPath, message) {
        const matchError = messages_1.hasMessage(this._messages, constructPath, constructMessage('warning', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching warning of a given construct path and message.
     *
     * @param constructPath the construct path to the warning.
     * @param message the warning message as should be expected.
     * @stability stable
     */
    findWarning(constructPath, message) {
        return convertMessagesTypeToArray(messages_1.findMessage(this._messages, constructPath, constructMessage('warning', message)));
    }
    /**
     * Assert that an info with the given message exists in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the info.
     * @param message the info message as should be expected.
     * @stability stable
     */
    hasInfo(constructPath, message) {
        const matchError = messages_1.hasMessage(this._messages, constructPath, constructMessage('info', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching infos of a given construct path and message.
     *
     * @param constructPath the construct path to the info.
     * @param message the info message as should be expected.
     * @stability stable
     */
    findInfo(constructPath, message) {
        return convertMessagesTypeToArray(messages_1.findMessage(this._messages, constructPath, constructMessage('info', message)));
    }
}
exports.Annotations = Annotations;
_a = JSII_RTTI_SYMBOL_1;
Annotations[_a] = { fqn: "@aws-cdk/assertions.Annotations", version: "1.142.0" };
function constructMessage(type, message) {
    return {
        level: type,
        entry: {
            data: message,
        },
    };
}
function convertArrayToMessagesType(messages) {
    return messages.reduce((obj, item) => {
        return {
            ...obj,
            [item.id]: item,
        };
    }, {});
}
function convertMessagesTypeToArray(messages) {
    return Object.values(messages);
}
function toMessages(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    // to support incremental assertions (i.e. "expect(stack).toNotContainSomething(); doSomething(); expect(stack).toContainSomthing()")
    const force = true;
    const assembly = root.synth({ force });
    return assembly.getStackArtifact(stack.artifactId).messages;
}
//# sourceMappingURL=data:application/json;base64,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