"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workspace = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_grafana_1 = require("aws-cdk-lib/aws-grafana");
const workspace_base_1 = require("./workspace-base");
;
/**
 * Specifies a workspace. In a workspace, you can create Grafana dashboards and visualizations to
 * analyze your metrics, logs, and traces. You don't have to build, package, or deploy any hardware
 * to run the Grafana server.
 */
class Workspace extends workspace_base_1.WorkspaceBase {
    static fromWorkspaceAttributes(scope, id, attrs) {
        class Import extends workspace_base_1.WorkspaceBase {
            constructor(attributes) {
                super(scope, id);
                this.accountAccessType = attributes.accountAccessType;
                this.authenticationProviders = attributes.authenticationProviders;
                this.clientToken = attributes.clientToken;
                this.dataSources = attributes.dataSources;
                this.description = attributes.description;
                this.name = attributes.name;
                this.networkAccessControl = attributes.networkAccessControl;
                this.notificationDestinations = attributes.notificationDestinations;
                this.organizationalUnits = attributes.organizationalUnits;
                this.organizationRoleName = attributes.organizationRoleName;
                this.permissionType = attributes.permissionType;
                this.pluginAdminEnabled = attributes.pluginAdminEnabled;
                this.role = attributes.role;
                this.samlConfiguration = attributes.samlConfiguration;
                this.stackSetName = attributes.stackSetName;
                this.vpcConfiguration = attributes.vpcConfiguration;
                this.workspaceArn = attributes.workspaceArn;
                this.workspaceId = this.getWorkspaceId(attributes.workspaceArn);
            }
        }
        return new Import(attrs);
    }
    /**
     * Validates the clientToken property.
     *
     * @param token - The client token to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be a string
     * - Must be between 1 and 64 characters long
     * - Must contain only printable ASCII characters
     */
    static validateClientToken(token) {
        const errors = [];
        if (typeof token !== 'string') {
            errors.push('clientToken must be a string');
            return errors; // No need to check further if not a string
        }
        const regex = /^[!-~]{1,64}$/;
        if (!regex.test(token)) {
            if (token.length < 1 || token.length > 64) {
                errors.push('clientToken must be between 1 and 64 characters long');
            }
            if (!/^[!-~]*$/.test(token)) {
                errors.push('clientToken must contain only printable ASCII characters');
            }
        }
        return errors;
    }
    /**
     * Validates the description property.
     *
     * @param description - The description to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be a string
     * - Maximum length of 2048 characters
     */
    static validateDescription(description) {
        const errors = [];
        if (typeof description !== 'string') {
            errors.push('description must be a string');
            return errors; // No need to check further if not a string
        }
        if (description.length > 2048) {
            errors.push('description cannot exceed 2048 characters');
        }
        return errors;
    }
    /**
     * Validates the grafanaVersion property.
     *
     * @param version - The Grafana version to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be a string
     * - Must be between 1 and 255 characters long
     */
    static validateGrafanaVersion(version) {
        const errors = [];
        if (typeof version !== 'string') {
            errors.push('grafanaVersion must be a string');
            return errors; // No need to check further if not a string
        }
        if (version.length < 1) {
            errors.push('grafanaVersion cannot be empty');
        }
        if (version.length > 255) {
            errors.push('grafanaVersion cannot exceed 255 characters');
        }
        return errors;
    }
    /**
     * Validates the name property.
     *
     * @param name - The workspace name to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be a string
     * - Must be between 1 and 255 characters long
     * - Can only contain alphanumeric characters, hyphens, dots, underscores, and tildes
     */
    static validateName(name) {
        const errors = [];
        if (typeof name !== 'string') {
            errors.push('name must be a string');
            return errors; // No need to check further if not a string
        }
        const regex = /^[a-zA-Z0-9\-._~]{1,255}$/;
        if (!regex.test(name)) {
            if (name.length < 1 || name.length > 255) {
                errors.push('name must be between 1 and 255 characters long');
            }
            if (!/^[a-zA-Z0-9\-._~]*$/.test(name)) {
                errors.push('name can only contain alphanumeric characters, hyphens, dots, underscores, and tildes');
            }
        }
        return errors;
    }
    /**
     * Validates the networkAccessControl property.
     *
     * @param nac - The network access control configuration to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be an object
     * - prefixLists (if present) must be an array with at most 5 items
     * - vpcEndpoints (if present) must be an array with at most 5 items
     */
    static validateNetworkAccessControl(nac) {
        const errors = [];
        if (!nac || typeof nac !== 'object') {
            errors.push('networkAccessControl must be an object');
            return errors;
        }
        const networkAccessControl = nac;
        // Check prefixLists if present
        if (networkAccessControl.prefixLists !== undefined) {
            if (!Array.isArray(networkAccessControl.prefixLists)) {
                errors.push('prefixLists must be an array');
            }
            else if (networkAccessControl.prefixLists.length > 5) {
                errors.push('prefixLists can have at most 5 items');
            }
        }
        // Check vpcEndpoints if present
        if (networkAccessControl.vpcEndpoints !== undefined) {
            if (!Array.isArray(networkAccessControl.vpcEndpoints)) {
                errors.push('vpcEndpoints must be an array');
            }
            else if (networkAccessControl.vpcEndpoints.length > 5) {
                errors.push('vpcEndpoints can have at most 5 items');
            }
        }
        return errors;
    }
    /**
     * Validates the organizationRoleName property.
     *
     * @param roleName - The organization role name to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be a string
     * - Must be between 1 and 2048 characters long
     */
    static validateOrganizationRoleName(roleName) {
        const errors = [];
        if (typeof roleName !== 'string') {
            errors.push('organizationRoleName must be a string');
            return errors; // No need to check further if not a string
        }
        if (roleName.length < 1) {
            errors.push('organizationRoleName cannot be empty');
        }
        if (roleName.length > 2048) {
            errors.push('organizationRoleName cannot exceed 2048 characters');
        }
        return errors;
    }
    /**
     * Validates the SAML assertion attributes.
     *
     * @param obj - The SAML assertion attributes to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be an object
     * - Each attribute must be a string
     * - Each attribute must be between 1 and 256 characters long
     * - Valid attribute keys are: 'email', 'groups', 'login', 'name', 'org', 'role'
     */
    static validateSamlAssertionAttributes(obj) {
        const errors = [];
        if (!obj || typeof obj !== 'object') {
            return ['Input is not an object'];
        }
        const attributes = obj;
        for (const key in attributes) {
            const value = attributes[key];
            if (value === undefined) {
                continue; // Optional properties can be undefined
            }
            if (typeof value !== 'string') {
                errors.push(`Property '${key}' must be a string`);
            }
            else if (value.length < 1) {
                errors.push(`Property '${key}' cannot be empty`);
            }
            else if (value.length > 256) {
                errors.push(`Property '${key}' exceeds maximum length of 256 characters`);
            }
        }
        return errors;
    }
    /**
     * Validates the SAML IdP metadata.
     *
     * @param obj - The SAML IdP metadata to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be an object
     * - url (if present) must be a string between 1 and 2048 characters long
     * - xml (if present) must be a string
     */
    static validateSamlIdpMetadata(obj) {
        const errors = [];
        if (!obj || typeof obj !== 'object') {
            return ['Input is not an object'];
        }
        const metadata = obj;
        // Check url property if present
        if (metadata.url !== undefined) {
            if (typeof metadata.url !== 'string') {
                errors.push("Property 'url' must be a string");
            }
            else if (metadata.url.length < 1) {
                errors.push("Property 'url' cannot be empty");
            }
            else if (metadata.url.length > 2048) {
                errors.push("Property 'url' exceeds maximum length of 2048 characters");
            }
        }
        // Check xml property if present
        if (metadata.xml !== undefined && typeof metadata.xml !== 'string') {
            errors.push("Property 'xml' must be a string");
        }
        return errors;
    }
    /**
     * Validates the SAML configuration.
     *
     * @param config - The SAML configuration to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be an object
     * - idpMetadata is required and must be valid
     * - assertionAtrributes (if present) must be valid
     * - allowedOrganizations (if present) must be an array of strings with 1-256 elements
     * - loginValidityDuration (if present) must be a positive number
     * - roleValues (if present) must be an object with valid admin and editor arrays
     */
    static validateSamlConfiguration(config) {
        const errors = [];
        if (!config || typeof config !== 'object') {
            errors.push('samlConfiguration must be an object');
            return errors;
        }
        const samlConfig = config;
        // Check idpMetadata (required)
        if (samlConfig.idpMetadata === undefined) {
            errors.push('idpMetadata is required in samlConfiguration');
        }
        else {
            const idpMetadataErrors = Workspace.validateSamlIdpMetadata(samlConfig.idpMetadata);
            if (idpMetadataErrors.length > 0) {
                errors.push(...idpMetadataErrors.map((err) => `idpMetadata: ${err}`));
            }
        }
        // Check assertionAtrributes if present
        if (samlConfig.assertionAtrributes !== undefined) {
            const attributeErrors = Workspace.validateSamlAssertionAttributes(samlConfig.assertionAtrributes);
            if (attributeErrors.length > 0) {
                errors.push(...attributeErrors.map((err) => `assertionAtrributes: ${err}`));
            }
        }
        // Check allowedOrganizations if present
        if (samlConfig.allowedOrganizations !== undefined) {
            if (!Array.isArray(samlConfig.allowedOrganizations)) {
                errors.push('allowedOrganizations must be an array');
            }
            else {
                if (samlConfig.allowedOrganizations.length < 1) {
                    errors.push('allowedOrganizations must have at least 1 element');
                }
                if (samlConfig.allowedOrganizations.length > 256) {
                    errors.push('allowedOrganizations cannot have more than 256 elements');
                }
                for (let i = 0; i < samlConfig.allowedOrganizations.length; i++) {
                    const org = samlConfig.allowedOrganizations[i];
                    if (typeof org !== 'string') {
                        errors.push(`allowedOrganizations[${i}] must be a string`);
                    }
                }
            }
        }
        // Check loginValidityDuration if present
        if (samlConfig.loginValidityDuration !== undefined) {
            if (typeof samlConfig.loginValidityDuration !== 'number') {
                errors.push('loginValidityDuration must be a number');
            }
            else if (samlConfig.loginValidityDuration <= 0) {
                errors.push('loginValidityDuration must be positive');
            }
        }
        // Check roleValues if present
        if (samlConfig.roleValues !== undefined) {
            if (!samlConfig.roleValues || typeof samlConfig.roleValues !== 'object') {
                errors.push('roleValues must be an object');
            }
            else {
                // Check admin array if present
                if (samlConfig.roleValues.admin !== undefined) {
                    if (!Array.isArray(samlConfig.roleValues.admin)) {
                        errors.push('roleValues.admin must be an array');
                    }
                    else {
                        for (let i = 0; i < samlConfig.roleValues.admin.length; i++) {
                            if (typeof samlConfig.roleValues.admin[i] !== 'string') {
                                errors.push(`roleValues.admin[${i}] must be a string`);
                            }
                        }
                        if (samlConfig.roleValues.admin.length > 256) {
                            errors.push('roleValues.admin cannot have more than 256 elements');
                        }
                    }
                }
                // Check editor array if present
                if (samlConfig.roleValues.editor !== undefined) {
                    if (!Array.isArray(samlConfig.roleValues.editor)) {
                        errors.push('roleValues.editor must be an array');
                    }
                    else {
                        for (let i = 0; i < samlConfig.roleValues.editor.length; i++) {
                            if (typeof samlConfig.roleValues.editor[i] !== 'string') {
                                errors.push(`roleValues.editor[${i}] must be a string`);
                            }
                        }
                        if (samlConfig.roleValues.editor.length > 256) {
                            errors.push('roleValues.editor cannot have more than 256 elements');
                        }
                    }
                }
            }
        }
        return errors;
    }
    /**
     * Validates the vpcConfiguration property.
     *
     * @param config - The VPC configuration to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * Validation rules:
     * - Must be an object
     * - securityGroups is required and must be an array with 1-5 items
     * - subnets is required and must be an array with 2-6 items
     */
    static validateVpcConfiguration(config) {
        const errors = [];
        if (!config || typeof config !== 'object') {
            errors.push('vpcConfiguration must be an object');
            return errors;
        }
        const vpcConfig = config;
        // Check securityGroups (required)
        if (vpcConfig.securityGroups === undefined) {
            errors.push('securityGroups is required in vpcConfiguration');
        }
        else if (!Array.isArray(vpcConfig.securityGroups)) {
            errors.push('securityGroups must be an array');
        }
        else {
            if (vpcConfig.securityGroups.length < 1) {
                errors.push('securityGroups must have at least 1 item');
            }
            if (vpcConfig.securityGroups.length > 5) {
                errors.push('securityGroups cannot have more than 5 items');
            }
        }
        // Check subnets (required)
        if (vpcConfig.subnets === undefined) {
            errors.push('subnets is required in vpcConfiguration');
        }
        else if (!Array.isArray(vpcConfig.subnets)) {
            errors.push('subnets must be an array');
        }
        else {
            if (vpcConfig.subnets.length < 2) {
                errors.push('subnets must have at least 2 items');
            }
            if (vpcConfig.subnets.length > 6) {
                errors.push('subnets cannot have more than 6 items');
            }
        }
        return errors;
    }
    /**
     * Validates all workspace properties.
     *
     * @param props - The workspace properties to validate
     * @returns An array of error messages if validation fails, or an empty array if valid
     *
     * This method aggregates validation results from all individual property validators.
     * It throws an error if props is not an object.
     */
    static validateProps(props) {
        const errors = [];
        if (!props || typeof props !== 'object') {
            throw new Error('Props is not an object');
        }
        const workspaceProps = props;
        if (workspaceProps.clientToken !== undefined) {
            const clientTokenErrors = Workspace.validateClientToken(workspaceProps.clientToken);
            if (clientTokenErrors.length > 0) {
                errors.push(...clientTokenErrors.map((err) => `clientToken: ${err}`));
            }
        }
        if (workspaceProps.description !== undefined) {
            const descriptionErrors = Workspace.validateDescription(workspaceProps.description);
            if (descriptionErrors.length > 0) {
                errors.push(...descriptionErrors.map((err) => `description: ${err}`));
            }
        }
        if (workspaceProps.grafanaVersion !== undefined) {
            const grafanaVersionErrors = Workspace.validateGrafanaVersion(workspaceProps.grafanaVersion);
            if (grafanaVersionErrors.length > 0) {
                errors.push(...grafanaVersionErrors.map((err) => `grafanaVersion: ${err}`));
            }
        }
        if (workspaceProps.name !== undefined) {
            const nameErrors = Workspace.validateName(workspaceProps.name);
            if (nameErrors.length > 0) {
                errors.push(...nameErrors.map((err) => `name: ${err}`));
            }
        }
        if (workspaceProps.networkAccessControl !== undefined) {
            const networkAccessControlErrors = Workspace.validateNetworkAccessControl(workspaceProps.networkAccessControl);
            if (networkAccessControlErrors.length > 0) {
                errors.push(...networkAccessControlErrors.map((err) => `networkAccessControl: ${err}`));
            }
        }
        if (workspaceProps.organizationRoleName !== undefined) {
            const organizationRoleNameErrors = Workspace.validateOrganizationRoleName(workspaceProps.organizationRoleName);
            if (organizationRoleNameErrors.length > 0) {
                errors.push(...organizationRoleNameErrors.map((err) => `organizationRoleName: ${err}`));
            }
        }
        if (workspaceProps.samlConfiguration !== undefined) {
            const samlConfigurationErrors = Workspace.validateSamlConfiguration(workspaceProps.samlConfiguration);
            if (samlConfigurationErrors.length > 0) {
                errors.push(...samlConfigurationErrors.map((err) => `samlConfiguration: ${err}`));
            }
        }
        if (workspaceProps.vpcConfiguration !== undefined) {
            const vpcConfigurationErrors = Workspace.validateVpcConfiguration(workspaceProps.vpcConfiguration);
            if (vpcConfigurationErrors.length > 0) {
                errors.push(...vpcConfigurationErrors.map((err) => `vpcConfiguration: ${err}`));
            }
        }
        return errors;
    }
    constructor(scope, id, props) {
        super(scope, id);
        const errors = Workspace.validateProps(props);
        if (props.accountAccessType === workspace_base_1.AccountAccessType.CURRENT_ACCOUNT &&
            !props.role) {
            errors.push('Role must be provided when accountAccessType is CURRENT_ACCOUNT');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid props:\n${errors.join('\n')}`);
        }
        this.accountAccessType = props.accountAccessType;
        this.authenticationProviders = props.authenticationProviders;
        this.clientToken = props.clientToken;
        this.dataSources = props.dataSources;
        this.description = props.description;
        this.networkAccessControl = props.networkAccessControl;
        this.notificationDestinations = props.notificationDestinations;
        this.organizationalUnits = props.organizationalUnits;
        this.organizationRoleName = props.organizationRoleName;
        this.permissionType = props.permissionType;
        this.pluginAdminEnabled = props.pluginAdminEnabled;
        this.name = props.name;
        this.role = props.role;
        this.samlConfiguration = props.samlConfiguration;
        this.stackSetName = props.stackSetName;
        this.vpcConfiguration = props.vpcConfiguration;
        let cfnWorkspaceProps = {
            accountAccessType: props.accountAccessType,
            authenticationProviders: props.authenticationProviders,
            clientToken: props.clientToken,
            dataSources: props.dataSources,
            description: props.description,
            grafanaVersion: props.grafanaVersion,
            name: props.name,
            notificationDestinations: props.notificationDestinations,
            organizationalUnits: props.organizationalUnits,
            organizationRoleName: props.organizationRoleName,
            permissionType: props.permissionType,
            pluginAdminEnabled: props.pluginAdminEnabled,
            roleArn: props.role?.roleArn,
            samlConfiguration: props.samlConfiguration,
            stackSetName: props.stackSetName,
            vpcConfiguration: props.vpcConfiguration
                ? {
                    securityGroupIds: props.vpcConfiguration.securityGroups.map((sg) => sg.securityGroupId),
                    subnetIds: props.vpcConfiguration.subnets.map((subnet) => subnet.subnetId),
                }
                : undefined,
        };
        this.workspace = new aws_grafana_1.CfnWorkspace(this, 'Resource', cfnWorkspaceProps);
        this.node.defaultChild = this.workspace;
        this.creationTimestamp = this.workspace.attrCreationTimestamp;
        this.endpoint = this.workspace.attrEndpoint;
        this.grafanaVersion = this.workspace.attrGrafanaVersion;
        this.workspaceId = this.workspace.attrId;
        this.modificationTimestamp = this.workspace.attrModificationTimestamp;
        this.samlConfigurationStatus = this.workspace
            .attrSamlConfigurationStatus;
        this.ssoClientId = this.workspace.attrSsoClientId;
        this.status = this.workspace.attrStatus;
        this.workspaceArn = this.getWorkspaceArn(this.workspaceId);
    }
}
exports.Workspace = Workspace;
_a = JSII_RTTI_SYMBOL_1;
Workspace[_a] = { fqn: "@robhan-cdk-lib/aws_grafana.Workspace", version: "0.0.52" };
//# sourceMappingURL=data:application/json;base64,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