# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xair_api']

package_data = \
{'': ['*']}

install_requires = \
['python-osc>=1.8.0,<2.0.0']

extras_require = \
{':python_version < "3.11"': ['tomli>=2.0.1,<3.0.0']}

entry_points = \
{'console_scripts': ['obs = scripts:ex_obs',
                     'x32 = scripts:test_x32',
                     'xair = scripts:test_xair']}

setup_kwargs = {
    'name': 'xair-api',
    'version': '2.2.1',
    'description': 'Remote control Behringer X-Air | Midas MR mixers through OSC',
    'long_description': '[![PyPI version](https://badge.fury.io/py/xair-api.svg)](https://badge.fury.io/py/xair-api)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://github.com/onyx-and-iris/xair-api-python/blob/dev/LICENSE)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Imports: isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n![Tests Status](./tests/xair/MR18.svg?dummy=8484744)\n\n# Xair API\n\nThis package offers a python interface for the [Behringer XAir](https://www.behringer.com/series.html?category=R-BEHRINGER-XAIRSERIES), [Midas MR](https://www.midasconsoles.com/catalog.html?catalog=Category&category=C-MIDAS-MIXERS-DIGITALSTAGEBOXMIXERS) series of digital rack mixers. I only have access to an MR18 for testing so if there is an error in the kind maps feel free to raise an issue or PR.\n\nFor an outline of past/future changes refer to: [CHANGELOG](CHANGELOG.md)\n\n## Prerequisites\n\n-   Python 3.10 or greater\n\n## Installation\n\n```\npip install xair-api\n```\n\n## Usage\n\n### Connection\n\nA toml file named config.toml, placed into the current working directory of your code may be used to configure the mixers ip. A valid `config.toml` may resemble:\n\n```toml\n[connection]\nip = "<ip address>"\n```\n\nAlternatively you may pass it as a keyword argument.\n\n### Example\n\n```python\nimport xair_api\n\n\ndef main():\n    kind_id = "XR18"\n    ip = "<ip address>"\n\n    with xair_api.connect(kind_id, ip=ip) as mixer:\n        mixer.strip[8].config.name = "sm7b"\n        mixer.strip[8].mix.on = True\n        print(\n            f"strip 09 ({mixer.strip[8].config.name}) on has been set to {mixer.strip[8].mix.on}"\n        )\n\n\nif __name__ == "__main__":\n    main()\n```\n\n#### `xair_api.connect(kind_id, ip=ip, delay=delay)`\n\nCurrently the following devices are supported:\n\n-   `MR18`\n-   `XR18`\n-   `XR16`\n-   `XR12`\n\nThe `X32` is partially supported. However, this document covers specifically the `XAir` series.\n\nThe following keyword arguments may be passed:\n\n-   `ip`: ip address of the mixer\n-   `port`: mixer port, defaults to 10023 for x32 and 10024 for xair\n-   `delay`: a delay between each command, defaults to 20ms.\n    -   a note about delay, stability may rely on network connection. For wired connections the delay can be safely reduced.\n\n## API\n\n### XAirRemote class (higher level)\n\n`mixer.lr`\n\nA class representing Main LR channel\n\n`mixer.strip`\n\nA Strip tuple containing a class for each input strip channel\n\n`mixer.bus`\n\nA Bus tuple containing a class for each output bus channel\n\n`mixer.dca`\n\nA DCA tuple containing a class for each DCA group\n\n`mixer.fx`\n\nAn FX tuple containing a class for each FX channel\n\n`mixer.fxsend`\n\nAn FXSend tuple containing a class for each FX Send channel\n\n`mixer.fxreturn`\n\nAn FXReturn tuple containing a class for each FX Return channel\n\n`mixer.auxreturn`\n\nA class representing auxreturn channel\n\n`mixer.config`\n\nA class representing the main config settings\n\n### `LR`\n\nContains the subclasses:\n(`Config`, `Dyn`, `Insert`, `EQ`, `Mix`)\n\n### `Strip`\n\nContains the subclasses:\n(`Config`, `Preamp`, `Gate`, `Dyn`, `Insert`, `GEQ`, `EQ`, `Mix`, `Group`, `Automix`)\n\n### `Bus`\n\nContains the subclasses:\n(`Config`, `Dyn`, `Insert`, `EQ`, `Mix`, `Group`)\n\n### `FXSend`\n\nContains the subclasses:\n(`Config`, `Mix`, `Group`)\n\n### `FXRtn`\n\nContains the subclasses:\n(`Config`, `Preamp`, `EQ`, `Mix`, `Group`)\n\n### `AuxRtn`\n\nContains the subclasses:\n(`Config`, `Preamp`, `EQ`, `Mix`, `Group`)\n\n### `Subclasses`\n\nFor each subclass the corresponding properties are available.\n\n`Config`\n\n-   `name`: string\n-   `color`: int, from 0, 16\n-   `inputsource`: int\n-   `usbreturn`: int\n\n`Preamp`\n\n-   `on`: bool\n-   `usbtrim`: float, from -18.0 to 18.0\n-   `usbinput`: bool\n-   `invert`: bool\n-   `highpasson`: bool\n-   `highpassfilter`: int, from 20 to 400\n\n`Gate`\n\n-   `on`: bool\n-   `mode`: str, one of (\'gate\', \'exp2\', \'exp3\', \'exp4\', \'duck\')\n-   `threshold`: float, from -80.0 to 0.0\n-   `range`: int, from 3 to 60\n-   `attack`: int, from 0 to 120\n-   `hold`: float, from 0.02 to 2000\n-   `release`: int, from 5 to 4000\n-   `keysource`, from 0 to 22\n-   `filteron`: bool\n-   `filtertype`: int, from 0 to 8\n-   `filterfreq`: float, from 20 to 20000\n\n`Dyn`\n\n-   `on`: bool\n-   `mode`: str, one of (\'comp\', \'exp\')\n-   `det`: str, one of (\'peak\', \'rms\')\n-   `env`: str, one of (\'lin\', \'log\')\n-   `threshold`: float, from -60.0 to 0.0\n-   `ratio`: int, from 0 to 11\n-   `knee`: int, from 0 to 5\n-   `mgain`: float, from 0.0 to 24.0\n-   `attack`: int, from 0 to 120\n-   `hold`: float, from 0.02 to 2000\n-   `release`: int, from 5 to 4000\n-   `mix`: int, from 0 to 100\n-   `keysource`: int, from 0 to 22\n-   `auto`: bool\n-   `filteron`: bool\n-   `filtertype`: int, from 0 to 8\n-   `filterfreq`: float, from 20 to 20000\n\n`Insert`\n\n-   `on`: bool\n-   `sel`: int\n\n`GEQ`\nThe following method names preceded by `slider_`\n\n-   `20`, `25`, `31_5`, `40`, `50`, `63`, `80`, `100`, `125`, `160`,\n-   `200`, `250`, `315`, `400`, `500`, `630`, `800`, `1k`, `1k25`, `1k6`, `2k`,\n-   `2k5`, `3k15`, `4k`, `5k`, `6k3`, `8k`, `10k`, `12k5`, `16k`, `20k`: float, from -15.0 to 15.0\n\nfor example: `slider_20`, `slider_6k3` etc..\n\n`EQ`\n\n-   `on`: bool\n-   `mode`: str, one of (\'peq\', \'geq\', \'teq\')\n\nFor the subclasses: `low`, `low2`, `lomid`, `himid`, `high2`, `high` the following properties are available:\n\n-   `type`: int, from 0 to 5\n-   `frequency`: float, from 20.0 to 20000.0\n-   `gain`: float, -15.0 to 15.0\n-   `quality`: float, from 0.3 to 10.0\n\nfor example: `eq.low2.type`\n\n`Mix`\n\n-   `on`: bool\n-   `fader`: float, -inf, to 10.0\n-   `lr`: bool\n\n`Group`\n\n-   `dca`: int, from 0 to 15\n-   `mute`: int, from 0 to 15\n\n`Automix`\n\n-   `group`: int, from 0 to 2\n-   `weight`: float, from -12.0 to 12.0\n\n### `DCA`\n\n-   `on`: bool\n-   `name`: str\n-   `color`: int, from 0 to 15\n\n### `Config`\n\nThe following method names preceded by `chlink`\n\n-   `1_2`, `3_4`, `5_6`, `7_8`, `9_10`, `11_12`, `13_14`, `15_16`\n\nThe following method names preceded by `buslink`\n\n-   `1_2`, `3_4`, `5_6`\n\nfor example: `chlink1_2`, `buslink5_6` etc..\n\n-   `link_eq`: bool\n-   `link_dyn`: bool\n-   `link_fader_mute`: bool\n-   `amixenable`: bool\n-   `amixlock`: bool\n\nFor the subclass `monitor` the following properties are available\n\n-   `level`: float, -inf to 10.0\n-   `source`: int, from 0 to 14\n-   `sourcetrim`: float, from -18.0 to 18.0\n-   `chmode`: bool\n-   `busmode`: bool\n-   `dim`: bool\n-   `dimgain`: float, from -40.0 to 0.0\n-   `mono`: bool\n-   `mute`: bool\n-   `dimfpl`: bool\n\nfor example: `config.monitor.chmode`\n\n#### `mute_group`\n\ntuple containing a class for each mute group\n\n-   `on`: bool, from 0 to 3\n\nfor example: `config.mute_group[0].on = True`\n\n### XAirRemote class (lower level)\n\nSend an OSC command directly to the mixer\n\n-   `send(osc command, value)`\n\nfor example:\n\n```python\nmixer.send("/ch/01/mix/on", 1)\nmixer.send("/bus/2/config/name", "somename")\n```\n\nQuery the value of a command:\n\n-   `query(osc command)`\n\nfor example:\n\n```python\nprint(mixer.query("/ch/01/mix/on"))\n```\n\n### `Tests`\n\nUnplug any expensive equipment before running tests.\nSave your current settings to a snapshot first.\n\nFirst make sure you installed the [development dependencies](https://github.com/onyx-and-iris/xair-api-python#installation)\n\nTo run all tests:\n\n`pytest -v`.\n\n## License\n\nThis project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details\n\n## Documentation\n\n[XAir OSC Commands](https://behringer.world/wiki/doku.php?id=x-air_osc)\n\n[X32 OSC Commands](https://wiki.munichmakerlab.de/images/1/17/UNOFFICIAL_X32_OSC_REMOTE_PROTOCOL_%281%29.pdf)\n\n## Special Thanks\n\n[Peter Dikant](https://github.com/peterdikant) for writing the base class\n',
    'author': 'onyx-and-iris',
    'author_email': 'code@onyxandiris.online',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/onyx-and-iris/xair-api-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
