"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unquote = void 0;
const pipeVariable_1 = require("./pipeVariable");
// enrich the pipe variable values with quotes so that the regex match includes the quotes, otherwise the replace function will not find the quotes
const pipeVariableValuesWithQuotes = Object.values(pipeVariable_1.PipeVariable).map(value => `"${value}"`);
// build a regex that matches the pipe variable values with quotes like "<aws.pipes.pipe-name>"
const pipeVariableRegex = new RegExp(`(${pipeVariableValuesWithQuotes.join('|')})`, 'g');
// A regex that matches the event path expressions values with quotes like "<$.foo.bar>"
const eventPathRegex = new RegExp('"(<\\$.+?>)"', 'g');
/**
 * Removes the quotes from a string
 * @param match a string that eventually contains quotes
 * @returns a string without quotes
 */
const removeQuotes = (match) => match.replace(/^"|"$/g, '');
/**
 * Removes the quotes from the matched pipe variable values
 * @param stringWithQuotes a string that eventually contains pipe variable values with quotes
 * @param replacer a function that replaces the matched pipe variable values with quotes
 * @returns a string without quotes
 */
const removeQuotesFromVariables = (stringWithQuotes, replacer) => stringWithQuotes.replace(pipeVariableRegex, replacer);
const removeQuotesFromEventPathExpression = (stringWithQuotes, replacer) => stringWithQuotes.replace(eventPathRegex, replacer);
/**
 * Removes the quotes from PipeVariables and EventPathExpressions
 * @param stringWithQuotes a string that eventually contains pipe variable or event path expression values with quotes
 * @returns a string were pipe variables and event path expressions don't have quotes
 */
const unquote = (stringWithQuotes) => {
    const stringWithoutVariablesQuotes = removeQuotesFromVariables(stringWithQuotes, removeQuotes);
    const stringWithoutEventPathQuotes = removeQuotesFromEventPathExpression(stringWithoutVariablesQuotes, removeQuotes);
    return stringWithoutEventPathQuotes;
};
exports.unquote = unquote;
//# sourceMappingURL=data:application/json;base64,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