import { Construct } from '@aws-cdk/core';
import { IRole } from '@aws-cdk/aws-iam';
import { IVpc, IInstance } from '@aws-cdk/aws-ec2';
export interface GitlabContainerRunnerProps {
    /**
     * Gitlab token for the Register Runner .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN' });
     *
     * @default - You must to give the token !!!
     *
     */
    readonly gitlabtoken: string;
    /**
     * Runner default EC2 instance type.
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', ec2type: 't3.small' });
     *
     * @default - t3.micro
     *
     */
    readonly ec2type?: string;
    /**
     * VPC for the Gitlab Runner .
     *
     * @example
     * const newvpc = new Vpc(stack, 'NEWVPC', {
     *   cidr: '10.1.0.0/16',
     *   maxAzs: 2,
     *   subnetConfiguration: [{
     *     cidrMask: 26,
     *     name: 'RunnerVPC',
     *     subnetType: SubnetType.PUBLIC,
     *   }],
     *   natGateways: 0,
     * });
     *
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', selfvpc: newvpc });
     *
     * @default - new VPC will be created , 1 Vpc , 2 Public Subnet .
     *
     */
    readonly selfvpc?: IVpc;
    /**
     * IAM role for the Gitlab Runner Instance .
     *
     * @example
     * const role = new Role(stack, 'runner-role', {
     *   assumedBy: new ServicePrincipal('ec2.amazonaws.com'),
     *   description: 'For Gitlab EC2 Runner Test Role',
     *   roleName: 'Myself-Runner-Role',
     * });
     *
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', ec2iamrole: role });
     *
     * @default - new Role for Gitlab Runner Instance , attach AmazonSSMManagedInstanceCore Policy .
     *
     */
    readonly ec2iamrole?: IRole;
    /**
     * Gitlab Runner register tag1  .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', tag1: 'aa' });
     *
     * @default - tag1: gitlab .
     *
     */
    readonly tag1?: string;
    /**
     * Gitlab Runner register tag2  .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', tag2: 'bb' });
     *
     * @default - tag2: awscdk .
     *
     */
    readonly tag2?: string;
    /**
     * Gitlab Runner register tag3  .
     *
     * @example
     * new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN', tag3: 'cc' });
     *
     * @default - tag3: runner .
     *
     */
    readonly tag3?: string;
    /**
     * Gitlab Runner register url .
     *
     * @example
     * const runner = new GitlabContainerRunner(stack, 'runner', { gitlabtoken: 'GITLAB_TOKEN',gitlaburl: 'https://gitlab.com/'});
     *
     * @default - gitlaburl='https://gitlab.com/'
     *
     */
    readonly gitlaburl?: string;
}
export declare class GitlabContainerRunner extends Construct {
    /**
     * The IAM role assumed by the Runner instance .
     */
    readonly runnerRole: IRole;
    /**
    * This represents a Runner EC2 instance .
    */
    readonly runnerEc2: IInstance;
    constructor(scope: Construct, id: string, props: GitlabContainerRunnerProps);
}
