###############################################

# x3d.py X3D Package for Python

# generator:  X3duomToX3dPythonPackage.xslt
# X3DUOM:     X3dUnifiedObjectModel-4.0.xml
# Python X3D: https://www.web3d.org/x3d/stylesheets/python/python.html
"""
The x3d.py Python X3D Package supports programmers with Python interfaces and objects for standards-based X3D programming, all as open source.

This work is part of the X3D Python Scene Access Interface Library (X3DPSAIL).
"""

# Include regular expression (regex) library: re is built into Python
# https://docs.python.org/3/library/re.html
# https://docs.python.org/3/howto/regex.html#regex-howto
# https://www.web3d.org/specifications/X3dRegularExpressions.html

import re

_DEBUG = False       # options True False

###############################################

# SimpleType Enumerations

ACCESSTYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'initializeOnly', # A field with accessType initializeOnly can be initialized, but cannot send or receive events.
    'inputOnly', # A field with accessType inputOnly cannot be initialized or included in a scene file, but can receive input event values via a ROUTE.
    'outputOnly', # A field with accessType outputOnly cannot be initialized or included in a scene file, but can send output event values via a ROUTE.
    'inputOutput' # A field with accessType inputOutput can be initialized, and can also send or receive events.
)
def assertValidAccessType(fieldName, value):
    """
    Utility function to assert type validity of accessTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in ACCESSTYPECHOICES:
        # print('*** assertValidAccessType ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in ACCESSTYPECHOICES:
        # print('*** assertValidAccessType ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in ACCESSTYPECHOICES=' + str(ACCESSTYPECHOICES))

ALPHAMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'AUTO', # Material transparency is applied to texture transparency
    'OPAQUE', # Ignore alpha channel texture transparency, opaque
    'MASK', # Alpha-testing mode for transparent when alpha value less than 0.5 and opaque when greater than or equal to 0.5
    'BLEND' # Blend combines partial transparency of textures and materials
)
def assertValidAlphaMode(fieldName, value):
    """
    Utility function to assert type validity of alphaModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in ALPHAMODECHOICES:
        # print('*** assertValidAlphaMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in ALPHAMODECHOICES:
        # print('*** assertValidAlphaMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in ALPHAMODECHOICES=' + str(ALPHAMODECHOICES))

APPLIEDPARAMETERSCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"BOUNCE"', # The bounce field value is used.
    '"USER_FRICTION"', # The system will normally calculate the friction direction vector that is perpendicular to the contact normal. This setting indicates that the user-supplied value in this contact should be used.
    '"FRICTION_COEFFICIENT-2"', # Apply frictionCoefficients values
    '"ERROR_REDUCTION"', # Apply softnessErrorCorrection value
    '"CONSTANT_FORCE"', # Apply softnessConstantForceMix value
    '"SPEED-1"', # Apply first component of surfaceSpeed array
    '"SPEED-2"', # Apply second component of surfaceSpeed array
    '"SLIP-1"', # Apply first component of slipFactors array
    '"SLIP-2"' # Apply second component of slipFactors array
)
def assertValidAppliedParameters(fieldName, value):
    """
    Utility function to assert type validity of appliedParametersChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in APPLIEDPARAMETERSCHOICES:
        # print('*** assertValidAppliedParameters ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in APPLIEDPARAMETERSCHOICES:
        # print('*** assertValidAppliedParameters ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in APPLIEDPARAMETERSCHOICES=' + str(APPLIEDPARAMETERSCHOICES))

BIQUADTYPEFILTERCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'lowpass',
    'highpass',
    'bandpass',
    'lowshelf',
    'highshelf',
    'peaking',
    'notch',
    'allpass'
)
def assertValidBiquadTypeFilter(fieldName, value):
    """
    Utility function to assert type validity of biquadTypeFilterChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in BIQUADTYPEFILTERCHOICES:
        # print('*** assertValidBiquadTypeFilter ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in BIQUADTYPEFILTERCHOICES:
        # print('*** assertValidBiquadTypeFilter ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in BIQUADTYPEFILTERCHOICES=' + str(BIQUADTYPEFILTERCHOICES))

CHANNELCOUNTMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'max',
    'clamped-max',
    'explicit'
)
def assertValidChannelCountMode(fieldName, value):
    """
    Utility function to assert type validity of channelCountModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in CHANNELCOUNTMODECHOICES:
        # print('*** assertValidChannelCountMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in CHANNELCOUNTMODECHOICES:
        # print('*** assertValidChannelCountMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in CHANNELCOUNTMODECHOICES=' + str(CHANNELCOUNTMODECHOICES))

CHANNELINTERPRETATIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'speakers',
    'discrete'
)
def assertValidChannelInterpretation(fieldName, value):
    """
    Utility function to assert type validity of channelInterpretationChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in CHANNELINTERPRETATIONCHOICES:
        # print('*** assertValidChannelInterpretation ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in CHANNELINTERPRETATIONCHOICES:
        # print('*** assertValidChannelInterpretation ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in CHANNELINTERPRETATIONCHOICES=' + str(CHANNELINTERPRETATIONCHOICES))

CLOSURETYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'PIE', # Connects arc endpoints to center, forming a pie wedge
    'CHORD' # Connects arc endpoints directly to each other, as in chord on a circle
)
def assertValidClosureType(fieldName, value):
    """
    Utility function to assert type validity of closureTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in CLOSURETYPECHOICES:
        # print('*** assertValidClosureType ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in CLOSURETYPECHOICES:
        # print('*** assertValidClosureType ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in CLOSURETYPECHOICES=' + str(CLOSURETYPECHOICES))

COMPONENTNAMECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'Core', # The Core component supplies the base functionality for the X3D run-time system, including the abstract base node type, field types, the event model, and routing.
    'CADGeometry', # The CADGeometry component is provided for Computer-Aided Design (CAD) nodes.
    'CubeMapTexturing', # The Cube Map Environmental Texturing component describes how additional texturing effects are defined to produce environmental effects such as reflections from objects.
    'DIS', # The Distributed Interactive Simulation (DIS) component provides networked interoperability with the IEEE DIS protocol for sharing state and conducting real-time platform-level simulations across multiple host computers.
    'EnvironmentalEffects', # Nodes in the Environmental effects component support the creation of realistic environmental effects such as panoramic backgrounds and fog.
    'EnvironmentalSensor', # The Environment Sensor nodes emit events indicating activity in the scene environment, usually based on interactions between the viewer and the world.
    'EventUtilities', # The Event Utility nodes provide the capability to filter, trigger, convert, or sequence numerous event-types for common interactive applications without the use of a Script node.
    'Followers', # The Follower nodes (Chasers and Dampers) support dynamic creation of smooth parameter transitions at run time.
    'Geometry2D', # The Geometry2D component defines how two-dimensional geometry is specified and what shapes are available.
    'Geometry3D', # The Geometry3D component describes how three-dimensional geometry is specified and defines ElevationGrid, Extrusion, IndexedFaceSet, and most primitive geometry nodes (Box, Cone, Cylinder, Sphere).
    'Geospatial', # The Geospatial component defines how to associate real-world locations in an X3D scene and specifies nodes particularly tuned for geospatial applications.
    'Grouping', # The Grouping component describes how nodes are organized into groups to establish a transformation hierarchy for the X3D scene graph.
    'HAnim', # The Humanoid Animation (HAnim) component for X3D defines node bindings and other details for implementing ISO/IEC 19774, the HAnim International Specification. Original name was H-Anim for X3D versions 3.0 through 3.3, both enumeration values HAnim and H-Anim are allowed to pass validation.
    'H-Anim', # Legacy enumeration H-Anim for X3D versions 3.0-3.3 provides backwards compatibility with Humanoid Animation (HAnim) version 1, preferred form of enumeration value is HAnim.
    'Interpolation', # Interpolator nodes provide keyframe-based animation capability.
    'KeyDeviceSensor', # The Key Device Sensor defines how keyboard keystrokes are inserted into an X3D world.
    'Layering', # The Layering component describes how to layer a set of subscene layers into a composite scene.
    'Layout', # The Layout component defines how to precisely position content in a scene in relation to the rendered results, especially for integrating 2D content with 3D content.
    'Lighting', # The Lighting component specifies how light sources are defined and positioned, as well as how lights effect the rendered image.
    'Navigation', # The Navigation component specifies how a user can effectively and intuitively move through and around a 3D scene.
    'Networking', # The Networking component defines node types and other features used to access file-based and streaming resources on the World Wide Web.
    'NURBS', # The NURBS component describes Non-uniform Rational B-Spline (NURBS) geometry and interpolation nodes.
    'ParticleSystems', # The Particle Systems component specifies how to model particles and their interactions through the application of basic physics principles to affect motion.
    'Picking', # The Picking component provides the ability to test for arbitrary object collision and provide basic capabilities to detecting object intersections and interactions.
    'PointingDeviceSensor', # Pointing device sensor nodes detect pointing events from user-interface devices, defining activities such as a user selecting a piece of geometry.
    'TextureProjector', # TextureProjector nodes project textures onto geometry in a scene.
    'Rendering', # The Rendering component includes fundamental rendering primitives such as TriangleSet and PointSet nodes, as well as geometric properties nodes that define how coordinate indices, colors, normals and texture coordinates are specified.
    'RigidBodyPhysics', # The Rigid Body Physics component describes how to model rigid bodies and their interactions through the application of basic physics principles to effect motion.
    'Scripting', # The Scripting component describes how Script nodes are used to effect changes in X3D worlds.
    'Shaders', # The Programmable Shaders component describes how programmable shaders are specified and how they affect the visual appearance of geometry.
    'Shape', # The Shape component defines nodes for associating geometry with their visible properties and the scene environment.
    'Sound', # The Sound component defines how sound is delivered to an X3D world as well as how sounds are accessed.
    'Text', # The Text component defines how text strings are rendered in an X3D scene.
    'Texturing', # The Texturing component specifies how 2D texture images are defined and then positioned on associated geometry.
    'Texturing3D', # The Texturing3D component specifies how 3D volumetric textures describe surface properties as data points in a volume of space, rather than a flat surface.
    'Time', # The Time component defines how time is sensed, computed and associated with events in an X3D scene.
    'VolumeRendering' # The Volume Rendering component provides the ability to specify and render volumetric data sets.
)
def assertValidComponentName(fieldName, value):
    """
    Utility function to assert type validity of componentNameChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in COMPONENTNAMECHOICES:
        # print('*** assertValidComponentName ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in COMPONENTNAMECHOICES:
        # print('*** assertValidComponentName ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in COMPONENTNAMECHOICES=' + str(COMPONENTNAMECHOICES))

DISTANCEMODELCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'linear',
    'inverse',
    'exponential'
)
def assertValidDistanceModel(fieldName, value):
    """
    Utility function to assert type validity of distanceModelChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in DISTANCEMODELCHOICES:
        # print('*** assertValidDistanceModel ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in DISTANCEMODELCHOICES:
        # print('*** assertValidDistanceModel ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in DISTANCEMODELCHOICES=' + str(DISTANCEMODELCHOICES))

FIELDTYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'SFBool', # Single Field (singleton) Boolean
    'MFBool', # Multiple Field (list) Boolean
    'SFColor', # Single Field (singleton) color value, red-green-blue
    'MFColor', # Multiple Field (list) color value, red-green-blue
    'SFColorRGBA', # Single Field (singleton) color value, red-green-blue alpha (opacity)
    'MFColorRGBA', # Multiple Field (list) color value, red-green-blue alpha (opacity)
    'SFDouble', # Single Field (singleton) double-precision (64-bit) float
    'MFDouble', # Multiple Field (list) 2-tuple double-precision (64-bit) float vector
    'SFFloat', # Single Field (singleton) single-precision (32-bit) float
    'MFFloat', # Multiple Field (list) single-precision (32-bit) float vector
    'SFImage', # Single Field (singleton) image value
    'MFImage', # Multiple Field (list) image values
    'SFInt32', # Single Field (singleton) 32-bit integer
    'MFInt32', # Multiple Field (list) 32-bit integer
    'SFNode', # Single Field (singleton) node
    'MFNode', # Multiple Field (list) nodes
    'SFRotation', # Single Field (singleton) rotation value using 3-tuple axis, radian angle
    'MFRotation', # Multiple Field (list) rotation values using 3-tuple axis, radian angle
    'SFString', # Single Field (singleton) string value
    'MFString', # Multiple Field (list) SFString array
    'SFTime', # Single Field (singleton) time value in seconds
    'MFTime', # Multiple Field (list) time array in seconds
    'SFVec2d', # Single Field (singleton) 2-tuple double-precision float vector
    'MFVec2d', # Multiple Field (list) 2-tuple double-precision float vectors
    'SFVec2f', # Single Field (singleton) 2-tuple single-precision float vector
    'MFVec2f', # Multiple Field (list) 2-tuple single-precision float vectors
    'SFVec3d', # Single Field (singleton) 3-tuple double-precision float vector
    'MFVec3d', # Multiple Field (list) 3-tuple double-precision float vectors
    'SFVec3f', # Single Field (singleton) 3-tuple single-precision float vector
    'MFVec3f', # Multiple Field (list) 3-tuple single-precision float vectors
    'SFVec4d', # Single Field (singleton) 4-tuple double-precision float vector
    'MFVec4d', # Multiple Field (list) 4-tuple double-precision float vectors
    'SFVec4f', # Single Field (singleton) 4-tuple single-precision float vector
    'MFVec4f', # Multiple Field (list) 4-tuple single-precision float vectors
    'SFMatrix3d', # Single Field (singleton) 3×3 matrix of double-precision floating point numbers
    'MFMatrix3d', # Multiple Field (list) 3×3 matrices of double-precision floating point numbers
    'SFMatrix3f', # Single Field (singleton) 3×3 matrix of single-precision floating point numbers
    'MFMatrix3f', # Multiple Field (list) 3×3 matrices of double-precision floating point numbers
    'SFMatrix4d', # Single Field (singleton) 4×4 matrix of double-precision floating point numbers
    'MFMatrix4d', # Multiple Field (list) 4×4 matric3w of double-precision floating point numbers
    'SFMatrix4f', # Single Field (singleton) 4×4 matrix of single-precision floating point numbers
    'MFMatrix4f' # Multiple Field (list) 4×4 matrices of single-precision floating point numbers
)
def assertValidFieldType(fieldName, value):
    """
    Utility function to assert type validity of fieldTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in FIELDTYPECHOICES:
        # print('*** assertValidFieldType ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in FIELDTYPECHOICES:
        # print('*** assertValidFieldType ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in FIELDTYPECHOICES=' + str(FIELDTYPECHOICES))

FOGTYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'LINEAR', # linear blending as a function of distance
    'EXPONENTIAL' # exponential blending as a function of distance
)
def assertValidFogType(fieldName, value):
    """
    Utility function to assert type validity of fogTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in FOGTYPECHOICES:
        # print('*** assertValidFogType ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in FOGTYPECHOICES:
        # print('*** assertValidFogType ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in FOGTYPECHOICES=' + str(FOGTYPECHOICES))

FONTFAMILYVALUES = (
    # specification-defined values follow, other values are also allowed
    '"SANS"', # default font family for sans-serif font such as Helvetica
    '"SERIF"', # default font family for serif font such as Times-Roman
    '"TYPEWRITER"' # default font family for a fixed-pitch font such as Courier
)

FONTSTYLECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'PLAIN', # default plain type
    'BOLD', # boldface type
    'ITALIC', # italic type
    'BOLDITALIC' # bold and italic type
)
def assertValidFontStyle(fieldName, value):
    """
    Utility function to assert type validity of fontStyleChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in FONTSTYLECHOICES:
        # print('*** assertValidFontStyle ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in FONTSTYLECHOICES:
        # print('*** assertValidFontStyle ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in FONTSTYLECHOICES=' + str(FONTSTYLECHOICES))

FORCEOUTPUTVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ALL"', # all forceOutput fields computed
    '"NONE"' # no forceOutput fields computed
)

GENERATEDCUBEMAPTEXTUREUPDATECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'NONE', # no further texture updates are rendered
    'NEXT_FRAME_ONLY', # render texture once at end of frame
    'ALWAYS' # texture to be rendered every frame
)
def assertValidGeneratedCubeMapTextureUpdate(fieldName, value):
    """
    Utility function to assert type validity of generatedCubeMapTextureUpdateChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in GENERATEDCUBEMAPTEXTUREUPDATECHOICES:
        # print('*** assertValidGeneratedCubeMapTextureUpdate ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in GENERATEDCUBEMAPTEXTUREUPDATECHOICES:
        # print('*** assertValidGeneratedCubeMapTextureUpdate ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in GENERATEDCUBEMAPTEXTUREUPDATECHOICES=' + str(GENERATEDCUBEMAPTEXTUREUPDATECHOICES))

GEOMETADATASUMMARYKEYVALUES = (
    # specification-defined values follow, other values are also allowed
    'title', # A name to succinctly identify the dataset to user. For example, "San Francisco, California".
    'description', # A brief textual description or summary of the content of the dataset. For example, "LANDSAT 7 satellite imagery taken over northern Scotland".
    'coordinateSystem', # The spatial reference frame used to represent the data (e.g., GD, UTM, or LCC). The list of valid codes that can be used in this field are defined in ISO/IEC 18026. In the case of UTM, the zone number should also be specified in the format "UTM Zx", where the zone number is in the range [1,60]. For example, "UTM Z11".
    'horizontalDatum', # The name of the geodetic datum. The list of valid codes that can be used in this field are defined in ISO/IEC 18026. For example, "W84".
    'verticalDatum', # The name of the vertical datum (geoid). The list of valid codes that can be used in this field are defined in ISO/IEC 18026. For example, "W84".
    'ellipsoid', # The name of the geodetic ellipsoid. The list of valid codes that can be used in this field are defined in ISO/IEC 18026. For example, "WE".
    'extent', # The bounding coordinates for the dataset given in spatial reference frame specified by the coordinateSystem keyword. These are provided in the order eastmost, southmost, westmost, northmost. An example for GD is: "-180.0 -90.0 180.0 90.0".
    'resolution', # SFFloat resolution, or ground sample distance, given in units of length base units. For example, "30".
    'originator', # A string defining the originator of the data, for example the author, agency, organization, publisher, etc. For example, "John Doe, Any Corporation, Some Town, Some Country"
    'copyright', # Any appropriate copyright declaration that pertains to the data. For example, "(c) Copyright 2000, Any Corporation. All rights reserved. Freely distributable."
    'date', # A single date/time, or a date/time range, defining the valid time period to which the data pertains. Dates are specified in the format "YYYY MM DD [HH:MM]". Years in the current time period should be specified using four digits (EXAMPLE "1999" or "2001"). Years can have other than four digits and can be negative. A date range is specified by supplying two values separated by a "-" (hyphen) character. An optional time can be supplied should this level of accuracy be required. Times are to be specified in 24-hour format with respect to GMT. For example, "1999 01 01 00:00 - 1999 12 31 23:59".
    'metadataFormat', # A string that specifies the format of the external metadata description specified by the url field of the GeoMetadata node. For example, "FGDC", "ISO TC211", "CEN TC287", or "OGC".
    'dataUrl', # A hypertext link to the source data used to create the X3D node(s) to which this metadata pertains. Multiple dataUrl keyword/value pairs can be specified in order to provide alternative locations for the same source data. For example, "https://www.foo.bar/data/sf1".
    'dataFormat' # A free-text string that describes the format of the source data used to create the X3D node(s) to which this metadata pertains. This refers to the source data specified by the dataUrl keyword (if present). For example, "USGS 5.5-min DEM".
)

GEOSYSTEMEARTHELLIPSOIDVALUES = (
    # specification-defined values follow, other values are also allowed
    'AM', # Modified Airy
    'AN', # Australian National
    'BN', # Bessel 1841 (Namibia)
    'BR', # Bessel 1841 (Ethiopia Indonesia ...)
    'CC', # Clarke 1866
    'CD', # Clarke 1880
    'EA', # Everest (India 1830)
    'EB', # Everest (Sabah & Sarawak)
    'EC', # Everest (India 1956)
    'ED', # Everest (W. Malaysia 1969)
    'EE', # Everest (W. Malaysia & Singapore 1948)
    'EF', # Everest (Pakistan)
    'FA', # Modified Fischer 1960
    'HE', # Helmert 1906
    'HO', # Hough 1960
    'ID', # Indonesia 1974
    'IN', # International 1924
    'KA', # Krassovsky 1940
    'RF', # Geodetic Reference System 1980 (GRS 80)
    'SA', # South American 1969
    'WD', # WGS 72
    'WE', # WGS 84
    'WGS84', # WGS84 geoid
    'Zn', # Zone number (1..60) (only used with UTM)
    'S' # Southern hemisphere (only used with UTM)
)

GEOSYSTEMSPATIALREFERENCEFRAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'GD', # Geodetic spatial reference frame (latitude/longitude), to be followed by ellipsoid
    'UTM', # Universal Transverse Mercator. One further required argument must be supplied for UTM in order to specify the zone number (1..60) with optional suffix of "S" may be appended in order to specify that the coordinates are in the southern hemisphere. Optional arguments can follow.
    'GC', # Earth-fixed Geocentric with respect to the WGS84 ellipsoid. No additional arguments are supported.
    'GDC', # Synonymous to GD, but may be subject to future deprecation
    'GCC' # Synonymous to GC, but may be subject to future deprecation
)

HANIMFEATUREPOINTNAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'skull_vertex', # CAESAR 2003 skull_vertex matches ISO 7250-1 part 5.22 Vertex (top of head). No corresponding landmark provided in CAESAR 2018.
    'glabella', # glabella is between the eyebrows and above the nose
    'sellion', # osseocartilaginous junction of the nasal dorsum
    'l_infraorbitale', # Left Infraorbitale foramen is opening in maxillary bone of skull located below the infraorbital margin of the orbit.
    'l_tragion', # notch just above the tragus of the ear
    'l_gonion', # Left Gonion is midpoint of mandibular angle of the jaw.
    'r_infraorbitale', # Right Infraorbitale foramen is opening in maxillary bone of skull located below the infraorbital margin of the orbit.
    'r_tragion', # notch just above the tragus of the ear
    'r_gonion', # Right Gonion is midpoint of the mandibular angle of the jaw.
    'supramenton', # center point above tip of chin
    'cervicale',
    'adams_apple',
    'suprasternale', # Suprasternale
    'substernale',
    'l_clavicle',
    'l_acromion',
    'l_axilla_proximal', # Left Axilla Proximal (Anterior)
    'l_axilla_distal', # Left Axilla Distal (Posterior)
    'l_axilla_posterior_folds',
    'r_clavicle',
    'r_acromion',
    'r_axilla_proximal', # Right Axilla Proximal (Anterior)
    'r_axilla_distal', # Right Axilla Distal (Posterior)
    'r_axilla_posterior_folds', # Right Posterior Axillary Folds
    'spine_1_middle_back',
    'spine_2_lower_back',
    'waist_preferred_anterior',
    'waist_preferred_posterior',
    'l_rib10',
    'l_thelion',
    'r_rib10',
    'r_thelion',
    'l_asis',
    'l_iliocristale',
    'l_psis',
    'r_asis',
    'r_iliocristale',
    'r_psis',
    'crotch',
    'l_femoral_lateral_epicondyle',
    'l_femoral_medial_epicondyle',
    'l_suprapatella',
    'l_trochanterion',
    'r_femoral_lateral_epicondyle',
    'r_femoral_medial_epicondyle',
    'r_suprapatella',
    'r_trochanterion',
    'l_tibiale',
    'l_medial_malleolus',
    'l_lateral_malleolus',
    'l_sphyrion',
    'r_tibiale',
    'r_medial_malleolus',
    'r_lateral_malleolus',
    'r_sphyrion',
    'l_metatarsal_phalanx_1',
    'l_metatarsal_phalanx_5',
    'l_dactylion',
    'l_calcaneus_posterior',
    'r_metatarsal_phalanx_1',
    'r_metatarsal_phalanx_5',
    'r_dactylion',
    'r_calcaneus_posterior',
    'l_humeral_lateral_epicondyle',
    'l_humeral_medial_epicondyle',
    'l_olecranon',
    'r_humeral_lateral_epicondyle',
    'r_humeral_medial_epicondyle',
    'r_olecranon',
    'l_radiale',
    'l_ulnar_styloid',
    'l_radial_styloid',
    'r_radiale',
    'r_ulnar_styloid',
    'r_radial_styloid',
    'l_metacarpal_phalanx_2',
    'l_metacarpal_phalanx_3',
    'l_metacarpal_phalanx_5',
    'r_metacarpal_phalanx_2',
    'r_metacarpal_phalanx_3',
    'r_metacarpal_phalanx_5',
    'nuchale',
    'l_neck_base',
    'r_neck_base',
    'navel',
    'l_ectocanthus',
    'r_ectocanthus',
    'menton',
    'mesosternale',
    'opisthocranion',
    'l_knee_crease',
    'r_knee_crease',
    'rear_center_midsagittal_plane',
    'buttocks_standing_wall_contact_point',
    'l_chest_midsagittal_plane',
    'r_chest_midsagittal_plane',
    'l_bideltoid',
    'r_bideltoid',
    'l_carpal_distal_phalanx_1',
    'l_carpal_distal_phalanx_2',
    'l_carpal_distal_phalanx_3',
    'l_carpal_distal_phalanx_4',
    'l_carpal_distal_phalanx_5',
    'r_carpal_distal_phalanx_1',
    'r_carpal_distal_phalanx_2',
    'r_carpal_distal_phalanx_3',
    'r_carpal_distal_phalanx_4',
    'r_carpal_distal_phalanx_5',
    'l_tarsal_distal_phalanx_1',
    'l_tarsal_distal_phalanx_2',
    'l_tarsal_distal_phalanx_3',
    'l_tarsal_distal_phalanx_4',
    'l_tarsal_distal_phalanx_5',
    'r_tarsal_distal_phalanx_1',
    'r_tarsal_distal_phalanx_2',
    'r_tarsal_distal_phalanx_3',
    'r_tarsal_distal_phalanx_4',
    'r_tarsal_distal_phalanx_5'
)

HANIMHUMANOIDINFOKEYVALUES = (
    # specification-defined values follow, other values are also allowed
    'authorName', # Name of Humanoid author.
    'authorEmail', # Email address of Humanoid author.
    'copyright', # Copyright information for this Humanoid model (if any).
    'creationDate', # Creation data for this for this Humanoid model.
    'usageRestrictions', # Usage restrictions for this Humanoid model (if any).
    'humanoidVersion', # The humanoidVersion term refers to the version of the humanoid being used, in order to track revisions to the data. It is not the same as the version field of the Humanoid object, which refers to the version of the HAnim specification that was used when building the humanoid.
    'age', # Description of Humanoid age (not necessarily numeric).
    'gender', # The gender term typically has a value of female, male or neuter.
    'height', # SFFloat Humanoid height in base units (typically meters).
    'weight' # SFFloat Humanoid weight in base units (typically kilograms).
)

HANIMJOINTNAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'humanoid_root',
    'sacroiliac',
    'l_hip',
    'l_knee',
    'l_talocrural',
    'l_talocalcaneonavicular',
    'l_cuneonavicular_1',
    'l_tarsometatarsal_1',
    'l_metatarsophalangeal_1',
    'l_tarsal_interphalangeal_1',
    'l_cuneonavicular_2',
    'l_tarsometatarsal_2',
    'l_metatarsophalangeal_2',
    'l_tarsal_proximal_interphalangeal_2',
    'l_tarsal_distal_interphalangeal_2',
    'l_cuneonavicular_3',
    'l_tarsometatarsal_3',
    'l_metatarsophalangeal_3',
    'l_tarsal_proximal_interphalangeal_3',
    'l_tarsal_distal_interphalangeal_3',
    'l_calcaneocuboid',
    'l_transversetarsal',
    'l_tarsometatarsal_4',
    'l_metatarsophalangeal_4',
    'l_tarsal_proximal_interphalangeal_4',
    'l_tarsal_distal_interphalangeal_4',
    'l_tarsometatarsal_5',
    'l_metatarsophalangeal_5',
    'l_tarsal_proximal_interphalangeal_5',
    'l_tarsal_distal_interphalangeal_5',
    'r_hip',
    'r_knee',
    'r_talocrural',
    'r_talocalcaneonavicular',
    'r_cuneonavicular_1',
    'r_tarsometatarsal_1',
    'r_metatarsophalangeal_1',
    'r_tarsal_interphalangeal_1',
    'r_cuneonavicular_2',
    'r_tarsometatarsal_2',
    'r_metatarsophalangeal_2',
    'r_tarsal_proximal_interphalangeal_2',
    'r_tarsal_distal_interphalangeal_2',
    'r_cuneonavicular_3',
    'r_tarsometatarsal_3',
    'r_metatarsophalangeal_3',
    'r_tarsal_proximal_interphalangeal_3',
    'r_tarsal_distal_interphalangeal_3',
    'r_calcaneocuboid',
    'r_transversetarsal',
    'r_tarsometatarsal_4',
    'r_metatarsophalangeal_4',
    'r_tarsal_proximal_interphalangeal_4',
    'r_tarsal_distal_interphalangeal_4',
    'r_tarsometatarsal_5',
    'r_metatarsophalangeal_5',
    'r_tarsal_proximal_interphalangeal_5',
    'r_tarsal_distal_interphalangeal_5',
    'vl5',
    'vl4',
    'vl3',
    'vl2',
    'vl1',
    'vt12',
    'vt11',
    'vt10',
    'vt9',
    'vt8',
    'vt7',
    'vt6',
    'vt5',
    'vt4',
    'vt3',
    'vt2',
    'vt1',
    'vc7',
    'vc6',
    'vc5',
    'vc4',
    'vc3',
    'vc2',
    'vc1',
    'skullbase',
    'l_eyelid_joint',
    'r_eyelid_joint',
    'l_eyeball_joint',
    'r_eyeball_joint',
    'l_eyebrow_joint',
    'r_eyebrow_joint',
    'temporomandibular',
    'l_sternoclavicular',
    'l_acromioclavicular',
    'l_shoulder',
    'l_elbow',
    'l_radiocarpal',
    'l_midcarpal_1',
    'l_carpometacarpal_1',
    'l_metacarpophalangeal_1',
    'l_carpal_interphalangeal_1',
    'l_midcarpal_2',
    'l_carpometacarpal_2',
    'l_metacarpophalangeal_2',
    'l_carpal_proximal_interphalangeal_2',
    'l_carpal_distal_interphalangeal_2',
    'l_midcarpal_3',
    'l_carpometacarpal_3',
    'l_metacarpophalangeal_3',
    'l_carpal_proximal_interphalangeal_3',
    'l_carpal_distal_interphalangeal_3',
    'l_midcarpal_4_5',
    'l_carpometacarpal_4',
    'l_metacarpophalangeal_4',
    'l_carpal_proximal_interphalangeal_4',
    'l_carpal_distal_interphalangeal_4',
    'l_carpometacarpal_5',
    'l_metacarpophalangeal_5',
    'l_carpal_proximal_interphalangeal_5',
    'l_carpal_distal_interphalangeal_5',
    'r_sternoclavicular',
    'r_acromioclavicular',
    'r_shoulder',
    'r_elbow',
    'r_radiocarpal',
    'r_midcarpal_1',
    'r_carpometacarpal_1',
    'r_metacarpophalangeal_1',
    'r_carpal_interphalangeal_1',
    'r_midcarpal_2',
    'r_carpometacarpal_2',
    'r_metacarpophalangeal_2',
    'r_carpal_proximal_interphalangeal_2',
    'r_carpal_distal_interphalangeal_2',
    'r_midcarpal_3',
    'r_carpometacarpal_3',
    'r_metacarpophalangeal_3',
    'r_carpal_proximal_interphalangeal_3',
    'r_carpal_distal_interphalangeal_3',
    'r_midcarpal_4_5',
    'r_carpometacarpal_4',
    'r_metacarpophalangeal_4',
    'r_carpal_proximal_interphalangeal_4',
    'r_carpal_distal_interphalangeal_4',
    'r_carpometacarpal_5',
    'r_metacarpophalangeal_5',
    'r_carpal_proximal_interphalangeal_5',
    'r_carpal_distal_interphalangeal_5'
)

HANIMSEGMENTNAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'sacrum',
    'pelvis',
    'l_thigh',
    'l_calf',
    'l_talus',
    'l_navicular',
    'l_cuneiform_1',
    'l_metatarsal_1',
    'l_tarsal_proximal_phalanx_1',
    'l_tarsal_distal_phalanx_1',
    'l_cuneiform_2',
    'l_metatarsal_2',
    'l_tarsal_proximal_phalanx_2',
    'l_tarsal_middle_phalanx_2',
    'l_tarsal_distal_phalanx_2',
    'l_cuneiform_3',
    'l_metatarsal_3',
    'l_tarsal_proximal_phalanx_3',
    'l_tarsal_middle_phalanx_3',
    'l_tarsal_distal_phalanx_3',
    'l_calcaneus',
    'l_cuboid',
    'l_metatarsal_4',
    'l_tarsal_proximal_phalanx_4',
    'l_tarsal_middle_phalanx_4',
    'l_tarsal_distal_phalanx_4',
    'l_metatarsal_5',
    'l_tarsal_proximal_phalanx_5',
    'l_tarsal_middle_phalanx_5',
    'l_tarsal_distal_phalanx_5',
    'r_thigh',
    'r_calf',
    'r_talus',
    'r_navicular',
    'r_cuneiform_1',
    'r_metatarsal_1',
    'r_tarsal_proximal_phalanx_1',
    'r_tarsal_distal_phalanx_1',
    'r_cuneiform_2',
    'r_metatarsal_2',
    'r_tarsal_proximal_phalanx_2',
    'r_tarsal_middle_phalanx_2',
    'r_tarsal_distal_phalanx_2',
    'r_cuneiform_3',
    'r_metatarsal_3',
    'r_tarsal_proximal_phalanx_3',
    'r_tarsal_middle_phalanx_3',
    'r_tarsal_distal_phalanx_3',
    'r_calcaneus',
    'r_cuboid',
    'r_metatarsal_4',
    'r_tarsal_proximal_phalanx_4',
    'r_tarsal_middle_phalanx_4',
    'r_tarsal_distal_phalanx_4',
    'r_metatarsal_5',
    'r_tarsal_proximal_phalanx_5',
    'r_tarsal_middle_phalanx_5',
    'r_tarsal_distal_phalanx_5',
    'l5',
    'l4',
    'l3',
    'l2',
    'l1',
    't12',
    't11',
    't10',
    't9',
    't8',
    't7',
    't6',
    't5',
    't4',
    't3',
    't2',
    't1',
    'c7',
    'c6',
    'c5',
    'c4',
    'c3',
    'c2',
    'c1',
    'skull',
    'l_eyelid',
    'r_eyelid',
    'l_eyeball',
    'r_eyeball',
    'l_eyebrow',
    'r_eyebrow',
    'jaw',
    'l_clavicle',
    'l_scapula',
    'l_upperarm',
    'l_forearm',
    'l_carpal',
    'l_trapezium',
    'l_metacarpal_1',
    'l_carpal_proximal_phalanx_1',
    'l_carpal_distal_phalanx_1',
    'l_trapezoid',
    'l_metacarpal_2',
    'l_carpal_proximal_phalanx_2',
    'l_carpal_middle_phalanx_2',
    'l_carpal_distal_phalanx_2',
    'l_capitate',
    'l_metacarpal_3',
    'l_carpal_proximal_phalanx_3',
    'l_carpal_middle_phalanx_3',
    'l_carpal_distal_phalanx_3',
    'l_hamate',
    'l_metacarpal_4',
    'l_carpal_proximal_phalanx_4',
    'l_carpal_middle_phalanx_4',
    'l_carpal_distal_phalanx_4',
    'l_metacarpal_5',
    'l_carpal_proximal_phalanx_5',
    'l_carpal_middle_phalanx_5',
    'l_carpal_distal_phalanx_5',
    'r_clavicle',
    'r_scapula',
    'r_upperarm',
    'r_forearm',
    'r_carpal',
    'r_trapezium',
    'r_metacarpal_1',
    'r_carpal_proximal_phalanx_1',
    'r_carpal_distal_phalanx_1',
    'r_trapezoid',
    'r_metacarpal_2',
    'r_carpal_proximal_phalanx_2',
    'r_carpal_middle_phalanx_2',
    'r_carpal_distal_phalanx_2',
    'r_capitate',
    'r_metacarpal_3',
    'r_carpal_proximal_phalanx_3',
    'r_carpal_middle_phalanx_3',
    'r_carpal_distal_phalanx_3',
    'r_hamate',
    'r_metacarpal_4',
    'r_carpal_proximal_phalanx_4',
    'r_carpal_middle_phalanx_4',
    'r_carpal_distal_phalanx_4',
    'r_metacarpal_5',
    'r_carpal_proximal_phalanx_5',
    'r_carpal_middle_phalanx_5',
    'r_carpal_distal_phalanx_5'
)

HANIMVERSIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    '1.0', # International standard HAnim 19774 version 1 approved by ISO in 2006. Note that HAnim version 2.0 has more capabilties, while version 1.0 includes several small incompatibilities. Since no other versions were formally approved, no other values are allowed for earlier HAnim versions.
    '2.0' # Revised standard HAnim 19774 version 2 (parts 1 and 2) were approved by ISO in November 2019, published by Web3D Consortium May 2020.
)
def assertValidHanimVersion(fieldName, value):
    """
    Utility function to assert type validity of hanimVersionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in HANIMVERSIONCHOICES:
        # print('*** assertValidHanimVersion ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in HANIMVERSIONCHOICES:
        # print('*** assertValidHanimVersion ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in HANIMVERSIONCHOICES=' + str(HANIMVERSIONCHOICES))

INTERSECTIONTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    'BOUNDS', # TODO undefined in X3D specification
    'GEOMETRY' # TODO undefined in X3D specification
)

JUSTIFYCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"MIDDLE"',
    '"MIDDLE" "BEGIN"',
    '"MIDDLE" "END"',
    '"MIDDLE" "FIRST"',
    '"MIDDLE" "MIDDLE"',
    '"BEGIN"',
    '"BEGIN" "BEGIN"',
    '"BEGIN" "END"',
    '"BEGIN" "FIRST"',
    '"BEGIN" "MIDDLE"',
    '"END"',
    '"END" "BEGIN"',
    '"END" "END"',
    '"END" "FIRST"',
    '"END" "MIDDLE"',
    '"FIRST"',
    '"FIRST" "BEGIN"',
    '"FIRST" "END"',
    '"FIRST" "FIRST"',
    '"FIRST" "MIDDLE"'
)
def assertValidJustify(fieldName, value):
    """
    Utility function to assert type validity of justifyChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in JUSTIFYCHOICES:
        # print('*** assertValidJustify ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in JUSTIFYCHOICES:
        # print('*** assertValidJustify ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in JUSTIFYCHOICES=' + str(JUSTIFYCHOICES))

LAYOUTALIGNCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"LEFT" "BOTTOM"',
    '"LEFT" "CENTER"',
    '"LEFT" "TOP"',
    '"CENTER" "BOTTOM"',
    '"CENTER" "CENTER"',
    '"CENTER" "TOP"',
    '"RIGHT" "BOTTOM"',
    '"RIGHT" "CENTER"',
    '"RIGHT" "TOP"'
)
def assertValidLayoutAlign(fieldName, value):
    """
    Utility function to assert type validity of layoutAlignChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in LAYOUTALIGNCHOICES:
        # print('*** assertValidLayoutAlign ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in LAYOUTALIGNCHOICES:
        # print('*** assertValidLayoutAlign ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in LAYOUTALIGNCHOICES=' + str(LAYOUTALIGNCHOICES))

LAYOUTSCALEMODECHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"NONE" "NONE"',
    '"NONE" "FRACTION"',
    '"NONE" "STRETCH"',
    '"NONE" "PIXEL"',
    '"FRACTION" "NONE"',
    '"FRACTION" "FRACTION"',
    '"FRACTION" "STRETCH"',
    '"FRACTION" "PIXEL"',
    '"STRETCH" "NONE"',
    '"STRETCH" "FRACTION"',
    '"STRETCH" "STRETCH"',
    '"STRETCH" "PIXEL"',
    '"PIXEL" "NONE"',
    '"PIXEL" "FRACTION"',
    '"PIXEL" "STRETCH"',
    '"PIXEL" "PIXEL"'
)
def assertValidLayoutScaleMode(fieldName, value):
    """
    Utility function to assert type validity of layoutScaleModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in LAYOUTSCALEMODECHOICES:
        # print('*** assertValidLayoutScaleMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in LAYOUTSCALEMODECHOICES:
        # print('*** assertValidLayoutScaleMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in LAYOUTSCALEMODECHOICES=' + str(LAYOUTSCALEMODECHOICES))

LAYOUTUNITSCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"WORLD" "WORLD"',
    '"WORLD" "FRACTION"',
    '"WORLD" "PIXEL"',
    '"FRACTION" "WORLD"',
    '"FRACTION" "FRACTION"',
    '"FRACTION" "PIXEL"',
    '"PIXEL" "WORLD"',
    '"PIXEL" "FRACTION"',
    '"PIXEL" "PIXEL"'
)
def assertValidLayoutUnits(fieldName, value):
    """
    Utility function to assert type validity of layoutUnitsChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in LAYOUTUNITSCHOICES:
        # print('*** assertValidLayoutUnits ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in LAYOUTUNITSCHOICES:
        # print('*** assertValidLayoutUnits ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in LAYOUTUNITSCHOICES=' + str(LAYOUTUNITSCHOICES))

METADIRECTIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'rtl', # right-to-left
    'ltr' # left-to-right
)
def assertValidMetaDirection(fieldName, value):
    """
    Utility function to assert type validity of metaDirectionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in METADIRECTIONCHOICES:
        # print('*** assertValidMetaDirection ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in METADIRECTIONCHOICES:
        # print('*** assertValidMetaDirection ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in METADIRECTIONCHOICES=' + str(METADIRECTIONCHOICES))

METANAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'accessRights', # permission required to access resource or security status
    'author', # name of individual author
    'contributor', # name of individual contributing to this resource
    'created', # date of initial version
    'creator', # name of original author
    'description', # summary overview describing this resource
    'disclaimer', # statement of denial or disavowal regarding potential claims or responsiblity
    'drawing', # name or reference link to a supporting drawing or sketch file
    'error', # information about an error (or known problem) that can prevent proper operation
    'generator', # authoring tool or translation tool
    'hint', # user hint about resource features or operation
    'identifier', # url address or unique Uniform Resource Identifier (URI) for resource
    'Image', # name or reference link to supporting image file
    'info', # additional info of interest
    'information', # additional information of interest
    'isVersionOf', # Related resource of which the described resource is a version, edition, or adaptation.
    'keywords', # comma-separated tokens, each of which is a keyword of interest
    'license', # content or software license
    'mediator', # entity that mediates access to resource and for whom resource is intended or useful
    'modified', # date of modified version
    'movie', # name or reference link to supporting movie file (note that Dublin Core term is MovingImage)
    'MovingImage', # name or reference link to supporting movie
    'original', # name or reference link to original file or resource
    'photo', # name or reference link to supporting photo file (note that Dublin Core term is Image)
    'photograph', # name or reference link to supporting photograph file (note that Dublin Core term is Image)
    'publisher', # entity responsible for making the resource available
    'reference', # name or reference link to supporting reference
    'requires', # prerequisites for operation or viewing
    'rights', # intellectual property rights (IPR)
    'robots', # search engine and web-spider guidance value: noindex to block page indexing, nofollow to block following links
    'Sound', # name or reference link to supporting sound file
    'source', # related resource from which the described resource is derived
    'specificationSection', # title of relevant specification section
    'specificationUrl', # url for relevant specification section
    'subject', # search-index subject keywords, key phrases, or classification codes
    'Text', # resource consisting primarily of words for reading
    'title', # file name for this resource
    'TODO', # action item "to do" that still needs to be performed
    'translator', # name of person performing translation from another format or language
    'translated', # date of translation from another format or language
    'version', # current version number or ID of this resource
    'warning' # warning information about a known problem that impedes proper operation
)

MULTITEXTUREFUNCTIONVALUES = (
    # specification-defined values follow, other values are also allowed
    '"COMPLEMENT"', # Invert argument x as (1 - x)
    '"ALPHAREPLICATE"', # Replicate alpha information to all color channels before operation completes.
    '""' # No function is applied - empty SFString is allowed value within MFString array
)

MULTITEXTUREMODEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ADD"',
    '"ADDSIGNED"',
    '"ADDSIGNED2X"',
    '"ADDSMOOTH"',
    '"BLENDCURRENTALPHA"',
    '"BLENDDIFFUSEALPHA"',
    '"BLENDFACTORALPHA"',
    '"BLENDTEXTUREALPHA"',
    '"DOTPRODUCT3"',
    '"MODULATE"',
    '"MODULATE2X"',
    '"MODULATE4X"',
    '"MODULATEALPHA_ADDCOLOR"',
    '"MODULATEINVALPHA_ADDCOLOR"',
    '"MODULATEINVCOLOR_ADDALPHA"',
    '"OFF"',
    '"REPLACE"',
    '"SELECTARG1"',
    '"SELECTARG2"',
    '"SUBTRACT"'
)

MULTITEXTURESOURCEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"DIFFUSE"',
    '"FACTOR"',
    '"SPECULAR"',
    '""'
)

NAVIGATIONTRANSITIONTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"TELEPORT"', # immediate transition
    '"LINEAR"', # transition may proceed directly through intervening objects
    '"ANIMATE"' # rowser-specific transition
)

NAVIGATIONTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ANY"', # browser can offer any type for user to choose
    '"WALK"', # free navigation, avatar remains on ground, collision detection
    '"EXAMINE"', # view an individual object by rotating view about center
    '"FLY"', # free navigation, collision detection
    '"LOOKAT"', # navigate to particular object
    '"NONE"', # disables all navigation interfaces
    '"EXPLORE"' # consistent keystroke navigation for both geospatial and Cartesian modes
)

NETWORKMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'standAlone', # ignore network but still respond to events in local scene
    'networkReader', # listen to network and read PDU packets at readInterval, act as remotely linked copy of entity
    'networkWriter' # send PDU packets to network at writeInterval, act as master entity
)
def assertValidNetworkMode(fieldName, value):
    """
    Utility function to assert type validity of networkModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in NETWORKMODECHOICES:
        # print('*** assertValidNetworkMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in NETWORKMODECHOICES:
        # print('*** assertValidNetworkMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in NETWORKMODECHOICES=' + str(NETWORKMODECHOICES))

PARTICLESYSTEMGEOMETRYTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    'LINE', # line is drawn along current velocity vector of particle
    'POINT', # point geometry is rendered at particle position
    'QUAD', # quad geometry is rendered at particle position facing direction traveled
    'SPRITE', # quad geometry is rendered at particle position facing screen
    'TRIANGLE', # pair of triangles creating quad geometry is rendered at particle position facing direction traveled
    'GEOMETRY' # geometry field is used for rendering each particle
)

PERIODICWAVETYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'sine',
    'square',
    'sawtooth',
    'triangle',
    'custom'
)
def assertValidPeriodicWaveType(fieldName, value):
    """
    Utility function to assert type validity of periodicWaveTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in PERIODICWAVETYPECHOICES:
        # print('*** assertValidPeriodicWaveType ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in PERIODICWAVETYPECHOICES:
        # print('*** assertValidPeriodicWaveType ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in PERIODICWAVETYPECHOICES=' + str(PERIODICWAVETYPECHOICES))

PHASEFUNCTIONVALUES = (
    # specification-defined values follow, other values are also allowed
    'Henyey-Greenstein', # Henyey-Greenstein phase function for scattering model
    'NONE' # no scattering
)

PICKABLEOBJECTTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ALL"', # each node is available for picking
    '"NONE"', # no node is available for picking
    '"TERRAIN"' # TERRAIN is an example value
)

PICKSENSORMATCHCRITERIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'MATCH_ANY', # any match of objectType values is acceptable
    'MATCH_EVERY', # every objectType value in X3DPickSensorNode and X3DPickableObject shall match
    'MATCH_ONLY_ONE' # one and only one objectType value can match
)
def assertValidPickSensorMatchCriterion(fieldName, value):
    """
    Utility function to assert type validity of pickSensorMatchCriterionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in PICKSENSORMATCHCRITERIONCHOICES:
        # print('*** assertValidPickSensorMatchCriterion ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in PICKSENSORMATCHCRITERIONCHOICES:
        # print('*** assertValidPickSensorMatchCriterion ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in PICKSENSORMATCHCRITERIONCHOICES=' + str(PICKSENSORMATCHCRITERIONCHOICES))

PICKSENSORSORTORDERVALUES = (
    # specification-defined values follow, other values are also allowed
    'ANY', # any single object that can satisfy picking conditions
    'CLOSEST', # return closest object by distance that satisfies conditions of this pick sensor
    'ALL', # every object that satisfies picking conditions for this pick sensor is returned
    'ALL_SORTED' # every object that satisfies picking conditions for this pick sensor is returned, in sorted order
)

PROFILENAMECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'Core', # Core Profile includes no nodes and is provided as the basis for custom componentization. Allowed X3D statements for all profiles are: connect ExternProtoDeclare EXPORT field fieldValue IMPORT IS ProtoBody ProtoDeclare ProtoInterface ProtoInstance ROUTE X3D. Allowed X3D nodes for this profile are: MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString.
    'Interchange', # Interchange Profile equals the minimum subset of nodes needed to display lightweight compelling content. Allowed X3D nodes for this profile are: Appearance Background Box Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder DirectionalLight Group ImageTexture IndexedFaceSet IndexedLineSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator PixelTexture PointSet PositionInterpolator ScalarInterpolator Shape Sphere TextureCoordinate TextureCoordinateGenerator TextureTransform TimeSensor Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint WorldInfo.
    'CADInterchange', # CADInterchange Profile adds support for CADGeometry component nodes to Interchange Profile. Allowed X3D nodes for this profile are: Anchor Appearance CADAssembly CADFace CADLayer CADPart Billboard Collision Color ColorRGBA Coordinate DirectionalLight FragmentShader Group ImageTexture IndexedLineSet IndexedQuadSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet Inline LineProperties LineSet LOD Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiShader MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal PixelTexture PointSet QuadSet Shader ShaderAppearance Shape TextureCoordinate TextureCoordinateGenerator TextureTransform Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint VertexShader WorldInfo.
    'Interactive', # Interactive Profile adds interaction nodes (Anchor, KeySensor) to the minimum subset of nodes needed to display lightweight compelling content. Allowed X3D nodes for this profile are: Anchor Appearance Background BooleanFilter BooleanSequencer BooleanToggle BooleanTrigger Box Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder CylinderSensor DirectionalLight ElevationGrid Group ImageTexture IndexedFaceSet IndexedLineSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger KeySensor LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger KeySensor LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator PixelTexture PlaneSensor PointLight PointSet PositionInterpolator ProximitySensor ScalarInterpolator Shape Sphere SphereSensor SpotLight StringSensor Switch TextureCoordinate TextureCoordinateGenerator TextureTransform TimeSensor TimeTrigger TouchSensor Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint VisibilitySensor WorldInfo.
    'Immersive', # Immersive Profile equals all of the nodes in the VRML97 Specification, plus various X3D node additions including KeySensor, StringSensor and Scene. Allowed X3D nodes for this profile are: Anchor Appearance AudioClip Background Billboard BooleanFilter BooleanSequencer BooleanToggle BooleanTrigger Box Collision Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder CylinderSensor DirectionalLight ElevationGrid Extrusion Fog FontStyle Group ImageTexture IndexedFaceSet IndexedLineSet IndexedTriangleFan IndexedTriangleSet IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger KeySensor LineProperties LineSet LoadSensor LOD Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MovieTexture MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator PixelTexture PlaneSensor PointLight PointSet Polyline2D Polypoint2D PositionInterpolator ProximitySensor Rectangle2D ScalarInterpolator Script Shape Sound Sphere SphereSensor SpotLight StringSensor Switch Text TextureCoordinate TextureCoordinateGenerator TextureTransform TimeSensor TimeTrigger TouchSensor TriangleFanSet TriangleSet TriangleSet2D TriangleStripSet Transform Viewpoint VisibilitySensor WorldInfo.
    'MedicalInterchange', # The MedicalInterchange profile adds support for VolumeRendering component to Interchange profile. Allowed X3D nodes for this profile are: Anchor Arc2D ArcClose2D Appearance Background Billboard BlendedVolumeStyle BooleanFilter BooleanSequencer BooleanToggle BooleanTrigger BoundaryEnhancementVolumeStyle Box CartoonVolumeStyle Circle2D ClipPlane Collision Color ColorInterpolator ColorRGBA ComposedVolumeStyle CompositeTexture3D Cone Coordinate CoordinateDouble CoordinateInterpolator Cylinder DirectionalLight Disk2D EdgeEnhancementVolumeStyle FillProperties FontStyle Group ImageTexture ImageTexture3D IndexedFaceSet IndexedLineSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger IsoSurfaceVolumeData LineProperties LineSet LOD Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OctTree OpacityMapVolumeStyle OrientationInterpolator OrthoViewpoint PixelTexture PixelTexture3D PointSet Polyline2D Polypoint2D PositionInterpolator ProjectionVolumeStyle Rectangle2D ScalarInterpolator SegmentedVolumeData ShadedVolumeStyle Shape SilhouetteEnhancementVolumeStyle Sphere StaticGroup Switch Text TextureCoordinate TextureCoordinate3D TextureCoordinate4D TextureCoordinateGenerator TextureMatrixTransform TextureProperties TextureTransform TextureTransform3D TimeSensor TimeTrigger ToneMappedVolumeStyle Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint ViewpointGroup VolumeData WorldInfo.
    'MPEG4Interactive', # MPEGInteractive Profile defines base interoperability with MPEG4 standards to a small subset of nodes needed to display lightweight compelling content. Allowed X3D nodes for this profile are: Anchor Appearance Background Box Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder CylinderSensor DirectionalLight ElevationGrid Group ImageTexture IndexedFaceSet IndexedLineSet Inline LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString NavigationInfo NormalInterpolator OrientationInterpolator PixelTexture PlaneSensor PointLight PointSet PositionInterpolator ProximitySensor ScalarInterpolator Shape Sphere SphereSensor SpotLight Switch TextureCoordinate TextureTransform TimeSensor TouchSensor Transform Viewpoint WorldInfo.
    'Full' # The Full Profile corresponds to all Immersive X3D nodes plus all approved/implemented extensions. All X3D nodes and statements are allowed in this profile.
)
def assertValidProfileName(fieldName, value):
    """
    Utility function to assert type validity of profileNameChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in PROFILENAMECHOICES:
        # print('*** assertValidProfileName ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in PROFILENAMECHOICES:
        # print('*** assertValidProfileName ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in PROFILENAMECHOICES=' + str(PROFILENAMECHOICES))

PROJECTIONVOLUMESTYLETYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'MAX', # Maximum Intensity Projection (MIP) or Least MIP (LMIP) algorithm is used to generate output color
    'MIN', # Minimum Intensity Projection algorithm is used to generate output color
    'AVERAGE' # All voxels along ray are averaged to generate output color
)
def assertValidProjectionVolumeStyleType(fieldName, value):
    """
    Utility function to assert type validity of projectionVolumeStyleTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in PROJECTIONVOLUMESTYLETYPECHOICES:
        # print('*** assertValidProjectionVolumeStyleType ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in PROJECTIONVOLUMESTYLETYPECHOICES:
        # print('*** assertValidProjectionVolumeStyleType ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in PROJECTIONVOLUMESTYLETYPECHOICES=' + str(PROJECTIONVOLUMESTYLETYPECHOICES))

SHADERLANGUAGEVALUES = (
    # specification-defined values follow, other values are also allowed
    'Cg', # nVidia Cg shading language
    'GLSL', # OpenGL shading language (GLSL)
    'HLSL' # Microsoft High Level Shading Language (HLSL)
)

SHADERPARTTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    'VERTEX', # vertex shader
    'FRAGMENT' # fragment shader
)

TEXTUREBOUNDARYMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'CLAMP', # Clamp texture coordinates to range [0,1]
    'CLAMP_TO_EDGE', # Clamp texture coordinates such that a border texel is never sampled
    'CLAMP_TO_BOUNDARY', # Clamp texture coordinates such that texture samples are border texels for fragments
    'MIRRORED_REPEAT', # Texture coordinates are mirrored and then clamped as in CLAMP_TO_EDGE
    'REPEAT' # Repeat a texture across the fragment
)
def assertValidTextureBoundaryMode(fieldName, value):
    """
    Utility function to assert type validity of textureBoundaryModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in TEXTUREBOUNDARYMODECHOICES:
        # print('*** assertValidTextureBoundaryMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in TEXTUREBOUNDARYMODECHOICES:
        # print('*** assertValidTextureBoundaryMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in TEXTUREBOUNDARYMODECHOICES=' + str(TEXTUREBOUNDARYMODECHOICES))

TEXTURECOMPRESSIONMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'DEFAULT', # browser-specified default compression mode
    'FASTEST', # fastest method available
    'HIGH', # greatest amount of compression
    'LOW', # least amount of compression
    'MEDIUM', # moderate amount of compressions
    'NICEST' # highest quality method available
)
def assertValidTextureCompressionMode(fieldName, value):
    """
    Utility function to assert type validity of textureCompressionModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in TEXTURECOMPRESSIONMODECHOICES:
        # print('*** assertValidTextureCompressionMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in TEXTURECOMPRESSIONMODECHOICES:
        # print('*** assertValidTextureCompressionMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in TEXTURECOMPRESSIONMODECHOICES=' + str(TEXTURECOMPRESSIONMODECHOICES))

TEXTURECOORDINATEGENERATORMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'SPHERE', # Creates texture coordinates for a spherical environment
    'CAMERASPACENORMAL', # Use vertex normal, transformed to camera space, as input texture coordinates
    'CAMERASPACEPOSITION', # Use vertex position, transformed to camera space, as input texture coordinates
    'CAMERASPACEREFLECTIONVECTOR', # Use reflection vector, transformed to camera space, as input texture coordinates
    'SPHERE-LOCAL', # Sphere mapping but in local coordinates
    'COORD', # Use vertex coordinates
    'COORD-EYE', # Use vertex coordinates transformed to camera space
    'NOISE', # Apply Perlin solid noise function on vertex coordinates
    'NOISE-EYE', # Apply Perlin solid noise function on vertex coordinates transformed to camera space
    'SPHERE-REFLECT', # similar to CAMERASPACEREFLECTIONVECTOR with optional index of refraction
    'SPHERE-REFLECT-LOCAL' # Similar to SPHERE-REFLECT transformed to camera space
)
def assertValidTextureCoordinateGeneratorMode(fieldName, value):
    """
    Utility function to assert type validity of textureCoordinateGeneratorModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in TEXTURECOORDINATEGENERATORMODECHOICES:
        # print('*** assertValidTextureCoordinateGeneratorMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in TEXTURECOORDINATEGENERATORMODECHOICES:
        # print('*** assertValidTextureCoordinateGeneratorMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in TEXTURECOORDINATEGENERATORMODECHOICES=' + str(TEXTURECOORDINATEGENERATORMODECHOICES))

TEXTUREMAGNIFICATIONMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'AVG_PIXEL', # weighted average of four texture elements closest to center of pixel being textured
    'DEFAULT', # browser-specified default magnification mode
    'FASTEST', # fastest method available
    'NEAREST_PIXEL', # texture element nearest to the center of pixel being textured
    'NICEST' # highest quality method available
)
def assertValidTextureMagnificationMode(fieldName, value):
    """
    Utility function to assert type validity of textureMagnificationModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in TEXTUREMAGNIFICATIONMODECHOICES:
        # print('*** assertValidTextureMagnificationMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in TEXTUREMAGNIFICATIONMODECHOICES:
        # print('*** assertValidTextureMagnificationMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in TEXTUREMAGNIFICATIONMODECHOICES=' + str(TEXTUREMAGNIFICATIONMODECHOICES))

TEXTUREMINIFICATIONMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'AVG_PIXEL', # weighted average of four texture elements closest to center of pixel being textured
    'AVG_PIXEL_AVG_MIPMAP', # tri-linear mipmap filtering
    'AVG_PIXEL_NEAREST_MIPMAP', # choose mipmap that most closely matches size of pixel being textured, use weighted average of four texture elements closest to center of pixel
    'DEFAULT', # browser-specified default minification mode
    'FASTEST', # fastest method available, use mipmaps if possible
    'NEAREST_PIXEL', # texture element nearest to center of pixel being textured
    'NEAREST_PIXEL_AVG_MIPMAP', # texture element nearest to center of pixel being textured, use average of two nearest mipmaps
    'NEAREST_PIXEL_NEAREST_MIPMAP', # texture element nearest to center of pixel being textured, use nearest mipmap
    'NICEST' # highest quality method available
)
def assertValidTextureMinificationMode(fieldName, value):
    """
    Utility function to assert type validity of textureMinificationModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in TEXTUREMINIFICATIONMODECHOICES:
        # print('*** assertValidTextureMinificationMode ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in TEXTUREMINIFICATIONMODECHOICES:
        # print('*** assertValidTextureMinificationMode ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in TEXTUREMINIFICATIONMODECHOICES=' + str(TEXTUREMINIFICATIONMODECHOICES))

UNITCATEGORYCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'angle', # angle default is radians
    'force', # force default is newtons
    'length', # length default is meters
    'mass' # mass default is kilograms
)
def assertValidUnitCategory(fieldName, value):
    """
    Utility function to assert type validity of unitCategoryChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in UNITCATEGORYCHOICES:
        # print('*** assertValidUnitCategory ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in UNITCATEGORYCHOICES:
        # print('*** assertValidUnitCategory ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in UNITCATEGORYCHOICES=' + str(UNITCATEGORYCHOICES))

VOLUMERENDERINGWEIGHTFUNCTIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'CONSTANT', # Use weightConstant1
    'ALPHA1', # Use O_v
    'ALPHA2', # Use O_blend
    'ONE_MINUS_ALPHA1', # Use 1 - O_v
    'ONE_MINUS_ALPHA2', # Use 1 - O_blend
    'TABLE' # Use table lookup value
)
def assertValidVolumeRenderingWeightFunction(fieldName, value):
    """
    Utility function to assert type validity of volumeRenderingWeightFunctionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES:
        # print('*** assertValidVolumeRenderingWeightFunction ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES:
        # print('*** assertValidVolumeRenderingWeightFunction ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES=' + str(VOLUMERENDERINGWEIGHTFUNCTIONCHOICES))

WAVESHAPEROVERSAMPLECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'none', # No oversampling
    '2x', # Double sampling rate
    '4x' # Quadruple sampling rate
)
def assertValidWaveShaperOversample(fieldName, value):
    """
    Utility function to assert type validity of waveShaperOversampleChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in WAVESHAPEROVERSAMPLECHOICES:
        # print('*** assertValidWaveShaperOversample ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in WAVESHAPEROVERSAMPLECHOICES:
        # print('*** assertValidWaveShaperOversample ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in WAVESHAPEROVERSAMPLECHOICES=' + str(WAVESHAPEROVERSAMPLECHOICES))

X3DVERSIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    '3.0', # X3D version 3.0 approved by ISO in 2004.
    '3.1', # X3D version 3.1 Amendment 1 approved by ISO in 2005. Backwards compatibility maintained with version 3.0.
    '3.2', # X3D version 3.2 Amendment 2 approved by ISO in 2007. Backwards compatibility maintained with versions 3.0 and 3.1.
    '3.3', # X3D version 3.3 approved by ISO in 2013 as International Standard (IS). Backwards compatibility maintained with versions 3.0, 3.1 and 3.2.
    '4.0' # X3D version 4.0 under final development by Web3D Consortium. Backwards compatibility maintained with versions 3.0, 3.1, 3.2 and 3.3.
)
def assertValidX3dVersion(fieldName, value):
    """
    Utility function to assert type validity of x3dVersionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if str(MFString(value).XML()) in X3DVERSIONCHOICES:
        # print('*** assertValidX3dVersion ' + fieldName + ' str(MFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    if isinstance(value, (SFString, str)) and str(SFString(value).XML()) in X3DVERSIONCHOICES:
        # print('*** assertValidX3dVersion ' + fieldName + ' str(SFString(' + str(value) + ').XML())=' + str(MFString(value).XML()), flush=True) # diagnostic
        return True
    raise X3DTypeError(fieldName + ' value=\'' + MFString(value).XML() + '\' does not match allowed enumerations in X3DVERSIONCHOICES=' + str(X3DVERSIONCHOICES))

###############################################

# Utility Functions

def metaDiagnostics(self, headElement=None):
    """
    Utility function to return any meta info, hint, warning, error, TODO values in this model.
    """
    if  headElement is None:
        headElement = self
    if  isinstance(headElement, X3D):
        headElement = headElement.head
    # print('type(headElement)=' + str(type(headElement)), flush=True) # diagnostic
    if  isinstance(headElement, head):
        result = "meta "
        for each in headElement.children:
            if isinstance(each, meta) and each.name in ('info', 'hint', 'warning', 'error', 'TODO'):
                result += each.name.strip() + ': ' + each.content.strip()
                if  result.endswith('.') or result.endswith(','):
                    result += ' '
                else:
                    result += ', '
        if  result.strip() != "meta":
            return result.rstrip(', ').strip()
    return ''

def prependLineNumbers(someText="", lineNumber=0, blockInterval = 3):
    """
    Utility function to prepend line numbers to block of text, can optionally define lineNumber and blockInterval.
    """
    # https://stackoverflow.com/questions/64621076/how-to-add-line-numbers-to-multiline-string
    result = ''
    count = 1
    splitLines = someText.splitlines()
    if not lineNumber:
        lineNumber = 0
    if lineNumber == 0:
        first = 0
        last  = len(splitLines)
    else:
        first = max(lineNumber - blockInterval - 1, 0)
        last  = min(lineNumber + blockInterval,     len(splitLines))
        print("*** issue in line", str(lineNumber), "of", str(len(splitLines)), "***") # "first=" + str(first), "last=" + str(last) +
    for line in splitLines[first:last]:
        result += ("{}{:2d}{}{}".format("[",first + count,"] ",line)) + '\n'
        count += 1
    return result

###############################################

# Field Validation Functions

# Type-specific functions to check for valid values, throw exception if illegal value is found

def fixBoolean(value, default=None):
    """
    Utility function to convert boolean to corresponding Python value.
    """
    # if _DEBUG: print('fixBoolean(value=' + str(value) + ', type=' + str(type(value)) + ', default=' + str(default) + ')', flush=True)
    if  value is None:
        # if _DEBUG: print('...DEBUG... fixBoolean set value to default=' + str(default))
        return default
#   print('fixBoolean #0a')
#    if isinstance(value, MFBool):
#        value = value.value # dereference
#       print('fixBoolean #0a dereferenced')
#   print('fixBoolean #0b')
    if isinstance(value, list) and len(value) == 0:
        return value
#   print('fixBoolean #1') # debug
    if isinstance(value, list) and len(value) == 1:
        # if _DEBUG: print('fixBoolean downcasting by resetting singleton list value=' + str(value) + ' as value=' + str(value[0]), flush=True)
        return value[0] # dereference
    if isinstance(value, SFBool):
        return value.value # dereference
    if isinstance(value, MFBool) and len(value) == 1:
        return value.value[0] # dereference
#   print('fixBoolean #2') # debug
    if value in ('true', 'True'):
        return True
    if value in ('false', 'False'):
        return False
#   print('fixBoolean #3') # debug
    if isinstance(value, bool):
        return value
#   print('fixBoolean #4') # debug
    if isinstance(value, MFBool):
        value = value.value # dereference value from base type
#       print('fixBoolean #4a, dereference MFBool') # debug
    if isinstance(value, list):
#       print('fixBoolean #4b found list, value=' + str(value), ' length=' + str(len(value))) # debug
        index = 0
        result = value
        for each in value:
            if  each in ('true', 'True'):
                result[index] = True
            elif each in ('false', 'False'):
                result[index] = False
            while isinstance(each, list) and len(each) == 1:
                # if _DEBUG: print('fixBoolean downcasting by extracting singleton list value[' + str(index) + ']=' + str(each) + ' as each[0]=' + str(each[0]), flush=True)
                result[index] = each[0]
#           print('fixBoolean #4c found each=' + str(each), 'result=' + str(result), flush=True) # debug
            if not isinstance(result[index], bool):
                # print(flush=True)
                raise X3DTypeError('fixBoolean(value=' + str(value) + ') MFBool value[' + str(index) + ']=' + str(each) + ', result[' + str(index) + ']=' + str(result[index]) + ' with type=' + str(type(value)) + ' is not a valid Python bool expression')
            index += 1
        # if _DEBUG: print('...DEBUG...fixBoolean result=' + str(result), flush=True)
        return result
#   print('fixBoolean #5, unhandled case: value=' + str(value), ' length=' + str(len(value))) # debug
    print(flush=True)
    raise X3DTypeError('fixBoolean(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python bool')

def isPositive(value):
    """
    Utility function to confirm positive value(s) greater than or equal to zero.
    """
    if  value is None:
        return None
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element <= 0:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each <= 0:
                return False
        return True
    if isinstance(value, (int, float)):
        return value > 0
    raise X3DTypeError('isPositive(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertPositive(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isPositive(value).
    """
    # if _DEBUG: print('...DEBUG... assertPositive(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isPositive(value), str(fieldName) + '=' + str(value) + ' fails assertPositive requirements: value(s) must be greater than or equal to zero'

def isNonNegative(value):
    """
    Utility function to confirm nonnegative value(s) greater than or equal to zero.
    """
    if  value is None:
        return None
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element < 0:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        # if _DEBUG: print('isNonNegative: ', value, flush=True)
        for each in value:
            if each < 0:
                return False
        return True
    if isinstance(value, (int, float)):
        return value >= 0
    raise X3DTypeError('isNonNegative(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertNonNegative(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isNonNegative(value).
    """
    # if _DEBUG: print('...DEBUG... assertNonNegative(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isNonNegative(value), str(fieldName) + '=' + str(value) + ' fails assertNonNegative requirements: value(s) must be greater than or equal to zero'

def isZeroToOne(value):
    """
    Utility function to confirm value(s) in range [0..1]
    """
    # if _DEBUG: print('...DEBUG... isZeroToOne(' + str(value) + ')', flush=True)
    if  value is None:
        return None
    if isinstance(value,_X3DField):
        value = value.value # dereference
    if isinstance(value, (list, tuple)):
        for each in value:
            if isinstance(each, (list, tuple)):
                for another in each:
                    if not 0 <= another <= 1:
                        return False
            elif not 0 <= each <= 1:
                return False
        return True
    if isinstance(value, (int, float)):
        return 0 <= value <= 1
    raise X3DTypeError('isZeroToOne(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertZeroToOne(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isZeroToOne(value)
    """
    # if _DEBUG: print('...DEBUG... assertZeroToOne(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isZeroToOne(value), str(fieldName) + '=' + str(value) + ' fails assertZeroToOne requirements: value(s) must be in range [0..1]'

def isLessThanEquals(value, maximum):
    """
    Utility function to confirm value(s) less than or equal to maximum.
    """
    # if True or _DEBUG: print('* debug: isLessThanEquals(' + str(value) + ')', flush=True)
    if  value is None:
        return None
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element > maximum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each > maximum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value <= maximum
    raise X3DTypeError('isLessThanEquals(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertLessThanEquals(fieldName, value, maximum):
    """
    Utility function to raise X3DTypeError if not isLessThanEquals(value)
    """
    # if _DEBUG: print('...DEBUG... assertLessThanEquals(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isLessThanEquals(value, maximum), fieldName + '=' + str(value) + ' fails assertLessThanEquals maximum=' + str(maximum)

def isLessThan(value, maximum):
    """
    Utility function to confirm value(s) less than maximum.
    """
    # if True or _DEBUG: print('* debug: isLessThan(' + str(value) + ')', flush=True)
    if  value is None:
        return None
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element >= maximum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each >= maximum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value < maximum
    raise X3DTypeError('isLessThan(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertLessThan(fieldName, value, maximum):
    """
    Utility function to raise X3DTypeError if not isLessThan(value)
    """
    # if _DEBUG: print('...DEBUG... assertLessThan(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isLessThan(value, maximum), str(fieldName) + '=' + str(value) + ' fails assertLessThan maximum=' + str(maximum)

######
def isGreaterThanEquals(value, minimum):
    """
    Utility function to confirm value(s) less than or equal to minimum.
    """
    # if True or _DEBUG: print('* debug: isGreaterThanEquals(' + str(value) + ')', flush=True)
    if  value is None:
        return None
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element < minimum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each < minimum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value >= minimum
    raise X3DTypeError('isGreaterThanEquals(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertGreaterThanEquals(fieldName, value, minimum):
    """
    Utility function to raise X3DTypeError if not isGreaterThanEquals(value)
    """
    # if _DEBUG: print('...DEBUG... assertGreaterThanEquals(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isGreaterThanEquals(value, minimum), str(fieldName) + '=' + str(value) + ' fails assertGreaterThanEquals minimum=' + str(minimum)

def isGreaterThan(value, minimum):
    """
    Utility function to confirm value(s) less than minimum.
    """
    # if True or _DEBUG: print('* debug: isGreaterThan(' + str(value) + ')', flush=True)
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element <= minimum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each <= minimum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value > minimum
    raise X3DTypeError('isGreaterThan(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python number')

def assertGreaterThan(fieldName, value, minimum):
    """
    Utility function to raise X3DTypeError if not isGreaterThan(value)
    """
    # if _DEBUG: print('...DEBUG... assertGreaterThan(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isGreaterThan(value, minimum), str(fieldName) + '=' + str(value) + ' fails assertGreaterThan minimum=' + str(minimum)

def isBoundingBox(value):
    """
    Utility function to confirm legal X3D bounding box value of (-1 -1 -1) or nonnegative triple.
    """
    if value is None:
        return None
    # if True or _DEBUG: print('* debug: isBoundingBox(' + str(value) + ')', 'isinstance(value, tuple)=' + str(isinstance(value, tuple)), 'len(value)=' + str(len(value)), flush=True)
    if isinstance(value, (list, tuple)):
        if len(value) != 3:
            return False
        if value[0] == -1 and value[1] == -1 and value[2] == -1:
            return True
        return isNonNegative(value) # legal bounding box tuple
    raise X3DTypeError('isBoundingBox(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python numeric triple')

def assertBoundingBox(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isBoundingBox(value)
    """
    # if True or _DEBUG: print('* debug: assertBoundingBox(' + str(fieldName) + ', ' + str(value) + ')', flush=True)
    assert isBoundingBox(value), str(fieldName) + '=' + str(value) + ' fails assertBoundingBox requirements: must be (-1, -1, -1) or non-negative 3-tuple'

def isValidSFBool(value):
    """
    Utility function to determine type validity of a SFBool value.
    """
    if re.fullmatch(SFBool().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFBool().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFBool(value)
        return True
    except ValueError:
        print('isValidSFBool failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFBool) and not isinstance(value, MFBool):
    ###         # if _DEBUG: print('SFBool type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFBool)=' + str(isinstance(value, SFBool)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFBool):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFBool().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFBool) and (len(value) == 1) and isValidMFBool(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, bool):
    ###     return False
    ### return True

def assertValidSFBool(value):
    """
    Utility function to assert type validity of a SFBool value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFBool(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFBool') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFBool) and not isinstance(value, MFBool):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFBool')
    ### if isinstance(value, SFBool):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFBool) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    ### if not isinstance(value, bool):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid bool value (True or False) for SFBool')
    ### if not isValidSFBool(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python bool value (True or False) for SFBool')
    return True

def isValidMFBool(value):
    """
    Utility function to determine type validity of a MFBool value.
    """
    if re.fullmatch(MFBool().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFBool().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFBool(value)
        return True
    except ValueError:
        print('isValidMFBool failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFBool) and not isinstance(value, MFBool):
    ###         # if _DEBUG: print('MFBool type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFBool)=' + str(isinstance(value, MFBool)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFBool):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFBool().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFBool):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFBool):
    ###         each = each.value # dereference
    ###     if not isinstance(each, bool):
    ###         return False
    ### return True

def assertValidMFBool(value):
    """
    Utility function to assert type validity of a MFBool value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFBool(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFBool') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFBool) and not isinstance(value, MFBool):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFBool')
    ### if isinstance(value, MFBool):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFBool)) and not isinstance(value, list):
        ### value = list(SFBool(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFBool')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFBool):
    ###         each = each.value # dereference
    ###     if not isinstance(each, bool):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFBool list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid bool')
    ### if not isValidMFBool(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFBool')
    return True

def isValidSFColor(value):
    """
    Utility function to determine type validity of a SFColor value.
    """
    if re.fullmatch(SFColor().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFColor().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFColor(value)
        return True
    except ValueError:
        print('isValidSFColor failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFColor) and not isinstance(value, MFColor):
    ###         # if _DEBUG: print('SFColor type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFColor)=' + str(isinstance(value, SFColor)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFColor):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFColor().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFColor) and (len(value) == 1) and isValidMFColor(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFColor):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ###     if (each < 0) or (each > 1):
    ###         return False
    ### if tupleCount != 3:
    ###     return False
    ### if not isZeroToOne(value):
    ###     return False
    ### return True

def assertValidSFColor(value):
    """
    Utility function to assert type validity of a SFColor value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFColor(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFColor') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFColor) and not isinstance(value, MFColor):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFColor')
    ### if isinstance(value, SFColor):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFColor(value)
    ###     print('found string for SFColor, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFColor) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFColor encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFColor')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColor')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFColor):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFColor list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ###     if (each < 0) or (each > 1):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFColor' + str(value)[:100] + ' has value ' + str(each) + ' with type=' + str(type(value)) + ' is  out of range [0..1] and is not a valid SFColor')
    ### if tupleCount != 3:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFColor ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')

        ###                 assertZeroToOne('SFColor', value)
    ### if not isValidSFColor(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColor')
    return True

def isValidMFColor(value):
    """
    Utility function to determine type validity of a MFColor value.
    """
    if re.fullmatch(MFColor().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFColor().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFColor(value)
        return True
    except ValueError:
        print('isValidMFColor failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFColor) and not isinstance(value, MFColor):
    ###         # if _DEBUG: print('MFColor type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFColor)=' + str(isinstance(value, MFColor)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFColor):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFColor().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFColor):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFColor().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFColor tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColor().TUPLE_SIZE() =' + str(MFColor().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ###         if (element < 0) or (element > 1):
    ###             return False
    ### if not isZeroToOne(value):
    ###     return False
    ### return True

def assertValidMFColor(value):
    """
    Utility function to assert type validity of a MFColor value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFColor(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFColor') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFColor) and not isinstance(value, MFColor):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFColor')
    ### if isinstance(value, MFColor):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFColor(value)
    ###     print('found string for MFColor, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFColor)) and not isinstance(value, list):
        ### value = list(SFColor(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColor')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFColor().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFColor tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColor().TUPLE_SIZE() =' + str(MFColor().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFColor)):
#                # print(flush=True)
#                raise X3DTypeError('MFColor element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFColor element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
        ###             if (element < 0) or (element > 1):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFColor' + str(value)[:100] + ' has value ' + str(element) + ' with type=' + str(type(value)) + ' out of range [0..1] and is not a valid MFColor')

        ###                 assertZeroToOne('MFColor', value)
    ### if not isValidMFColor(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColor')
    return True

def isValidSFColorRGBA(value):
    """
    Utility function to determine type validity of a SFColorRGBA value.
    """
    if re.fullmatch(SFColorRGBA().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFColorRGBA().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFColorRGBA(value)
        return True
    except ValueError:
        print('isValidSFColorRGBA failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
    ###         # if _DEBUG: print('SFColorRGBA type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFColorRGBA)=' + str(isinstance(value, SFColorRGBA)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFColorRGBA):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFColorRGBA().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFColorRGBA) and (len(value) == 1) and isValidMFColorRGBA(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFColorRGBA):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ###     if (each < 0) or (each > 1):
    ###         return False
    ### if tupleCount != 4:
    ###     return False
    ### if not isZeroToOne(value):
    ###     return False
    ### return True

def assertValidSFColorRGBA(value):
    """
    Utility function to assert type validity of a SFColorRGBA value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFColorRGBA(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFColorRGBA') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFColorRGBA')
    ### if isinstance(value, SFColorRGBA):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFColorRGBA(value)
    ###     print('found string for SFColorRGBA, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFColorRGBA) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFColorRGBA encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFColorRGBA')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColorRGBA')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFColorRGBA):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFColorRGBA list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ###     if (each < 0) or (each > 1):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFColorRGBA' + str(value)[:100] + ' has value ' + str(each) + ' with type=' + str(type(value)) + ' is  out of range [0..1] and is not a valid SFColorRGBA')
    ### if tupleCount != 4:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFColorRGBA ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')

        ###                 assertZeroToOne('SFColorRGBA', value)
    ### if not isValidSFColorRGBA(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColorRGBA')
    return True

def isValidMFColorRGBA(value):
    """
    Utility function to determine type validity of a MFColorRGBA value.
    """
    if re.fullmatch(MFColorRGBA().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFColorRGBA().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFColorRGBA(value)
        return True
    except ValueError:
        print('isValidMFColorRGBA failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
    ###         # if _DEBUG: print('MFColorRGBA type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFColorRGBA)=' + str(isinstance(value, MFColorRGBA)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFColorRGBA):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFColorRGBA().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFColorRGBA):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFColorRGBA().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFColorRGBA tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColorRGBA().TUPLE_SIZE() =' + str(MFColorRGBA().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ###         if (element < 0) or (element > 1):
    ###             return False
    ### if not isZeroToOne(value):
    ###     return False
    ### return True

def assertValidMFColorRGBA(value):
    """
    Utility function to assert type validity of a MFColorRGBA value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFColorRGBA(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFColorRGBA') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFColorRGBA')
    ### if isinstance(value, MFColorRGBA):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFColorRGBA(value)
    ###     print('found string for MFColorRGBA, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFColorRGBA)) and not isinstance(value, list):
        ### value = list(SFColorRGBA(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColorRGBA')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFColorRGBA().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFColorRGBA tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColorRGBA().TUPLE_SIZE() =' + str(MFColorRGBA().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFColorRGBA)):
#                # print(flush=True)
#                raise X3DTypeError('MFColorRGBA element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFColorRGBA element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
        ###             if (element < 0) or (element > 1):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFColorRGBA' + str(value)[:100] + ' has value ' + str(element) + ' with type=' + str(type(value)) + ' out of range [0..1] and is not a valid MFColorRGBA')

        ###                 assertZeroToOne('MFColorRGBA', value)
    ### if not isValidMFColorRGBA(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColorRGBA')
    return True

def isValidSFDouble(value):
    """
    Utility function to determine type validity of a SFDouble value.
    """
    if re.fullmatch(SFDouble().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFDouble().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFDouble(value)
        return True
    except ValueError:
        print('isValidSFDouble failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
    ###         # if _DEBUG: print('SFDouble type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFDouble)=' + str(isinstance(value, SFDouble)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFDouble):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFDouble().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFDouble) and (len(value) == 1) and isValidMFDouble(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, float) and not isinstance(value, int):
    ###     return False
    ### return True

def assertValidSFDouble(value):
    """
    Utility function to assert type validity of a SFDouble value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFDouble(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFDouble') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFDouble')
    ### if isinstance(value, SFDouble):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFDouble(value)
    ###     print('found string for SFDouble, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFDouble) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFDouble encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFDouble')
    ### if not isinstance(value, float) and not isinstance(value, int):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFDouble')
    ### if not isValidSFDouble(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFDouble')
    return True

def isValidMFDouble(value):
    """
    Utility function to determine type validity of a MFDouble value.
    """
    if re.fullmatch(MFDouble().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFDouble().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFDouble(value)
        return True
    except ValueError:
        print('isValidMFDouble failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
    ###         # if _DEBUG: print('MFDouble type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFDouble)=' + str(isinstance(value, MFDouble)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFDouble):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFDouble().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFDouble):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFDouble):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### return True

def assertValidMFDouble(value):
    """
    Utility function to assert type validity of a MFDouble value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFDouble(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFDouble') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFDouble')
    ### if isinstance(value, MFDouble):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFDouble(value)
    ###     print('found string for MFDouble, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFDouble)) and not isinstance(value, list):
        ### value = list(SFDouble(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFDouble')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFDouble):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFDouble list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if not isValidMFDouble(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFDouble')
    return True

def isValidSFFloat(value):
    """
    Utility function to determine type validity of a SFFloat value.
    """
    if re.fullmatch(SFFloat().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFFloat().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFFloat(value)
        return True
    except ValueError:
        print('isValidSFFloat failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
    ###         # if _DEBUG: print('SFFloat type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFFloat)=' + str(isinstance(value, SFFloat)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFFloat):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFFloat().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFFloat) and (len(value) == 1) and isValidMFFloat(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, float) and not isinstance(value, int):
    ###     return False
    ### return True

def assertValidSFFloat(value):
    """
    Utility function to assert type validity of a SFFloat value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFFloat(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFFloat') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFFloat')
    ### if isinstance(value, SFFloat):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFFloat(value)
    ###     print('found string for SFFloat, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFFloat) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFFloat encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFFloat')
    ### if not isinstance(value, float) and not isinstance(value, int):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFFloat')
    ### if not isValidSFFloat(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFFloat')
    return True

def isValidMFFloat(value):
    """
    Utility function to determine type validity of a MFFloat value.
    """
    if re.fullmatch(MFFloat().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFFloat().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFFloat(value)
        return True
    except ValueError:
        print('isValidMFFloat failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
    ###         # if _DEBUG: print('MFFloat type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFFloat)=' + str(isinstance(value, MFFloat)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFFloat):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFFloat().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFFloat):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFFloat):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### return True

def assertValidMFFloat(value):
    """
    Utility function to assert type validity of a MFFloat value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFFloat(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFFloat') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFFloat')
    ### if isinstance(value, MFFloat):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFFloat(value)
    ###     print('found string for MFFloat, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFFloat)) and not isinstance(value, list):
        ### value = list(SFFloat(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFFloat')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFFloat):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFFloat list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if not isValidMFFloat(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFFloat')
    return True

def isValidSFImage(value):
    """
    Utility function to determine type validity of a SFImage value.
    """
    if re.fullmatch(SFImage().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFImage().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFImage(value)
        return True
    except ValueError:
        print('isValidSFImage failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFImage) and not isinstance(value, MFImage):
    ###         # if _DEBUG: print('SFImage type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFImage)=' + str(isinstance(value, SFImage)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFImage):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFImage().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFImage) and (len(value) == 1) and isValidMFImage(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFImage):
    ###         each = each.value # dereference
    ###     if not isinstance(each, int):
    ###         return False
    ### if len(value) > 0: # SFImage list length checks
    ###     if 0 < len(value) < 3:
    ###         return False # SFImage list must start with width, height and number of components (0..4)
    ###     width = value[0]
    ###     height = value[1]
    ###     # numberComponents = value[2]
    ###     if  len(value) != (width * height) + 3: # assumes each value in array has all component values in single integer
    ###         print('SFImage array length of ' + str(len(value)) + ' does not equal (width=' + str(width)+ ' * height=' + str(height)+ ') + 3) = ' + str(width * height * numberComponents + 3) + ' (numberComponents=' + numberComponents + ')', flush=True)
    ###         return False # SFImage has invalid list length
    ### return True

def assertValidSFImage(value):
    """
    Utility function to assert type validity of a SFImage value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFImage(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFImage') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFImage) and not isinstance(value, MFImage):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFImage')
    ### if isinstance(value, SFImage):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFImage) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFImage encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFImage')
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for SFImage')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFImage):
    ###         each = each.value # dereference
    ###     if not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFImage list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    ### if not isValidSFImage(value):
    ###     # print(flush=True)
    ###     diagnostic = ''
    ###     if len(value) > 0: # SFImage list length checks
    ###         if 0 < len(value) < 3:
    ###             diagnostic = 'SFImage list must start with width, height and number of components (0..4)'
    ###         else:
    ###             width = value[0]
    ###             height = value[1]
    ###             numberComponents = value[2]
    ###             diagnostic = ' array length of ' + str(len(value)) + ' does not equal (width=' + str(width)+ ' * height=' + str(height)+ ') + 3) = ' + str(width * height + 3)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for SFImage' + diagnostic)
    return True

def isValidMFImage(value):
    """
    Utility function to determine type validity of a MFImage value.
    """
    if re.fullmatch(MFImage().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFImage().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFImage(value)
        return True
    except ValueError:
        print('isValidMFImage failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFImage) and not isinstance(value, MFImage):
    ###         # if _DEBUG: print('MFImage type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFImage)=' + str(isinstance(value, MFImage)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFImage):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFImage().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFImage):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFImage):
    ###         each = each.value # dereference
    ###     if not isinstance(each, int):
    ###         return False
    ### return True

def assertValidMFImage(value):
    """
    Utility function to assert type validity of a MFImage value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFImage(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFImage') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFImage) and not isinstance(value, MFImage):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFImage')
    ### if isinstance(value, MFImage):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFImage)) and not isinstance(value, list):
        ### value = list(SFImage(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFImage')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFImage):
    ###         each = each.value # dereference
    ###     if not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFImage list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    ### if not isValidMFImage(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFImage')
    return True

def isValidSFInt32(value):
    """
    Utility function to determine type validity of a SFInt32 value.
    """
    if re.fullmatch(SFInt32().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFInt32().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFInt32(value)
        return True
    except ValueError:
        print('isValidSFInt32 failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
    ###         # if _DEBUG: print('SFInt32 type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFInt32)=' + str(isinstance(value, SFInt32)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFInt32):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFInt32().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFInt32) and (len(value) == 1) and isValidMFInt32(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, int):
    ###     return False
    ### return True

def assertValidSFInt32(value):
    """
    Utility function to assert type validity of a SFInt32 value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFInt32(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFInt32') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFInt32')
    ### if isinstance(value, SFInt32):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     try:
    ###         value = SFInt32(value)
    ###         print('found string for SFInt32, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ###         return True
    ###     except:
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid int value for SFInt32')
    ### elif isinstance(value, MFInt32) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         int(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFInt32 encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid int value for SFInt32')
    ### if not isinstance(value, int):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid int value for SFInt32')
    ### if not isValidSFInt32(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python int value for SFInt32')
    return True

def isValidMFInt32(value):
    """
    Utility function to determine type validity of a MFInt32 value.
    """
    if re.fullmatch(MFInt32().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFInt32().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFInt32(value)
        return True
    except ValueError:
        print('isValidMFInt32 failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
    ###         # if _DEBUG: print('MFInt32 type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFInt32)=' + str(isinstance(value, MFInt32)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFInt32):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFInt32().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFInt32):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFInt32):
    ###         each = each.value # dereference
    ###     if not isinstance(each, int):
    ###         return False
    ### return True

def assertValidMFInt32(value):
    """
    Utility function to assert type validity of a MFInt32 value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFInt32(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFInt32') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFInt32')
    ### if isinstance(value, MFInt32):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     try:
    ###         value = SFInt32(value)
    ###         print('found string for MFInt32, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ###         return True
    ###     except:
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid int value for SFInt32')
    ### elif (isinstance(value, SFInt32)) and not isinstance(value, list):
        ### value = list(SFInt32(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFInt32')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFInt32):
    ###         each = each.value # dereference
    ###     if not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFInt32 list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    ### if not isValidMFInt32(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFInt32')
    return True

def isValidSFMatrix3d(value):
    """
    Utility function to determine type validity of a SFMatrix3d value.
    """
    if re.fullmatch(SFMatrix3d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFMatrix3d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFMatrix3d(value)
        return True
    except ValueError:
        print('isValidSFMatrix3d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
    ###         # if _DEBUG: print('SFMatrix3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix3d)=' + str(isinstance(value, SFMatrix3d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFMatrix3d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFMatrix3d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFMatrix3d) and (len(value) == 1) and isValidMFMatrix3d(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix3d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 9:
    ###     return False
    ### return True

def assertValidSFMatrix3d(value):
    """
    Utility function to assert type validity of a SFMatrix3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFMatrix3d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFMatrix3d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix3d')
    ### if isinstance(value, SFMatrix3d):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFMatrix3d) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFMatrix3d encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFMatrix3d')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3d')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix3d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFMatrix3d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 9:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFMatrix3d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 9')
    ### if not isValidSFMatrix3d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3d')
    return True

def isValidMFMatrix3d(value):
    """
    Utility function to determine type validity of a MFMatrix3d value.
    """
    if re.fullmatch(MFMatrix3d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFMatrix3d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFMatrix3d(value)
        return True
    except ValueError:
        print('isValidMFMatrix3d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
    ###         # if _DEBUG: print('MFMatrix3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix3d)=' + str(isinstance(value, MFMatrix3d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFMatrix3d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFMatrix3d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFMatrix3d):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFMatrix3d().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFMatrix3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3d().TUPLE_SIZE() =' + str(MFMatrix3d().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFMatrix3d(value):
    """
    Utility function to assert type validity of a MFMatrix3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFMatrix3d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFMatrix3d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix3d')
    ### if isinstance(value, MFMatrix3d):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFMatrix3d)) and not isinstance(value, list):
        ### value = list(SFMatrix3d(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFMatrix3d().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFMatrix3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3d().TUPLE_SIZE() =' + str(MFMatrix3d().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix3d)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix3d element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFMatrix3d element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFMatrix3d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3d')
    return True

def isValidSFMatrix3f(value):
    """
    Utility function to determine type validity of a SFMatrix3f value.
    """
    if re.fullmatch(SFMatrix3f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFMatrix3f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFMatrix3f(value)
        return True
    except ValueError:
        print('isValidSFMatrix3f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
    ###         # if _DEBUG: print('SFMatrix3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix3f)=' + str(isinstance(value, SFMatrix3f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFMatrix3f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFMatrix3f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFMatrix3f) and (len(value) == 1) and isValidMFMatrix3f(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix3f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 9:
    ###     return False
    ### return True

def assertValidSFMatrix3f(value):
    """
    Utility function to assert type validity of a SFMatrix3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFMatrix3f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFMatrix3f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix3f')
    ### if isinstance(value, SFMatrix3f):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFMatrix3f) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFMatrix3f encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFMatrix3f')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3f')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix3f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFMatrix3f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 9:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFMatrix3f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 9')
    ### if not isValidSFMatrix3f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3f')
    return True

def isValidMFMatrix3f(value):
    """
    Utility function to determine type validity of a MFMatrix3f value.
    """
    if re.fullmatch(MFMatrix3f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFMatrix3f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFMatrix3f(value)
        return True
    except ValueError:
        print('isValidMFMatrix3f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
    ###         # if _DEBUG: print('MFMatrix3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix3f)=' + str(isinstance(value, MFMatrix3f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFMatrix3f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFMatrix3f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFMatrix3f):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFMatrix3f().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFMatrix3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3f().TUPLE_SIZE() =' + str(MFMatrix3f().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFMatrix3f(value):
    """
    Utility function to assert type validity of a MFMatrix3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFMatrix3f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFMatrix3f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix3f')
    ### if isinstance(value, MFMatrix3f):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFMatrix3f)) and not isinstance(value, list):
        ### value = list(SFMatrix3f(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFMatrix3f().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFMatrix3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3f().TUPLE_SIZE() =' + str(MFMatrix3f().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix3f)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix3f element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFMatrix3f element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFMatrix3f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3f')
    return True

def isValidSFMatrix4d(value):
    """
    Utility function to determine type validity of a SFMatrix4d value.
    """
    if re.fullmatch(SFMatrix4d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFMatrix4d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFMatrix4d(value)
        return True
    except ValueError:
        print('isValidSFMatrix4d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
    ###         # if _DEBUG: print('SFMatrix4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix4d)=' + str(isinstance(value, SFMatrix4d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFMatrix4d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFMatrix4d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFMatrix4d) and (len(value) == 1) and isValidMFMatrix4d(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix4d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 16:
    ###     return False
    ### return True

def assertValidSFMatrix4d(value):
    """
    Utility function to assert type validity of a SFMatrix4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFMatrix4d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFMatrix4d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix4d')
    ### if isinstance(value, SFMatrix4d):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFMatrix4d) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFMatrix4d encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFMatrix4d')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4d')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix4d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFMatrix4d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 16:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFMatrix4d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 16')
    ### if not isValidSFMatrix4d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4d')
    return True

def isValidMFMatrix4d(value):
    """
    Utility function to determine type validity of a MFMatrix4d value.
    """
    if re.fullmatch(MFMatrix4d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFMatrix4d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFMatrix4d(value)
        return True
    except ValueError:
        print('isValidMFMatrix4d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
    ###         # if _DEBUG: print('MFMatrix4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix4d)=' + str(isinstance(value, MFMatrix4d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFMatrix4d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFMatrix4d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFMatrix4d):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFMatrix4d().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFMatrix4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4d().TUPLE_SIZE() =' + str(MFMatrix4d().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFMatrix4d(value):
    """
    Utility function to assert type validity of a MFMatrix4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFMatrix4d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFMatrix4d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix4d')
    ### if isinstance(value, MFMatrix4d):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFMatrix4d)) and not isinstance(value, list):
        ### value = list(SFMatrix4d(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFMatrix4d().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFMatrix4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4d().TUPLE_SIZE() =' + str(MFMatrix4d().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix4d)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix4d element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFMatrix4d element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFMatrix4d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4d')
    return True

def isValidSFMatrix4f(value):
    """
    Utility function to determine type validity of a SFMatrix4f value.
    """
    if re.fullmatch(SFMatrix4f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFMatrix4f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFMatrix4f(value)
        return True
    except ValueError:
        print('isValidSFMatrix4f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
    ###         # if _DEBUG: print('SFMatrix4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix4f)=' + str(isinstance(value, SFMatrix4f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFMatrix4f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFMatrix4f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFMatrix4f) and (len(value) == 1) and isValidMFMatrix4f(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix4f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 16:
    ###     return False
    ### return True

def assertValidSFMatrix4f(value):
    """
    Utility function to assert type validity of a SFMatrix4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFMatrix4f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFMatrix4f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix4f')
    ### if isinstance(value, SFMatrix4f):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFMatrix4f) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFMatrix4f encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFMatrix4f')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4f')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFMatrix4f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFMatrix4f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 16:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFMatrix4f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 16')
    ### if not isValidSFMatrix4f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4f')
    return True

def isValidMFMatrix4f(value):
    """
    Utility function to determine type validity of a MFMatrix4f value.
    """
    if re.fullmatch(MFMatrix4f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFMatrix4f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFMatrix4f(value)
        return True
    except ValueError:
        print('isValidMFMatrix4f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
    ###         # if _DEBUG: print('MFMatrix4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix4f)=' + str(isinstance(value, MFMatrix4f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFMatrix4f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFMatrix4f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFMatrix4f):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFMatrix4f().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFMatrix4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4f().TUPLE_SIZE() =' + str(MFMatrix4f().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFMatrix4f(value):
    """
    Utility function to assert type validity of a MFMatrix4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFMatrix4f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFMatrix4f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix4f')
    ### if isinstance(value, MFMatrix4f):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFMatrix4f)) and not isinstance(value, list):
        ### value = list(SFMatrix4f(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFMatrix4f().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFMatrix4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4f().TUPLE_SIZE() =' + str(MFMatrix4f().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix4f)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix4f element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFMatrix4f element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFMatrix4f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4f')
    return True

def isValidSFNode(value):
    """
    Utility function to determine type validity of a SFNode value.
    """
    try:
        SFNode(value)
        return True
    except ValueError:
        print('isValidSFNode failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFNode) and not isinstance(value, MFNode):
    ###         # if _DEBUG: print('SFNode type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFNode)=' + str(isinstance(value, SFNode)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFNode):
    ###     value = value.value # dereference value from base type
    ###     return True
    ### if isinstance(value, MFNode) and (len(value) == 1) and isValidMFNode(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, object):
    ###     return False
    ### return True

def assertValidSFNode(value):
    """
    Utility function to assert type validity of a SFNode value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFNode(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFNode') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFNode) and not isinstance(value, MFNode):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFNode')
    ### if isinstance(value, SFNode):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFNode) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    ### if not isinstance(value, object):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid object value for SFNode')
    ### if not isValidSFNode(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python object value for SFNode')
    return True

def isValidMFNode(value):
    """
    Utility function to determine type validity of a MFNode value.
    """
    try:
        MFNode(value)
        return True
    except ValueError:
        print('isValidMFNode failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFNode) and not isinstance(value, MFNode):
    ###         # if _DEBUG: print('MFNode type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFNode)=' + str(isinstance(value, MFNode)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFNode):
    ###     value = value.value # dereference value from base type
    ###     return True
    ### if isinstance(value, SFNode):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if value and isinstance(value,list):
    ###     for each in value:
    ###         if not isinstance(each, (_X3DNode, _X3DStatement)):
    ###             return False
    ### return True

def assertValidMFNode(value):
    """
    Utility function to assert type validity of a MFNode value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFNode(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFNode') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFNode) and not isinstance(value, MFNode):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFNode')
    ### if isinstance(value, MFNode):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFNode)) and not isinstance(value, list):
        ### value = list(SFNode(value).value) # dereference value from this SF type, convert to list #2c
    ### if value and isinstance(value,list):
    ###     for each in value:
    ###         if not isinstance(each, _X3DNode) and not isinstance(each, _X3DStatement):
    ###             print(flush=True)
    ###             raise X3DTypeError(str(value)[:100] + ' element ' + str(each) + ', type=' + str(type(each)) + ' is not a valid _X3DNode or _X3DStatement for MFNode')
    ### if not isValidMFNode(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid _X3DNode or _X3DStatement for MFNode')
    return True

def isValidSFRotation(value):
    """
    Utility function to determine type validity of a SFRotation value.
    """
    if re.fullmatch(SFRotation().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFRotation().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFRotation(value)
        return True
    except ValueError:
        print('isValidSFRotation failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
    ###         # if _DEBUG: print('SFRotation type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFRotation)=' + str(isinstance(value, SFRotation)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFRotation):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFRotation().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFRotation) and (len(value) == 1) and isValidMFRotation(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFRotation):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 4:
    ###     return False
    ### return True

def assertValidSFRotation(value):
    """
    Utility function to assert type validity of a SFRotation value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFRotation(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFRotation') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFRotation')
    ### if isinstance(value, SFRotation):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFRotation(value)
    ###     print('found string for SFRotation, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFRotation) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFRotation encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFRotation')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFRotation')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFRotation):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFRotation list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 4:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFRotation ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    ### if not isValidSFRotation(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFRotation')
    return True

def isValidMFRotation(value):
    """
    Utility function to determine type validity of a MFRotation value.
    """
    if re.fullmatch(MFRotation().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFRotation().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFRotation(value)
        return True
    except ValueError:
        print('isValidMFRotation failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
    ###         # if _DEBUG: print('MFRotation type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFRotation)=' + str(isinstance(value, MFRotation)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFRotation):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFRotation().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFRotation):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFRotation().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFRotation tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFRotation().TUPLE_SIZE() =' + str(MFRotation().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFRotation(value):
    """
    Utility function to assert type validity of a MFRotation value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFRotation(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFRotation') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFRotation')
    ### if isinstance(value, MFRotation):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFRotation(value)
    ###     print('found string for MFRotation, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFRotation)) and not isinstance(value, list):
        ### value = list(SFRotation(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFRotation')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFRotation().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFRotation tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFRotation().TUPLE_SIZE() =' + str(MFRotation().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFRotation)):
#                # print(flush=True)
#                raise X3DTypeError('MFRotation element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFRotation element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFRotation(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFRotation')
    return True

def isValidSFString(value):
    """
    Utility function to determine type validity of a SFString value.
    """
    if re.fullmatch(SFString().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFString().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFString(value)
        return True
    except ValueError:
        print('isValidSFString failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFString) and not isinstance(value, MFString):
    ###         # if _DEBUG: print('SFString type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFString)=' + str(isinstance(value, SFString)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFString):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFString().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFString) and (len(value) == 1) and isValidMFString(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, str):
    ###     return False
    ### return True

def assertValidSFString(value):
    """
    Utility function to assert type validity of a SFString value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFString(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFString') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFString) and not isinstance(value, MFString):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFString')
    ### if isinstance(value, SFString):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFString) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    ### if not isinstance(value, str):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid str value for SFString')
    ### if not isValidSFString(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python str value for SFString')
    return True

def isValidMFString(value):
    """
    Utility function to determine type validity of a MFString value.
    """
    if re.fullmatch(MFString().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFString().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFString(value)
        return True
    except ValueError:
        print('isValidMFString failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFString) and not isinstance(value, MFString):
    ###         # if _DEBUG: print('MFString type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFString)=' + str(isinstance(value, MFString)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFString):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFString().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFString):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFString):
    ###         each = each.value # dereference
    ###     if not isinstance(each, str):
    ###         return False
    ### return True

def assertValidMFString(value):
    """
    Utility function to assert type validity of a MFString value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFString(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFString') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFString) and not isinstance(value, MFString):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFString')
    ### if isinstance(value, MFString):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFString) or isinstance(value, str)) and not isinstance(value, list):
        ### value = list(SFString(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFString')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFString):
    ###         each = each.value # dereference
    ###     if not isinstance(each, str):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFString list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid str')
    ### if not isValidMFString(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFString')
    return True

def isValidSFTime(value):
    """
    Utility function to determine type validity of a SFTime value.
    """
    if re.fullmatch(SFTime().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFTime().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFTime(value)
        return True
    except ValueError:
        print('isValidSFTime failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFTime) and not isinstance(value, MFTime):
    ###         # if _DEBUG: print('SFTime type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFTime)=' + str(isinstance(value, SFTime)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFTime):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFTime().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFTime) and (len(value) == 1) and isValidMFTime(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, float) and not isinstance(value, int):
    ###     return False
    ### return True

def assertValidSFTime(value):
    """
    Utility function to assert type validity of a SFTime value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFTime(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFTime') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFTime) and not isinstance(value, MFTime):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFTime')
    ### if isinstance(value, SFTime):
    ###     value = value.value # dereference value from this base type #2a
    ### elif isinstance(value, MFTime) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFTime encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFTime')
    ### if not isinstance(value, float) and not isinstance(value, int):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFTime')
    ### if not isValidSFTime(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFTime')
    return True

def isValidMFTime(value):
    """
    Utility function to determine type validity of a MFTime value.
    """
    if re.fullmatch(MFTime().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFTime().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFTime(value)
        return True
    except ValueError:
        print('isValidMFTime failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFTime) and not isinstance(value, MFTime):
    ###         # if _DEBUG: print('MFTime type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFTime)=' + str(isinstance(value, MFTime)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFTime):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFTime().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFTime):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFTime):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### return True

def assertValidMFTime(value):
    """
    Utility function to assert type validity of a MFTime value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFTime(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFTime') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFTime) and not isinstance(value, MFTime):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFTime')
    ### if isinstance(value, MFTime):
    ###     value = value.value # dereference value from this base type #2a
    ### elif (isinstance(value, SFTime)) and not isinstance(value, list):
        ### value = list(SFTime(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFTime')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### for each in value:
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFTime):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('MFTime list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if not isValidMFTime(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFTime')
    return True

def isValidSFVec2d(value):
    """
    Utility function to determine type validity of a SFVec2d value.
    """
    if re.fullmatch(SFVec2d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFVec2d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFVec2d(value)
        return True
    except ValueError:
        print('isValidSFVec2d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
    ###         # if _DEBUG: print('SFVec2d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec2d)=' + str(isinstance(value, SFVec2d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFVec2d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFVec2d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFVec2d) and (len(value) == 1) and isValidMFVec2d(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec2d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 2:
    ###     return False
    ### return True

def assertValidSFVec2d(value):
    """
    Utility function to assert type validity of a SFVec2d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFVec2d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFVec2d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec2d')
    ### if isinstance(value, SFVec2d):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFVec2d(value)
    ###     print('found string for SFVec2d, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFVec2d) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFVec2d encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFVec2d')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2d')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec2d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFVec2d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 2:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFVec2d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 2')
    ### if not isValidSFVec2d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2d')
    return True

def isValidMFVec2d(value):
    """
    Utility function to determine type validity of a MFVec2d value.
    """
    if re.fullmatch(MFVec2d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFVec2d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFVec2d(value)
        return True
    except ValueError:
        print('isValidMFVec2d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
    ###         # if _DEBUG: print('MFVec2d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec2d)=' + str(isinstance(value, MFVec2d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFVec2d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFVec2d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFVec2d):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFVec2d().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFVec2d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2d().TUPLE_SIZE() =' + str(MFVec2d().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFVec2d(value):
    """
    Utility function to assert type validity of a MFVec2d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFVec2d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFVec2d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec2d')
    ### if isinstance(value, MFVec2d):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFVec2d(value)
    ###     print('found string for MFVec2d, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFVec2d)) and not isinstance(value, list):
        ### value = list(SFVec2d(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFVec2d().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFVec2d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2d().TUPLE_SIZE() =' + str(MFVec2d().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec2d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec2d element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFVec2d element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFVec2d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2d')
    return True

def isValidSFVec2f(value):
    """
    Utility function to determine type validity of a SFVec2f value.
    """
    if re.fullmatch(SFVec2f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFVec2f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFVec2f(value)
        return True
    except ValueError:
        print('isValidSFVec2f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
    ###         # if _DEBUG: print('SFVec2f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec2f)=' + str(isinstance(value, SFVec2f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFVec2f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFVec2f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFVec2f) and (len(value) == 1) and isValidMFVec2f(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec2f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 2:
    ###     return False
    ### return True

def assertValidSFVec2f(value):
    """
    Utility function to assert type validity of a SFVec2f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFVec2f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFVec2f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec2f')
    ### if isinstance(value, SFVec2f):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFVec2f(value)
    ###     print('found string for SFVec2f, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFVec2f) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFVec2f encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFVec2f')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2f')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec2f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFVec2f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 2:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFVec2f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 2')
    ### if not isValidSFVec2f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2f')
    return True

def isValidMFVec2f(value):
    """
    Utility function to determine type validity of a MFVec2f value.
    """
    if re.fullmatch(MFVec2f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFVec2f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFVec2f(value)
        return True
    except ValueError:
        print('isValidMFVec2f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
    ###         # if _DEBUG: print('MFVec2f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec2f)=' + str(isinstance(value, MFVec2f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFVec2f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFVec2f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFVec2f):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFVec2f().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFVec2f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2f().TUPLE_SIZE() =' + str(MFVec2f().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFVec2f(value):
    """
    Utility function to assert type validity of a MFVec2f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFVec2f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFVec2f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec2f')
    ### if isinstance(value, MFVec2f):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFVec2f(value)
    ###     print('found string for MFVec2f, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFVec2f)) and not isinstance(value, list):
        ### value = list(SFVec2f(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFVec2f().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFVec2f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2f().TUPLE_SIZE() =' + str(MFVec2f().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec2f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec2f element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFVec2f element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFVec2f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2f')
    return True

def isValidSFVec3d(value):
    """
    Utility function to determine type validity of a SFVec3d value.
    """
    if re.fullmatch(SFVec3d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFVec3d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFVec3d(value)
        return True
    except ValueError:
        print('isValidSFVec3d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
    ###         # if _DEBUG: print('SFVec3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec3d)=' + str(isinstance(value, SFVec3d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFVec3d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFVec3d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFVec3d) and (len(value) == 1) and isValidMFVec3d(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec3d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 3:
    ###     return False
    ### return True

def assertValidSFVec3d(value):
    """
    Utility function to assert type validity of a SFVec3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFVec3d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFVec3d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec3d')
    ### if isinstance(value, SFVec3d):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFVec3d(value)
    ###     print('found string for SFVec3d, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFVec3d) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFVec3d encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFVec3d')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3d')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec3d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFVec3d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 3:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFVec3d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    ### if not isValidSFVec3d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3d')
    return True

def isValidMFVec3d(value):
    """
    Utility function to determine type validity of a MFVec3d value.
    """
    if re.fullmatch(MFVec3d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFVec3d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFVec3d(value)
        return True
    except ValueError:
        print('isValidMFVec3d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
    ###         # if _DEBUG: print('MFVec3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec3d)=' + str(isinstance(value, MFVec3d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFVec3d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFVec3d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFVec3d):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFVec3d().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFVec3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3d().TUPLE_SIZE() =' + str(MFVec3d().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFVec3d(value):
    """
    Utility function to assert type validity of a MFVec3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFVec3d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFVec3d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec3d')
    ### if isinstance(value, MFVec3d):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFVec3d(value)
    ###     print('found string for MFVec3d, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFVec3d)) and not isinstance(value, list):
        ### value = list(SFVec3d(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFVec3d().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFVec3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3d().TUPLE_SIZE() =' + str(MFVec3d().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec3d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec3d element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFVec3d element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFVec3d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3d')
    return True

def isValidSFVec3f(value):
    """
    Utility function to determine type validity of a SFVec3f value.
    """
    if re.fullmatch(SFVec3f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFVec3f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFVec3f(value)
        return True
    except ValueError:
        print('isValidSFVec3f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
    ###         # if _DEBUG: print('SFVec3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec3f)=' + str(isinstance(value, SFVec3f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFVec3f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFVec3f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFVec3f) and (len(value) == 1) and isValidMFVec3f(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec3f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 3:
    ###     return False
    ### return True

def assertValidSFVec3f(value):
    """
    Utility function to assert type validity of a SFVec3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFVec3f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFVec3f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec3f')
    ### if isinstance(value, SFVec3f):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFVec3f(value)
    ###     print('found string for SFVec3f, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFVec3f) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFVec3f encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFVec3f')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3f')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec3f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFVec3f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 3:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFVec3f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    ### if not isValidSFVec3f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3f')
    return True

def isValidMFVec3f(value):
    """
    Utility function to determine type validity of a MFVec3f value.
    """
    if re.fullmatch(MFVec3f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFVec3f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFVec3f(value)
        return True
    except ValueError:
        print('isValidMFVec3f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
    ###         # if _DEBUG: print('MFVec3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec3f)=' + str(isinstance(value, MFVec3f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFVec3f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFVec3f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFVec3f):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFVec3f().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFVec3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3f().TUPLE_SIZE() =' + str(MFVec3f().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFVec3f(value):
    """
    Utility function to assert type validity of a MFVec3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFVec3f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFVec3f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec3f')
    ### if isinstance(value, MFVec3f):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFVec3f(value)
    ###     print('found string for MFVec3f, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFVec3f)) and not isinstance(value, list):
        ### value = list(SFVec3f(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFVec3f().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFVec3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3f().TUPLE_SIZE() =' + str(MFVec3f().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec3f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec3f element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFVec3f element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFVec3f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3f')
    return True

def isValidSFVec4d(value):
    """
    Utility function to determine type validity of a SFVec4d value.
    """
    if re.fullmatch(SFVec4d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFVec4d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFVec4d(value)
        return True
    except ValueError:
        print('isValidSFVec4d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
    ###         # if _DEBUG: print('SFVec4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec4d)=' + str(isinstance(value, SFVec4d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFVec4d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFVec4d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFVec4d) and (len(value) == 1) and isValidMFVec4d(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec4d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 4:
    ###     return False
    ### return True

def assertValidSFVec4d(value):
    """
    Utility function to assert type validity of a SFVec4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFVec4d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFVec4d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec4d')
    ### if isinstance(value, SFVec4d):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFVec4d(value)
    ###     print('found string for SFVec4d, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFVec4d) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFVec4d encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFVec4d')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4d')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec4d):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFVec4d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 4:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFVec4d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    ### if not isValidSFVec4d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4d')
    return True

def isValidMFVec4d(value):
    """
    Utility function to determine type validity of a MFVec4d value.
    """
    if re.fullmatch(MFVec4d().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFVec4d().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFVec4d(value)
        return True
    except ValueError:
        print('isValidMFVec4d failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
    ###         # if _DEBUG: print('MFVec4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec4d)=' + str(isinstance(value, MFVec4d)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFVec4d):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFVec4d().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFVec4d):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFVec4d().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFVec4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4d().TUPLE_SIZE() =' + str(MFVec4d().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFVec4d(value):
    """
    Utility function to assert type validity of a MFVec4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFVec4d(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFVec4d') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec4d')
    ### if isinstance(value, MFVec4d):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFVec4d(value)
    ###     print('found string for MFVec4d, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFVec4d)) and not isinstance(value, list):
        ### value = list(SFVec4d(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFVec4d().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFVec4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4d().TUPLE_SIZE() =' + str(MFVec4d().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec4d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec4d element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFVec4d element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFVec4d(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4d')
    return True

def isValidSFVec4f(value):
    """
    Utility function to determine type validity of a SFVec4f value.
    """
    if re.fullmatch(SFVec4f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch SFVec4f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        SFVec4f(value)
        return True
    except ValueError:
        print('isValidSFVec4f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
    ###         # if _DEBUG: print('SFVec4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec4f)=' + str(isinstance(value, SFVec4f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, SFVec4f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(SFVec4f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, MFVec4f) and (len(value) == 1) and isValidMFVec4f(value):
    ###     value = value.value[0] # dereference value from this MF type
    ###     return True
    ### if not isinstance(value, tuple):
    ###     return False
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec4f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         return False
    ### if tupleCount != 4:
    ###     return False
    ### return True

def assertValidSFVec4f(value):
    """
    Utility function to assert type validity of a SFVec4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        SFVec4f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid SFVec4f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec4f')
    ### if isinstance(value, SFVec4f):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = SFVec4f(value)
    ###     print('found string for SFVec4f, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif isinstance(value, MFVec4f) and len(value) == 1:
    ###     value = value.value[0] # dereference value from this MF type #2b
    # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
    ### elif isinstance(value, str):
    ###     try:
    ###         float(value) # checks but does not set value, may throw exception
    ###     except ValueError:
    ###         print('SFVec4f encountered string with illegal value=' + str(value))
    ###         raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFVec4f')
    ### if not isinstance(value, tuple):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4f')
    ### # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    ### tupleCount = 0
    ### for each in value:
    ###     tupleCount += 1
    ###     while isinstance(each, list) and len(each) == 1:
    ###         each = each[0] # dereference
    ###     if isinstance(each, SFVec4f):
    ###         each = each.value # dereference
    ###     if not isinstance(each, float) and not isinstance(each, int):
    ###         # print(flush=True)
    ###         raise X3DTypeError('SFVec4f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    ### if tupleCount != 4:
    ###     # print(flush=True)
    ###     raise X3DTypeError('SFVec4f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    ### if not isValidSFVec4f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4f')
    return True

def isValidMFVec4f(value):
    """
    Utility function to determine type validity of a MFVec4f value.
    """
    if re.fullmatch(MFVec4f().REGEX_PYTHON(),str(value)) is None:
        print('* regex mismatch MFVec4f().REGEX_PYTHON(),' + str(value) + ')')
        return False
    try:
        MFVec4f(value)
        return True
    except ValueError:
        print('isValidMFVec4f failed with illegal value=' + str(value))
        return False
    ### if isinstance(value, _X3DField):
    ###     if not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
    ###         # if _DEBUG: print('MFVec4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec4f)=' + str(isinstance(value, MFVec4f)), flush=True)
    ###        return False # type mismatch!
    ### if isinstance(value, MFVec4f):
    ###     value = value.value # dereference value from base type
    ###     if re.fullmatch(MFVec4f().REGEX_PYTHON(),str(value)) is None:
    ###         return False
    ###     return True
    ### if isinstance(value, SFVec4f):
    ###     value = list(value.value) # dereference value from this SF type, convert to list #1
    ###     return True
    ### if not isinstance(value, list):
    ###     return False
    ### index = 0
    ### for each in value:
    ###     index += 1
   ###      if len(each) % MFVec4f().TUPLE_SIZE()  != 0:
    ###         # if _DEBUG:
    ###         print('* isValidMFVec4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4f().TUPLE_SIZE() =' + str(MFVec4f().TUPLE_SIZE() ) + ' for value=' + str(value), flush=True)
    ###         return False
    ###     for element in each:
    ###         if not isinstance(element, float) and not isinstance(element, int):
    ###             return False
    ### return True

def assertValidMFVec4f(value):
    """
    Utility function to assert type validity of a MFVec4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    try:
        MFVec4f(value)
    except Exception as error:
        # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
        print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' but is not a valid MFVec4f') from error
    # if _DEBUG: print('...DEBUG... debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    # if _DEBUG: print('value=', value, 'str(value)=', str(value))
    ### if isinstance(value, _X3DField) and not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec4f')
    ### if isinstance(value, MFVec4f):
    ###     value = value.value # dereference value from this base type #2a
    ### if isinstance(value,str):
    ###     value = MFVec4f(value)
    ###     print('found string for MFVec4f, isinstance(value,list)=' + str(isinstance(value,list)),flush=True)
    ### elif (isinstance(value, SFVec4f)) and not isinstance(value, list):
        ### value = list(SFVec4f(value).value) # dereference value from this SF type, convert to list #2c
    ### if not isinstance(value, list):
    ###     print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    ### if isinstance(value, list):
    ###     index = 0
    ###     for each in value:
    ###         if len(each) % MFVec4f().TUPLE_SIZE()  != 0:
                # print(flush=True)
    ###             raise X3DValueError('MFVec4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4f().TUPLE_SIZE() =' + str(MFVec4f().TUPLE_SIZE() ) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec4f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec4f element #' + str(index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
        ###     index += 1
        ###     if isinstance(each, tuple):
        ###         for element in each:
        ###             if not isinstance(element, float) and not isinstance(element, int):
        ###                 # print(flush=True)
        ###                 raise X3DTypeError('MFVec4f element #' + str(index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    ### if not isValidMFVec4f(value):
    ###     # print(flush=True)
    ###     raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4f')
    return True


def assertValidFieldInitializationValue(name, fieldType, value, parent=''):
    """
    Utility function to assert fieldType validity of a field initialization value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('...DEBUG... assertValidFieldInitializationValue name=' + str(name) + ', fieldType=' + str(fieldType) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
    # note that ExternProtoDeclare field definitions do not have any value
    if name is None:
        print('* assertValidFieldInitializationValue improper invocation: name=' + str(name) + ', fieldType=' + str(fieldType) + ', value=' + str(value)[:100] + ', parent=' + parent + ', ignored', flush=True)
        return # ignore
    if value is None or (not(fieldType == bool) and not value):
        return # ignore
    if fieldType == 'SFString':
        assertValidSFString(value)
    elif fieldType == 'MFString':
        assertValidMFString(value)
    elif (fieldType == 'SFBool') or (fieldType == bool) or isinstance(value, bool):
        assertValidSFBool(value)
    elif fieldType == 'MFBool':
        assertValidMFBool(value)
    elif (fieldType == 'SFInt32') or (fieldType == int) or isinstance(value, int):
        assertValidSFInt32(value)
    elif fieldType == 'MFInt32':
        assertValidMFInt32(value)
    elif (fieldType == 'SFFloat') or (fieldType == float) or isinstance(value, float):
        assertValidSFFloat(value)
    elif fieldType == 'MFFloat':
        assertValidMFFloat(value)
    elif fieldType == 'SFDouble':
        assertValidSFDouble(value)
    elif fieldType == 'MFDouble':
        assertValidMFDouble(value)
    elif fieldType == 'SFTime':
        assertValidSFTime(value)
    elif fieldType == 'MFTime':
        assertValidMFTime(value)
    elif fieldType == 'SFColor':
        assertValidSFColor(value)
    elif fieldType == 'MFColorRGBA':
        assertValidMFColorRGBA(value)
    elif fieldType == 'SFRotation':
        assertValidSFRotation(value)
    elif fieldType == 'MFRotation':
        assertValidMFRotation(value)
    elif fieldType == 'SFImage':
        assertValidSFImage(value)
    elif fieldType == 'MFImage':
        assertValidMFImage(value)
    elif fieldType == 'SFNode':
        assertValidSFNode(value)
    elif fieldType == 'MFNode':
        assertValidMFNode(value)
    elif fieldType == 'SFVec2f':
        assertValidSFVec2f(value)
    elif fieldType == 'MFVec2f':
        assertValidMFVec2f(value)
    elif fieldType == 'SFVec3f':
        assertValidSFVec3f(value)
    elif fieldType == 'MFVec3f':
        assertValidMFVec3f(value)
    elif fieldType == 'SFVec4f':
        assertValidSFVec4f(value)
    elif fieldType == 'MFVec4f':
        assertValidMFVec4f(value)
    elif fieldType == 'SFVec2d':
        assertValidSFVec2d(value)
    elif fieldType == 'MFVec2d':
        assertValidMFVec2d(value)
    elif fieldType == 'SFVec3d':
        assertValidSFVec3d(value)
    elif fieldType == 'MFVec3d':
        assertValidMFVec3d(value)
    elif fieldType == 'SFVec4d':
        assertValidSFVec4d(value)
    elif fieldType == 'MFVec4d':
        assertValidMFVec4d(value)
    elif fieldType == 'SFMatrix3d':
        assertValidSFMatrix3f(value)
    elif fieldType == 'MFMatrix3f':
        assertValidMFMatrix3f(value)
    elif fieldType == 'SFMatrix4f':
        assertValidSFMatrix4f(value)
    elif fieldType == 'MFMatrix4f':
        assertValidMFMatrix4f(value)
    elif fieldType == 'SFMatrix3d':
        assertValidSFMatrix3d(value)
    elif fieldType == 'MFMatrix3d':
        assertValidMFMatrix3d(value)
    elif fieldType == 'SFMatrix4d':
        assertValidSFMatrix4d(value)
    elif fieldType == 'MFMatrix4d':
        assertValidMFMatrix4d(value)
    elif (fieldType == str)   or isinstance(value, str):
        assertValidSFString(value)
    elif str(parent) == 'fieldValue':
        return True # TODO check further if possible
    elif (fieldType == list) or isinstance(value, list):
        try:
            if isinstance(value[0], tuple):
                print('*** assertValidFieldInitializationValue TODO validate list fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
                return True # TODO check further
            initialListItemType = fieldType(value[0])
            # https://stackoverflow.com/questions/522563/accessing-the-index-in-for-loops/28072982#28072982
            # https://stackoverflow.com/questions/1952464/in-python-how-do-i-determine-if-an-object-is-iterable
            for index, each in enumerate(value):
                assertValidFieldInitializationValue(name + '[' + str(index) + ']', initialListItemType, value[index], parent)
            return True
        except TypeError:
            return False # TODO check further if possible
    elif (fieldType == tuple) or isinstance(value, tuple):
        print('*** assertValidFieldInitializationValue TODO validate tuple fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
        return True # TODO check further if possible
#       initialListItemType = fieldType(value[0])
#       for index, each in enumerate(value):
#           assertValidFieldInitializationValue(name + '[' + str(index) + '], fieldType(value[index])', value[index], parent)
    else:
        print('*** assertValidFieldInitializationValue unknown fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
        return False # TODO check further if possible
    return True
###############################################

class _X3DField:
    """
    X3DField is the abstract field type from which all single values field types are derived. All fields directly derived from X3DField have names beginning with SF (single-valued field). SF fields may only contain a single value of the type indicated by the name of the field type.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return '_X3DField'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldTypes.html#X3DField'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FieldTypesTable'
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide value of this field type. """
        return self.__value
    def __repl__(self):
        # if _DEBUG: print('...DEBUG... type(self.value)=' + str(type(self.value)), flush=True)
        if isinstance(self.value, (SFString, str)):
            return "'" + self.value + "'"
        if  isinstance(self.value, tuple) and 'SF' in str(type(self)): # avoid X3DTypeError if value is not iterable
            result = '('
            if self.value: # walk each child in list, if any (avoid empty list recursion)
                for each in self.value:
                    result += str(each) + ', '
                    # if _DEBUG: print('...DEBUG... _X3DField debug: str(each)=' + str(each), flush=True)
            return result.rstrip(', ') + ')'
        if  isinstance(self.value, list) and 'MF' in str(type(self)): # avoid X3DTypeError if value is not iterable
            # isinstance(self.value, MFNode): not working, what got passed in was not an MFNode object apparently
            result = '['
            if self.value: # walk each child in list, if any (avoid empty list recursion)
                for each in self.value:
                    result += str(each) + ', '
                    # if _DEBUG: print('...DEBUG... _X3DField debug: str(each)=' + str(each), flush=True)
            return result.rstrip(', ') + ']'
        return str(self.value)
    def __str__(self):
        return self.__repl__()

class _X3DArrayField(_X3DField):
    """
    X3DArrayField is the abstract field type from which all field types that can contain multiple values are derived, implementing the X3DField interface. All fields derived from X3DArrayField have names beginning with MF (multiple-valued field). MF fields may zero or more values, each of which shall be of the type indicated by the corresponding SF field type. It is illegal for any MF field to mix values of different SF field types.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DArrayField'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldTypes.html#X3DArrayField'

def isX3DField(value):
    """
    Determine whether object is an instance of _X3DField.
    """
    return isinstance(value, _X3DField)

# Access Types

class AccessType(_X3DField):
    """
    accessType determines whether a field corresponds to event input, event output, or persistent state information. Events are strictly typed values with a corresponding timestamp. ROUTE connections must match accessType between source field and target field.
    """
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/concepts.html#FieldSemantics'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#accessType'
    @staticmethod
    def initializeOnly():
        """ initializeOnly: can be initialized, but cannot send or receive events. This is usually the case for fields that are considered too computationally expensive to change at run time. """
        return 'initializeOnly'
    @staticmethod
    def inputOutput():
        """ inputOutput: can be initialized, and can also send or receive events during run-time operations. """
        return 'inputOutput'
    @staticmethod
    def inputOnly():
        """ inputOnly: cannot be initialized or included in a scene file, but can receive input event values via a ROUTE during run-time operations. """
        return 'inputOnly'
    @staticmethod
    def outputOnly():
        """ outputOnly: cannot be initialized or included in a scene file, but can send output event values via a ROUTE during run-time operations. """
        return 'outputOnly'


# Field Types

class FieldType(_X3DField):
    """
    The X3D Architecture specification of field types classify the possible values for a field.
    Each field in each node (i.e. each XML attribute) has a strictly defined data type.
    Multiple data types are provided for boolean, integer, floating-point and string values.
    X3D is a strongly typed language, meaning that all data must strictly conform to these data types in order for a scene to be correct.
    """
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/fieldsDef.html'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#type'
    # string constants listing each allowed type
    @staticmethod
    def SFBool():
        """ Type SFBool https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFBool """
        return 'SFBool'
    @staticmethod
    def MFBool():
        """ Type MFBool https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFBool """
        return 'MFBool'
    @staticmethod
    def SFColor():
        """ Type SFColor https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColor """
        return 'SFColor'
    @staticmethod
    def MFColor():
        """ Type MFColor https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColor """
        return 'MFColor'
    @staticmethod
    def SFColorRGBA():
        """ Type SFColorRGBA https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColorRGBA """
        return 'SFColorRGBA'
    @staticmethod
    def MFColorRGBA():
        """ Type MFColorRGBA https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColorRGBA """
        return 'MFColorRGBA'
    @staticmethod
    def SFDouble():
        """ Type SFDouble https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFDouble """
        return 'SFDouble'
    @staticmethod
    def MFDouble():
        """ Type MFDouble https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFDouble """
        return 'MFDouble'
    @staticmethod
    def SFFloat():
        """ Type SFFloat https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFFloat """
        return 'SFFloat'
    @staticmethod
    def MFFloat():
        """ Type MFFloat https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFFloat """
        return 'MFFloat'
    @staticmethod
    def SFImage():
        """ Type SFImage https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFImage """
        return 'SFImage'
    @staticmethod
    def MFImage():
        """ Type MFImage https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFImage """
        return 'MFImage'
    @staticmethod
    def SFInt32():
        """ Type SFInt32 https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFInt32 """
        return 'SFInt32'
    @staticmethod
    def MFInt32():
        """ Type MFInt32 https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFInt32 """
        return 'MFInt32'
    @staticmethod
    def SFMatrix3d():
        """ Type SFMatrix3d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3d """
        return 'SFMatrix3d'
    @staticmethod
    def MFMatrix3d():
        """ Type MFMatrix3d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3d """
        return 'MFMatrix3d'
    @staticmethod
    def SFMatrix3f():
        """ Type SFMatrix3f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3f """
        return 'SFMatrix3f'
    @staticmethod
    def MFMatrix3f():
        """ Type MFMatrix3f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3f """
        return 'MFMatrix3f'
    @staticmethod
    def SFMatrix4d():
        """ Type SFMatrix4d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4d """
        return 'SFMatrix4d'
    @staticmethod
    def MFMatrix4d():
        """ Type MFMatrix4d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4d """
        return 'MFMatrix4d'
    @staticmethod
    def SFMatrix4f():
        """ Type SFMatrix4f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4f """
        return 'SFMatrix4f'
    @staticmethod
    def MFMatrix4f():
        """ Type MFMatrix4f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4f """
        return 'MFMatrix4f'
    @staticmethod
    def SFNode():
        """ Type SFNode https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFNode """
        return 'SFNode'
    @staticmethod
    def MFNode():
        """ Type MFNode https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFNode """
        return 'MFNode'
    @staticmethod
    def SFRotation():
        """ Type SFRotation https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFRotation """
        return 'SFRotation'
    @staticmethod
    def MFRotation():
        """ Type MFRotation https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFRotation """
        return 'MFRotation'
    @staticmethod
    def SFString():
        """ Type SFString https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFString """
        return 'SFString'
    @staticmethod
    def MFString():
        """ Type MFString https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFString """
        return 'MFString'
    @staticmethod
    def SFTime():
        """ Type SFTime https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFTime """
        return 'SFTime'
    @staticmethod
    def MFTime():
        """ Type MFTime https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFTime """
        return 'MFTime'
    @staticmethod
    def SFVec2d():
        """ Type SFVec2d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2d """
        return 'SFVec2d'
    @staticmethod
    def MFVec2d():
        """ Type MFVec2d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2d """
        return 'MFVec2d'
    @staticmethod
    def SFVec2f():
        """ Type SFVec2f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2f """
        return 'SFVec2f'
    @staticmethod
    def MFVec2f():
        """ Type MFVec2f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2f """
        return 'MFVec2f'
    @staticmethod
    def SFVec3d():
        """ Type SFVec3d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3d """
        return 'SFVec3d'
    @staticmethod
    def MFVec3d():
        """ Type MFVec3d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3d """
        return 'MFVec3d'
    @staticmethod
    def SFVec3f():
        """ Type SFVec3f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3f """
        return 'SFVec3f'
    @staticmethod
    def MFVec3f():
        """ Type MFVec3f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3f """
        return 'MFVec3f'
    @staticmethod
    def SFVec4d():
        """ Type SFVec4d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4d """
        return 'SFVec4d'
    @staticmethod
    def MFVec4d():
        """ Type MFVec4d https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4d """
        return 'MFVec4d'
    @staticmethod
    def SFVec4f():
        """ Type SFVec4f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4f """
        return 'SFVec4f'
    @staticmethod
    def MFVec4f():
        """ Type MFVec4f https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4f """
        return 'MFVec4f'

class SFBool(_X3DField):
    """
    Field type Python Boolean values are capitalized as True or False. SFBool is a logical type with possible values (true|false) to match the XML boolean type. Hint: XML boolean values are lower case (true|false) in order to maintain compatibility with HTML and other XML documents.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFBool'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFBoolAndMFBool'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFBool'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return True
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(true|false|True|False)\s*' # (less than fully strict Python: allows lower-case strings true, false)
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(true|false)\s*'
    # - - - - - - - - - -
    def __init__(self, value=True):
        # print('*** SFBool __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        value = fixBoolean(value, default=SFBool.DEFAULT_VALUE())
        if isinstance(value,SFBool):
            value = value.value # dereference
        elif value is None:
            value = SFBool.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFBool.DEFAULT_VALUE()=' + str(SFBool.DEFAULT_VALUE()))
        elif isinstance(value, MFBool) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFBool self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower()
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).upper()
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower()

class MFBool(_X3DArrayField):
    """
    Field type Python Boolean values are capitalized as True or False. MFBool is an array of boolean values. Type MFBool was previously undefined in the VRML97 Specification, but nevertheless needed for event utilities and scripting. Example use: MFBool is useful for defining a series of behavior states using a BooleanSequencer prototype. Hint: XML boolean values are lower case (true|false) in order to maintain compatibility with HTML and other XML documents.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFBool'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFBoolAndMFBool'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFBool'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*((true|false|True|False)\s*,?\s*)*\]?\s*' # (less than fully strict Python: allows lower-case strings true, false)
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((true|false)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFBool __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        value = fixBoolean(value, default=MFBool.DEFAULT_VALUE())
        if not isinstance(value,list):
            _newValue = [ value ]
        else:
            _newValue = []
            for each in value:
                _newValue.append(SFBool(each).value)
        value = _newValue
        if isinstance(value,SFBool):
            value = value.value # dereference
        elif value is None:
            value = MFBool.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFBool.DEFAULT_VALUE()=' + str(MFBool.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFBool(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFBool(SFBool(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFBool(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ bool(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFBool):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFBool):
                self.__value.append(SFBool(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFBool):
                for each in value:
                    self.__value.append(SFBool(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFBool(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFBool(value):
    ###             if isinstance(value, SFBool):
    ###                 value = SFBool(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFBool(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFBool):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFBool(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFBool self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).upper().replace(',', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('[', '').replace(']', '')

class SFColor(_X3DField):
    """
    Field type SFColor specifies one RGB (red-green-blue) color triple, where each color value is an RGB triple of floating point numbers in range [0,1]. The default value of an uninitialized SFColor field is (0 0 0). Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFColor'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFColorAndMFColor'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColor'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0, 0, 0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 3
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*\,?\s*){2}([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s+){2}([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None):
        # print('*** SFColor __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        if isinstance(value,SFColor):
            value = value.value # dereference
        elif value is None:
            value = SFColor.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFColor.DEFAULT_VALUE()=' + str(SFColor.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif isinstance(value, MFColor) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFColor encountered string with illegal value=' + str(value)) from error
        assertZeroToOne(SFColor,value)
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFColor self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFColor(_X3DArrayField):
    """
    Field type MFColor specifies zero or more SFColor RGB triples, where each color value is an RGB triple of floating point numbers in range [0,1]. The default value of an uninitialized MFColor field is the empty list. Individual SFColor array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFColor'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFColorAndMFColor'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColor'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 3
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*\,?\s*){2}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFColor __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        if isinstance(value,SFColor):
            value = value.value # dereference
        elif value is None:
            value = MFColor.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFColor.DEFAULT_VALUE()=' + str(MFColor.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
    ###     elif not isinstance(value, list) and isValidSFColor(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFColor(SFColor(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFColor(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        assertZeroToOne(MFColor,value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFColor):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFColor):
                self.__value.append(SFColor(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFColor):
                for each in value:
                    self.__value.append(SFColor(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFColor(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFColor(value):
    ###             if isinstance(value, SFColor):
    ###                 value = SFColor(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFColor(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFColor):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFColor(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFColor self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFColorRGBA(_X3DField):
    """
    Field type SFColorRGBA specifies one RGBA (red-green-blue-alpha) color 4-tuple, where each color value is an RGBA 4-tuple of floating point numbers in range [0,1]. Alpha (opacity) values = (1 - transparency). The default value of an uninitialized SFColorRGBA field is (0 0 0 0). Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFColorRGBA'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFColorRGBAAndMFColorRGBA'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColorRGBA'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0, 0, 0, 0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        # print('*** SFColorRGBA __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFColorRGBA):
            value = value.value # dereference
        elif value is None:
            value = SFColorRGBA.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFColorRGBA.DEFAULT_VALUE()=' + str(SFColorRGBA.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFColorRGBA) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFColorRGBA encountered string with illegal value=' + str(value)) from error
        assertZeroToOne(SFColorRGBA,value)
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFColorRGBA self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFColorRGBA(_X3DArrayField):
    """
    Field type MFColorRGBA specifies zero or more SFColorRGBA 4-tuples, where each color value is an RGBA 4-tuple of floating point numbers in range [0,1]. Alpha (opacity) values = (1 - transparency). The default value of an uninitialized MFColor field is the empty list. Individual SFColorRGBA array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFColorRGBA'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFColorRGBAAndMFColorRGBA'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColorRGBA'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFColorRGBA __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFColorRGBA):
            value = value.value # dereference
        elif value is None:
            value = MFColorRGBA.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFColorRGBA.DEFAULT_VALUE()=' + str(MFColorRGBA.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
    ###     elif not isinstance(value, list) and isValidSFColorRGBA(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFColorRGBA(SFColorRGBA(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFColorRGBA(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        assertZeroToOne(MFColorRGBA,value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFColorRGBA):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFColorRGBA):
                self.__value.append(SFColorRGBA(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFColorRGBA):
                for each in value:
                    self.__value.append(SFColorRGBA(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFColorRGBA(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFColorRGBA(value):
    ###             if isinstance(value, SFColorRGBA):
    ###                 value = SFColorRGBA(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFColorRGBA(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFColorRGBA):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFColorRGBA(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFColorRGBA self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFDouble(_X3DField):
    """
    Field type SFDouble is a double-precision floating-point type. Array values are optionally separated by commas in XML syntax. See GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision for rationale.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFDouble'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFDoubleAndMFDouble'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFDouble'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return 0.0
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=0.0):
        # print('*** SFDouble __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFDouble):
            value = value.value # dereference
        elif value is None:
            value = SFDouble.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFDouble.DEFAULT_VALUE()=' + str(SFDouble.DEFAULT_VALUE()))
        elif isinstance(value, MFDouble) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFDouble encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFDouble self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFDouble(_X3DArrayField):
    """
    Field type MFDouble is an array of Double values, meaning a double-precision floating-point array type. See GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision for rationale. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFDouble'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFDoubleAndMFDouble'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFDouble'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFDouble __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFDouble):
            value = value.value # dereference
        elif value is None:
            value = MFDouble.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFDouble.DEFAULT_VALUE()=' + str(MFDouble.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFDouble(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFDouble(SFDouble(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFDouble(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFDouble):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFDouble):
                self.__value.append(SFDouble(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFDouble):
                for each in value:
                    self.__value.append(SFDouble(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFDouble(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFDouble(value):
    ###             if isinstance(value, SFDouble):
    ###                 value = SFDouble(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFDouble(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFDouble):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFDouble(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFDouble self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFFloat(_X3DField):
    """
    Field type SFFloat is a single-precision floating-point type.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFFloat'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFFloatAndMFFloat'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFFloat'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return 0.0
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=0.0):
        # print('*** SFFloat __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFFloat):
            value = value.value # dereference
        elif value is None:
            value = SFFloat.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFFloat.DEFAULT_VALUE()=' + str(SFFloat.DEFAULT_VALUE()))
        elif isinstance(value, MFFloat) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFFloat encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFFloat self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFFloat(_X3DArrayField):
    """
    Field type MFFloat is an array of SFFloat values, meaning a single-precision floating-point array type. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFFloat'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFFloatAndMFFloat'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFFloat'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFFloat __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFFloat):
            value = value.value # dereference
        elif value is None:
            value = MFFloat.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFFloat.DEFAULT_VALUE()=' + str(MFFloat.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFFloat(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFFloat(SFFloat(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFFloat(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFFloat):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFFloat):
                self.__value.append(SFFloat(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFFloat):
                for each in value:
                    self.__value.append(SFFloat(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFFloat(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFFloat(value):
    ###             if isinstance(value, SFFloat):
    ###                 value = SFFloat(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFFloat(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFFloat):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFFloat(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFFloat self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFImage(_X3DField):
    """
    Field type SFImage specifies a single uncompressed 2-dimensional pixel image. SFImage fields contain three integers representing the width, height and number of components in the image, followed by (width x height) hexadecimal or integer values representing the pixels in the image.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFImage'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFImageAndMFImage'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFImage'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [0, 0, 0]
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*'
    # - - - - - - - - - -
    def __init__(self, value=[0, 0, 0]):
        # print('*** SFImage __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFImage):
            value = value.value # dereference
        elif value is None:
            value = SFImage.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFImage.DEFAULT_VALUE()=' + str(SFImage.DEFAULT_VALUE()))
        elif isinstance(value, MFImage) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFImage encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFImage self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFImage(_X3DArrayField):
    """
    Field type MFImage is an array of SFImage values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFImage'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFImageAndMFImage'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFImage'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFImage __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFImage):
            value = value.value # dereference
        elif value is None:
            value = MFImage.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFImage.DEFAULT_VALUE()=' + str(MFImage.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFImage(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFImage(SFImage(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFImage(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ int(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFImage):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFImage):
                self.__value.append(SFImage(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFImage):
                for each in value:
                    self.__value.append(SFImage(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFImage(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFImage(value):
    ###             if isinstance(value, SFImage):
    ###                 value = SFImage(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFImage(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFImage):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFImage(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFImage self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFInt32(_X3DField):
    """
    Field type SFInt32 specifies one 32-bit signed integer.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFInt32'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFInt32AndMFInt32'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFInt32'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return 0
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*[+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*[+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?\s*'
    # - - - - - - - - - -
    def __init__(self, value=0):
        # print('*** SFInt32 __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = int(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFInt32):
            value = value.value # dereference
        elif value is None:
            value = SFInt32.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFInt32.DEFAULT_VALUE()=' + str(SFInt32.DEFAULT_VALUE()))
        elif isinstance(value, MFInt32) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                int(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', int(value)=' + str(int(value)), flush=True)
                value = int(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFInt32 encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFInt32 self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFInt32(_X3DArrayField):
    """
    Field type MFInt32 defines an array of 32-bit signed integers. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFInt32'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFInt32AndMFInt32'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFInt32'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*([+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFInt32 __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFInt32):
            value = value.value # dereference
        elif value is None:
            value = MFInt32.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFInt32.DEFAULT_VALUE()=' + str(MFInt32.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFInt32(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFInt32(SFInt32(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFInt32(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ int(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFInt32):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFInt32):
                self.__value.append(SFInt32(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFInt32):
                for each in value:
                    self.__value.append(SFInt32(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFInt32(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFInt32(value):
    ###             if isinstance(value, SFInt32):
    ###                 value = SFInt32(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFInt32(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFInt32):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFInt32(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFInt32 self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFMatrix3d(_X3DField):
    """
    Field type SFMatrix3d specifies a 3x3 matrix of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFMatrix3d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix3dAndMFMatrix3d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (1, 0, 0, 0, 1, 0, 0, 0, 1)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 9
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=(1, 0, 0, 0, 1, 0, 0, 0, 1)):
        # print('*** SFMatrix3d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix3d):
            value = value.value # dereference
        elif value is None:
            value = SFMatrix3d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFMatrix3d.DEFAULT_VALUE()=' + str(SFMatrix3d.DEFAULT_VALUE()))
        elif isinstance(value, MFMatrix3d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFMatrix3d encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFMatrix3d self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFMatrix3d(_X3DArrayField):
    """
    Field type MFMatrix3d specifies zero or more 3x3 matrices of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 9-tuple values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFMatrix3d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix3dAndMFMatrix3d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 9
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFMatrix3d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix3d):
            value = value.value # dereference
        elif value is None:
            value = MFMatrix3d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFMatrix3d.DEFAULT_VALUE()=' + str(MFMatrix3d.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFMatrix3d(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFMatrix3d(SFMatrix3d(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFMatrix3d(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFMatrix3d):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFMatrix3d):
                self.__value.append(SFMatrix3d(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFMatrix3d):
                for each in value:
                    self.__value.append(SFMatrix3d(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFMatrix3d(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFMatrix3d(value):
    ###             if isinstance(value, SFMatrix3d):
    ###                 value = SFMatrix3d(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFMatrix3d(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFMatrix3d):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFMatrix3d(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFMatrix3d self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFMatrix3f(_X3DField):
    """
    Field type SFMatrix3f specifies a 3x3 matrix of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFMatrix3f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix3fAndMFMatrix3f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (1, 0, 0, 0, 1, 0, 0, 0, 1)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 9
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=(1, 0, 0, 0, 1, 0, 0, 0, 1)):
        # print('*** SFMatrix3f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix3f):
            value = value.value # dereference
        elif value is None:
            value = SFMatrix3f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFMatrix3f.DEFAULT_VALUE()=' + str(SFMatrix3f.DEFAULT_VALUE()))
        elif isinstance(value, MFMatrix3f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFMatrix3f encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFMatrix3f self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFMatrix3f(_X3DArrayField):
    """
    Field type MFMatrix3f specifies zero or more 3x3 matrices of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 9-tuple values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFMatrix3f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix3fAndMFMatrix3f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 9
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFMatrix3f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix3f):
            value = value.value # dereference
        elif value is None:
            value = MFMatrix3f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFMatrix3f.DEFAULT_VALUE()=' + str(MFMatrix3f.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFMatrix3f(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFMatrix3f(SFMatrix3f(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFMatrix3f(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFMatrix3f):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFMatrix3f):
                self.__value.append(SFMatrix3f(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFMatrix3f):
                for each in value:
                    self.__value.append(SFMatrix3f(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFMatrix3f(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFMatrix3f(value):
    ###             if isinstance(value, SFMatrix3f):
    ###                 value = SFMatrix3f(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFMatrix3f(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFMatrix3f):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFMatrix3f(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFMatrix3f self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFMatrix4d(_X3DField):
    """
    Field type SFMatrix4d specifies a 4x4 matrix of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFMatrix4d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix4dAndMFMatrix4d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 16
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)):
        # print('*** SFMatrix4d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix4d):
            value = value.value # dereference
        elif value is None:
            value = SFMatrix4d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFMatrix4d.DEFAULT_VALUE()=' + str(SFMatrix4d.DEFAULT_VALUE()))
        elif isinstance(value, MFMatrix4d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFMatrix4d encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFMatrix4d self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFMatrix4d(_X3DArrayField):
    """
    Field type MFMatrix4d specifies zero or more 4x4 matrices of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 16-tuple values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFMatrix4d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix4dAndMFMatrix4d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 16
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFMatrix4d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix4d):
            value = value.value # dereference
        elif value is None:
            value = MFMatrix4d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFMatrix4d.DEFAULT_VALUE()=' + str(MFMatrix4d.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFMatrix4d(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFMatrix4d(SFMatrix4d(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFMatrix4d(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFMatrix4d):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFMatrix4d):
                self.__value.append(SFMatrix4d(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFMatrix4d):
                for each in value:
                    self.__value.append(SFMatrix4d(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFMatrix4d(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFMatrix4d(value):
    ###             if isinstance(value, SFMatrix4d):
    ###                 value = SFMatrix4d(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFMatrix4d(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFMatrix4d):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFMatrix4d(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFMatrix4d self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFMatrix4f(_X3DField):
    """
    Field type SFMatrix4f specifies a 4x4 matrix of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFMatrix4f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix4fAndMFMatrix4f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 16
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=(1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)):
        # print('*** SFMatrix4f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix4f):
            value = value.value # dereference
        elif value is None:
            value = SFMatrix4f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFMatrix4f.DEFAULT_VALUE()=' + str(SFMatrix4f.DEFAULT_VALUE()))
        elif isinstance(value, MFMatrix4f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFMatrix4f encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFMatrix4f self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFMatrix4f(_X3DArrayField):
    """
    Field type MFMatrix4f specifies zero or more 4x4 matrices of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 16-tuple values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFMatrix4f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFMatrix4fAndMFMatrix4f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 16
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFMatrix4f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFMatrix4f):
            value = value.value # dereference
        elif value is None:
            value = MFMatrix4f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFMatrix4f.DEFAULT_VALUE()=' + str(MFMatrix4f.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFMatrix4f(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFMatrix4f(SFMatrix4f(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFMatrix4f(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFMatrix4f):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFMatrix4f):
                self.__value.append(SFMatrix4f(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFMatrix4f):
                for each in value:
                    self.__value.append(SFMatrix4f(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFMatrix4f(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFMatrix4f(value):
    ###             if isinstance(value, SFMatrix4f):
    ###                 value = SFMatrix4f(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFMatrix4f(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFMatrix4f):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFMatrix4f(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFMatrix4f self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFNode(_X3DField):
    """
    Field type SFNode specifies an X3D node; the default empty value of an uninitialized SFNode field is sometimes described as NULL.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFNodeAndMFNode'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFNode'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return None
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [('value', 'None', FieldType.SFNode, AccessType.inputOutput, 'SFNode')]
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r''
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r''
    # - - - - - - - - - -
    def __init__(self, value=None):
        # print('*** SFNode __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFNode):
            value = value.value # dereference
        elif value is None:
            value = SFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFNode.DEFAULT_VALUE()=' + str(SFNode.DEFAULT_VALUE()))
        elif isinstance(value, MFNode) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFNode self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        if not self.__value is None:
            return self.__value.XML()
        return None
    def VRML(self):
        """ Provide VRML value for this field type. """
        if not self.__value is None:
            return self.__value.VRML()
        return None
    def JSON(self):
        """ Provide JSON value for this field type. """
        if not self.__value is None:
            return self.__value.JSON()
        return None

class MFNode(_X3DArrayField):
    """
    Field type MFNode specifies zero or more nodes; the default value of an MFNode field is the empty list.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFNodeAndMFNode'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFNode'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [('value', None, FieldType.MFNode, AccessType.inputOutput, 'MFNode')]
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r''
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r''
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFNode __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFNode):
            value = value.value # dereference
        elif value is None:
            value = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFNode(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFNode(SFNode(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFNode(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ list(value) ]
        self.__value = value
    def __repl__(self):
        result = '['
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                result += str(each) + ', '
        return result.rstrip(', ') + ']'
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFNode):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFNode):
                self.__value.append(SFNode(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFNode):
                for each in value:
                    self.__value.append(SFNode(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFNode(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFNode(value):
    ###             if isinstance(value, SFNode):
    ###                 value = SFNode(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFNode(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFNode):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFNode(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFNode self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        result = ''
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                if not self.__value is None:
                    result += each.XML() # has line break '\n' at end, which is OK
        result = result.rstrip('\n')
        return result
    def VRML(self):
        """ Provide VRML value for this field type. """
        result = ''
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                if not self.__value is None:
                    result += each.VRML() # has line break '\n' at end, which is OK
        result = result.rstrip('\n')
        return result
    def JSON(self):
        """ Provide JSON value for this field type. """
        result = ''
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                if not self.__value is None:
                    result += each.JSON() # TODO? has line break '\n' at end, which is OK
        result = result.rstrip('\n')
        return result

class SFRotation(_X3DField):
    """
    Field type SFRotation is an axis-angle 4-tuple, indicating X-Y-Z direction axis plus angle orientation about that axis. The first three values specify a normalized axis vector about which the rotation takes place, so the first three values shall be within the range [-1..+1] in order to represent a normalized unit vector. The fourth value specifies the amount of right-handed rotation about that axis in radians. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFRotation'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFRotationAndMFRotation'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFRotation'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0, 0, 1, 0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        # print('*** SFRotation __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFRotation):
            value = value.value # dereference
        elif value is None:
            value = SFRotation.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFRotation.DEFAULT_VALUE()=' + str(SFRotation.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFRotation) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFRotation encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFRotation self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFRotation(_X3DArrayField):
    """
    Field type MFRotation is an array of SFRotation values. Individual singleton SFRotation array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFRotation'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFRotationAndMFRotation'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFRotation'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFRotation __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFRotation):
            value = value.value # dereference
        elif value is None:
            value = MFRotation.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFRotation.DEFAULT_VALUE()=' + str(MFRotation.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
    ###     elif not isinstance(value, list) and isValidSFRotation(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFRotation(SFRotation(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFRotation(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFRotation):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFRotation):
                self.__value.append(SFRotation(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFRotation):
                for each in value:
                    self.__value.append(SFRotation(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFRotation(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFRotation(value):
    ###             if isinstance(value, SFRotation):
    ###                 value = SFRotation(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFRotation(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFRotation):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFRotation(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFRotation self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFString(_X3DField):
    """
    Field type SFString defines a single string encoded with the UTF-8 universal character set.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFString'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFStringAndMFString'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFString'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return ''
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'(\s|\S)*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'(\s|\S)*'
    # - - - - - - - - - -
    def __init__(self, value=''):
        # print('*** SFString __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFString):
            value = value.value # dereference
        elif value is None:
            value = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        elif isinstance(value, MFString) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFString self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).replace("'","&apos;").replace("& ","&amp; ").replace("<","&lt;").replace(">","&gt;")
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '"' + str(self.__value) + '"'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).replace('"','&quot;')

class MFString(_X3DArrayField):
    """
    Field type MFString is an array of SFString values, each "quoted" and separated by whitespace. Individual SFString array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFString'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFStringAndMFString'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFString'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?(\s|\S)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'(\s|\S)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFString __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFString):
            value = value.value # dereference
        elif value is None:
            value = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFString(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFString(SFString(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFString(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ str(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFString):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFString):
                self.__value.append(SFString(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFString):
                for each in value:
                    self.__value.append(SFString(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFString(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFString(value):
    ###             if isinstance(value, SFString):
    ###                 value = SFString(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFString(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFString):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFString(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFString self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        result = ''
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                result += '"' + str(each).replace("'","&apos;").replace("& ","&amp; ").replace("<","&lt;").replace(">","&gt;") + '"' + ' '
        result = result.rstrip(' ')
        return result
    def VRML(self):
        """ Provide VRML value for this field type. """
        result = ''
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                result += '"' + str(each) + '"' + ' '
        result = '[' + result.rstrip(' ') + ']'
        return result
    def JSON(self):
        """ Provide JSON value for this field type. """
        result = ''
        if self.__value: # walk each child in list, if any (avoid empty list recursion)
            for each in self.__value:
                result += '"' + str(each).replace('"','&quot;') + '"' + ' '
        result = result.rstrip(' ')
        return result

class SFTime(_X3DField):
    """
    Field type SFTime specifies a single time value, expressed as a double-precision floating point number. Typically, SFTime fields represent the number of seconds since Jan 1, 1970, 00:00:00 GMT.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFTime'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFTimeAndMFTime'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFTime'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return -1.0
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=-1.0):
        # print('*** SFTime __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFTime):
            value = value.value # dereference
        elif value is None:
            value = SFTime.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFTime.DEFAULT_VALUE()=' + str(SFTime.DEFAULT_VALUE()))
        elif isinstance(value, MFTime) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFTime encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFTime self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFTime(_X3DArrayField):
    """
    Field type MFTime is an array of SFTime values. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFTime'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFTimeAndMFTime'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFTime'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 1
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFTime __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed and sized values. """
        if isinstance(value,SFTime):
            value = value.value # dereference
        elif value is None:
            value = MFTime.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFTime.DEFAULT_VALUE()=' + str(MFTime.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFTime(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFTime(SFTime(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFTime(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFTime):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFTime):
                self.__value.append(SFTime(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFTime):
                for each in value:
                    self.__value.append(SFTime(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFTime(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFTime(value):
    ###             if isinstance(value, SFTime):
    ###                 value = SFTime(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFTime(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFTime):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFTime(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFTime self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFVec2d(_X3DField):
    """
    Field type SFVec2d is a 2-tuple pair of SFDouble values. Hint: SFVec2d can be used to specify a 2D double-precision coordinate. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFVec2d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec2dAndMFVec2d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0.0, 0.0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 2
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None):
        # print('*** SFVec2d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None:
            value = (float(value),float(value2))
        if isinstance(value,SFVec2d):
            value = value.value # dereference
        elif value is None:
            value = SFVec2d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFVec2d.DEFAULT_VALUE()=' + str(SFVec2d.DEFAULT_VALUE()))
        elif isinstance(value, MFVec2d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFVec2d encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFVec2d self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFVec2d(_X3DArrayField):
    """
    Field type MFVec2d is an array of SFVec2d values. Individual singleton SFVec2d array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFVec2d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec2dAndMFVec2d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 2
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFVec2d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None:
            value = (float(value),float(value2))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None:
            value = (float(value),float(value2))
        if isinstance(value,SFVec2d):
            value = value.value # dereference
        elif value is None:
            value = MFVec2d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFVec2d.DEFAULT_VALUE()=' + str(MFVec2d.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFVec2d(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFVec2d(SFVec2d(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFVec2d(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFVec2d):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFVec2d):
                self.__value.append(SFVec2d(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFVec2d):
                for each in value:
                    self.__value.append(SFVec2d(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFVec2d(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFVec2d(value):
    ###             if isinstance(value, SFVec2d):
    ###                 value = SFVec2d(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFVec2d(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFVec2d):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFVec2d(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFVec2d self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFVec2f(_X3DField):
    """
    Field type SFVec2f is a 2-tuple pair of SFFloat values. Hint: SFVec2f can be used to specify a 2D single-precision coordinate. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFVec2f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec2fAndMFVec2f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0.0, 0.0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 2
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None):
        # print('*** SFVec2f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None:
            value = (float(value),float(value2))
        if isinstance(value,SFVec2f):
            value = value.value # dereference
        elif value is None:
            value = SFVec2f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFVec2f.DEFAULT_VALUE()=' + str(SFVec2f.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 2-tuples
                value = [(x, y) for x, y, in value]
        elif isinstance(value, MFVec2f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFVec2f encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFVec2f self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFVec2f(_X3DArrayField):
    """
    Field type MFVec2f is an array of SFVec2f values. Individual singleton SFVec2f array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFVec2f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec2fAndMFVec2f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 2
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFVec2f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None:
            value = (float(value),float(value2))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None:
            value = (float(value),float(value2))
        if isinstance(value,SFVec2f):
            value = value.value # dereference
        elif value is None:
            value = MFVec2f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFVec2f.DEFAULT_VALUE()=' + str(MFVec2f.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 2-tuples
                value = [(x, y) for x, y, in value]
    ###     elif not isinstance(value, list) and isValidSFVec2f(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFVec2f(SFVec2f(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFVec2f(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFVec2f):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFVec2f):
                self.__value.append(SFVec2f(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFVec2f):
                for each in value:
                    self.__value.append(SFVec2f(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFVec2f(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFVec2f(value):
    ###             if isinstance(value, SFVec2f):
    ###                 value = SFVec2f(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFVec2f(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFVec2f):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFVec2f(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFVec2f self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFVec3d(_X3DField):
    """
    Field type SFVec3d is a 3-tuple triplet of SFDouble values. See GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision. Hint: SFVec3d can be used to specify a georeferenced 3D coordinate. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFVec3d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec3dAndMFVec3d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0.0, 0.0, 0.0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 3
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None):
        # print('*** SFVec3d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        if isinstance(value,SFVec3d):
            value = value.value # dereference
        elif value is None:
            value = SFVec3d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFVec3d.DEFAULT_VALUE()=' + str(SFVec3d.DEFAULT_VALUE()))
        elif isinstance(value, MFVec3d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFVec3d encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFVec3d self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFVec3d(_X3DArrayField):
    """
    Field type MFVec3d is an array of SFVec3d values. Individual singleton SFVec3d array values are optionally separated by commas in XML syntax. Original rationale for inclusion: GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision. Hint: MFVec3d can be used to specify a list of georeferenced 3D coordinates.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFVec3d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec3dAndMFVec3d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 3
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFVec3d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        if isinstance(value,SFVec3d):
            value = value.value # dereference
        elif value is None:
            value = MFVec3d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFVec3d.DEFAULT_VALUE()=' + str(MFVec3d.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFVec3d(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFVec3d(SFVec3d(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFVec3d(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFVec3d):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFVec3d):
                self.__value.append(SFVec3d(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFVec3d):
                for each in value:
                    self.__value.append(SFVec3d(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFVec3d(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFVec3d(value):
    ###             if isinstance(value, SFVec3d):
    ###                 value = SFVec3d(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFVec3d(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFVec3d):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFVec3d(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFVec3d self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFVec3f(_X3DField):
    """
    Field type SFVec3f is a 3-tuple triplet of SFFloat values. Hint: SFVec3f can be used to specify a 3D coordinate or a 3D scale value. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFVec3f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec3fAndMFVec3f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0.0, 0.0, 0.0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 3
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None):
        # print('*** SFVec3f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        if isinstance(value,SFVec3f):
            value = value.value # dereference
        elif value is None:
            value = SFVec3f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFVec3f.DEFAULT_VALUE()=' + str(SFVec3f.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif isinstance(value, MFVec3f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFVec3f encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFVec3f self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFVec3f(_X3DArrayField):
    """
    Field type MFVec3f is an array of SFVec3f values. Individual singleton SFVec3f array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFVec3f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec3fAndMFVec3f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 3
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFVec3f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None:
            value = (float(value),float(value2),float(value3))
        if isinstance(value,SFVec3f):
            value = value.value # dereference
        elif value is None:
            value = MFVec3f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFVec3f.DEFAULT_VALUE()=' + str(MFVec3f.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
    ###     elif not isinstance(value, list) and isValidSFVec3f(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFVec3f(SFVec3f(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFVec3f(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFVec3f):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFVec3f):
                self.__value.append(SFVec3f(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFVec3f):
                for each in value:
                    self.__value.append(SFVec3f(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFVec3f(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFVec3f(value):
    ###             if isinstance(value, SFVec3f):
    ###                 value = SFVec3f(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFVec3f(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFVec3f):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFVec3f(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFVec3f self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFVec4d(_X3DField):
    """
    Field type SFVec4d is a 4-tuple set of double-precision floating-point values, specifying a 3D homogeneous vector. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFVec4d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec4dAndMFVec4d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0.0, 0.0, 0.0, 0.0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        # print('*** SFVec4d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFVec4d):
            value = value.value # dereference
        elif value is None:
            value = SFVec4d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFVec4d.DEFAULT_VALUE()=' + str(SFVec4d.DEFAULT_VALUE()))
        elif isinstance(value, MFVec4d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFVec4d encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFVec4d self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFVec4d(_X3DArrayField):
    """
    Field type MFVec4d is zero or more SFVec4d values. Individual singleton SFVec4d array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFVec4d'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec4dAndMFVec4d'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4d'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFVec4d __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFVec4d):
            value = value.value # dereference
        elif value is None:
            value = MFVec4d.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFVec4d.DEFAULT_VALUE()=' + str(MFVec4d.DEFAULT_VALUE()))
    ###     elif not isinstance(value, list) and isValidSFVec4d(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFVec4d(SFVec4d(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFVec4d(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFVec4d):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFVec4d):
                self.__value.append(SFVec4d(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFVec4d):
                for each in value:
                    self.__value.append(SFVec4d(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFVec4d(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFVec4d(value):
    ###             if isinstance(value, SFVec4d):
    ###                 value = SFVec4d(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFVec4d(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFVec4d):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFVec4d(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFVec4d self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

class SFVec4f(_X3DField):
    """
    Field type SFVec4f is a 4-tuple set of single-precision floating-point values, specifying a 3D homogeneous vector. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'SFVec4f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec4fAndMFVec4f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return (0.0, 0.0, 0.0, 0.0)
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return False
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\(\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\)\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        # print('*** SFVec4f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if isinstance(value,str):
            value = float(value)
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFVec4f):
            value = value.value # dereference
        elif value is None:
            value = SFVec4f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFVec4f.DEFAULT_VALUE()=' + str(SFVec4f.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFVec4f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        # https://stackoverflow.com/questions/354038/how-do-i-check-if-a-string-is-a-number-float
        if isinstance(value, str):
            try:
                float(value) # this statement checks but does not set value, may throw exception
                print('*** string value provided, value=' + str(value) + ', float(value)=' + str(float(value)), flush=True)
                value = float(value)
            except ValueError as error:
                # https://stackoverflow.com/questions/66995878/consider-explicitly-re-raising-using-the-from-keyword-pylint-suggestion
                raise X3DTypeError('SFVec4f encountered string with illegal value=' + str(value)) from error
        self.__value = value
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, SFVec4f self.__value type=' + str(type(self.__value)) + ' is not a list')
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace(',', '').replace('(', '').replace(')', '').replace('[', '').replace(']', '')

class MFVec4f(_X3DArrayField):
    """
    Field type MFVec4f is zero or more SFVec4f values. Individual singleton SFVec4f array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Field class. """
        return 'MFVec4f'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/fieldsDef.html#SFVec4fAndMFVec4f'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4f'
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value defined for this data type by the X3D Specification """
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    @classmethod
    def ARRAY_TYPE(cls):
        """ Whether or not this field class is array based. """
        return True
    @classmethod
    def TUPLE_SIZE(cls):
        """ How many values make up each data tuple. """
        return 4
    @classmethod
    def REGEX_PYTHON(cls):
        """ Regular expression for validating Python values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*\[?\s*(\s*\(?\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*\,?\s*){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*\)?\s*\,?)*\s*\]?\s*'
    @classmethod
    def REGEX_XML(cls):
        """ Regular expression for validating XML values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*'
    # - - - - - - - - - -
    def __init__(self, value=None,value2=None,value3=None,value4=None):
        if value is None:
            value = self.DEFAULT_VALUE()
        # print('*** MFVec4f __init__ value=' + str(value), 'type=' + str(type(value))) # debug
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value,value2=None,value3=None,value4=None):
        """ The value setter only allows correctly typed and sized values. """
        if value2 is not None and value3 is not None and value4 is not None:
            value = (float(value),float(value2),float(value3),float(value4))
        if isinstance(value,SFVec4f):
            value = value.value # dereference
        elif value is None:
            value = MFVec4f.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFVec4f.DEFAULT_VALUE()=' + str(MFVec4f.DEFAULT_VALUE()))
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
    ###     elif not isinstance(value, list) and isValidSFVec4f(value):
    ###         print(' upcast to MF type', value)
    ###         value = MFVec4f(SFVec4f(value))
        elif isinstance(value, list):
            if not value is None and not (isinstance(value,list) and len(value) == 0):
                _newValue = []
                for each in value:
                    _newValue.append(SFVec4f(each).value)
                # if _DEBUG: print('...DEBUG... assign list, value=' + str(value), ', type=' + str(type(value)), ', _newValue=' + str(_newValue),flush=True)
                value = _newValue
        elif isinstance(value, str):
            value = [ float(value) ]
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            # if _DEBUG: print('...DEBUG... append to list, value=' + str(self.__value), ', type=' + str(type(self.__value)), ', value=' + str(value),flush=True)
            if isinstance(value,SFVec4f):
                self.__value.append(value.value) # dereference
            elif not isinstance(value,list) and not isinstance(value,MFVec4f):
                self.__value.append(SFVec4f(value).value) # checks validity
            elif (isinstance(value,list) and len(value) > 0) or isinstance(value,MFVec4f):
                for each in value:
                    self.__value.append(SFVec4f(each).value) # checks validity
            elif isinstance(value,str):
                self.__value.append(SFVec4f(value).value) # checks validity
    ###     if  not value is None:
    ###         if isValidSFVec4f(value):
    ###             if isinstance(value, SFVec4f):
    ###                 value = SFVec4f(value).value # dereference value from base type
    ###             self.__value.append(value)
    ###         elif isValidMFVec4f(value):
    ###             for each in value:
    ###                 while isinstance(each, list) and len(each) == 1:
    ###                     each = each[0] # dereference
    ###                 if isinstance(each, SFVec4f):
    ###                     each = each.value # dereference
    ###                 self.__value.append(each)
    ###         else:
    ###             assertValidMFVec4f(value) # report type failure
    def __bool__(self):
        if not isinstance(self.__value,list):
            print('*** x3d.py internal error, MFVec4f self.__value type=' + str(type(self.__value)) + ' is not a list', flush=True)
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '') + ']'
    def JSON(self):
        """ Provide JSON value for this field type. """
        return str(self.__value).lower().replace('(', '').replace(')', '').replace(',', '').replace('[', '').replace(']', '')

###############################################

# Abstract Node Types

# Note that these package-internal class names are preceded by an underscore _ character for hidden scope, since X3D authors are not expected to use them.

class _X3DNode:
    """
    All instantiable nodes implement X3DNode, which corresponds to SFNode type in the X3D specification.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#X3DNode'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html'
    # Field declarations for this node are performed by implementing node
    def __init__(self, DEF="", USE="", class_="", id_="", style_="", IS=None, metadata=None):
        self.DEF = DEF
        self.USE = USE
        self.class_ = class_
        self.id_ = id_
        self.style_ = style_
        self.IS = IS
        self.metadata = metadata
        # if _DEBUG: print('...DEBUG... in X3DNode __init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
    @property # getter - - - - - - - - - -
    def DEF(self):
        """ Unique ID name for this node, referenceable by other X3D nodes. """
        return self.__DEF
    @DEF.setter
    def DEF(self, DEF):
        if  DEF is None:
            DEF = SFString.DEFAULT_VALUE()
        assertValidSFString(DEF)
        self.__DEF = str(DEF)
        if self.__DEF:
            self.__USE = None # DEF and USE are mutually exclusive
    @property # getter - - - - - - - - - -
    def USE(self):
        """ Reuse an already DEF-ed node ID, excluding all child nodes and all other attributes. """
        return self.__USE
    @USE.setter
    def USE(self, USE):
        if  USE is None:
            USE = SFString.DEFAULT_VALUE()
        assertValidSFString(USE)
        self.__USE = str(USE)
        if self.__USE:
            self.__DEF = None # DEF and USE are mutually exclusive
    @property # getter - - - - - - - - - -
    def class_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading stylesheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__class_
    @class_.setter
    def class_(self, class_):
        if  class_ is None:
            class_ = SFString.DEFAULT_VALUE()
        assertValidSFString(class_)
        self.__class_ = class_
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
        assertValidSFString(id_)
        self.__id = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
        assertValidSFString(style_)
        self.__style_ = style_
    @property # getter - - - - - - - - - -
    def IS(self):
        """ The IS statement connects node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields. """
        return self.__IS
    @IS.setter
    def IS(self, IS):
        if  IS is None:
            self.__IS = SFNode.DEFAULT_VALUE()
        assertValidSFNode(IS)
        if not isinstance(IS, object) and not isinstance(IS, [ IS ] ): # TODO disambiguate IS naming
            # print(flush=True)
            raise X3DTypeError(str(IS) + ' IS.setter does not have a valid node type object, must be an IS node')
        self.__IS = IS
    @property # getter - - - - - - - - - -
    def metadata(self):
        """ The metadata field can contain a single MetadataBoolean, MetadataInteger, MetadataFloat, MetadataDouble, MetadataString or MetadataSet node. """
        return self.__metadata
    @metadata.setter
    def metadata(self, metadata):
        if  metadata is None:
            metadata = SFNode.DEFAULT_VALUE()
        assertValidSFNode(metadata)
        if not isinstance(metadata, object) and not isinstance(metadata, ( MetadataBoolean, MetadataInteger, MetadataFloat, MetadataDouble, MetadataString, MetadataSet, ProtoInstance ) ):
            # print(flush=True)
            raise X3DTypeError(str(metadata) + ' metadata.setter does not have a valid node type object, must be a Metadata* node or ProtoInstance')
        self.__metadata = metadata
    def __repl__(self):
        result = self.NAME() + '('
        # TODO put DEF first, matching canonical form
        if self.FIELD_DECLARATIONS():
            for each in self.FIELD_DECLARATIONS():
                # if _DEBUG: print(self.NAME() + ' for each in self.FIELD_DECLARATIONS(): each=' + str(each))
                name = each[0]
                default = each[1]
                type_ = each[2]
                accessType = each[3]
                value = getattr(self, name)
                # if _DEBUG: print('gettattr(self, ' + str(name) + ') value="' + str(value)[:100] + '" for FIELD_DECLARATIONS() ' + str(each) + ')', flush=True)
                if value != default:
                    # consider whether indentation is useful; probably not
                    # print("\n\t")
                    if  isinstance(value, list): # avoid X3DTypeError if value is not iterable
                        result += str(name) + '=['
                        for each in value:
                            # if _DEBUG: print('...DEBUG... X3DNode debug: str(each)=' + str(each), flush=True)
                            result += str(each) + ', '
                        result = result.rstrip(', ')
                        result += '],'
                    elif isinstance(value, str) and "'" in value:
                        result += str(name) + '=' + '"' + str(value)[:100] + '"' + ','
                    elif isinstance(value, str) and value != default:
                        result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
                    elif value != default:
                        result += str(name) + '='       + str(value)[:100]       + ','
                    # elif _DEBUG:
                    #   result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
        return result.strip().rstrip(',').rstrip(', ') + ')'
    def __str__(self):
        return self.__repl__().strip() # X3DNode

class _X3DChildNode(_X3DNode):
    """
    A node that implements X3DChildNode is one of the legal children for a X3DGroupingNode parent.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DChildNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#X3DChildNode'

class _X3DSoundNode(_X3DChildNode):
    """
    Base type for all sound nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSoundNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#X3DSoundNode'

class _X3DSoundChannelNode(_X3DSoundNode):
    """
    Base type for all sound destination nodes, which represent the final destination of an audio signal and are what the user can ultimately hear.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSoundChannelNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#X3DSoundChannelNode'

class _X3DTimeDependentNode(_X3DChildNode):
    """
    Base type from which all time-dependent nodes are derived.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTimeDependentNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/time.html#X3DTimeDependentNode'

class _X3DGeometryNode(_X3DNode):
    """
    Geometry nodes produce renderable geometry and are contained by a Shape node.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DGeometryNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rendering.html#X3DGeometryNode'

class _X3DParametricGeometryNode(_X3DGeometryNode):
    """
    Base type for all geometry node types that are created parametrically and use control points to describe the final shape of the surface.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DParametricGeometryNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/nurbs.html#X3DParametricGeometryNode'

class _X3DAppearanceChildNode(_X3DNode):
    """
    Nodes of this type can be used as child nodes for Appearance.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DAppearanceChildNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#X3DAppearanceChildNode'

class _X3DTextureNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes which specify sources for texture images.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTextureNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DTextureNode'

class _X3DSensorNode(_X3DChildNode):
    """
    Base type for all sensors.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#X3DSensorNode'

class _X3DPointingDeviceSensorNode(_X3DSensorNode):
    """
    Base type for all pointing device sensors.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DPointingDeviceSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/pointingDeviceSensor.html#X3DPointingDeviceSensorNode'

class _X3DVolumeRenderStyleNode(_X3DNode):
    """
    The X3DVolumeRenderStyleNode abstract node type is the base type for all node types that specify a specific visual rendering style to be used when rendering volume data.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DVolumeRenderStyleNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/volume.html#X3DVolumeRenderStyleNode'

class _X3DGeometricPropertyNode(_X3DNode):
    """
    Base type for all geometric property node types.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DGeometricPropertyNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rendering.html#X3DGeometricPropertyNode'

class _X3DFollowerNode(_X3DChildNode):
    """
    X3DFollowerNode is the abstract base class for all nodes in the Followers component.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DFollowerNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/followers.html#X3DFollowerNode'

class _X3DBindableNode(_X3DChildNode):
    """
    Bindable nodes implement the binding stack, so that only one of each node type is active at a given time.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DBindableNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#X3DBindableNode'

class _X3DAppearanceNode(_X3DNode):
    """
    Base type for all Appearance nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DAppearanceNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#X3DAppearanceNode'

class _X3DBackgroundNode(_X3DBindableNode):
    """
    Abstract type from which all backgrounds inherit, also defining a background binding stack.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DBackgroundNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/environmentalEffects.html#X3DBackgroundNode'

class _X3DChaserNode(_X3DFollowerNode):
    """
    The X3DChaserNode abstract node type calculates the output on value_changed as a finite impulse response (FIR) based on the events received on set_destination field.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DChaserNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/followers.html#X3DChaserNode'

class _X3DColorNode(_X3DGeometricPropertyNode):
    """
    Base type for color specifications in X3D.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DColorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rendering.html#X3DColorNode'

class _X3DComposableVolumeRenderStyleNode(_X3DVolumeRenderStyleNode):
    """
    The X3DComposableVolumeRenderStyleNode abstract node type is the base type for all node types that allow rendering styles to be sequentially composed together to form a single renderable output.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DComposableVolumeRenderStyleNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/volume.html#X3DComposableVolumeRenderStyleNode'

class _X3DComposedGeometryNode(_X3DGeometryNode):
    """
    Composed geometry nodes produce renderable geometry, can contain Color Coordinate Normal TextureCoordinate, and are contained by a Shape node.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DComposedGeometryNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rendering.html#X3DComposedGeometryNode'

class _X3DCoordinateNode(_X3DGeometricPropertyNode):
    """
    Base type for all coordinate node types in X3D.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DCoordinateNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rendering.html#X3DCoordinateNode'

class _X3DDamperNode(_X3DFollowerNode):
    """
    The X3DDamperNode abstract node type creates an IIR response that approaches the destination value according to the shape of the e-function only asymptotically but very quickly.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DDamperNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/followers.html#X3DDamperNode'

class _X3DDragSensorNode(_X3DPointingDeviceSensorNode):
    """
    Base type for all drag-style pointing device sensors.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DDragSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/pointingDeviceSensor.html#X3DDragSensorNode'

class _X3DEnvironmentalSensorNode(_X3DSensorNode):
    """
    Base type for the environmental sensor nodes ProximitySensor, TransformSensor and VisibilitySensor.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DEnvironmentalSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/environmentalSensor.html#X3DEnvironmentalSensorNode'

class _X3DEnvironmentTextureNode(_X3DTextureNode):
    """
    Base type for all nodes that specify cubic environment map sources for texture images.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DEnvironmentTextureNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/environmentalTexturing#X3DEnvironmentTextureNode'

class _X3DFontStyleNode(_X3DNode):
    """
    Base type for all font style nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DFontStyleNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/text.html#X3DFontStyleNode'

class _X3DGroupingNode(_X3DChildNode):
    """
    Grouping nodes can contain other nodes as children, thus making up the backbone of a scene graph.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DGroupingNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/grouping.html#X3DGroupingNode'

class _X3DInfoNode(_X3DChildNode):
    """
    Base type for all nodes that contain only information without visual semantics.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DInfoNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/grouping.html#X3DInfoNode'

class _X3DInterpolatorNode(_X3DChildNode):
    """
    Interpolator nodes are designed for linear keyframed animation. Interpolators are driven by an input key ranging [0..1] and produce corresponding piecewise-linear output functions.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DInterpolatorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/interpolators.html#X3DInterpolatorNode'

class _X3DKeyDeviceSensorNode(_X3DSensorNode):
    """
    Base type for all sensor node types that operate using key devices.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DKeyDeviceSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/keyDeviceSensor.html#X3DKeyDeviceSensorNode'

class _X3DLayerNode(_X3DNode):
    """
    The X3DLayerNode abstract node type is the base node type for layer nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DLayerNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/layering.html#X3DLayerNode'

class _X3DLayoutNode(_X3DChildNode):
    """
    X3DLayoutNode is the base node type for layout nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DLayoutNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/layout.html#X3DLayoutNode'

class _X3DLightNode(_X3DChildNode):
    """
    Light nodes provide illumination for rendering geometry in the scene. Implementing nodes must include a global field with type SFBool and accessType inputOutput.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DLightNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/lighting.html#X3DLightNode'

class _X3DMaterialNode(_X3DAppearanceChildNode):
    """
    Base type for all Material nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DMaterialNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#X3DMaterialNode'

class _X3DNBodyCollidableNode(_X3DChildNode):
    """
    The X3DNBodyCollidableNode abstract node type represents objects that act as the interface between the rigid body physics, collision geometry proxy, and renderable objects in the scene graph hierarchy.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNBodyCollidableNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rigidBodyPhysics.html#X3DNBodyCollidableNode'

class _X3DNBodyCollisionSpaceNode(_X3DNode):
    """
    The X3DNBodyCollisionSpaceNode abstract node type represents objects that act as a self-contained spatial collection of objects that can interact through collision detection routines.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNBodyCollisionSpaceNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rigidBodyPhysics.html#X3DNBodyCollisionSpaceNode'

class _X3DNetworkSensorNode(_X3DSensorNode):
    """
    Base typefor all sensors that generate events based on network activity.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNetworkSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/networking.html#X3DNetworkSensorNode'

class _X3DNormalNode(_X3DGeometricPropertyNode):
    """
    Base type for all normal node types in X3D.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNormalNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rendering.html#X3DNormalNode'

class _X3DNurbsControlCurveNode(_X3DNode):
    """
    Base type for all nodes that provide control curve information in 2D space.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNurbsControlCurveNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/nurbs.html#X3DNurbsControlCurveNode'

class _X3DNurbsSurfaceGeometryNode(_X3DParametricGeometryNode):
    """
    Abstract geometry type for all types of NURBS surfaces.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DNurbsSurfaceGeometryNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/nurbs.html#X3DNurbsSurfaceGeometryNode'

class _X3DOneSidedMaterialNode(_X3DMaterialNode):
    """
    Base type for material nodes that describe how the shape looks like from one side. A different number of contanied texture nodes are allowed by each of the implementing nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DOneSidedMaterialNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#X3DOneSidedMaterialNode'

class _X3DParticleEmitterNode(_X3DNode):
    """
    The X3DParticleEmitterNode abstract type represents any node that is an emitter of particles.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DParticleEmitterNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/particleSystems.html#X3DParticleEmitterNode'

class _X3DParticlePhysicsModelNode(_X3DNode):
    """
    The X3DParticlePhysicsModelNode abstract type represents any node that applies a form of constraints on the particles after they have been generated.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DParticlePhysicsModelNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/particleSystems.html#X3DParticlePhysicsModelNode'

class _X3DPickSensorNode(_X3DSensorNode):
    """
    The X3DPickSensorNode abstract node type is the base node type that represents the lowest common denominator of picking capabilities.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DPickSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/picking.html#X3DPickSensorNode'

class _X3DProductStructureChildNode(_X3DChildNode):
    """
    Base type marking nodes that are valid product structure children for the CADGeometry component.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DProductStructureChildNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/CADGeometry.html#X3DProductStructureChildNode'

class _X3DPrototypeInstance(_X3DNode):
    """
    Base type for all prototype instances. Note that direct children nodes are disallowed, instead let fieldValue with type SFNode/MFNode contain them. Current practice is that, if desired, prototype authors must explicitly add the metadata SFNode field in the ProtoInterface.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DPrototypeInstance'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#X3DPrototypeInstance'

class _X3DRigidJointNode(_X3DNode):
    """
    The X3DRigidJointNode abstract node type is the base type for all joint types.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DRigidJointNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rigidBodyPhysics.html#X3DRigidJointNode'

class _X3DScriptNode(_X3DChildNode):
    """
    Base type for scripting nodes (but not shader nodes).
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DScriptNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/scripting.html#X3DScriptNode'

class _X3DSequencerNode(_X3DChildNode):
    """
    Base type from which all Sequencers are derived.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSequencerNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/eventUtilities.html#X3DSequencerNode'

class _X3DShaderNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes that specify a programmable shader.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DShaderNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shaders.html#X3DShaderNode'

class _X3DShapeNode(_X3DChildNode):
    """
    Base type for all Shape nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DShapeNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#X3DShapeNode'

class _X3DSoundDestinationNode(_X3DSoundNode):
    """
    Base type for all sound destination nodes, which represent the final destination of an audio signal and are what the user can ultimately hear.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSoundDestinationNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#X3DSoundDestinationNode'

class _X3DSoundProcessingNode(_X3DTimeDependentNode):
    """
    Base type for all sound processing nodes, which are used to enhance audio with filtering, delaying, changing gain, etc.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSoundProcessingNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#X3DSoundProcessingNode'

class _X3DSoundSourceNode(_X3DTimeDependentNode):
    """
    Nodes implementing X3DSoundSourceNode provide signal inputs to the audio graph.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSoundSourceNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#X3DSoundSourceNode'

class _X3DTexture3DNode(_X3DTextureNode):
    """
    Base type for all nodes that specify 3D sources for texture images.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTexture3DNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texture3D.html#X3DTexture3DNode'

class _X3DTextureProjectorNode(_X3DLightNode):
    """
    Base type for all node types that specify texture projector nodes, which provide a form of lighting.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTextureProjectorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/TextureProjector.html#X3DTextureProjectorNode'

class _X3DTouchSensorNode(_X3DPointingDeviceSensorNode):
    """
    Base type for all touch-style pointing device sensors.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTouchSensorNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/pointingDeviceSensor.html#X3DTouchSensorNode'

class _X3DTriggerNode(_X3DChildNode):
    """
    Base type from which all trigger nodes are derived.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTriggerNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/eventUtilities.html#X3DTriggerNode'

class _X3DVertexAttributeNode(_X3DGeometricPropertyNode):
    """
    Base type for all nodes that specify per-vertex attribute information to the shader.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DVertexAttributeNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shaders.html#X3DVertexAttributeNode'

class _X3DViewpointNode(_X3DBindableNode):
    """
    Node type X3DViewpointNode defines a specific location in the local coordinate system from which the user may view the scene, and also defines a viewpoint binding stack.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DViewpointNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/navigation.html#X3DViewpointNode'

class _X3DViewportNode(_X3DGroupingNode):
    """
    The X3DViewportNode abstract node type is the base node type for viewport nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DViewportNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/layering.html#X3DViewportNode'

class _X3DVolumeDataNode(_X3DChildNode):
    """
    The X3DVolumeDataNode abstract node type is the base type for all node types that describe volumetric data to be rendered.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DVolumeDataNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/volume.html#X3DVolumeDataNode'

class _X3DTextureTransformNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes which specify a transformation of texture coordinates.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTextureTransformNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DTextureTransformNode'

class _X3DSingleTextureTransformNode(_X3DTextureTransformNode):
    """
    Base type for all texture transform nodes which specify texture coordinate transformation for a single texture.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSingleTextureTransformNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DSingleTextureTransformNode'

class _X3DTextureCoordinateNode(_X3DGeometricPropertyNode):
    """
    Base type for all nodes which specify texture coordinates.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTextureCoordinateNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DTextureCoordinateNode'

class _X3DSingleTextureCoordinateNode(_X3DTextureCoordinateNode):
    """
    Base type for all texture coordinate nodes which specify texture coordinates for a single texture.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSingleTextureCoordinateNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DSingleTextureCoordinateNode'

class _X3DSingleTextureNode(_X3DTextureNode):
    """
    Base type for all texture node types that define a single texture. A single texture can be used to influence a parameter of various material nodes in the Shape component, and it can be a child of MultiTexture.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DSingleTextureNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DSingleTextureNode'

class _X3DTexture2DNode(_X3DSingleTextureNode):
    """
    Base type for all nodes which specify 2D sources for texture images.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DTexture2DNode'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/texturing.html#X3DTexture2DNode'

###############################################

# Abstract Object Types

# Note that these package-internal class names are preceded by an underscore _ character since X3D authors are not expected to use them

class _X3DBoundedObject(_X3DNode):
    """
    X3DBoundedObject indicates that bounding box values can be provided (or computed) to encompass this node and any children.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DBoundedObject'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/grouping.html#X3DBoundedObject'

class _X3DFogObject(_X3DNode):
    """
    Abstract type describing a node that influences the lighting equation through the use of fog semantics.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DFogObject'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/environmentalEffects.html#X3DFogOjbect'

class _X3DMetadataObject(_X3DNode):
    """
    Each node inheriting the X3DMetadataObject interface contains a single array of strictly typed values: MFBool, MFInt32, MFFloat, MFDouble, MFString, or MFNode, the latter having children that are all Metadata nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DMetadataObject'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#X3DMetadataObject'

class _X3DPickableObject(_X3DNode):
    """
    The X3DPickableObject abstract interface marks a node as being capable of having customized picking performed on its contents or children.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DPickableObject'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/picking.html#X3DPickableObject'

class _X3DProgrammableShaderObject(_X3DNode):
    """
    Base type for all nodes that specify arbitrary fields for interfacing with per-object attribute values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DProgrammableShaderObject'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shaders.html#X3DProgrammableShaderObject'

class _X3DUrlObject(_X3DNode):
    """
    X3DUrlObject indicates that a node has content loaded from a Uniform Resource Locator (URL) and can be tracked via a LoadSensor. Such child nodes have containerField='children' to indicate their relationship to the parent LoadSensor node.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Abstract Type class. """
        return '_X3DUrlObject'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/networking.html#X3DUrlObject'

###############################################

# Statements

class _X3DStatement:
    """
    All X3D statements implement _X3DStatement abstract type.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Statement class. """
        return '_X3DStatement'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return []
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/components/core.html#AbstractX3DStructure'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html'
    def __init__(self, class_="", id_="", style_=""):
        self.class_ = class_
        self.id_ = id_
        self.style_ = style_
        # if _DEBUG: print('...DEBUG... in X3DNode __init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + ')', flush=True)
    @property # getter - - - - - - - - - -
    def class_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading stylesheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__class_
    @class_.setter
    def class_(self, class_):
        if  class_ is None:
            class_ = SFString.DEFAULT_VALUE()
        assertValidSFString(class_)
        self.__class_ = class_
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
        assertValidSFString(style_)
        self.__style_ = style_
    def __repl__(self):
        result = self.NAME() + '('
        # if _DEBUG: print(self.NAME() + ' self.FIELD_DECLARATIONS(): ' + str(self.FIELD_DECLARATIONS))
        if self.FIELD_DECLARATIONS():
            for each in self.FIELD_DECLARATIONS():
                # if _DEBUG: print(self.NAME() + ' for each in self.FIELD_DECLARATIONS(): each=' + str(each))
                name = each[0]
                default = each[1]
                type_ = each[2]
                accessType = each[3]
                value = getattr(self, name)
                # if _DEBUG: print('gettattr(self, ' + str(name) + ') value="' + str(value)[:100] + '" for FIELD_DECLARATIONS() ' + str(each) + ')', flush=True)
                if value != default:
                    if  isinstance(value, list): # avoid X3DTypeError if value is not iterable
                        result += str(name) + '=['
                        for each in value:
                            result += str(each) + ', '
                            # if _DEBUG: print('...DEBUG... _X3DStatement debug: str(each)=' + str(each), flush=True)
                        result = result.rstrip(', ')
                        result += '],'
                    elif isinstance(value, str) and "'" in value:
                        result += str(name) + '=' + '"' + str(value)[:100] + '"' + ','
                    elif isinstance(value, str) and value != default:
                        result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
                    elif value != default:
                        result += str(name) + '='       + str(value)[:100]       + ','
                    # elif _DEBUG:
        #   result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
        return result.strip().rstrip(',').rstrip(', ') + ')'
    def __str__(self):
        return self.__repl__().strip() # _X3DStatement

def isX3DStatement(value):
    """
    Whether or not value is an _X3DStatement object.
    """
    return isinstance(value, _X3DStatement)

class Comment(_X3DStatement):
    """
    X3D statement containing zero or more comment strings.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Statement class. """
        return 'Comment'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#Organization'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html'
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def DEFAULT_VALUE(cls):
        """ Default value for comments is empty string """
        return ''
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return []
    @classmethod
    def REGEX_PYTHON(cls):
        """' Regular expression for validating values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'(\s|\S)*' # (includes lower-case true, false)
    @classmethod
    def REGEX_XML(cls):
        """' Regular expression for validating values, for more information see https://www.web3d.org/specifications/X3dRegularExpressions.html """
        return r'(\s|\S)*' # (includes lower-case true, false)
    def __init__(self, value=''):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide list of comment strings. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = SFString.DEFAULT_VALUE()
        self.__value = str(value)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ <!-- XML comments are wrapped in special delimiters --> """
        result = ''
        indent = '  ' * indentLevel
        if self.value:
            result = indent + '<!-- ' + self.value + ' -->' + '\n'
        return result
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ VRML comments begin with # effective to end of line """
        result = ''
        indent = '  ' * indentLevel
        if self.value:
            result = '\n' + indent + '# ' + self.value + '\n' + indent
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ JSON does not support comments, so X3D JSON created a specific object for them """
        result = ''
        indent = '  ' * indentLevel
        if self.value:
            result = indent + '{ "#comment" : "' + self.value + '" }' + '\n'
        return result

def isComment(value):
    """
    Whether or not value is a Comment object.
    """
    return isinstance(value, Comment)

class component(_X3DStatement):
    """
    Functional summary: each added component statement indicates needed scene functionality support above the given X3D profile.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'component'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/concepts.html#Components'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#component'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('level', 1, FieldType.SFInt32, AccessType.inputOutput, 'component'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'component'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        level=1,
        name='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement component __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.level = level
        self.name = name
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def level(self):
        """Necessary level of support for this scene, as defined in corresponding Support table for a given node's component."""
        return self.__level
    @level.setter
    def level(self, level):
        if  level is None:
            level = 1 # default
        assertValidSFInt32(level)
        assertGreaterThanEquals('level', level, 1)
        assertLessThanEquals('level', level, 5)
        self.__level = level
    @property # getter - - - - - - - - - -
    def name(self):
        """Provides name of this component, as defined in corresponding X3D Specification component Introduction."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        assertValidComponentName('name', name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function component.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<component'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.level != 1:
            result += " level='" + SFInt32(self.level).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></component>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</component>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function component.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"component":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.level != 1:
            attributeResult += "        " + '"@level":"' + SFInt32(self.level).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        result += 'COMPONENT ' + self.name + ':' + str(self.level) + '\n'
#       print('VRML serialization complete.', flush=True)
        return result

class connect(_X3DStatement):
    """
    Functional summary: connect statements define event-routing connections between node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'connect'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#IS_ConnectStatementSyntax'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#connect'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('nodeField', '', FieldType.SFString, AccessType.inputOutput, 'connect'),
        ('protoField', '', FieldType.SFString, AccessType.inputOutput, 'connect'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        nodeField='',
        protoField='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement connect __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.nodeField = nodeField
        self.protoField = protoField
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def nodeField(self):
        """Name of field within this node which IS CONNECTed to the ancestor ProtoDeclare field definition."""
        return self.__nodeField
    @nodeField.setter
    def nodeField(self, nodeField):
        if  nodeField is None:
            nodeField = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(nodeField)
        self.__nodeField = nodeField
    @property # getter - - - - - - - - - -
    def protoField(self):
        """Name of parent ProtoDeclare field definition connecting to field in this node."""
        return self.__protoField
    @protoField.setter
    def protoField(self, protoField):
        if  protoField is None:
            protoField = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(protoField)
        self.__protoField = protoField
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function connect.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<connect'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.nodeField:
            result += " nodeField='" + SFString(self.nodeField).XML() + "'"
        if self.protoField:
            result += " protoField='" + SFString(self.protoField).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></connect>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</connect>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function connect.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"connect":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.nodeField:
            attributeResult += "        " + '"@nodeField":"' + SFString(self.nodeField).JSON() + '"' + ',\n'
        if self.protoField:
            attributeResult += "        " + '"@protoField":"' + SFString(self.protoField).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function connect.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        result += 'connect' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.nodeField:
            result += '\n' + indent + '  ' + "nodeField " +  '"' + self.nodeField + '"' + ""
        if self.protoField:
            result += '\n' + indent + '  ' + "protoField " +  '"' + self.protoField + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class EXPORT(_X3DStatement):
    """
    Functional summary: EXPORT exposes a local node for ROUTE passing of event values when the current Scene is included via Inline by a parent external world. These connections allow event values to be exchanged via ROUTE statements between a parent model and a child Inline model.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'EXPORT'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/concepts.html#EXPORTSemantics'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EXPORT'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('AS', '', FieldType.SFString, AccessType.inputOutput, 'EXPORT'),
        ('localDEF', '', FieldType.SFString, AccessType.inputOutput, 'EXPORT'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        AS='',
        localDEF='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement EXPORT __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.AS = AS
        self.localDEF = localDEF
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def AS(self):
        """rename localDEF node AS a different name when exporting."""
        return self.__AS
    @AS.setter
    def AS(self, AS):
        if  AS is None:
            AS = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(AS)
        self.__AS = AS
    @property # getter - - - - - - - - - -
    def localDEF(self):
        """localDEF is the DEF name of the local node to be EXPORTed."""
        return self.__localDEF
    @localDEF.setter
    def localDEF(self, localDEF):
        if  localDEF is None:
            localDEF = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(localDEF)
        self.__localDEF = localDEF
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EXPORT.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EXPORT'
        if self.AS:
            result += " AS='" + SFString(self.AS).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.localDEF:
            result += " localDEF='" + SFString(self.localDEF).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></EXPORT>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</EXPORT>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EXPORT.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"EXPORT":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.AS:
            attributeResult += "        " + '"@AS":"' + SFString(self.AS).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.localDEF:
            attributeResult += "        " + '"@localDEF":"' + SFString(self.localDEF).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function EXPORT.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        result += 'EXPORT' + ' {'
        if self.AS:
            result += '\n' + indent + '  ' + "AS " +  '"' + self.AS + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.localDEF:
            result += '\n' + indent + '  ' + "localDEF " +  '"' + self.localDEF + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class ExternProtoDeclare(_X3DStatement):
    """
    ExternProtoDeclare refers to a ProtoDeclare node declaration provided in another file. ExternProtoDeclare interfaces are defined by field statements (and without IS/connect statements).
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'ExternProtoDeclare'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/concepts.html#Externalprototypesemantics'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ExternProtoDeclare'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('url', [], FieldType.MFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('field', [], FieldType.MFNode, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        appinfo='',
        documentation='',
        name='',
        url=None,
        field=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement ExternProtoDeclare __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.url = url
        self.field = field
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """name of the ExternProtoDeclare (External Prototype Declaration) being referenced."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def url(self):
        """Location and filename of ProtoDeclare source declaration of interest."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in the corresponding original ProtoDeclare."""
        return self.__field
    @field.setter
    def field(self, field):
        if  field is None:
            field = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        # TODO type-aware checks for field
        if field: # walk each child in list, if any (avoid empty list recursion)
            for each in field:
                assertValidFieldInitializationValue(each.name, type(each.value), each.value, parent='ExternProtoDeclare/field')
        self.__field = field
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.field) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ExternProtoDeclare.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ExternProtoDeclare'
        if self.appinfo:
            result += " appinfo='" + SFString(self.appinfo).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.documentation:
            result += " documentation='" + SFString(self.documentation).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.url != []:
            result += " url='" + MFString(self.url).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ExternProtoDeclare>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.field: # walk each child in list, if any
            ### print('* ExternProtoDeclare found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.field: # walk each child in list, if any (avoid empty list recursion)
                for each in self.field:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</ExternProtoDeclare>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ExternProtoDeclare.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"ExternProtoDeclare":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.appinfo:
            attributeResult += "        " + '"@appinfo":"' + SFString(self.appinfo).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.documentation:
            attributeResult += "        " + '"@documentation":"' + SFString(self.documentation).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.url != []:
            attributeResult += "        " + '"@url":"' + MFString(self.url).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.field: # walk each child in list, if any (avoid empty list recursion)
            ### print('* ExternProtoDeclare found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.field: # walk each child in list, if any (avoid empty list recursion)
                for each in self.field:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        if indentLevel == 0:
            result += '\n'
        result += 'ExternProtoDeclare' + ' {'
        if self.appinfo:
            result += '\n' + indent + '  ' + "appinfo " +  '"' + self.appinfo + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.documentation:
            result += '\n' + indent + '  ' + "documentation " +  '"' + self.documentation + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.name:
            result += '\n' + indent + '  ' + "name " +  '"' + self.name + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.url != []:
            result += '\n' + indent + '  ' + "url " + MFString(self.url).VRML() + ""
        if self.field: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.field:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class field(_X3DStatement):
    """
    Functional summary: a field statement defines an interface attribute or node. Each field statement can contain either attribute-value or node content.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'field'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#NodeAndFieldStatementSyntax'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#field'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('accessType', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('type', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('value', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'field'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        accessType='',
        appinfo='',
        documentation='',
        name='',
        type='',
        value='',
        children=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement field __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.accessType = accessType
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.type = type
        self.value = value
        self.children = children
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def accessType(self):
        """Event-model semantics for field set/get capabilities."""
        return self.__accessType
    @accessType.setter
    def accessType(self, accessType):
        if  accessType is None:
            accessType = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(accessType)
        assertValidAccessType('accessType', accessType)
        self.__accessType = accessType
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """Name of this field declaration."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def type(self):
        """Base type of this field variable."""
        return self.__type
    @type.setter
    def type(self, type):
        if  type is None:
            type = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(type)
        assertValidFieldType('type', type)
        self.__type = type
    @property # getter - - - - - - - - - -
    def value(self):
        """Provide default initialization value for this field variable (which may be re-initialized later by instantiation value of a named ProtoInstance fieldValue)."""
        return self.__value
    @value.setter
    def value(self, value):
        if  value is None:
            value = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidFieldInitializationValue(self.name, self.type, value, parent='field/@value')
        self.__value = value
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DNode] If this field definition has type SFNode or MFNode, then initialization node (or nodes) of any appropriate type may be provided as children of the field definition."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function field.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<field'
        if self.accessType:
            result += " accessType='" + SFString(self.accessType).XML() + "'"
        if self.appinfo:
            result += " appinfo='" + SFString(self.appinfo).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.documentation:
            result += " documentation='" + SFString(self.documentation).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.type:
            result += " type='" + SFString(self.type).XML() + "'"
        if self.value:
            # print('field name=' + self.name + ' type=' + str(self.type) + ' type(value)=' + str(type(self.value)) + ' value=' + str(self.value))
            if (self.type   == 'SFBool'):
                _newValue =  SFBool(self.value).XML()
            elif (self.type == 'MFBool'):
                _newValue =  MFBool(self.value).XML()
            elif (self.type == 'SFInt32'):
                _newValue =  SFInt32(self.value).XML()
            elif (self.type == 'MFInt32'):
                _newValue =  MFInt32(self.value).XML()
            elif (self.type == 'SFFloat'):
                _newValue =  SFFloat(self.value).XML()
            elif (self.type == 'MFFloat'):
                _newValue =  MFFloat(self.value).XML()
            elif (self.type == 'SFDouble'):
                _newValue =  SFDouble(self.value).XML()
            elif (self.type == 'MFDouble'):
                _newValue =  MFDouble(self.value).XML()

            elif (self.type == 'SFString'):
                _newValue =  SFString(self.value).XML()
            elif (self.type == 'MFString'):
                _newValue =  MFString(self.value).XML()

            elif (self.type == 'SFRotation'):
                _newValue =  SFRotation(self.value).XML()
            elif (self.type == 'MFRotation'):
                _newValue =  MFRotation(self.value).XML()
            elif (self.type == 'SFColor'):
                _newValue =  SFColor(self.value).XML()
            elif (self.type == 'MFColor'):
                _newValue =  MFColor(self.value).XML()
            elif (self.type == 'SFColorRGBA'):
                _newValue =  SFColorRGBA(self.value).XML()
            elif (self.type == 'MFColorRGBA'):
                _newValue =  MFColorRGBA(self.value).XML()

            elif (self.type == 'SFVec2f'):
                _newValue =  SFVec2f(self.value).XML()
            elif (self.type == 'MFVec2f'):
                _newValue =  MFVec2f(self.value).XML()
            elif (self.type == 'SFVec2d'):
                _newValue =  SFVec2d(self.value).XML()
            elif (self.type == 'MFVec2d'):
                _newValue =  MFVec2d(self.value).XML()

            elif (self.type == 'SFVec3f'):
                _newValue =  SFVec3f(self.value).XML()
            elif (self.type == 'MFVec3f'):
                _newValue =  MFVec3f(self.value).XML()
            elif (self.type == 'SFVec3d'):
                _newValue =  SFVec3d(self.value).XML()
            elif (self.type == 'MFVec3d'):
                _newValue =  MFVec3d(self.value).XML()

            elif (self.type == 'SFVec4f'):
                _newValue =  SFVec4f(self.value).XML()
            elif (self.type == 'MFVec4f'):
                _newValue =  MFVec4f(self.value).XML()
            elif (self.type == 'SFVec4d'):
                _newValue =  SFVec4d(self.value).XML()
            elif (self.type == 'MFVec4d'):
                _newValue =  MFVec4d(self.value).XML()
            # TODO matrices
            else:
                _newValue = SFString(self.value).XML()
            result += " value=" + "'" + _newValue.rstrip() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></field>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in list, if any
            ### print('* field found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</field>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function field.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"field":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.accessType:
            attributeResult += "        " + '"@accessType":"' + SFString(self.accessType).JSON() + '"' + ',\n'
        if self.appinfo:
            attributeResult += "        " + '"@appinfo":"' + SFString(self.appinfo).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.documentation:
            attributeResult += "        " + '"@documentation":"' + SFString(self.documentation).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.type:
            attributeResult += "        " + '"@type":"' + SFString(self.type).JSON() + '"' + ',\n'
        if self.value:
            attributeResult += "        " + '"@value":"' + str(self.value) + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* field found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function field.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        result += 'field' + ' {'
        if self.accessType:
            result += '\n' + indent + '  ' + "accessType " +  '"' + self.accessType + '"' + ""
        if self.appinfo:
            result += '\n' + indent + '  ' + "appinfo " +  '"' + self.appinfo + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.documentation:
            result += '\n' + indent + '  ' + "documentation " +  '"' + self.documentation + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.name:
            result += '\n' + indent + '  ' + "name " +  '"' + self.name + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.type:
            result += '\n' + indent + '  ' + "type " +  '"' + self.type + '"' + ""
        if self.value:
            result += '\n' + indent + '  ' + "value " + str(self.value) + ""
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class fieldValue(_X3DStatement):
    """
    Functional summary: a fieldValue statement re-initializes the default value of a field in a ProtoInstance. Each fieldValue statement can contain either attribute-value or node content.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'fieldValue'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#ProtoInstanceAndFieldValueStatement'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#fieldValue'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'fieldValue'),
        ('value', '', FieldType.SFString, AccessType.inputOutput, 'fieldValue'),
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'fieldValue'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        name='',
        value='',
        children=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement fieldValue __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.name = name
        self.value = value
        self.children = children
        self.id_ = id_
        self.style_ = style_
        self.type='SFString' # convenience property matching corresponding field declaration
    @property # getter - - - - - - - - - -
    # convenience property matching corresponding field declaration
    def type(self):
        """ Computed type of this fieldValue corresponding to corresponding field declaration. """
        if self.__type is None:
            self.__type = 'SFString'
        #print('*** need to find fieldValue type, using type=' + str(self.__type))
        return self.__type
    @type.setter
    def type(self, type):
        if  type is None:
            type = SFString.NAME()
            # if _DEBUG: print('...DEBUG... set type to SFString.NAME()=' + str(SFString.NAME()))
        assertValidSFString(type)
    ### assertValidFieldType('type', type) # something strange is happening here
        self.__type = type
    @property # getter - - - - - - - - - -
    def name(self):
        """Name of the ProtoInstance field being re-initialized (corresponding to field name already defined in ProtoDeclare or ExternProtoDeclare)."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def value(self):
        """Initial value for this field, which overrides default initialization value defined in original ProtoDeclare field."""
        return self.__value
    @value.setter
    def value(self, value):
        if  value is None:
            value = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidFieldInitializationValue(self.name, type(value), value, parent='fieldValue')
        if isinstance(value,list) and isinstance(value[0],str):
            # print('*** found MFString when setting fieldValue name=' + self.name) # hack, better would be matching proto declaration
            self.type = 'MFString'
        self.__value = value
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DNode] If this field definition has type SFNode or MFNode, then initialization node (or nodes) of any appropriate type may be provided as children of the field definition."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function fieldValue.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<fieldValue'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.value:
            # heuristic (i.e. hack) to infer MFString type from Python object, if not deduced from (Extern)ProtoDeclare
            if not (self.type == 'MFString') and isinstance(self.value,list) and isinstance(self.value[0],str):
                # print('*** found MFString when preparing XML output, fieldValue name=' + self.name)
                self.type = 'MFString'
            # print('fieldValue name=' + self.name + ' type=' + str(self.type) + ' type(value)=' + str(type(self.value)) + ' value=' + str(self.value))
            if (self.type == 'MFString'):
                _newValue = "'" + MFString(self.value).XML() + "'"
                # print('*** computed fieldValue MFString',_newValue)
            elif isinstance(self.value,str):    ### (self.type == 'SFString'):
                _newValue = "'" + SFString(self.value).XML() + "'" ## here!
                # print('*** computed fieldValue (str)',_newValue)
            elif isinstance(self.value,(list,tuple)):
                # print('*** found',str(type(self.value)))
                _newValue = "'"
                for each in self.value:
                    _newValue += str(each).replace('(','').replace(')','').replace(',',' ') + ' '
                _newValue = _newValue.strip()
                _newValue += "'"
                # print('*** computed fieldValue (list,tuple)',_newValue)
            else:
                _newValue = "'" + str(self.value).replace('(','').replace(')','').replace(',',' ')
                _newValue = _newValue.rstrip() + "'"
                # print('*** computed fieldValue default',_newValue)
            result += " value=" + _newValue
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></fieldValue>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in list, if any
            ### print('* fieldValue found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</fieldValue>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function fieldValue.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"fieldValue":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.value:
            attributeResult += "        " + '"@value":"' + str(self.value) + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* fieldValue found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function fieldValue.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        result += 'fieldValue' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.name:
            result += '\n' + indent + '  ' + "name " +  '"' + self.name + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.value:
            result += '\n' + indent + '  ' + "value " + str(self.value) + ""
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class head(_X3DStatement):
    """
    Functional summary: each X3D scene includes a head statement that can contain component, unit and meta statements.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'head'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#Header'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#head'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [('children', None, FieldType.MFNode, AccessType.inputOutput, 'head')]
    def __init__(self, class_="", id_="", style_="", children=None):
        self.class_ = class_
        self.id_ = id_
        self.style_ = style_
        self.children = children
    @property # getter - - - - - - - - - -
    def class_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading stylesheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__class_
    @class_.setter
    def class_(self, class_):
        if  class_ is None:
            class_ = SFString.DEFAULT_VALUE()
        assertValidSFString(class_)
        self.__class_ = class_
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
        assertValidSFString(id_)
        self.__id = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
        assertValidSFString(style_)
        self.__style_ = style_
    @property # getter - - - - - - - - - -
    def children(self):
        """ The head statement has children consisting of component, unit and meta statements. """
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return bool(self.children)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function head.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<head'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></head>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.__children: # walk each child in list, if any (avoid empty list recursion)
                ## print('* head found self.children, now invoking XML(' + str(indentLevel+1) + ')', flush=True)
                # order is significant for component, unit, meta statements
                if self.children: # walk each child in list, if any (avoid empty list recursion)
                    for each in self.children:
                        # TODO check order of output: component unit meta
                        result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</head>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function head.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"head":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.__children: # walk each child in list, if any
                ## print('* head found self.children, now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
                # order is significant for component, unit, meta statements
                if self.children: # walk each child in list, if any (avoid empty list recursion)
                    for each in self.children:
                        # TODO check order of output: component unit meta
                        result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            ## print('* head found self.children, now invoking VRML(' + str(indentLevel+1) + ', ' + VRML97 + ')', flush=True)
            # order is significant for component, unit, meta statements
            for each in self.children:
                # TODO check order of output: component unit meta
                result += each.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
#       print('VRML serialization complete.', flush=True)
        return result

class IMPORT(_X3DStatement):
    """
    Functional summary: IMPORT provides ROUTE access to a node that has a corresponding EXPORT statement within an Inline scene. These connections allow event values to be exchanged via ROUTE statements between a parent model and a child Inline model.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'IMPORT'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/concepts.html#ImportExportsemantics'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#IMPORT'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('AS', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('importedDEF', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('inlineDEF', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        AS='',
        importedDEF='',
        inlineDEF='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement IMPORT __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.AS = AS
        self.importedDEF = importedDEF
        self.inlineDEF = inlineDEF
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def AS(self):
        """map importedDEF name AS a new name in current scene."""
        return self.__AS
    @AS.setter
    def AS(self, AS):
        if  AS is None:
            AS = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(AS)
        self.__AS = AS
    @property # getter - - - - - - - - - -
    def importedDEF(self):
        """importedDEF is DEF name of the node of interest that is contained in the remote inlineDEF scene."""
        return self.__importedDEF
    @importedDEF.setter
    def importedDEF(self, importedDEF):
        if  importedDEF is None:
            importedDEF = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(importedDEF)
        self.__importedDEF = importedDEF
    @property # getter - - - - - - - - - -
    def inlineDEF(self):
        """inlineDEF is the DEF name of Inline node in the same scene as this IMPORT statement."""
        return self.__inlineDEF
    @inlineDEF.setter
    def inlineDEF(self, inlineDEF):
        if  inlineDEF is None:
            inlineDEF = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(inlineDEF)
        self.__inlineDEF = inlineDEF
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IMPORT.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<IMPORT'
        if self.AS:
            result += " AS='" + SFString(self.AS).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.importedDEF:
            result += " importedDEF='" + SFString(self.importedDEF).XML() + "'"
        if self.inlineDEF:
            result += " inlineDEF='" + SFString(self.inlineDEF).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></IMPORT>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</IMPORT>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IMPORT.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"IMPORT":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.AS:
            attributeResult += "        " + '"@AS":"' + SFString(self.AS).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.importedDEF:
            attributeResult += "        " + '"@importedDEF":"' + SFString(self.importedDEF).JSON() + '"' + ',\n'
        if self.inlineDEF:
            attributeResult += "        " + '"@inlineDEF":"' + SFString(self.inlineDEF).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function IMPORT.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        result += 'IMPORT' + ' {'
        if self.AS:
            result += '\n' + indent + '  ' + "AS " +  '"' + self.AS + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.importedDEF:
            result += '\n' + indent + '  ' + "importedDEF " +  '"' + self.importedDEF + '"' + ""
        if self.inlineDEF:
            result += '\n' + indent + '  ' + "inlineDEF " +  '"' + self.inlineDEF + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class IS(_X3DStatement):
    """
    Functional summary: the IS statement connects node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields. IS/connect statements can be added if the parent node is within a ProtoBody and connect statements define correspondences between prototype fields and built-in node fields.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'IS'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/concepts.html#PROTOdefinitionsemantics'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#IS'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('connect', [], FieldType.MFNode, AccessType.inputOutput, 'IS'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        connect=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement IS __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.connect = connect
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def connect(self):
        """When inside a ProtoBody declaration and an IS statement, add a connect statement to define event-routing connections between a parent node's field to a corresponding ProtoInterface field."""
        return self.__connect
    @connect.setter
    def connect(self, connect):
        if  connect is None:
            connect = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(connect)
        self.__connect = connect
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.connect) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IS.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<IS'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></IS>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.connect: # walk each child in list, if any
            ### print('* IS found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(connect)=' + str(len(self.connect)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.connect: # walk each child in list, if any (avoid empty list recursion)
                for each in self.connect:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</IS>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IS.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"IS":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.connect: # walk each child in list, if any (avoid empty list recursion)
            ### print('* IS found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(connect)=' + str(len(self.connect)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.connect: # walk each child in list, if any (avoid empty list recursion)
                for each in self.connect:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function IS.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        result += 'IS' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.connect: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.connect:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class meta(_X3DStatement):
    """
    Functional summary: the meta statement provides metadata information about a scene, where name and content attributes provide attribute=value metadata pairs.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'meta'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#Header'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#meta'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('content', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('dir', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('httpequiv', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('lang', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('scheme', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        content='',
        dir='',
        httpequiv='',
        lang='',
        name='',
        scheme='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement meta __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.content = content
        self.dir = dir
        self.httpequiv = httpequiv
        self.lang = lang
        self.name = name
        self.scheme = scheme
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def content(self):
        """The content attribute provides metadata information relevant to the name attribute provided."""
        return self.__content
    @content.setter
    def content(self, content):
        if  content is None:
            content = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(content)
        self.__content = content
    @property # getter - - - - - - - - - -
    def dir(self):
        """Direction for weak/neutral text (ltr=left-to-right, rtl=right-to-left)."""
        return self.__dir
    @dir.setter
    def dir(self, dir):
        if  dir is None:
            dir = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(dir)
        assertValidMetaDirection('dir', dir)
        self.__dir = dir
    @property # getter - - - - - - - - - -
    def httpequiv(self):
        """The http-equiv attribute provides an HTTP header for the value of the content attribute."""
        return self.__httpequiv
    @httpequiv.setter
    def httpequiv(self, httpequiv):
        if  httpequiv is None:
            httpequiv = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(httpequiv)
        self.__httpequiv = httpequiv
    @property # getter - - - - - - - - - -
    def lang(self):
        """Language code, as per [IETF BCP47/RFC5646]."""
        return self.__lang
    @lang.setter
    def lang(self, lang):
        if  lang is None:
            lang = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(lang)
        self.__lang = lang
    @property # getter - - - - - - - - - -
    def name(self):
        """Keyword name of the meta attribute, following the same naming conventions as HTML's meta tag."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def scheme(self):
        """The scheme attribute allows authors to provide user agents more context for the correct interpretation of meta information."""
        return self.__scheme
    @scheme.setter
    def scheme(self, scheme):
        if  scheme is None:
            scheme = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(scheme)
        self.__scheme = scheme
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function meta.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<meta'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.content:
            result += " content='" + SFString(self.content).XML() + "'"
        if self.dir:
            result += " dir='" + SFString(self.dir).XML() + "'"
        if self.httpequiv:
            result += " httpequiv='" + SFString(self.httpequiv).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.lang:
            result += " lang='" + SFString(self.lang).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.scheme:
            result += " scheme='" + SFString(self.scheme).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></meta>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</meta>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function meta.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"meta":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.content:
            attributeResult += "        " + '"@content":"' + SFString(self.content).JSON() + '"' + ',\n'
        if self.dir:
            attributeResult += "        " + '"@dir":"' + SFString(self.dir).JSON() + '"' + ',\n'
        if self.httpequiv:
            attributeResult += "        " + '"@httpequiv":"' + SFString(self.httpequiv).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.lang:
            attributeResult += "        " + '"@lang":"' + SFString(self.lang).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.scheme:
            attributeResult += "        " + '"@scheme":"' + SFString(self.scheme).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        result += 'META "' + self.name + '" "' + self.content + '"' + '\n'
#       print('VRML serialization complete.', flush=True)
        return result

class ProtoBody(_X3DStatement):
    """
    ProtoBody contains the definition nodes for new Prototype nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'ProtoBody'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#PrototypeAndFieldDeclarationSyntax'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoBody'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'ProtoBody'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        children=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement ProtoBody __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.children = children
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DNode] ProtoBody can contain nodes, statements and comments."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoBody.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoBody'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ProtoBody>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in list, if any
            ### print('* ProtoBody found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</ProtoBody>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoBody.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"ProtoBody":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* ProtoBody found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        if indentLevel == 0:
            result += '\n'
        result += 'ProtoBody' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class ProtoDeclare(_X3DStatement):
    """
    ProtoDeclare defines new Prototype nodes. Nested ProtoDeclares and ProtoInstances are allowed by the specification.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'ProtoDeclare'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#PrototypeAndFieldDeclarationSyntax'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoDeclare'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('ProtoBody', None, FieldType.SFNode, AccessType.inputOutput, 'ProtoDeclare'),
        ('ProtoInterface', None, FieldType.SFNode, AccessType.inputOutput, 'ProtoDeclare'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        appinfo='',
        documentation='',
        name='',
        ProtoBody=None,
        ProtoInterface=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement ProtoDeclare __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.ProtoBody = ProtoBody
        self.ProtoInterface = ProtoInterface
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """name of this prototype being declared."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def ProtoBody(self):
        """Include one ProtoBody statement after the ProtoInterface statement."""
        return self.__ProtoBody
    @ProtoBody.setter
    def ProtoBody(self, ProtoBody):
        if  ProtoBody is None:
            ProtoBody = None # default
        assertValidSFNode(ProtoBody)
        self.__ProtoBody = ProtoBody
    @property # getter - - - - - - - - - -
    def ProtoInterface(self):
        """Include an optional ProtoInterface statement if this ProtoDeclare has field declarations."""
        return self.__ProtoInterface
    @ProtoInterface.setter
    def ProtoInterface(self, ProtoInterface):
        if  ProtoInterface is None:
            ProtoInterface = None # default
        assertValidSFNode(ProtoInterface)
        self.__ProtoInterface = ProtoInterface
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.ProtoBody or self.ProtoInterface
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoDeclare.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoDeclare'
        if self.appinfo:
            result += " appinfo='" + SFString(self.appinfo).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.documentation:
            result += " documentation='" + SFString(self.documentation).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ProtoDeclare>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.ProtoInterface: # output this SFNode
                result += self.ProtoInterface.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.ProtoBody: # output this SFNode
                result += self.ProtoBody.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</ProtoDeclare>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoDeclare.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"ProtoDeclare":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.appinfo:
            attributeResult += "        " + '"@appinfo":"' + SFString(self.appinfo).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.documentation:
            attributeResult += "        " + '"@documentation":"' + SFString(self.documentation).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.ProtoInterface: # output this SFNode
                result += self.ProtoInterface.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.ProtoBody: # output this SFNode
                result += self.ProtoBody.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        if indentLevel == 0:
            result += '\n'
        result += 'ProtoDeclare' + ' {'
        if self.appinfo:
            result += '\n' + indent + '  ' + "appinfo " +  '"' + self.appinfo + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.documentation:
            result += '\n' + indent + '  ' + "documentation " +  '"' + self.documentation + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.name:
            result += '\n' + indent + '  ' + "name " +  '"' + self.name + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.ProtoInterface: # output this SFNode
            result += '\n' + '  ' + indent + 'ProtoInterface ' + self.ProtoInterface.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.ProtoBody: # output this SFNode
            result += '\n' + '  ' + indent + 'ProtoBody ' + self.ProtoBody.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class ProtoInterface(_X3DStatement):
    """
    ProtoInterface defines fields for new Prototype nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'ProtoInterface'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#PrototypeAndFieldDeclarationSyntax'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoInterface'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('field', [], FieldType.MFNode, AccessType.inputOutput, 'ProtoInterface'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        field=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement ProtoInterface __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.field = field
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in this ProtoDeclare's ProtoInterface."""
        return self.__field
    @field.setter
    def field(self, field):
        if  field is None:
            field = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        # TODO type-aware checks for field
        if field: # walk each child in list, if any (avoid empty list recursion)
            for each in field:
                assertValidFieldInitializationValue(each.name, each.type, each.value, parent='ProtoInterface')
        self.__field = field
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.field) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoInterface.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoInterface'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ProtoInterface>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.field: # walk each child in list, if any
            ### print('* ProtoInterface found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.field: # walk each child in list, if any (avoid empty list recursion)
                for each in self.field:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</ProtoInterface>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoInterface.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"ProtoInterface":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.field: # walk each child in list, if any (avoid empty list recursion)
            ### print('* ProtoInterface found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.field: # walk each child in list, if any (avoid empty list recursion)
                for each in self.field:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        if indentLevel == 0:
            result += '\n'
        result += 'ProtoInterface' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.field: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.field:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class ROUTE(_X3DStatement):
    """
    ROUTE connects output fields of event-producing nodes to input fields of event-consuming nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'ROUTE'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/concepts.html#ModifyingObjectsRoutes'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ROUTE'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('fromField', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('fromNode', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('toField', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('toNode', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        fromField='',
        fromNode='',
        toField='',
        toNode='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement ROUTE __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.fromField = fromField
        self.fromNode = fromNode
        self.toField = toField
        self.toNode = toNode
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def fromField(self):
        """fromField is the field name in the source node which is originating an event."""
        return self.__fromField
    @fromField.setter
    def fromField(self, fromField):
        if  fromField is None:
            fromField = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(fromField)
        self.__fromField = fromField
    @property # getter - - - - - - - - - -
    def fromNode(self):
        """fromNode is the DEF name of the node originating an event."""
        return self.__fromNode
    @fromNode.setter
    def fromNode(self, fromNode):
        if  fromNode is None:
            fromNode = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(fromNode)
        self.__fromNode = fromNode
    @property # getter - - - - - - - - - -
    def toField(self):
        """toField is the field name in the destination node which is receiving an event."""
        return self.__toField
    @toField.setter
    def toField(self, toField):
        if  toField is None:
            toField = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(toField)
        self.__toField = toField
    @property # getter - - - - - - - - - -
    def toNode(self):
        """toNode is the DEF name of the destination node receiving an event."""
        return self.__toNode
    @toNode.setter
    def toNode(self, toNode):
        if  toNode is None:
            toNode = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(toNode)
        self.__toNode = toNode
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ROUTE.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ROUTE'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.fromField:
            result += " fromField='" + SFString(self.fromField).XML() + "'"
        if self.fromNode:
            result += " fromNode='" + SFString(self.fromNode).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.toField:
            result += " toField='" + SFString(self.toField).XML() + "'"
        if self.toNode:
            result += " toNode='" + SFString(self.toNode).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ROUTE>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</ROUTE>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ROUTE.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"ROUTE":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.fromField:
            attributeResult += "        " + '"@fromField":"' + SFString(self.fromField).JSON() + '"' + ',\n'
        if self.fromNode:
            attributeResult += "        " + '"@fromNode":"' + SFString(self.fromNode).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.toField:
            attributeResult += "        " + '"@toField":"' + SFString(self.toField).JSON() + '"' + ',\n'
        if self.toNode:
            attributeResult += "        " + '"@toNode":"' + SFString(self.toNode).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result += '\n' + indent + 'ROUTE ' + self.fromNode + '.' + self.fromField + ' TO ' + self.toNode + '.' + self.toField + '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class Scene(_X3DStatement):
    """
    Scene is the implicit root node of the X3D scene graph.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'Scene'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#Header'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Scene'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'Scene'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        children=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement Scene __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.children = children
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DNode] Scene can contain nodes, statements and comments."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Scene.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Scene'
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Scene>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in list, if any
            ### print('* Scene found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</Scene>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Scene.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"Scene":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* Scene found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
#       print('VRML serialization complete.', flush=True)
        return result

class unit(_X3DStatement):
    """
    Functional summary: unit statement defines data-conversion factors for typed values defined in a scene.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'unit'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/core.html#UNITStatement'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#unit'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('category', '', FieldType.SFString, AccessType.initializeOnly, 'unit'),
        ('conversionFactor', 1.0, FieldType.SFDouble, AccessType.inputOutput, 'unit'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'unit'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        category='',
        conversionFactor=1.0,
        name='',
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement unit __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.category = category
        self.conversionFactor = conversionFactor
        self.name = name
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def category(self):
        """Base-unit category as defined in X3D Specification."""
        return self.__category
    @category.setter
    def category(self, category):
        if  category is None:
            category = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(category)
        assertValidUnitCategory('category', category)
        self.__category = category
    @property # getter - - - - - - - - - -
    def conversionFactor(self):
        """[0,+infinity) Positive double-precision factor that converts new base unit to default base unit."""
        return self.__conversionFactor
    @conversionFactor.setter
    def conversionFactor(self, conversionFactor):
        if  conversionFactor is None:
            conversionFactor = 1.0 # default
        assertValidSFDouble(conversionFactor)
        assertPositive('conversionFactor', conversionFactor)
        self.__conversionFactor = conversionFactor
    @property # getter - - - - - - - - - -
    def name(self):
        """Author-defined name for this unit conversionFactor value (for example, FeetToMeters)."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function unit.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<unit'
        if self.category:
            result += " category='" + SFString(self.category).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.conversionFactor != 1.0:
            result += " conversionFactor='" + SFDouble(self.conversionFactor).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.name:
            result += " name='" + SFString(self.name).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></unit>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</unit>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function unit.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"unit":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.category:
            attributeResult += "        " + '"@category":"' + SFString(self.category).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.conversionFactor != 1.0:
            attributeResult += "        " + '"@conversionFactor":"' + SFDouble(self.conversionFactor).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.name:
            attributeResult += "        " + '"@name":"' + SFString(self.name).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        result += 'UNIT ' + self.category + ' ' + self.name + ' ' + str(self.conversionFactor) + '\n'
#       print('VRML serialization complete.', flush=True)
        return result

class X3D(_X3DNode):
    """
    X3D is the root node for an Extensible 3D (X3D) Graphics model.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'X3D'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/documents/specifications/19776-1/V3.3/Part01/concepts.html#Header'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#X3D'
    XML_HEADER = '<?xml version="1.0" encoding="UTF-8"?>'
    XML_DOCTYPE_X3D_3_0 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.0//EN" "https://www.web3d.org/specifications/x3d-3.0.dtd">'
    XML_DOCTYPE_X3D_3_1 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.1//EN" "https://www.web3d.org/specifications/x3d-3.1.dtd">'
    XML_DOCTYPE_X3D_3_2 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.2//EN" "https://www.web3d.org/specifications/x3d-3.2.dtd">'
    XML_DOCTYPE_X3D_3_3 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.3//EN" "https://www.web3d.org/specifications/x3d-3.3.dtd">'
    XML_DOCTYPE_X3D_4_0 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 4.0//EN" "https://www.web3d.org/specifications/x3d-4.0.dtd">'
    XML_DOCTYPE_X3D_4_1 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 4.1//EN" "https://www.web3d.org/specifications/x3d-4.1.dtd">'
    X3D_XML_SCHEMA_ATTRIBUTES_3_0 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='https://www.web3d.org/specifications/x3d-3.0.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_1 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='https://www.web3d.org/specifications/x3d-3.1.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_2 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='https://www.web3d.org/specifications/x3d-3.2.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_3 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='https://www.web3d.org/specifications/x3d-3.3.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_4_0 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='https://www.web3d.org/specifications/x3d-4.0.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_4_1 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='https://www.web3d.org/specifications/x3d-4.1.xsd'"
    VRML97_HEADER = '#VRML V2.0 utf8'
    CLASSIC_VRML_HEADER_PREFIX = '#VRML V' # followed by X3D version number
    CLASSIC_VRML_HEADER_SUFFIX = ' utf8'

    # TODO confirm JSON Schema header
    JSON_HEADER = '''{
    "X3D":,
    {
        "encoding":"UTF-8",
        "$id":   "https://www.web3d.org/specifications/x3d-4.0-JSONSchema.json",
        "$schema": "https://json-schema.org/draft/2020-12/schema",
'''

    X3D_XML_VALIDATOR  = "https://savage.nps.edu/X3dValidator"
    X3D_JSON_VALIDATOR = "https://coderextreme.net/X3DJSONLD/src/main/html/validator.html"

    X3DOM_HEADER = """<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">

<!-- =================================================================== -->
<!-- embedded X3D scene appears after html/head/script and style entries -->
<!-- =================================================================== -->
<html>
   <head>
      <title></title>
      <meta http-equiv="X-UA-Compatible" content="chrome=1,IE=edge"/>
      <meta http-equiv="Content-Type" content="text/html;charset=utf-8"/>
<script type="text/javascript" src="https://ajax.googleapis.com/ajax/libs/jquery/3.2.1/jquery.min.js"></script>
<!-- Numbered X3DOM release versions: https://doc.x3dom.org/download -->
<!-- Developer X3DOM release version: https://doc.x3dom.org/download/dev -->
<link rel="stylesheet"
            type="text/css"
            href="https://x3dom.org/release/x3dom.css"/>
      <script type="text/javascript" src="https://x3dom.org/release/x3dom-full.js"/>
      <meta name="warning"
            content="Webfonts must be loaded prior to using Text node in X3D scene... see https://x3dom.org/x3dom/example/x3dom_text.html"/>
      <!-- X3DOM needs Web Fonts when an X3D Text node is included -->
      <!-- adapted from https://x3dom.org/x3dom/example/x3dom_text.html and http://web.mit.edu/jmorzins/www/fonts.html -->
      <style type="text/css">
/* ============================================================================= */
@font-face {
  font-family: 'SERIF'; /* default original */
  font-style: normal;
  font-weight: 700;
  src: local('Roman'), url('Roman.ttf') format('truetype');
}
@font-face {
  font-family: 'SERIF'; /* default alternate */
  font-style: normal;
  font-weight: 700;
  src: local('Times New Roman'), local('TimesNewRoman'), url('Times New Roman.ttf') format('truetype');
}
/* ============================================================================= */
@font-face {
  font-family: 'SANS'; /* default original */
  font-style: normal;
  font-weight: 400;
  src: local('Arial'), url('Arial.ttf') format('truetype');
}
@font-face {
  font-family: 'SANS'; /* default alternate */
  font-style: normal;
  font-weight: 400;
  src: local('Helvetica'), url('Helvetica.ttf') format('truetype');
}
/* ============================================================================= */
@font-face {
  font-family: 'TYPEWRITER'; /* default original */
  font-style: normal;
  font-weight: 900;
  src: local('Courier'), url('Courier.ttf') format('truetype');
}
@font-face {
  font-family: 'TYPEWRITER'; /* default alternate */
  font-style: normal;
  font-weight: 900;
  src: local('Courier New'), url('Courier New.ttf') format('truetype');
}
/* ============================================================================= */
</style>
   </head>
   <body>"""
    X3DOM_FOOTER = """
   </body>
</html>
"""
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('profile', 'Full', FieldType.SFString, AccessType.inputOutput, 'X3D'),
        ('version', '4.0', FieldType.SFString, AccessType.inputOutput, 'X3D'),
        ('head', None, FieldType.SFNode, AccessType.inputOutput, 'X3D'),
        ('Scene', None, FieldType.SFNode, AccessType.inputOutput, 'X3D'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DStatement')]
    def __init__(self,
        profile='Full',
        version='4.0',
        head=None,
        Scene=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in Statement X3D __init__ calling super.__init__(' + str(class_) + ',' + str(id_) + ',' + str(style_) +  + ')', flush=True)
        super().__init__(class_, id_, style_) # fields for _X3DStatement only
        self.profile = profile
        self.version = version
        self.head = head
        self.Scene = Scene
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def profile(self):
        """profile attribute is required and defines the player or tool support needed for this model."""
        return self.__profile
    @profile.setter
    def profile(self, profile):
        if  profile is None:
            profile = 'Full' # default
        assertValidSFString(profile)
        assertValidProfileName('profile', profile)
        self.__profile = profile
    @property # getter - - - - - - - - - -
    def version(self):
        """Default is highest value matching schema and DOCTYPE in the scene."""
        return self.__version
    @version.setter
    def version(self, version):
        if  version is None:
            version = '4.0' # default
        assertValidSFString(version)
        assertValidX3dVersion('version', version)
        self.__version = version
    @property # getter - - - - - - - - - -
    def head(self):
        """Include a head element to contain component, unit or meta statements for this X3D model."""
        return self.__head
    @head.setter
    def head(self, head):
        if  head is None:
            head = None # default
        assertValidSFNode(head)
        if not head is None and not isinstance(head,object):
            # print(flush=True)
            raise X3DTypeError(str(head) + ' does not match required node type object and is invalid')
        self.__head = head
    @property # getter - - - - - - - - - -
    def Scene(self):
        """Include a Scene element to contain scene-graph nodes for this X3D model."""
        return self.__Scene
    @Scene.setter
    def Scene(self, Scene):
        if  Scene is None:
            Scene = None # default
        assertValidSFNode(Scene)
        if not Scene is None and not isinstance(Scene,object):
            # print(flush=True)
            raise X3DTypeError(str(Scene) + ' does not match required node type object and is invalid')
        self.__Scene = Scene
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.head or self.Scene
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function X3D.XML(indentLevel=' + str(indentLevel) + '), indent="' + indent + '"' + '\n'
        result += indent + self.XML_HEADER + '\n'
        if self.version == '3.0':
            result += indent + self.XML_DOCTYPE_X3D_3_0 + '\n'
        elif self.version == '3.1':
            result += indent + self.XML_DOCTYPE_X3D_3_1 + '\n'
        elif self.version == '3.2':
            result += indent + self.XML_DOCTYPE_X3D_3_2 + '\n'
        elif self.version == '3.3':
            result += indent + self.XML_DOCTYPE_X3D_3_3 + '\n'
        elif self.version == '4.0':
            result += indent + self.XML_DOCTYPE_X3D_4_0 + '\n'
        elif self.version == '4.1':
            result += indent + self.XML_DOCTYPE_X3D_4_1 + '\n'
        result += indent + "<X3D profile='" + self.profile +"' version='" + self.version +"' "
        if self.version == '3.0':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_0
        elif self.version == '3.1':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_1
        elif self.version == '3.2':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_2
        elif self.version == '3.3':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_3
        elif self.version == '4.0':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_4_0
        elif self.version == '4.1':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_4_1
        result += '>' + '\n' # finish open tag
        if self.head and self.head.hasChild():
            result += str(self.head.XML(indentLevel=indentLevel+1, syntax=syntax))
        if self.Scene and self.Scene.hasChild():
            result += str(self.Scene.XML(indentLevel=indentLevel+1, syntax=syntax))
        result += '</X3D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def XMLvalidate(self,otherX3dVersion=""):
        """
        XML well-formed test and XML Schema validation test
        """
        if otherX3dVersion:
            validationVersion = str(otherX3dVersion)
            print("XMLvalidate validationVersion=" + validationVersion)
        else:
            validationVersion = str(self.version)
        try:
            selfX3dXmlText = ''
            import xmlschema
            x3dSchemaUrl = 'https://www.web3d.org/specifications/x3d-' + validationVersion + '.xsd'
            x3dschema = xmlschema.XMLSchema(x3dSchemaUrl)
            try:
                from xml.etree import ElementTree
                selfX3dXmlText = self.XML()
                selfX3dXmlTree = ElementTree.fromstring(selfX3dXmlText)
                print("Python-to-XML well-formed XML document  test of XML output complete")
                x3dschema.is_valid(selfX3dXmlTree)
                print("Python-to-XML X3D", str(self.version), "schema validation test of XML output complete")
            except SyntaxError as err: # Exception
                # https://stackoverflow.com/questions/18176602/how-to-get-the-name-of-an-exception-that-was-caught-in-python
                print("*** Python-to-XML X3D", str(self.version), "schema validation test of XML output failed.")
                print("    x3dSchemaUrl=", x3dSchemaUrl)
                if hasattr(err,'position') and err.position[0]:
                    print("   ", type(err).__name__, "(line=" + str(err.lineno) + ')', err)
                if selfX3dXmlText: # might have failed to generate
                    print(prependLineNumbers(selfX3dXmlText,err.lineno))
        except Exception as err: # usually ParseError
            # https://docs.python.org/3/library/xml.etree.elementtree.html#exceptions
            print("*** Python-to-XML well-formed XML document test failed.")
            print("    x3dSchemaUrl=", x3dSchemaUrl)
            print("    ", type(err).__name__, err)
            if hasattr(err,'position') and err.position[0]:
                lineNumber = err.position[0]
                print('type(err.position)=' + str(type(err.position)), 'lineNumber=' + str(lineNumber))
            else:
                lineNumber = 1
            if selfX3dXmlText: # might have failed to generate
                print(prependLineNumbers(selfX3dXmlText,lineNumber))
    # output function - - - - - - - - - -
    def VRML97(self, indentLevel=0):
        """ Provide VRML97 output serialization suitable for .wrl file. """
        return VRML(self, indentLevel=0, VRML97=True)
    # output function - - - - - - - - - -
    def ClassicVRML(self, indentLevel=0):
        """ Provide ClassicVRML output serialization suitable for .x3dv file. """
        return VRML(self, indentLevel=0, VRML97=False)
    # output function - - - - - - - - - -
#    def X_ITE(self): # TODO implement
#        """ Provide X_ITE output serialization suitable for .html file. """
#        return X3D.X_ITE_HEADER + result + self.XML(indentLevel=0, syntax="XML") + X3D.X_ITE_FOOTER:
    # output function - - - - - - - - - -
    def X3DOM(self, indentLevel=0):
        """ Provide X3DOM output serialization suitable for .html file. """
        return X3D.X3DOM_HEADER + self.XML(indentLevel=0, syntax="HTML5") + X3D.X3DOM_FOOTER
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function X3D.JSON(indentLevel=' + str(indentLevel) + '), indent="' + indent + '"' + '\n'
        result += indent + self.JSON_HEADER
        indent = '    ' * 2
        result += indent + '"@version":"' + self.version + '",' + '\n'
        result += indent + '"@profile":"' + self.profile + '",' + '\n'
        if self.head and self.head.hasChild():
            result += str(self.head.JSON(indentLevel=indentLevel+1, syntax=syntax))
        if self.Scene and self.Scene.hasChild():
            result += str(self.Scene.JSON(indentLevel=indentLevel+1, syntax=syntax))
        result += '}' + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        if VRML97:
            result += self.VRML97_HEADER + '\n'
            result += '# X3D-to-VRML97 serialization autogenerated by X3DPSAIL x3d.py' + '\n'
        else:
            result += self.CLASSIC_VRML_HEADER_PREFIX + str(self.version) + self.CLASSIC_VRML_HEADER_SUFFIX + '\n'
        result += '#X3D-to-ClassicVRML serialization autogenerated by X3DPSAIL x3d.py' + '\n'
        result += '\n'
        if not VRML97:
            result += 'PROFILE '
            if not self.profile or VRML97:
                result += 'IMMERSIVE' + '\n'
            else:
                result += self.profile + '\n'
        if self.head and self.head.hasChild():
            result += str(self.head.VRML(indentLevel=indentLevel+1, VRML97=VRML97))
        if self.Scene and self.Scene.hasChild():
            result += str(self.Scene.VRML(indentLevel=indentLevel+1, VRML97=VRML97))
        result += '\n'
#       print('VRML serialization complete.', flush=True)
        return result

###############################################

# Concrete Nodes

def isX3DNode(value):
    """
    Whether or not value is a concrete node (Shape WorldInfo etc.) meaning any _X3DNode object.
    """
    return isinstance(value, _X3DNode)

class AcousticProperties(_X3DAppearanceChildNode):
    """
    AcousticProperties specifies the interaction of sound waves with characteristics of geometric objects in the scene.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'AcousticProperties'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#AcousticProperties'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#AcousticProperties'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('absorption', 0, FieldType.SFFloat, AccessType.inputOutput, 'AcousticProperties'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'AcousticProperties'),
        ('diffuse', 0, FieldType.SFFloat, AccessType.inputOutput, 'AcousticProperties'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'AcousticProperties'),
        ('refraction', 0, FieldType.SFFloat, AccessType.inputOutput, 'AcousticProperties'),
        ('specular', 0, FieldType.SFFloat, AccessType.inputOutput, 'AcousticProperties'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        absorption=0,
        description='',
        diffuse=0,
        enabled=True,
        refraction=0,
        specular=0,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode AcousticProperties __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.absorption = absorption
        self.description = description
        self.diffuse = diffuse
        self.enabled = enabled
        self.refraction = refraction
        self.specular = specular
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def absorption(self):
        """[0,1] specifies the sound absorption coefficient of a surface, meaning the ratio of sound intensity not reflected by a surface."""
        return self.__absorption
    @absorption.setter
    def absorption(self, absorption):
        if  absorption is None:
            absorption = 0 # default
        assertValidSFFloat(absorption)
        assertZeroToOne('absorption', absorption)
        self.__absorption = absorption
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided prose that describes intended purpose of the url asset."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def diffuse(self):
        """[0,1] diffuse coefficient of sound reflection indicates how much of the incident sound energy is reflected back in multiple directions."""
        return self.__diffuse
    @diffuse.setter
    def diffuse(self, diffuse):
        if  diffuse is None:
            diffuse = 0 # default
        assertValidSFFloat(diffuse)
        assertZeroToOne('diffuse', diffuse)
        self.__diffuse = diffuse
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def refraction(self):
        """[0,1] sound refraction coefficient of a medium, which determines change in propagation direction of sound wave when obliquely crossing boundary between two mediums where its speed is different."""
        return self.__refraction
    @refraction.setter
    def refraction(self, refraction):
        if  refraction is None:
            refraction = 0 # default
        assertValidSFFloat(refraction)
        assertZeroToOne('refraction', refraction)
        self.__refraction = refraction
    @property # getter - - - - - - - - - -
    def specular(self):
        """[0,1] specular coefficient of sound reflection striking a plane surface, directly reflected back into space, where angle of reflection equals angle of incidence."""
        return self.__specular
    @specular.setter
    def specular(self, specular):
        if  specular is None:
            specular = 0 # default
        assertValidSFFloat(specular)
        assertZeroToOne('specular', specular)
        self.__specular = specular
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AcousticProperties.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<AcousticProperties'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.absorption != 0:
            result += " absorption='" + SFFloat(self.absorption).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.description:
            result += " description='" + SFString(self.description).XML() + "'"
        if self.diffuse != 0:
            result += " diffuse='" + SFFloat(self.diffuse).XML() + "'"
        if not self.enabled: # default=true
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.refraction != 0:
            result += " refraction='" + SFFloat(self.refraction).XML() + "'"
        if self.specular != 0:
            result += " specular='" + SFFloat(self.specular).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></AcousticProperties>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</AcousticProperties>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AcousticProperties.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"AcousticProperties":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.absorption != 0:
            attributeResult += "        " + '"@absorption":"' + SFFloat(self.absorption).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.description:
            attributeResult += "        " + '"@description":"' + SFString(self.description).JSON() + '"' + ',\n'
        if self.diffuse != 0:
            attributeResult += "        " + '"@diffuse":"' + SFFloat(self.diffuse).JSON() + '"' + ',\n'
        if not self.enabled: # default=true
            attributeResult += "        " + '"@enabled":"' + SFBool(self.enabled).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.refraction != 0:
            attributeResult += "        " + '"@refraction":"' + SFFloat(self.refraction).JSON() + '"' + ',\n'
        if self.specular != 0:
            attributeResult += "        " + '"@specular":"' + SFFloat(self.specular).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function AcousticProperties.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'AcousticProperties' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'AcousticProperties' + ' {'
        if self.absorption != 0:
            result += '\n' + indent + '  ' + "absorption " + SFFloat(self.absorption).VRML() + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += '\n' + indent + '  ' + "description " +  '"' + self.description + '"' + ""
        if self.diffuse != 0:
            result += '\n' + indent + '  ' + "diffuse " + SFFloat(self.diffuse).VRML() + ""
        if not self.enabled: # default=true
            result += '\n' + indent + '  ' + "enabled " + SFBool(self.enabled).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.refraction != 0:
            result += '\n' + indent + '  ' + "refraction " + SFFloat(self.refraction).VRML() + ""
        if self.specular != 0:
            result += '\n' + indent + '  ' + "specular " + SFFloat(self.specular).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class Analyser(_X3DSoundProcessingNode):
    """
    Analyser provides real-time frequency and time-domain analysis information, without any change to the input.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'Analyser'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#Analyser'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Analyser'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('channelCountMode', 'max', FieldType.SFString, AccessType.inputOutput, 'X3DSoundProcessingNode'),
        ('channelInterpretation', 'speakers', FieldType.SFString, AccessType.inputOutput, 'X3DSoundProcessingNode'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSoundProcessingNode'),
        ('fftSize', 2048, FieldType.SFInt32, AccessType.inputOutput, 'Analyser'),
        ('frequencyBinCount', 1024, FieldType.SFInt32, AccessType.inputOutput, 'Analyser'),
        ('gain', 1, FieldType.SFFloat, AccessType.inputOutput, 'X3DSoundProcessingNode'),
        ('maxDecibels', -30, FieldType.SFFloat, AccessType.inputOutput, 'Analyser'),
        ('minDecibels', -100, FieldType.SFFloat, AccessType.inputOutput, 'Analyser'),
        ('pauseTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('resumeTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('smoothingTimeConstant', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'Analyser'),
        ('startTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('stopTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('tailTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DSoundProcessingNode'),
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'Analyser'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        channelCountMode='max',
        channelInterpretation='speakers',
        description='',
        enabled=True,
        fftSize=2048,
        frequencyBinCount=1024,
        gain=1,
        maxDecibels=-30,
        minDecibels=-100,
        pauseTime=0,
        resumeTime=0,
        smoothingTimeConstant=0.8,
        startTime=0,
        stopTime=0,
        tailTime=0,
        children=None,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode Analyser __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.channelCountMode = channelCountMode
        self.channelInterpretation = channelInterpretation
        self.description = description
        self.enabled = enabled
        self.fftSize = fftSize
        self.frequencyBinCount = frequencyBinCount
        self.gain = gain
        self.maxDecibels = maxDecibels
        self.minDecibels = minDecibels
        self.pauseTime = pauseTime
        self.resumeTime = resumeTime
        self.smoothingTimeConstant = smoothingTimeConstant
        self.startTime = startTime
        self.stopTime = stopTime
        self.tailTime = tailTime
        self.children = children
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def channelCountMode(self):
        """channelCountMode determines how individual channels are counted when up-mixing and down-mixing connections to any inputs."""
        return self.__channelCountMode
    @channelCountMode.setter
    def channelCountMode(self, channelCountMode):
        if  channelCountMode is None:
            channelCountMode = 'max' # default
        assertValidSFString(channelCountMode)
        assertValidChannelCountMode('channelCountMode', channelCountMode)
        self.__channelCountMode = channelCountMode
    @property # getter - - - - - - - - - -
    def channelInterpretation(self):
        """channelInterpretation determines how individual channels are treated when up-mixing and down-mixing connections to any inputs."""
        return self.__channelInterpretation
    @channelInterpretation.setter
    def channelInterpretation(self, channelInterpretation):
        if  channelInterpretation is None:
            channelInterpretation = 'speakers' # default
        assertValidSFString(channelInterpretation)
        assertValidChannelInterpretation('channelInterpretation', channelInterpretation)
        self.__channelInterpretation = channelInterpretation
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided prose that describes intended purpose of the url asset."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def fftSize(self):
        """(0,+infinity) fftSize represents size of Fast Fourier Transform (FFT) used to determine frequency domain."""
        return self.__fftSize
    @fftSize.setter
    def fftSize(self, fftSize):
        if  fftSize is None:
            fftSize = 2048 # default
        assertValidSFInt32(fftSize)
        assertNonNegative('fftSize', fftSize)
        self.__fftSize = fftSize
    @property # getter - - - - - - - - - -
    def frequencyBinCount(self):
        """(0,+infinity) frequencyBinCount is half of fftSize and generally equates to number of data values available for the visualization."""
        return self.__frequencyBinCount
    @frequencyBinCount.setter
    def frequencyBinCount(self, frequencyBinCount):
        if  frequencyBinCount is None:
            frequencyBinCount = 1024 # default
        assertValidSFInt32(frequencyBinCount)
        assertNonNegative('frequencyBinCount', frequencyBinCount)
        self.__frequencyBinCount = frequencyBinCount
    @property # getter - - - - - - - - - -
    def gain(self):
        """(-infinity,+infinity) The gain field is a factor that represents the amount of linear amplification to apply to the output of the node."""
        return self.__gain
    @gain.setter
    def gain(self, gain):
        if  gain is None:
            gain = 1 # default
        assertValidSFFloat(gain)
        self.__gain = gain
    @property # getter - - - - - - - - - -
    def maxDecibels(self):
        """(-infinity,+infinity) maxDecibels represents maximum power value in scaling range for FFT analysis data."""
        return self.__maxDecibels
    @maxDecibels.setter
    def maxDecibels(self, maxDecibels):
        if  maxDecibels is None:
            maxDecibels = -30 # default
        assertValidSFFloat(maxDecibels)
        self.__maxDecibels = maxDecibels
    @property # getter - - - - - - - - - -
    def minDecibels(self):
        """(-infinity,+infinity) minDecibels represents minimum power value in scaling range for FFT analysis data."""
        return self.__minDecibels
    @minDecibels.setter
    def minDecibels(self, minDecibels):
        if  minDecibels is None:
            minDecibels = -100 # default
        assertValidSFFloat(minDecibels)
        self.__minDecibels = minDecibels
    @property # getter - - - - - - - - - -
    def pauseTime(self):
        """When time now >= pauseTime, isPaused becomes true and AudioClip becomes paused."""
        return self.__pauseTime
    @pauseTime.setter
    def pauseTime(self, pauseTime):
        if  pauseTime is None:
            pauseTime = 0 # default
        assertValidSFTime(pauseTime)
        self.__pauseTime = pauseTime
    @property # getter - - - - - - - - - -
    def resumeTime(self):
        """When resumeTime becomes <= time now, isPaused becomes false and AudioClip becomes active."""
        return self.__resumeTime
    @resumeTime.setter
    def resumeTime(self, resumeTime):
        if  resumeTime is None:
            resumeTime = 0 # default
        assertValidSFTime(resumeTime)
        self.__resumeTime = resumeTime
    @property # getter - - - - - - - - - -
    def smoothingTimeConstant(self):
        """(0,+infinity) smoothingTimeConstant represents averaging constant during last analysis frame."""
        return self.__smoothingTimeConstant
    @smoothingTimeConstant.setter
    def smoothingTimeConstant(self, smoothingTimeConstant):
        if  smoothingTimeConstant is None:
            smoothingTimeConstant = 0.8 # default
        assertValidSFFloat(smoothingTimeConstant)
        assertNonNegative('smoothingTimeConstant', smoothingTimeConstant)
        self.__smoothingTimeConstant = smoothingTimeConstant
    @property # getter - - - - - - - - - -
    def startTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__startTime
    @startTime.setter
    def startTime(self, startTime):
        if  startTime is None:
            startTime = 0 # default
        assertValidSFTime(startTime)
        self.__startTime = startTime
    @property # getter - - - - - - - - - -
    def stopTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__stopTime
    @stopTime.setter
    def stopTime(self, stopTime):
        if  stopTime is None:
            stopTime = 0 # default
        assertValidSFTime(stopTime)
        self.__stopTime = stopTime
    @property # getter - - - - - - - - - -
    def tailTime(self):
        """[0,+infinity) tailTime is duration of time that a node continues to provide output signal after the input signal becomes silent."""
        return self.__tailTime
    @tailTime.setter
    def tailTime(self, tailTime):
        if  tailTime is None:
            tailTime = 0 # default
        assertValidSFTime(tailTime)
        assertNonNegative('tailTime', tailTime)
        self.__tailTime = tailTime
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DSoundChannelNode|X3DSoundProcessingNode|X3DSoundSourceNode] The children field specifies audio-graph sound sources providing input signals for this node."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata or (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Analyser.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Analyser'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.channelCountMode != 'max':
            result += " channelCountMode='" + SFString(self.channelCountMode).XML() + "'"
        if self.channelInterpretation != 'speakers':
            result += " channelInterpretation='" + SFString(self.channelInterpretation).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.description:
            result += " description='" + SFString(self.description).XML() + "'"
        if not self.enabled: # default=true
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.fftSize != 2048:
            result += " fftSize='" + SFInt32(self.fftSize).XML() + "'"
        if self.frequencyBinCount != 1024:
            result += " frequencyBinCount='" + SFInt32(self.frequencyBinCount).XML() + "'"
        if self.gain != 1:
            result += " gain='" + SFFloat(self.gain).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.maxDecibels != -30:
            result += " maxDecibels='" + SFFloat(self.maxDecibels).XML() + "'"
        if self.minDecibels != -100:
            result += " minDecibels='" + SFFloat(self.minDecibels).XML() + "'"
        if self.pauseTime != 0:
            result += " pauseTime='" + SFTime(self.pauseTime).XML() + "'"
        if self.resumeTime != 0:
            result += " resumeTime='" + SFTime(self.resumeTime).XML() + "'"
        if self.smoothingTimeConstant != 0.8:
            result += " smoothingTimeConstant='" + SFFloat(self.smoothingTimeConstant).XML() + "'"
        if self.startTime != 0:
            result += " startTime='" + SFTime(self.startTime).XML() + "'"
        if self.stopTime != 0:
            result += " stopTime='" + SFTime(self.stopTime).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.tailTime != 0:
            result += " tailTime='" + SFTime(self.tailTime).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Analyser>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.children: # walk each child in list, if any
            ### print('* Analyser found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</Analyser>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Analyser.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"Analyser":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.channelCountMode != 'max':
            attributeResult += "        " + '"@channelCountMode":"' + SFString(self.channelCountMode).JSON() + '"' + ',\n'
        if self.channelInterpretation != 'speakers':
            attributeResult += "        " + '"@channelInterpretation":"' + SFString(self.channelInterpretation).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.description:
            attributeResult += "        " + '"@description":"' + SFString(self.description).JSON() + '"' + ',\n'
        if not self.enabled: # default=true
            attributeResult += "        " + '"@enabled":"' + SFBool(self.enabled).JSON() + '"' + ',\n'
        if self.fftSize != 2048:
            attributeResult += "        " + '"@fftSize":"' + SFInt32(self.fftSize).JSON() + '"' + ',\n'
        if self.frequencyBinCount != 1024:
            attributeResult += "        " + '"@frequencyBinCount":"' + SFInt32(self.frequencyBinCount).JSON() + '"' + ',\n'
        if self.gain != 1:
            attributeResult += "        " + '"@gain":"' + SFFloat(self.gain).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.maxDecibels != -30:
            attributeResult += "        " + '"@maxDecibels":"' + SFFloat(self.maxDecibels).JSON() + '"' + ',\n'
        if self.minDecibels != -100:
            attributeResult += "        " + '"@minDecibels":"' + SFFloat(self.minDecibels).JSON() + '"' + ',\n'
        if self.pauseTime != 0:
            attributeResult += "        " + '"@pauseTime":"' + SFTime(self.pauseTime).JSON() + '"' + ',\n'
        if self.resumeTime != 0:
            attributeResult += "        " + '"@resumeTime":"' + SFTime(self.resumeTime).JSON() + '"' + ',\n'
        if self.smoothingTimeConstant != 0.8:
            attributeResult += "        " + '"@smoothingTimeConstant":"' + SFFloat(self.smoothingTimeConstant).JSON() + '"' + ',\n'
        if self.startTime != 0:
            attributeResult += "        " + '"@startTime":"' + SFTime(self.startTime).JSON() + '"' + ',\n'
        if self.stopTime != 0:
            attributeResult += "        " + '"@stopTime":"' + SFTime(self.stopTime).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.tailTime != 0:
            attributeResult += "        " + '"@tailTime":"' + SFTime(self.tailTime).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* Analyser found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Analyser.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Analyser' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Analyser' + ' {'
        if self.channelCountMode != 'max':
            result += '\n' + indent + '  ' + "channelCountMode " +  '"' + self.channelCountMode + '"' + ""
        if self.channelInterpretation != 'speakers':
            result += '\n' + indent + '  ' + "channelInterpretation " +  '"' + self.channelInterpretation + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += '\n' + indent + '  ' + "description " +  '"' + self.description + '"' + ""
        if not self.enabled: # default=true
            result += '\n' + indent + '  ' + "enabled " + SFBool(self.enabled).VRML() + ""
        if self.fftSize != 2048:
            result += '\n' + indent + '  ' + "fftSize " + SFInt32(self.fftSize).VRML() + ""
        if self.frequencyBinCount != 1024:
            result += '\n' + indent + '  ' + "frequencyBinCount " + SFInt32(self.frequencyBinCount).VRML() + ""
        if self.gain != 1:
            result += '\n' + indent + '  ' + "gain " + SFFloat(self.gain).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.maxDecibels != -30:
            result += '\n' + indent + '  ' + "maxDecibels " + SFFloat(self.maxDecibels).VRML() + ""
        if self.minDecibels != -100:
            result += '\n' + indent + '  ' + "minDecibels " + SFFloat(self.minDecibels).VRML() + ""
        if self.pauseTime != 0:
            result += '\n' + indent + '  ' + "pauseTime " + SFTime(self.pauseTime).VRML() + ""
        if self.resumeTime != 0:
            result += '\n' + indent + '  ' + "resumeTime " + SFTime(self.resumeTime).VRML() + ""
        if self.smoothingTimeConstant != 0.8:
            result += '\n' + indent + '  ' + "smoothingTimeConstant " + SFFloat(self.smoothingTimeConstant).VRML() + ""
        if self.startTime != 0:
            result += '\n' + indent + '  ' + "startTime " + SFTime(self.startTime).VRML() + ""
        if self.stopTime != 0:
            result += '\n' + indent + '  ' + "stopTime " + SFTime(self.stopTime).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.tailTime != 0:
            result += '\n' + indent + '  ' + "tailTime " + SFTime(self.tailTime).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class Anchor(_X3DGroupingNode, _X3DUrlObject):
    """
    Anchor is a Grouping node that can contain most nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'Anchor'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/networking.html#Anchor'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Anchor'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('autoRefresh', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DUrlObject'),
        ('autoRefreshTimeLimit', 3600, FieldType.SFTime, AccessType.inputOutput, 'X3DUrlObject'),
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxDisplay', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('load', True, FieldType.SFBool, AccessType.inputOutput, 'X3DUrlObject'),
        ('parameter', [], FieldType.MFString, AccessType.inputOutput, 'Anchor'),
        ('url', [], FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        autoRefresh=0,
        autoRefreshTimeLimit=3600,
        bboxCenter=(0, 0, 0),
        bboxDisplay=False,
        bboxSize=(-1, -1, -1),
        description='',
        load=True,
        parameter=None,
        url=None,
        visible=True,
        children=None,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode Anchor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.autoRefresh = autoRefresh
        self.autoRefreshTimeLimit = autoRefreshTimeLimit
        self.bboxCenter = bboxCenter
        self.bboxDisplay = bboxDisplay
        self.bboxSize = bboxSize
        self.description = description
        self.load = load
        self.parameter = parameter
        self.url = url
        self.visible = visible
        self.children = children
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def autoRefresh(self):
        """The [autoRefresh field has no effect, Anchor operation is only triggered by user selection."""
        return self.__autoRefresh
    @autoRefresh.setter
    def autoRefresh(self, autoRefresh):
        if  autoRefresh is None:
            autoRefresh = 0 # default
        assertValidSFTime(autoRefresh)
        assertNonNegative('autoRefresh', autoRefresh)
        self.__autoRefresh = autoRefresh
    @property # getter - - - - - - - - - -
    def autoRefreshTimeLimit(self):
        """The [autoRefreshTimeLimit field has no effect, Anchor operation is only triggered by user selection."""
        return self.__autoRefreshTimeLimit
    @autoRefreshTimeLimit.setter
    def autoRefreshTimeLimit(self, autoRefreshTimeLimit):
        if  autoRefreshTimeLimit is None:
            autoRefreshTimeLimit = 3600 # default
        assertValidSFTime(autoRefreshTimeLimit)
        assertNonNegative('autoRefreshTimeLimit', autoRefreshTimeLimit)
        self.__autoRefreshTimeLimit = autoRefreshTimeLimit
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxDisplay(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__bboxDisplay
    @bboxDisplay.setter
    def bboxDisplay(self, bboxDisplay):
        if  bboxDisplay is None:
            bboxDisplay = False # default
        assertValidSFBool(bboxDisplay)
        self.__bboxDisplay = bboxDisplay
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """or [0,+infinity) Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided prose that describes intended purpose of the url asset."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def load(self):
        """The load field has no effect, Anchor operation is only triggered by user selection."""
        return self.__load
    @load.setter
    def load(self, load):
        if  load is None:
            load = True # default
        assertValidSFBool(load)
        self.__load = load
    @property # getter - - - - - - - - - -
    def parameter(self):
        """If provided, parameter tells the X3D player where to to redirect the loaded url."""
        return self.__parameter
    @parameter.setter
    def parameter(self, parameter):
        if  parameter is None:
            parameter = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(parameter)
        self.__parameter = parameter
    @property # getter - - - - - - - - - -
    def url(self):
        """Address of replacement world, or #ViewpointDEFName within the current scene, or alternate Web resource, activated by the user selecting Shape geometry within the Anchor children nodes."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata or (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Anchor.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Anchor'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.autoRefresh != 0:
            result += " autoRefresh='" + SFTime(self.autoRefresh).XML() + "'"
        if self.autoRefreshTimeLimit != 3600:
            result += " autoRefreshTimeLimit='" + SFTime(self.autoRefreshTimeLimit).XML() + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxDisplay: # default=false
            result += " bboxDisplay='" + SFBool(self.bboxDisplay).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.description:
            result += " description='" + SFString(self.description).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if not self.load: # default=true
            result += " load='" + SFBool(self.load).XML() + "'"
        if self.parameter != []:
            result += " parameter='" + MFString(self.parameter).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.url != []:
            result += " url='" + MFString(self.url).XML() + "'"
        if not self.visible: # default=true
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Anchor>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.children: # walk each child in list, if any
            ### print('* Anchor found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</Anchor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Anchor.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"Anchor":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.autoRefresh != 0:
            attributeResult += "        " + '"@autoRefresh":"' + SFTime(self.autoRefresh).JSON() + '"' + ',\n'
        if self.autoRefreshTimeLimit != 3600:
            attributeResult += "        " + '"@autoRefreshTimeLimit":"' + SFTime(self.autoRefreshTimeLimit).JSON() + '"' + ',\n'
        if self.bboxCenter != (0, 0, 0):
            attributeResult += "        " + '"@bboxCenter":"' + SFVec3f(self.bboxCenter).JSON() + '"' + ',\n'
        if self.bboxDisplay: # default=false
            attributeResult += "        " + '"@bboxDisplay":"' + SFBool(self.bboxDisplay).JSON() + '"' + ',\n'
        if self.bboxSize != (-1, -1, -1):
            attributeResult += "        " + '"@bboxSize":"' + SFVec3f(self.bboxSize).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.description:
            attributeResult += "        " + '"@description":"' + SFString(self.description).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if not self.load: # default=true
            attributeResult += "        " + '"@load":"' + SFBool(self.load).JSON() + '"' + ',\n'
        if self.parameter != []:
            attributeResult += "        " + '"@parameter":"' + MFString(self.parameter).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.url != []:
            attributeResult += "        " + '"@url":"' + MFString(self.url).JSON() + '"' + ',\n'
        if not self.visible: # default=true
            attributeResult += "        " + '"@visible":"' + SFBool(self.visible).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* Anchor found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Anchor.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Anchor' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Anchor' + ' {'
        if self.autoRefresh != 0:
            result += '\n' + indent + '  ' + "autoRefresh " + SFTime(self.autoRefresh).VRML() + ""
        if self.autoRefreshTimeLimit != 3600:
            result += '\n' + indent + '  ' + "autoRefreshTimeLimit " + SFTime(self.autoRefreshTimeLimit).VRML() + ""
        if self.bboxCenter != (0, 0, 0):
            result += '\n' + indent + '  ' + "bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxDisplay: # default=false
            result += '\n' + indent + '  ' + "bboxDisplay " + SFBool(self.bboxDisplay).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += '\n' + indent + '  ' + "bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += '\n' + indent + '  ' + "description " +  '"' + self.description + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if not self.load: # default=true
            result += '\n' + indent + '  ' + "load " + SFBool(self.load).VRML() + ""
        if self.parameter != []:
            result += '\n' + indent + '  ' + "parameter " + MFString(self.parameter).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.url != []:
            result += '\n' + indent + '  ' + "url " + MFString(self.url).VRML() + ""
        if not self.visible: # default=true
            result += '\n' + indent + '  ' + "visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class Appearance(_X3DAppearanceNode):
    """
    Appearance specifies the visual properties of geometry by containing the Material, ImageTexture/MovieTexture/PixelTexture, FillProperties, LineProperties, programmable shader nodes (ComposedShader, PackagedShader, ProgramShader) and TextureTransform nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'Appearance'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/shape.html#Appearance'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Appearance'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('alphaCutoff', 0.5, FieldType.SFFloat, AccessType.inputOutput, 'Appearance'),
        ('alphaMode', 'AUTO', FieldType.SFString, AccessType.inputOutput, 'Appearance'),
        ('acousticProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('fillProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('lineProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('material', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('pointProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('texture', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('textureTransform', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('shaders', [], FieldType.MFNode, AccessType.inputOutput, 'Appearance'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        alphaCutoff=0.5,
        alphaMode='AUTO',
        acousticProperties=None,
        fillProperties=None,
        lineProperties=None,
        material=None,
        pointProperties=None,
        texture=None,
        textureTransform=None,
        shaders=None,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode Appearance __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.alphaCutoff = alphaCutoff
        self.alphaMode = alphaMode
        self.acousticProperties = acousticProperties
        self.fillProperties = fillProperties
        self.lineProperties = lineProperties
        self.material = material
        self.pointProperties = pointProperties
        self.texture = texture
        self.textureTransform = textureTransform
        self.shaders = shaders
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def alphaCutoff(self):
        """[0,1] Threshold value used for pixel rendering either transparent or opaque, used when alphaMode="MASK"."""
        return self.__alphaCutoff
    @alphaCutoff.setter
    def alphaCutoff(self, alphaCutoff):
        if  alphaCutoff is None:
            alphaCutoff = 0.5 # default
        assertValidSFFloat(alphaCutoff)
        assertZeroToOne('alphaCutoff', alphaCutoff)
        self.__alphaCutoff = alphaCutoff
    @property # getter - - - - - - - - - -
    def alphaMode(self):
        """Provides options for control of alpha transparency handling for textures."""
        return self.__alphaMode
    @alphaMode.setter
    def alphaMode(self, alphaMode):
        if  alphaMode is None:
            alphaMode = 'AUTO' # default
        assertValidSFString(alphaMode)
        assertValidAlphaMode('alphaMode', alphaMode)
        self.__alphaMode = alphaMode
    @property # getter - - - - - - - - - -
    def acousticProperties(self):
        """[AcousticProperties] Single contained acousticProperties node that can specify additional acoustic attributes applied to associated surface geometry."""
        return self.__acousticProperties
    @acousticProperties.setter
    def acousticProperties(self, acousticProperties):
        if  acousticProperties is None:
            acousticProperties = None # default
        assertValidSFNode(acousticProperties)
        if not acousticProperties is None and not isinstance(acousticProperties,(AcousticProperties,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(acousticProperties) + ' does not match required node type (AcousticProperties,ProtoInstance) and is invalid')
        self.__acousticProperties = acousticProperties
    @property # getter - - - - - - - - - -
    def fillProperties(self):
        """[FillProperties] Single contained FillProperties node that can specify additional visual attributes applied to polygonal areas of corresponding geometry, on top of whatever other appearance is already defined."""
        return self.__fillProperties
    @fillProperties.setter
    def fillProperties(self, fillProperties):
        if  fillProperties is None:
            fillProperties = None # default
        assertValidSFNode(fillProperties)
        if not fillProperties is None and not isinstance(fillProperties,(FillProperties,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(fillProperties) + ' does not match required node type (FillProperties,ProtoInstance) and is invalid')
        self.__fillProperties = fillProperties
    @property # getter - - - - - - - - - -
    def lineProperties(self):
        """[LineProperties] Single contained LineProperties node that can specify additional visual attributes applied to corresponding line geometry."""
        return self.__lineProperties
    @lineProperties.setter
    def lineProperties(self, lineProperties):
        if  lineProperties is None:
            lineProperties = None # default
        assertValidSFNode(lineProperties)
        if not lineProperties is None and not isinstance(lineProperties,(LineProperties,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(lineProperties) + ' does not match required node type (LineProperties,ProtoInstance) and is invalid')
        self.__lineProperties = lineProperties
    @property # getter - - - - - - - - - -
    def material(self):
        """[X3DMaterialNode] Single contained Material node that can specify visual attributes for lighting response (color types, transparency, etc."""
        return self.__material
    @material.setter
    def material(self, material):
        if  material is None:
            material = None # default
        assertValidSFNode(material)
        if not material is None and not isinstance(material,(_X3DMaterialNode,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(material) + ' does not match required node type (_X3DMaterialNode,ProtoInstance) and is invalid')
        self.__material = material
    @property # getter - - - - - - - - - -
    def pointProperties(self):
        """[LineProperties] Single contained PointProperties node that can specify additional visual attributes applied to corresponding point geometry."""
        return self.__pointProperties
    @pointProperties.setter
    def pointProperties(self, pointProperties):
        if  pointProperties is None:
            pointProperties = None # default
        assertValidSFNode(pointProperties)
        if not pointProperties is None and not isinstance(pointProperties,(PointProperties,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(pointProperties) + ' does not match required node type (PointProperties,ProtoInstance) and is invalid')
        self.__pointProperties = pointProperties
    @property # getter - - - - - - - - - -
    def texture(self):
        """[X3DTextureNode] Single contained texture node (ImageTexture, MovieTexture, PixelTexture, MultiTexture) that maps image(s) to surface geometry."""
        return self.__texture
    @texture.setter
    def texture(self, texture):
        if  texture is None:
            texture = None # default
        assertValidSFNode(texture)
        if not texture is None and not isinstance(texture,(_X3DTextureNode,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(texture) + ' does not match required node type (_X3DTextureNode,ProtoInstance) and is invalid')
        self.__texture = texture
    @property # getter - - - - - - - - - -
    def textureTransform(self):
        """[X3DTextureTransformNode] Single contained TextureTransform node that defines 2D transformation applied to texture coordinates."""
        return self.__textureTransform
    @textureTransform.setter
    def textureTransform(self, textureTransform):
        if  textureTransform is None:
            textureTransform = None # default
        assertValidSFNode(textureTransform)
        if not textureTransform is None and not isinstance(textureTransform,(_X3DTextureTransformNode,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(textureTransform) + ' does not match required node type (_X3DTextureTransformNode,ProtoInstance) and is invalid')
        self.__textureTransform = textureTransform
    @property # getter - - - - - - - - - -
    def shaders(self):
        """[X3DShaderNode] Zero or more contained programmable shader nodes (ComposedShader, PackagedShader, ProgramShader) that specify, in order of preference, author-programmed rendering characteristics."""
        return self.__shaders
    @shaders.setter
    def shaders(self, shaders):
        if  shaders is None:
            shaders = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(shaders)
        self.__shaders = shaders
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.acousticProperties or self.fillProperties or self.IS or self.lineProperties or self.material or self.metadata or self.pointProperties or self.texture or self.textureTransform or (len(self.shaders) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Appearance.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Appearance'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.alphaCutoff != 0.5:
            result += " alphaCutoff='" + SFFloat(self.alphaCutoff).XML() + "'"
        if self.alphaMode != 'AUTO':
            result += " alphaMode='" + SFString(self.alphaMode).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Appearance>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.acousticProperties: # output this SFNode
                result += self.acousticProperties.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.fillProperties: # output this SFNode
                result += self.fillProperties.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.lineProperties: # output this SFNode
                result += self.lineProperties.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.material: # output this SFNode
                result += self.material.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.pointProperties: # output this SFNode
                result += self.pointProperties.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.texture: # output this SFNode
                result += self.texture.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.textureTransform: # output this SFNode
                result += self.textureTransform.XML(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.shaders: # walk each child in list, if any
            ### print('* Appearance found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(shaders)=' + str(len(self.shaders)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.shaders: # walk each child in list, if any (avoid empty list recursion)
                for each in self.shaders:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</Appearance>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Appearance.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"Appearance":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.alphaCutoff != 0.5:
            attributeResult += "        " + '"@alphaCutoff":"' + SFFloat(self.alphaCutoff).JSON() + '"' + ',\n'
        if self.alphaMode != 'AUTO':
            attributeResult += "        " + '"@alphaMode":"' + SFString(self.alphaMode).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.acousticProperties: # output this SFNode
                result += self.acousticProperties.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.fillProperties: # output this SFNode
                result += self.fillProperties.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.lineProperties: # output this SFNode
                result += self.lineProperties.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.material: # output this SFNode
                result += self.material.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.pointProperties: # output this SFNode
                result += self.pointProperties.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.texture: # output this SFNode
                result += self.texture.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.textureTransform: # output this SFNode
                result += self.textureTransform.JSON(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.shaders: # walk each child in list, if any (avoid empty list recursion)
            ### print('* Appearance found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(shaders)=' + str(len(self.shaders)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.shaders: # walk each child in list, if any (avoid empty list recursion)
                for each in self.shaders:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Appearance.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Appearance' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Appearance' + ' {'
        if self.alphaCutoff != 0.5:
            result += '\n' + indent + '  ' + "alphaCutoff " + SFFloat(self.alphaCutoff).VRML() + ""
        if self.alphaMode != 'AUTO':
            result += '\n' + indent + '  ' + "alphaMode " +  '"' + self.alphaMode + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.acousticProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'acousticProperties ' + self.acousticProperties.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.fillProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'fillProperties ' + self.fillProperties.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.lineProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'lineProperties ' + self.lineProperties.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.material: # output this SFNode
            result += '\n' + '  ' + indent + 'material ' + self.material.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.pointProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'pointProperties ' + self.pointProperties.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.texture: # output this SFNode
            result += '\n' + '  ' + indent + 'texture ' + self.texture.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.textureTransform: # output this SFNode
            result += '\n' + '  ' + indent + 'textureTransform ' + self.textureTransform.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.shaders: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.shaders:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class Arc2D(_X3DGeometryNode):
    """
    Arc2D is a line-based geometry node that defines a linear circular arc with center (0,0) in X-Y plane, with angles measured starting at positive x-axis and sweeping towards positive y-axis.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'Arc2D'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/geometry2D.html#Arc2D'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Arc2D'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('endAngle', 1.570796, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('startAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        endAngle=1.570796,
        radius=1,
        startAngle=0,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode Arc2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.endAngle = endAngle
        self.radius = radius
        self.startAngle = startAngle
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def endAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__endAngle
    @endAngle.setter
    def endAngle(self, endAngle):
        if  endAngle is None:
            endAngle = 1.570796 # default
        assertValidSFFloat(endAngle)
        assertGreaterThan('endAngle', endAngle, -6.2832)
        assertLessThan('endAngle', endAngle, 6.2832)
        self.__endAngle = endAngle
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius, of which the arc is a portion."""
        return self.__radius
    @radius.setter
    def radius(self, radius):
        if  radius is None:
            radius = 1 # default
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def startAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__startAngle
    @startAngle.setter
    def startAngle(self, startAngle):
        if  startAngle is None:
            startAngle = 0 # default
        assertValidSFFloat(startAngle)
        assertGreaterThan('startAngle', startAngle, -6.2832)
        assertLessThan('startAngle', startAngle, 6.2832)
        self.__startAngle = startAngle
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Arc2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Arc2D'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.endAngle != 1.570796:
            result += " endAngle='" + SFFloat(self.endAngle).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.radius != 1:
            result += " radius='" + SFFloat(self.radius).XML() + "'"
        if self.startAngle != 0:
            result += " startAngle='" + SFFloat(self.startAngle).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Arc2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</Arc2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Arc2D.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"Arc2D":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.endAngle != 1.570796:
            attributeResult += "        " + '"@endAngle":"' + SFFloat(self.endAngle).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.radius != 1:
            attributeResult += "        " + '"@radius":"' + SFFloat(self.radius).JSON() + '"' + ',\n'
        if self.startAngle != 0:
            attributeResult += "        " + '"@startAngle":"' + SFFloat(self.startAngle).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Arc2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Arc2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Arc2D' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.endAngle != 1.570796:
            result += '\n' + indent + '  ' + "endAngle " + SFFloat(self.endAngle).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.radius != 1:
            result += '\n' + indent + '  ' + "radius " + SFFloat(self.radius).VRML() + ""
        if self.startAngle != 0:
            result += '\n' + indent + '  ' + "startAngle " + SFFloat(self.startAngle).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class ArcClose2D(_X3DGeometryNode):
    """
    ArcClose2D is a polygonal geometry node that defines a linear circular arc, closed by PIE or CHORD line segments, with center (0,0) in X-Y plane, with angles measured starting at positive x-axis and sweeping towards positive y-axis.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'ArcClose2D'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/geometry2D.html#ArcClose2D'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ArcClose2D'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('closureType', 'PIE', FieldType.SFString, AccessType.initializeOnly, 'ArcClose2D'),
        ('endAngle', 1.570796, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('solid', False, FieldType.SFBool, AccessType.inputOutput, 'ArcClose2D'),
        ('startAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        closureType='PIE',
        endAngle=1.570796,
        radius=1,
        solid=False,
        startAngle=0,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode ArcClose2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.closureType = closureType
        self.endAngle = endAngle
        self.radius = radius
        self.solid = solid
        self.startAngle = startAngle
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def closureType(self):
        """Defines whether pair of line segments connect to center (PIE), or single line-segment chord connects arc endpoints (CHORD)."""
        return self.__closureType
    @closureType.setter
    def closureType(self, closureType):
        if  closureType is None:
            closureType = 'PIE' # default
        assertValidSFString(closureType)
        assertValidClosureType('closureType', closureType)
        self.__closureType = closureType
    @property # getter - - - - - - - - - -
    def endAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__endAngle
    @endAngle.setter
    def endAngle(self, endAngle):
        if  endAngle is None:
            endAngle = 1.570796 # default
        assertValidSFFloat(endAngle)
        assertGreaterThan('endAngle', endAngle, -6.2832)
        assertLessThan('endAngle', endAngle, 6.2832)
        self.__endAngle = endAngle
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius, of which the arc is a portion."""
        return self.__radius
    @radius.setter
    def radius(self, radius):
        if  radius is None:
            radius = 1 # default
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid):
        if  solid is None:
            solid = False # default
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def startAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__startAngle
    @startAngle.setter
    def startAngle(self, startAngle):
        if  startAngle is None:
            startAngle = 0 # default
        assertValidSFFloat(startAngle)
        assertGreaterThan('startAngle', startAngle, -6.2832)
        assertLessThan('startAngle', startAngle, 6.2832)
        self.__startAngle = startAngle
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ArcClose2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ArcClose2D'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.closureType != 'PIE':
            result += " closureType='" + SFString(self.closureType).XML() + "'"
        if self.endAngle != 1.570796:
            result += " endAngle='" + SFFloat(self.endAngle).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.radius != 1:
            result += " radius='" + SFFloat(self.radius).XML() + "'"
        if self.solid: # default=false
            result += " solid='" + SFBool(self.solid).XML() + "'"
        if self.startAngle != 0:
            result += " startAngle='" + SFFloat(self.startAngle).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ArcClose2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</ArcClose2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ArcClose2D.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"ArcClose2D":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.closureType != 'PIE':
            attributeResult += "        " + '"@closureType":"' + SFString(self.closureType).JSON() + '"' + ',\n'
        if self.endAngle != 1.570796:
            attributeResult += "        " + '"@endAngle":"' + SFFloat(self.endAngle).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.radius != 1:
            attributeResult += "        " + '"@radius":"' + SFFloat(self.radius).JSON() + '"' + ',\n'
        if self.solid: # default=false
            attributeResult += "        " + '"@solid":"' + SFBool(self.solid).JSON() + '"' + ',\n'
        if self.startAngle != 0:
            attributeResult += "        " + '"@startAngle":"' + SFFloat(self.startAngle).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ArcClose2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ArcClose2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ArcClose2D' + ' {'
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.closureType != 'PIE':
            result += '\n' + indent + '  ' + "closureType " +  '"' + self.closureType + '"' + ""
        if self.endAngle != 1.570796:
            result += '\n' + indent + '  ' + "endAngle " + SFFloat(self.endAngle).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.radius != 1:
            result += '\n' + indent + '  ' + "radius " + SFFloat(self.radius).VRML() + ""
        if self.solid: # default=false
            result += '\n' + indent + '  ' + "solid " + SFBool(self.solid).VRML() + ""
        if self.startAngle != 0:
            result += '\n' + indent + '  ' + "startAngle " + SFFloat(self.startAngle).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class AudioClip(_X3DSoundSourceNode, _X3DUrlObject):
    """
    AudioClip provides audio data used by parent Sound nodes.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'AudioClip'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#AudioClip'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#AudioClip'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('autoRefresh', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DUrlObject'),
        ('autoRefreshTimeLimit', 3600, FieldType.SFTime, AccessType.inputOutput, 'X3DUrlObject'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSoundSourceNode'),
        ('gain', 1, FieldType.SFFloat, AccessType.inputOutput, 'X3DSoundSourceNode'),
        ('load', True, FieldType.SFBool, AccessType.inputOutput, 'X3DUrlObject'),
        ('loop', False, FieldType.SFBool, AccessType.inputOutput, 'AudioClip'),
        ('pauseTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('pitch', 1.0, FieldType.SFFloat, AccessType.inputOutput, 'AudioClip'),
        ('resumeTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('startTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('stopTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('url', [], FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        autoRefresh=0,
        autoRefreshTimeLimit=3600,
        description='',
        enabled=True,
        gain=1,
        load=True,
        loop=False,
        pauseTime=0,
        pitch=1.0,
        resumeTime=0,
        startTime=0,
        stopTime=0,
        url=None,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode AudioClip __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.autoRefresh = autoRefresh
        self.autoRefreshTimeLimit = autoRefreshTimeLimit
        self.description = description
        self.enabled = enabled
        self.gain = gain
        self.load = load
        self.loop = loop
        self.pauseTime = pauseTime
        self.pitch = pitch
        self.resumeTime = resumeTime
        self.startTime = startTime
        self.stopTime = stopTime
        self.url = url
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def autoRefresh(self):
        """autoRefresh defines interval in seconds before automatic reload of current url asset is performed."""
        return self.__autoRefresh
    @autoRefresh.setter
    def autoRefresh(self, autoRefresh):
        if  autoRefresh is None:
            autoRefresh = 0 # default
        assertValidSFTime(autoRefresh)
        assertNonNegative('autoRefresh', autoRefresh)
        self.__autoRefresh = autoRefresh
    @property # getter - - - - - - - - - -
    def autoRefreshTimeLimit(self):
        """autoRefreshTimeLimit defines maximum duration that automatic refresh activity can occur."""
        return self.__autoRefreshTimeLimit
    @autoRefreshTimeLimit.setter
    def autoRefreshTimeLimit(self, autoRefreshTimeLimit):
        if  autoRefreshTimeLimit is None:
            autoRefreshTimeLimit = 3600 # default
        assertValidSFTime(autoRefreshTimeLimit)
        assertNonNegative('autoRefreshTimeLimit', autoRefreshTimeLimit)
        self.__autoRefreshTimeLimit = autoRefreshTimeLimit
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided prose that describes intended purpose of the url asset."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def gain(self):
        """(-infinity,+infinity) The gain field is a factor that represents the amount of linear amplification to apply to the output of the node."""
        return self.__gain
    @gain.setter
    def gain(self, gain):
        if  gain is None:
            gain = 1 # default
        assertValidSFFloat(gain)
        self.__gain = gain
    @property # getter - - - - - - - - - -
    def load(self):
        """load=true means load immediately, load=false means defer loading or else unload a previously loaded scene."""
        return self.__load
    @load.setter
    def load(self, load):
        if  load is None:
            load = True # default
        assertValidSFBool(load)
        self.__load = load
    @property # getter - - - - - - - - - -
    def loop(self):
        """Repeat indefinitely when loop=true, repeat only once when loop=false."""
        return self.__loop
    @loop.setter
    def loop(self, loop):
        if  loop is None:
            loop = False # default
        assertValidSFBool(loop)
        self.__loop = loop
    @property # getter - - - - - - - - - -
    def pauseTime(self):
        """When time now >= pauseTime, isPaused becomes true and AudioClip becomes paused."""
        return self.__pauseTime
    @pauseTime.setter
    def pauseTime(self, pauseTime):
        if  pauseTime is None:
            pauseTime = 0 # default
        assertValidSFTime(pauseTime)
        self.__pauseTime = pauseTime
    @property # getter - - - - - - - - - -
    def pitch(self):
        """(0,+infinity) Multiplier for the rate at which sampled sound is played."""
        return self.__pitch
    @pitch.setter
    def pitch(self, pitch):
        if  pitch is None:
            pitch = 1.0 # default
        assertValidSFFloat(pitch)
        assertPositive('pitch', pitch)
        self.__pitch = pitch
    @property # getter - - - - - - - - - -
    def resumeTime(self):
        """When resumeTime becomes <= time now, isPaused becomes false and AudioClip becomes active."""
        return self.__resumeTime
    @resumeTime.setter
    def resumeTime(self, resumeTime):
        if  resumeTime is None:
            resumeTime = 0 # default
        assertValidSFTime(resumeTime)
        self.__resumeTime = resumeTime
    @property # getter - - - - - - - - - -
    def startTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__startTime
    @startTime.setter
    def startTime(self, startTime):
        if  startTime is None:
            startTime = 0 # default
        assertValidSFTime(startTime)
        self.__startTime = startTime
    @property # getter - - - - - - - - - -
    def stopTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__stopTime
    @stopTime.setter
    def stopTime(self, stopTime):
        if  stopTime is None:
            stopTime = 0 # default
        assertValidSFTime(stopTime)
        self.__stopTime = stopTime
    @property # getter - - - - - - - - - -
    def url(self):
        """Location and filename of sound file or stream."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AudioClip.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<AudioClip'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.autoRefresh != 0:
            result += " autoRefresh='" + SFTime(self.autoRefresh).XML() + "'"
        if self.autoRefreshTimeLimit != 3600:
            result += " autoRefreshTimeLimit='" + SFTime(self.autoRefreshTimeLimit).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.description:
            result += " description='" + SFString(self.description).XML() + "'"
        if not self.enabled: # default=true
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.gain != 1:
            result += " gain='" + SFFloat(self.gain).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if not self.load: # default=true
            result += " load='" + SFBool(self.load).XML() + "'"
        if self.loop: # default=false
            result += " loop='" + SFBool(self.loop).XML() + "'"
        if self.pauseTime != 0:
            result += " pauseTime='" + SFTime(self.pauseTime).XML() + "'"
        if self.pitch != 1.0:
            result += " pitch='" + SFFloat(self.pitch).XML() + "'"
        if self.resumeTime != 0:
            result += " resumeTime='" + SFTime(self.resumeTime).XML() + "'"
        if self.startTime != 0:
            result += " startTime='" + SFTime(self.startTime).XML() + "'"
        if self.stopTime != 0:
            result += " stopTime='" + SFTime(self.stopTime).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.url != []:
            result += " url='" + MFString(self.url).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></AudioClip>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</AudioClip>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AudioClip.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"AudioClip":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.autoRefresh != 0:
            attributeResult += "        " + '"@autoRefresh":"' + SFTime(self.autoRefresh).JSON() + '"' + ',\n'
        if self.autoRefreshTimeLimit != 3600:
            attributeResult += "        " + '"@autoRefreshTimeLimit":"' + SFTime(self.autoRefreshTimeLimit).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.description:
            attributeResult += "        " + '"@description":"' + SFString(self.description).JSON() + '"' + ',\n'
        if not self.enabled: # default=true
            attributeResult += "        " + '"@enabled":"' + SFBool(self.enabled).JSON() + '"' + ',\n'
        if self.gain != 1:
            attributeResult += "        " + '"@gain":"' + SFFloat(self.gain).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if not self.load: # default=true
            attributeResult += "        " + '"@load":"' + SFBool(self.load).JSON() + '"' + ',\n'
        if self.loop: # default=false
            attributeResult += "        " + '"@loop":"' + SFBool(self.loop).JSON() + '"' + ',\n'
        if self.pauseTime != 0:
            attributeResult += "        " + '"@pauseTime":"' + SFTime(self.pauseTime).JSON() + '"' + ',\n'
        if self.pitch != 1.0:
            attributeResult += "        " + '"@pitch":"' + SFFloat(self.pitch).JSON() + '"' + ',\n'
        if self.resumeTime != 0:
            attributeResult += "        " + '"@resumeTime":"' + SFTime(self.resumeTime).JSON() + '"' + ',\n'
        if self.startTime != 0:
            attributeResult += "        " + '"@startTime":"' + SFTime(self.startTime).JSON() + '"' + ',\n'
        if self.stopTime != 0:
            attributeResult += "        " + '"@stopTime":"' + SFTime(self.stopTime).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.url != []:
            attributeResult += "        " + '"@url":"' + MFString(self.url).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function AudioClip.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'AudioClip' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'AudioClip' + ' {'
        if self.autoRefresh != 0:
            result += '\n' + indent + '  ' + "autoRefresh " + SFTime(self.autoRefresh).VRML() + ""
        if self.autoRefreshTimeLimit != 3600:
            result += '\n' + indent + '  ' + "autoRefreshTimeLimit " + SFTime(self.autoRefreshTimeLimit).VRML() + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += '\n' + indent + '  ' + "description " +  '"' + self.description + '"' + ""
        if not self.enabled: # default=true
            result += '\n' + indent + '  ' + "enabled " + SFBool(self.enabled).VRML() + ""
        if self.gain != 1:
            result += '\n' + indent + '  ' + "gain " + SFFloat(self.gain).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if not self.load: # default=true
            result += '\n' + indent + '  ' + "load " + SFBool(self.load).VRML() + ""
        if self.loop: # default=false
            result += '\n' + indent + '  ' + "loop " + SFBool(self.loop).VRML() + ""
        if self.pauseTime != 0:
            result += '\n' + indent + '  ' + "pauseTime " + SFTime(self.pauseTime).VRML() + ""
        if self.pitch != 1.0:
            result += '\n' + indent + '  ' + "pitch " + SFFloat(self.pitch).VRML() + ""
        if self.resumeTime != 0:
            result += '\n' + indent + '  ' + "resumeTime " + SFTime(self.resumeTime).VRML() + ""
        if self.startTime != 0:
            result += '\n' + indent + '  ' + "startTime " + SFTime(self.startTime).VRML() + ""
        if self.stopTime != 0:
            result += '\n' + indent + '  ' + "stopTime " + SFTime(self.stopTime).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.url != []:
            result += '\n' + indent + '  ' + "url " + MFString(self.url).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class AudioDestination(_X3DSoundDestinationNode):
    """
    AudioDestination node represents the final audio destination and is what user ultimately hears, typically from the speakers of user device.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'AudioDestination'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/sound.html#AudioDestination'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#AudioDestination'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('channelCountMode', 'max', FieldType.SFString, AccessType.inputOutput, 'X3DSoundDestinationNode'),
        ('channelInterpretation', 'speakers', FieldType.SFString, AccessType.inputOutput, 'X3DSoundDestinationNode'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DSoundNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSoundNode'),
        ('gain', 1, FieldType.SFFloat, AccessType.inputOutput, 'X3DSoundDestinationNode'),
        ('maxChannelCount', 2, FieldType.SFInt32, AccessType.inputOutput, 'AudioDestination'),
        ('mediaDeviceID', '', FieldType.SFString, AccessType.inputOutput, 'X3DSoundDestinationNode'),
        ('children', [], FieldType.MFNode, AccessType.inputOutput, 'AudioDestination'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        channelCountMode='max',
        channelInterpretation='speakers',
        description='',
        enabled=True,
        gain=1,
        maxChannelCount=2,
        mediaDeviceID='',
        children=None,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode AudioDestination __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.channelCountMode = channelCountMode
        self.channelInterpretation = channelInterpretation
        self.description = description
        self.enabled = enabled
        self.gain = gain
        self.maxChannelCount = maxChannelCount
        self.mediaDeviceID = mediaDeviceID
        self.children = children
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def channelCountMode(self):
        """channelCountMode determines how individual channels are counted when up-mixing and down-mixing connections to any inputs."""
        return self.__channelCountMode
    @channelCountMode.setter
    def channelCountMode(self, channelCountMode):
        if  channelCountMode is None:
            channelCountMode = 'max' # default
        assertValidSFString(channelCountMode)
        assertValidChannelCountMode('channelCountMode', channelCountMode)
        self.__channelCountMode = channelCountMode
    @property # getter - - - - - - - - - -
    def channelInterpretation(self):
        """channelInterpretation determines how individual channels are treated when up-mixing and down-mixing connections to any inputs."""
        return self.__channelInterpretation
    @channelInterpretation.setter
    def channelInterpretation(self, channelInterpretation):
        if  channelInterpretation is None:
            channelInterpretation = 'speakers' # default
        assertValidSFString(channelInterpretation)
        assertValidChannelInterpretation('channelInterpretation', channelInterpretation)
        self.__channelInterpretation = channelInterpretation
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided prose that describes intended purpose of the url asset."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def gain(self):
        """(-infinity,+infinity) The gain field is a factor that represents the amount of linear amplification to apply to the output of the node."""
        return self.__gain
    @gain.setter
    def gain(self, gain):
        if  gain is None:
            gain = 1 # default
        assertValidSFFloat(gain)
        self.__gain = gain
    @property # getter - - - - - - - - - -
    def maxChannelCount(self):
        """[0,+infinity) [maxChannelCount."""
        return self.__maxChannelCount
    @maxChannelCount.setter
    def maxChannelCount(self, maxChannelCount):
        if  maxChannelCount is None:
            maxChannelCount = 2 # default
        assertValidSFInt32(maxChannelCount)
        assertNonNegative('maxChannelCount', maxChannelCount)
        self.__maxChannelCount = maxChannelCount
    @property # getter - - - - - - - - - -
    def mediaDeviceID(self):
        """mediaDeviceID field provides ID parameter functionality."""
        return self.__mediaDeviceID
    @mediaDeviceID.setter
    def mediaDeviceID(self, mediaDeviceID):
        if  mediaDeviceID is None:
            mediaDeviceID = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(mediaDeviceID)
        self.__mediaDeviceID = mediaDeviceID
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DSoundChannelNode|X3DSoundProcessingNode|X3DSoundSourceNode] The children field specifies audio-graph sound sources providing input signals for this node."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFNode.DEFAULT_VALUE()=' + str(MFNode.DEFAULT_VALUE()))
        assertValidMFNode(children)
        self.__children = children
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata or (len(self.children) > 0)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AudioDestination.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<AudioDestination'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.channelCountMode != 'max':
            result += " channelCountMode='" + SFString(self.channelCountMode).XML() + "'"
        if self.channelInterpretation != 'speakers':
            result += " channelInterpretation='" + SFString(self.channelInterpretation).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.description:
            result += " description='" + SFString(self.description).XML() + "'"
        if not self.enabled: # default=true
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.gain != 1:
            result += " gain='" + SFFloat(self.gain).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.maxChannelCount != 2:
            result += " maxChannelCount='" + SFInt32(self.maxChannelCount).XML() + "'"
        if self.mediaDeviceID:
            result += " mediaDeviceID='" + SFString(self.mediaDeviceID).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></AudioDestination>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.children: # walk each child in list, if any
            ### print('* AudioDestination found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</AudioDestination>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AudioDestination.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"AudioDestination":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.channelCountMode != 'max':
            attributeResult += "        " + '"@channelCountMode":"' + SFString(self.channelCountMode).JSON() + '"' + ',\n'
        if self.channelInterpretation != 'speakers':
            attributeResult += "        " + '"@channelInterpretation":"' + SFString(self.channelInterpretation).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.description:
            attributeResult += "        " + '"@description":"' + SFString(self.description).JSON() + '"' + ',\n'
        if not self.enabled: # default=true
            attributeResult += "        " + '"@enabled":"' + SFBool(self.enabled).JSON() + '"' + ',\n'
        if self.gain != 1:
            attributeResult += "        " + '"@gain":"' + SFFloat(self.gain).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.maxChannelCount != 2:
            attributeResult += "        " + '"@maxChannelCount":"' + SFInt32(self.maxChannelCount).JSON() + '"' + ',\n'
        if self.mediaDeviceID:
            attributeResult += "        " + '"@mediaDeviceID":"' + SFString(self.mediaDeviceID).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            ### if self.children: # walk each child in list, if any (avoid empty list recursion)
            ### print('* AudioDestination found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking JSON(' + str(indentLevel+1) + ')', flush=True)
            if self.children: # walk each child in list, if any (avoid empty list recursion)
                for each in self.children:
                    result += each.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function AudioDestination.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'AudioDestination' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'AudioDestination' + ' {'
        if self.channelCountMode != 'max':
            result += '\n' + indent + '  ' + "channelCountMode " +  '"' + self.channelCountMode + '"' + ""
        if self.channelInterpretation != 'speakers':
            result += '\n' + indent + '  ' + "channelInterpretation " +  '"' + self.channelInterpretation + '"' + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += '\n' + indent + '  ' + "description " +  '"' + self.description + '"' + ""
        if not self.enabled: # default=true
            result += '\n' + indent + '  ' + "enabled " + SFBool(self.enabled).VRML() + ""
        if self.gain != 1:
            result += '\n' + indent + '  ' + "gain " + SFFloat(self.gain).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.maxChannelCount != 2:
            result += '\n' + indent + '  ' + "maxChannelCount " + SFInt32(self.maxChannelCount).VRML() + ""
        if self.mediaDeviceID:
            result += '\n' + indent + '  ' + "mediaDeviceID " +  '"' + self.mediaDeviceID + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.children: # walk each child in list, if any (avoid empty list recursion)
            result += '\n' + indent + '  ' + 'children [' + '\n' + indent + '  ' + '  '
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+2, VRML97=VRML97)
            result += '\n' + indent + '  ' + ']' + '\n' + indent
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class Background(_X3DBackgroundNode):
    """
    Background simulates ground and sky, using vertical arrays of wraparound color values.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'Background'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/environmentalEffects.html#Background'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Background'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('backUrl', [], FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('bottomUrl', [], FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('frontUrl', [], FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('groundAngle', [], FieldType.MFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('groundColor', [], FieldType.MFColor, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('leftUrl', [], FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('rightUrl', [], FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('skyAngle', [], FieldType.MFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('skyColor', [(0, 0, 0)], FieldType.MFColor, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('topUrl', [], FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('transparency', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        backUrl=None,
        bottomUrl=None,
        frontUrl=None,
        groundAngle=None,
        groundColor=None,
        leftUrl=None,
        rightUrl=None,
        skyAngle=None,
        skyColor=None,
        topUrl=None,
        transparency=0,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode Background __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.backUrl = backUrl
        self.bottomUrl = bottomUrl
        self.frontUrl = frontUrl
        self.groundAngle = groundAngle
        self.groundColor = groundColor
        self.leftUrl = leftUrl
        self.rightUrl = rightUrl
        self.skyAngle = skyAngle
        self.skyColor = skyColor
        self.topUrl = topUrl
        self.transparency = transparency
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def backUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__backUrl
    @backUrl.setter
    def backUrl(self, backUrl):
        if  backUrl is None:
            backUrl = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(backUrl)
        self.__backUrl = backUrl
    @property # getter - - - - - - - - - -
    def bottomUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__bottomUrl
    @bottomUrl.setter
    def bottomUrl(self, bottomUrl):
        if  bottomUrl is None:
            bottomUrl = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(bottomUrl)
        self.__bottomUrl = bottomUrl
    @property # getter - - - - - - - - - -
    def frontUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__frontUrl
    @frontUrl.setter
    def frontUrl(self, frontUrl):
        if  frontUrl is None:
            frontUrl = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(frontUrl)
        self.__frontUrl = frontUrl
    @property # getter - - - - - - - - - -
    def groundAngle(self):
        """[0,pi/2] The angle array values increase from 0."""
        return self.__groundAngle
    @groundAngle.setter
    def groundAngle(self, groundAngle):
        if  groundAngle is None:
            groundAngle = MFFloat.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFFloat.DEFAULT_VALUE()=' + str(MFFloat.DEFAULT_VALUE()))
        assertValidMFFloat(groundAngle)
        assertGreaterThanEquals('groundAngle', groundAngle, 0)
        assertLessThanEquals('groundAngle', groundAngle, 1.5708)
        self.__groundAngle = groundAngle
    @property # getter - - - - - - - - - -
    def groundColor(self):
        """Color of the ground at the various angles on the ground partial sphere."""
        return self.__groundColor
    @groundColor.setter
    def groundColor(self, groundColor):
        if  groundColor is None:
            groundColor = MFColor.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFColor.DEFAULT_VALUE()=' + str(MFColor.DEFAULT_VALUE()))
        assertValidMFColor(groundColor)
        assertZeroToOne('groundColor', groundColor)
        self.__groundColor = groundColor
    @property # getter - - - - - - - - - -
    def leftUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__leftUrl
    @leftUrl.setter
    def leftUrl(self, leftUrl):
        if  leftUrl is None:
            leftUrl = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(leftUrl)
        self.__leftUrl = leftUrl
    @property # getter - - - - - - - - - -
    def rightUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__rightUrl
    @rightUrl.setter
    def rightUrl(self, rightUrl):
        if  rightUrl is None:
            rightUrl = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(rightUrl)
        self.__rightUrl = rightUrl
    @property # getter - - - - - - - - - -
    def skyAngle(self):
        """[0,pi] The angle array values increase from 0."""
        return self.__skyAngle
    @skyAngle.setter
    def skyAngle(self, skyAngle):
        if  skyAngle is None:
            skyAngle = MFFloat.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFFloat.DEFAULT_VALUE()=' + str(MFFloat.DEFAULT_VALUE()))
        assertValidMFFloat(skyAngle)
        assertGreaterThanEquals('skyAngle', skyAngle, 0)
        assertLessThanEquals('skyAngle', skyAngle, 3.1416)
        self.__skyAngle = skyAngle
    @property # getter - - - - - - - - - -
    def skyColor(self):
        """Color of the sky at various angles on the sky sphere."""
        return self.__skyColor
    @skyColor.setter
    def skyColor(self, skyColor):
        if  skyColor is None:
            skyColor = [(0, 0, 0)] # default
        assertValidMFColor(skyColor)
        assertZeroToOne('skyColor', skyColor)
        self.__skyColor = skyColor
    @property # getter - - - - - - - - - -
    def topUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__topUrl
    @topUrl.setter
    def topUrl(self, topUrl):
        if  topUrl is None:
            topUrl = MFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to MFString.DEFAULT_VALUE()=' + str(MFString.DEFAULT_VALUE()))
        assertValidMFString(topUrl)
        self.__topUrl = topUrl
    @property # getter - - - - - - - - - -
    def transparency(self):
        """[0,1] how "clear" the background is, allows underlying page to show through: 1."""
        return self.__transparency
    @transparency.setter
    def transparency(self, transparency):
        if  transparency is None:
            transparency = 0 # default
        assertValidSFFloat(transparency)
        assertZeroToOne('transparency', transparency)
        self.__transparency = transparency
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.IS or self.metadata
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Background.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Background'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.backUrl != []:
            result += " backUrl='" + MFString(self.backUrl).XML() + "'"
        if self.bottomUrl != []:
            result += " bottomUrl='" + MFString(self.bottomUrl).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.frontUrl != []:
            result += " frontUrl='" + MFString(self.frontUrl).XML() + "'"
        if self.groundAngle != []:
            result += " groundAngle='" + MFFloat(self.groundAngle).XML() + "'"
        if self.groundColor != []:
            result += " groundColor='" + MFColor(self.groundColor).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.leftUrl != []:
            result += " leftUrl='" + MFString(self.leftUrl).XML() + "'"
        if self.rightUrl != []:
            result += " rightUrl='" + MFString(self.rightUrl).XML() + "'"
        if self.skyAngle != []:
            result += " skyAngle='" + MFFloat(self.skyAngle).XML() + "'"
        if self.skyColor != [(0, 0, 0)]:
            result += " skyColor='" + MFColor(self.skyColor).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if self.topUrl != []:
            result += " topUrl='" + MFString(self.topUrl).XML() + "'"
        if self.transparency != 0:
            result += " transparency='" + SFFloat(self.transparency).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Background>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</Background>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Background.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"Background":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.backUrl != []:
            attributeResult += "        " + '"@backUrl":"' + MFString(self.backUrl).JSON() + '"' + ',\n'
        if self.bottomUrl != []:
            attributeResult += "        " + '"@bottomUrl":"' + MFString(self.bottomUrl).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.frontUrl != []:
            attributeResult += "        " + '"@frontUrl":"' + MFString(self.frontUrl).JSON() + '"' + ',\n'
        if self.groundAngle != []:
            attributeResult += "        " + '"@groundAngle":"' + MFFloat(self.groundAngle).JSON() + '"' + ',\n'
        if self.groundColor != []:
            attributeResult += "        " + '"@groundColor":"' + MFColor(self.groundColor).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.leftUrl != []:
            attributeResult += "        " + '"@leftUrl":"' + MFString(self.leftUrl).JSON() + '"' + ',\n'
        if self.rightUrl != []:
            attributeResult += "        " + '"@rightUrl":"' + MFString(self.rightUrl).JSON() + '"' + ',\n'
        if self.skyAngle != []:
            attributeResult += "        " + '"@skyAngle":"' + MFFloat(self.skyAngle).JSON() + '"' + ',\n'
        if self.skyColor != [(0, 0, 0)]:
            attributeResult += "        " + '"@skyColor":"' + MFColor(self.skyColor).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"' + ',\n'
        if self.topUrl != []:
            attributeResult += "        " + '"@topUrl":"' + MFString(self.topUrl).JSON() + '"' + ',\n'
        if self.transparency != 0:
            attributeResult += "        " + '"@transparency":"' + SFFloat(self.transparency).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Background.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Background' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Background' + ' {'
        if self.backUrl != []:
            result += '\n' + indent + '  ' + "backUrl " + MFString(self.backUrl).VRML() + ""
        if self.bottomUrl != []:
            result += '\n' + indent + '  ' + "bottomUrl " + MFString(self.bottomUrl).VRML() + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.frontUrl != []:
            result += '\n' + indent + '  ' + "frontUrl " + MFString(self.frontUrl).VRML() + ""
        if self.groundAngle != []:
            result += '\n' + indent + '  ' + "groundAngle " + MFFloat(self.groundAngle).VRML() + ""
        if self.groundColor != []:
            result += '\n' + indent + '  ' + "groundColor " + MFColor(self.groundColor).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.leftUrl != []:
            result += '\n' + indent + '  ' + "leftUrl " + MFString(self.leftUrl).VRML() + ""
        if self.rightUrl != []:
            result += '\n' + indent + '  ' + "rightUrl " + MFString(self.rightUrl).VRML() + ""
        if self.skyAngle != []:
            result += '\n' + indent + '  ' + "skyAngle " + MFFloat(self.skyAngle).VRML() + ""
        if self.skyColor != [(0, 0, 0)]:
            result += '\n' + indent + '  ' + "skyColor " + MFColor(self.skyColor).VRML() + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        if self.topUrl != []:
            result += '\n' + indent + '  ' + "topUrl " + MFString(self.topUrl).VRML() + ""
        if self.transparency != 0:
            result += '\n' + indent + '  ' + "transparency " + SFFloat(self.transparency).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1, VRML97=VRML97)
        else:
            result += ' '
        if not self.USE:
            result += '\n' + indent + '}' +  '\n' + indent
#       print('VRML serialization complete.', flush=True)
        return result

class BallJoint(_X3DRigidJointNode):
    """
    BallJoint represents an unconstrained joint between two bodies that pivot about a common anchor point.
    """
    # immutable constant functions have getter but no setter - - - - - - - - - -
    @classmethod
    def NAME(cls):
        """ Name of this X3D Node class. """
        return 'BallJoint'
    @classmethod
    def SPECIFICATION_URL(cls):
        """ Extensible 3D (X3D) Graphics International Standard governs X3D architecture for all file formats and programming languages. """
        return 'https://www.web3d.org/specifications/X3Dv4Draft/ISO-IEC19775-1v4-CD1/Part01/components/rigidBodyPhysics.html#BallJoint'
    @classmethod
    def TOOLTIP_URL(cls):
        """ X3D Tooltips provide authoring tips, hints and warnings for each node and field in X3D. """
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BallJoint'
    @classmethod
    def FIELD_DECLARATIONS(cls):
        """ Field declarations for this node: name, defaultValue, type, accessType, inheritedFrom """
        return [
        ('anchorPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'BallJoint'),
        ('forceOutput', ["NONE"], FieldType.MFString, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('id_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('style_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
        anchorPoint=(0, 0, 0),
        forceOutput=None,
        body1=None,
        body2=None,
        DEF='',
        USE='',
        IS=None,
        metadata=None,
        class_='',
        id_='',
        style_=''):
        # if _DEBUG: print('...DEBUG... in ConcreteNode BallJoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(id_) + ',' + str(style_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, id_, style_, IS, metadata) # fields for _X3DNode only
        self.anchorPoint = anchorPoint
        self.forceOutput = forceOutput
        self.body1 = body1
        self.body2 = body2
        self.id_ = id_
        self.style_ = style_
    @property # getter - - - - - - - - - -
    def anchorPoint(self):
        """anchorPoint is joint center, specified in world coordinates."""
        return self.__anchorPoint
    @anchorPoint.setter
    def anchorPoint(self, anchorPoint):
        if  anchorPoint is None:
            anchorPoint = (0, 0, 0) # default
        assertValidSFVec3f(anchorPoint)
        self.__anchorPoint = anchorPoint
    @property # getter - - - - - - - - - -
    def forceOutput(self):
        """forceOutput controls which output fields are generated for the next frame."""
        return self.__forceOutput
    @forceOutput.setter
    def forceOutput(self, forceOutput):
        if  forceOutput is None:
            forceOutput = ["NONE"] # default
        assertValidMFString(forceOutput)
        self.__forceOutput = forceOutput
    @property # getter - - - - - - - - - -
    def body1(self):
        """[RigidBody] The body1 and body2 fields indicate the two RigidBody nodes connected by this joint."""
        return self.__body1
    @body1.setter
    def body1(self, body1):
        if  body1 is None:
            body1 = None # default
        assertValidSFNode(body1)
        if not body1 is None and not isinstance(body1,(RigidBody,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not match required node type (RigidBody,ProtoInstance) and is invalid')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        """[RigidBody] The body1 and body2 fields indicate the two RigidBody nodes connected by this joint."""
        return self.__body2
    @body2.setter
    def body2(self, body2):
        if  body2 is None:
            body2 = None # default
        assertValidSFNode(body2)
        if not body2 is None and not isinstance(body2,(RigidBody,ProtoInstance)):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not match required node type (RigidBody,ProtoInstance) and is invalid')
        self.__body2 = body2
    @property # getter - - - - - - - - - -
    def id_(self):
        """ id_ attribute is a unique identifier for use within HTML pages. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__id_
    @id_.setter
    def id_(self, id_):
        if  id_ is None:
            id_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(id_)
        self.__id_ = id_
    @property # getter - - - - - - - - - -
    def style_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading style_sheets. Appended underscore to field name to avoid naming collision with Python reserved word. """
        return self.__style_
    @style_.setter
    def style_(self, style_):
        if  style_ is None:
            style_ = SFString.DEFAULT_VALUE()
            # if _DEBUG: print('...DEBUG... set value to SFString.DEFAULT_VALUE()=' + str(SFString.DEFAULT_VALUE()))
        assertValidSFString(style_)
        self.__style_ = style_
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        """ Whether or not this node has any child node or statement """
        return self.body1 or self.body2 or self.IS or self.metadata
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BallJoint.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BallJoint'
        if self.DEF:
            result += " DEF='" + SFString(self.DEF).XML() + "'"
        if self.USE:
            result += " USE='" + SFString(self.USE).XML() + "'"
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint='" + SFVec3f(self.anchorPoint).XML() + "'"
        if self.class_:
            result += " class='" + SFString(self.class_).XML() + "'"
        if self.forceOutput != ["NONE"]:
            result += " forceOutput='" + MFString(self.forceOutput).XML() + "'"
        if self.id_:
            result += " id='" + SFString(self.id_).XML() + "'"
        if self.style_:
            result += " style='" + SFString(self.style_).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BallJoint>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.body1: # output this SFNode
                result += self.body1.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.body2: # output this SFNode
                result += self.body2.XML(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '</BallJoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def JSON(self, indentLevel=0, syntax="JSON"):
        """ Provide X3D output serialization using JSON encoding. """
        result = ''
        indent = '    ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BallJoint.JSON(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '"BallJoint":\n'
        result += indent + '{\n'
        attributeResult = ''
        if self.DEF:
            attributeResult += "        " + '"@DEF":"' + SFString(self.DEF).JSON() + '"' + ',\n'
        if self.USE:
            attributeResult += "        " + '"@USE":"' + SFString(self.USE).JSON() + '"' + ',\n'
        if self.anchorPoint != (0, 0, 0):
            attributeResult += "        " + '"@anchorPoint":"' + SFVec3f(self.anchorPoint).JSON() + '"' + ',\n'
        if self.class_:
            attributeResult += "        " + '"@class":"' + SFString(self.class_).JSON() + '"' + ',\n'
        if self.forceOutput != ["NONE"]:
            attributeResult += "        " + '"@forceOutput":"' + MFString(self.forceOutput).JSON() + '"' + ',\n'
        if self.id_:
            attributeResult += "        " + '"@id":"' + SFString(self.id_).JSON() + '"' + ',\n'
        if self.style_:
            attributeResult += "        " + '"@style":"' + SFString(self.style_).JSON() + '"'
        # print("attributeResult=" + attributeResult) # debug
        attributeResult = attributeResult.rstrip()
        if attributeResult.endswith(","):
            attributeResult = attributeResult[:-1] # remove trailing comma from last element of list
        if attributeResult:
            result += "      {\n" + attributeResult + '\n' + "      " + '}\n'
        if not self.hasChild():
            if syntax.upper() == "JSON":
                result += '    },' + '\n'
            else:
                raise X3DValueError('.toJSON(syntax=' + syntax + ') is incorrect, allowed value is "JSON"')
        else:
            if self.IS: # output this SFNode
                result += self.IS.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.body1: # output this SFNode
                result += self.body1.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.body2: # output this SFNode
                result += self.body2.JSON(indentLevel=indentLevel+1, syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.JSON(indentLevel=indentLevel+1, syntax=syntax)
            result += indent + '}' ### here? + '\n'
#       print('JSON serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel=indentLevel+1, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BallJoint.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BallJoint' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BallJoint' + ' {'
        if self.anchorPoint != (0, 0, 0):
            result += '\n' + indent + '  ' + "anchorPoint " + SFVec3f(self.anchorPoint).VRML() + ""
        if self.class_:
            result += '\n' + indent + '  ' + "class " +  '"' + self.class_ + '"' + ""
        if self.forceOutput != ["NONE"]:
            result += '\n' + indent + '  ' + "forceOutput " + MFString(self.forceOutput).VRML() + ""
        if self.id_:
            result += '\n' + indent + '  ' + "id " +  '"' + self.id_ + '"' + ""
        if self.style_:
            result += '\n' + indent + '  ' + "style " +  '"' + self.style_ + '"' + ""
        