# pf-reasoning-tool-proj/pf_reasoning_tool/tools/vision_llm_tool.py

from typing import Union, List, Optional

# PromptFlow specific imports
from promptflow.contracts.types import PromptTemplate
from promptflow.connections import AzureOpenAIConnection, OpenAIConnection
from promptflow.tools.exception import InvalidConnectionType
# Import the internal AOAI/OpenAI GPT4V helper classes used by the built-in tool
from promptflow.tools.aoai_gpt4v import AzureOpenAI as PFAzureOpenAIWrapper
from promptflow.tools.openai_gpt4v import OpenAI as PFOpenAIWrapper
# Error handling decorator
from promptflow.tools.common import handle_openai_error
# Avoid circular dependencies: Use import 'from promptflow._internal'
from promptflow._internal import tool


# Use the @tool decorator and error handler like the built-in tool
# Add streaming capability like the built-in tool if needed
@tool(streaming_option_parameter="stream")
@handle_openai_error()
def vision_llm( # Keep function name consistent
    connection: Union[AzureOpenAIConnection, OpenAIConnection],
    prompt: PromptTemplate, # <<< Input is PromptTemplate
    deployment_name: str = "",  # For Azure OpenAI
    model: str = "",           # For OpenAI
    temperature: float = 0.7,  # Your default
    top_p: float = 1.0,
    # stream is hidden to the end user, set by the executor.
    stream: bool = False,
    stop: Optional[List[str]] = None, # Use Optional for lists/dicts
    max_tokens: Optional[int] = 1000, # Set your default, use Optional
    presence_penalty: float = 0,
    frequency_penalty: float = 0,
    seed: Optional[int] = None,
    detail: str = 'auto', # Added detail parameter like built-in tool
    **kwargs, # Essential to capture rendered template variables
):
    """
    Calls a vision-capable LLM using the PromptFlow custom_llm pattern.

    Relies on the PromptTemplate containing image references (e.g., ![image]({{image_url}}))
    and uses internal PromptFlow helpers to handle the API call via chat.completions.

    Args:
        connection: The AzureOpenAI or OpenAI connection object.
        prompt: The PromptTemplate containing text and image references.
        deployment_name: The specific deployment name for Azure OpenAI vision model.
        model: The specific model name for OpenAI vision model (e.g., gpt-4-vision-preview).
        temperature (float): Controls randomness (0.0 to 2.0).
        top_p (float): Nucleus sampling parameter.
        stream (bool): Whether to stream the response (controlled by executor).
        stop (Optional[List[str]]): List of sequences to stop generation at.
        max_tokens (Optional[int]): Maximum number of tokens to generate.
        presence_penalty (float): Penalty for new tokens based on presence.
        frequency_penalty (float): Penalty for new tokens based on frequency.
        seed (Optional[int]): Seed for deterministic outputs (if supported).
        detail (str): Controls the detail level for image processing ('low', 'high', 'auto').
        **kwargs: Additional parameters passed from the rendered prompt template.

    Returns:
        The text response generated by the vision model's chat completion.
    """

    # Use the PromptFlow internal wrappers, similar to the built-in llm_vision tool
    if isinstance(connection, AzureOpenAIConnection):
        # Instantiate the PromptFlow AzureOpenAI wrapper for GPT-4V
        aoai_wrapper = PFAzureOpenAIWrapper(connection)
        # Call its chat method
        return aoai_wrapper.chat(
            prompt=prompt,
            deployment_name=deployment_name,
            temperature=temperature,
            top_p=top_p,
            stream=stream,
            stop=stop,
            max_tokens=max_tokens,
            presence_penalty=presence_penalty,
            frequency_penalty=frequency_penalty,
            seed=seed,
            detail=detail,
            **kwargs # Pass through any extra arguments from the prompt template render
        )
    elif isinstance(connection, OpenAIConnection):
        # Instantiate the PromptFlow OpenAI wrapper for GPT-4V
        openai_wrapper = PFOpenAIWrapper(connection)
        # Call its chat method
        return openai_wrapper.chat(
            prompt=prompt,
            model=model, # Use model parameter for OpenAI
            temperature=temperature,
            top_p=top_p,
            stream=stream,
            stop=stop,
            max_tokens=max_tokens,
            presence_penalty=presence_penalty,
            frequency_penalty=frequency_penalty,
            seed=seed,
            detail=detail,
            **kwargs # Pass through any extra arguments from the prompt template render
        )
    else:
        # Same error handling as the built-in tool
        error_message = f"Not Support connection type '{type(connection).__name__}' for vision llm. " \
                        "Connection type should be in [AzureOpenAIConnection, OpenAIConnection]."
        raise InvalidConnectionType(message=error_message)