# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['datargs', 'datargs.compat']

package_data = \
{'': ['*']}

extras_require = \
{':python_version == "3.6"': ['dataclasses>=0.6,<0.7'],
 'attrs': ['attrs>=20.2.0,<21.0.0']}

setup_kwargs = {
    'name': 'datargs',
    'version': '0.1.1',
    'description': 'Declerative, type-safe command line argument parsers from dataclasses and attrs classes',
    'long_description': '# datargs\nCreate type-safe command line argument parsers from attrs and dataclass classes.\n\n## Usage\nCreate a dataclass (or an `attrs` class) describing your command line interface, and just call\n`datargs.parse()` with the class:\n\n```python\n# script.py\nfrom dataclasses import dataclass\nfrom pathlib import Path\nfrom datargs import parse\n\n@dataclass\nclass Args:\n    url: str\n    output_path: Path\n    verbose: bool\n    retries: int = 3\n\ndef main():\n    args: Args = parse(Args)\n    print(args)\n\nif __name__ == "__main__":\n    main()\n```\n\nMypy is happy (and so is Pycharm):\n```bash\n$ mypy script.py\nSuccess: no issues found in 1 source file\n```\n\nYour script is good to go!\n```bash\n$ python script.py -h\nusage: test.py [-h] --url URL --output-path OUTPUT_PATH [--retries RETRIES]\n               [--verbose]\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --url URL\n  --output-path OUTPUT_PATH\n  --retries RETRIES\n  --verbose\n$ python script.py --url "https://..." --output-path out --retries 4 --verbose\nArgs(url="https://...", output_path=Path("out"), retries=4, verbose=True)\n```\n\nMypy/Pycharm have your back when you when you make a mistake:\n```python\n...\ndef main():\n    args = parse(Args)\n    args.urll  # typo\n...\n```\nPycharm says: `Unresolved attribute reference \'urll\' for class \'Args\'`.\n\nMypy says: `script.py:15: error: "Args" has no attribute "urll"; maybe "url"?`\n\nYou can use `attr.s` if you prefer:\n```pycon\n>>> import attr, datargs\n>>> @attr.s\n... class Args:\n...     flag: bool = attr.ib()\n>>> datargs.parse(Args, [])\nArgs(flag=False)\n```\n\nAdditional `ArgumentParser.add_argument()` parameters are taken from `metadata`:\n\n```pycon\n>>> from dataclasses import dataclass, field\n>>> from datargs import parse\n>>> @dataclass\n... class Args:\n...     retries: int = field(default=3, metadata=dict(help="number of retries", aliases=["-r"], metavar="RETRIES"))\n>>> parse(Args, ["-h"])\nusage: ...\noptional arguments:\n  -h, --help            show this help message and exit\n  --retries RETRIES, -r RETRIES\n>>> parse(Args, ["-r", "4"])\nArgs(retries=4)\n```\n\n`arg` is a replacement for field that puts `add_argument()` parameters in `metadata`. Use it to save precious keystrokes:\n```pycon\n>>> from dataclasses import dataclass\n>>> from datargs import parse, arg\n>>> @dataclass\n... class Args:\n...     retries: int = arg(default=3, help="number of retries", aliases=["-r"], metavar="RETRIES")\n...     # perhaps many more...\n>>> parse(Args, ["-h"])\n# exactly the same as before\n```\n\nAnd `argsclass` is a `dataclass` alias for extra style points:\n```python\nfrom datargs import argsclass, args\n@argsclass\nclass Args:\n    flag: bool = arg(help="MY FLAG")\n```\n\nTo add program descriptions etc. pass your own parser to `parse()`:\n```pycon\n>>> from argparse import ArgumentParser\n>>> from datargs import parse, argsclass\n>>> @argsclass\n... class Args:\n...     flag: bool\n>>> parser = ArgumentParser(description="Romans go home!", prog="messiah.py")\n>>> parse(Args, ["-h"], parser=parser)\nusage: messiah.py [-h] [--flag]\nRomans go home!\n...\n```\n\nUse `make_parser()` to create a parser and save it for later:\n```pycon\n>>> from datargs import make_parser\n>>> @datargs\n... class Args:\n...     ...\n>>> parser = make_parser(Args)  # pass `parser=...` to modify an existing parser\n```\n\n## Features\n- supports typing: code is statically checked to be correct\n- comptability with both `dataclass` and `attrs`\n- `args` supports all `field` and `attr.ib` arguments.\n- support for enums (passed by name):\n    ```pycon\n    >>> import enum, attr, datargs\n    >>> class FoodEnum(enum.Enum):\n    ...     ham = 0\n    ...     spam = 1\n    >>> @attr.dataclass\n    ... class Args:\n    ...     food: FoodEnum\n    >>> datargs.parse(Args, ["--food", "eggs"])\n    Args(food=<FoodEnum.ham: 0>)\n    >>> datargs.parse(Args, ["--food", "eggs"])\n    usage: enum_test.py [-h] --food {ham,spam}\n    enum_test.py: error: argument --food: invalid choice: \'eggs\' (choose from [\'ham\', \'spam\'])\n    ```\n\n## "Why not"s and design choices\nThere are many libraries out there that do similar things. This list serves as documentation for existing solutions and differences.\n\nSo, why not...\n\n### Just use argparse?\nThat\'s easy. The interface is clumsy and repetitive, a.k.a boilerplate. Additionally, `ArgumentParser.parse_args()` returns a `Namespace`, which is basically \nequivalent to `Any`, meaning that it any attribute access is legal when type checking. Alas, invalid attribute access will fail at runtime. For example:\n```python\ndef parse_args():\n    parser = ArgumentParser()\n    parser.add_argument("--url")\n    return parser.parse_args()\n \ndef main():\n    args = parse_args()\n    print(args.url)\n```\nLet\'s say you for some reason `--url` is changed to `--uri`:\n```python\nparser.add_argument("--uri")\n...\nprint(args.url)  # oops\n```\nYou won\'t discover you made a mistake until you run the code. With `datargs`, a static type checker will issue an error.\nAlso, why use a carriage when you have a spaceship?\n\n### Use [`click`](https://click.palletsprojects.com/en/7.x/)?\n`click` is a great library, but I believe user interface should not be coupled with implementation.\n\n### Use [`simple-parsing`](https://pypi.org/project/simple-parsing/)?\nThis is another impressive libarary. The drawbacks for me are:\n* argument documentation uses introspection hacks and has multiple ways to be used\n* options are always nested\n* underscores in argument names (`--like_this`)\nAn upside is that it lets you use your own parser, an important feature for composability and easy modification.\n\n### Use [`argparse-dataclass`](https://pypi.org/project/argparse-dataclass/)?\nIt\'s very similar to this library. The main differences I found are:\n* no `attrs` support\n* not on github, so who you gonna call?\n\n### Use [`argparse-dataclasses`](https://pypi.org/project/argparse-dataclasses/)?\nSams points `argparse-dataclass` but also [Uses inheritance](https://refactoring.guru/replace-inheritance-with-delegation).\n',
    'author': 'Roee Nizan',
    'author_email': 'roeen30@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/roee30/datargs',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
