import time

import requests
from tqdm import tqdm

from .web.urls import TASK_STATUS_URL

FAILURE = "FAILURE"
PENDING = "PENDING"
COMPLETE = "COMPLETE"
STOPPED = "STOPPED"

TASK_DONE_LIST = (FAILURE, COMPLETE, STOPPED)

import threading

bar_suffix = "{desc}: {percentage:.1f}%|{bar}| remaining: {remaining} || elapsed: {elapsed} "
one_hundred_percent = 100


class Task:

    def __init__(self, task_id, callback=None, show=True):
        self.task_id = task_id
        self.task_result = ""
        self.task_type = ""
        self.progress = 0
        self.task_state = PENDING
        self.callback = callback
        self.cache = None
        self._logs = {}
        self.prints = []
        if self.callback:
            t = threading.Thread(target=self.callback_thread)
            t.start()
        if show:
            self.update()
            print("Started {0}. View status at https://dashboard.neuro-ai.co.uk/tasks/{1}".format(self.task_type.lower(), task_id))

    def wait(self):
        if self.finished():
            return
        with tqdm(desc=self.task_type, total=one_hundred_percent,
                  bar_format=bar_suffix) as bar:
            while not self.finished():
                time.sleep(0.1)
                [bar.write(log) for log in self.prints]
                self.prints = []
                bar.n = self.progress * one_hundred_percent
                bar.refresh()
            if self.task_state == FAILURE:
                self.update(include_result=True)
                raise Exception("ERROR for task {}: {}".format(self.task_id, self.task_result))
            if self.task_state == STOPPED:
                print("Task has been stopped.")
                return
            bar.n = one_hundred_percent

    def callback_thread(self):
        self.get_result()
        self.callback(self)

    def get_result(self):
        self.wait()
        self.update(include_result=True)
        return self.task_result

    def update(self, include_result=False):
        from .common import getResponse
        params = {"include_result": include_result}
        response = requests.get(TASK_STATUS_URL + self.task_id, params=params)
        response = getResponse(response)
        self.task_state = response["state"]
        self.task_type = response["taskType"]
        self.progress = response["progress"]
        if "result" in response:
            self.task_result = response["result"]
        if "metrics" in response:
            self._logs = response["metrics"]
        if "print" in response:
            self.prints = response["print"]

    def __str__(self):
        return str(self.get_result())

    def finished(self):
        self.update()
        return self.task_state in TASK_DONE_LIST

    def logs(self):
        self.get_result()
        return self._logs


