from pathlib import Path
from typing import Union

BASE_OFFSET = 0x10  # the size of the rom header identifying the rom

TILE_LEVEL_1 = 0x03
TILE_LEVEL_2 = 0x04
TILE_LEVEL_3 = 0x05
TILE_LEVEL_4 = 0x06
TILE_LEVEL_5 = 0x07
TILE_LEVEL_6 = 0x08
TILE_LEVEL_7 = 0x09
TILE_LEVEL_8 = 0x0A
TILE_LEVEL_9 = 0x0B
TILE_LEVEL_10 = 0x0C
TILE_MUSHROOM_HOUSE_1 = 0x50
TILE_MUSHROOM_HOUSE_2 = 0xE0
TILE_STAR_1 = 0x55
TILE_STAR_2 = 0xE9
TILE_SPIRAL_TOWER_1 = 0x5F
TILE_SPIRAL_TOWER_2 = 0xDF
TILE_DUNGEON_1 = 0x67
TILE_DUNGEON_2 = 0xEB
TILE_QUICKSAND = 0x68
TILE_PYRAMID = 0x69
TILE_PIPE = 0xBC
TILE_POND = 0xBF
TILE_CASTLE_BOTTOM = 0xC9
TILE_BOWSER_CASTLE = 0xCC  # TILE_BOWSERCASTLELL
TILE_HAND_TRAP = 0xE6
TILE_SPADE_HOUSE = 0xE8

OBJ_AUTOSCROLL = 0xD3

POWERUP_MUSHROOM = 0x01
POWERUP_FIREFLOWER = 0x02
POWERUP_RACCOON = 0x03
POWERUP_FROG = 0x04
POWERUP_TANOOKI = 0x05
POWERUP_HAMMER = 0x06
POWERUP_ADDITION_JUDGEMS = 0x07
POWERUP_ADDITION_PWING = 0x08
POWERUP_ADDITION_STARMAN = 0x09


def update_global_offsets(path_to_global_list: Union[str, Path]):
    path_to_global_list = Path(path_to_global_list)

    if not path_to_global_list.exists():
        return

    with open(path_to_global_list, "r") as label_file:
        for line in label_file.readlines():
            label_name, global_address = line.split(":")

            if label_name.startswith("PRG0") or label_name.startswith("_"):
                continue

            global_address = int(global_address, 16)

            globals()[label_name] = BASE_OFFSET + global_address


# default values for all labels in the stock ROM after this line

AND7F = 0x381BC
ASFloat_VVelLimit = 0x13E41
ASFloat_VelAccel = 0x13D7D
ASFloat_VertLimit = 0x13E43
ASMLT_0 = 0x13B70
ASMLT_3 = 0x13B7D
ASMLT_4 = 0x13B81
ASMLT_5 = 0x13B84
ASMLT_6 = 0x13B8F
ASMLT_Airship = 0x13B78
ASMLT_Battleship = 0x13B74
ASML_0 = 0x13B42
ASML_3 = 0x13B4F
ASML_4 = 0x13B53
ASML_5 = 0x13B56
ASML_6 = 0x13B61
ASML_Airship = 0x13B4A
ASML_Battleship = 0x13B46
ASMR_CoinShip = 0x13B2C
ASMR_Terminator = 0x13B34
ASMR_UNK12 = 0x13B2D
ASMR_UNK4 = 0x13A6E
ASMR_UNK8 = 0x13AA9
ASMR_UNK9 = 0x13AB8
ASMR_W1Airship = 0x13AD7
ASMR_W1CoinHeaven = 0x13B2B
ASMR_W3_Airship = 0x13A4A
ASMR_W4Airship = 0x13A7A
ASMR_W5_Airship = 0x13A5C
ASMR_W6Airship = 0x13A95
ASMR_W7Airship = 0x13AEC
ASMR_W8Airship = 0x13B0D
ASMR_W8Battleship = 0x13B10
ASMR_World8Tank1 = 0x13B2F
ASMR_World8Tank2 = 0x13B32
ASMR_World_36_14 = 0x13A47
ASMR_World_56 = 0x13AA7
ASMR_World_62 = 0x13A5B
ASMR_World_67 = 0x13ACC
ASMR_World_74 = 0x13B16
ASM_CoinShip = 0x13A3F
ASM_UNK12 = 0x13A40
ASM_UNK4 = 0x13980
ASM_UNK8 = 0x139BC
ASM_UNK9 = 0x139CB
ASM_W1Airship = 0x139EA
ASM_W1CoinHeaven = 0x13A3E
ASM_W3_Airship = 0x1395C
ASM_W4Airship = 0x1398C
ASM_W5_Airship = 0x1396E
ASM_W6Airship = 0x139A7
ASM_W7Airship = 0x139FF
ASM_W8Airship = 0x13A20
ASM_W8Battleship = 0x13A23
ASM_World8Tank1 = 0x13A42
ASM_World8Tank2 = 0x13A45
ASM_World_36_14 = 0x13959
ASM_World_56 = 0x139BA
ASM_World_62 = 0x1396D
ASM_World_67 = 0x139DF
ASM_World_74 = 0x13A29
AScrlURDiag_CheckWrapping = 0x1EA4
AScrlURDiag_HandleWrap = 0x1E4E
AScrlURDiag_NoWrapAbort = 0x1EC0
AScroll_CoinShipCoinGlowPal = 0x13B9D
AScroll_HorizontalInitMove = 0x1ECC
AScroll_MoveEndLoopSelect = 0x13944
AScroll_MovePlayer = 0x13CB0
AScroll_Movement = 0x13959
AScroll_MovementLoop = 0x13B42
AScroll_MovementLoopStart = 0x13B39
AScroll_MovementLoopTicks = 0x13B70
AScroll_MovementRepeat = 0x13A47
AScroll_SpikeCeilVLimits = 0x13D81
AScroll_SpikeCeilVVelLimit = 0x13D7F
AScroll_VelAccel = 0x13B35
AirshipCtl_Catch = 0x3B2EB
AirshipCtl_DoNothing = 0x3B2CD
AirshipCtl_HoldAnchor = 0x3B312
AirshipCtl_LandOnDeck = 0x3B32C
AirshipCtl_RunAndJump = 0x3B2CE
Airship_JumpFrameByPup = 0x100AC
Airship_Layouts = 0x19291
Airship_Objects = 0x192A1
ArrowPlat_Attrs = 0xB039
ArrowPlat_ByTile = 0xE010
ArrowPlat_CheckWorldCollide = 0xB0B1
ArrowPlat_DirByTile = 0xE019
ArrowPlat_Draw = 0xB042
ArrowPlat_FlickerMask = 0xB03E
ArrowPlat_Patterns = 0xB025
ArrowPlat_XOff = 0xE022
ArrowPlat_XVel = 0xAF39
ArrowPlat_YVel = 0xAF3D
AutoScroll_ApplyHVel = 0x13F32
AutoScroll_ApplyVVel = 0x13F30
AutoScroll_BooRoom = 0x13D30
AutoScroll_CalcPlayerY = 0x11229
AutoScroll_Do = 0x13910
AutoScroll_Floating = 0x13E51
AutoScroll_Horizontal = 0x13BC4
AutoScroll_SpikeCeiling = 0x13D85
AutoScroll_URDiagonal = 0x13EB9
AutoScroll_URDiagonalLimits = 0x13EB1
Background_Layout = 0x2C503
Background_Tiles = 0x2C524
Bank2_HotFootHaltAction = 0x4869
Bank2_PiranhaSpikeHaltAction = 0x4F43
Bank2_SpikeHaltAction = 0x4DAE
BattleEnemy_ByEnterID = 0x8478
BigBerthaLCC_Draw = 0x72B9
BigBerthaLCC_XVelAccel = 0x7207
BigBerthaLCC_XVelLimit = 0x7209
BigBerthaLCC_XVelLowLimit = 0x720B
BigBertha_Accel = 0x5860
BigBertha_Draw = 0x59CA
BigBertha_EatLimit = 0x5864
BigBertha_HVisByFlip = 0x72B5
BigBertha_ProxLimit = 0x585C
BigBertha_SpitOutCheepCheep = 0x7358
BigBertha_Splash = 0x59AF
BigBertha_SpriteXOff = 0x72B7
BigBertha_XCoarseLimit = 0x5862
BigBertha_XHiMove = 0x725E
BigBertha_XMove = 0x725C
BigBertha_XVelLimit = 0x585E
BigCannonBall_Draw = 0xA3BE
BigQBlock1O = 0xC986
BigQBlock2O = 0xC988
BigQBlock3O = 0xC98D
BigQBlock4O = 0xC998
BigQBlock5O = 0xC9A0
BigQBlock6O = 0xC9A8
BigQBlock7O = 0xC9B3
BigQBlock8O = 0xC9BB
BigQBlock_Draw = 0xB769
BigQBlock_EmergePowerup = 0xB71E
BigQBlock_GotItBits = 0xB621
BigQBlock_Item = 0xB6E2
BigQBlock_LowerPats = 0xB761
BigQBlock_Open = 0xB6F0
BigQBlock_PlayerPushXVel = 0xB6E0
BigQBlock_StarManFlash = 0xB6E9
BigQBlock_UpperPats = 0xB759
BigQBlock_YVelByTimer = 0xB63F
Bill_CPXOff = 0xFDD6
Bill_CPYOff = 0xFDEC
Bill_FlipTowardsPlayer = 0xFF3B
Bill_Var4TowardsPlayer = 0xFF3D
Bill_XVelTowardsPlayer = 0xFF39
BlockBump_Init = 0x1DB5
BlockChange_Do = 0x3BC10
BlooperKid_Data = 0x787A
BlooperKid_SpriteYOff = 0xF06F
BlooperKid_UNKD = 0x787E
BlooperKid_VelAccel = 0xF06B
BlooperKid_VelLimit = 0xF06D
BlooperKid_XVel = 0x7876
BlooperKid_YVel = 0x7872
Blooper_FlipTowardsPlayer = 0x7778
Blooper_LaunchKids = 0x7882
Blooper_YVelAccel = 0x776F
Blooper_YVelLimit = 0x7771
BobOmbExp_StartXVel = 0x66C3
BobOmb_CalcULOffXY = 0x68E3
BobOmb_DoExplosion = 0x6838
BobOmb_Explode = 0x6789
BobOmb_FlashToExplode = 0x6777
BobOmb_Kill = 0x6801
BobOmb_StartXVel = 0x66C1
BobOmb_WalkAround = 0x66F9
BobombKickXVel = 0xE17
Bolt_CheckOnThread = 0xAC6A
Bolt_ToBoltCollide = 0xACA4
BombStars_XOff = 0x6820
BombStars_YOff = 0x682C
BonusCoin_ApplyVel = 0x2CE00
BonusCoin_ApplyXVel = 0x2CDE4
BonusCoin_ApplyYVel = 0x2CDF2
BonusCoins_Draw = 0x2CE2D
BonusCoins_Init = 0x2CD89
BonusCoins_UpdateAndDraw = 0x2CDBB
BonusGameBox_SpriteXs = 0x2C7DF
BonusGameBox_SpriteYs = 0x2C7D7
BonusGame_Do = 0x2C883
BonusGame_HostDraw = 0x2C6F6
BonusGame_Loop = 0x3CD1C
BonusGame_NSpade_Text = 0x2CAAB
BonusGame_PlayerPal = 0x37808
BonusGame_Spade_Text = 0x2CA6E
BonusGame_UNUSED0_Text = 0x2CA44
BonusGame_UNUSED3_Text = 0x2CAF4
BonusGame_UNUSED4_Text = 0x2CB05
BonusGame_UNUSED5_Text = 0x2CB0E
BonusGame_UNUSED6_Text = 0x2CB30
BonusGame_UNUSED7_Text = 0x2CB51
BonusGame_UNUSED_Etc0 = 0x2CB59
BonusGame_UNUSED_Etc1 = 0x2CB63
BonusGame_UNUSED_Etc2 = 0x2CB6F
BonusGame_UNUSED_Etc3 = 0x2CB90
BonusGame_UNUSED_Etc4 = 0x2CB9A
BonusGame_UNUSED_Etc5 = 0x2CB9F
BonusGame_UNUSED_Etc6 = 0x2CBAB
BonusGame_UNUSED_Etc7 = 0x2CBCC
BonusTable_Tiles = 0x2C4B8
BonusText_HostGreetPtrH = 0x2C86B
BonusText_HostGreetPtrL = 0x2C863
BonusText_Round2PtrH = 0x2C87B
BonusText_Round2PtrL = 0x2C873
BonusUNKTALL_Tiles = 0x2C5CE
Bonus_DieFlyAway = 0x2CCFE
Bonus_DieRotate = 0x2CC0F
Bonus_DoHostText = 0x2C917
Bonus_DoNothing = 0x2CA02
Bonus_DrawDie = 0x2CC6A
Bonus_GetDiePrize = 0x2CD61
Bonus_Init = 0x2C8CA
Bonus_InitRound2 = 0x2C9A5
Bonus_InstBoxBot = 0x2CA37
Bonus_InstBoxLine1 = 0x2CA10
Bonus_InstBoxLine2 = 0x2CA1D
Bonus_InstBoxLine3 = 0x2CA2A
Bonus_InstBoxTop = 0x2CA03
Bonus_KTPrizeFlipBits = 0x2C7E7
Bonus_KTPrizeGame = 0x2C9F0
Bonus_KTPrizePattern = 0x2C7E9
Bonus_LayoutData = 0x2C42F
Bonus_Prize1 = 0x3D623
Bonus_Return2Col = 0x2CBD7
Bonus_Return2Row = 0x2CBD3
Bonus_Return2_SetMapPos = 0x2CBDB
Bonus_Wait80 = 0x2CE57
Bonus_WaitA0 = 0x2C900
Boo_CheckPlayerSight = 0x48C9
Boo_VelAccel = 0x48DD
Boo_VelLimit = 0x48DF
BoomBoom_CalcFlightPath = 0x7CA8
BoomBoom_Death = 0x6E98
BoomBoom_Draw = 0x6DC6
BoomBoom_ExecuteFlight = 0x6CD6
BoomBoom_FinalAttack = 0x6D14
BoomBoom_FinalAttackXVel = 0x6A12
BoomBoom_FlightToPlan = 0x6C9D
BoomBoom_FlyAccel = 0x6C7C
BoomBoom_FlyXVelLimit = 0x6C7A
BoomBoom_FlyYVelAccel = 0x6C7E
BoomBoom_GetupFrames = 0x6A16
BoomBoom_HitTest = 0x6E4E
BoomBoom_Init = 0x6AFA
BoomBoom_PatternLeft = 0x6D5D
BoomBoom_PatternRight = 0x6D73
BoomBoom_PlanFlight = 0x6D4F
BoomBoom_PrimaryAttack = 0x6B3A
BoomBoom_ResetVar6 = 0x6B29
BoomBoom_SeconaryAttack = 0x6BEF
BoomBoom_SetXVelTowardsPlayer = 0x6C71
BoomBoom_SprYOffArmLeft = 0x6D89
BoomBoom_SprYOffArmRight = 0x6D9F
BoomBoom_TowardsPlayerXVel = 0x6A10
BoomBoom_Unused = 0x6B2F
BoomBoom_XVelAccel = 0x6B36
BoomBoom_XVelFastAccel = 0x6B38
BoomBoom_XVelFastLimit = 0x6B34
BoomBoom_XVelLimit = 0x6B32
BoomBoom_XVelSlide = 0x6B7A
BoomerangBro_FacePlayerFlip = 0x84A7
BoomerangBro_ThrowBoomerang = 0x8630
BoomerangBro_WalkLeft = 0x84F1
BoomerangBro_WalkRight = 0x84F5
BoomerangBro_WalkThenStop = 0x84E0
Boomerang_ArrayValLoad = 0x862E
Boomerang_Attributes = 0xF8B1
Boomerang_Patterns = 0xF8B5
Boomerang_XVel = 0x862C
Boomerang_XVelDelta = 0xF8A9
Boomerang_XVelLimit = 0xF8AB
Boomerang_YVelAccel = 0xF8AD
Boomerang_YVelLimit = 0xF8AF
BorderOut_Bottom = 0x34F0E
BorderOut_Do = 0x34E7D
BorderOut_Left = 0x34F5F
BorderOut_Right = 0x34ECC
BorderOut_Top = 0x34E8B
BorderOut_VHLimitB = 0x34E78
BorderOut_VHLimitTRL = 0x34E73
Border_Bottom = 0x34DCB
Border_BottomCorners = 0x14B8E
Border_Do = 0x34D78
Border_Left = 0x34DE6
Border_Right = 0x34DB6
Border_Top = 0x34D86
Border_TopCorners = 0x14B88
Border_VAttrMask = 0x14B31
Border_VAttrs = 0x14B3D
Border_VertEdges = 0x14B8B
BossAttack_Draw = 0x8460
BossAttack_Frame = 0x845C
BossAttack_HitOffCeiling = 0x8454
BossAttack_HitOffWall = 0x843A
BossAttack_Poof = 0x8352
BossAttack_SprRAMOff = 0x82BB
BossAttack_XVelTowardsPlayer = 0x829D
BounceBlock_Pal = 0x2611
BounceBlock_SprOff = 0x2619
BounceBlock_Tile = 0x261B
BounceBlock_Update = 0x262B
Bounce_TileReplacements = 0x2504
Bouncer_PUp = 0x24FC
Bouncer_PUpVel = 0x250C
Bowser_AlignAndFall = 0x3A87
Bowser_BreatheFire = 0x3B63
Bowser_BustFloor = 0x3C40
Bowser_BustFloorLookAround = 0x3AB9
Bowser_CalcPlayersSide = 0x3E64
Bowser_Counter3Do = 0x3BD7
Bowser_DetectTiles = 0x3BF0
Bowser_DoMovements = 0x3939
Bowser_DoTimeBonus = 0x3F1A
Bowser_DoVar5Action = 0x38F6
Bowser_DoorAppear = 0x3F27
Bowser_Draw = 0x3D9D
Bowser_FacePlayer = 0x3B26
Bowser_FallAndSplat = 0x3EA9
Bowser_FallToFloor = 0x396F
Bowser_FinalDoorColorCycle = 0x3F51
Bowser_FireballXOff = 0x3B59
Bowser_FireballXVel = 0x3B57
Bowser_FlipToFace = 0x3B30
Bowser_HandleIfDead = 0x3E73
Bowser_HeadBounceXVel = 0x3899
Bowser_HopAndBreatheFire = 0x3B3A
Bowser_HoppingFrames = 0x3B32
Bowser_JumpAndLandOnFloor = 0x3A0A
Bowser_LookAroundFlipBits = 0x3AF9
Bowser_LookAroundFrames = 0x3AF1
Bowser_SprHVisLUT = 0x3D71
Bowser_SprPats = 0x3CB1
Bowser_SprRAMOff = 0x3D81
Bowser_SprVVisLUT = 0x3D75
Bowser_SprXHFlipped = 0x3D4D
Bowser_SprXNotHFlipped = 0x3D41
Bowser_SprYNotVFlipped = 0x3D59
Bowser_SprYVFlipped = 0x3D65
Bowser_TileOffsets = 0x3C3E
Bowser_VsPlayerXVelNegBit = 0x3A08
Bowser_WaitAndVictory = 0x3F0D
Bowser_WaitForPlayer = 0x3907
Bowser_XVelByDist = 0x39B5
BoxOut_ByVStart = 0x3D545
BoxOut_CalcOffsets = 0x3D5BE
BoxOut_CalcWhich8x8 = 0x3D5EA
BoxOut_InitVAddrH = 0x3D54A
BoxOut_InitVAddrL0 = 0x3D54F
BoxOut_InitVAddrL1 = 0x3D554
BoxOut_InitVAddrL2 = 0x3D559
BoxOut_InitVAddrL3 = 0x3D55E
BoxOut_PutPatternInStrip = 0x3D649
BoxOut_SetThisBorderVRAM = 0x3D56E
BrickBust_DrawAndUpdate = 0xEBE4
BrickBust_MicroGoomba = 0x64EB
BrickBust_MoveOver = 0xDB4
BrickBust_SprRAMOff = 0xEBE0
BrickBusts_DrawAndUpdate = 0xEBD5
BrickPoof_SprRAMOff = 0xEBE2
Bubble_Draw = 0xE93F
Bubble_SprRAMOff = 0xE938
Bubble_UpdateAndDraw = 0xE888
Bubble_XOff = 0xE934
Bubbles_UpdateAndDraw = 0xE878
BulletBill_FlipBits = 0x9167
BulletBill_XAccel = 0x9163
BulletBill_XLimit = 0x9165
BustBlock_Segment = 0x6521
BustBlock_XVelByInput = 0x651C
BustBlock_YHiOffByInput = 0x650D
BustBlock_YOffByInput = 0x6508
BustBlock_YVelByInput = 0x6517
Buster_BlockLiftXOff = 0x460B
Buster_BlockLiftYOff = 0x4623
Buster_DrawHoldingIceBrick = 0x463B
Buster_XVel = 0x44D6
CBig_ChngTiles = 0x3BEDC
CBig_GiantBlock_BlockHit = 0x3BEA0
CBig_GiantBlock_BrickBust = 0x3BE83
CBig_GiantBlock_BrickRestore = 0x3BEBD
CBig_Offsets = 0x3BE44
CFire_4Way = 0xFDEE
CFire_BulletBill = 0xFF3F
CFire_Cannonball = 0xFC29
CFire_GoombaPipe = 0xFD25
CFire_Laser = 0xFBD7
CFire_RockyWrench = 0xFEF2
CannonFire_DrawAndUpdate = 0xFB91
CannonFire_Init = 0xBA56
CannonFire_NoiseAndSmoke = 0xFEBB
CannonFire_UpdateAndDraw = 0xFB7F
CannonPoof_XOffs = 0xFDC2
CannonPoof_YOffs = 0xFDD8
Cannonball_YDiffLimit = 0xF3B6
Cannonball_YOffset = 0xF3B4
Cannons_CPXOff = 0xFDCA
Cannons_CPYOff = 0xFDE0
CardBack_BufCmds = 0x2DDC1
CardBase_BufCmds = 0x2DD88
CardFlipSprite_GetPattern = 0x2DD2A
CardFlip_Attributes = 0x2D703
CardFlip_Draw = 0x2DCB3
CardFlip_Frame0 = 0x2D718
CardFlip_Frame1 = 0x2D714
CardFlip_Frame2 = 0x2D710
CardFlip_Frame3 = 0x2D70C
CardFlip_Frame4 = 0x2D708
CardFlip_FrameDefOff = 0x2D71C
CardFlip_FrameDefs = 0x2D708
CardLL = 0x3FCBE
CardLR = 0x3FCC2
CardUL = 0x3FCB6
CardUR = 0x3FCBA
CardVStartL = 0x3FCC9
CardVStartU = 0x3FCC6
Card_BeginFlip = 0x2D7C1
Card_BeginFlipBack = 0x2D88E
Card_Blackout = 0x2DB05
Card_BottomLeft = 0x2D78C
Card_BottomRight = 0x2D78F
Card_CheckForMatch = 0x2DB4E
Card_CursorXs = 0x2D6DF
Card_CursorYs = 0x2D6F1
Card_DoFlip = 0x2D7EC
Card_DoSelection = 0x2DBAC
Card_DrawCursor = 0x2DC4C
Card_DrawSelectSprite = 0x2DC2E
Card_FaceAttribute = 0x2D76F
Card_FacePatterns = 0x2D757
Card_FlipBackCard = 0x2D8AE
Card_HandleMatch = 0x2D81A
Card_HandleMismatch = 0x2D84E
Card_InitDeck = 0x2D662
Card_InitGame = 0x2D656
Card_InitVars = 0x2D69F
Card_InitialSet = 0x2D721
Card_MFSToInventory = 0x2DD3C
Card_MatchPairReward = 0x2DD3F
Card_Patterns = 0x12113
Card_RenderBack = 0x2D9FF
Card_RenderDeck = 0x2D681
Card_RenderFace = 0x2D931
Card_SelectAttribute = 0x2D6D5
Card_SelectCard = 0x2D7BA
Card_SelectPattern = 0x2D6CB
Card_Shuffle = 0x2D8EE
Card_TwoCardResponse = 0x2D813
Card_VRAMAttrLo = 0x2D775
Card_VRAMHi = 0x2D733
Card_VRAMLo = 0x2D745
Card_WaitThenExit = 0x2D886
Card_XOffset = 0x12110
CarryClearIfAliveAndNoHalt = 0x44C8
ChainChomp_BreakFree = 0x9A07
ChainChomp_ChaseXVel = 0x9A51
ChainChomp_ChaseYVel = 0x9A4D
ChainChomp_ChooseLunge = 0x9A87
ChainChomp_DoMove = 0x9AD9
ChainChomp_Draw = 0x9C69
ChainChomp_Drop = 0x9BA5
ChainChomp_LinkVisibleTest = 0x9E42
ChainChomp_LungeMask = 0x9BA1
ChainChomp_MoveChain = 0x9CCB
ChainChomp_XDelta = 0x9A59
ChainChomp_XVel = 0x9BA3
ChainChomp_YDelta = 0x9A55
CheepCheepCharge_Accel = 0x5A1C
CheepCheepCharge_ProxLimit = 0x5A18
CheepCheepCharge_XCoarseLimit = 0x5A1E
CheepCheepCharge_XVelLimit = 0x5A1A
CheepCheepHopper_InitXVel = 0x744C
CheepCheep_RandomXVels = 0xBD09
CheepCheep_RandomXs = 0xBD05
CheepP2P_XVel = 0x4776
CheepP2P_YVel = 0x477C
Chng4Way_45Degrees = 0x3BE66
Chng4Way_90Degrees = 0x3BE49
ChngTile_32x32 = 0x3BEF0
Chomp_JumpYVels = 0x78FB
Chomp_XVels = 0x78FD
CineKing_DoWandReturn = 0x36010
Cinematic_ToadAndKing = 0x30010
ClearPattern_ByTileset = 0x3C3A2
Clear_Nametable = 0x3FDD7
Clear_Nametable_Short = 0x3FE12
Clear_PPU_CTL2_Copy = 0x3FDCF
Clear_RAM_thru_ZeroPage = 0x3D6DE
CoinPUp_Attributes = 0xEDBC
CoinPUp_Patterns = 0xEDB8
CoinPUp_UpdateAndDraw = 0xEDD0
CoinPUps_DrawAndUpdate = 0xEDC0
CoinShipO = 0xDA14
CoinShip_BossO = 0xDA1F
CoinShip_CoinGlow = 0x13BA3
CoinShip_Layouts = 0x19337
CoinShip_Objects = 0x19347
ColorRotation_Do = 0xE19F
ConveyorEnable = 0x3D4
ConveyorSlide = 0x11C5A
Conveyor_CarryX = 0x4E2
Conveyor_CarryXHi = 0x4E4
CurrentDownward_YAccel = 0x7136
CurrentUpward_YAccel = 0x712D
Current_GenerateBubble = 0x71A8
DMC01 = 0x3E010
DMC02 = 0x3E090
DMC02_Bad = 0x3E490
DMC03 = 0x15B90
DMC04 = 0x3E990
DMC05 = 0x3F290
DMC05_B = 0x3F390
DMC05_C = 0x3F3D0
DMC06 = 0x3EA90
DMC07 = 0x15810
DMC08 = 0x15C10
DMC09 = 0x3EE10
DMC_MODADDR_LUT = 0x3E31A
DMC_MODCTL_LUT = 0x3E33A
DMC_MODLEN_LUT = 0x3E32A
Debug_DownUp = 0x30810
DeleteIfOffAndDrawWide = 0x4B11
DiagBox2_R1 = 0x3620B
DiagBox2_R2 = 0x36221
DiagBox2_R3 = 0x36237
DiagBox2_RowOffs = 0x3624D
DiagBox_R1 = 0x3006D
DiagBox_R2 = 0x30083
DiagBox_R3 = 0x30099
DiagBox_RowOffs = 0x300AF
DiagText_BySuit_H = 0x364AB
DiagText_BySuit_L = 0x364A4
Die_OddEven = 0x2CC0B
Die_OddEvenMatch = 0x2CC0D
DoTimeBonus = 0x422
Do_2PVsChallenge = 0x3D35C
Do_Ending2_IntCmd = 0x31D48
Do_Title_Screen = 0x308BF
DonutLift_ChangeBlock = 0x66A2
DoorAppear_X = 0x3F09
DoorAppear_YUpr = 0x3F07
DrawLemmyBall_AndHit = 0x3716
Draw_KTPrizeGameBox = 0x2C7ED
Draw_KoopalingBody = 0x3466
Draw_KoopalingWand = 0x33E1
Draw_SpreadAndColorful = 0xEA2B
DryBones_AnimCntrMask = 0x566E
DryBones_BumpOffOthers = 0x5791
DryBones_CrumblePats = 0x56E1
DryBones_Draw = 0x56EC
DryBones_HeadYOff = 0x56E3
DryBones_ReassembleFrames = 0x5671
DryBones_ReassembleTimers = 0x5679
DryBones_XVelConveyor = 0x566B
DynJump = 0x3FEA9
Empty_ObjLayout = 0xC016
EndLevelCard_3CMMsg2XUP = 0x5C62
EndLevelCard_AnimateSpinAway = 0x5C28
EndLevelCard_Appear = 0x5C45
EndLevelCard_AsTwinkleFrames = 0x5B2A
EndLevelCard_AttributeByFrame = 0x5B1B
EndLevelCard_CardBackLower = 0x5B40
EndLevelCard_CardBackOffset = 0x5B32
EndLevelCard_CardBackUpper = 0x5B35
EndLevelCard_CardFace = 0x5E7A
EndLevelCard_ClearPlayerIfOff = 0x5ECA
EndLevelCard_DoXUPs = 0x5D1B
EndLevelCard_Draw = 0x5FB0
EndLevelCard_DrawCard = 0x5E22
EndLevelCard_DrawCardBack = 0x5E5C
EndLevelCard_DrawCardT2 = 0x5C74
EndLevelCard_DrawFlashing = 0x5E92
EndLevelCard_Exit = 0x5D59
EndLevelCard_Frame = 0x5EE9
EndLevelCard_Give1UpsAndCycle = 0x5D82
EndLevelCard_InitForFanfare = 0x5B7E
EndLevelCard_MsgAndCard = 0x5CBC
EndLevelCard_PalData = 0x3C320
EndLevelCard_PlayerRunOff = 0x5CA1
EndLevelCard_SetFrameToCard = 0x5EB9
EndLevelCard_SetupXUP = 0x5CF1
EndLevelCard_Untouched = 0x5B60
EndLevelCard_Var1 = 0x5EE6
EndLevelCard_WaitFlyAway = 0x5C8A
EndLevelCard_xUpByMatch = 0x5CB8
EndPicByWorld_H = 0x32126
EndPicByWorld_L = 0x3212E
EndPic_VRAMStart_H = 0x325DA
EndPic_VRAMStart_L = 0x325E2
EndPic_World1 = 0x32136
EndPic_World2 = 0x321CA
EndPic_World3 = 0x32269
EndPic_World4 = 0x322F4
EndPic_World5 = 0x32390
EndPic_World6 = 0x32421
EndPic_World7 = 0x324B8
EndPic_World8 = 0x32558
EndSeq_CurtainFadeIn1 = 0x334D9
EndSeq_CurtainFadeIn2 = 0x334ED
EndSeq_CurtainFadeIn3 = 0x33501
EndSeq_Fade0 = 0x334A1
EndSeq_Fade1 = 0x3347D
EndSeq_Fade2 = 0x33459
EndSeq_Fade3 = 0x33435
EndSeq_Fade4 = 0x33411
EndSeq_PrincessChamber = 0x33198
EndSeq_World1Pal = 0x33078
EndSeq_World2Pal = 0x3309C
EndSeq_World3Pal = 0x330C0
EndSeq_World4Pal = 0x330E4
EndSeq_World5Pal = 0x33108
EndSeq_World6Pal = 0x3312C
EndSeq_World7Pal = 0x33150
EndSeq_World8Pal = 0x33174
EndSeq_WorldFadeIn1 = 0x32D02
EndSeq_WorldFadeIn2 = 0x32D26
EndSeq_WorldFadeIn3 = 0x32D4A
EndSeq_WorldFadeIn4 = 0x32D6E
EndText = 0x31AD9
EndText_DoPrincessText = 0x31B34
EndText_DrawDiagBox = 0x31A81
EndText_Wait = 0x31B81
EndWorldLetter_GenerateText = 0x36B77
Ending2_AddSprites = 0x31E89
Ending2_ClearScreen = 0x31D99
Ending2_CommitPicture = 0x31E49
Ending2_CurtainRaise = 0x31D70
Ending2_DoEndPic = 0x31D53
Ending2_EndPicPatTable2 = 0x31F6E
Ending2_EndPicPatTable3 = 0x31F76
Ending2_EndPicPatTable4 = 0x31F7E
Ending2_EndPicPatTable5 = 0x31F86
Ending2_EndPicSpriteListH = 0x31F8E
Ending2_EndPicSpriteListL = 0x31F96
Ending2_EndPicSpriteListLen = 0x31F9E
Ending2_EndPicSprites1 = 0x31FA6
Ending2_EndPicSprites2 = 0x31FCE
Ending2_EndPicSprites3 = 0x32006
Ending2_EndPicSprites4 = 0x3202A
Ending2_EndPicSprites5 = 0x3205A
Ending2_EndPicSprites6 = 0x3208E
Ending2_EndPicSprites7 = 0x320BA
Ending2_EndPicSprites8 = 0x320DE
Ending2_FadeIn = 0x31EE1
Ending2_FadeOut = 0x31F2B
Ending2_FadeOutQs = 0x31F24
Ending2_PicVRAM_NextLineWrap = 0x31F48
Ending2_PrepClear = 0x31D81
Ending2_PrepEndPic = 0x31DF0
Ending2_SeriesQueue = 0x31E7C
Ending2_SetFlag1 = 0x31D94
Ending2_SetFlag2 = 0x31E34
Ending2_THEEND = 0x31D28
Ending2_Wait = 0x31F0C
Ending_ChamberScene = 0x318BF
Ending_Credits = 0x31B8C
Ending_CurtainExtension = 0x32B40
Ending_DoChamberScene = 0x318FD
Ending_FadeIn = 0x31943
Ending_FadeOut = 0x31A1E
Ending_Init = 0x31910
Ending_LightsOn = 0x3198A
Ending_LightsOnPalSet = 0x31976
Ending_MarioAppears = 0x31956
Ending_PrincessSpeech = 0x31A16
Ending_WalkToCenter = 0x319DA
EnemyBro_DrawAndHandleBump = 0x85C2
EnemyEnterFlip = 0x4CCD
EnemyEnterXVel = 0x4CCF
EnemyKill_XVels = 0x1AFC
Enemy_CollideWithWorld = 0x4AB7
Enemy_DeleteIfOffAndDrawTail = 0x7A1B
Enemy_Kill = 0x1BA0
Entropy_BySlot = 0x4805
Event_Countdown_Init = 0x10399
ExplodeStar_VelMask = 0xF1B1
Explode_Stars = 0x6EFF
FX_MonoFlash_By_MapTick = 0x149E6
FX_World_8_Darkness = 0x17B7D
FacePlayer_FlipBitsStart = 0x4937
Fill_Tile_AttrTable_ByTileset = 0x3D52B
FireBro_FacePlayerFlip = 0x889C
FireBro_FireballXVel = 0x89D1
FireBro_SpitFire = 0x89D3
FireBro_WaitState = 0x8989
FireBro_WalkLeft = 0x8976
FireBro_WalkRight = 0x8998
FireCannonBall = 0xFE6E
FireChompTail_Attributes = 0x7ADF
FireChompTail_Patterns = 0x7ADB
FireChomp_FlipTowardsPlayer = 0x7A29
FireChomp_MoveAndExplodeDeath = 0x7A2F
FireChomp_SpitFire = 0x7C7E
FireChomp_XVelAccel = 0x7A25
FireChomp_XVelLimit = 0x7A21
FireChomp_YVelAccel = 0x7A2D
FireChomp_YVelLimit = 0x7A2B
FireJetLR_Patterns = 0xA579
FireJetLR_SpriteVisibleTest = 0xA96C
FireJet_Frame = 0xA459
FireJet_PlayerHitTest = 0xA624
FireJet_TimerReload = 0xA453
FireSnake_FlipForTick = 0x7D73
FireSnake_FrameForTick = 0x7D77
FireSnake_JumpYVel = 0x7D7B
FireSnake_RandomTimer3Vals = 0x7D7D
FireSnake_XVelTowardsPlayer = 0x7D71
Fireball_Attributes = 0xFA04
Fireball_DetectWorld = 0xE4B3
Fireball_Patterns = 0xFA00
Fireball_PoofPattern = 0xE711
Fireball_ThawTile = 0xE5BD
Fish_FixedYIfAppro = 0x157A
Fish_FixedY_ExceptHitFloor = 0x1574
Fish_Repawn = 0x582F
Flip_BotBarMid = 0x341D1
Flip_BottomBarCards = 0x3424B
Flip_BottomBarInv = 0x341B3
Flip_END = 0x34264
Flip_EraseBotBarMid = 0x341E5
Flip_EraseTopBarMid = 0x341E0
Flip_MidBItems = 0x3418F
Flip_MidBStatCards = 0x34227
Flip_MidTItems = 0x3416B
Flip_MidTStatCards = 0x34203
Flip_TopBarCards = 0x341EA
Flip_TopBarInv = 0x3415C
Flip_TopBarMid = 0x341C2
Flip_Video_Data_Closing = 0x341C2
Flip_Video_Data_Opening = 0x3415C
Flip_Video_Ends = 0x34278
Flip_Video_Offsets = 0x34268
FloatLevel_StatCheck = 0x10678
Float_WaterBottomLimitY = 0x13D2E
Float_YAccel = 0x4B49
Float_YVelAdj = 0x4B46
FloatingBGCloud_Patterns = 0x697C
FloatingCloud_Var5 = 0xBC52
FloatingCloud_XVel = 0xBC56
FortressFXBase_ByWorld = 0x148A8
FortressFX_MapCompIdx = 0x147EF
FortressFX_MapLocation = 0x14866
FortressFX_MapLocationRow = 0x14855
FortressFX_MapTileReplace = 0x14877
FortressFX_Patterns = 0x14811
FortressFX_VAddrH = 0x147CD
FortressFX_VAddrL = 0x147DE
FortressFX_W1 = 0x14888
FortressFX_W2 = 0x1488C
FortressFX_W3 = 0x14890
FortressFX_W4 = 0x14894
FortressFX_W5 = 0x14898
FortressFX_W6 = 0x1489C
FortressFX_W7 = 0x148A0
FortressFX_W8 = 0x148A4
FourWay_CPXOff = 0xFDC2
FourWay_CPYOff = 0xFDD8
FourWay_CannonballXVel = 0xFD9A
FourWay_CannonballYVel = 0xFDAE
Frog_BaseFrame = 0x10A36
Frog_FrameOffset = 0x10A39
Frog_SwimSoundMask = 0x10A34
Frog_Velocity = 0x10A41
GBCtl_BrickBump = 0x4395
GBCtl_BrickBust = 0x43DE
GBCtl_CoinBlock = 0x4398
GBCtl_LeafBlock = 0x43AB
GameOver_Complete = 0x146E8
GameOver_DeadPlayerSprite = 0x14765
GameOver_DoMenu = 0x14725
GameOver_Loop = 0x14702
GameOver_MLName = 0x14319
GameOver_PatchPlayerName = 0x14325
GameOver_PlayerQuitCleanup = 0x3D48C
GameOver_ReturnToStartX = 0x1674D
GameOver_Timeout = 0x1478E
GameOver_TwirlFromAfar = 0x166CF
GameOver_TwirlToStart = 0x165ED
GameOver_WaitOnBGM = 0x1471A
GameOver_WhiteMapObjs = 0x3D489
GamePlay_BGM = 0x3D7BC
GamePlay_TimeStart = 0x3D7B8
GamePlay_VStart = 0x3D7B0
GamePlay_YHiStart = 0x3D7A0
GamePlay_YStart = 0x3D7A8
Gameplay_UpdateAndDrawMisc = 0xE239
GiantBlockCtl_Attrs = 0x4450
GiantBlockCtl_BlkBump = 0x4280
GiantBlockCtl_BlockStarts = 0x4284
GiantBlockCtl_DebrisXOff = 0x4325
GiantBlockCtl_DebrisXVel = 0x432D
GiantBlockCtl_DebrisYOff = 0x4329
GiantBlockCtl_Draw = 0x4498
GiantBlockCtl_Frames = 0x4287
GiantBlockCtl_Pats = 0x4440
GiantEnemy_Draw = 0x95C2
GiantPiranha_Chomp = 0x97FB
GiantPiranha_Draw = 0x9820
GiantPiranha_Emerge = 0x97C3
GiantPiranha_HideInPipe = 0x980E
GiantPiranha_Retract = 0x97DE
GiantPiranha_TimerReloads = 0x9787
Giant_HVisBit = 0x8B4C
Giant_HXOff = 0x8B4E
GndMov_Big = 0x109BE
GndMov_FireHammer = 0x109C9
GndMov_Frog = 0x109F1
GndMov_Leaf = 0x109E3
GndMov_Small = 0x10993
GndMov_Tanooki = 0x10AE5
Goomba_InitFlipBits = 0xFD23
GraphicsBuf_Prep_And_WaitVSyn2 = 0x30816
GraphicsBuf_Prep_And_WaitVSync = 0x3D6F5
GroundTroop_BumpOffOthers = 0x9422
GroundTroop_Draw = 0x94B8
GroundTroop_DrawMirrored = 0x993E
GroundTroop_DrawNormal = 0x9944
GroundTroop_DrawOffsetInY = 0x9946
GroundTroop_FlipTowardsPlayer = 0x922D
GroundTroop_XVel = 0x9368
HT_DoRaiseAndFall = 0x164E5
HT_Flash = 0x16495
HT_GrabPlayer = 0x164A7
HT_Init = 0x1647A
HT_RaiseFall = 0x164E3
HammerBro_FacePlayerFlip = 0x8534
HammerBro_JumpYVel = 0x852B
HammerBro_ThrowHammer = 0x85F6
HammerBro_TimerReload = 0x852E
Hammer_Attributes = 0xF867
Hammer_Draw = 0xF87B
Hammer_Patterns = 0xF86F
Hammer_XOff = 0xF6E2
Hammer_XVel = 0x85F4
Hammer_YOff = 0xF6E8
HandleLevelJunction = 0x34943
HeavyBro_Draw = 0x8B50
HeavyBro_FacePlayerFlip = 0x8A16
HeavyBro_WaitState = 0x8B17
HeavyBro_WalkLeft = 0x8B04
HeavyBro_WalkRight = 0x8B29
HostHammerBro_Draw = 0x2C715
HostHammerBro_DrawSprites = 0x2C7AD
HostHammerBro_SpriteXs = 0x2C7A2
HostHammerBro_SpriteYs = 0x2C797
HostToad_Draw = 0x2C704
HostToad_DrawSprites = 0x2C72F
HostToad_SpriteXs = 0x2C724
HostToad_SpriteYs = 0x2C719
HostTroopa_Draw = 0x2C711
HostTroopa_DrawSprites = 0x2C76D
HostTroopa_DrawWithGameBits = 0x2C708
HostTroopa_SpriteXs = 0x2C763
HostTroopa_SpriteYs = 0x2C759
InitPal_Per_MapPowerup = 0x37822
InitPals_Per_MapPUp = 0x3782B
IntIRQ = 0x3F7A5
IntIRQ_32PixPart_HideSprites = 0x3F9F3
IntIRQ_32PixelPartition = 0x3F9C3
IntIRQ_32PixelPartition_Part2 = 0x3DFB0
IntIRQ_32PixelPartition_Part3 = 0x3FA1F
IntIRQ_32PixelPartition_Part5 = 0x3DF8E
IntIRQ_A0FIXME = 0x3FB80
IntIRQ_Finish = 0x3F8D7
IntIRQ_Finish_NoDis = 0x3F8DA
IntIRQ_SpadeGame = 0x3FAE1
IntIRQ_Standard = 0x3F8EB
IntIRQ_TitleEnding = 0x30836
IntIRQ_Vertical = 0x3F7F7
IntNMI = 0x3F496
IntNMI_Raster_Table = 0x3F493
IntNMI_UpdSel_Table = 0x3F490
IntReset = 0x3FF50
IntReset_Part2 = 0x3C42D
InvCard_Tile_Layout = 0x342C4
InvFlipFrame_DoNothing = 0x343A8
InvFlipFrame_DrawMLLivesScore = 0x343DD
InvFlipFrame_DrawWorldCoins = 0x343C1
InvFlipFrame_UpdateStatusBar = 0x343A9
InvFlip_TileLayout_Sel = 0x34288
InvFlip_VAddrLo_LUT = 0x340DB
InvGBuf_By_Open = 0x34264
InvItem_AddSub = 0x3440C
InvItem_HiliteMinMax = 0x34416
InvItem_HiliteOORX = 0x34414
InvItem_Hilite_Layout = 0x3485C
InvItem_IndexOOR = 0x3440E
InvItem_NextPrior = 0x34412
InvItem_Pal = 0x34524
InvItem_PerPlayerOff = 0x3441C
InvItem_PerPowerUp_Disp = 0x34575
InvItem_PerPowerUp_L1Sound = 0x3456C
InvItem_PerPowerUp_Palette = 0x3457F
InvItem_PerPowerUp_Palette2 = 0x345A3
InvItem_RightLeft = 0x34418
InvItem_RightLeftMinMax = 0x3441A
InvItem_SetColor = 0x34532
InvItem_Tile_Layout = 0x3428C
InvItem_Wrap = 0x34410
Inv_Display_Hilite = 0x34878
Inv_OpenPal_Changes = 0x34120
Inv_UseItem = 0x3454A
Inv_UseItem_Anchor = 0x34692
Inv_UseItem_Denial = 0x34697
Inv_UseItem_Hammer = 0x346CB
Inv_UseItem_MusicBox = 0x346B1
Inv_UseItem_Powerup = 0x345C6
Inv_UseItem_ShiftOver = 0x3462B
Inv_UseItem_Starman = 0x34681
Inv_UseItem_WarpWhistle = 0x34770
InventoryRow_BaseIndex = 0x1DDC
Inventory_Close = 0x340DF
Inventory_DestX = 0x1DE0
Inventory_DoFlipPalChange = 0x34128
Inventory_DoFlipVideoUpd = 0x342D4
Inventory_DoHilites = 0x340A2
Inventory_DoPowerupUse = 0x3441E
Inventory_DrawItemsOrCards = 0x34338
Inventory_ForceFlip = 0x34436
Inventory_ForceUpdate_AndFlip = 0x3448C
Inventory_OffsetByPlayer = 0x162BD
InvisiLift_Draw = 0x4A14
KDefeat_CloudWidths = 0x35B9
KDefeat_CloudXs = 0x3536
KDefeat_CloudYs = 0x353E
KDefeat_DrawClouds = 0x35BE
KDefeat_MoveClouds = 0x356B
KDefeat_PositionClouds = 0x3546
KDefeat_VanishMask = 0x34DE
KINGO = 0xD2BF
KS_BonkMario_HitFloor = 0x31578
KS_Carry_Xoff = 0x31655
KS_HitMario = 0x31610
KS_Kick_LuigiCatch = 0x3159C
KS_LuigiKick_MarioCatch = 0x315BE
KS_MarioCarry_KickWrapAround = 0x315E9
KS_ShellCarry = 0x31665
KS_SpinAnimAndDraw = 0x31638
KingAndToad_Sprites = 0x365AF
KingAttr_ByWorld_Bottom = 0x3652D
KingAttr_ByWorld_Top = 0x36526
KingHelpMsg1 = 0x3012A
KingHelpMsg2 = 0x301A2
KingPatIndexOff_ByWorld = 0x36534
KingPatternTable_ByWorld = 0x3651F
KingText_Frog = 0x3633C
KingText_Hammer = 0x3642C
KingText_Tanooki = 0x363B4
KingText_Typical = 0x362C4
KingToad_PatOffset = 0x303AA
KingToad_Patterns = 0x30398
KingToad_Sprites = 0x30380
KingWand_ApplyVel = 0x3671B
KingWand_ApplyVelAndFall = 0x36706
King_Animate = 0x304B2
King_DoDialog = 0x3005D
King_NumPatsByWorld = 0x302A1
King_PalByWorld = 0x3029A
King_PatTableByWorld = 0x30293
King_Patterns = 0x3653B
King_SprDataOffByWorldAndFrame = 0x302A8
King_SprDataX = 0x302E7
King_SprDataYOff = 0x302B6
King_SprPatOffByWorldAndFrame = 0x30318
King_SprPats = 0x30326
King_W1347 = 0x30531
King_W1Pat0 = 0x3032C
King_W1Pat1 = 0x30326
King_W1X = 0x302E7
King_W1YOff = 0x302B6
King_W2 = 0x304FF
King_W2Pat0 = 0x30356
King_W2Pat1 = 0x30358
King_W2X = 0x30309
King_W2YOff = 0x302D8
King_W2_YLimit = 0x304FD
King_W3Pat0 = 0x3035F
King_W3Pat1 = 0x3035A
King_W3X0 = 0x30312
King_W3X1 = 0x3030C
King_W3YOff0 = 0x302E1
King_W3YOff1 = 0x302DB
King_W4Pat0 = 0x30350
King_W4Pat1 = 0x3034A
King_W4X = 0x30302
King_W4YOff = 0x302D1
King_W5 = 0x30541
King_W5Pat0 = 0x3033E
King_W5Pat1 = 0x30344
King_W5X = 0x302FB
King_W5YOff = 0x302CA
King_W6 = 0x304E6
King_W6Crown_Attr = 0x30374
King_W6Crown_Pattern = 0x3036C
King_W6Crown_YOff = 0x30364
King_W6Pat0 = 0x3033A
King_W6Pat1 = 0x3033C
King_W6X = 0x302F8
King_W6YOff = 0x302C7
King_W7Pat0 = 0x30336
King_W7Pat1 = 0x30332
King_W7X1 = 0x302EE
King_W7X2 = 0x302F3
King_W7YOff0 = 0x302C2
King_W7YOff1 = 0x302BD
King_YDelta = 0x304FA
KingsRoomLayout_ByWorld = 0x19260
KingsRoomObjLayout = 0x19270
KoopalingAnimSpd = 0x3702
KoopalingImpact_Count = 0x3606
KoopalingPatSet4 = 0x2E82
KoopalingPatSet5 = 0x2E89
KoopalingPats_Iggy = 0x3281
KoopalingPats_Larry = 0x324B
KoopalingPats_Lemmy = 0x32A5
KoopalingPats_Ludwig = 0x32B7
KoopalingPats_Morton = 0x325D
KoopalingPats_Roy = 0x3293
KoopalingPats_Wendy = 0x326F
Koopaling_AirshipVanish = 0x34E2
Koopaling_DetectWorld = 0x3609
Koopaling_DieByFire = 0x3030
Koopaling_DoPlayerFall = 0x3510
Koopaling_DrawAndAnimate = 0x3310
Koopaling_FacePlayerBit = 0x367F
Koopaling_FinalHit = 0x2EF9
Koopaling_FrameLoopEnd = 0x31DA
Koopaling_FrameLoopStart = 0x31DC
Koopaling_JumpChanceMask = 0x2EAA
Koopaling_JumpYVels = 0x2EBF
Koopaling_JumpYVelsBase = 0x2E92
Koopaling_LRByFrame = 0x31DE
Koopaling_Normal = 0x2F3B
Koopaling_OffXHi = 0x32F1
Koopaling_OffXLo = 0x32D9
Koopaling_OffYHi = 0x32FD
Koopaling_OffYLo = 0x32CD
Koopaling_OffYOff = 0x3309
Koopaling_Palettes = 0x2E14
Koopaling_PatLookup = 0x31F0
Koopaling_TimeBonus = 0x34D2
Koopaling_WandFall = 0x34D1
Koopaling_WandFrame = 0x32C9
Koopaling_WandGrab = 0x34C9
Koopaling_XVelMidAir = 0x2E90
LATP_10Coin = 0x118AD
LATP_1up = 0x118C1
LATP_Brick = 0x11850
LATP_BrickAltClear = 0x1191D
LATP_Coin = 0x11820
LATP_CoinCommon = 0x11927
LATP_CoinStar = 0x11840
LATP_Flower = 0x117FC
LATP_GNote = 0x11683
LATP_GetCoinAboveBlock = 0x1193A
LATP_HNote = 0x11684
LATP_HandleSpecialBounceTiles = 0x1179E
LATP_InvisCoin = 0x1169E
LATP_InvisNote = 0x116A1
LATP_JumpTable = 0x117E1
LATP_Leaf = 0x1180A
LATP_None = 0x117F9
LATP_Notes = 0x11685
LATP_PSwitch = 0x118CC
LATP_PWrksJct = 0x116A2
LATP_PowerUps = 0x11683
LATP_QBlocks = 0x1168D
LATP_Star = 0x11818
LATP_Vine = 0x118A0
LATP_Woodblocks = 0x11689
LATR_BlockResult = 0x116A3
LATR_GNote = 0x116A3
LATR_HNote = 0x116A4
LATR_InvisCoin = 0x116BE
LATR_InvisNote = 0x116C1
LATR_Notes = 0x116A5
LATR_PWrksJct = 0x116C2
LATR_QBlocks = 0x116AD
LATR_Woodblocks = 0x116A9
LBGHM_BitMasks = 0x1DF71
LL17_ReturnTileAndNextRow = 0x227C5
LL19_GetLayoutByte_AndBackup = 0x26A38
LL19_ReturnTileAndNextRow = 0x26A54
LL20_GetLayoutByte_AndBackup = 0x28EF6
LL20_GetLayoutByte_AndBackup2 = 0x28F12
LL20_MiscTiles = 0x289B4
LL20_ReturnTileAndNextRow = 0x28F1B
LL21_Background = 0x2A744
LL21_InitLongRun = 0x2A6A7
LL21_LongRunNextRow = 0x2A6CD
LL23_GetLayoutByte_AndBackup = 0x2EBD0
LL23_ReturnTileAndNextRow = 0x2EBEC
LL_225SlopesB2T_Lower = 0x1D682
LL_225SlopesB2T_LowerCeiling = 0x1D7A1
LL_225SlopesB2T_Upper = 0x1D67F
LL_225SlopesB2T_UpperCeiling = 0x1D79E
LL_225SlopesT2B_Lower = 0x1D613
LL_225SlopesT2B_LowerCeiling = 0x1D713
LL_225SlopesT2B_Upper = 0x1D610
LL_225SlopesT2B_UpperCeiling = 0x1D716
LL_45SlopesB2T = 0x1D46A
LL_45SlopesB2TCeiling = 0x1D470
LL_45SlopesT2B = 0x1D467
LL_45SlopesT2BCeiling = 0x1D46D
LL_4WayCannon45 = 0x2E8B6
LL_4WayCannon90 = 0x2E8B0
LL_AirshipMisc = 0x2E89A
LL_ArrowLifts = 0x24A0A
LL_BGMid = 0x1D95B
LL_BGPipeHEnd = 0x24721
LL_BGPipeHMid = 0x2471F
LL_BGTop = 0x1D958
LL_BigBlockSky_Left = 0x1D188
LL_BigBlockSky_Middle = 0x1D194
LL_BigBlockSky_Right = 0x1D1A0
LL_BigBlockSky_Shadow = 0x1D1AC
LL_BigBlock_ApprShadow = 0x1D084
LL_BigBlock_Bot = 0x1D05F
LL_BigBlock_Left = 0x2260A
LL_BigBlock_Mid = 0x1D053
LL_BigBlock_Middle = 0x2260E
LL_BigBlock_Right = 0x22612
LL_BigBlock_ShadowChks = 0x1D06B
LL_BigBlock_Top = 0x1D047
LL_BigBlock_Top_ColorOff = 0x1D0A1
LL_BigBlocks = 0x28548
LL_BigCannon = 0x2E962
LL_BigSandBrick = 0x286F2
LL_BigWoodBlocks = 0x22557
LL_BiggerBlock = 0x28591
LL_BiggerSandBrick = 0x28749
LL_BiggestBlock = 0x285E8
LL_Bridge = 0x1CEFA
LL_Bush1 = 0x1CB61
LL_Bush2 = 0x1CB76
LL_Bush3 = 0x1CB83
LL_Bush_Width = 0x1CBB4
LL_CannonPlatform = 0x28E40
LL_Chest = 0x24B13
LL_Cloud = 0x1D3A9
LL_Cloud17 = 0x226FC
LL_Cloud3 = 0x1DAA6
LL_CloudGoal = 0x268B4
LL_Conveyor = 0x1D007
LL_Corners = 0x1D44C
LL_Crate = 0x2E805
LL_DarkSky = 0x2677A
LL_DarkSkyStar = 0x2677C
LL_DecoCeiling = 0x1D8D8
LL_DecoCeilingMid = 0x1D8D5
LL_DecoGroundMid = 0x1D856
LL_DecoGroundTop = 0x1D850
LL_DesertChains = 0x289DD
LL_DesertTree = 0x28D64
LL_DiamondBrick = 0x2A684
LL_DoubleCloud = 0x2682A
LL_EndDoor = 0x2A7D6
LL_FollowBlock = 0x1CF5F
LL_GhostBlock = 0x2A654
LL_GiantBlocks = 0x26526
LL_GiantPipe = 0x2658E
LL_Goal_TSetAddr = 0x1CA77
LL_Goal_TSetList = 0x1CA85
LL_GrayLeft = 0x2489D
LL_GrayMiddle = 0x2489F
LL_GrayRight = 0x248A1
LL_HPipeTB = 0x2666E
LL_HorzPipe = 0x1C77F
LL_IceBlocks = 0x2269F
LL_IntroAirship = 0x2EAA5
LL_LBGC2_Rows = 0x269DC
LL_LBGC2_StartIdx = 0x269DA
LL_LargeBGCloudChecks = 0x266EA
LL_LargeBGClouds = 0x2669D
LL_LargeBGClouds2 = 0x269B0
LL_LargeBGClouds2B = 0x269C8
LL_LargeIceBlock = 0x226C3
LL_Lava = 0x1CD39
LL_LeftEdge = 0x1D2B3
LL_LongCloud = 0x2265F
LL_LongCloudWB = 0x267DC
LL_MetalPlate = 0x2E854
LL_MiniPipeV = 0x2663E
LL_MiscBG = 0x1DB24
LL_MuncherAlternators = 0x2662C
LL_OrangeLeft = 0x248FB
LL_OrangeMiddle = 0x248FD
LL_OrangeRight = 0x248FF
LL_PUPClouds = 0x1D233
LL_Path45B2T = 0x1CC89
LL_Path45T2B = 0x1CC69
LL_Path625B2T = 0x1CCDB
LL_Path625T2B = 0x1CCA9
LL_PathHorz = 0x1CC3D
LL_PathInit = 0x1CD23
LL_PathVert = 0x1CC4F
LL_Pillar = 0x1CD9A
LL_PillarBrickChk = 0x1CDA4
LL_PillarShadow = 0x1CDA0
LL_PipeElbows = 0x247FE
LL_PipeworkXtra = 0x28E9B
LL_PipeworksGround = 0x28EAD
LL_PitTile = 0x1D2AF
LL_PlatformPuller = 0x22529
LL_PowerBlocks = 0x1CAD4
LL_RightEdge = 0x1D2AB
LL_RockyTiles = 0x2EB51
LL_RoundCloudReplace = 0x268FE
LL_RoundCloudSearch = 0x268F9
LL_RoundCloudTops = 0x268F7
LL_RunGroundMid = 0x1D30E
LL_RunGroundTop = 0x1D308
LL_RunGround_Offset = 0x1D314
LL_SandBrick = 0x286A9
LL_SceneryHPipe_CheckCross = 0x24661
LL_SceneryHPipe_CrossB = 0x2465E
LL_SceneryHPipe_CrossChk = 0x24658
LL_SceneryHPipe_CrossT = 0x2465B
LL_SceneryHPipe_End = 0x2461B
LL_SceneryHPipe_Mid = 0x24619
LL_SceneryVPipe_CheckCrossL = 0x245EF
LL_SceneryVPipe_CheckCrossR = 0x24604
LL_SceneryVPipe_CrossChk = 0x2458B
LL_SceneryVPipe_CrossL = 0x24590
LL_SceneryVPipe_CrossR = 0x24595
LL_ShadowBrick = 0x2A55F
LL_SlopeMidGround = 0x1D464
LL_Spike = 0x2A703
LL_Spikes = 0x2483F
LL_Stars = 0x26777
LL_SupportSqr = 0x2E8C4
LL_SuspensionCables = 0x22758
LL_TankPrefab1 = 0x2EA1C
LL_TankPrefab2 = 0x2EA3F
LL_ThroneRoom = 0x1DDB5
LL_ToadHouseBlocks = 0x24AA2
LL_TopBlock = 0x1CF57
LL_Tunnel = 0x1DAFD
LL_UnderwaterScenery = 0x247C9
LL_VertGroundL = 0x1D807
LL_VertGroundR = 0x1D80A
LL_VertPipe = 0x1C652
LL_VertPipe4 = 0x1C65A
LL_VertPipeTransit = 0x1C658
LL_W8MiniShip1 = 0x2EB5E
LL_W8MiniShip2 = 0x2EB6A
LL_W8MiniShipIndex = 0x2EB7C
LL_W8MiniShipRows = 0x2EB7E
LL_WallCannonB = 0x2E8C0
LL_WallCannonF = 0x2E8BC
LL_WarGround = 0x2E7C8
LL_WindowHeight = 0x2A586
LL_WoodBody = 0x2E531
LL_WoodFloor = 0x2E76D
LL_WoodThick = 0x2E5ED
LL_WoodTipsUnderside = 0x2E909
LOSBSV_LookAhead = 0xBAF6
LOSBSV_LookAheadHi = 0xBAF8
LOSBS_LookAhead = 0xB842
LOSBS_LookAheadHi = 0xB844
LT0 = 0x16418
LT0B = 0x173DA
Lakitu_TossEnemy = 0x8E3A
Lakitu_XVelAccel = 0x8D04
Lakitu_XVelLimit = 0x8D06
Lakitu_YOff = 0x8D08
Laser_PrepSpritesAndHit = 0xF053
LavaLotusFire_XOff = 0x70C2
LavaLotusFire_XVel = 0x70C7
LavaLotusFire_YVel = 0x70CC
LavaLotus_RightEdgePatBottom = 0x701D
LavaLotus_RightEdgePatTop = 0x701A
LavaLotus_SpitFire = 0x70D1
Leaf_TimerVals = 0x2BA3
Leaf_XVelByOsc = 0x2BDD
Leaf_XVelLimit = 0x2BDF
Leaf_YVels = 0x2B9F
LemmyBall_Attrs = 0x3712
LemmyBall_FloorBounceYVel = 0x82B3
LemmyBall_Left = 0x370A
LemmyBall_Right = 0x370E
LemmyBall_RotateBits = 0x82B7
LemmyBall_XVel = 0x37AA
Lemmy_SpawnBall = 0x37AC
Lemmy_XLimits = 0x3679
Lemmy_XVelAccel = 0x367D
Lemmy_XVelLimits = 0x367B
LetterItem_ByWorld = 0x360DE
LetterRow_VRAM_H = 0x36B17
LetterRow_VRAM_L = 0x36B0E
Letter_Bodies = 0x36784
Letter_GiveIncludedItem = 0x361E9
Letter_Header = 0x36B20
Letter_HeaderBowser = 0x36B45
Letter_HeaderOffset = 0x36B68
Letter_ItemAttr_L = 0x360FF
Letter_ItemAttr_R = 0x3610C
Letter_ItemPat_L = 0x360E5
Letter_ItemPat_R = 0x360F2
Letter_PalKludges = 0x361B8
Letter_Palette = 0x3608D
Letter_PaletteKludge = 0x361BA
Letter_PrincessSprites = 0x3611C
Letter_VRAMHi = 0x36B6F
Letter_VRAMLo = 0x36B73
Letter_World1 = 0x36792
Letter_World2 = 0x3680F
Letter_World3 = 0x36876
Letter_World4 = 0x368F9
Letter_World5 = 0x36979
Letter_World6 = 0x36A04
Letter_World7 = 0x36A97
LeveLoad_FixedSizeGen_TS051113 = 0x264C5
LeveLoad_FixedSizeGen_TS1 = 0x1E4A8
LeveLoad_FixedSizeGen_TS10 = 0x2E4BE
LeveLoad_FixedSizeGen_TS14 = 0x1A4F4
LeveLoad_FixedSizeGen_TS151617 = 0x2C4A4
LeveLoad_FixedSizeGen_TS18 = 0x1C499
LeveLoad_FixedSizeGen_TS2 = 0x2A4FE
LeveLoad_FixedSizeGen_TS3 = 0x204F4
LeveLoad_FixedSizeGen_TS4_TS12 = 0x224C8
LeveLoad_FixedSizeGen_TS678 = 0x2452A
LeveLoad_FixedSizeGen_TS9 = 0x284CC
LeveLoad_FixedSizeGens = 0x3DA85
LeveLoad_Generators = 0x3DA59
LevelEvent_Cancel = 0xBBCA
LevelEvent_CheepCheep = 0xBD0D
LevelEvent_CloudsinBG = 0xBC5A
LevelEvent_Do = 0xBBAB
LevelEvent_DoNothing = 0x13B0
LevelEvent_LakituFlee = 0xBBC5
LevelEvent_Parabeetles = 0xBDC5
LevelEvent_SpikeCheep = 0xBD68
LevelEvent_TreasureBox = 0xBCB3
LevelEvent_WoodPlatforms = 0xBBD4
LevelInit_Airship = 0x10309
LevelInit_Airship_Board = 0x10315
LevelInit_DoNothing = 0x102EC
LevelInit_EnableSlopes = 0x1033D
LevelInit_PipeExitBottom = 0x102F4
LevelInit_PipeExitLeft = 0x10302
LevelInit_PipeExitRight = 0x102FB
LevelInit_PipeExitTop = 0x102ED
LevelInit_StartSliding = 0x102E0
LevelJctBQ_Layout = 0x3491B
LevelJctBQ_Objects = 0x3492B
LevelJctBQ_Tileset = 0x3493B
LevelJctGE_Layout = 0x34B27
LevelJctGE_Objects = 0x34B37
LevelJctGE_Tileset = 0x34B47
LevelJct_BigQuestionBlock = 0x349E9
LevelJct_General = 0x34A97
LevelJct_GenericExit = 0x34B4F
LevelJct_GetVScreenH = 0x3DE8C
LevelJct_GetVScreenH2 = 0x3DE9F
LevelJct_SpecialToadHouse = 0x34B92
LevelJct_VertStarts = 0x34A8F
LevelJct_YLHStarts = 0x34A87
LevelJunction_PartialInit = 0x1035B
LevelLoad = 0x3D7C7
LevelLoad_ByTileset = 0x3DA2D
LevelLoad_CopyObjectList = 0x3550F
LevelLoad_TS1 = 0x1E418
LevelLoad_TS10 = 0x2E418
LevelLoad_TS13 = 0x26429
LevelLoad_TS14 = 0x1A418
LevelLoad_TS15_TS16_TS17 = 0x2C410
LevelLoad_TS18 = 0x1C418
LevelLoad_TS2 = 0x2A418
LevelLoad_TS3 = 0x20418
LevelLoad_TS4_TS12 = 0x22418
LevelLoad_TS5 = 0x26418
LevelLoad_TS6 = 0x24418
LevelLoad_TS7 = 0x24429
LevelLoad_TS8 = 0x24485
LevelLoad_TS9 = 0x28418
Level_ActionTiles = 0x116D3
Level_ActionTiles_HitEnable = 0x116DB
Level_ActionTiles_OffFix = 0x116CB
Level_ActionTiles_Range = 0x116C3
Level_ActionTiles_TIHitBit = 0x116E3
Level_BG_Pages1 = 0x3D772
Level_BG_Pages2 = 0x3D789
Level_ChangeTile_ByTempVars = 0x1C6F
Level_CheckGndLR_TileGTAttr = 0x115CC
Level_CheckIfTileUnderwater = 0x1067A
Level_CountNotDeadObjs = 0xBE1D
Level_DoBumpBlocks = 0x116E8
Level_DoChangeReset = 0xBE3F
Level_DoCommonSpecialTiles = 0x115E1
Level_EndFlipBits = 0x3B465
Level_IceBlock_GrabNew = 0x11665
Level_InitAction_Do = 0x102C9
Level_InitAction_JumpTable = 0x102B9
Level_Initialize = 0x1024D
Level_JctCtl_Do = 0x349A8
Level_JctInit = 0x349BA
Level_MainLoop = 0x3CDE8
Level_MinTileUWByQuad = 0x210
Level_ObjCalcXDiffs = 0x1D3C
Level_ObjCalcYDiffs = 0x1D4D
Level_ObjectsSpawnByScroll = 0xB846
Level_ObjectsSpawnByScrollV = 0xBAFA
Level_Opening_Effect = 0x34E01
Level_PipeEnter = 0x3B37E
Level_PipeExitDirTable = 0x3B5E8
Level_PipeTransitOrExit = 0x3B4C7
Level_PrepareNewObject = 0x14A9
Level_QueueChangeBlock = 0x11F91
Level_RecordBlockHit = 0x3D9BA
Level_SetPlayerPUpPal = 0x10559
Level_SlopeQuad00 = 0x18
Level_SlopeQuad40 = 0x38
Level_SlopeQuad80 = 0x55
Level_SlopeQuadC0 = 0x7C
Level_SlopeSetByQuad = 0x10
Level_SpawnObj = 0xBE10
Level_SpawnObjSetMax = 0xBE12
Level_SpawnObjsAndBounce = 0xB800
Level_TilesetIdx_ByTileset = 0x10010
Level_XStarts = 0x10249
LoadLevel21_Background = 0x2A746
LoadLevel_18UNK = 0x24A28
LoadLevel_18UNK_B = 0x24B89
LoadLevel_20UNK = 0x28AED
LoadLevel_4WayCannon45 = 0x2E8CD
LoadLevel_4WayCannon90 = 0x2E8C8
LoadLevel_AirshipMisc = 0x2E8A4
LoadLevel_AltBackground = 0x2A791
LoadLevel_AltClearToCloud = 0x2675C
LoadLevel_ArrowLifts = 0x24A12
LoadLevel_BGBush = 0x1DA85
LoadLevel_BGHMinipipe = 0x2478E
LoadLevel_BGHPipeLeft = 0x24723
LoadLevel_BGOrWater = 0x1D95E
LoadLevel_BGVMinipipe = 0x246F8
LoadLevel_BGVPipe = 0x246BD
LoadLevel_Background = 0x2C527
LoadLevel_BackgroundPyramid = 0x28DD0
LoadLevel_BigBlock_Pyramid = 0x28B84
LoadLevel_BigBlocks = 0x28550
LoadLevel_BigBlocksDiagRect = 0x28850
LoadLevel_BigCannonLeft = 0x2E966
LoadLevel_BigCannonRight = 0x2E999
LoadLevel_BigSandBricks = 0x28704
LoadLevel_BigSizeBush = 0x1CBC5
LoadLevel_BigWoodBlocks = 0x22563
LoadLevel_BiggerBlock_Pyramid = 0x28C1F
LoadLevel_BiggerBlocks = 0x285A3
LoadLevel_BiggerBlocksDiagRect = 0x288AF
LoadLevel_BiggerSandBricks = 0x28769
LoadLevel_BiggestBlockDiaRect = 0x28910
LoadLevel_BiggestBlock_Pyramid = 0x28CBF
LoadLevel_BiggestBlocks = 0x28608
LoadLevel_Black14Rows = 0x2EA01
LoadLevel_BlockRun = 0x1CB2F
LoadLevel_Blocks = 0x1CB26
LoadLevel_BonusFloor = 0x2C684
LoadLevel_BonusPlayer = 0x2C591
LoadLevel_Border = 0x2C5F9
LoadLevel_BottomBoltThreadV = 0x2E69C
LoadLevel_BowserStatue = 0x2A7BF
LoadLevel_BrickAndShadow = 0x1CE6F
LoadLevel_BrightDiamond = 0x2A680
LoadLevel_BrightDiamondLong = 0x2A6E8
LoadLevel_BushBunch = 0x225E9
LoadLevel_CCBridge = 0x1CEFE
LoadLevel_Candle = 0x2A7B7
LoadLevel_Cannon = 0x1CEBA
LoadLevel_CannonPlatform = 0x28E43
LoadLevel_ChainLeft = 0x289D9
LoadLevel_ChainRight = 0x289DF
LoadLevel_CheckBGHMem = 0x1DF79
LoadLevel_CloudGoal = 0x268BA
LoadLevel_CloudRun = 0x1D3AF
LoadLevel_CloudRun17 = 0x22702
LoadLevel_CloudRun3 = 0x1DAAC
LoadLevel_CloudSpace = 0x26A12
LoadLevel_Conveyor = 0x1D00A
LoadLevel_Corner = 0x1D458
LoadLevel_Crate = 0x2E80E
LoadLevel_DecoCeiling = 0x1D8DB
LoadLevel_DecoGround = 0x1D85C
LoadLevel_DesertAltBG = 0x28E8C
LoadLevel_DesertCloud = 0x28DC1
LoadLevel_DesertPillar = 0x28991
LoadLevel_DesertTree = 0x28D67
LoadLevel_DonutLifts = 0x1D033
LoadLevel_Door1 = 0x1C624
LoadLevel_Door2 = 0x1C63B
LoadLevel_DoubleCloud = 0x26836
LoadLevel_DoubleTipBodyWood = 0x2E93C
LoadLevel_DownwardSpikes = 0x24841
LoadLevel_EndDoor = 0x2A7DA
LoadLevel_EndGoal = 0x1C872
LoadLevel_EndGoalDecoSquare = 0x1CA8C
LoadLevel_FatTrunk = 0x22783
LoadLevel_FillBackground = 0x1D9DD
LoadLevel_FloatingBigBlock = 0x1D1B8
LoadLevel_GenerateBigBlock = 0x1D0A5
LoadLevel_Generator_TS051113 = 0x26442
LoadLevel_Generator_TS1 = 0x1E431
LoadLevel_Generator_TS10 = 0x2E431
LoadLevel_Generator_TS14 = 0x1A43D
LoadLevel_Generator_TS151617 = 0x2C495
LoadLevel_Generator_TS18 = 0x1C43C
LoadLevel_Generator_TS2 = 0x2A46D
LoadLevel_Generator_TS3 = 0x2043D
LoadLevel_Generator_TS4_TS12 = 0x2243F
LoadLevel_Generator_TS678 = 0x2449B
LoadLevel_Generator_TS9 = 0x2843B
LoadLevel_GetLayoutByte = 0x227B2
LoadLevel_GhostBlock = 0x2A65A
LoadLevel_GhostBlockUNSURE = 0x2A650
LoadLevel_GiantBlock = 0x26542
LoadLevel_GiantCoral = 0x265D1
LoadLevel_GiantHill = 0x26940
LoadLevel_GiantPipe = 0x26597
LoadLevel_GrayBlock = 0x248A3
LoadLevel_GroundRun = 0x1D316
LoadLevel_HLeftWallPipeRun = 0x1C7EA
LoadLevel_HMinipipe = 0x247AB
LoadLevel_HPipeRun = 0x26670
LoadLevel_HRightWallPipeRun = 0x1C785
LoadLevel_HRightWallPipeRun3 = 0x1C849
LoadLevel_HangingGlobes = 0x2A619
LoadLevel_IceBlocks = 0x226A2
LoadLevel_IceBricks = 0x1CB5C
LoadLevel_IntroAirship = 0x2EB22
LoadLevel_InvisibleSolid = 0x2EBB4
LoadLevel_Jelectro = 0x1CFFF
LoadLevel_LargeGreenOrSnow = 0x22616
LoadLevel_LargeIceBlocks = 0x226C7
LoadLevel_LargeWater = 0x24A30
LoadLevel_Lava = 0x1CD3B
LoadLevel_Ledge = 0x2EB4C
LoadLevel_LeftBoltThreadH = 0x2E665
LoadLevel_LilCloud = 0x2273C
LoadLevel_LittleBlocks = 0x2852D
LoadLevel_LittleBlocksDiagRect = 0x287ED
LoadLevel_LittleBushRun = 0x1D293
LoadLevel_LittleBushRun17 = 0x225B5
LoadLevel_LittleCannon = 0x2E9CB
LoadLevel_LittleCloudRun = 0x1CFE6
LoadLevel_LittleCloudSolidRun = 0x1CFE0
LoadLevel_LittleVBlocks = 0x28651
LoadLevel_LongCloud = 0x22665
LoadLevel_LongCloud_WB = 0x267E8
LoadLevel_LongWoodBlock = 0x2253B
LoadLevel_MetalPlate = 0x2E85D
LoadLevel_MetalSquare = 0x2E904
LoadLevel_MetalSupportHRun = 0x2E6DF
LoadLevel_MetalSupportLVRun = 0x2E6F3
LoadLevel_MetalSupportRVRun = 0x2E730
LoadLevel_MidSizeBush = 0x1CBB7
LoadLevel_MiniPipe_V = 0x26640
LoadLevel_MiniShip = 0x2EB80
LoadLevel_MiscBG = 0x1DB27
LoadLevel_MiscDesertTiles = 0x289B8
LoadLevel_Muncher17 = 0x2279F
LoadLevel_NextColumn = 0x1DFC3
LoadLevel_Nothing = 0x1D9D9
LoadLevel_Nothing18 = 0x247C8
LoadLevel_OrangeBlock = 0x24901
LoadLevel_Path45B2T = 0x1CC8B
LoadLevel_Path45T2B = 0x1CC6B
LoadLevel_Path625B2T = 0x1CCDF
LoadLevel_Path625T2B = 0x1CCAD
LoadLevel_PathHorizontal = 0x1CC3F
LoadLevel_PathVertical = 0x1CC51
LoadLevel_Pillar = 0x1CD96
LoadLevel_Pillar_BG = 0x1CDA6
LoadLevel_PipeElbows = 0x2480E
LoadLevel_PipeworkXtra = 0x28E9D
LoadLevel_Pipeworks = 0x28A14
LoadLevel_PipeworksGround = 0x28EB0
LoadLevel_Pitfall = 0x1D2A7
LoadLevel_PitfallW = 0x1D2B7
LoadLevel_PlatformPuller = 0x2252B
LoadLevel_PointyCloud = 0x2687B
LoadLevel_Portholes = 0x2E925
LoadLevel_PowerBlock = 0x1CAED
LoadLevel_PrefabBGClouds = 0x266F2
LoadLevel_PrefabBGClouds2 = 0x269DE
LoadLevel_PrefabBlock = 0x1DD50
LoadLevel_PutLittleBGCloud = 0x1D02B
LoadLevel_QBoxBlue = 0x2C661
LoadLevel_QBoxOrange = 0x2C63A
LoadLevel_RandomPUpClouds = 0x1D236
LoadLevel_RightBoltThreadH = 0x2E680
LoadLevel_RopeRailing = 0x2E7B0
LoadLevel_RotoDisc = 0x2A7AF
LoadLevel_RoundCloudTop = 0x26903
LoadLevel_SandBrickBotCrnrs = 0x2868E
LoadLevel_SandBrickRightDiags = 0x287B2
LoadLevel_SandBricks = 0x286B1
LoadLevel_ScenHPipe_LeftWall = 0x24685
LoadLevel_ScenHPipe_RightWall = 0x2461D
LoadLevel_SceneryVPipe = 0x2459A
LoadLevel_Set_TileMemAddr = 0x3D945
LoadLevel_ShadowBGBrick = 0x2A561
LoadLevel_ShadowFlood = 0x2A71E
LoadLevel_SkyH = 0x28AD9
LoadLevel_SkyH_Unused = 0x28A00
LoadLevel_SkyV = 0x28ABD
LoadLevel_Slope225B2T = 0x1D685
LoadLevel_Slope225B2TCeiling = 0x1D7A4
LoadLevel_Slope225T2B = 0x1D616
LoadLevel_Slope225T2BCeiling = 0x1D719
LoadLevel_Slope45B2T = 0x1D4CC
LoadLevel_Slope45B2TCeiling = 0x1D5BD
LoadLevel_Slope45T2B = 0x1D473
LoadLevel_Slope45T2BCeiling = 0x1D54C
LoadLevel_SmallBlock_Pyramid = 0x28B09
LoadLevel_SmallSizeBush = 0x1CBBE
LoadLevel_SolidBrick = 0x2A686
LoadLevel_SpikeDown = 0x2A705
LoadLevel_SpikeUp = 0x2A6FF
LoadLevel_Stars = 0x2677E
LoadLevel_StoreJctStart = 0x1DFDD
LoadLevel_SuspensionCableL = 0x2275C
LoadLevel_SuspensionCableR = 0x22753
LoadLevel_Table = 0x2C4BE
LoadLevel_TankPrefab1 = 0x2EA17
LoadLevel_TankPrefab2 = 0x2EA7B
LoadLevel_TankTread = 0x2E9E4
LoadLevel_ThinGround = 0x28971
LoadLevel_ThroneRoom = 0x1DF46
LoadLevel_TileMemNextRow = 0x1D3FE
LoadLevel_ToadBlackBGH = 0x24963
LoadLevel_ToadBlackBGV = 0x24977
LoadLevel_ToadBlockAndShadow = 0x24AA4
LoadLevel_ToadBlockV = 0x24A86
LoadLevel_ToadChest = 0x24B17
LoadLevel_ToadMiniChest = 0x24B60
LoadLevel_ToadShadowV = 0x24AE1
LoadLevel_TopBoltThreadV = 0x2E6BC
LoadLevel_TopDecoBlocks = 0x1CF67
LoadLevel_Tunnel = 0x1DAFF
LoadLevel_TwoRowsWater = 0x24864
LoadLevel_UNKTALL = 0x2C5D6
LoadLevel_UnderwaterCirclesH = 0x24993
LoadLevel_UnderwaterScenery = 0x247CD
LoadLevel_UpwardSpikes = 0x2483B
LoadLevel_VCeilingPipeRun = 0x1C6B6
LoadLevel_VGroundPipe5Run = 0x1C773
LoadLevel_VGroundPipeRun = 0x1C65C
LoadLevel_VTransitPipeRun = 0x1C70C
LoadLevel_VertGroundL = 0x1D80D
LoadLevel_VertGroundR = 0x1D836
LoadLevel_VerticalAltClear = 0x24858
LoadLevel_VineToGround = 0x1D27B
LoadLevel_VsBounceBlocks = 0x1C586
LoadLevel_VsBrickTiles = 0x1C581
LoadLevel_VsCoins = 0x1C84E
LoadLevel_VsDiamondBlocks = 0x1C58E
LoadLevel_VsLadder = 0x1C5A5
LoadLevel_VsLeftPipes = 0x1C5BB
LoadLevel_VsQBlock = 0x1C5E6
LoadLevel_VsRightPipes = 0x1C5C8
LoadLevel_VsStatusBar = 0x1C602
LoadLevel_WallCannonB = 0x2E8FB
LoadLevel_WallCannonF = 0x2E8F6
LoadLevel_WarGround = 0x2E7CC
LoadLevel_WaterFill = 0x249A7
LoadLevel_WideCloud = 0x22744
LoadLevel_WindowShort = 0x2A582
LoadLevel_WindowTall = 0x2A588
LoadLevel_WindowVeryShort = 0x2A5F8
LoadLevel_WoodBody = 0x2E539
LoadLevel_WoodBodyLong = 0x2E56F
LoadLevel_WoodFloor = 0x2E771
LoadLevel_WoodFloor2 = 0x2E79C
LoadLevel_WoodThick = 0x2E5F5
LoadLevel_WoodTipsUnderside = 0x2E90D
LoadLevel_WoodVert = 0x2E582
LoadLevel_WoodVertShortRepeat = 0x2E5BB
LogPlat_Draw = 0x55BA
LostShoe_Attribute = 0xF504
LostShoe_Pattern = 0xF4FC
M12ASegData00 = 0x389B7
M12ASegData01 = 0x389D6
M12ASegData02 = 0x389F2
M12ASegData03 = 0x389A0
M12ASegData04 = 0x38A4F
M12ASegData05 = 0x38CA9
M12ASegData06 = 0x38D4B
M12ASegData07 = 0x38D12
M12ASegData08 = 0x38DD0
M12ASegData09 = 0x38F4B
M12ASegData0A = 0x38DDE
M12ASegData0B = 0x38E64
M12ASegData0C = 0x39034
M12ASegData0D = 0x38F0E
M12ASegData0E = 0x38F25
M12ASegData0F = 0x390D9
M12ASegData10 = 0x3911A
M12ASegData11 = 0x39147
M12ASegData12 = 0x38A9E
M12ASegData13 = 0x38B75
M12ASegData14 = 0x38BC1
M12ASegData15 = 0x38AF7
M12ASegData16 = 0x38A07
M12ASegData17 = 0x388BD
M12ASegData18 = 0x38B45
M12ASegData19 = 0x38C0F
M12ASegData1A = 0x388AA
M12ASegData1B = 0x3A28A
M12ASegData1C = 0x3A2B6
M12ASegData1D = 0x3A2E8
M12ASegData1E = 0x3A343
M12ASegData1F = 0x3A3C2
M12ASegData20 = 0x3A40E
M12ASegData21 = 0x3A486
M12ASegData22 = 0x3A544
M12ASegData23 = 0x3A595
M12ASegData24 = 0x388E0
M2BSegData00 = 0x399A9
M2BSegData01 = 0x399D1
M2BSegData02 = 0x39A12
M2BSegData03 = 0x39A63
M2BSegData04 = 0x39B14
M2BSegData05 = 0x39B4A
M2BSegData06 = 0x39B89
M2BSegData07 = 0x3A228
M2BSegData08 = 0x3A261
M2BSegData09 = 0x3A134
M2BSegData0A = 0x3A177
M2BSegData0B = 0x39BF7
M2BSegData0C = 0x39C17
M2BSegData0D = 0x39C6F
M2BSegData0E = 0x39564
M2BSegData0F = 0x395B9
M2BSegData10 = 0x395D3
M2BSegData11 = 0x39639
M2BSegData12 = 0x3969A
M2BSegData13 = 0x3972D
M2BSegData14 = 0x3A010
M2BSegData15 = 0x3A048
M2BSegData16 = 0x3988A
M2BSegData17 = 0x398B5
M2BSegData18 = 0x398B8
M2BSegData19 = 0x398FD
M2BSegData1A = 0x3A083
M2BSegData1B = 0x39CD5
M2BSegData1C = 0x39D4F
M2BSegData1D = 0x39DB7
M2BSegData1E = 0x397D3
M2BSegData1F = 0x39808
M2BSegHedr00 = 0x3943D
M2BSegHedr01 = 0x39444
M2BSegHedr02 = 0x3944B
M2BSegHedr03 = 0x39452
M2BSegHedr04 = 0x39459
M2BSegHedr05 = 0x39460
M2BSegHedr06 = 0x39467
M2BSegHedr07 = 0x3946E
M2BSegHedr08 = 0x39475
M2BSegHedr09 = 0x3947C
M2BSegHedr0A = 0x39483
M2BSegHedr0B = 0x3948A
M2BSegHedr0C = 0x39491
M2BSegHedr0D = 0x39498
M2BSegHedr0E = 0x3949F
M2BSegHedr0F = 0x394A6
M2BSegHedr10 = 0x394AD
M2BSegHedr11 = 0x394B4
M2BSegHedr12 = 0x394BB
M2BSegHedr13 = 0x394C2
M2BSegHedr14 = 0x394C9
M2BSegHedr15 = 0x394D0
M2BSegHedr16 = 0x394D7
M2BSegHedr17 = 0x394DE
M2BSegHedr18 = 0x394E5
M2BSegHedr19 = 0x394EC
M2BSegHedr1A = 0x394F3
M2BSegHedr1B = 0x394FA
M2BSegHedr1C = 0x39501
M2BSegHedr1D = 0x39508
M2BSegHedr1E = 0x3950F
M2BSegHedr1F = 0x39516
M2BSegHedr20 = 0x3951D
M2BSegHedr21 = 0x39524
M2BSegHedr22 = 0x3952B
M2BSegHedr23 = 0x39532
M2BSegHedr24 = 0x39539
MAPOBJ_UNK0C_X = 0x16D71
MAPOBJ_UNK0C_Y = 0x16D70
MAT_X_W1A = 0x1709E
MAT_X_W1B = 0x170A4
MAT_X_W1C = 0x170AA
MAT_X_W2A = 0x170B0
MAT_X_W2B = 0x170B6
MAT_X_W2C = 0x170BC
MAT_X_W3A = 0x170C2
MAT_X_W3B = 0x170C8
MAT_X_W3C = 0x170CE
MAT_X_W4A = 0x170D4
MAT_X_W4B = 0x170DA
MAT_X_W4C = 0x170E0
MAT_X_W5A = 0x170E6
MAT_X_W5B = 0x170EC
MAT_X_W5C = 0x170F2
MAT_X_W6A = 0x170F8
MAT_X_W6B = 0x170FE
MAT_X_W6C = 0x17104
MAT_X_W7A = 0x1710A
MAT_X_W7B = 0x17110
MAT_X_W7C = 0x17116
MAT_X_W8A = 0x1711C
MAT_X_W8B = 0x17122
MAT_X_W8C = 0x17128
MAT_Y_W1A = 0x1700E
MAT_Y_W1B = 0x17014
MAT_Y_W1C = 0x1701A
MAT_Y_W2A = 0x17020
MAT_Y_W2B = 0x17026
MAT_Y_W2C = 0x1702C
MAT_Y_W3A = 0x17032
MAT_Y_W3B = 0x17038
MAT_Y_W3C = 0x1703E
MAT_Y_W4A = 0x17044
MAT_Y_W4B = 0x1704A
MAT_Y_W4C = 0x17050
MAT_Y_W5A = 0x17056
MAT_Y_W5B = 0x1705C
MAT_Y_W5C = 0x17062
MAT_Y_W6A = 0x17068
MAT_Y_W6B = 0x1706E
MAT_Y_W6C = 0x17074
MAT_Y_W7A = 0x1707A
MAT_Y_W7B = 0x17080
MAT_Y_W7C = 0x17086
MAT_Y_W8A = 0x1708C
MAT_Y_W8B = 0x17092
MAT_Y_W8C = 0x17098
MO_Airship = 0x192B1
MO_CheckForBonus = 0x16C63
MO_CheckForBonusRules = 0x16C83
MO_CoinShip = 0x19357
MO_DoFortressFX = 0x148B9
MO_DoLevelClear = 0x169FB
MO_HammerBroMarch = 0x149F5
MO_HandTrap = 0x1646E
MO_NSpade = 0x192D6
MO_NormalMoveEnter = 0x14DDA
MO_Shop = 0x19318
MO_SkidAfarFinish = 0x16976
MO_SkidAfarPrep = 0x16929
MO_SkidToPrev = 0x16766
MO_SkidToPrevAfar = 0x168D9
MO_SwitchToMO_D = 0x147B0
MO_Unused = 0x192D0
MO_UnusedMapObj = 0x1937A
MO_Wait14Ticks = 0x147B8
MO_Wait8Proceed = 0x14A3B
MO_WorldXIntro = 0x1450A
MS1_01SegHedr = 0x38809
MS1_02SegHedr = 0x38822
MS1_04SegHedr = 0x38810
MS1_08SegHedr = 0x38829
MS1_10SegHedr = 0x38876
MS1_20SegHedr = 0x38817
MS1_40SegHedr = 0x38802
MS1_80SegHedr = 0x3881E
MS2ASegHedr00 = 0x3877C
MS2ASegHedr01 = 0x38783
MS2ASegHedr02 = 0x3878A
MS2ASegHedr03 = 0x38791
MS2ASegHedr04 = 0x38798
MS2ASegHedr05 = 0x3879F
MS2ASegHedr06 = 0x387A6
MS2ASegHedr07 = 0x387AD
MS2ASegHedr08 = 0x387B4
MS2ASegHedr09 = 0x387BB
MS2ASegHedr0A = 0x387C2
MS2ASegHedr0B = 0x387C9
MS2ASegHedr0C = 0x387D0
MS2ASegHedr0D = 0x387D7
MS2ASegHedr0E = 0x387DE
MS2ASegHedr0F = 0x387E6
MS2ASegHedr10 = 0x387ED
MS2ASegHedr11 = 0x387F4
MS2ASegHedr12 = 0x387FB
MS2ASegHedr1A = 0x38830
MS2ASegHedr1B = 0x38837
MS2ASegHedr1C = 0x3883E
MS2ASegHedr1D = 0x38845
MS2ASegHedr1E = 0x3884C
MS2ASegHedr1F = 0x38853
MS2ASegHedr20 = 0x3885A
MS2ASegHedr21 = 0x38861
MS2ASegHedr22 = 0x38868
MS2ASegHedr23 = 0x3886F
MSI_CalcDeltaAndSteps = 0x17A04
MSI_CalcStarsXY = 0x17903
MSI_CalcXOffset = 0x1799B
MSI_DefaultRadii = 0x17788
MSI_DrawStar = 0x179BA
MSI_RadSize = 0x17790
MapBonusChk_CoinShip = 0x16D41
MapBonusChk_MAPOBJ_UNK0C = 0x16D72
MapBonusChk_NSpade = 0x16C90
MapBonusChk_WhiteToadHouse = 0x16D01
MapMove_DeltaX = 0x14D8A
MapMove_DeltaXC = 0x14DA5
MapMove_DeltaXHi = 0x14D93
MapMove_DeltaXHiC = 0x14DAE
MapMove_DeltaY = 0x14D81
MapMove_DeltaYC = 0x14D9C
MapObj_DrawAndEnter = 0x175CE
MapObject_Attr1 = 0x174DF
MapObject_Attr2 = 0x1750F
MapObject_DrawSleepEnter = 0x1759D
MapObject_Pat1 = 0x1747F
MapObject_Pat2 = 0x174AF
MapObjects_UpdateDrawEnter = 0x1754B
MapSound_Play2FreqL = 0x3817B
MapSound_PlayFreqL = 0x3811F
MapSound_Playing = 0x380F0
MapSound_Queued = 0x380BC
MapSound_SetLen = 0x38112
MapSound_Stop = 0x38102
MapStarsIntro_Do = 0x17830
MapStarsIntro_DoStarFX = 0x1777C
MapStarsIntro_Init = 0x177A2
MapStarsIntro_Patterns = 0x177A0
MapTile_Get_By_Offset = 0x15379
Map_Airship_Dest_XSets = 0x16FDE
Map_Airship_Dest_YSets = 0x16FAE
Map_Airship_Pick_Travel = 0x162A9
Map_Airship_Travel_BaseIdx = 0x16291
Map_Airship_Travel_Random = 0x16299
Map_Airship_Travel_To = 0x17162
Map_AnimCHRROM = 0x17C35
Map_AnimSpeeds = 0x17C11
Map_BorderSprites = 0x14FF2
Map_Bottom_Tiles = 0x18464
Map_ByRowType = 0x193EC
Map_ByScrCol = 0x193FE
Map_ByXHi_InitIndex = 0x193DA
Map_Calc_NT2Addr_By_XY = 0x3D604
Map_CanoeCheckXHiOff = 0x1528C
Map_CanoeCheckXOff = 0x15288
Map_CanoeCheckYOff = 0x15284
Map_CheckDoMove = 0x15290
Map_Check_PanL = 0x15123
Map_Check_PanR = 0x150FD
Map_ClearLevelFXPatterns = 0x169C8
Map_Clear_EntTranMem = 0x3D563
Map_Completable_Tiles = 0x18457
Map_CompleteBit = 0x17A3D
Map_CompleteByML_Tiles = 0x1845C
Map_CompleteTile = 0x169DA
Map_CompleteY = 0x17A36
Map_Complete_Bits = 0x1843F
Map_Completion_Bit = 0x348DC
Map_ConfigWorldIntro = 0x1435D
Map_DoAnimations = 0x17C39
Map_DoEnterViaID = 0x19211
Map_DoInventory_And_PoofFX = 0x34080
Map_DoMap = 0x143D3
Map_DoMap_WarpWind_FX = 0x16325
Map_DoOperation = 0x144BC
Map_DoPlayer_As_Marker = 0x150AF
Map_DoPlayer_As_Twirl = 0x150CB
Map_DoPlayer_Edge_Scroll = 0x150EA
Map_DoVBorders = 0x14B94
Map_Do_Borders = 0x14AB2
Map_DrawAndPan = 0x14A9C
Map_DrawBorderForPlayer = 0x166FF
Map_DrawBridgeCheck = 0x1527C
Map_DrawBridgeCheckV = 0x15280
Map_DrawClearLevelPoof = 0x16BDF
Map_DrawPlayer = 0x1500A
Map_EnterLevel_Effect = 0x34CE1
Map_EnterSpecialTiles = 0x14DBF
Map_FindEmptyObjectSlot = 0x16DAD
Map_ForcePoofTiles = 0x169E5
Map_GetTile = 0x1520E
Map_HoldPadDir = 0x15248
Map_Init = 0x161E8
Map_IntroAttrSave = 0x142D4
Map_Intro_Erase1Strip = 0x14576
Map_LevelLayouts = 0x19422
Map_List_Object_IDs = 0x16050
Map_List_Object_Items = 0x16060
Map_List_Object_XHis = 0x16030
Map_List_Object_XLos = 0x16040
Map_List_Object_Ys = 0x16020
Map_MarchValidateTravel = 0x173AD
Map_MarchXtraForbidTiles = 0x173A9
Map_March_InitValues = 0x16DB9
Map_MarkLevelComplete = 0x17A45
Map_Marker_MorL = 0x14FDC
Map_NoAnimUpdate = 0x17C80
Map_NoLoseTurnTiles = 0x169F6
Map_No_Pan = 0x1513C
Map_ObjSets = 0x19410
Map_Object_Airship = 0x17136
Map_Object_Canoe = 0x17AC6
Map_Object_CheckVisibility = 0x17A9A
Map_Object_ColorSets = 0x18436
Map_Object_Do = 0x16E05
Map_Object_Do_All = 0x16DCD
Map_Object_Forbid_LandingTiles = 0x17398
Map_Object_March = 0x16E8C
Map_Object_March_NonNormal = 0x16EE2
Map_Object_March_Offset = 0x16E4C
Map_Object_March_OffsetXHi = 0x16E6C
Map_Object_March_PickTravel = 0x1744B
Map_Object_March_UpdateXs = 0x17373
Map_Object_March_XHiOffByDir = 0x17390
Map_Object_March_XOffByDir = 0x17388
Map_Object_March_YOffByDir = 0x17380
Map_Object_MusicBoxCheck = 0x16E9E
Map_Object_Stationary = 0x16FA4
Map_Object_Travel_X = 0x16E44
Map_Object_Travel_XHi = 0x16E48
Map_Object_Travel_Y = 0x16E40
Map_Object_Valid_Down = 0x1526A
Map_Object_Valid_Left = 0x15258
Map_Object_Valid_Right = 0x15261
Map_Object_Valid_Tiles = 0x15250
Map_Object_Valid_Up = 0x15273
Map_PanInit = 0x14C4F
Map_PanLeft = 0x14CF4
Map_PanRight = 0x14C5C
Map_PanelCompletePats = 0x169EA
Map_PlayerPalFix = 0x37820
Map_Poof_Tiles = 0x347C4
Map_Poof_To_Row = 0x348D5
Map_Poof_Update = 0x347D0
Map_PostJC_PUpPML = 0x14DD1
Map_PostJC_PUpPP1 = 0x14DCA
Map_PostJC_PUpPP2 = 0x14DD3
Map_Power_Attrib_F1 = 0x14F94
Map_Power_Attrib_F2 = 0x14FB8
Map_Power_Pats_F1 = 0x14F4C
Map_Power_Pats_F2 = 0x14F70
Map_PrepareLevel = 0x1910F
Map_Reload_with_Completions = 0x1846D
Map_Removable_Tiles = 0x18447
Map_RemoveTo_Tiles = 0x1844F
Map_RevDir = 0x1524C
Map_ScrollDeltaX = 0x143C6
Map_ScrollDeltaXHi = 0x143C8
Map_SetCompletion_By_Poof = 0x348E4
Map_Set_Pan = 0x15130
Map_SpriteRAM_Offset = 0x1753F
Map_StateNothing = 0x14D49
Map_Tile_ColorSets = 0x1842D
Map_Tile_Layouts = 0x185A8
Map_Unused7EEA_Vals = 0x16018
Map_W8D_XOffTable = 0x153B1
Map_W8D_YOffTable = 0x153B5
Map_W8Dark_IntroCover = 0x14560
Map_W8DarknessFill = 0x143A2
Map_W8DarknessUpdate = 0x153B9
Map_WWOrHT_StartX = 0x3476E
Map_WW_DeltaX = 0x16306
Map_WW_IslandX = 0x1631C
Map_WW_IslandY = 0x1630A
Map_WW_Pal = 0x1650D
Map_WW_Pattern = 0x1650F
Map_WW_StartX = 0x16304
Map_WW_TargetX = 0x16308
Map_WarpZone_DrawNumbers = 0x151B0
Map_WarpZone_Numbers = 0x15194
Map_WhiteObjects = 0x16C5F
Map_WorldIntro_DrawPlayer = 0x1453B
Map_WorldIntro_PSPat = 0x14534
Map_WorldIntro_PSpr = 0x1452C
Map_Y_Starts = 0x3C39A
Microgoomba_Draw = 0xF69E
Microgoomba_SprRAMAlt = 0xF5C6
Microgoomba_XAccel = 0xF5C2
Microgoomba_XLimit = 0xF5C4
Modulus_ByTempVar4 = 0x9B86
Move_Kuribo = 0x10B02
MuncherJelectroSet = 0x3E2
Mushroom_SetFall = 0x295A
Mushroom_SetUpVel = 0x286F
Mushroom_SetXVel = 0x2874
Music_CalcNoteLen = 0x3E766
Music_EndSegment = 0x3E4EB
Music_GetRestTicks = 0x3E34B
Music_NoiseLUTA = 0x3E75A
Music_NoiseLUTB = 0x3E75E
Music_NoiseLUTC = 0x3E762
Music_NseNoteOn = 0x3E6F3
Music_NseTrack = 0x3E6CD
Music_PCMTrack = 0x3E70D
Music_PatchGetCTL = 0x3E795
Music_PatchTableLong = 0x3E775
Music_PatchTableShort = 0x3E785
Music_PlayDMC = 0x3E2D0
Music_RestH_LUT = 0x3E884
Music_Set1_Set2A_Headers = 0x3877C
Music_Set1_Set2A_IndexOffs = 0x38750
Music_Set2A_Ends = 0x3888C
Music_Set2A_Loops = 0x3889B
Music_Set2A_Starts = 0x3887D
Music_Set2B_Ends = 0x3954C
Music_Set2B_Headers = 0x3943D
Music_Set2B_IndexOffs = 0x39410
Music_Set2B_Loops = 0x39558
Music_Set2B_Starts = 0x39540
Music_Sq1NoteOn = 0x3E5FC
Music_Sq1Track = 0x3E5BC
Music_Sq2NoteOn = 0x3E54D
Music_Sq2Track = 0x3E4D5
Music_StopAll = 0x3E511
Music_StopDMC = 0x3E2E6
Music_TriNoteOn = 0x3E69E
Music_TriTrack = 0x3E675
Music_UpdateBend = 0x3E867
NSpade_DoGame = 0x2D792
Negate = 0x1D5E
NonSlope_LUT_Addr = 0x20C
OTDO_G1Alt = 0x2BE
OTDO_G1R1 = 0x25E
OTDO_Water = 0x2C2
Obj2Obj_EnByState = 0x1BFF
ObjAutoScroller_Init = 0xB967
ObjB3_AttrByFrame = 0xA2FF
ObjHalt_DoNothing = 0x13B0
ObjHalt_DoNothingNotNormal = 0x1559
ObjHit_BoundLR = 0x2B2A
ObjHit_Card = 0x2DE2
ObjHit_CloudPlat = 0x4A6B
ObjHit_DoNothing = 0x13B0
ObjHit_DryBones = 0x5767
ObjHit_EndLevelCard = 0x5EEC
ObjHit_FireFlower = 0x2A09
ObjHit_GoombaInShoe = 0x510E
ObjHit_Koopaling = 0x3148
ObjHit_Obj01 = 0x22CC
ObjHit_Obj04 = 0x23DB
ObjHit_Obj05 = 0x2474
ObjHit_Obj0A = 0x2734
ObjHit_Obj1A = 0x2A43
ObjHit_PUp1UpMush = 0x27AA
ObjHit_PUpMush = 0x28A3
ObjHit_StarOrSuit = 0x280E
ObjHit_SuperLeaf = 0x2C3B
ObjInit_AirshipProp = 0xA8F7
ObjInit_Anchor = 0x269E
ObjInit_BigBerthaBirther = 0x71F9
ObjInit_BigCannonBall = 0xA373
ObjInit_BigQBlock = 0xB629
ObjInit_Blooper = 0x7765
ObjInit_BlooperWithKids = 0x7766
ObjInit_BobOmb = 0x66C5
ObjInit_BobOmbExplode = 0x66CE
ObjInit_BoltLift = 0xAB5F
ObjInit_BoomBoom = 0x69FA
ObjInit_BoomBoomQBall = 0x696B
ObjInit_BoomerangBro = 0x849A
ObjInit_BossAttack = 0x829F
ObjInit_BounceDU = 0x24BE
ObjInit_BounceLR = 0x2A59
ObjInit_Bowser = 0x3890
ObjInit_BusterBeatle = 0x44D8
ObjInit_ChainChomp = 0x99CB
ObjInit_ChasingFish = 0x5843
ObjInit_CheepCheepHopper = 0x744E
ObjInit_CheepCheepP2P = 0x476C
ObjInit_CheepCheepP2P2 = 0x4768
ObjInit_CloudPlatFast = 0x4A45
ObjInit_CloudPlatform = 0x4A52
ObjInit_DoNothing = 0x13B0
ObjInit_DonutLift = 0x6626
ObjInit_FallingPlatform = 0x4BBD
ObjInit_FireBro = 0x8890
ObjInit_FireChomp = 0x7976
ObjInit_FireFlower = 0x29C8
ObjInit_FireJetLeft = 0xA999
ObjInit_FireJetRight = 0xA9A2
ObjInit_FireJetUpsideDown = 0xA448
ObjInit_FireJetUpward = 0xA43A
ObjInit_FireSnake = 0x796D
ObjInit_FloatWoodenPlat = 0x4B31
ObjInit_FloatingBGCloud = 0x6A0F
ObjInit_GiantGreenPiranha = 0x9764
ObjInit_GiantRedPiranha = 0x9768
ObjInit_GiantTroop = 0x9237
ObjInit_GreenPiranha = 0xA648
ObjInit_GreenPiranhaFlip = 0xA634
ObjInit_GroundTroop = 0x923A
ObjInit_HammerBro = 0x8482
ObjInit_HeavyBro = 0x8A12
ObjInit_HotFoot = 0x480A
ObjInit_HotFootShy = 0x480F
ObjInit_InvisibleLift = 0x49C6
ObjInit_Koopaling = 0x2E30
ObjInit_Lakitu = 0x8CF4
ObjInit_LavaLotus = 0x7121
ObjInit_NipperHopping = 0x51AC
ObjInit_Obj01 = 0x228B
ObjInit_Obj02 = 0x2331
ObjInit_Obj04 = 0x23B6
ObjInit_Obj05 = 0x23EE
ObjInit_Obj0A = 0x2713
ObjInit_Obj1C = 0x2B4D
ObjInit_ObjB3 = 0xA2F6
ObjInit_OrangeCheep = 0x889B
ObjInit_OscillatingShort = 0x4A5A
ObjInit_PDoor = 0x2D34
ObjInit_PUpMush = 0x284A
ObjInit_ParaBeetle = 0xB0FD
ObjInit_Patooie = 0x4E45
ObjInit_PileDriver = 0x63E8
ObjInit_PipewayCtlr = 0x469C
ObjInit_PiranhaSidewaysL = 0x6F47
ObjInit_PiranhaSidewaysR = 0x6F40
ObjInit_PiranhaSpikeBall = 0x4E3B
ObjInit_Podoboo = 0xA22E
ObjInit_PodobooCeiling = 0x637E
ObjInit_RedPiranha = 0xA644
ObjInit_RedPiranhaFlip = 0xA62E
ObjInit_RockyWrench = 0xA9A5
ObjInit_RotatePlatform = 0xB1CA
ObjInit_RotatePlatformPer = 0xB1C5
ObjInit_RotoDiscDualCCW = 0x7E92
ObjInit_RotoDiscDualCW = 0x7E8B
ObjInit_Set3DoNothing = 0x9243
ObjInit_SpikeCheep = 0x9244
ObjInit_SpinyCheep = 0x2211
ObjInit_StarOrSuit = 0x27B9
ObjInit_Stretch = 0x493A
ObjInit_StretchFlip = 0x4953
ObjInit_Sun = 0xAD16
ObjInit_SuperLeaf = 0x2BA7
ObjInit_Thwomp = 0x867B
ObjInit_ThwompLRSlide = 0x8751
ObjInit_ThwompUDSlide = 0x87E4
ObjInit_TinyCheepCheep = 0x7456
ObjInit_Toad = 0x5220
ObjInit_Tornado = 0x748D
ObjInit_TowardsPlayer = 0x4CD1
ObjInit_TreasureBox = 0x62A7
ObjInit_Twirling = 0x657A
ObjInit_Vine = 0x2C6A
ObjInit_WarpHide = 0x2D3D
ObjInit_WaterCurrent = 0x712C
ObjInit_WoodenFallingPlat = 0x4BBB
ObjInit_WoodenPlat = 0x4A49
ObjInit_WoodenPlatform = 0x4A56
ObjKickXvel = 0xC7C
ObjLRFlags = 0xB9AA
ObjNorm_AirshipPropellar = 0xA903
ObjNorm_Anchor = 0x26A5
ObjNorm_ArrowPlatform = 0xAF41
ObjNorm_BigBertha = 0x5866
ObjNorm_BigBerthaBirther = 0x720D
ObjNorm_BigCannonBall = 0xA3A7
ObjNorm_BigPiranha = 0x978B
ObjNorm_BigQBlock = 0xB649
ObjNorm_Blooper = 0x777A
ObjNorm_BobOmb = 0x66DA
ObjNorm_BoltLift = 0xAB75
ObjNorm_Boo = 0x48E1
ObjNorm_BoomBoom = 0x6A26
ObjNorm_BoomBoomQBall = 0x68F8
ObjNorm_BoomerangBro = 0x84A9
ObjNorm_BossAttack = 0x82BF
ObjNorm_BounceDU = 0x2517
ObjNorm_BounceLR = 0x2A86
ObjNorm_Bowser = 0x389B
ObjNorm_BulletBill = 0x9169
ObjNorm_BusterBeatle = 0x44E6
ObjNorm_Card = 0x2DCE
ObjNorm_ChainChomp = 0x9A5D
ObjNorm_ChainChompFree = 0x78FF
ObjNorm_CharginCheepCheep = 0x5A20
ObjNorm_CheepCheepHopper = 0x7457
ObjNorm_CheepCheepP2P = 0x4788
ObjNorm_CheepCheepP2P2 = 0x47F6
ObjNorm_CloudPlat = 0x4A5E
ObjNorm_DoNothing = 0x13B0
ObjNorm_DonutLift = 0x6627
ObjNorm_DryBones = 0x5681
ObjNorm_EndLevelCard = 0x5B4B
ObjNorm_FireBro = 0x889E
ObjNorm_FireChomp = 0x79DA
ObjNorm_FireFlower = 0x29EA
ObjNorm_FireJet = 0xA45F
ObjNorm_FireSnake = 0x7D85
ObjNorm_FloatingBGCloud = 0x696C
ObjNorm_FlyingGreenPara = 0x92B9
ObjNorm_FlyingRedTroopa = 0x931A
ObjNorm_GiantBlockCtl = 0x4296
ObjNorm_GoombaInShoe = 0x503B
ObjNorm_GroundTroop = 0x9374
ObjNorm_HammerBro = 0x8536
ObjNorm_HeavyBro = 0x8A18
ObjNorm_Hotfoot = 0x4899
ObjNorm_HotfootShy = 0x4816
ObjNorm_IceBlock = 0x62A2
ObjNorm_InvisibleLift = 0x49D2
ObjNorm_JumpingCheepCheep = 0x90B3
ObjNorm_Koopaling = 0x2ED4
ObjNorm_Lakitu = 0x8D0C
ObjNorm_LavaLotus = 0x7020
ObjNorm_Nipper = 0x51AF
ObjNorm_NipperFireBreathe = 0x55EE
ObjNorm_Obj01 = 0x2294
ObjNorm_Obj02 = 0x237C
ObjNorm_Obj04 = 0x23C9
ObjNorm_Obj05 = 0x2401
ObjNorm_Obj0A = 0x2719
ObjNorm_Obj1A = 0x2A31
ObjNorm_Obj1C = 0x2B8B
ObjNorm_ObjB3 = 0xA303
ObjNorm_OrangeCheep = 0x8846
ObjNorm_OscillatingH = 0x4ADC
ObjNorm_OscillatingV = 0x4B17
ObjNorm_PDoor = 0x2D8F
ObjNorm_PUp1UpMush = 0x278E
ObjNorm_PUpMush = 0x2881
ObjNorm_ParaBeetle = 0xB106
ObjNorm_ParaGoomba = 0x8EA0
ObjNorm_ParaGoombaBomber = 0x8F83
ObjNorm_PathFollowPlat = 0x4C29
ObjNorm_Patooie = 0x4EBF
ObjNorm_PipewayCtlr = 0x470A
ObjNorm_Piranha = 0xA671
ObjNorm_PiranhaSideways = 0x6F56
ObjNorm_PiranhaSpikeBall = 0x4E4D
ObjNorm_Podoboo = 0xA254
ObjNorm_PodobooCeiling = 0x6387
ObjNorm_RedTroopa = 0x9263
ObjNorm_RockyWrench = 0xA9B7
ObjNorm_RotoDisc = 0x7FAA
ObjNorm_RotoDiscDual = 0x7F48
ObjNorm_RotoDiscDualOpp = 0x7ED0
ObjNorm_RotoDiscDualOpp2 = 0x7F0C
ObjNorm_Spike = 0x4D2B
ObjNorm_SpikeCheep = 0x90E9
ObjNorm_SpinyCheep = 0x222F
ObjNorm_SpinyEgg = 0x8C84
ObjNorm_SpinyEggDud = 0x8BF6
ObjNorm_StarOrSuit = 0x27E6
ObjNorm_Stretch = 0x4970
ObjNorm_Sun = 0xAD1F
ObjNorm_SuperLeaf = 0x2BE4
ObjNorm_Thwomp = 0x8687
ObjNorm_ThwompLRSlide = 0x8757
ObjNorm_ThwompVertical = 0x87F8
ObjNorm_TiltingPlatform = 0xB24E
ObjNorm_TinyCheepCheep = 0x73C2
ObjNorm_Toad = 0x5251
ObjNorm_ToadHouseItem = 0x54E9
ObjNorm_Tornado = 0x753A
ObjNorm_TreasureBox = 0x62D6
ObjNorm_TwirlingPlatCW = 0xB1FD
ObjNorm_TwirlingPlatCWNS = 0xB1E1
ObjNorm_TwirlingShell = 0x6593
ObjNorm_Vine = 0x2C72
ObjNorm_WarpHide = 0x2D48
ObjNorm_WaterCurrent = 0x713F
ObjNorm_WoodenPlatFloat = 0x4B4C
ObjNorm_WoodenPlatRider = 0x4A96
ObjNorm_WoodenPlatform = 0x4BA0
ObjP00 = 0x21C0
ObjP01 = 0x21C6
ObjP02 = 0x21C8
ObjP03 = 0x21C0
ObjP04 = 0x21CE
ObjP05 = 0x21D6
ObjP06 = 0x21DC
ObjP07 = 0x21C0
ObjP08 = 0x2205
ObjP09 = 0x21E4
ObjP0A = 0x21E7
ObjP0B = 0x21F3
ObjP0C = 0x21EB
ObjP0D = 0x21F3
ObjP0E = 0x21C0
ObjP0F = 0x21C0
ObjP10 = 0x21C0
ObjP11 = 0x21C0
ObjP12 = 0x21C0
ObjP13 = 0x21C0
ObjP14 = 0x21C0
ObjP15 = 0x21C0
ObjP16 = 0x21C0
ObjP17 = 0x21C0
ObjP18 = 0x21C0
ObjP19 = 0x21F5
ObjP1A = 0x21F7
ObjP1B = 0x21DC
ObjP1C = 0x21F9
ObjP1D = 0x21C0
ObjP1E = 0x21FB
ObjP1F = 0x21FD
ObjP20 = 0x21C0
ObjP21 = 0x21FF
ObjP22 = 0x2201
ObjP23 = 0x2203
ObjP24 = 0x421A
ObjP25 = 0x41C0
ObjP26 = 0x41D4
ObjP27 = 0x41D4
ObjP28 = 0x41D4
ObjP29 = 0x4232
ObjP2A = 0x423C
ObjP2B = 0x41DA
ObjP2C = 0x421A
ObjP2D = 0x41F0
ObjP2E = 0x4252
ObjP2F = 0x41C0
ObjP30 = 0x41C4
ObjP31 = 0x41CA
ObjP32 = 0x41CA
ObjP33 = 0x41DE
ObjP34 = 0x41E6
ObjP35 = 0x41C0
ObjP36 = 0x41D4
ObjP37 = 0x41D4
ObjP38 = 0x41D4
ObjP39 = 0x41DE
ObjP3A = 0x424C
ObjP3B = 0x4246
ObjP3C = 0x41D4
ObjP3D = 0x41EA
ObjP3E = 0x41D4
ObjP3F = 0x4220
ObjP40 = 0x4278
ObjP41 = 0x4256
ObjP42 = 0x4246
ObjP43 = 0x4246
ObjP44 = 0x41D4
ObjP45 = 0x41C4
ObjP46 = 0x423C
ObjP47 = 0x41C0
ObjP48 = 0x620C
ObjP49 = 0x61C0
ObjP4A = 0x620A
ObjP4B = 0x6212
ObjP4C = 0x6212
ObjP4D = 0x61C0
ObjP4E = 0x61C0
ObjP4F = 0x61C0
ObjP50 = 0x6202
ObjP51 = 0x6286
ObjP52 = 0x61C6
ObjP53 = 0x61E2
ObjP54 = 0x61F6
ObjP55 = 0x61FA
ObjP56 = 0x626A
ObjP57 = 0x626A
ObjP58 = 0x627A
ObjP59 = 0x6280
ObjP5A = 0x6286
ObjP5B = 0x6286
ObjP5C = 0x61E4
ObjP5D = 0x61C0
ObjP5E = 0x6286
ObjP5F = 0x6286
ObjP60 = 0x6286
ObjP61 = 0x628A
ObjP62 = 0x628A
ObjP63 = 0x6290
ObjP64 = 0x629C
ObjP65 = 0x61C0
ObjP66 = 0x61C0
ObjP67 = 0x626E
ObjP68 = 0x61EA
ObjP69 = 0x61F0
ObjP6A = 0x628A
ObjP6B = 0x61E8
ObjP6C = 0x8209
ObjP6D = 0x8209
ObjP6E = 0x8209
ObjP6F = 0x8209
ObjP70 = 0x8217
ObjP71 = 0x8225
ObjP72 = 0x8233
ObjP73 = 0x8233
ObjP74 = 0x8233
ObjP75 = 0x81C3
ObjP76 = 0x8245
ObjP77 = 0x8245
ObjP78 = 0x823B
ObjP79 = 0x823B
ObjP7A = 0x81ED
ObjP7B = 0x81ED
ObjP7C = 0x81F9
ObjP7D = 0x81E5
ObjP7E = 0x81ED
ObjP7F = 0x81E5
ObjP80 = 0x8209
ObjP81 = 0x8285
ObjP82 = 0x8285
ObjP83 = 0x825D
ObjP84 = 0x8251
ObjP85 = 0x8257
ObjP86 = 0x8269
ObjP87 = 0x8279
ObjP88 = 0x8245
ObjP89 = 0x81D3
ObjP8A = 0x81D9
ObjP8B = 0x81D9
ObjP8C = 0x81D9
ObjP8D = 0x81D9
ObjP8E = 0x81D9
ObjP8F = 0x81D9
ObjP90 = 0xA1C0
ObjP91 = 0xA1C0
ObjP92 = 0xA1C0
ObjP93 = 0xA1C0
ObjP94 = 0xA1C0
ObjP95 = 0xA1C0
ObjP96 = 0xA1C0
ObjP97 = 0xA1C0
ObjP98 = 0xA1C0
ObjP99 = 0xA1C0
ObjP9A = 0xA1C0
ObjP9B = 0xA1C0
ObjP9C = 0xA1C0
ObjP9D = 0xA1D2
ObjP9E = 0xA1C0
ObjP9F = 0xA204
ObjPA0 = 0xA1E4
ObjPA1 = 0xA1E4
ObjPA2 = 0xA1E4
ObjPA3 = 0xA1E4
ObjPA4 = 0xA1F4
ObjPA5 = 0xA1F4
ObjPA6 = 0xA1F4
ObjPA7 = 0xA1F4
ObjPA8 = 0xA1C0
ObjPA9 = 0xA1C0
ObjPAA = 0xA22C
ObjPAB = 0xA22A
ObjPAC = 0xA1C0
ObjPAD = 0xA21A
ObjPAE = 0xA212
ObjPAF = 0xA20A
ObjPB0 = 0xA1CE
ObjPB1 = 0xA1C0
ObjPB2 = 0xA1D2
ObjPB3 = 0xA1C6
ObjShell_AnimFlipBits = 0xDAC
ObjShell_AnimFrame = 0xDB0
ObjState_DeadEmpty = 0x13B0
ObjState_Held = 0xE31
ObjState_Initializing = 0x13B1
ObjState_Kicked = 0xC7E
ObjState_Killed = 0xFC1
ObjState_Normal = 0x1517
ObjState_PoofDying = 0xAB6
ObjState_Shelled = 0xB09
ObjState_Squashed = 0x107B
ObjWakeUp_FeetYOff = 0xB07
ObjectGroup00_Attributes = 0x20E8
ObjectGroup00_Attributes2 = 0x210C
ObjectGroup00_Attributes3 = 0x2130
ObjectGroup00_CollideJumpTable = 0x20A0
ObjectGroup00_InitJumpTable = 0x2010
ObjectGroup00_KillAction = 0x2178
ObjectGroup00_NormalJumpTable = 0x2058
ObjectGroup00_PatTableSel = 0x2154
ObjectGroup00_PatternSets = 0x21C0
ObjectGroup00_PatternStarts = 0x219C
ObjectGroup01_Attributes = 0x40E8
ObjectGroup01_Attributes2 = 0x410C
ObjectGroup01_Attributes3 = 0x4130
ObjectGroup01_CollideJumpTable = 0x40A0
ObjectGroup01_InitJumpTable = 0x4010
ObjectGroup01_KillAction = 0x4178
ObjectGroup01_NormalJumpTable = 0x4058
ObjectGroup01_PatTableSel = 0x4154
ObjectGroup01_PatternSets = 0x41C0
ObjectGroup01_PatternStarts = 0x419C
ObjectGroup02_Attributes = 0x60E8
ObjectGroup02_Attributes2 = 0x610C
ObjectGroup02_Attributes3 = 0x6130
ObjectGroup02_CollideJumpTable = 0x60A0
ObjectGroup02_InitJumpTable = 0x6010
ObjectGroup02_KillAction = 0x6178
ObjectGroup02_NormalJumpTable = 0x6058
ObjectGroup02_PatTableSel = 0x6154
ObjectGroup02_PatternSets = 0x61C0
ObjectGroup02_PatternStarts = 0x619C
ObjectGroup03_Attributes = 0x80E8
ObjectGroup03_Attributes2 = 0x810C
ObjectGroup03_Attributes3 = 0x8130
ObjectGroup03_CollideJumpTable = 0x80A0
ObjectGroup03_InitJumpTable = 0x8010
ObjectGroup03_KillAction = 0x8178
ObjectGroup03_NormalJumpTable = 0x8058
ObjectGroup03_PatTableSel = 0x8154
ObjectGroup03_PatternSets = 0x81C0
ObjectGroup03_PatternStarts = 0x819C
ObjectGroup04_Attributes = 0xA0E8
ObjectGroup04_Attributes2 = 0xA10C
ObjectGroup04_Attributes3 = 0xA130
ObjectGroup04_CollideJumpTable = 0xA0A0
ObjectGroup04_InitJumpTable = 0xA010
ObjectGroup04_KillAction = 0xA178
ObjectGroup04_NormalJumpTable = 0xA058
ObjectGroup04_PatTableSel = 0xA154
ObjectGroup04_PatternSets = 0xA1C0
ObjectGroup04_PatternStarts = 0xA19C
ObjectHeld_WakeUpDir = 0x10EF
ObjectHoldXHiOff = 0xE24
ObjectHoldXOff = 0xE19
ObjectID_BaseVals = 0xA51
ObjectKickXVelMoving = 0xE13
ObjectKill_SetShellKillVars = 0xD93
ObjectObject_Intersect = 0x198B
ObjectToObject_HitTest = 0x1C08
ObjectToObject_HitXVel = 0xE2F
Object_AboutFace = 0x116B
Object_AddVelFrac = 0x1D0B
Object_AnySprOffscreen = 0x1D64
Object_ApplyXVel = 0x1CF4
Object_ApplyYVel = 0x1CDD
Object_ApplyYVel_NoLimit = 0x1CE9
Object_AttrFlags = 0x304
Object_BoundBox = 0x2C4
Object_BumpBlocks = 0xDE2
Object_BumpOffBlocks = 0x1198A
Object_BumpOffOthers = 0xC34
Object_CalcAndDrawKilled = 0x1028
Object_CalcBoundBox = 0x193F
Object_CalcBoundBox2 = 0x1965
Object_CalcCoarseXDiff = 0x1CB2
Object_CalcHomingVels = 0x308C
Object_CalcSpriteXY_NoHi = 0x1564
Object_CheckIfNormalState = 0x63EB
Object_Delete = 0x1442
Object_DeleteOffScreen = 0x13F0
Object_DeleteOffScreen_N2 = 0x13EC
Object_DeleteOffScreen_N4 = 0x13E8
Object_DeleteOrDraw = 0x492F
Object_DetectTile = 0x7B9
Object_DetermineHorzVis = 0x17AE
Object_DetermineHorzVisY = 0x17BB
Object_DetermineVertVis = 0x17F2
Object_DetermineVertVisY = 0x17FD
Object_DoCollision = 0x19C8
Object_DoHaltedAction = 0x1530
Object_DoKillAction = 0xFD9
Object_DoKilledAction = 0x102B
Object_DoMoveBounceConveyor = 0x6940
Object_DoNormal = 0x151E
Object_DoStateAction = 0xA56
Object_Draw16x16Sprite = 0x1690
Object_Draw16x32Killed = 0x101C
Object_Draw16x32Sprite = 0x1628
Object_Draw24x16Sprite = 0x16D7
Object_DrawAndMoveNotHalt = 0x1010
Object_DrawMirroredKilled = 0x1022
Object_DrawShelled = 0xB71
Object_DrawTallAndHFlip = 0x1611
Object_DrawTallHFlipped = 0x1016
Object_DrawWide = 0x165C
Object_FallAndDelete = 0x1065
Object_FlipByXVel = 0x9E31
Object_FlipFace = 0x1172
Object_GetAttrAndMoveTiles = 0x6AD
Object_GetRandNearUnusedSpr = 0x1746
Object_GiantKilled = 0xFFF
Object_Gravity = 0x10A5
Object_HandleBumpUnderneath = 0x117D
Object_HandleConveyorBounceVel = 0x506
Object_HandleConveyorCarry = 0x4E6
Object_HitByTailOrBouncer = 0x1B02
Object_HitFloorAlign = 0x4ACF
Object_HitGround = 0x525
Object_HitGroundAlign = 0x2E0B
Object_HitTest = 0x1846
Object_HitTestRespond = 0x184B
Object_HoldKickOrHurtPlayer = 0x1341
Object_InitTailBuffer = 0x7984
Object_InteractWithWorld = 0x2976
Object_MaxFalls = 0x10A3
Object_Move = 0x10A7
Object_MoveAndReboundOffWall = 0x2DF3
Object_MoveNotHalted = 0x105D
Object_NormalAndKilled = 0xFF6
Object_NormalWhileKilled = 0xFFC
Object_PoofDie = 0x1005
Object_ResetXToSlotIndex = 0x5EB6
Object_RespondToTailAttack = 0x1B29
Object_SetDeadAndNotSpawned = 0x1461
Object_SetDeadEmpty = 0x146E
Object_SetHFlipByXVel = 0x78C6
Object_SetPaletteFromAttr = 0x13CC
Object_SetShakeAwakeTimer = 0xB7E
Object_SetShellState = 0x1336
Object_SetXVelByFacingDir = 0x4E2D
Object_ShakeAndCalcSprite = 0x1599
Object_ShakeAndDraw = 0x15F7
Object_ShakeAndDrawMirrored = 0x1600
Object_ShellDoWakeUp = 0x10F1
Object_StandardKill = 0xFF0
Object_TileDetectOffsets = 0x256
Object_ToggleFrameBySpd = 0x1670
Object_WaterSplash = 0x8A7
Object_WidthFlags = 0x17A8
Object_Widths = 0x17A2
Object_WorldDetect4 = 0x55A
Object_WorldDetect8 = 0x556
Object_WorldDetectN1 = 0x552
Objects_HandleScrollAndUpdate = 0x925
OneTile_ChangeToPatterns = 0x3BCD9
OneTile_ChangeToTile = 0x3BCC7
OrangeCheep_Accel = 0x8842
OrangeCheep_Limit = 0x8844
OscTimerSets = 0x4AE9
OscXVel = 0x4ADA
OscXVelLimit = 0x4AD8
PAGE_A000_ByTileset = 0x3C3F9
PAGE_C000_ByTileset = 0x3C3E6
PAUSE_Sprites = 0x3C410
PChg_C000_To_0 = 0x10241
PChg_C000_To_29 = 0x10239
PDiagBox_R1 = 0x31A46
PDiagBox_R2 = 0x31A57
PDiagBox_R3 = 0x31A68
PDiagBox_RowOffs = 0x31A79
PF00 = 0x3AC61
PF01 = 0x3AC67
PF02 = 0x3AC6D
PF03 = 0x3AC73
PF04 = 0x3AC79
PF05 = 0x3AC7F
PF06 = 0x3AC85
PF07 = 0x3AC8B
PF08 = 0x3AC91
PF09 = 0x3AC97
PF0A = 0x3AC9D
PF0B = 0x3ACA3
PF0C = 0x3ACA9
PF0D = 0x3ACAF
PF0E = 0x3ACB5
PF0F = 0x3ACBB
PF10 = 0x3ACC1
PF11 = 0x3ACC7
PF12 = 0x3ACCD
PF13 = 0x3ACD3
PF14 = 0x3ACD9
PF15 = 0x3ACDF
PF16 = 0x3ACE5
PF17 = 0x3ACEB
PF18 = 0x3ACF1
PF19 = 0x3ACF7
PF1B = 0x3AD03
PF1C = 0x3AD09
PF1D = 0x3AD0F
PF1E = 0x3AD15
PF1F = 0x3AD1B
PF20 = 0x3AD21
PF21 = 0x3AD27
PF22 = 0x3AD2D
PF23 = 0x3AD33
PF24 = 0x3AD39
PF25 = 0x3AD3F
PF26 = 0x3AD45
PF27 = 0x3AD4B
PF28 = 0x3AD51
PF29 = 0x3AD57
PF2A = 0x3AD5D
PF2B = 0x3AD63
PF2C = 0x3AD69
PF2D = 0x3AD6F
PF2E = 0x3AD75
PF2F = 0x3AD7B
PF30 = 0x3AD81
PF31 = 0x3AD87
PF32 = 0x3AD8D
PF33 = 0x3AD93
PF34 = 0x3AD99
PF35 = 0x3AD9F
PF36 = 0x3ADA5
PF37 = 0x3ADAB
PF38 = 0x3ADB1
PF39 = 0x3ADB7
PF3A = 0x3ADBD
PF3B = 0x3ADC3
PF3C = 0x3ADC9
PF3D = 0x3ADCF
PF3E = 0x3ADD5
PF3F = 0x3ADDB
PF40 = 0x3ADE1
PF41 = 0x3ADE7
PF42 = 0x3ADED
PF43 = 0x3ADF3
PF44 = 0x3ADF9
PF45 = 0x3ADFF
PF46 = 0x3AE05
PF47 = 0x3AE0B
PF48 = 0x3AE11
PF49 = 0x3AE17
PF4A = 0x3AE1D
PF4B = 0x3AE23
PF4C = 0x3AE29
PF4D = 0x3AE2F
PF4E = 0x3AE35
PF4F = 0x3AE3B
PF50 = 0x3AE41
PRGROM_Change_A000 = 0x3FFD2
PRGROM_Change_Both = 0x3FC7F
PRGROM_Change_Both2 = 0x3FFCF
PRGROM_Change_C000 = 0x3FFE1
PSwitch_SubstTileAndAttr = 0x853
PT2_Anim = 0x3C40C
PT2_Full_CHRROM_Loop = 0x3FFBF
PT2_Full_CHRROM_Switch = 0x3FFBD
PUpCoin_Attributes = 0xEFE6
PUpCoin_Patterns = 0xEFE2
PUp_GeneralCollect = 0x28C0
PUp_StarOrSuitFrames = 0x280A
PalSet_2PVs = 0x37662
PalSet_Airship = 0x37362
PalSet_Bonus = 0x37722
PalSet_Desert = 0x372A2
PalSet_Fort = 0x36D62
PalSet_Giant = 0x37422
PalSet_HighUp = 0x36EE2
PalSet_HillsUnder = 0x36E22
PalSet_Ice = 0x374E2
PalSet_Maps = 0x36BE2
PalSet_PipeMaze = 0x371E2
PalSet_Plains = 0x36CA2
PalSet_Plant = 0x36FA2
PalSet_Sky = 0x375A2
PalSet_Toad = 0x37122
PalSet_Water = 0x37062
Palette_By_Tileset = 0x377E2
Palette_DoFadeIn = 0x34BEC
Palette_DoFadeInTK = 0x37942
Palette_DoFadeOut = 0x34C3A
Palette_DoFadeOutTK = 0x3799D
Palette_FadeIn = 0x34C79
Palette_FadeOut = 0x34CBA
Palette_PrepareFadeIn = 0x34BB3
Palette_PrepareFadeInTK = 0x3790A
Palette_PrepareFadeOut = 0x34C30
Palette_PrepareFadeOutTK = 0x37993
Palette_PrepareFadeOutTK_Entry = 0x3790B
Palette_PrepareFadeOut_Entry = 0x34BB4
ParaBeetle_X = 0xBDC1
ParaBeetle_XVel = 0xBDC3
ParaBeetle_XVelTowardsPlayer = 0xB0FB
ParaGoomba_Draw = 0x8F2E
ParaTroopaFly_Accel = 0x92B5
ParaTroopaFly_Limit = 0x92B7
Parabeetle_FlipByXVel = 0xB0EF
Paragoomba_DropMicrogoombas = 0x902B
Paragoomba_FlutterDown = 0x8FB1
Paragoomba_FlyAbovePlayer = 0x8FFF
Paragoomba_SpawnMicroGoomba = 0x9084
Paragoomba_XVelAccel = 0x8F7F
Paragoomba_XVelLimit = 0x8F81
PatL1 = 0x391A6
PatL2 = 0x391FD
PatL3 = 0x39254
PatL4 = 0x392AC
PatL5 = 0x3931A
PatL6 = 0x39371
PatL8 = 0x393B1
PatS1 = 0x391E6
PatS2 = 0x3923D
PatS3 = 0x39294
PatS4 = 0x392EC
PatS5 = 0x39303
PatS6 = 0x3935A
PatS8 = 0x393F1
PatooiePiranha_StartTileX = 0x502D
Patooie_BallXJiggle = 0x4EBB
Patooie_BallYDelta = 0x4EA3
PileDriver_TowardsPlayerXVel = 0x63E9
Piledriver_YOff = 0x6502
PipeEnter_XYOffs = 0x11F26
PipeEntryPrepare = 0x11F2E
PipeMove_Frame = 0x1004B
PipeMove_LR = 0x3B4C3
PipeMove_LRH = 0x3B4C5
PipeMove_LR_Dir = 0x3B4C1
PipeMove_LeftRight = 0x3B789
PipeMove_SetPlayerFrame = 0x11F14
PipeMove_UpDown = 0x3B7AB
PipeMove_XVel = 0x3B4BD
PipeMove_YVel = 0x3B4BF
PipeTile_EnableByTileset = 0x11C45
PipeTransit_XDelta = 0x3B5E6
PipeTransit_XYOffsets = 0x3B5D4
PipeTransit_YDelta = 0x3B5E4
PipeTransit_YOff = 0x3B56D
Pipe_Move_Down = 0x3B4FA
Pipe_Move_JumpTable = 0x3B4B1
Pipe_Move_Left = 0x3B55F
Pipe_Move_Right = 0x3B51F
Pipe_Move_Up = 0x3B4DF
Pipe_PadDirForEnter = 0x11C56
Pipe_Transit = 0x3B56F
PipewayCtlr_MapScrlXHi = 0x46F2
PipewayCtlr_MapX = 0x46C2
PipewayCtlr_MapXHi = 0x46AA
PipewayCtlr_MapY = 0x46DA
PipewyCtl_GetUpperValue = 0x475D
PipewyCtl_SetCarryByPPos = 0x4751
PiranhaFireball_XVel = 0xA88A
PiranhaFireball_YVel = 0xA888
Piranha_Attack = 0xA7F1
Piranha_Emerge = 0xA7B9
Piranha_FacePlayerFlip = 0xA66D
Piranha_HideInPipe = 0xA844
Piranha_Retract = 0xA7D4
Piranha_SpitFire = 0xA88C
Piranha_Style = 0xA665
Piranha_VFlip = 0xA66F
PlantInfest_ACnt_MaxConst = 0x3C3BB
PlantInfest_MiniPipes = 0x11C60
PlantInfest_PatTablePerACnt = 0x3C3BC
PlantInfest_PiranhaTiles = 0x11C5E
Platform_Oscillate = 0x4AEB
Platform_SplitVar5 = 0xB56B
PlayerFall_Frame = 0x3673F
PlayerFireball_FlipBits = 0xE32B
PlayerFireball_Pats = 0xE327
PlayerHammer_FlipBits = 0xE32F
PlayerHammer_XOff = 0xE334
PlayerHammer_YOff = 0xE333
PlayerKickFlipBits = 0xE15
PlayerPlatform_Collide = 0x5AB7
PlayerProj_ChangeToPoof = 0xE4AD
PlayerProj_HitEnemies = 0xE656
PlayerProj_HitObject = 0xE69D
PlayerProj_ThrowWeapon = 0xE2BF
PlayerProj_UpdateAndDraw = 0xE343
PlayerProjs_UpdateAndDraw = 0xE339
PlayerPushWithPlatform_XVel = 0x5B19
PlayerSnd_Bump = 0x38299
PlayerSnd_CounterUpd = 0x3835F
PlayerSnd_FirBmpCont = 0x382AA
PlayerSnd_Fire = 0x38290
PlayerSnd_Frog = 0x3825E
PlayerSnd_FrogCont = 0x38276
PlayerSnd_Jump = 0x38265
PlayerSnd_JumpCont = 0x3830F
PlayerSnd_Kick = 0x38337
PlayerSnd_KickCont = 0x3834A
PlayerSnd_Pipe = 0x38306
PlayerSnd_Pipe2 = 0x38374
PlayerSnd_PipeCont = 0x38309
PlayerSnd_PipeCont2 = 0x3837C
PlayerSnd_Stop = 0x38364
PlayerSnd_Swim = 0x38312
PlayerSnd_SwimCont = 0x38323
PlayerStand_Frame = 0x36746
PlayerSuit_TileBaseIndex = 0x2C58E
PlayerSuit_ToIndex = 0x2C58B
PlayerY_HeightOff = 0x113BA
Player_AnimTailWag = 0x11046
Player_ApplyVelocity = 0x11FC4
Player_ApplyXVelocity = 0x11FAD
Player_ApplyYVelocity = 0x11FFD
Player_BonusTiles_Big = 0x2C567
Player_BonusTiles_Raccoon = 0x2C573
Player_BonusTiles_Small = 0x2C57F
Player_BoundBox = 0x183C
Player_ClimbFrame = 0x100A5
Player_CommonGroundAnims = 0x10DE9
Player_Control = 0x106CB
Player_ControlJmp = 0x106BA
Player_DetectSolids = 0x11483
Player_Die = 0x1A8C
Player_Die_Dying = 0x3B6DD
Player_Die_FellOff = 0x3B706
Player_Die_JumpTable = 0x3B6C4
Player_Die_NotDying = 0x3B4DE
Player_Die_TimeUp = 0x3B721
Player_DoClimbAnim = 0x1102A
Player_DoGameplay = 0x10151
Player_DoLavaDonutArrowBounce = 0xE02B
Player_DoScrolling = 0x1112F
Player_DoSpecialTiles = 0x11C62
Player_DoVibration = 0x104B1
Player_Draw = 0x3AEAF
Player_Draw29 = 0x1021D
Player_DrawAndDoActions = 0x3B1CA
Player_DrawAndDoActions29 = 0x10226
Player_DuckFrame = 0x10071
Player_FireInAir = 0x10076
Player_FireOnGround = 0x10073
Player_FramePageOff = 0x3AE47
Player_FrogHopFrames = 0x10079
Player_GetCard = 0x3FD54
Player_GetCardAndUpdate = 0x3FCCC
Player_GetHurt = 0x19E3
Player_GetInventoryOffset = 0x5E54
Player_GetItem = 0x3FD7C
Player_GetTileAndSlope = 0x113D7
Player_GetTileAndSlope_Normal = 0x3DEAD
Player_GetTileSlopeAndQuad = 0x11C27
Player_GetTileV = 0x3DE4C
Player_GroundHControl = 0x10B54
Player_GrowFrames = 0x1003F
Player_HitEnemy = 0x11CA
Player_HoldingFrames = 0x10091
Player_HurtIfNotDieOffBehind = 0x7D62
Player_JumpFlyFlutter = 0x10C38
Player_KickObject = 0xE42
Player_KuriboFrame = 0x1006F
Player_LoseKuribo = 0x1AC5
Player_PUpRootPage = 0x3AEA8
Player_PowerUpdate = 0x104D2
Player_RootJumpVel = 0x10657
Player_SetSpecialFrames = 0x10ED3
Player_ShootAnim = 0x1111A
Player_SkidFrame = 0x1004F
Player_SoarJumpFallFrame = 0x110B0
Player_SomersaultFlipBits = 0x1066E
Player_SpeedJumpInc = 0x10658
Player_SpreadEagleFrames = 0x1007D
Player_StandOnPlatform = 0x4A74
Player_StopMovement = 0x3B7BF
Player_SuitLost_DoPoof = 0x3B7CF
Player_SwimActiveFrames = 0x10051
Player_SwimAnim = 0x10D8F
Player_SwimIdleFrames = 0x1005D
Player_SwimV = 0x10D3F
Player_TailAttackAnim = 0x110D6
Player_TailAttackDo = 0x1B56
Player_TailAttackFrames = 0x10065
Player_TailAttackXOff = 0x1AFE
Player_TailAttack_HitBlocks = 0x11960
Player_TailAttack_Offsets = 0x1195C
Player_TailWagFlyFrames = 0x10085
Player_TanookiStatue = 0x10FBE
Player_Twirl_Attribs = 0x14FEA
Player_Twirl_Tiles = 0x14FE2
Player_TwisterSpinFrames = 0x10099
Player_UnderwaterHControl = 0x10C12
Player_Update = 0x103DA
Player_UphillSpeedVals = 0x10B49
Player_VibeDisableFrame = 0x100B3
Player_WalkAnimTickMax = 0x1064E
Player_WalkFramesByPUp = 0x10023
Player_WaterOrWaterfallVizFX = 0xE78A
Player_WaterSplash = 0x100C2
Player_XAccelMain = 0x105AC
Player_XAccelMain_UW = 0x105E4
Player_XAccelPseudoFrac = 0x105FC
Player_XAccelPseudoFrac_UW = 0x10634
Podoboo_Splash = 0x8A9
Podoboo_YVelByHeight = 0xA23E
PoofDeath_Pats = 0xAB2
Poof_Patterns = 0xFAA3
PostPSwitchAttr = 0x84F
PostPSwitchTile = 0x84B
PowerUpMovement_JumpTable = 0x10975
PowerUp_Ability = 0x3F0
PowerUp_BounceXVel = 0x296A
PowerUp_DoHitTest = 0x289A
PowerUp_DoRaise = 0x28D4
PowerUp_Palettes = 0x10539
PowerUp_PlaySound = 0x28CB
PrePSwitchTile = 0x847
PrepareNewObjectOrAbort = 0xFFDD
Princess_FacePatterns = 0x36119
Projectile_BBoxX = 0xE68D
Projectile_BBoxY = 0xE67D
Prop_LowerYOff = 0xA901
Prop_UpperYOff = 0xA8FF
QBoxBlue_Tiles = 0x2C65D
QBoxOrange_Tiles = 0x2C636
Randomize = 0x3D994
Read_Joypad = 0x3FF22
Read_Joypad_Loop = 0x3FF2D
Read_Joypads = 0x3FECE
Read_Joypads_UnkTable = 0x3FEBE
RecordBlockHitBits = 0x3D9B2
Rescue_Princess = 0x3186A
Reset_PPU_Clear_Nametables = 0x3FDB9
Reset_PPU_Clear_Nametables2 = 0x3FDC1
Respawn_XHi = 0x582D
Respawn_XLo = 0x582B
RockBreak_Replace = 0x346C1
RockBreak_TileFix = 0x346C3
RockyWrench_FlipBits = 0xA9A3
Rocky_DieOrWaitRevive = 0xAA76
Rocky_Draw = 0xAB49
Rocky_FaceAndPopup = 0xAA21
Rocky_FacePlayer = 0xA9A7
Rocky_InitAttr = 0xFEF0
Rocky_KillOrStandOn = 0xAAAC
Rocky_Killed = 0xAA9B
Rocky_ReadyThrowWrench = 0xAA5E
Rocky_RiseUp = 0xAA49
Rocky_ThrowWrench = 0xAB01
Rocky_TimerReload = 0xA9B1
Rocky_UpdFrameAdvIntState = 0xAA60
Rocky_WaitTimer = 0xAA39
Rocky_WrenchToss = 0xAA66
Rocky_WrenchYVel = 0xAAF9
Rotation_Colors = 0xE19B
RotoDiscSpin = 0x7C08
RotoDisc_CollideAndCycle = 0x7FB8
RotoDisc_VelAccels = 0x79BF
RotoDisc_VelLimits = 0x79CE
RouletteRow_DoStopState = 0x2D102
RouletteRow_EndTurn = 0x2D22A
RouletteRow_HitLockPos = 0x2D17B
RouletteRow_LockDecide = 0x2D1B1
RouletteRow_Move = 0x2D114
RouletteRow_Slow = 0x2D13A
RouletteRow_WaitToStop = 0x2D12F
Roulette_BorderSprites = 0x2CE75
Roulette_ConfigPPUCTL2 = 0x2D04E
Roulette_ConfigTurns = 0x2D056
Roulette_Configure = 0x2D044
Roulette_DoFadeOut = 0x2D31C
Roulette_DoGame = 0x2D03A
Roulette_DrawBorderSprites = 0x2CE8D
Roulette_DrawShapes = 0x2D586
Roulette_DrawXUpReward = 0x2D2E2
Roulette_ExitToMap = 0x2D26C
Roulette_FadeOut = 0x2D255
Roulette_Flower = 0x2D4DE
Roulette_GiveReward = 0x2D285
Roulette_Init = 0x2D06F
Roulette_MatchReward = 0x2D1AD
Roulette_MatchSets = 0x2D1A5
Roulette_MoveRow = 0x2D34D
Roulette_Mushroom = 0x2D436
Roulette_RasterDelay = 0x3FAD9
Roulette_RasterDiv = 0x3FADD
Roulette_RewardPattern = 0x2D280
Roulette_Run = 0x2D097
Roulette_RunGame = 0x2D05F
Roulette_Shapes = 0x2D388
Roulette_Star = 0x2D38E
Round2LineBlank_VRAMH = 0x2C99F
Round2LineBlank_VRAML = 0x2C9A2
SMB3J_SuitLossFrame = 0x19DC
SObjYOff_PlayerSize = 0xF6EE
SObj_AddXVelFrac = 0xFB2D
SObj_AddYVelFrac = 0xFB51
SObj_ApplyXYVelsWithGravity = 0xF39E
SObj_BlooperKid = 0xF07F
SObj_Boomerang = 0xF8BB
SObj_Bubble = 0xF331
SObj_Cannonball = 0xF3B8
SObj_CheckHitSolid = 0xEE60
SObj_CoinOrDebris = 0xF10A
SObj_DoNothing = 0xEF33
SObj_Draw16x16 = 0xF573
SObj_ExplodeStar = 0xF1B5
SObj_Fireball = 0xFA08
SObj_GetSprRAMOffChkVScreen = 0xFAF1
SObj_Hammer = 0xF6F4
SObj_KuriboShoe = 0xF549
SObj_Laser = 0xEFEA
SObj_LavaLotusFire = 0xF2A2
SObj_Microgoomba = 0xF5D0
SObj_OffsetYForRaster = 0xF486
SObj_PlayerCollide = 0xF7E1
SObj_Poof = 0xFAA7
SObj_SetSpriteXYRelative = 0xF4E8
SObj_Spikeball = 0xF586
SObj_SprRAMBase = 0xFAE8
SObj_UNKNOWN = 0xF992
SObj_UNKNOWN_XAccel = 0xF98A
SObj_UNKNOWN_XLimit = 0xF98C
SObj_UNKNOWN_YAccel = 0xF98E
SObj_UNKNOWN_YLimit = 0xF990
SObj_VLimit = 0xF6F0
SObj_Wand = 0xF209
SObj_WandBlast = 0xF508
SObj_Wrench = 0xF4AA
SPPF_Offsets = 0x3AC10
SPPF_Table = 0x3AC61
ScoreAdd_H = 0xEA84
ScoreAdd_L = 0xEA77
Score_FindFreeSlot = 0x49E
Score_Get100PlusPts = 0x46E
Score_Get100PlusPtsY = 0x46A
Score_GiveAndDraw = 0xEAC1
Score_PatternLeft = 0xEA5D
Score_PatternRight = 0xEA6A
Score_PopUp = 0x477
Score_RiseCounterMask = 0xEADB
Score_SetAttribute = 0xEBC6
Scores_GiveAndDraw = 0xEA91
Scroll_ColumnLOff = 0x14C59
Scroll_Commit_Column = 0x352EC
Scroll_Dirty_Update = 0x3DACA
Scroll_DoColumn = 0x3DBE5
Scroll_Do_AttrColumn = 0x3DC88
Scroll_Do_AttrRow = 0x3DDCC
Scroll_Map_SpriteBorder = 0x151BC
Scroll_PPU_Reset = 0x3FDA7
Scroll_ToVRAM_Apply = 0x3539F
Scroll_Update = 0x3DB83
Scroll_Update_Ranges = 0x3D69E
Search_XHiOff = 0x15371
Search_XOff = 0x15375
Search_YOff = 0x1536D
SetPages_ByTileset = 0x3D4CB
SetRotatingColor = 0x67A2
Setup32PixPartWater = 0x13E8E
Setup_PalData = 0x3784F
Shoe_DrawGoomba = 0x5148
Shoe_EjectGoomba = 0x50C5
Shoe_GoombaYOff = 0x5139
SidePiranha_Chomp = 0x6FA0
SidePiranha_Draw = 0x6FB7
SidePiranha_Emerge = 0x6F83
SidePiranha_EndSprXOff = 0x6FB5
SidePiranha_HideInPipe = 0x6FAD
SidePiranha_Receed = 0x6F90
SlideThwomp_InitXVel = 0x87DE
SlideThwomp_InitYVel = 0x87E1
Slope_ChkFeetOrHead = 0x11C25
Slope_CorrectH = 0x119E4
Slope_CorrectL = 0x119E6
Slope_IsNotFloorShape = 0x113C2
Slope_LUT = 0x97
Slope_LUT_Addr = 0x20E
Slope_ObjectVel_Effect = 0x1FC
Slope_PlayerVel_Effect = 0x1E7
SndLev1_1up = 0x384B6
SndLev1_1upCont = 0x384BE
SndLev1_1upData = 0x38395
SndLev1_Boom = 0x38401
SndLev1_Boom_Cont = 0x384B0
SndLev1_Boom_Cont2 = 0x3840F
SndLev1_Coin = 0x38450
SndLev1_Coin2 = 0x383D9
SndLev1_Coin_Cont = 0x384AD
SndLev1_Coin_Cont2 = 0x383F3
SndLev1_Data = 0x3856B
SndLev1_DataLongWag = 0x385D7
SndLev1_DataLostShoe = 0x385C2
SndLev1_DataPoof = 0x3858A
SndLev1_DataUnk = 0x3856B
SndLev1_PUp = 0x3841F
SndLev1_PUpData = 0x3839B
SndLev1_PUpRise = 0x3844D
SndLev1_PUpRise2 = 0x384DF
SndLev1_PUpRiseData = 0x383BA
SndLev1_PUpRise_Cont = 0x384F8
SndLev1_PUp_1up = 0x384B3
SndLev1_PUp_Cont = 0x38427
SndLev1_PUp_Cont2 = 0x38438
SndLev1_SuitLost = 0x38456
SndLev1_SuitLost2 = 0x38517
SndLev1_SuitLost_Cont = 0x384A7
SndLev1_SuitLost_Cont2 = 0x38542
SndLev1_Text = 0x384AA
SndLev1_Text2 = 0x383E2
SndLev1_Text_Cont = 0x384AD
SndLev1_VineRise = 0x38453
SndLev1_VineRise2 = 0x384E6
SndLev2_Airship = 0x38707
SndLev2_AirshipCont = 0x3870F
SndLev2_Boomerang = 0x386E8
SndLev2_BoomerangCont = 0x386F0
SndLev2_BoomerangData = 0x385FE
SndLev2_Crumble = 0x3863C
SndLev2_CrumbleCont = 0x38644
SndLev2_Flame = 0x386B5
SndLev2_FlameCont = 0x386BD
SndLev2_March = 0x386C9
SndLev2_MarchCont = 0x386D1
SndLev2_MarchData = 0x385F0
SndLev2_Skid = 0x38620
SndLev2_SkidCont = 0x38628
SndLev2_SkidNCtl = 0x38739
SndLev2_SkidNFreq = 0x3860A
SndLev2_SkidTFreq = 0x38619
SndMapH_Bonus = 0x381CF
SndMapH_Deny = 0x381D1
SndMapH_Enter = 0x381CB
SndMapH_Entrance = 0x381C7
SndMapH_Flip = 0x381CD
SndMapH_Move = 0x381C9
SndMapH_Unused = 0x381D3
SndMap_Data = 0x381D5
SndMap_Data_Bonus = 0x38211
SndMap_Data_Deny = 0x38219
SndMap_Data_Entr = 0x381E6
SndMap_Data_Entr2 = 0x381F5
SndMap_Data_Flip = 0x38206
SndMap_Data_Move = 0x3820C
SndMap_Data_Unused = 0x381E5
SndMap_Data_WEnt = 0x381D5
SndMap_Entr_VolData = 0x381AC
SndMus2A_LoadNext = 0x3E46A
SndMus2A_Next = 0x3E442
SndMus2A_NextOrStop = 0x3E43F
SndMus2B_LoadNext = 0x3E3C6
SndMus2B_Next = 0x3E3B3
SndMus2B_NextOrStop = 0x3E3B0
SndMus_Queue1 = 0x3E455
SndMus_QueueCommon = 0x3E3FE
SndMus_QueueCommonJ = 0x3E359
SndMus_QueueSet2B = 0x3E38B
SndPause = 0x38027
Sound1_XCTL_YRAMP = 0x3E7BB
Sound2_XCTL_YRAMP = 0x3E7C2
Sound_Engine_Begin = 0x38010
Sound_FullPowerRing = 0x104EB
Sound_Map_LUT = 0x381BF
Sound_PlayLevel1 = 0x38459
Sound_PlayLevel2 = 0x38671
Sound_PlayMapSounds = 0x380B1
Sound_PlayMusic = 0x3E35F
Sound_PlayPlayer = 0x382BB
Sound_Process = 0x38088
Sound_Sq1_NoteOn = 0x3E7C9
Sound_Sq1_NoteOn_NoPAPURAMP = 0x3E7CF
Sound_Sq2_NoteOn = 0x3E83B
Sound_Sq2_NoteOn_NoPAPURAMP = 0x3E841
Sound_StatueSwitch = 0x11021
Sound_Tri_NoteOn = 0x3E845
Spawn3TroopsOrCheeps = 0xB9B6
Spawn3Var4 = 0xB9B3
Spawn3XHiOff = 0xB9AE
Spawn3XOff = 0xB9AC
Spawn3YVels = 0xB9B0
SpecialObj_FindEmptyAbort = 0x457
SpecialObj_FindEmptyAbortY = 0x459
SpecialObj_Remove = 0xF84E
SpecialObj_RemoveInd = 0xEFAB
SpecialObj_UpdateAndDraw = 0xEF34
SpecialObjs_UpdateAndDraw = 0xEF24
SpikeBall_Patterns = 0x422E
SpikeCheepX = 0xBD64
SpikeCheepXVel = 0xBD66
SpikeCheep_XVelTowardsPlayer = 0x922F
SpikeCheep_YAccel = 0x90E5
SpikeCheep_YLimit = 0x90E7
Spike_Frames = 0x4CE0
Spike_TossSpikeBall = 0x4DFE
Spike_XOff = 0x4CEF
Spike_YOff = 0x4D0D
SpikesEnable = 0x3C6
SpinyCheep_XVel = 0x220F
SpinyCheep_YVelAccel = 0x222B
SpinyCheep_YVelMax = 0x222D
SpinyEggDud_FlipTowardsPlayer = 0x8BF4
SpinyEgg_HatchFacePlayerFlip = 0x8C82
SpinyEgg_TowardsPlayer = 0xA2F4
SplashBubble_XOff = 0x10125
SplashBubble_YOff = 0x10122
Splash_Draw = 0xE9C0
Splash_Patterns = 0xE9B4
Splash_Remove = 0xE9BA
Splash_UpdateAndDraw = 0xE989
SprRamOffsets = 0x411
SpriteHideCHR_1000 = 0x3F7A1
SpriteHideCHR_1400 = 0x3F7A2
SpriteHideCHR_1800 = 0x3F7A3
SpriteHideCHR_1C00 = 0x3F7A4
SpriteMTCHR_1000 = 0x3F79B
SpriteMTCHR_1400 = 0x3F79C
SpriteMTCHR_1800 = 0x3F79D
SpriteMTCHR_1C00 = 0x3F79E
Sprite_NoCarryIfVisible = 0x78D7
Sprite_RAM_Clear = 0x3FD94
Square1_Table_Notes = 0x3E849
Star_Palettes = 0x27B5
Star_Vel = 0x6EF7
StatusBarCHR_0000 = 0x3F79F
StatusBarCHR_0800 = 0x3F7A0
StatusBarMTCHR_0000 = 0x3F799
StatusBarMTCHR_0800 = 0x3F79A
StatusBar_DrawCardPiece = 0x3FCE8
StatusBar_Fill_Coins = 0x35083
StatusBar_Fill_Lives = 0x3504C
StatusBar_Fill_MorL = 0x35118
StatusBar_Fill_PowerMT = 0x35266
StatusBar_Fill_Score = 0x35185
StatusBar_Fill_Time = 0x34FAD
StatusBar_Fill_World = 0x350D4
StatusBar_UpdTemplate = 0x35438
StatusBar_UpdateValues = 0x3545A
StatusBar_Update_Cards = 0x3FCCF
Stretch_TimerByIntState = 0x496C
Stretch_XVelStart = 0x4935
SuitLost_Poof_Patterns = 0x3B7CB
Sun_DoMovement = 0xAE32
Sun_Draw = 0xAE6B
Sun_Patterns = 0xAE67
Sun_SpriteYOffs = 0xAE69
Sun_TimerReload = 0xADBE
Sun_VelAccel = 0xAD17
Sun_VelLimits = 0xAD1B
Sun_WaitAndResetTimer20 = 0xAE21
Sun_WaitFarEnough = 0xAD79
Sun_WaitForUpperReturn = 0xADC2
Sun_WaitForUpperReturn2 = 0xADF4
Sun_WaitThenAttackLeft = 0xADA7
Sun_WaitThenAttackRight = 0xAD8E
SwimBubble_XOff = 0xE782
SwimBubble_YOff = 0xE77A
SwimCTL1_LUT = 0x3821F
Swim_FireHammer = 0x109D6
Swim_Frog = 0x10A43
Swim_SmallBigLeaf = 0x109B4
Swim_Tanooki = 0x10AF5
TAndK_DoKingText = 0x364B2
TAndK_DoState = 0x3603A
TAndK_DoToadText = 0x3021A
TAndK_DrawDiagBox = 0x300B7
TAndK_DrawDiagBox2 = 0x36255
TAndK_FadeOutAndGetItem = 0x360AD
TAndK_Init = 0x3604A
TAndK_PlayerFallLandDraw = 0x3674D
TAndK_WaitForA = 0x36140
TAndK_WaitPlayerButtonA = 0x30271
TBoxItem_MirrorFlags = 0x62C8
TDiagBox_R1 = 0x5290
TDiagBox_R2 = 0x52A1
TDiagBox_R3 = 0x52B2
TDiagBox_RowOffs = 0x52C3
THouse_ChestColumns = 0x24B0B
THouse_ChestOpenBit = 0x24B0F
TIMEUP_SpriteXs = 0x3B71A
TIMEUP_Sprites = 0x3B713
TOADO = 0xC043
TOAD_SpecO = 0xC048
T_SP_00 = 0x30DE5
T_SP_01 = 0x30DEB
T_SP_02 = 0x30DF1
T_SP_03 = 0x30DF7
T_SP_04 = 0x30DFD
T_SP_05 = 0x30E03
T_SP_06 = 0x30E09
T_SP_07 = 0x30E0F
T_SP_08 = 0x30E15
T_SP_09 = 0x30E1B
T_SP_0A = 0x30E21
T_SP_0B = 0x30E27
T_SP_0C = 0x30E2D
T_SP_0D = 0x30E33
T_SP_0E = 0x30E39
T_SP_0F = 0x30E3F
T_SP_10 = 0x30E45
T_SP_11 = 0x30E4B
T_SP_12 = 0x30E51
T_SP_13 = 0x30E57
T_SP_14 = 0x30E5D
T_SP_15 = 0x30E63
T_SP_16 = 0x30E69
T_SP_17 = 0x30E6F
T_SP_18 = 0x30E75
T_SP_19 = 0x30E7B
T_SP_1A = 0x30E81
T_SP_1B = 0x30E87
TailEnemy_DoDeath = 0x7E4D
TailEnemy_DoStandard = 0x7DFC
TailEnemy_DrawAndWaitTimer2 = 0x7E3A
TailEnemy_InitOrHalt = 0x7E13
TailEnemy_MoveTailWithScroll = 0x7AB3
Tail_CheckHurtPlayer = 0x7D21
Tail_DrawAndHurtPlayer = 0x7AE3
Tail_PlayerYLimit = 0x7D1F
Tail_PlayerYOff = 0x7D1D
ThwompLR_Return = 0x87C2
ThwompLR_ReturnXVel = 0x87A0
ThwompLR_Slide = 0x87A2
ThwompLR_WaitForPlayer = 0x876F
ThwompLR_XVelAccel = 0x879C
ThwompLR_XVelLimit = 0x879E
ThwompVert_Reverse = 0x8814
ThwompVert_Slide = 0x882F
Thwomp_Center = 0x867F
Thwomp_Draw = 0x86FD
Thwomp_FallToGround = 0x86B7
Thwomp_ReturnToOrigin = 0x86E5
TileAttrAndQuad_OffsFlat = 0x11356
TileAttrAndQuad_OffsFlat_Sm = 0x11376
TileAttrAndQuad_OffsSlopeEdge = 0x113B6
TileAttrAndQuad_OffsSloped = 0x11396
TileAttrAndQuad_OffsSloped_Sm = 0x113A6
TileAttribute_ByTileset = 0x3D501
TileChng_DoNothing = 0x3BD69
TileChng_DoorAppear = 0x3BD73
TileChng_DoorBufCmds = 0x3BD6A
TileChng_OneTile = 0x3BD21
TileChng_ToadBoxBufCmds = 0x3BDD8
TileChng_ToadBoxOpen = 0x3BDE7
TileChng_VRAMCommit = 0x353F6
TileLayoutPage_ByTileset = 0x3DAB7
TileLayout_ByTileset = 0x3DA07
TileLayout_GetBaseAddr = 0x3DE22
Tile_Attributes_TS0 = 0x18410
Tile_Attributes_TS1 = 0x1E410
Tile_Attributes_TS10 = 0x2E410
Tile_Attributes_TS14 = 0x1A410
Tile_Attributes_TS15_TS16_TS17 = 0x3D527
Tile_Attributes_TS18 = 0x1C410
Tile_Attributes_TS2 = 0x2A410
Tile_Attributes_TS3 = 0x20410
Tile_Attributes_TS4_TS12 = 0x22410
Tile_Attributes_TS5_TS11_TS13 = 0x26410
Tile_Attributes_TS6_TS7_TS8 = 0x24410
Tile_Attributes_TS9 = 0x28410
Tile_Layout_TS0 = 0x18010
Tile_Layout_TS1 = 0x1E010
Tile_Layout_TS10 = 0x2E010
Tile_Layout_TS14 = 0x1A010
Tile_Layout_TS15_TS16_TS17 = 0x2C010
Tile_Layout_TS18 = 0x1C010
Tile_Layout_TS2 = 0x2A010
Tile_Layout_TS3 = 0x20010
Tile_Layout_TS4_TS12 = 0x22010
Tile_Layout_TS5_TS11_TS13 = 0x26010
Tile_Layout_TS6_TS7_TS8 = 0x24010
Tile_Layout_TS9 = 0x28010
Tile_Mem_Addr = 0x3C010
Tile_Mem_AddrVH = 0x3C03E
Tile_Mem_AddrVL = 0x3C02E
Tile_Mem_Clear = 0x1841E
Tile_Mem_ClearA = 0x3D715
Tile_Mem_ClearB = 0x3D744
Tile_Mem_ClearV = 0x24453
TiltPlat_XVelAccel = 0xB24A
TiltPlat_XVelLimit = 0xB24C
TimeBonus_Score = 0x420
Timer_NoChange = 0x35000
TinyCheepCheep_XVelLimit = 0x73BE
TinyCheep_Flip = 0x7356
TinyCheep_XOff = 0x7354
TinyCheep_XVel = 0x7352
TitleScreen_DarkPal = 0x32A97
TitleScreen_LogoAttr1 = 0x32A4F
TitleScreen_LogoAttr2 = 0x32A73
TitleScreen_LogoLine01 = 0x3277D
TitleScreen_LogoLine02 = 0x327A1
TitleScreen_LogoLine03 = 0x327C5
TitleScreen_LogoLine04 = 0x327E9
TitleScreen_LogoLine05 = 0x327EE
TitleScreen_LogoLine06 = 0x3280B
TitleScreen_LogoLine07 = 0x32828
TitleScreen_LogoLine08 = 0x32846
TitleScreen_LogoLine09 = 0x32864
TitleScreen_LogoLine10 = 0x32888
TitleScreen_LogoLine11 = 0x328AC
TitleScreen_LogoLine12 = 0x328D0
TitleScreen_LogoLine13 = 0x328F4
TitleScreen_LogoLine14 = 0x32918
TitleScreen_LogoLine15 = 0x32932
TitleScreen_LogoLine16 = 0x3294C
TitleScreen_LogoLine17 = 0x32966
TitleScreen_LogoLine18 = 0x3298A
TitleScreen_LogoLine19 = 0x329AE
TitleScreen_LogoLine20 = 0x329D2
TitleScreen_LogoLine21 = 0x329E9
TitleScreen_LogoLine22 = 0x32A00
TitleScreen_LogoLine23 = 0x32A19
TitleScreen_LogoLine24 = 0x32A34
TitleState_CurtainRaise = 0x309DF
TitleState_OpeningSequence = 0x30AC2
Title_3Glow = 0x30A8D
Title_3GlowColors = 0x30ABA
Title_AddVel_toPos = 0x31081
Title_AnimateMarioLuigi = 0x310D1
Title_ApplyYVel = 0x310BC
Title_ApplyYVelFall = 0x31786
Title_Checkerboard_Floor = 0x326CD
Title_DebugMenu = 0x30CD9
Title_Display_Curtain = 0x309A0
Title_Do1P2PMenu = 0x30C45
Title_DoBuzzyBeatle = 0x3151C
Title_DoEvent = 0x30B91
Title_DoGoomba = 0x314D0
Title_DoKoopaShell = 0x31566
Title_DoLeaf = 0x3146A
Title_DoMushroom = 0x31454
Title_DoNothing = 0x30BAA
Title_DoStarman = 0x31441
Title_DoState = 0x309C4
Title_DrawMarioLuigi = 0x3121B
Title_DrawMenu = 0x30C23
Title_Drop = 0x30BBD
Title_FadeIn_1 = 0x32ABF
Title_FadeIn_2 = 0x32AD3
Title_FadeIn_3 = 0x32AE7
Title_FadeIn_4 = 0x32AFB
Title_InitObjects = 0x30BF4
Title_IntroSkip = 0x30B04
Title_LAS_CmdFF = 0x30B8C
Title_LAS_DoNextEvent = 0x30B6E
Title_LActionScript = 0x30A70
Title_Leaf_XVelAdj = 0x31463
Title_Leaf_XVelLimit = 0x31465
Title_Leaf_YVel = 0x31467
Title_LoadGraphics = 0x30A09
Title_LoadMenu = 0x32B0F
Title_LoadSMB3 = 0x30BAB
Title_Load_Palette = 0x326AA
Title_LogoShakeDown = 0x30BDA
Title_LogoShakeUp = 0x30BD0
Title_MAS_CmdFE = 0x30B67
Title_MAS_CmdFF = 0x30B58
Title_MAS_DoNextEvent = 0x30B36
Title_MActionScript = 0x30A18
Title_MLAccelFactors = 0x30F7A
Title_MLDetermineSpriteVis = 0x31350
Title_MLSpriteVis_BitVal = 0x3134A
Title_MLSpriteVis_XHiOff = 0x3133E
Title_MLSpriteVis_XOff = 0x31344
Title_MLXAccel = 0x30F7D
Title_MLYAccel = 0x310B0
Title_ML_DoGravity = 0x31008
Title_MarioDoPoof = 0x3137E
Title_Menu_1P2PCursorY = 0x30C43
Title_Menu_InitKoopa = 0x317B8
Title_Menu_Koopa_Attr = 0x317D4
Title_Menu_Koopa_Patterns1 = 0x317DA
Title_Menu_Koopa_Patterns2 = 0x317E0
Title_Menu_Koopa_SignalX = 0x317D0
Title_Menu_Koopa_SprYOff = 0x317CD
Title_Menu_Koopa_SpriteRAMOff = 0x3179A
Title_Menu_UpdateKoopa = 0x317E6
Title_Menu_UpdateKoopas = 0x3179E
Title_ObjCommonFinale = 0x316BD
Title_ObjCommonXYVel = 0x316A1
Title_ObjDraw = 0x316FE
Title_ObjInitFlags = 0x30ED3
Title_ObjInitIdx_Time = 0x308B9
Title_ObjInitX = 0x30ED1
Title_ObjMLCarrySprite = 0x30EC1
Title_ObjMLDuckSprite = 0x30EB9
Title_ObjMLSkidSprite = 0x30EBC
Title_ObjMLTailWagSprite = 0x30EBE
Title_ObjMLWalkSprite = 0x30EA9
Title_ObjMarioPowerDown = 0x30EC5
Title_ObjPal = 0x316F8
Title_ObjPatOff = 0x316EC
Title_ObjPatterns = 0x316D8
Title_ObjS1_Goomba = 0x3142B
Title_ObjS1_Leaf = 0x3141A
Title_ObjS1_None = 0x31419
Title_ObjSprRAMOff = 0x316F2
Title_ObjStartX = 0x313F2
Title_ObjXLimit = 0x316B9
Title_ObjXLimitHi = 0x316BB
Title_ObjXVelApply = 0x3177E
Title_Obj_InitIdx = 0x308B3
Title_PalFadeIn = 0x30C07
Title_Poof_Patterns = 0x3137A
Title_PrepForMenu = 0x30C2A
Title_PrepForWorldMap = 0x30CCB
Title_SpriteOffX = 0x31215
Title_SpritePattern = 0x30DE5
Title_SpritePatternIndex = 0x30DC9
Title_SpriteVROMPage = 0x30E8D
Title_SpriteVROMPwrOff = 0x31218
Title_UpdateAllObjs = 0x30ED5
Title_UpdateObj = 0x31431
Title_UpdateObjState1 = 0x313F8
Title_UpdateOtherObjs = 0x313D3
ToadFrame0 = 0x30398
ToadFrame1 = 0x3039E
ToadFrame2 = 0x303A4
ToadHouse_Box_X = 0x3B174
ToadHouse_ChestPressB = 0x3B177
ToadHouse_GiveItem = 0x1DE7
ToadHouse_Item2Inventory = 0x3B14B
ToadHouse_ItemOff = 0x3B15A
ToadHouse_RandomItem = 0x3B164
ToadItem_AttrRight = 0x54DC
ToadItem_PalPerItem = 0x248
ToadItem_PatternLeft = 0x54C2
ToadItem_PatternRight = 0x54CF
ToadMsg_AnchorPWing = 0x53EA
ToadMsg_High = 0x5447
ToadMsg_Low = 0x5444
ToadMsg_Standard = 0x5336
ToadMsg_WarpWhistle = 0x5390
ToadShop_Layouts = 0x192F8
ToadShop_Objects = 0x19308
Toad_AnimDelayByFrame = 0x365F7
Toad_DoToadText = 0x544A
Toad_DrawDiagBox = 0x52CB
Toad_Frame0 = 0x365DB
Toad_Frame1 = 0x365DF
Toad_Frame2 = 0x365E3
Toad_Frame3 = 0x365E7
Toad_FrameOffset = 0x365EB
Toad_FramePatterns = 0x365DB
Toad_Speak = 0x5285
Tornado_DrawParticle = 0x76DE
Tornado_InitXVel = 0x748B
Tornado_MaskOffset = 0x7532
Tornado_Masks = 0x76D6
Tornado_ParticleDivCnt = 0x752A
Tornado_ParticleOffsets = 0x749A
Tornado_PlayerXVelAdj = 0x7631
Tornado_PlayerXVelAdj2 = 0x7633
Tornado_PlayerYAccel = 0x762F
Tornado_PlayerYLimit = 0x762D
Tornado_PlayerYVelAccelByRow = 0x763B
Tornado_ScatterParticle = 0x764B
Tornado_ScatterY = 0x751A
Tornado_SpreadFreq = 0x7522
TreasureBox_Poof = 0x6361
Troopa_Draw = 0x94EC
Troopa_FootByEvenOddFrame = 0x94B2
Troopa_YOffByFrame = 0x94B0
TwirlShell_Draw = 0x6611
Twirler_InitXVel = 0x6578
TwisterSpin_FlipBits = 0x10ED1
Unused11O = 0xCAE8
Unused12 = 0xCB46
Unused13 = 0xCB54
Unused1O = 0xC1BB
Unused2O = 0xC231
Unused2_ExitO = 0xC58F
Unused3O = 0xC6EC
Unused3_CHO = 0xCFDD
Unused4_InnerO = 0xC7B7
Unused5O = 0xD411
Unused7O = 0xCCB0
Unused7_EndO = 0xCCE5
Unused8_EndO = 0xD165
Unused9O = 0xCECD
UnusedMapObj_Layout = 0x19376
UnusedMapObj_Objects = 0x19378
UpdSel_32PixPart = 0x3F692
UpdSel_Roulette = 0x2CEDF
UpdSel_Title = 0x3F709
UpdSel_Vertical = 0x3F5A9
VBlank_Wait_Loop = 0x3FF5E
VBorder_Offset = 0x14B91
VScroll_CalcAttributeVRAMAddr = 0x3DDA9
VScroll_CalcPatternVRAMAddr = 0x3DD24
VScroll_DoPatternAndAttrRow = 0x3DD4F
VScroll_PageAndDoPatAttrRow = 0x3DBD0
VScroll_TileQuads2Attrs = 0x3DE34
Vector_Table = 0x4000A
VertLevel_ScreenH = 0x3FC96
VertLevel_ScreenL = 0x3FCA6
VibrationOffset = 0x104A9
Video_3CMAppear1 = 0x3C2A6
Video_3CMAppear2 = 0x3C31D
Video_3CMAppear3 = 0x3C2B7
Video_3CMAppear4 = 0x3C2C2
Video_3CMAppear5 = 0x3C2CD
Video_3CMAppear6 = 0x3C2EA
Video_3CMAppear7 = 0x3C307
Video_3CMFlowDiag = 0x1F82
Video_3CMFlowMid = 0x1FA3
Video_3CMFlowStem = 0x1FC4
Video_3CMFlowTop = 0x1F61
Video_3CMMushBot = 0x1F42
Video_3CMMushLeft = 0x1EFB
Video_3CMMushMid = 0x1F18
Video_3CMMushRight = 0x1F2D
Video_3CMMushTop = 0x1EE2
Video_3CMStarBot1 = 0x3C278
Video_3CMStarBot2 = 0x3C28D
Video_3CMStarSide = 0x3C25F
Video_3CMStarTip = 0x3C246
Video_3CMStarTop = 0x3C229
Video_Blackout = 0x3C318
Video_CourseClear = 0x3C35A
Video_DoGameOver00 = 0x140E1
Video_DoGameOver80 = 0x14140
Video_DoStatusBar = 0x3C12B
Video_DoStatusBarHM = 0x3C1AA
Video_DoStatusBarV = 0x3C0AC
Video_DoW2WZ = 0x141CB
Video_DoWXLuigi00 = 0x14203
Video_DoWXLuigi80 = 0x14258
Video_DoWXMario00 = 0x14010
Video_DoWXMario80 = 0x14065
Video_Do_Update = 0x3D4DD
Video_Misc_Updates = 0x352A3
Video_Misc_Updates2 = 0x30869
Video_NSpadeBG = 0x2DDDD
Video_RoulBordAttr = 0x2CFD3
Video_Upd_Table = 0x3C04E
Video_Upd_Table2 = 0x325EA
Video_Upd_TableRoulette = 0x2CFC5
Video_YouGotCard = 0x3C36C
Video_YouGotCardH = 0x3C32E
Vine_NTHigh = 0x2C70
Vine_Set8x8s = 0x2CA5
VsES_FFlyAll = 0x12B7F
VsES_FFlySStep = 0x12B8E
VsES_SStepAll = 0x12B93
VsES_SpinyAll = 0x12B7A
VsES_SpinyFFly = 0x12B84
VsES_SpinySStep = 0x12B89
VsPlayer_BounceOffOtherDizzy = 0x12A9F
VsPlayer_BounceOffOtherSnd = 0x12AA2
VsPlayer_BounceOffOtherXVel = 0x12A98
VsPlayer_BounceOffOtherYVel = 0x12A9C
VsPlayer_BumpEachother = 0x12A5E
VsPlayer_CalcBoundBox = 0x12B40
VsPlayer_CalcObjBoundBox = 0x13337
VsPlayer_Climbing = 0x126EC
VsPlayer_Dying = 0x1278B
VsPlayer_Init = 0x12469
VsPlayer_Normal = 0x124B8
Vs_2PVsInit = 0x12210
Vs_2PVsPauseHandler = 0x121B7
Vs_2PVsRun = 0x122D1
Vs_5EnemySets = 0x12B7A
Vs_ApplyXVel = 0x138A0
Vs_ApplyYVel = 0x138AC
Vs_Battlefields = 0x1C4AE
Vs_BottomPipeEntranceX = 0x12F0E
Vs_BottomPipeLimitX = 0x12F10
Vs_BumpBlockApplyYVel = 0x13888
Vs_BumpBlockUpdateAndDraw = 0x137CE
Vs_BumpBlocksUpdateAndDraw = 0x13799
Vs_CardAwardLives = 0x12028
Vs_CardFlower = 0x12DD5
Vs_CardMushroomOrStar = 0x12DDA
Vs_CardStompYVel = 0x12390
Vs_CardWeight = 0x1204A
Vs_CheckBoxCollision = 0x12B5D
Vs_Coin = 0x12E3F
Vs_CoinLayoutX = 0x122AA
Vs_CoinLayoutY = 0x122AF
Vs_CoinPats = 0x12E33
Vs_CoinTwinklePats = 0x12E37
Vs_CollideAgainstWorld = 0x127E2
Vs_DetStatLoop = 0x127D6
Vs_DetStatLoopUnderHit = 0x127DC
Vs_DetStatLoopXOff = 0x127C4
Vs_DetStatLoopYOff = 0x127D0
Vs_DoNothing = 0x1224C
Vs_DrawPlayerCards = 0x12116
Vs_DrawxUps = 0x120AC
Vs_EndGame = 0x133E0
Vs_EnemyHandleHitUnder = 0x13045
Vs_EnemyMove = 0x12F14
Vs_EnemyPipeXVel = 0x12F12
Vs_EnemySetByGameType = 0x12204
Vs_EnemyUsePipes = 0x12F8F
Vs_EraseBlockHitTile = 0x13828
Vs_EraseLastTile = 0x1295D
Vs_FighterFly = 0x12F2E
Vs_FireballBaseAnimCnt = 0x13117
Vs_FireballXLimit = 0x136C8
Vs_FireballYVelAccel = 0x136C4
Vs_FireballYVelLimit = 0x136C6
Vs_Fireballs = 0x136CA
Vs_FountainSpray = 0x12338
Vs_GameCoins = 0x1231A
Vs_GameFountain = 0x1230C
Vs_GameLadder = 0x12305
Vs_GameTypical = 0x122F5
Vs_GetPlayerDirFromObj = 0x1332B
Vs_GetTileAndCheckSolid = 0x129A6
Vs_GrabCardObject = 0x134D0
Vs_HFireballX = 0x12BA0
Vs_HFireballXVel = 0x12BA2
Vs_HiddenCoinX = 0x1224D
Vs_HiddenCoinY = 0x1225B
Vs_HideCoins = 0x12269
Vs_KickedBlock = 0x12DB1
Vs_LayoutCoins = 0x122B4
Vs_LivesReward = 0x12010
Vs_NewEnemy = 0x12C9E
Vs_ObjPatterns = 0x130F5
Vs_ObjStateDying = 0x135A9
Vs_ObjStateFlippedOver = 0x13511
Vs_ObjStateNormal = 0x12D4B
Vs_ObjectApplyXVel = 0x1388E
Vs_ObjectApplyYVel = 0x13894
Vs_ObjectBasePatIndex = 0x130DD
Vs_ObjectBumpOffEachother = 0x13275
Vs_ObjectBumpOffOthers = 0x1325D
Vs_ObjectCollideAgainstWorld = 0x129C5
Vs_ObjectDestroy = 0x135BF
Vs_ObjectDoState = 0x12D2B
Vs_ObjectDraw = 0x13127
Vs_ObjectDrawStyle = 0x130E9
Vs_ObjectGetUpXVel = 0x1350B
Vs_ObjectHitFloor = 0x13037
Vs_ObjectKickedXVel = 0x1335D
Vs_ObjectMoveAndFall = 0x13593
Vs_ObjectMoveFaster = 0x13532
Vs_ObjectPalette = 0x130D1
Vs_ObjectPlayerCollide = 0x1335F
Vs_ObjectReverseXVel = 0x1331F
Vs_ObjectSprRAMOffsets = 0x12CEB
Vs_ObjectsUpdateAndDraw = 0x12D06
Vs_POWHeight = 0x135EF
Vs_POWPatterns = 0x135E9
Vs_POWVertShakes = 0x135E5
Vs_PatCoin = 0x13105
Vs_PatFBall = 0x130FD
Vs_PatFBall2 = 0x1310F
Vs_PatFFly = 0x13101
Vs_PatFlower = 0x13109
Vs_PatKBlock = 0x1310D
Vs_PatMush = 0x13107
Vs_PatSStep = 0x130F9
Vs_PatSpiny = 0x130F5
Vs_PatSpinyFlip = 0x13113
Vs_PatStar = 0x1310B
Vs_PlayerAddCoinToStatus = 0x12191
Vs_PlayerBlockXVel = 0x124AA
Vs_PlayerCoinStatBaseVRAM = 0x1210E
Vs_PlayerCoinStatCmds = 0x12109
Vs_PlayerDoMovements = 0x1242A
Vs_PlayerDraw = 0x129E3
Vs_PlayerFlipBits = 0x129E1
Vs_PlayerInitDir = 0x12467
Vs_PlayerInitX = 0x12463
Vs_PlayerJumpHeightBySpd = 0x125AF
Vs_PlayerMove = 0x12401
Vs_PlayerPopCard = 0x12393
Vs_PlayerSprPats = 0x129D1
Vs_PlayerWallEnable = 0x124AC
Vs_PrepareNewObject = 0x12C2A
Vs_Randomize = 0x138E0
Vs_RestoreBlockHitTile = 0x13815
Vs_SpawnEnemies = 0x12BA4
Vs_SpawnEnemyStartX = 0x12B98
Vs_SpawnEnemyStartXVel = 0x12B9A
Vs_SpawnEnemyStartXVelFast = 0x12B9D
Vs_SpawnFighterFly = 0x12C90
Vs_SpawnSidestepper = 0x12C74
Vs_SpawnSpiny = 0x12C9C
Vs_SpinyAndSidesteppers = 0x12FC9
Vs_StatusBarTiles = 0x1C5EE
Vs_UNKGAME = 0x12324
Vs_UpdateAndDrawPOW = 0x135F2
Vs_WalkCntRate = 0x124A2
Vs_xUpDraw = 0x120B7
Vs_xUpInitCount = 0x12020
Vs_xUpPattern_L = 0x120A0
Vs_xUpPattern_R = 0x120A5
Vs_xUpSndTable = 0x12018
Vs_xUp_SprRAMOff = 0x120AA
W101O = 0xC537
W102O = 0xC6BA
W103O = 0xC2FE
W103_CoinHeavO = 0xCFE2
W104O = 0xCC43
W104_EndO = 0xC594
W105O = 0xC93B
W105_CoinHeavO = 0xD03C
W106O = 0xCC1D
W10FO = 0xD33B
W10F_AltO = 0xD361
W1AirshipO = 0xD6FA
W1Airship_BossO = 0xD9F6
W1HBO = 0xC73B
W1K_Pat = 0x3653B
W1UNO = 0xCBF5
W1_Map_Layout = 0x185BA
W1_Obj_IDs = 0x16148
W1_Obj_Items = 0x16190
W1_Obj_XHis = 0x160B8
W1_Obj_XLos = 0x16100
W1_Obj_Ys = 0x16070
W201O = 0xD199
W202O = 0xC8A5
W202_ExitO = 0xD2B5
W203O = 0xD200
W203_ExitO = 0xD2BA
W204O = 0xD26F
W205O = 0xD1D1
W20FO = 0xD232
W20F_AltO = 0xD264
W2AirshipO = 0xD72C
W2Airship_BossO = 0xD9FB
W2FBO = 0xD152
W2HBO = 0xD15D
W2K_Pat = 0x36547
W2P1O = 0xC153
W2PYO = 0xD107
W2PY_InsideO = 0xC5CC
W2STO = 0xC853
W2UNO = 0xC012
W2_Map_Layout = 0x1864B
W2_Obj_IDs = 0x16151
W2_Obj_Items = 0x16199
W2_Obj_XHis = 0x160C1
W2_Obj_XLos = 0x16109
W2_Obj_Ys = 0x16079
W301O = 0xCE35
W301_ExitO = 0xC4BF
W302O = 0xCA33
W302_EndO = 0xCA68
W303O = 0xC59E
W303_EndO = 0xC5C7
W304O = 0xC9C3
W305O = 0xCDFB
W305_EndO = 0xC32D
W306O = 0xCA73
W306_EndO = 0xCA96
W307O = 0xC47E
W307_CoinHeavO = 0xCE99
W308O = 0xC964
W308_EndO = 0xC56B
W309O = 0xC39F
W309_EndO = 0xC3E3
W3AirshipO = 0xD767
W3Airship_BossO = 0xDA00
W3F1O = 0xD3A3
W3F1_AltO = 0xD3D5
W3F2O = 0xD372
W3F2_AltO = 0xD38F
W3HBO = 0xC650
W3K_Pat = 0x36553
W3P1O = 0xC158
W3P2O = 0xC15D
W3P3O = 0xC162
W3_Map_Layout = 0x1876C
W3_Obj_IDs = 0x1615A
W3_Obj_Items = 0x161A2
W3_Obj_XHis = 0x160CA
W3_Obj_XLos = 0x16112
W3_Obj_Ys = 0x16082
W401O = 0xCEA7
W401_BonusO = 0xD041
W402O = 0xCF24
W403O = 0xCF1C
W403_UnderO = 0xC873
W404O = 0xCFC9
W405O = 0xCFFC
W405_BonusO = 0xC335
W40F_BonusO = 0xC978
W46AO = 0xC267
W46BO = 0xC1EA
W4AirshipO = 0xD7A8
W4Airship_BossO = 0xD6B1
W4F1O = 0xD538
W4F2O = 0xD518
W4F2_BonusO = 0xCE27
W4HBO = 0xD0FA
W4K_Pat = 0x3655F
W4P1O = 0xC167
W4P2O = 0xC16C
W4_Map_Layout = 0x1891D
W4_Obj_IDs = 0x16163
W4_Obj_Items = 0x161AB
W4_Obj_XHis = 0x160D3
W4_Obj_XLos = 0x1611B
W4_Obj_Ys = 0x1608B
W501O = 0xC3F2
W501_BonusO = 0xC424
W502O = 0xC8CE
W502_UnderO = 0xCE5B
W503O = 0xC016
W503_EndO = 0xC570
W503_InsideO = 0xC2AE
W504O = 0xD049
W504_EndO = 0xCE91
W505O = 0xCB1A
W506O = 0xCF50
W507O = 0xCEED
W507_LowerO = 0xC176
W508O = 0xCE80
W509O = 0xCF8B
W5AirshipO = 0xD6B6
W5Airship_BossO = 0xDA05
W5F1O = 0xD3E0
W5F1_AltO = 0xD40C
W5F2O = 0xD2C4
W5F2_AltO = 0xD2C9
W5K_Pat = 0x3656B
W5P1O = 0xC171
W5T1O = 0xD4FC
W5T2O = 0xCFF4
W5T3O = 0xD510
W5T4O = 0xCFEC
W5TDO = 0xCFE7
W5_Map_Layout = 0x18A3E
W5_Obj_IDs = 0x1616C
W5_Obj_Items = 0x161B4
W5_Obj_XHis = 0x160DC
W5_Obj_XLos = 0x16124
W5_Obj_Ys = 0x16094
W601O = 0xCBD2
W602O = 0xCB62
W602_EndO = 0xCC18
W603O = 0xCA9E
W604O = 0xCC7B
W604_CoinHeavO = 0xD037
W605O = 0xCAE3
W605_UnderO = 0xC5FB
W606O = 0xCCF0
W606_UnderO = 0xC65B
W607O = 0xCD27
W607_EndO = 0xCD35
W608O = 0xC9F5
W609O = 0xCD3D
W609_UnderO = 0xC61E
W610O = 0xCCF8
W6AirshipO = 0xD7F5
W6Airship_BossO = 0xDA0A
W6Crown_Frames = 0x304DE
W6F1O = 0xD4C0
W6F1_AltO = 0xD4E3
W6F2O = 0xCABB
W6F2_AltO = 0xCADE
W6F3O = 0xD480
W6F3_AltO = 0xD4B2
W6K_Pat = 0x36577
W6P1O = 0xC149
W6P2O = 0xC14E
W6Seal_Frames = 0x304D6
W6_Map_Layout = 0x18B5F
W6_Obj_IDs = 0x16175
W6_Obj_Items = 0x161BD
W6_Obj_XHis = 0x160E5
W6_Obj_XLos = 0x1612D
W6_Obj_Ys = 0x1609D
W701O = 0xC25D
W701_MazeO = 0xCDA3
W702O = 0xD114
W703O = 0xC703
W704O = 0xC3ED
W704_WaterO = 0xCDD2
W705O = 0xC566
W705_UnderO = 0xC181
W706O = 0xC3E8
W706_MazeO = 0xCE66
W707O = 0xC262
W707_MainO = 0xCD45
W708O = 0xC33D
W708_CoinHeavO = 0xD102
W709O = 0xD16A
W709_EndO = 0xD10F
W7AirshipO = 0xD824
W7Airship_BossO = 0xDA0F
W7F1O = 0xD4F4
W7F2O = 0xD42B
W7F2_AltO = 0xD46C
W7I1O = 0xD08C
W7I1_PrizeO = 0xD0F2
W7I2O = 0xD078
W7I2_PrizeO = 0xD0EA
W7K_Pat = 0x36583
W7P1O = 0xC121
W7P2O = 0xC126
W7P3O = 0xC12B
W7P4O = 0xC130
W7P5O = 0xC135
W7P6O = 0xC13A
W7P7O = 0xC13F
W7P8O = 0xC144
W7_Map_Layout = 0x18D10
W7_Obj_IDs = 0x1617E
W7_Obj_Items = 0x161C6
W7_Obj_XHis = 0x160EE
W7_Obj_XLos = 0x16136
W7_Obj_Ys = 0x160A6
W801O = 0xC434
W802O = 0xC90F
W80FO = 0xD561
W80F_AltO = 0xD599
W8AirshipO = 0xD877
W8Airship_BossO = 0xDA34
W8BCO = 0xD5DD
W8BC_AltO = 0xD61B
W8BSO = 0xD8DC
W8BS_BossO = 0xDA2F
W8D_Calc_RC = 0x154A9
W8D_Calc_TileOff = 0x1550E
W8D_Calc_VRAM_Addr = 0x154EF
W8D_CircSprs = 0x17B0F
W8D_FillGraphicsBuffer = 0x15449
W8D_GetNext8x8 = 0x154BF
W8H1O = 0xD0C0
W8H2O = 0xD0A3
W8H3O = 0xD0D1
W8HBO = 0xC04D
W8H_PrizeO = 0xD0DF
W8P1O = 0xC103
W8P2O = 0xC108
W8P3O = 0xC10D
W8P4O = 0xC112
W8P5O = 0xC117
W8P6O = 0xC11C
W8T1O = 0xD97F
W8T2O = 0xD92C
W8Tank1_BossO = 0xDA39
W8Tank2_BossO = 0xDA2A
W8_Map_Layout = 0x18E31
W8_Obj_IDs = 0x16187
W8_Obj_Items = 0x161CF
W8_Obj_XHis = 0x160F7
W8_Obj_XLos = 0x1613F
W8_Obj_Ys = 0x160AF
W9_Map_Layout = 0x19072
WAirship_IntroO = 0xD6AC
WWFX_WarpDoWind = 0x163A9
WWFX_WarpIslandInit = 0x163F7
WWFX_WarpLanding = 0x1642F
WWFX_WarpWhistleFlash = 0x16396
WWFX_WarpWhistleInit = 0x16350
WandAndToad_DrawAndUpdate = 0x36603
Wand_Attributes = 0xF201
Wand_FrameAdd = 0x365AD
Wand_FrameChangeMask = 0x365AB
Wand_Offset_BySuit = 0x3B414
Wand_Pattern1 = 0xF1F1
Wand_Pattern2 = 0xF1F9
Wand_SpinAttrs = 0x3659B
Wand_SpinPats = 0x3658F
Wand_XOff_BySuitL = 0x3B3FF
Wand_XOff_BySuitR = 0x3B406
Wand_YOff_BySuit = 0x3B40D
WarpWhistle_Flash = 0x1633B
WarpWindHandTrap_Draw = 0x16530
Wendy_RingXVel = 0x37FA
WoodenPlat_NextTileIdx = 0x4BD7
WoodenPlat_PathTiles = 0x4BE7
WoodenPlat_PathTilesAlt = 0x4BF7
WoodenPlat_ScanIndices = 0x4BCA
WoodenPlat_ScanIndices_End = 0x4BD7
WoodenPlat_XVel = 0x4C18
WoodenPlat_YVel = 0x4C07
WoodenPlatform_XVel = 0xBBD0
World5_Sky_AddCloudDeco = 0x144F0
World5_Sky_CloudDeco = 0x144E0
WorldIntro_BoxTimer = 0x14516
WorldIntro_BoxTimer_NoSym = 0x14519
WorldIntro_CompleteStars = 0x146D4
WorldIntro_EraseAndStars = 0x1455A
WorldMap_Loop = 0x3C6EE
WorldMap_UpdateAndDraw = 0x14F39
WorldMap_UpdateAndDrawInd = 0x16301
World_BGM = 0x3C424
World_BGM_Arrival = 0x143CA
World_Map_Max_PanR = 0x14F44
Wrench_Attributes = 0xF4A6
Wrench_Patterns = 0xF4A2
XUp_NumPats = 0x5D55
XUp_Patterns = 0x5D4F
XUp_XOff = 0x5D52
