"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
describe('UserPoolIdentityProvider', () => {
    describe('facebook', () => {
        test('defaults', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
            });
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                ProviderName: 'Facebook',
                ProviderType: 'Facebook',
                ProviderDetails: {
                    client_id: 'fb-client-id',
                    client_secret: 'fb-client-secret',
                    authorize_scopes: 'public_profile',
                },
            });
        });
        test('scopes & api_version', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
                scopes: ['scope1', 'scope2'],
                apiVersion: 'version1',
            });
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                ProviderName: 'Facebook',
                ProviderType: 'Facebook',
                ProviderDetails: {
                    client_id: 'fb-client-id',
                    client_secret: 'fb-client-secret',
                    authorize_scopes: 'scope1,scope2',
                    api_version: 'version1',
                },
            });
        });
        test('registered with user pool', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            const provider = new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
            });
            // THEN
            expect(pool.identityProviders).toContain(provider);
        });
        test('attribute mapping', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderFacebook(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'fb-client-id',
                clientSecret: 'fb-client-secret',
                attributeMapping: {
                    givenName: lib_1.ProviderAttribute.FACEBOOK_NAME,
                    address: lib_1.ProviderAttribute.other('fb-address'),
                    custom: {
                        customAttr1: lib_1.ProviderAttribute.FACEBOOK_EMAIL,
                        customAttr2: lib_1.ProviderAttribute.other('fb-custom-attr'),
                    },
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                AttributeMapping: {
                    given_name: 'name',
                    address: 'fb-address',
                    customAttr1: 'email',
                    customAttr2: 'fb-custom-attr',
                },
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmFjZWJvb2sudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImZhY2Vib29rLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxnQ0FBOEI7QUFDOUIsd0NBQXNDO0FBQ3RDLG1DQUEwRjtBQUUxRixRQUFRLENBQUMsMEJBQTBCLEVBQUUsR0FBRyxFQUFFO0lBQ3hDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsR0FBRyxFQUFFO1FBQ3hCLElBQUksQ0FBQyxVQUFVLEVBQUUsR0FBRyxFQUFFO1lBQ3BCLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBQzFCLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztZQUU3QyxPQUFPO1lBQ1AsSUFBSSxzQ0FBZ0MsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO2dCQUN6RCxRQUFRLEVBQUUsSUFBSTtnQkFDZCxRQUFRLEVBQUUsY0FBYztnQkFDeEIsWUFBWSxFQUFFLGtCQUFrQjthQUNqQyxDQUFDLENBQUM7WUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLHdDQUF3QyxFQUFFO2dCQUNyRSxZQUFZLEVBQUUsVUFBVTtnQkFDeEIsWUFBWSxFQUFFLFVBQVU7Z0JBQ3hCLGVBQWUsRUFBRTtvQkFDZixTQUFTLEVBQUUsY0FBYztvQkFDekIsYUFBYSxFQUFFLGtCQUFrQjtvQkFDakMsZ0JBQWdCLEVBQUUsZ0JBQWdCO2lCQUNuQzthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHNCQUFzQixFQUFFLEdBQUcsRUFBRTtZQUNoQyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFFN0MsT0FBTztZQUNQLElBQUksc0NBQWdDLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtnQkFDekQsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsUUFBUSxFQUFFLGNBQWM7Z0JBQ3hCLFlBQVksRUFBRSxrQkFBa0I7Z0JBQ2hDLE1BQU0sRUFBRSxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUM7Z0JBQzVCLFVBQVUsRUFBRSxVQUFVO2FBQ3ZCLENBQUMsQ0FBQztZQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsd0NBQXdDLEVBQUU7Z0JBQ3JFLFlBQVksRUFBRSxVQUFVO2dCQUN4QixZQUFZLEVBQUUsVUFBVTtnQkFDeEIsZUFBZSxFQUFFO29CQUNmLFNBQVMsRUFBRSxjQUFjO29CQUN6QixhQUFhLEVBQUUsa0JBQWtCO29CQUNqQyxnQkFBZ0IsRUFBRSxlQUFlO29CQUNqQyxXQUFXLEVBQUUsVUFBVTtpQkFDeEI7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywyQkFBMkIsRUFBRSxHQUFHLEVBQUU7WUFDckMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBRTdDLE9BQU87WUFDUCxNQUFNLFFBQVEsR0FBRyxJQUFJLHNDQUFnQyxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7Z0JBQzFFLFFBQVEsRUFBRSxJQUFJO2dCQUNkLFFBQVEsRUFBRSxjQUFjO2dCQUN4QixZQUFZLEVBQUUsa0JBQWtCO2FBQ2pDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3JELENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG1CQUFtQixFQUFFLEdBQUcsRUFBRTtZQUM3QixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFFN0MsT0FBTztZQUNQLElBQUksc0NBQWdDLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRTtnQkFDekQsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsUUFBUSxFQUFFLGNBQWM7Z0JBQ3hCLFlBQVksRUFBRSxrQkFBa0I7Z0JBQ2hDLGdCQUFnQixFQUFFO29CQUNoQixTQUFTLEVBQUUsdUJBQWlCLENBQUMsYUFBYTtvQkFDMUMsT0FBTyxFQUFFLHVCQUFpQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUM7b0JBQzlDLE1BQU0sRUFBRTt3QkFDTixXQUFXLEVBQUUsdUJBQWlCLENBQUMsY0FBYzt3QkFDN0MsV0FBVyxFQUFFLHVCQUFpQixDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztxQkFDdkQ7aUJBQ0Y7YUFDRixDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyx3Q0FBd0MsRUFBRTtnQkFDckUsZ0JBQWdCLEVBQUU7b0JBQ2hCLFVBQVUsRUFBRSxNQUFNO29CQUNsQixPQUFPLEVBQUUsWUFBWTtvQkFDckIsV0FBVyxFQUFFLE9BQU87b0JBQ3BCLFdBQVcsRUFBRSxnQkFBZ0I7aUJBQzlCO2FBQ0YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgUHJvdmlkZXJBdHRyaWJ1dGUsIFVzZXJQb29sLCBVc2VyUG9vbElkZW50aXR5UHJvdmlkZXJGYWNlYm9vayB9IGZyb20gJy4uLy4uL2xpYic7XG5cbmRlc2NyaWJlKCdVc2VyUG9vbElkZW50aXR5UHJvdmlkZXInLCAoKSA9PiB7XG4gIGRlc2NyaWJlKCdmYWNlYm9vaycsICgpID0+IHtcbiAgICB0ZXN0KCdkZWZhdWx0cycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ3VzZXJwb29sJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBVc2VyUG9vbElkZW50aXR5UHJvdmlkZXJGYWNlYm9vayhzdGFjaywgJ3VzZXJwb29saWRwJywge1xuICAgICAgICB1c2VyUG9vbDogcG9vbCxcbiAgICAgICAgY2xpZW50SWQ6ICdmYi1jbGllbnQtaWQnLFxuICAgICAgICBjbGllbnRTZWNyZXQ6ICdmYi1jbGllbnQtc2VjcmV0JyxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sSWRlbnRpdHlQcm92aWRlcicsIHtcbiAgICAgICAgUHJvdmlkZXJOYW1lOiAnRmFjZWJvb2snLFxuICAgICAgICBQcm92aWRlclR5cGU6ICdGYWNlYm9vaycsXG4gICAgICAgIFByb3ZpZGVyRGV0YWlsczoge1xuICAgICAgICAgIGNsaWVudF9pZDogJ2ZiLWNsaWVudC1pZCcsXG4gICAgICAgICAgY2xpZW50X3NlY3JldDogJ2ZiLWNsaWVudC1zZWNyZXQnLFxuICAgICAgICAgIGF1dGhvcml6ZV9zY29wZXM6ICdwdWJsaWNfcHJvZmlsZScsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3Njb3BlcyAmIGFwaV92ZXJzaW9uJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAndXNlcnBvb2wnKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgbmV3IFVzZXJQb29sSWRlbnRpdHlQcm92aWRlckZhY2Vib29rKHN0YWNrLCAndXNlcnBvb2xpZHAnLCB7XG4gICAgICAgIHVzZXJQb29sOiBwb29sLFxuICAgICAgICBjbGllbnRJZDogJ2ZiLWNsaWVudC1pZCcsXG4gICAgICAgIGNsaWVudFNlY3JldDogJ2ZiLWNsaWVudC1zZWNyZXQnLFxuICAgICAgICBzY29wZXM6IFsnc2NvcGUxJywgJ3Njb3BlMiddLFxuICAgICAgICBhcGlWZXJzaW9uOiAndmVyc2lvbjEnLFxuICAgICAgfSk7XG5cbiAgICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xJZGVudGl0eVByb3ZpZGVyJywge1xuICAgICAgICBQcm92aWRlck5hbWU6ICdGYWNlYm9vaycsXG4gICAgICAgIFByb3ZpZGVyVHlwZTogJ0ZhY2Vib29rJyxcbiAgICAgICAgUHJvdmlkZXJEZXRhaWxzOiB7XG4gICAgICAgICAgY2xpZW50X2lkOiAnZmItY2xpZW50LWlkJyxcbiAgICAgICAgICBjbGllbnRfc2VjcmV0OiAnZmItY2xpZW50LXNlY3JldCcsXG4gICAgICAgICAgYXV0aG9yaXplX3Njb3BlczogJ3Njb3BlMSxzY29wZTInLFxuICAgICAgICAgIGFwaV92ZXJzaW9uOiAndmVyc2lvbjEnLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdyZWdpc3RlcmVkIHdpdGggdXNlciBwb29sJywgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAndXNlcnBvb2wnKTtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgcHJvdmlkZXIgPSBuZXcgVXNlclBvb2xJZGVudGl0eVByb3ZpZGVyRmFjZWJvb2soc3RhY2ssICd1c2VycG9vbGlkcCcsIHtcbiAgICAgICAgdXNlclBvb2w6IHBvb2wsXG4gICAgICAgIGNsaWVudElkOiAnZmItY2xpZW50LWlkJyxcbiAgICAgICAgY2xpZW50U2VjcmV0OiAnZmItY2xpZW50LXNlY3JldCcsXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHBvb2wuaWRlbnRpdHlQcm92aWRlcnMpLnRvQ29udGFpbihwcm92aWRlcik7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhdHRyaWJ1dGUgbWFwcGluZycsICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ3VzZXJwb29sJyk7XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIG5ldyBVc2VyUG9vbElkZW50aXR5UHJvdmlkZXJGYWNlYm9vayhzdGFjaywgJ3VzZXJwb29saWRwJywge1xuICAgICAgICB1c2VyUG9vbDogcG9vbCxcbiAgICAgICAgY2xpZW50SWQ6ICdmYi1jbGllbnQtaWQnLFxuICAgICAgICBjbGllbnRTZWNyZXQ6ICdmYi1jbGllbnQtc2VjcmV0JyxcbiAgICAgICAgYXR0cmlidXRlTWFwcGluZzoge1xuICAgICAgICAgIGdpdmVuTmFtZTogUHJvdmlkZXJBdHRyaWJ1dGUuRkFDRUJPT0tfTkFNRSxcbiAgICAgICAgICBhZGRyZXNzOiBQcm92aWRlckF0dHJpYnV0ZS5vdGhlcignZmItYWRkcmVzcycpLFxuICAgICAgICAgIGN1c3RvbToge1xuICAgICAgICAgICAgY3VzdG9tQXR0cjE6IFByb3ZpZGVyQXR0cmlidXRlLkZBQ0VCT09LX0VNQUlMLFxuICAgICAgICAgICAgY3VzdG9tQXR0cjI6IFByb3ZpZGVyQXR0cmlidXRlLm90aGVyKCdmYi1jdXN0b20tYXR0cicpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpDb2duaXRvOjpVc2VyUG9vbElkZW50aXR5UHJvdmlkZXInLCB7XG4gICAgICAgIEF0dHJpYnV0ZU1hcHBpbmc6IHtcbiAgICAgICAgICBnaXZlbl9uYW1lOiAnbmFtZScsXG4gICAgICAgICAgYWRkcmVzczogJ2ZiLWFkZHJlc3MnLFxuICAgICAgICAgIGN1c3RvbUF0dHIxOiAnZW1haWwnLFxuICAgICAgICAgIGN1c3RvbUF0dHIyOiAnZmItY3VzdG9tLWF0dHInLFxuICAgICAgICB9LFxuICAgICAgfSk7XG4gICAgfSk7XG4gIH0pO1xufSk7Il19