"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
describe('UserPoolIdentityProvider', () => {
    describe('amazon', () => {
        test('defaults', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderAmazon(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'amzn-client-id',
                clientSecret: 'amzn-client-secret',
            });
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                ProviderName: 'LoginWithAmazon',
                ProviderType: 'LoginWithAmazon',
                ProviderDetails: {
                    client_id: 'amzn-client-id',
                    client_secret: 'amzn-client-secret',
                    authorize_scopes: 'profile',
                },
            });
        });
        test('scopes', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderAmazon(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'amzn-client-id',
                clientSecret: 'amzn-client-secret',
                scopes: ['scope1', 'scope2'],
            });
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                ProviderName: 'LoginWithAmazon',
                ProviderType: 'LoginWithAmazon',
                ProviderDetails: {
                    client_id: 'amzn-client-id',
                    client_secret: 'amzn-client-secret',
                    authorize_scopes: 'scope1 scope2',
                },
            });
        });
        test('registered with user pool', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            const provider = new lib_1.UserPoolIdentityProviderAmazon(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'amzn-client-id',
                clientSecret: 'amzn-client-secret',
            });
            // THEN
            expect(pool.identityProviders).toContain(provider);
        });
        test('attribute mapping', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'userpool');
            // WHEN
            new lib_1.UserPoolIdentityProviderAmazon(stack, 'userpoolidp', {
                userPool: pool,
                clientId: 'amazn-client-id',
                clientSecret: 'amzn-client-secret',
                attributeMapping: {
                    givenName: lib_1.ProviderAttribute.AMAZON_NAME,
                    address: lib_1.ProviderAttribute.other('amzn-address'),
                    custom: {
                        customAttr1: lib_1.ProviderAttribute.AMAZON_EMAIL,
                        customAttr2: lib_1.ProviderAttribute.other('amzn-custom-attr'),
                    },
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::Cognito::UserPoolIdentityProvider', {
                AttributeMapping: {
                    given_name: 'name',
                    address: 'amzn-address',
                    customAttr1: 'email',
                    customAttr2: 'amzn-custom-attr',
                },
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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