"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('custom domain name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            customDomain: {
                domainName: 'test-domain.example.com',
                certificate,
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'test-domain.example.com',
            CustomDomainConfig: {
                CertificateArn: 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8',
            },
        });
    });
    test('cognito domain prefix', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
            UserPoolId: stack.resolve(pool.userPoolId),
            Domain: 'cognito-domain-prefix',
        });
    });
    test('fails when neither cognitoDomain nor customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const certificate = aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:eu-west-1:0123456789:certificate/7ec3e4ac-808a-4649-b805-66ae02346ad8');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
            customDomain: {
                domainName: 'mydomain.com',
                certificate,
            },
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when both cognitoDomain and customDomain are specified', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => new lib_1.UserPoolDomain(stack, 'Domain', {
            userPool: pool,
        })).toThrow(/cognitoDomain or customDomain must be specified/);
    });
    test('fails when domainPrefix has characters outside the allowed charset', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addDomain('Domain1', {
            cognitoDomain: { domainPrefix: 'domain.prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain2', {
            cognitoDomain: { domainPrefix: 'Domain-Prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
        expect(() => pool.addDomain('Domain3', {
            cognitoDomain: { domainPrefix: 'dómäin-prefix' },
        })).toThrow(/lowercase alphabets, numbers and hyphens/);
    });
    test('does not fail when domainPrefix is a token', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const parameter = new core_1.CfnParameter(stack, 'Paraeter');
        expect(() => pool.addDomain('Domain', {
            cognitoDomain: { domainPrefix: parameter.valueAsString },
        })).not.toThrow();
    });
    test('custom resource is added when cloudFrontDomainName property is used', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const domain = pool.addDomain('Domain', {
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        // WHEN
        const cfDomainName = domain.cloudFrontDomainName;
        // THEN
        expect(stack.resolve(cfDomainName)).toEqual({
            'Fn::GetAtt': [
                'PoolDomainCloudFrontDomainName340BF87E',
                'DomainDescription.CloudFrontDistribution',
            ],
        });
        expect(stack).toHaveResource('Custom::UserPoolCloudFrontDomainName');
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [{
                        Action: 'cognito-idp:DescribeUserPoolDomain',
                        Effect: 'Allow',
                        Resource: '*',
                    }],
                Version: '2012-10-17',
            },
        });
    });
    test('cloudFrontDomainName property can be called multiple times', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const domain = pool.addDomain('Domain', {
            cognitoDomain: {
                domainPrefix: 'cognito-domain-prefix',
            },
        });
        const cfDomainNameFirst = domain.cloudFrontDomainName;
        const cfDomainNameSecond = domain.cloudFrontDomainName;
        expect(cfDomainNameSecond).toEqual(cfDomainNameFirst);
    });
    describe('signInUrl', () => {
        test('returns the expected URL', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'Pool');
            const domain = pool.addDomain('Domain', {
                cognitoDomain: {
                    domainPrefix: 'cognito-domain-prefix',
                },
            });
            const client = pool.addClient('Client', {
                oAuth: {
                    callbackUrls: ['https://example.com'],
                },
            });
            // WHEN
            const signInUrl = domain.signInUrl(client, {
                redirectUri: 'https://example.com',
            });
            // THEN
            expect(stack.resolve(signInUrl)).toEqual({
                'Fn::Join': [
                    '', [
                        'https://',
                        { Ref: 'PoolDomainCFC71F56' },
                        '.auth.',
                        { Ref: 'AWS::Region' },
                        '.amazoncognito.com/login?client_id=',
                        { Ref: 'PoolClient8A3E5EB7' },
                        '&response_type=code&redirect_uri=https://example.com',
                    ],
                ],
            });
        });
        test('correctly uses the signInPath', () => {
            // GIVEN
            const stack = new core_1.Stack();
            const pool = new lib_1.UserPool(stack, 'Pool');
            const domain = pool.addDomain('Domain', {
                cognitoDomain: {
                    domainPrefix: 'cognito-domain-prefix',
                },
            });
            const client = pool.addClient('Client', {
                oAuth: {
                    callbackUrls: ['https://example.com'],
                },
            });
            // WHEN
            const signInUrl = domain.signInUrl(client, {
                redirectUri: 'https://example.com',
                signInPath: '/testsignin',
            });
            // THEN
            expect(signInUrl).toMatch(/amazoncognito\.com\/testsignin\?/);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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