"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool Client', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthScopes: ['profile', 'phone', 'email', 'openid', 'aws.cognito.signin.user.admin'],
            CallbackURLs: ['https://example.com'],
            SupportedIdentityProviders: ['COGNITO'],
        });
    });
    test('client name', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        const client1 = new lib_1.UserPoolClient(stack, 'Client1', {
            userPool: pool,
            userPoolClientName: 'myclient',
        });
        const client2 = new lib_1.UserPoolClient(stack, 'Client2', {
            userPool: pool,
        });
        // THEN
        expect(client1.userPoolClientName).toEqual('myclient');
        expect(() => client2.userPoolClientName).toThrow(/available only if specified on the UserPoolClient during initialization/);
    });
    test('import', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const client = lib_1.UserPoolClient.fromUserPoolClientId(stack, 'Client', 'client-id-1');
        // THEN
        expect(client.userPoolClientId).toEqual('client-id-1');
    });
    test('ExplicitAuthFlows is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: assert_1.ABSENT,
        });
    });
    test('ExplicitAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            authFlows: {
                adminUserPassword: true,
                custom: true,
                refreshToken: true,
                userPassword: true,
                userSrp: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ExplicitAuthFlows: [
                'ALLOW_USER_PASSWORD_AUTH',
                'ALLOW_ADMIN_USER_PASSWORD_AUTH',
                'ALLOW_CUSTOM_AUTH',
                'ALLOW_USER_SRP_AUTH',
                'ALLOW_REFRESH_TOKEN_AUTH',
            ],
        });
    });
    test('AllowedOAuthFlows are correctly named', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    implicitCodeGrant: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
    });
    test('callbackUrl defaults are correctly chosen', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addClient('Client1', {
            oAuth: {
                flows: {
                    clientCredentials: true,
                },
            },
        });
        pool.addClient('Client2', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
            },
        });
        pool.addClient('Client3', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['client_credentials'],
            CallbackURLs: assert_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['implicit'],
            CallbackURLs: ['https://example.com'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthFlows: ['code'],
            CallbackURLs: ['https://example.com'],
        });
    });
    test('fails when callbackUrls is empty for codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: { implicitCodeGrant: true },
                callbackUrls: [],
            },
        })).toThrow(/callbackUrl must not be empty/);
        expect(() => pool.addClient('Client3', {
            oAuth: {
                flows: { authorizationCodeGrant: true },
                callbackUrls: [],
            },
        })).toThrow(/callbackUrl must not be empty/);
        expect(() => pool.addClient('Client4', {
            oAuth: {
                flows: { clientCredentials: true },
                callbackUrls: [],
            },
        })).not.toThrow();
    });
    test('fails when clientCredentials OAuth flow is selected along with codeGrant or implicitGrant', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client1', {
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
        expect(() => pool.addClient('Client2', {
            oAuth: {
                flows: {
                    implicitCodeGrant: true,
                    clientCredentials: true,
                },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        })).toThrow(/clientCredentials OAuth flow cannot be selected/);
    });
    test('OAuth scopes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client', {
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [
                    lib_1.OAuthScope.PHONE,
                    lib_1.OAuthScope.EMAIL,
                    lib_1.OAuthScope.OPENID,
                    lib_1.OAuthScope.PROFILE,
                    lib_1.OAuthScope.COGNITO_ADMIN,
                    lib_1.OAuthScope.custom('my-resource-server/my-own-scope'),
                ],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            AllowedOAuthScopes: [
                'phone',
                'email',
                'openid',
                'profile',
                'aws.cognito.signin.user.admin',
                'my-resource-server/my-own-scope',
            ],
        });
    });
    test('OAuthScope - openid is included when email or phone is specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('Client1', {
            userPoolClientName: 'Client1',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PHONE],
            },
        });
        pool.addClient('Client2', {
            userPoolClientName: 'Client2',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.EMAIL],
            },
        });
        pool.addClient('Client3', {
            userPoolClientName: 'Client3',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.PROFILE],
            },
        });
        pool.addClient('Client4', {
            userPoolClientName: 'Client4',
            oAuth: {
                flows: { clientCredentials: true },
                scopes: [lib_1.OAuthScope.COGNITO_ADMIN],
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client1',
            AllowedOAuthScopes: ['phone', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client2',
            AllowedOAuthScopes: ['email', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client3',
            AllowedOAuthScopes: ['profile', 'openid'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
            ClientName: 'Client4',
            AllowedOAuthScopes: ['aws.cognito.signin.user.admin'],
        });
    });
    test('enable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'ENABLED',
        });
    });
    test('disable user existence errors prevention', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
            preventUserExistenceErrors: false,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: 'LEGACY',
        });
    });
    test('user existence errors prevention is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            UserPoolId: stack.resolve(pool.userPoolId),
            PreventUserExistenceErrors: assert_1.ABSENT,
        });
    });
    test('default supportedIdentityProviders', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        const idp = lib_1.UserPoolIdentityProvider.fromProviderName(stack, 'imported', 'userpool-idp');
        pool.registerIdentityProvider(idp);
        // WHEN
        new lib_1.UserPoolClient(stack, 'Client', {
            userPool: pool,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            SupportedIdentityProviders: [
                'userpool-idp',
                'COGNITO',
            ],
        });
    });
    test('supportedIdentityProviders', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('AllEnabled', {
            userPoolClientName: 'AllEnabled',
            supportedIdentityProviders: [
                lib_1.UserPoolClientIdentityProvider.COGNITO,
                lib_1.UserPoolClientIdentityProvider.FACEBOOK,
                lib_1.UserPoolClientIdentityProvider.AMAZON,
            ],
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'AllEnabled',
            SupportedIdentityProviders: ['COGNITO', 'Facebook', 'LoginWithAmazon'],
        });
    });
    test('disableOAuth', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        // WHEN
        pool.addClient('OAuthDisabled', {
            userPoolClientName: 'OAuthDisabled',
            disableOAuth: true,
        });
        pool.addClient('OAuthEnabled', {
            userPoolClientName: 'OAuthEnabled',
            disableOAuth: false,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'OAuthDisabled',
            AllowedOAuthFlows: assert_1.ABSENT,
            AllowedOAuthScopes: assert_1.ABSENT,
            AllowedOAuthFlowsUserPoolClient: false,
        });
        expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
            ClientName: 'OAuthEnabled',
            AllowedOAuthFlows: ['implicit', 'code'],
            AllowedOAuthScopes: ['profile', 'phone', 'email', 'openid', 'aws.cognito.signin.user.admin'],
            AllowedOAuthFlowsUserPoolClient: true,
        });
    });
    test('fails when oAuth is specified but is disableOAuth is set', () => {
        const stack = new core_1.Stack();
        const pool = new lib_1.UserPool(stack, 'Pool');
        expect(() => pool.addClient('Client', {
            disableOAuth: true,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
            },
        })).toThrow(/disableOAuth is set/);
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLWNsaWVudC50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidXNlci1wb29sLWNsaWVudC50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsNENBQXlDO0FBQ3pDLGdDQUE4QjtBQUM5Qix3Q0FBc0M7QUFDdEMsZ0NBQXdIO0FBRXhILFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLEVBQUU7SUFDaEMsSUFBSSxDQUFDLGVBQWUsRUFBRSxHQUFHLEVBQUU7UUFDekIsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLE9BQU87UUFDUCxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNsQyxRQUFRLEVBQUUsSUFBSTtTQUNmLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLDhCQUE4QixFQUFFO1lBQzNELFVBQVUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7WUFDMUMsaUJBQWlCLEVBQUUsQ0FBQyxVQUFVLEVBQUUsTUFBTSxDQUFDO1lBQ3ZDLGtCQUFrQixFQUFFLENBQUMsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLCtCQUErQixDQUFDO1lBQzVGLFlBQVksRUFBRSxDQUFDLHFCQUFxQixDQUFDO1lBQ3JDLDBCQUEwQixFQUFFLENBQUMsU0FBUyxDQUFDO1NBQ3hDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGFBQWEsRUFBRSxHQUFHLEVBQUU7UUFDdkIsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLE9BQU87UUFDUCxNQUFNLE9BQU8sR0FBRyxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRTtZQUNuRCxRQUFRLEVBQUUsSUFBSTtZQUNkLGtCQUFrQixFQUFFLFVBQVU7U0FDL0IsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxPQUFPLEdBQUcsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7WUFDbkQsUUFBUSxFQUFFLElBQUk7U0FDZixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN2RCxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsT0FBTyxDQUFDLGtCQUFrQixDQUFDLENBQUMsT0FBTyxDQUFDLHlFQUF5RSxDQUFDLENBQUM7SUFDOUgsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsUUFBUSxFQUFFLEdBQUcsRUFBRTtRQUNsQixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsb0JBQWMsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRW5GLE9BQU87UUFDUCxNQUFNLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3pELENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtRQUNsRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFekIsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw4QkFBOEIsRUFBRTtZQUMvRCxpQkFBaUIsRUFBRSxlQUFNO1NBQzFCLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtRQUNqRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFO1lBQ3ZCLFNBQVMsRUFBRTtnQkFDVCxpQkFBaUIsRUFBRSxJQUFJO2dCQUN2QixNQUFNLEVBQUUsSUFBSTtnQkFDWixZQUFZLEVBQUUsSUFBSTtnQkFDbEIsWUFBWSxFQUFFLElBQUk7Z0JBQ2xCLE9BQU8sRUFBRSxJQUFJO2FBQ2Q7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsOEJBQThCLEVBQUU7WUFDL0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLDBCQUEwQjtnQkFDMUIsZ0NBQWdDO2dCQUNoQyxtQkFBbUI7Z0JBQ25CLHFCQUFxQjtnQkFDckIsMEJBQTBCO2FBQzNCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsdUNBQXVDLEVBQUUsR0FBRyxFQUFFO1FBQ2pELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUV6QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUU7WUFDeEIsS0FBSyxFQUFFO2dCQUNMLEtBQUssRUFBRTtvQkFDTCxzQkFBc0IsRUFBRSxJQUFJO29CQUM1QixpQkFBaUIsRUFBRSxJQUFJO2lCQUN4QjtnQkFDRCxNQUFNLEVBQUUsQ0FBQyxnQkFBVSxDQUFDLEtBQUssQ0FBQzthQUMzQjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUU7b0JBQ0wsaUJBQWlCLEVBQUUsSUFBSTtpQkFDeEI7Z0JBQ0QsTUFBTSxFQUFFLENBQUMsZ0JBQVUsQ0FBQyxLQUFLLENBQUM7YUFDM0I7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDhCQUE4QixFQUFFO1lBQy9ELGlCQUFpQixFQUFFLENBQUMsVUFBVSxFQUFFLE1BQU0sQ0FBQztZQUN2QywrQkFBK0IsRUFBRSxJQUFJO1NBQ3RDLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw4QkFBOEIsRUFBRTtZQUMvRCxpQkFBaUIsRUFBRSxDQUFDLG9CQUFvQixDQUFDO1lBQ3pDLCtCQUErQixFQUFFLElBQUk7U0FDdEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMkNBQTJDLEVBQUUsR0FBRyxFQUFFO1FBQ3JELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUU7b0JBQ0wsaUJBQWlCLEVBQUUsSUFBSTtpQkFDeEI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUU7b0JBQ0wsc0JBQXNCLEVBQUUsSUFBSTtpQkFDN0I7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUU7b0JBQ0wsaUJBQWlCLEVBQUUsSUFBSTtpQkFDeEI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw4QkFBOEIsRUFBRTtZQUMvRCxpQkFBaUIsRUFBRSxDQUFDLG9CQUFvQixDQUFDO1lBQ3pDLFlBQVksRUFBRSxlQUFNO1NBQ3JCLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw4QkFBOEIsRUFBRTtZQUMvRCxpQkFBaUIsRUFBRSxDQUFDLFVBQVUsQ0FBQztZQUMvQixZQUFZLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQztTQUN0QyxDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsOEJBQThCLEVBQUU7WUFDL0QsaUJBQWlCLEVBQUUsQ0FBQyxNQUFNLENBQUM7WUFDM0IsWUFBWSxFQUFFLENBQUMscUJBQXFCLENBQUM7U0FDdEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsaUVBQWlFLEVBQUUsR0FBRyxFQUFFO1FBQzNFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUNyQyxLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFO2dCQUNsQyxZQUFZLEVBQUUsRUFBRTthQUNqQjtTQUNGLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBRTdDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUNyQyxLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFLEVBQUUsc0JBQXNCLEVBQUUsSUFBSSxFQUFFO2dCQUN2QyxZQUFZLEVBQUUsRUFBRTthQUNqQjtTQUNGLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBRTdDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUNyQyxLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFO2dCQUNsQyxZQUFZLEVBQUUsRUFBRTthQUNqQjtTQUNGLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUNwQixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywyRkFBMkYsRUFBRSxHQUFHLEVBQUU7UUFDckcsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3JDLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUU7b0JBQ0wsc0JBQXNCLEVBQUUsSUFBSTtvQkFDNUIsaUJBQWlCLEVBQUUsSUFBSTtpQkFDeEI7Z0JBQ0QsTUFBTSxFQUFFLENBQUMsZ0JBQVUsQ0FBQyxLQUFLLENBQUM7YUFDM0I7U0FDRixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsaURBQWlELENBQUMsQ0FBQztRQUUvRCxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUU7WUFDckMsS0FBSyxFQUFFO2dCQUNMLEtBQUssRUFBRTtvQkFDTCxpQkFBaUIsRUFBRSxJQUFJO29CQUN2QixpQkFBaUIsRUFBRSxJQUFJO2lCQUN4QjtnQkFDRCxNQUFNLEVBQUUsQ0FBQyxnQkFBVSxDQUFDLEtBQUssQ0FBQzthQUMzQjtTQUNGLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO0lBQ2pFLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7UUFDeEIsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRTtZQUN2QixLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFO2dCQUNsQyxNQUFNLEVBQUU7b0JBQ04sZ0JBQVUsQ0FBQyxLQUFLO29CQUNoQixnQkFBVSxDQUFDLEtBQUs7b0JBQ2hCLGdCQUFVLENBQUMsTUFBTTtvQkFDakIsZ0JBQVUsQ0FBQyxPQUFPO29CQUNsQixnQkFBVSxDQUFDLGFBQWE7b0JBQ3hCLGdCQUFVLENBQUMsTUFBTSxDQUFDLGlDQUFpQyxDQUFDO2lCQUNyRDthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw4QkFBOEIsRUFBRTtZQUMvRCxrQkFBa0IsRUFBRTtnQkFDbEIsT0FBTztnQkFDUCxPQUFPO2dCQUNQLFFBQVE7Z0JBQ1IsU0FBUztnQkFDVCwrQkFBK0I7Z0JBQy9CLGlDQUFpQzthQUNsQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGtFQUFrRSxFQUFFLEdBQUcsRUFBRTtRQUM1RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLGtCQUFrQixFQUFFLFNBQVM7WUFDN0IsS0FBSyxFQUFFO2dCQUNMLEtBQUssRUFBRSxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBRTtnQkFDbEMsTUFBTSxFQUFFLENBQUMsZ0JBQVUsQ0FBQyxLQUFLLENBQUM7YUFDM0I7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUN4QixrQkFBa0IsRUFBRSxTQUFTO1lBQzdCLEtBQUssRUFBRTtnQkFDTCxLQUFLLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUU7Z0JBQ2xDLE1BQU0sRUFBRSxDQUFDLGdCQUFVLENBQUMsS0FBSyxDQUFDO2FBQzNCO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUU7WUFDeEIsa0JBQWtCLEVBQUUsU0FBUztZQUM3QixLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFO2dCQUNsQyxNQUFNLEVBQUUsQ0FBQyxnQkFBVSxDQUFDLE9BQU8sQ0FBQzthQUM3QjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLGtCQUFrQixFQUFFLFNBQVM7WUFDN0IsS0FBSyxFQUFFO2dCQUNMLEtBQUssRUFBRSxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBRTtnQkFDbEMsTUFBTSxFQUFFLENBQUMsZ0JBQVUsQ0FBQyxhQUFhLENBQUM7YUFDbkM7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDhCQUE4QixFQUFFO1lBQy9ELFVBQVUsRUFBRSxTQUFTO1lBQ3JCLGtCQUFrQixFQUFFLENBQUMsT0FBTyxFQUFFLFFBQVEsQ0FBQztTQUN4QyxDQUFDLENBQUM7UUFDSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsOEJBQThCLEVBQUU7WUFDL0QsVUFBVSxFQUFFLFNBQVM7WUFDckIsa0JBQWtCLEVBQUUsQ0FBQyxPQUFPLEVBQUUsUUFBUSxDQUFDO1NBQ3hDLENBQUMsQ0FBQztRQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyw4QkFBOEIsRUFBRTtZQUMvRCxVQUFVLEVBQUUsU0FBUztZQUNyQixrQkFBa0IsRUFBRSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUM7U0FDMUMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLDhCQUE4QixFQUFFO1lBQy9ELFVBQVUsRUFBRSxTQUFTO1lBQ3JCLGtCQUFrQixFQUFFLENBQUMsK0JBQStCLENBQUM7U0FDdEQsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseUNBQXlDLEVBQUUsR0FBRyxFQUFFO1FBQ25ELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUV6QyxPQUFPO1FBQ1AsSUFBSSxvQkFBYyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDbEMsUUFBUSxFQUFFLElBQUk7WUFDZCwwQkFBMEIsRUFBRSxJQUFJO1NBQ2pDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLDhCQUE4QixFQUFFO1lBQzNELFVBQVUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7WUFDMUMsMEJBQTBCLEVBQUUsU0FBUztTQUN0QyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywwQ0FBMEMsRUFBRSxHQUFHLEVBQUU7UUFDcEQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLE9BQU87UUFDUCxJQUFJLG9CQUFjLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUNsQyxRQUFRLEVBQUUsSUFBSTtZQUNkLDBCQUEwQixFQUFFLEtBQUs7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsOEJBQThCLEVBQUU7WUFDM0QsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztZQUMxQywwQkFBMEIsRUFBRSxRQUFRO1NBQ3JDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVEQUF1RCxFQUFFLEdBQUcsRUFBRTtRQUNqRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2xDLFFBQVEsRUFBRSxJQUFJO1NBQ2YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsOEJBQThCLEVBQUU7WUFDM0QsVUFBVSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztZQUMxQywwQkFBMEIsRUFBRSxlQUFNO1NBQ25DLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtRQUM5QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLGNBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFekMsTUFBTSxHQUFHLEdBQUcsOEJBQXdCLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUN6RixJQUFJLENBQUMsd0JBQXdCLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFbkMsT0FBTztRQUNQLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ2xDLFFBQVEsRUFBRSxJQUFJO1NBQ2YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsOEJBQThCLEVBQUU7WUFDM0QsMEJBQTBCLEVBQUU7Z0JBQzFCLGNBQWM7Z0JBQ2QsU0FBUzthQUNWO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO1FBQ3RDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUV6QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxZQUFZLEVBQUU7WUFDM0Isa0JBQWtCLEVBQUUsWUFBWTtZQUNoQywwQkFBMEIsRUFBRTtnQkFDMUIsb0NBQThCLENBQUMsT0FBTztnQkFDdEMsb0NBQThCLENBQUMsUUFBUTtnQkFDdkMsb0NBQThCLENBQUMsTUFBTTthQUN0QztTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLDhCQUE4QixFQUFFO1lBQzNELFVBQVUsRUFBRSxZQUFZO1lBQ3hCLDBCQUEwQixFQUFFLENBQUMsU0FBUyxFQUFFLFVBQVUsRUFBRSxpQkFBaUIsQ0FBQztTQUN2RSxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFO1FBQ3hCLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBUSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUV6QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxlQUFlLEVBQUU7WUFDOUIsa0JBQWtCLEVBQUUsZUFBZTtZQUNuQyxZQUFZLEVBQUUsSUFBSTtTQUNuQixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRTtZQUM3QixrQkFBa0IsRUFBRSxjQUFjO1lBQ2xDLFlBQVksRUFBRSxLQUFLO1NBQ3BCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLDhCQUE4QixFQUFFO1lBQzNELFVBQVUsRUFBRSxlQUFlO1lBQzNCLGlCQUFpQixFQUFFLGVBQU07WUFDekIsa0JBQWtCLEVBQUUsZUFBTTtZQUMxQiwrQkFBK0IsRUFBRSxLQUFLO1NBQ3ZDLENBQUMsQ0FBQztRQUNILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsOEJBQThCLEVBQUU7WUFDM0QsVUFBVSxFQUFFLGNBQWM7WUFDMUIsaUJBQWlCLEVBQUUsQ0FBQyxVQUFVLEVBQUUsTUFBTSxDQUFDO1lBQ3ZDLGtCQUFrQixFQUFFLENBQUMsU0FBUyxFQUFFLE9BQU8sRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLCtCQUErQixDQUFDO1lBQzVGLCtCQUErQixFQUFFLElBQUk7U0FDdEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsMERBQTBELEVBQUUsR0FBRyxFQUFFO1FBQ3BFLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxjQUFRLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXpDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsRUFBRTtZQUNwQyxZQUFZLEVBQUUsSUFBSTtZQUNsQixLQUFLLEVBQUU7Z0JBQ0wsS0FBSyxFQUFFO29CQUNMLHNCQUFzQixFQUFFLElBQUk7aUJBQzdCO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNyQyxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQUJTRU5UIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IE9BdXRoU2NvcGUsIFVzZXJQb29sLCBVc2VyUG9vbENsaWVudCwgVXNlclBvb2xDbGllbnRJZGVudGl0eVByb3ZpZGVyLCBVc2VyUG9vbElkZW50aXR5UHJvdmlkZXIgfSBmcm9tICcuLi9saWInO1xuXG5kZXNjcmliZSgnVXNlciBQb29sIENsaWVudCcsICgpID0+IHtcbiAgdGVzdCgnZGVmYXVsdCBzZXR1cCcsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgVXNlclBvb2xDbGllbnQoc3RhY2ssICdDbGllbnQnLCB7XG4gICAgICB1c2VyUG9vbDogcG9vbCxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgVXNlclBvb2xJZDogc3RhY2sucmVzb2x2ZShwb29sLnVzZXJQb29sSWQpLFxuICAgICAgQWxsb3dlZE9BdXRoRmxvd3M6IFsnaW1wbGljaXQnLCAnY29kZSddLFxuICAgICAgQWxsb3dlZE9BdXRoU2NvcGVzOiBbJ3Byb2ZpbGUnLCAncGhvbmUnLCAnZW1haWwnLCAnb3BlbmlkJywgJ2F3cy5jb2duaXRvLnNpZ25pbi51c2VyLmFkbWluJ10sXG4gICAgICBDYWxsYmFja1VSTHM6IFsnaHR0cHM6Ly9leGFtcGxlLmNvbSddLFxuICAgICAgU3VwcG9ydGVkSWRlbnRpdHlQcm92aWRlcnM6IFsnQ09HTklUTyddLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjbGllbnQgbmFtZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBjbGllbnQxID0gbmV3IFVzZXJQb29sQ2xpZW50KHN0YWNrLCAnQ2xpZW50MScsIHtcbiAgICAgIHVzZXJQb29sOiBwb29sLFxuICAgICAgdXNlclBvb2xDbGllbnROYW1lOiAnbXljbGllbnQnLFxuICAgIH0pO1xuICAgIGNvbnN0IGNsaWVudDIgPSBuZXcgVXNlclBvb2xDbGllbnQoc3RhY2ssICdDbGllbnQyJywge1xuICAgICAgdXNlclBvb2w6IHBvb2wsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KGNsaWVudDEudXNlclBvb2xDbGllbnROYW1lKS50b0VxdWFsKCdteWNsaWVudCcpO1xuICAgIGV4cGVjdCgoKSA9PiBjbGllbnQyLnVzZXJQb29sQ2xpZW50TmFtZSkudG9UaHJvdygvYXZhaWxhYmxlIG9ubHkgaWYgc3BlY2lmaWVkIG9uIHRoZSBVc2VyUG9vbENsaWVudCBkdXJpbmcgaW5pdGlhbGl6YXRpb24vKTtcbiAgfSk7XG5cbiAgdGVzdCgnaW1wb3J0JywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBjbGllbnQgPSBVc2VyUG9vbENsaWVudC5mcm9tVXNlclBvb2xDbGllbnRJZChzdGFjaywgJ0NsaWVudCcsICdjbGllbnQtaWQtMScpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChjbGllbnQudXNlclBvb2xDbGllbnRJZCkudG9FcXVhbCgnY2xpZW50LWlkLTEnKTtcbiAgfSk7XG5cbiAgdGVzdCgnRXhwbGljaXRBdXRoRmxvd3MgaXMgYWJzZW50IGJ5IGRlZmF1bHQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHBvb2wgPSBuZXcgVXNlclBvb2woc3RhY2ssICdQb29sJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgcG9vbC5hZGRDbGllbnQoJ0NsaWVudCcpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgRXhwbGljaXRBdXRoRmxvd3M6IEFCU0VOVCxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnRXhwbGljaXRBdXRoRmxvd3MgYXJlIGNvcnJlY3RseSBuYW1lZCcsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBwb29sLmFkZENsaWVudCgnQ2xpZW50Jywge1xuICAgICAgYXV0aEZsb3dzOiB7XG4gICAgICAgIGFkbWluVXNlclBhc3N3b3JkOiB0cnVlLFxuICAgICAgICBjdXN0b206IHRydWUsXG4gICAgICAgIHJlZnJlc2hUb2tlbjogdHJ1ZSxcbiAgICAgICAgdXNlclBhc3N3b3JkOiB0cnVlLFxuICAgICAgICB1c2VyU3JwOiB0cnVlLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgRXhwbGljaXRBdXRoRmxvd3M6IFtcbiAgICAgICAgJ0FMTE9XX1VTRVJfUEFTU1dPUkRfQVVUSCcsXG4gICAgICAgICdBTExPV19BRE1JTl9VU0VSX1BBU1NXT1JEX0FVVEgnLFxuICAgICAgICAnQUxMT1dfQ1VTVE9NX0FVVEgnLFxuICAgICAgICAnQUxMT1dfVVNFUl9TUlBfQVVUSCcsXG4gICAgICAgICdBTExPV19SRUZSRVNIX1RPS0VOX0FVVEgnLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnQWxsb3dlZE9BdXRoRmxvd3MgYXJlIGNvcnJlY3RseSBuYW1lZCcsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBwb29sLmFkZENsaWVudCgnQ2xpZW50MScsIHtcbiAgICAgIG9BdXRoOiB7XG4gICAgICAgIGZsb3dzOiB7XG4gICAgICAgICAgYXV0aG9yaXphdGlvbkNvZGVHcmFudDogdHJ1ZSxcbiAgICAgICAgICBpbXBsaWNpdENvZGVHcmFudDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgc2NvcGVzOiBbT0F1dGhTY29wZS5QSE9ORV0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHBvb2wuYWRkQ2xpZW50KCdDbGllbnQyJywge1xuICAgICAgb0F1dGg6IHtcbiAgICAgICAgZmxvd3M6IHtcbiAgICAgICAgICBjbGllbnRDcmVkZW50aWFsczogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICAgc2NvcGVzOiBbT0F1dGhTY29wZS5QSE9ORV0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgQWxsb3dlZE9BdXRoRmxvd3M6IFsnaW1wbGljaXQnLCAnY29kZSddLFxuICAgICAgQWxsb3dlZE9BdXRoRmxvd3NVc2VyUG9vbENsaWVudDogdHJ1ZSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgQWxsb3dlZE9BdXRoRmxvd3M6IFsnY2xpZW50X2NyZWRlbnRpYWxzJ10sXG4gICAgICBBbGxvd2VkT0F1dGhGbG93c1VzZXJQb29sQ2xpZW50OiB0cnVlLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjYWxsYmFja1VybCBkZWZhdWx0cyBhcmUgY29ycmVjdGx5IGNob3NlbicsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHBvb2wgPSBuZXcgVXNlclBvb2woc3RhY2ssICdQb29sJyk7XG5cbiAgICBwb29sLmFkZENsaWVudCgnQ2xpZW50MScsIHtcbiAgICAgIG9BdXRoOiB7XG4gICAgICAgIGZsb3dzOiB7XG4gICAgICAgICAgY2xpZW50Q3JlZGVudGlhbHM6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgcG9vbC5hZGRDbGllbnQoJ0NsaWVudDInLCB7XG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czoge1xuICAgICAgICAgIGF1dGhvcml6YXRpb25Db2RlR3JhbnQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgcG9vbC5hZGRDbGllbnQoJ0NsaWVudDMnLCB7XG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czoge1xuICAgICAgICAgIGltcGxpY2l0Q29kZUdyYW50OiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgQWxsb3dlZE9BdXRoRmxvd3M6IFsnY2xpZW50X2NyZWRlbnRpYWxzJ10sXG4gICAgICBDYWxsYmFja1VSTHM6IEFCU0VOVCxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgQWxsb3dlZE9BdXRoRmxvd3M6IFsnaW1wbGljaXQnXSxcbiAgICAgIENhbGxiYWNrVVJMczogWydodHRwczovL2V4YW1wbGUuY29tJ10sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2duaXRvOjpVc2VyUG9vbENsaWVudCcsIHtcbiAgICAgIEFsbG93ZWRPQXV0aEZsb3dzOiBbJ2NvZGUnXSxcbiAgICAgIENhbGxiYWNrVVJMczogWydodHRwczovL2V4YW1wbGUuY29tJ10sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2ZhaWxzIHdoZW4gY2FsbGJhY2tVcmxzIGlzIGVtcHR5IGZvciBjb2RlR3JhbnQgb3IgaW1wbGljaXRHcmFudCcsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHBvb2wgPSBuZXcgVXNlclBvb2woc3RhY2ssICdQb29sJyk7XG5cbiAgICBleHBlY3QoKCkgPT4gcG9vbC5hZGRDbGllbnQoJ0NsaWVudDEnLCB7XG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czogeyBpbXBsaWNpdENvZGVHcmFudDogdHJ1ZSB9LFxuICAgICAgICBjYWxsYmFja1VybHM6IFtdLFxuICAgICAgfSxcbiAgICB9KSkudG9UaHJvdygvY2FsbGJhY2tVcmwgbXVzdCBub3QgYmUgZW1wdHkvKTtcblxuICAgIGV4cGVjdCgoKSA9PiBwb29sLmFkZENsaWVudCgnQ2xpZW50MycsIHtcbiAgICAgIG9BdXRoOiB7XG4gICAgICAgIGZsb3dzOiB7IGF1dGhvcml6YXRpb25Db2RlR3JhbnQ6IHRydWUgfSxcbiAgICAgICAgY2FsbGJhY2tVcmxzOiBbXSxcbiAgICAgIH0sXG4gICAgfSkpLnRvVGhyb3coL2NhbGxiYWNrVXJsIG11c3Qgbm90IGJlIGVtcHR5Lyk7XG5cbiAgICBleHBlY3QoKCkgPT4gcG9vbC5hZGRDbGllbnQoJ0NsaWVudDQnLCB7XG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czogeyBjbGllbnRDcmVkZW50aWFsczogdHJ1ZSB9LFxuICAgICAgICBjYWxsYmFja1VybHM6IFtdLFxuICAgICAgfSxcbiAgICB9KSkubm90LnRvVGhyb3coKTtcbiAgfSk7XG5cbiAgdGVzdCgnZmFpbHMgd2hlbiBjbGllbnRDcmVkZW50aWFscyBPQXV0aCBmbG93IGlzIHNlbGVjdGVkIGFsb25nIHdpdGggY29kZUdyYW50IG9yIGltcGxpY2l0R3JhbnQnLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAnUG9vbCcpO1xuXG4gICAgZXhwZWN0KCgpID0+IHBvb2wuYWRkQ2xpZW50KCdDbGllbnQxJywge1xuICAgICAgb0F1dGg6IHtcbiAgICAgICAgZmxvd3M6IHtcbiAgICAgICAgICBhdXRob3JpemF0aW9uQ29kZUdyYW50OiB0cnVlLFxuICAgICAgICAgIGNsaWVudENyZWRlbnRpYWxzOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgICBzY29wZXM6IFtPQXV0aFNjb3BlLlBIT05FXSxcbiAgICAgIH0sXG4gICAgfSkpLnRvVGhyb3coL2NsaWVudENyZWRlbnRpYWxzIE9BdXRoIGZsb3cgY2Fubm90IGJlIHNlbGVjdGVkLyk7XG5cbiAgICBleHBlY3QoKCkgPT4gcG9vbC5hZGRDbGllbnQoJ0NsaWVudDInLCB7XG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czoge1xuICAgICAgICAgIGltcGxpY2l0Q29kZUdyYW50OiB0cnVlLFxuICAgICAgICAgIGNsaWVudENyZWRlbnRpYWxzOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgICBzY29wZXM6IFtPQXV0aFNjb3BlLlBIT05FXSxcbiAgICAgIH0sXG4gICAgfSkpLnRvVGhyb3coL2NsaWVudENyZWRlbnRpYWxzIE9BdXRoIGZsb3cgY2Fubm90IGJlIHNlbGVjdGVkLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ09BdXRoIHNjb3BlcycsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBwb29sLmFkZENsaWVudCgnQ2xpZW50Jywge1xuICAgICAgb0F1dGg6IHtcbiAgICAgICAgZmxvd3M6IHsgY2xpZW50Q3JlZGVudGlhbHM6IHRydWUgfSxcbiAgICAgICAgc2NvcGVzOiBbXG4gICAgICAgICAgT0F1dGhTY29wZS5QSE9ORSxcbiAgICAgICAgICBPQXV0aFNjb3BlLkVNQUlMLFxuICAgICAgICAgIE9BdXRoU2NvcGUuT1BFTklELFxuICAgICAgICAgIE9BdXRoU2NvcGUuUFJPRklMRSxcbiAgICAgICAgICBPQXV0aFNjb3BlLkNPR05JVE9fQURNSU4sXG4gICAgICAgICAgT0F1dGhTY29wZS5jdXN0b20oJ215LXJlc291cmNlLXNlcnZlci9teS1vd24tc2NvcGUnKSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBBbGxvd2VkT0F1dGhTY29wZXM6IFtcbiAgICAgICAgJ3Bob25lJyxcbiAgICAgICAgJ2VtYWlsJyxcbiAgICAgICAgJ29wZW5pZCcsXG4gICAgICAgICdwcm9maWxlJyxcbiAgICAgICAgJ2F3cy5jb2duaXRvLnNpZ25pbi51c2VyLmFkbWluJyxcbiAgICAgICAgJ215LXJlc291cmNlLXNlcnZlci9teS1vd24tc2NvcGUnLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnT0F1dGhTY29wZSAtIG9wZW5pZCBpcyBpbmNsdWRlZCB3aGVuIGVtYWlsIG9yIHBob25lIGlzIHNwZWNpZmllZCcsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBwb29sLmFkZENsaWVudCgnQ2xpZW50MScsIHtcbiAgICAgIHVzZXJQb29sQ2xpZW50TmFtZTogJ0NsaWVudDEnLFxuICAgICAgb0F1dGg6IHtcbiAgICAgICAgZmxvd3M6IHsgY2xpZW50Q3JlZGVudGlhbHM6IHRydWUgfSxcbiAgICAgICAgc2NvcGVzOiBbT0F1dGhTY29wZS5QSE9ORV0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHBvb2wuYWRkQ2xpZW50KCdDbGllbnQyJywge1xuICAgICAgdXNlclBvb2xDbGllbnROYW1lOiAnQ2xpZW50MicsXG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czogeyBjbGllbnRDcmVkZW50aWFsczogdHJ1ZSB9LFxuICAgICAgICBzY29wZXM6IFtPQXV0aFNjb3BlLkVNQUlMXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgcG9vbC5hZGRDbGllbnQoJ0NsaWVudDMnLCB7XG4gICAgICB1c2VyUG9vbENsaWVudE5hbWU6ICdDbGllbnQzJyxcbiAgICAgIG9BdXRoOiB7XG4gICAgICAgIGZsb3dzOiB7IGNsaWVudENyZWRlbnRpYWxzOiB0cnVlIH0sXG4gICAgICAgIHNjb3BlczogW09BdXRoU2NvcGUuUFJPRklMRV0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHBvb2wuYWRkQ2xpZW50KCdDbGllbnQ0Jywge1xuICAgICAgdXNlclBvb2xDbGllbnROYW1lOiAnQ2xpZW50NCcsXG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czogeyBjbGllbnRDcmVkZW50aWFsczogdHJ1ZSB9LFxuICAgICAgICBzY29wZXM6IFtPQXV0aFNjb3BlLkNPR05JVE9fQURNSU5dLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2duaXRvOjpVc2VyUG9vbENsaWVudCcsIHtcbiAgICAgIENsaWVudE5hbWU6ICdDbGllbnQxJyxcbiAgICAgIEFsbG93ZWRPQXV0aFNjb3BlczogWydwaG9uZScsICdvcGVuaWQnXSxcbiAgICB9KTtcbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2duaXRvOjpVc2VyUG9vbENsaWVudCcsIHtcbiAgICAgIENsaWVudE5hbWU6ICdDbGllbnQyJyxcbiAgICAgIEFsbG93ZWRPQXV0aFNjb3BlczogWydlbWFpbCcsICdvcGVuaWQnXSxcbiAgICB9KTtcbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpDb2duaXRvOjpVc2VyUG9vbENsaWVudCcsIHtcbiAgICAgIENsaWVudE5hbWU6ICdDbGllbnQzJyxcbiAgICAgIEFsbG93ZWRPQXV0aFNjb3BlczogWydwcm9maWxlJywgJ29wZW5pZCddLFxuICAgIH0pO1xuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZ25pdG86OlVzZXJQb29sQ2xpZW50Jywge1xuICAgICAgQ2xpZW50TmFtZTogJ0NsaWVudDQnLFxuICAgICAgQWxsb3dlZE9BdXRoU2NvcGVzOiBbJ2F3cy5jb2duaXRvLnNpZ25pbi51c2VyLmFkbWluJ10sXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2VuYWJsZSB1c2VyIGV4aXN0ZW5jZSBlcnJvcnMgcHJldmVudGlvbicsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcG9vbCA9IG5ldyBVc2VyUG9vbChzdGFjaywgJ1Bvb2wnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgVXNlclBvb2xDbGllbnQoc3RhY2ssICdDbGllbnQnLCB7XG4gICAgICB1c2VyUG9vbDogcG9vbCxcbiAgICAgIHByZXZlbnRVc2VyRXhpc3RlbmNlRXJyb3JzOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBVc2VyUG9vbElkOiBzdGFjay5yZXNvbHZlKHBvb2wudXNlclBvb2xJZCksXG4gICAgICBQcmV2ZW50VXNlckV4aXN0ZW5jZUVycm9yczogJ0VOQUJMRUQnLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkaXNhYmxlIHVzZXIgZXhpc3RlbmNlIGVycm9ycyBwcmV2ZW50aW9uJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAnUG9vbCcpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBVc2VyUG9vbENsaWVudChzdGFjaywgJ0NsaWVudCcsIHtcbiAgICAgIHVzZXJQb29sOiBwb29sLFxuICAgICAgcHJldmVudFVzZXJFeGlzdGVuY2VFcnJvcnM6IGZhbHNlLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBVc2VyUG9vbElkOiBzdGFjay5yZXNvbHZlKHBvb2wudXNlclBvb2xJZCksXG4gICAgICBQcmV2ZW50VXNlckV4aXN0ZW5jZUVycm9yczogJ0xFR0FDWScsXG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ3VzZXIgZXhpc3RlbmNlIGVycm9ycyBwcmV2ZW50aW9uIGlzIGFic2VudCBieSBkZWZhdWx0JywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAnUG9vbCcpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBVc2VyUG9vbENsaWVudChzdGFjaywgJ0NsaWVudCcsIHtcbiAgICAgIHVzZXJQb29sOiBwb29sLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBVc2VyUG9vbElkOiBzdGFjay5yZXNvbHZlKHBvb2wudXNlclBvb2xJZCksXG4gICAgICBQcmV2ZW50VXNlckV4aXN0ZW5jZUVycm9yczogQUJTRU5ULFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkZWZhdWx0IHN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAnUG9vbCcpO1xuXG4gICAgY29uc3QgaWRwID0gVXNlclBvb2xJZGVudGl0eVByb3ZpZGVyLmZyb21Qcm92aWRlck5hbWUoc3RhY2ssICdpbXBvcnRlZCcsICd1c2VycG9vbC1pZHAnKTtcbiAgICBwb29sLnJlZ2lzdGVySWRlbnRpdHlQcm92aWRlcihpZHApO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBVc2VyUG9vbENsaWVudChzdGFjaywgJ0NsaWVudCcsIHtcbiAgICAgIHVzZXJQb29sOiBwb29sLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBTdXBwb3J0ZWRJZGVudGl0eVByb3ZpZGVyczogW1xuICAgICAgICAndXNlcnBvb2wtaWRwJyxcbiAgICAgICAgJ0NPR05JVE8nLFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnc3VwcG9ydGVkSWRlbnRpdHlQcm92aWRlcnMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHBvb2wgPSBuZXcgVXNlclBvb2woc3RhY2ssICdQb29sJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgcG9vbC5hZGRDbGllbnQoJ0FsbEVuYWJsZWQnLCB7XG4gICAgICB1c2VyUG9vbENsaWVudE5hbWU6ICdBbGxFbmFibGVkJyxcbiAgICAgIHN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzOiBbXG4gICAgICAgIFVzZXJQb29sQ2xpZW50SWRlbnRpdHlQcm92aWRlci5DT0dOSVRPLFxuICAgICAgICBVc2VyUG9vbENsaWVudElkZW50aXR5UHJvdmlkZXIuRkFDRUJPT0ssXG4gICAgICAgIFVzZXJQb29sQ2xpZW50SWRlbnRpdHlQcm92aWRlci5BTUFaT04sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBDbGllbnROYW1lOiAnQWxsRW5hYmxlZCcsXG4gICAgICBTdXBwb3J0ZWRJZGVudGl0eVByb3ZpZGVyczogWydDT0dOSVRPJywgJ0ZhY2Vib29rJywgJ0xvZ2luV2l0aEFtYXpvbiddLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkaXNhYmxlT0F1dGgnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHBvb2wgPSBuZXcgVXNlclBvb2woc3RhY2ssICdQb29sJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgcG9vbC5hZGRDbGllbnQoJ09BdXRoRGlzYWJsZWQnLCB7XG4gICAgICB1c2VyUG9vbENsaWVudE5hbWU6ICdPQXV0aERpc2FibGVkJyxcbiAgICAgIGRpc2FibGVPQXV0aDogdHJ1ZSxcbiAgICB9KTtcbiAgICBwb29sLmFkZENsaWVudCgnT0F1dGhFbmFibGVkJywge1xuICAgICAgdXNlclBvb2xDbGllbnROYW1lOiAnT0F1dGhFbmFibGVkJyxcbiAgICAgIGRpc2FibGVPQXV0aDogZmFsc2UsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpDb2duaXRvOjpVc2VyUG9vbENsaWVudCcsIHtcbiAgICAgIENsaWVudE5hbWU6ICdPQXV0aERpc2FibGVkJyxcbiAgICAgIEFsbG93ZWRPQXV0aEZsb3dzOiBBQlNFTlQsXG4gICAgICBBbGxvd2VkT0F1dGhTY29wZXM6IEFCU0VOVCxcbiAgICAgIEFsbG93ZWRPQXV0aEZsb3dzVXNlclBvb2xDbGllbnQ6IGZhbHNlLFxuICAgIH0pO1xuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6Q29nbml0bzo6VXNlclBvb2xDbGllbnQnLCB7XG4gICAgICBDbGllbnROYW1lOiAnT0F1dGhFbmFibGVkJyxcbiAgICAgIEFsbG93ZWRPQXV0aEZsb3dzOiBbJ2ltcGxpY2l0JywgJ2NvZGUnXSxcbiAgICAgIEFsbG93ZWRPQXV0aFNjb3BlczogWydwcm9maWxlJywgJ3Bob25lJywgJ2VtYWlsJywgJ29wZW5pZCcsICdhd3MuY29nbml0by5zaWduaW4udXNlci5hZG1pbiddLFxuICAgICAgQWxsb3dlZE9BdXRoRmxvd3NVc2VyUG9vbENsaWVudDogdHJ1ZSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnZmFpbHMgd2hlbiBvQXV0aCBpcyBzcGVjaWZpZWQgYnV0IGlzIGRpc2FibGVPQXV0aCBpcyBzZXQnLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwb29sID0gbmV3IFVzZXJQb29sKHN0YWNrLCAnUG9vbCcpO1xuXG4gICAgZXhwZWN0KCgpID0+IHBvb2wuYWRkQ2xpZW50KCdDbGllbnQnLCB7XG4gICAgICBkaXNhYmxlT0F1dGg6IHRydWUsXG4gICAgICBvQXV0aDoge1xuICAgICAgICBmbG93czoge1xuICAgICAgICAgIGF1dGhvcml6YXRpb25Db2RlR3JhbnQ6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pKS50b1Rocm93KC9kaXNhYmxlT0F1dGggaXMgc2V0Lyk7XG4gIH0pO1xufSk7Il19