"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolClient = exports.UserPoolClientIdentityProvider = exports.OAuthScope = void 0;
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
/**
 * OAuth scopes that are allowed with this client.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 */
class OAuthScope {
    constructor(scopeName) {
        this.scopeName = scopeName;
    }
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     * The format is 'resource-server-identifier/scope'.
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     */
    static custom(name) {
        return new OAuthScope(name);
    }
}
exports.OAuthScope = OAuthScope;
/**
 * Grants access to the 'phone_number' and 'phone_number_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PHONE = new OAuthScope('phone');
/**
 * Grants access to the 'email' and 'email_verified' claims.
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.EMAIL = new OAuthScope('email');
/**
 * Returns all user attributes in the ID token that are readable by the client
 */
OAuthScope.OPENID = new OAuthScope('openid');
/**
 * Grants access to all user attributes that are readable by the client
 * Automatically includes access to `OAuthScope.OPENID`.
 */
OAuthScope.PROFILE = new OAuthScope('profile');
/**
 * Grants access to Amazon Cognito User Pool API operations that require access tokens,
 * such as UpdateUserAttributes and VerifyUserAttribute.
 */
OAuthScope.COGNITO_ADMIN = new OAuthScope('aws.cognito.signin.user.admin');
/**
 * Identity providers supported by the UserPoolClient
 */
class UserPoolClientIdentityProvider {
    constructor(name) {
        this.name = name;
    }
    /**
     * Specify a provider not yet supported by the CDK.
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`
     */
    static custom(name) {
        return new UserPoolClientIdentityProvider(name);
    }
}
exports.UserPoolClientIdentityProvider = UserPoolClientIdentityProvider;
/**
 * Allow users to sign in using 'Facebook Login'.
 * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.FACEBOOK = new UserPoolClientIdentityProvider('Facebook');
/**
 * Allow users to sign in using 'Login With Amazon'.
 * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
 */
UserPoolClientIdentityProvider.AMAZON = new UserPoolClientIdentityProvider('LoginWithAmazon');
/**
 * Allow users to sign in directly as a user of the User Pool
 */
UserPoolClientIdentityProvider.COGNITO = new UserPoolClientIdentityProvider('COGNITO');
/**
 * Define a UserPool App Client
 */
class UserPoolClient extends core_1.Resource {
    /*
     * Note to implementers: Two CloudFormation return values Name and ClientSecret are part of the spec.
     * However, they have been explicity not implemented here. They are not documented in CloudFormation, and
     * CloudFormation returns the following the string when these two attributes are 'GetAtt' - "attribute not supported
     * at this time, please use the CLI or Console to retrieve this value".
     * Awaiting updates from CloudFormation.
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        if (props.disableOAuth && props.oAuth) {
            throw new Error('OAuth settings cannot be specified when disableOAuth is set.');
        }
        this.oAuthFlows = (_b = (_a = props.oAuth) === null || _a === void 0 ? void 0 : _a.flows) !== null && _b !== void 0 ? _b : {
            implicitCodeGrant: true,
            authorizationCodeGrant: true,
        };
        let callbackUrls = (_c = props.oAuth) === null || _c === void 0 ? void 0 : _c.callbackUrls;
        if (this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) {
            if (callbackUrls === undefined) {
                callbackUrls = ['https://example.com'];
            }
            else if (callbackUrls.length === 0) {
                throw new Error('callbackUrl must not be empty when codeGrant or implicitGrant OAuth flows are enabled.');
            }
        }
        const resource = new cognito_generated_1.CfnUserPoolClient(this, 'Resource', {
            clientName: props.userPoolClientName,
            generateSecret: props.generateSecret,
            userPoolId: props.userPool.userPoolId,
            explicitAuthFlows: this.configureAuthFlows(props),
            allowedOAuthFlows: props.disableOAuth ? undefined : this.configureOAuthFlows(),
            allowedOAuthScopes: props.disableOAuth ? undefined : this.configureOAuthScopes(props.oAuth),
            callbackUrLs: callbackUrls && callbackUrls.length > 0 ? callbackUrls : undefined,
            allowedOAuthFlowsUserPoolClient: !props.disableOAuth,
            preventUserExistenceErrors: this.configurePreventUserExistenceErrors(props.preventUserExistenceErrors),
            supportedIdentityProviders: this.configureIdentityProviders(props),
        });
        this.userPoolClientId = resource.ref;
        this._userPoolClientName = props.userPoolClientName;
    }
    /**
     * Import a user pool client given its id.
     */
    static fromUserPoolClientId(scope, id, userPoolClientId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.userPoolClientId = userPoolClientId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization,
     * throws an error otherwise.
     */
    get userPoolClientName() {
        if (this._userPoolClientName === undefined) {
            throw new Error('userPoolClientName is available only if specified on the UserPoolClient during initialization');
        }
        return this._userPoolClientName;
    }
    configureAuthFlows(props) {
        var _a, _b, _c, _d, _e;
        const authFlows = [];
        if ((_a = props.authFlows) === null || _a === void 0 ? void 0 : _a.userPassword) {
            authFlows.push('ALLOW_USER_PASSWORD_AUTH');
        }
        if ((_b = props.authFlows) === null || _b === void 0 ? void 0 : _b.adminUserPassword) {
            authFlows.push('ALLOW_ADMIN_USER_PASSWORD_AUTH');
        }
        if ((_c = props.authFlows) === null || _c === void 0 ? void 0 : _c.custom) {
            authFlows.push('ALLOW_CUSTOM_AUTH');
        }
        if ((_d = props.authFlows) === null || _d === void 0 ? void 0 : _d.userSrp) {
            authFlows.push('ALLOW_USER_SRP_AUTH');
        }
        if ((_e = props.authFlows) === null || _e === void 0 ? void 0 : _e.refreshToken) {
            authFlows.push('ALLOW_REFRESH_TOKEN_AUTH');
        }
        if (authFlows.length === 0) {
            return undefined;
        }
        return authFlows;
    }
    configureOAuthFlows() {
        if ((this.oAuthFlows.authorizationCodeGrant || this.oAuthFlows.implicitCodeGrant) && this.oAuthFlows.clientCredentials) {
            throw new Error('clientCredentials OAuth flow cannot be selected along with codeGrant or implicitGrant.');
        }
        const oAuthFlows = [];
        if (this.oAuthFlows.clientCredentials) {
            oAuthFlows.push('client_credentials');
        }
        if (this.oAuthFlows.implicitCodeGrant) {
            oAuthFlows.push('implicit');
        }
        if (this.oAuthFlows.authorizationCodeGrant) {
            oAuthFlows.push('code');
        }
        if (oAuthFlows.length === 0) {
            return undefined;
        }
        return oAuthFlows;
    }
    configureOAuthScopes(oAuth) {
        var _a;
        const scopes = (_a = oAuth === null || oAuth === void 0 ? void 0 : oAuth.scopes) !== null && _a !== void 0 ? _a : [OAuthScope.PROFILE, OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.OPENID,
            OAuthScope.COGNITO_ADMIN];
        const scopeNames = new Set(scopes.map((x) => x.scopeName));
        const autoOpenIdScopes = [OAuthScope.PHONE, OAuthScope.EMAIL, OAuthScope.PROFILE];
        if (autoOpenIdScopes.reduce((agg, s) => agg || scopeNames.has(s.scopeName), false)) {
            scopeNames.add(OAuthScope.OPENID.scopeName);
        }
        return Array.from(scopeNames);
    }
    configurePreventUserExistenceErrors(prevent) {
        if (prevent === undefined) {
            return undefined;
        }
        return prevent ? 'ENABLED' : 'LEGACY';
    }
    configureIdentityProviders(props) {
        let providers;
        if (!props.supportedIdentityProviders) {
            const providerSet = new Set(props.userPool.identityProviders.map((p) => p.providerName));
            providerSet.add('COGNITO');
            providers = Array.from(providerSet);
        }
        else {
            providers = props.supportedIdentityProviders.map((p) => p.name);
        }
        if (providers.length === 0) {
            return undefined;
        }
        return Array.from(providers);
    }
}
exports.UserPoolClient = UserPoolClient;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXNlci1wb29sLWNsaWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVzZXItcG9vbC1jbGllbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQStEO0FBQy9ELDJEQUF3RDtBQTBGeEQ7OztHQUdHO0FBQ0gsTUFBYSxVQUFVO0lBNkNyQixZQUFvQixTQUFpQjtRQUNuQyxJQUFJLENBQUMsU0FBUyxHQUFHLFNBQVMsQ0FBQztJQUM3QixDQUFDO0lBakJEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVk7UUFDL0IsT0FBTyxJQUFJLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM5QixDQUFDOztBQXJDSCxnQ0FnREM7QUEvQ0M7OztHQUdHO0FBQ29CLGdCQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7QUFFdkQ7OztHQUdHO0FBQ29CLGdCQUFLLEdBQUcsSUFBSSxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7QUFFdkQ7O0dBRUc7QUFDb0IsaUJBQU0sR0FBRyxJQUFJLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztBQUV6RDs7O0dBR0c7QUFDb0Isa0JBQU8sR0FBRyxJQUFJLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQztBQUUzRDs7O0dBR0c7QUFDb0Isd0JBQWEsR0FBRyxJQUFJLFVBQVUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0FBc0J6Rjs7R0FFRztBQUNILE1BQWEsOEJBQThCO0lBNkJ6QyxZQUFvQixJQUFZO1FBQzlCLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDO0lBQ25CLENBQUM7SUFiRDs7O09BR0c7SUFDSSxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVk7UUFDL0IsT0FBTyxJQUFJLDhCQUE4QixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2xELENBQUM7O0FBeEJILHdFQWdDQztBQS9CQzs7O0dBR0c7QUFDb0IsdUNBQVEsR0FBRyxJQUFJLDhCQUE4QixDQUFDLFVBQVUsQ0FBQyxDQUFDO0FBRWpGOzs7R0FHRztBQUNvQixxQ0FBTSxHQUFHLElBQUksOEJBQThCLENBQUMsaUJBQWlCLENBQUMsQ0FBQztBQUV0Rjs7R0FFRztBQUNvQixzQ0FBTyxHQUFHLElBQUksOEJBQThCLENBQUMsU0FBUyxDQUFDLENBQUM7QUE4RmpGOztHQUVHO0FBQ0gsTUFBYSxjQUFlLFNBQVEsZUFBUTtJQW1CMUM7Ozs7OztPQU1HO0lBRUgsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUEwQjs7UUFDbEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksS0FBSyxDQUFDLEtBQUssRUFBRTtZQUNyQyxNQUFNLElBQUksS0FBSyxDQUFDLDhEQUE4RCxDQUFDLENBQUM7U0FDakY7UUFFRCxJQUFJLENBQUMsVUFBVSxlQUFHLEtBQUssQ0FBQyxLQUFLLDBDQUFFLEtBQUssbUNBQUk7WUFDdEMsaUJBQWlCLEVBQUUsSUFBSTtZQUN2QixzQkFBc0IsRUFBRSxJQUFJO1NBQzdCLENBQUM7UUFFRixJQUFJLFlBQVksU0FBeUIsS0FBSyxDQUFDLEtBQUssMENBQUUsWUFBWSxDQUFDO1FBQ25FLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxzQkFBc0IsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFpQixFQUFFO1lBQy9FLElBQUksWUFBWSxLQUFLLFNBQVMsRUFBRTtnQkFDOUIsWUFBWSxHQUFHLENBQUMscUJBQXFCLENBQUMsQ0FBQzthQUN4QztpQkFBTSxJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO2dCQUNwQyxNQUFNLElBQUksS0FBSyxDQUFDLHdGQUF3RixDQUFDLENBQUM7YUFDM0c7U0FDRjtRQUVELE1BQU0sUUFBUSxHQUFHLElBQUkscUNBQWlCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN2RCxVQUFVLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUNwQyxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsVUFBVTtZQUNyQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDO1lBQ2pELGlCQUFpQixFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLG1CQUFtQixFQUFFO1lBQzlFLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUM7WUFDM0YsWUFBWSxFQUFFLFlBQVksSUFBSSxZQUFZLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2hGLCtCQUErQixFQUFFLENBQUMsS0FBSyxDQUFDLFlBQVk7WUFDcEQsMEJBQTBCLEVBQUUsSUFBSSxDQUFDLG1DQUFtQyxDQUFDLEtBQUssQ0FBQywwQkFBMEIsQ0FBQztZQUN0RywwQkFBMEIsRUFBRSxJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDO1NBQ25FLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7SUFDdEQsQ0FBQztJQTlERDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxnQkFBd0I7UUFDdkYsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLHFCQUFnQixHQUFHLGdCQUFnQixDQUFDO1lBQ3RELENBQUM7U0FBQTtRQUVELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUF1REQ7OztPQUdHO0lBQ0gsSUFBVyxrQkFBa0I7UUFDM0IsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEtBQUssU0FBUyxFQUFFO1lBQzFDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0ZBQStGLENBQUMsQ0FBQztTQUNsSDtRQUNELE9BQU8sSUFBSSxDQUFDLG1CQUFtQixDQUFDO0lBQ2xDLENBQUM7SUFFTyxrQkFBa0IsQ0FBQyxLQUEwQjs7UUFDbkQsTUFBTSxTQUFTLEdBQWEsRUFBRSxDQUFDO1FBQy9CLFVBQUksS0FBSyxDQUFDLFNBQVMsMENBQUUsWUFBWSxFQUFFO1lBQUUsU0FBUyxDQUFDLElBQUksQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1NBQUU7UUFDbEYsVUFBSSxLQUFLLENBQUMsU0FBUywwQ0FBRSxpQkFBaUIsRUFBRTtZQUFFLFNBQVMsQ0FBQyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztTQUFFO1FBQzdGLFVBQUksS0FBSyxDQUFDLFNBQVMsMENBQUUsTUFBTSxFQUFFO1lBQUUsU0FBUyxDQUFDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1NBQUU7UUFDckUsVUFBSSxLQUFLLENBQUMsU0FBUywwQ0FBRSxPQUFPLEVBQUU7WUFBRSxTQUFTLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLENBQUM7U0FBRTtRQUN4RSxVQUFJLEtBQUssQ0FBQyxTQUFTLDBDQUFFLFlBQVksRUFBRTtZQUFFLFNBQVMsQ0FBQyxJQUFJLENBQUMsMEJBQTBCLENBQUMsQ0FBQztTQUFFO1FBRWxGLElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDMUIsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFDRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU8sbUJBQW1CO1FBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLHNCQUFzQixJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUMsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLGlCQUFpQixFQUFFO1lBQ3RILE1BQU0sSUFBSSxLQUFLLENBQUMsd0ZBQXdGLENBQUMsQ0FBQztTQUMzRztRQUNELE1BQU0sVUFBVSxHQUFhLEVBQUUsQ0FBQztRQUNoQyxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsaUJBQWlCLEVBQUU7WUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7U0FBRTtRQUNqRixJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsaUJBQWlCLEVBQUU7WUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQUU7UUFDdkUsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLHNCQUFzQixFQUFFO1lBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUFFO1FBRXhFLElBQUksVUFBVSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDM0IsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFDRCxPQUFPLFVBQVUsQ0FBQztJQUNwQixDQUFDO0lBRU8sb0JBQW9CLENBQUMsS0FBcUI7O1FBQ2hELE1BQU0sTUFBTSxTQUFHLEtBQUssYUFBTCxLQUFLLHVCQUFMLEtBQUssQ0FBRSxNQUFNLG1DQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxVQUFVLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLE1BQU07WUFDeEcsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQzVCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQzNELE1BQU0sZ0JBQWdCLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2xGLElBQUksZ0JBQWdCLENBQUMsTUFBTSxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsR0FBRyxJQUFJLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxFQUFFO1lBQ2xGLFVBQVUsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUM3QztRQUNELE9BQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRU8sbUNBQW1DLENBQUMsT0FBaUI7UUFDM0QsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFO1lBQ3pCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBQ0QsT0FBTyxPQUFPLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDO0lBQ3hDLENBQUM7SUFFTywwQkFBMEIsQ0FBQyxLQUEwQjtRQUMzRCxJQUFJLFNBQW1CLENBQUM7UUFDeEIsSUFBSSxDQUFDLEtBQUssQ0FBQywwQkFBMEIsRUFBRTtZQUNyQyxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUM7WUFDekYsV0FBVyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUMzQixTQUFTLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztTQUNyQzthQUFNO1lBQ0wsU0FBUyxHQUFHLEtBQUssQ0FBQywwQkFBMEIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNqRTtRQUNELElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFBRSxPQUFPLFNBQVMsQ0FBQztTQUFFO1FBQ2pELE9BQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMvQixDQUFDO0NBQ0Y7QUF2SUQsd0NBdUlDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIFJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5Vc2VyUG9vbENsaWVudCB9IGZyb20gJy4vY29nbml0by5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSVVzZXJQb29sIH0gZnJvbSAnLi91c2VyLXBvb2wnO1xuXG4vKipcbiAqIFR5cGVzIG9mIGF1dGhlbnRpY2F0aW9uIGZsb3dcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FtYXpvbi1jb2duaXRvLXVzZXItcG9vbHMtYXV0aGVudGljYXRpb24tZmxvdy5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXV0aEZsb3cge1xuICAvKipcbiAgICogRW5hYmxlIGFkbWluIGJhc2VkIHVzZXIgcGFzc3dvcmQgYXV0aGVudGljYXRpb24gZmxvd1xuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYWRtaW5Vc2VyUGFzc3dvcmQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBFbmFibGUgY3VzdG9tIGF1dGhlbnRpY2F0aW9uIGZsb3dcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEVuYWJsZSBhdXRoIHVzaW5nIHVzZXJuYW1lICYgcGFzc3dvcmRcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQYXNzd29yZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEVuYWJsZSBTUlAgYmFzZWQgYXV0aGVudGljYXRpb25cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJTcnA/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBFbmFibGUgYXV0aGZsb3cgdG8gcmVmcmVzaCB0b2tlbnNcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHJlZnJlc2hUb2tlbj86IGJvb2xlYW47XG59XG5cbi8qKlxuICogT0F1dGggc2V0dGluZ3MgdG8gY29uZmlndXJlIHRoZSBpbnRlcmFjdGlvbiBiZXR3ZWVuIHRoZSBhcHAgYW5kIHRoaXMgY2xpZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE9BdXRoU2V0dGluZ3Mge1xuXG4gIC8qKlxuICAgKiBPQXV0aCBmbG93cyB0aGF0IGFyZSBhbGxvd2VkIHdpdGggdGhpcyBjbGllbnQuXG4gICAqIEBzZWUgLSB0aGUgJ0FsbG93ZWQgT0F1dGggRmxvd3MnIHNlY3Rpb24gYXQgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hcHAtaWRwLXNldHRpbmdzLmh0bWxcbiAgICogQGRlZmF1bHQge2F1dGhvcml6YXRpb25Db2RlR3JhbnQ6dHJ1ZSxpbXBsaWNpdENvZGVHcmFudDp0cnVlfVxuICAgKi9cbiAgcmVhZG9ubHkgZmxvd3M/OiBPQXV0aEZsb3dzO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGFsbG93ZWQgcmVkaXJlY3QgVVJMcyBmb3IgdGhlIGlkZW50aXR5IHByb3ZpZGVycy5cbiAgICogQGRlZmF1bHQgLSBbJ2h0dHBzOi8vZXhhbXBsZS5jb20nXSBpZiBlaXRoZXIgYXV0aG9yaXphdGlvbkNvZGVHcmFudCBvciBpbXBsaWNpdENvZGVHcmFudCBmbG93cyBhcmUgZW5hYmxlZCwgbm8gY2FsbGJhY2sgVVJMcyBvdGhlcndpc2UuXG4gICAqL1xuICByZWFkb25seSBjYWxsYmFja1VybHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogT0F1dGggc2NvcGVzIHRoYXQgYXJlIGFsbG93ZWQgd2l0aCB0aGlzIGNsaWVudC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by11c2VyLXBvb2xzLWFwcC1pZHAtc2V0dGluZ3MuaHRtbFxuICAgKiBAZGVmYXVsdCBbT0F1dGhTY29wZS5QSE9ORSxPQXV0aFNjb3BlLkVNQUlMLE9BdXRoU2NvcGUuT1BFTklELE9BdXRoU2NvcGUuUFJPRklMRSxPQXV0aFNjb3BlLkNPR05JVE9fQURNSU5dXG4gICAqL1xuICByZWFkb25seSBzY29wZXM/OiBPQXV0aFNjb3BlW107XG59XG5cbi8qKlxuICogVHlwZXMgb2YgT0F1dGggZ3JhbnQgZmxvd3NcbiAqIEBzZWUgLSB0aGUgJ0FsbG93ZWQgT0F1dGggRmxvd3MnIHNlY3Rpb24gYXQgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hcHAtaWRwLXNldHRpbmdzLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBPQXV0aEZsb3dzIHtcbiAgLyoqXG4gICAqIEluaXRpYXRlIGFuIGF1dGhvcml6YXRpb24gY29kZSBncmFudCBmbG93LCB3aGljaCBwcm92aWRlcyBhbiBhdXRob3JpemF0aW9uIGNvZGUgYXMgdGhlIHJlc3BvbnNlLlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYXV0aG9yaXphdGlvbkNvZGVHcmFudD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBjbGllbnQgc2hvdWxkIGdldCB0aGUgYWNjZXNzIHRva2VuIGFuZCBJRCB0b2tlbiBkaXJlY3RseS5cbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGltcGxpY2l0Q29kZUdyYW50PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ2xpZW50IHNob3VsZCBnZXQgdGhlIGFjY2VzcyB0b2tlbiBhbmQgSUQgdG9rZW4gZnJvbSB0aGUgdG9rZW4gZW5kcG9pbnRcbiAgICogdXNpbmcgYSBjb21iaW5hdGlvbiBvZiBjbGllbnQgYW5kIGNsaWVudF9zZWNyZXQuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBjbGllbnRDcmVkZW50aWFscz86IGJvb2xlYW47XG59XG5cbi8qKlxuICogT0F1dGggc2NvcGVzIHRoYXQgYXJlIGFsbG93ZWQgd2l0aCB0aGlzIGNsaWVudC5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1hcHAtaWRwLXNldHRpbmdzLmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIE9BdXRoU2NvcGUge1xuICAvKipcbiAgICogR3JhbnRzIGFjY2VzcyB0byB0aGUgJ3Bob25lX251bWJlcicgYW5kICdwaG9uZV9udW1iZXJfdmVyaWZpZWQnIGNsYWltcy5cbiAgICogQXV0b21hdGljYWxseSBpbmNsdWRlcyBhY2Nlc3MgdG8gYE9BdXRoU2NvcGUuT1BFTklEYC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUEhPTkUgPSBuZXcgT0F1dGhTY29wZSgncGhvbmUnKTtcblxuICAvKipcbiAgICogR3JhbnRzIGFjY2VzcyB0byB0aGUgJ2VtYWlsJyBhbmQgJ2VtYWlsX3ZlcmlmaWVkJyBjbGFpbXMuXG4gICAqIEF1dG9tYXRpY2FsbHkgaW5jbHVkZXMgYWNjZXNzIHRvIGBPQXV0aFNjb3BlLk9QRU5JRGAuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVNQUlMID0gbmV3IE9BdXRoU2NvcGUoJ2VtYWlsJyk7XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYWxsIHVzZXIgYXR0cmlidXRlcyBpbiB0aGUgSUQgdG9rZW4gdGhhdCBhcmUgcmVhZGFibGUgYnkgdGhlIGNsaWVudFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBPUEVOSUQgPSBuZXcgT0F1dGhTY29wZSgnb3BlbmlkJyk7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBhY2Nlc3MgdG8gYWxsIHVzZXIgYXR0cmlidXRlcyB0aGF0IGFyZSByZWFkYWJsZSBieSB0aGUgY2xpZW50XG4gICAqIEF1dG9tYXRpY2FsbHkgaW5jbHVkZXMgYWNjZXNzIHRvIGBPQXV0aFNjb3BlLk9QRU5JRGAuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFBST0ZJTEUgPSBuZXcgT0F1dGhTY29wZSgncHJvZmlsZScpO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgYWNjZXNzIHRvIEFtYXpvbiBDb2duaXRvIFVzZXIgUG9vbCBBUEkgb3BlcmF0aW9ucyB0aGF0IHJlcXVpcmUgYWNjZXNzIHRva2VucyxcbiAgICogc3VjaCBhcyBVcGRhdGVVc2VyQXR0cmlidXRlcyBhbmQgVmVyaWZ5VXNlckF0dHJpYnV0ZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ09HTklUT19BRE1JTiA9IG5ldyBPQXV0aFNjb3BlKCdhd3MuY29nbml0by5zaWduaW4udXNlci5hZG1pbicpO1xuXG4gIC8qKlxuICAgKiBDdXN0b20gc2NvcGUgaXMgb25lIHRoYXQgeW91IGRlZmluZSBmb3IgeW91ciBvd24gcmVzb3VyY2Ugc2VydmVyIGluIHRoZSBSZXNvdXJjZSBTZXJ2ZXJzLlxuICAgKiBUaGUgZm9ybWF0IGlzICdyZXNvdXJjZS1zZXJ2ZXItaWRlbnRpZmllci9zY29wZScuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZ25pdG8vbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2NvZ25pdG8tdXNlci1wb29scy1kZWZpbmUtcmVzb3VyY2Utc2VydmVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGN1c3RvbShuYW1lOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gbmV3IE9BdXRoU2NvcGUobmFtZSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhpcyBzY29wZSBhcyByZWNvZ25pemVkIGJ5IENsb3VkRm9ybWF0aW9uLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1jb2duaXRvLXVzZXJwb29sY2xpZW50Lmh0bWwjY2ZuLWNvZ25pdG8tdXNlcnBvb2xjbGllbnQtYWxsb3dlZG9hdXRoc2NvcGVzXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc2NvcGVOYW1lOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihzY29wZU5hbWU6IHN0cmluZykge1xuICAgIHRoaXMuc2NvcGVOYW1lID0gc2NvcGVOYW1lO1xuICB9XG59XG5cbi8qKlxuICogSWRlbnRpdHkgcHJvdmlkZXJzIHN1cHBvcnRlZCBieSB0aGUgVXNlclBvb2xDbGllbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFVzZXJQb29sQ2xpZW50SWRlbnRpdHlQcm92aWRlciB7XG4gIC8qKlxuICAgKiBBbGxvdyB1c2VycyB0byBzaWduIGluIHVzaW5nICdGYWNlYm9vayBMb2dpbicuXG4gICAqIEEgYFVzZXJQb29sSWRlbnRpdHlQcm92aWRlckZhY2Vib29rYCBtdXN0IGJlIGF0dGFjaGVkIHRvIHRoZSB1c2VyIHBvb2wuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEZBQ0VCT09LID0gbmV3IFVzZXJQb29sQ2xpZW50SWRlbnRpdHlQcm92aWRlcignRmFjZWJvb2snKTtcblxuICAvKipcbiAgICogQWxsb3cgdXNlcnMgdG8gc2lnbiBpbiB1c2luZyAnTG9naW4gV2l0aCBBbWF6b24nLlxuICAgKiBBIGBVc2VyUG9vbElkZW50aXR5UHJvdmlkZXJBbWF6b25gIG11c3QgYmUgYXR0YWNoZWQgdG8gdGhlIHVzZXIgcG9vbC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQU1BWk9OID0gbmV3IFVzZXJQb29sQ2xpZW50SWRlbnRpdHlQcm92aWRlcignTG9naW5XaXRoQW1hem9uJyk7XG5cbiAgLyoqXG4gICAqIEFsbG93IHVzZXJzIHRvIHNpZ24gaW4gZGlyZWN0bHkgYXMgYSB1c2VyIG9mIHRoZSBVc2VyIFBvb2xcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQ09HTklUTyA9IG5ldyBVc2VyUG9vbENsaWVudElkZW50aXR5UHJvdmlkZXIoJ0NPR05JVE8nKTtcblxuICAvKipcbiAgICogU3BlY2lmeSBhIHByb3ZpZGVyIG5vdCB5ZXQgc3VwcG9ydGVkIGJ5IHRoZSBDREsuXG4gICAqIEBwYXJhbSBuYW1lIG5hbWUgb2YgdGhlIGlkZW50aXR5IHByb3ZpZGVyIGFzIHJlY29nbml6ZWQgYnkgQ2xvdWRGb3JtYXRpb24gcHJvcGVydHkgYFN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzYFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBjdXN0b20obmFtZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBVc2VyUG9vbENsaWVudElkZW50aXR5UHJvdmlkZXIobmFtZSk7XG4gIH1cblxuICAvKiogVGhlIG5hbWUgb2YgdGhlIGlkZW50aXR5IHByb3ZpZGVyIGFzIHJlY29nbml6ZWQgYnkgQ2xvdWRGb3JtYXRpb24gcHJvcGVydHkgYFN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzYCAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IobmFtZTogc3RyaW5nKSB7XG4gICAgdGhpcy5uYW1lID0gbmFtZTtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gY3JlYXRlIGEgVXNlclBvb2xDbGllbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBVc2VyUG9vbENsaWVudE9wdGlvbnMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgYXBwbGljYXRpb24gY2xpZW50XG4gICAqIEBkZWZhdWx0IC0gY2xvdWRmb3JtYXRpb24gZ2VuZXJhdGVkIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sQ2xpZW50TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciB0byBnZW5lcmF0ZSBhIGNsaWVudCBzZWNyZXRcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGdlbmVyYXRlU2VjcmV0PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIHNldCBvZiBPQXV0aCBhdXRoZW50aWNhdGlvbiBmbG93cyB0byBlbmFibGUgb24gdGhlIGNsaWVudFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2duaXRvL2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hbWF6b24tY29nbml0by11c2VyLXBvb2xzLWF1dGhlbnRpY2F0aW9uLWZsb3cuaHRtbFxuICAgKiBAZGVmYXVsdCAtIGFsbCBhdXRoIGZsb3dzIGRpc2FibGVkXG4gICAqL1xuICByZWFkb25seSBhdXRoRmxvd3M/OiBBdXRoRmxvdztcblxuICAvKipcbiAgICogVHVybnMgb2ZmIGFsbCBPQXV0aCBpbnRlcmFjdGlvbnMgZm9yIHRoaXMgY2xpZW50LlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgZGlzYWJsZU9BdXRoPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogT0F1dGggc2V0dGluZ3MgZm9yIHRoaXMgdG8gY2xpZW50IHRvIGludGVyYWN0IHdpdGggdGhlIGFwcC5cbiAgICogQW4gZXJyb3IgaXMgdGhyb3duIHdoZW4gdGhpcyBpcyBzcGVjaWZpZWQgYW5kIGBkaXNhYmxlT0F1dGhgIGlzIHNldC5cbiAgICogQGRlZmF1bHQgLSBzZWUgZGVmYXVsdHMgaW4gYE9BdXRoU2V0dGluZ3NgLiBtZWFuaW5nbGVzcyBpZiBgZGlzYWJsZU9BdXRoYCBpcyBzZXQuXG4gICAqL1xuICByZWFkb25seSBvQXV0aD86IE9BdXRoU2V0dGluZ3M7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgQ29nbml0byByZXR1cm5zIGEgVXNlck5vdEZvdW5kRXhjZXB0aW9uIGV4Y2VwdGlvbiB3aGVuIHRoZVxuICAgKiB1c2VyIGRvZXMgbm90IGV4aXN0IGluIHRoZSB1c2VyIHBvb2wgKGZhbHNlKSwgb3Igd2hldGhlciBpdCByZXR1cm5zXG4gICAqIGFub3RoZXIgdHlwZSBvZiBlcnJvciB0aGF0IGRvZXNuJ3QgcmV2ZWFsIHRoZSB1c2VyJ3MgYWJzZW5jZS5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29nbml0by9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvY29nbml0by11c2VyLXBvb2wtbWFuYWdpbmctZXJyb3JzLmh0bWxcbiAgICogQGRlZmF1bHQgdHJ1ZSBmb3IgbmV3IHN0YWNrc1xuICAgKi9cbiAgcmVhZG9ubHkgcHJldmVudFVzZXJFeGlzdGVuY2VFcnJvcnM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgbGlzdCBvZiBpZGVudGl0eSBwcm92aWRlcnMgdGhhdCB1c2VycyBzaG91bGQgYmUgYWJsZSB0byB1c2UgdG8gc2lnbiBpbiB1c2luZyB0aGlzIGNsaWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBzdXBwb3J0cyBhbGwgaWRlbnRpdHkgcHJvdmlkZXJzIHRoYXQgYXJlIHJlZ2lzdGVyZWQgd2l0aCB0aGUgdXNlciBwb29sLiBJZiB0aGUgdXNlciBwb29sIGFuZC9vclxuICAgKiBpZGVudGl0eSBwcm92aWRlcnMgYXJlIGltcG9ydGVkLCBlaXRoZXIgc3BlY2lmeSB0aGlzIG9wdGlvbiBleHBsaWNpdGx5IG9yIGVuc3VyZSB0aGF0IHRoZSBpZGVudGl0eSBwcm92aWRlcnMgYXJlXG4gICAqIHJlZ2lzdGVyZWQgd2l0aCB0aGUgdXNlciBwb29sIHVzaW5nIHRoZSBgVXNlclBvb2wucmVnaXN0ZXJJZGVudGl0eVByb3ZpZGVyKClgIEFQSS5cbiAgICovXG4gIHJlYWRvbmx5IHN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzPzogVXNlclBvb2xDbGllbnRJZGVudGl0eVByb3ZpZGVyW107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgdGhlIFVzZXJQb29sQ2xpZW50IGNvbnN0cnVjdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFVzZXJQb29sQ2xpZW50UHJvcHMgZXh0ZW5kcyBVc2VyUG9vbENsaWVudE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIFVzZXJQb29sIHJlc291cmNlIHRoaXMgY2xpZW50IHdpbGwgaGF2ZSBhY2Nlc3MgdG9cbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sOiBJVXNlclBvb2w7XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIENvZ25pdG8gdXNlciBwb29sIGNsaWVudC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJVXNlclBvb2xDbGllbnQgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgYXBwbGljYXRpb24gY2xpZW50XG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IHVzZXJQb29sQ2xpZW50SWQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBEZWZpbmUgYSBVc2VyUG9vbCBBcHAgQ2xpZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBVc2VyUG9vbENsaWVudCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVVzZXJQb29sQ2xpZW50IHtcbiAgLyoqXG4gICAqIEltcG9ydCBhIHVzZXIgcG9vbCBjbGllbnQgZ2l2ZW4gaXRzIGlkLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVXNlclBvb2xDbGllbnRJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCB1c2VyUG9vbENsaWVudElkOiBzdHJpbmcpOiBJVXNlclBvb2xDbGllbnQge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVVzZXJQb29sQ2xpZW50IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSB1c2VyUG9vbENsaWVudElkID0gdXNlclBvb2xDbGllbnRJZDtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IHVzZXJQb29sQ2xpZW50SWQ6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBPQXV0aCBmbG93cyBlbmFibGVkIGZvciB0aGlzIGNsaWVudC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBvQXV0aEZsb3dzOiBPQXV0aEZsb3dzO1xuICBwcml2YXRlIHJlYWRvbmx5IF91c2VyUG9vbENsaWVudE5hbWU/OiBzdHJpbmc7XG5cbiAgLypcbiAgICogTm90ZSB0byBpbXBsZW1lbnRlcnM6IFR3byBDbG91ZEZvcm1hdGlvbiByZXR1cm4gdmFsdWVzIE5hbWUgYW5kIENsaWVudFNlY3JldCBhcmUgcGFydCBvZiB0aGUgc3BlYy5cbiAgICogSG93ZXZlciwgdGhleSBoYXZlIGJlZW4gZXhwbGljaXR5IG5vdCBpbXBsZW1lbnRlZCBoZXJlLiBUaGV5IGFyZSBub3QgZG9jdW1lbnRlZCBpbiBDbG91ZEZvcm1hdGlvbiwgYW5kXG4gICAqIENsb3VkRm9ybWF0aW9uIHJldHVybnMgdGhlIGZvbGxvd2luZyB0aGUgc3RyaW5nIHdoZW4gdGhlc2UgdHdvIGF0dHJpYnV0ZXMgYXJlICdHZXRBdHQnIC0gXCJhdHRyaWJ1dGUgbm90IHN1cHBvcnRlZFxuICAgKiBhdCB0aGlzIHRpbWUsIHBsZWFzZSB1c2UgdGhlIENMSSBvciBDb25zb2xlIHRvIHJldHJpZXZlIHRoaXMgdmFsdWVcIi5cbiAgICogQXdhaXRpbmcgdXBkYXRlcyBmcm9tIENsb3VkRm9ybWF0aW9uLlxuICAgKi9cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVXNlclBvb2xDbGllbnRQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBpZiAocHJvcHMuZGlzYWJsZU9BdXRoICYmIHByb3BzLm9BdXRoKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09BdXRoIHNldHRpbmdzIGNhbm5vdCBiZSBzcGVjaWZpZWQgd2hlbiBkaXNhYmxlT0F1dGggaXMgc2V0LicpO1xuICAgIH1cblxuICAgIHRoaXMub0F1dGhGbG93cyA9IHByb3BzLm9BdXRoPy5mbG93cyA/PyB7XG4gICAgICBpbXBsaWNpdENvZGVHcmFudDogdHJ1ZSxcbiAgICAgIGF1dGhvcml6YXRpb25Db2RlR3JhbnQ6IHRydWUsXG4gICAgfTtcblxuICAgIGxldCBjYWxsYmFja1VybHM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkID0gcHJvcHMub0F1dGg/LmNhbGxiYWNrVXJscztcbiAgICBpZiAodGhpcy5vQXV0aEZsb3dzLmF1dGhvcml6YXRpb25Db2RlR3JhbnQgfHwgdGhpcy5vQXV0aEZsb3dzLmltcGxpY2l0Q29kZUdyYW50KSB7XG4gICAgICBpZiAoY2FsbGJhY2tVcmxzID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgY2FsbGJhY2tVcmxzID0gWydodHRwczovL2V4YW1wbGUuY29tJ107XG4gICAgICB9IGVsc2UgaWYgKGNhbGxiYWNrVXJscy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdjYWxsYmFja1VybCBtdXN0IG5vdCBiZSBlbXB0eSB3aGVuIGNvZGVHcmFudCBvciBpbXBsaWNpdEdyYW50IE9BdXRoIGZsb3dzIGFyZSBlbmFibGVkLicpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmblVzZXJQb29sQ2xpZW50KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNsaWVudE5hbWU6IHByb3BzLnVzZXJQb29sQ2xpZW50TmFtZSxcbiAgICAgIGdlbmVyYXRlU2VjcmV0OiBwcm9wcy5nZW5lcmF0ZVNlY3JldCxcbiAgICAgIHVzZXJQb29sSWQ6IHByb3BzLnVzZXJQb29sLnVzZXJQb29sSWQsXG4gICAgICBleHBsaWNpdEF1dGhGbG93czogdGhpcy5jb25maWd1cmVBdXRoRmxvd3MocHJvcHMpLFxuICAgICAgYWxsb3dlZE9BdXRoRmxvd3M6IHByb3BzLmRpc2FibGVPQXV0aCA/IHVuZGVmaW5lZCA6IHRoaXMuY29uZmlndXJlT0F1dGhGbG93cygpLFxuICAgICAgYWxsb3dlZE9BdXRoU2NvcGVzOiBwcm9wcy5kaXNhYmxlT0F1dGggPyB1bmRlZmluZWQgOiB0aGlzLmNvbmZpZ3VyZU9BdXRoU2NvcGVzKHByb3BzLm9BdXRoKSxcbiAgICAgIGNhbGxiYWNrVXJMczogY2FsbGJhY2tVcmxzICYmIGNhbGxiYWNrVXJscy5sZW5ndGggPiAwID8gY2FsbGJhY2tVcmxzIDogdW5kZWZpbmVkLFxuICAgICAgYWxsb3dlZE9BdXRoRmxvd3NVc2VyUG9vbENsaWVudDogIXByb3BzLmRpc2FibGVPQXV0aCxcbiAgICAgIHByZXZlbnRVc2VyRXhpc3RlbmNlRXJyb3JzOiB0aGlzLmNvbmZpZ3VyZVByZXZlbnRVc2VyRXhpc3RlbmNlRXJyb3JzKHByb3BzLnByZXZlbnRVc2VyRXhpc3RlbmNlRXJyb3JzKSxcbiAgICAgIHN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzOiB0aGlzLmNvbmZpZ3VyZUlkZW50aXR5UHJvdmlkZXJzKHByb3BzKSxcbiAgICB9KTtcblxuICAgIHRoaXMudXNlclBvb2xDbGllbnRJZCA9IHJlc291cmNlLnJlZjtcbiAgICB0aGlzLl91c2VyUG9vbENsaWVudE5hbWUgPSBwcm9wcy51c2VyUG9vbENsaWVudE5hbWU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGNsaWVudCBuYW1lIHRoYXQgd2FzIHNwZWNpZmllZCB2aWEgdGhlIGB1c2VyUG9vbENsaWVudE5hbWVgIHByb3BlcnR5IGR1cmluZyBpbml0aWFsaXphdGlvbixcbiAgICogdGhyb3dzIGFuIGVycm9yIG90aGVyd2lzZS5cbiAgICovXG4gIHB1YmxpYyBnZXQgdXNlclBvb2xDbGllbnROYW1lKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMuX3VzZXJQb29sQ2xpZW50TmFtZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3VzZXJQb29sQ2xpZW50TmFtZSBpcyBhdmFpbGFibGUgb25seSBpZiBzcGVjaWZpZWQgb24gdGhlIFVzZXJQb29sQ2xpZW50IGR1cmluZyBpbml0aWFsaXphdGlvbicpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fdXNlclBvb2xDbGllbnROYW1lO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVBdXRoRmxvd3MocHJvcHM6IFVzZXJQb29sQ2xpZW50UHJvcHMpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgY29uc3QgYXV0aEZsb3dzOiBzdHJpbmdbXSA9IFtdO1xuICAgIGlmIChwcm9wcy5hdXRoRmxvd3M/LnVzZXJQYXNzd29yZCkgeyBhdXRoRmxvd3MucHVzaCgnQUxMT1dfVVNFUl9QQVNTV09SRF9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy5hZG1pblVzZXJQYXNzd29yZCkgeyBhdXRoRmxvd3MucHVzaCgnQUxMT1dfQURNSU5fVVNFUl9QQVNTV09SRF9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy5jdXN0b20pIHsgYXV0aEZsb3dzLnB1c2goJ0FMTE9XX0NVU1RPTV9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy51c2VyU3JwKSB7IGF1dGhGbG93cy5wdXNoKCdBTExPV19VU0VSX1NSUF9BVVRIJyk7IH1cbiAgICBpZiAocHJvcHMuYXV0aEZsb3dzPy5yZWZyZXNoVG9rZW4pIHsgYXV0aEZsb3dzLnB1c2goJ0FMTE9XX1JFRlJFU0hfVE9LRU5fQVVUSCcpOyB9XG5cbiAgICBpZiAoYXV0aEZsb3dzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgcmV0dXJuIGF1dGhGbG93cztcbiAgfVxuXG4gIHByaXZhdGUgY29uZmlndXJlT0F1dGhGbG93cygpOiBzdHJpbmdbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCh0aGlzLm9BdXRoRmxvd3MuYXV0aG9yaXphdGlvbkNvZGVHcmFudCB8fCB0aGlzLm9BdXRoRmxvd3MuaW1wbGljaXRDb2RlR3JhbnQpICYmIHRoaXMub0F1dGhGbG93cy5jbGllbnRDcmVkZW50aWFscykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdjbGllbnRDcmVkZW50aWFscyBPQXV0aCBmbG93IGNhbm5vdCBiZSBzZWxlY3RlZCBhbG9uZyB3aXRoIGNvZGVHcmFudCBvciBpbXBsaWNpdEdyYW50LicpO1xuICAgIH1cbiAgICBjb25zdCBvQXV0aEZsb3dzOiBzdHJpbmdbXSA9IFtdO1xuICAgIGlmICh0aGlzLm9BdXRoRmxvd3MuY2xpZW50Q3JlZGVudGlhbHMpIHsgb0F1dGhGbG93cy5wdXNoKCdjbGllbnRfY3JlZGVudGlhbHMnKTsgfVxuICAgIGlmICh0aGlzLm9BdXRoRmxvd3MuaW1wbGljaXRDb2RlR3JhbnQpIHsgb0F1dGhGbG93cy5wdXNoKCdpbXBsaWNpdCcpOyB9XG4gICAgaWYgKHRoaXMub0F1dGhGbG93cy5hdXRob3JpemF0aW9uQ29kZUdyYW50KSB7IG9BdXRoRmxvd3MucHVzaCgnY29kZScpOyB9XG5cbiAgICBpZiAob0F1dGhGbG93cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIHJldHVybiBvQXV0aEZsb3dzO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVPQXV0aFNjb3BlcyhvQXV0aD86IE9BdXRoU2V0dGluZ3MpOiBzdHJpbmdbXSB7XG4gICAgY29uc3Qgc2NvcGVzID0gb0F1dGg/LnNjb3BlcyA/PyBbT0F1dGhTY29wZS5QUk9GSUxFLCBPQXV0aFNjb3BlLlBIT05FLCBPQXV0aFNjb3BlLkVNQUlMLCBPQXV0aFNjb3BlLk9QRU5JRCxcbiAgICAgIE9BdXRoU2NvcGUuQ09HTklUT19BRE1JTl07XG4gICAgY29uc3Qgc2NvcGVOYW1lcyA9IG5ldyBTZXQoc2NvcGVzLm1hcCgoeCkgPT4geC5zY29wZU5hbWUpKTtcbiAgICBjb25zdCBhdXRvT3BlbklkU2NvcGVzID0gW09BdXRoU2NvcGUuUEhPTkUsIE9BdXRoU2NvcGUuRU1BSUwsIE9BdXRoU2NvcGUuUFJPRklMRV07XG4gICAgaWYgKGF1dG9PcGVuSWRTY29wZXMucmVkdWNlKChhZ2csIHMpID0+IGFnZyB8fCBzY29wZU5hbWVzLmhhcyhzLnNjb3BlTmFtZSksIGZhbHNlKSkge1xuICAgICAgc2NvcGVOYW1lcy5hZGQoT0F1dGhTY29wZS5PUEVOSUQuc2NvcGVOYW1lKTtcbiAgICB9XG4gICAgcmV0dXJuIEFycmF5LmZyb20oc2NvcGVOYW1lcyk7XG4gIH1cblxuICBwcml2YXRlIGNvbmZpZ3VyZVByZXZlbnRVc2VyRXhpc3RlbmNlRXJyb3JzKHByZXZlbnQ/OiBib29sZWFuKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAocHJldmVudCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICByZXR1cm4gcHJldmVudCA/ICdFTkFCTEVEJyA6ICdMRUdBQ1knO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25maWd1cmVJZGVudGl0eVByb3ZpZGVycyhwcm9wczogVXNlclBvb2xDbGllbnRQcm9wcyk6IHN0cmluZ1tdIHwgdW5kZWZpbmVkIHtcbiAgICBsZXQgcHJvdmlkZXJzOiBzdHJpbmdbXTtcbiAgICBpZiAoIXByb3BzLnN1cHBvcnRlZElkZW50aXR5UHJvdmlkZXJzKSB7XG4gICAgICBjb25zdCBwcm92aWRlclNldCA9IG5ldyBTZXQocHJvcHMudXNlclBvb2wuaWRlbnRpdHlQcm92aWRlcnMubWFwKChwKSA9PiBwLnByb3ZpZGVyTmFtZSkpO1xuICAgICAgcHJvdmlkZXJTZXQuYWRkKCdDT0dOSVRPJyk7XG4gICAgICBwcm92aWRlcnMgPSBBcnJheS5mcm9tKHByb3ZpZGVyU2V0KTtcbiAgICB9IGVsc2Uge1xuICAgICAgcHJvdmlkZXJzID0gcHJvcHMuc3VwcG9ydGVkSWRlbnRpdHlQcm92aWRlcnMubWFwKChwKSA9PiBwLm5hbWUpO1xuICAgIH1cbiAgICBpZiAocHJvdmlkZXJzLmxlbmd0aCA9PT0gMCkgeyByZXR1cm4gdW5kZWZpbmVkOyB9XG4gICAgcmV0dXJuIEFycmF5LmZyb20ocHJvdmlkZXJzKTtcbiAgfVxufVxuIl19