"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * The type of the CodeBuild action that determines its CodePipeline Category -
 * Build, or Test.
 * The default is Build.
 */
var CodeBuildActionType;
(function (CodeBuildActionType) {
    /**
     * The action will have the Build Category.
     * This is the default.
     */
    CodeBuildActionType[CodeBuildActionType["BUILD"] = 0] = "BUILD";
    /**
     * The action will have the Test Category.
     */
    CodeBuildActionType[CodeBuildActionType["TEST"] = 1] = "TEST";
})(CodeBuildActionType = exports.CodeBuildActionType || (exports.CodeBuildActionType = {}));
/**
 * CodePipeline build action that uses AWS CodeBuild.
 */
class CodeBuildAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: props.type === CodeBuildActionType.TEST
                ? codepipeline.ActionCategory.TEST
                : codepipeline.ActionCategory.BUILD,
            provider: 'CodeBuild',
            artifactBounds: { minInputs: 1, maxInputs: 5, minOutputs: 0, maxOutputs: 5 },
            inputs: [props.input, ...props.extraInputs || []],
            resource: props.project,
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the CodeBuild project this action points to.
     * Variables in CodeBuild actions are defined using the 'exported-variables' subsection of the 'env'
     * section of the buildspec.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'exported-variables' section of the buildspec
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec-ref-syntax
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // check for a cross-account action if there are any outputs
        if ((this.actionProperties.outputs || []).length > 0) {
            const pipelineStack = cdk.Stack.of(scope);
            const projectStack = cdk.Stack.of(this.props.project);
            if (pipelineStack.account !== projectStack.account) {
                throw new Error('A cross-account CodeBuild action cannot have outputs. ' +
                    'This is a known CodeBuild limitation. ' +
                    'See https://github.com/aws/aws-cdk/issues/4169 for details');
            }
        }
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (this.props.project.role) {
            if ((this.actionProperties.outputs || []).length > 0) {
                options.bucket.grantReadWrite(this.props.project);
            }
            else {
                options.bucket.grantRead(this.props.project);
            }
        }
        if (this.props.project instanceof codebuild.Project) {
            this.props.project.bindToCodePipeline(scope, {
                artifactBucket: options.bucket,
            });
        }
        const configuration = {
            ProjectName: this.props.project.projectName,
            EnvironmentVariables: this.props.environmentVariables &&
                cdk.Stack.of(scope).toJsonString(codebuild.Project.serializeEnvVariables(this.props.environmentVariables)),
        };
        if ((this.actionProperties.inputs || []).length > 1) {
            // lazy, because the Artifact name might be generated lazily
            configuration.PrimarySource = cdk.Lazy.stringValue({ produce: () => this.props.input.artifactName });
        }
        return {
            configuration,
        };
    }
}
exports.CodeBuildAction = CodeBuildAction;
//# sourceMappingURL=data:application/json;base64,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