# -*- coding: utf-8 -*-
from typing import Type, Optional, Callable

from outflow.core.generic.lazy_object_proxy import LazyObjectProxy
from outflow.core.logging import logger
from outflow.core.pipeline import context
from outflow.core.workflow import as_workflow
from outflow.library.workflows.base_map_workflow import BaseMapWorkflow


def determine_map_class() -> Type[BaseMapWorkflow]:
    logger.debug(
        f"Generic MapWorkflow initializing a {context.backend_name} MapWorkflow"
    )

    if context.backend_name == "default":
        from outflow.library.workflows.sequential_map_workflow import (
            SequentialMapWorkflow,
        )

        return SequentialMapWorkflow

    elif context.backend_name == "parallel":
        from outflow.library.workflows.parallel_map_workflow import ParallelMapWorkflow

        return ParallelMapWorkflow

    elif context.backend_name == "slurm":
        from outflow.slurm.map_workflow import SlurmMapWorkflow

        return SlurmMapWorkflow

    else:
        raise AttributeError("Could not determine backend for generic MapWorkflow")


class LazyMapWorkflow(LazyObjectProxy):
    """
    This class is a LazyObjectProxy subclass used for generic MapWorkflow.
    as_workflow method is overriden for this class so that when called, the
    proxy function generated by this call will resolve the real MapWorkflow
    class _at_ the proxy call, ie inside setup_tasks / inside a pipeline context.

    MapWorkflow being a LazyObjectProxy, it will also resolve to the right
    class when used directly inside a pipeline context, when instantiating
    directly or through the with construct.
    """

    @staticmethod
    def as_workflow(run_func=None, **kwargs):
        return as_workflow(
            run_func=run_func,
            cls=lazy_map_type_helper(determine_map_class),
            **kwargs,
        )


def lazy_map_type_helper(
    factory: Optional[Callable[..., Type[BaseMapWorkflow]]] = None
) -> Type[BaseMapWorkflow]:
    """
    This function is only to provide the right type for MapWorkflow, especially
    for IDE hints.
    """

    return LazyMapWorkflow(factory)


MapWorkflow = lazy_map_type_helper(determine_map_class)
