import os
from typing import Literal
import pandas as pd
import logging


from capcruncher.tools.io import parse_bam
from capcruncher.tools.filter import CCSliceFilter, TriCSliceFilter, TiledCSliceFilter

SLICE_FILTERS = {
    "capture": CCSliceFilter,
    "tri": TriCSliceFilter,
    "tiled": TiledCSliceFilter,
}


# @get_timing(task_name="merging annotations with BAM input")
def merge_annotations(df: pd.DataFrame, annotations: os.PathLike) -> pd.DataFrame:
    """Combines annotations with the parsed bam file output.

    Uses pandas outer join on the indexes to merge annotations
    e.g. number of capture probe overlaps.

    Annotation tsv must have the index as the first column and this index
    must have intersecting keys with the first dataframe's index.


    Args:
     df (pd.DataFrame): Dataframe to merge with annotations
     annotations (os.PathLike): Filename of .tsv to read and merge with df

    Returns:
     pd.DataFrame: Merged dataframe
    """
    if annotations.endswith(".tsv"):
        df_ann = pd.read_csv(
            annotations,
            sep="\t",
            header=0,
            index_col=["slice_name", "chrom", "start"],
            low_memory=False,
        )

    elif annotations.endswith(".hdf5"):
        df_ann = pd.read_hdf(annotations, "annotation").set_index(
            ["slice_name", "chrom", "start"]
        )

    elif annotations.endswith(".parquet"):
        df_ann = pd.read_parquet(annotations).set_index(
            ["slice_name", "chrom", "start"]
        )

    df_ann = df_ann.drop(columns="end", errors="ignore")

    df = df.join(df_ann, how="inner").reset_index()

    return df


# @get_timing(task_name="analysis of bam file")
def filter(
    bam: os.PathLike,
    annotations: os.PathLike,
    custom_filtering: os.PathLike = None,
    output_prefix: os.PathLike = "reporters",
    output_format: Literal["tsv", "hdf5", "parquet"] = "parquet",
    stats_prefix: os.PathLike = "",
    method: str = "capture",
    sample_name: str = "",
    read_type: str = "",
    fragments: bool = True,
    read_stats: bool = True,
    slice_stats: bool = True,
    cis_and_trans_stats: bool = True,
):
    """
    Removes unwanted aligned slices and identifies reporters.

    Parses a BAM file and merges this with a supplied annotation to identify unwanted slices.
    Filtering can be tuned for Capture-C, Tri-C and Tiled-C data to ensure optimal filtering.

    Common filters include:

    - Removal of unmapped slices
    - Removal of excluded/blacklisted slices
    - Removal of non-capture fragments
    - Removal of multi-capture fragments
    - Removal of non-reporter fragments
    - Removal of fragments with duplicated coordinates.

    For specific filtering for each of the three methods see:

    - :class:`CCSliceFilter <capcruncher.tools.filter.CCSliceFilter>`
    - :class:`TriCSliceFilter <capcruncher.tools.filter.TriCSliceFilter>`
    - :class:`TiledCSliceFilter <capcruncher.tools.filter.TiledCSliceFilter>`


    In addition to outputting valid reporter fragments and slices separated by capture probe,
    this script also provides statistics on the number of read/slices filtered at each stage,
    and the number of cis/trans reporters for each probe.

    Notes:

     Whilst the script is capable of processing any annotations in tsv format, provided
     that the correct columns are present. It is highly recomended that the "annotate"
     subcomand is used to generate this file.

     Slice filtering is currently hard coded into each filtering class. This will be
     modified in a future update to enable custom filtering orders.


    \f
    Args:
     bam (os.PathLike): Input bam file to analyse
     annotations (os.PathLike): Annotations file generated by slices-annotate
     custom_filtering (os.PathLike): Allows for custom filtering to be performed. A yaml file is used to supply this ordering.
     output_prefix (os.PathLike, optional): Output file prefix. Defaults to "reporters".
     stats_prefix (os.PathLike, optional): Output stats prefix. Defaults to "".
     method (str, optional): Analysis method. Choose from (capture|tri|tiled). Defaults to "capture".
     sample_name (str, optional): Sample being processed e.g. DOX-treated_1. Defaults to "".
     read_type (str, optional): Process combined(flashed) or non-combined reads (pe) used for statistics. Defaults to "".
     gzip (bool, optional): Compress output with gzip. Defaults to False.
     fragments (bool, optional): Enables fragments to be output. Defaults to True.
     read_stats (bool, optional): Enables read level statistics to be output. Defaults to True.
     slice_stats (bool, optional): Enables slice level statistics to be output. Defaults to True.
     cis_and_trans_stats (bool, optional): Enables cis/trans statistics to be output. Defaults to True.
    """

    # Read bam file and merege annotations
    logging.info("Loading bam file")
    df_alignment = parse_bam(bam)
    logging.info("Merging bam file with annotations")
    df_alignment = merge_annotations(df_alignment, annotations)

    # Initialise SliceFilter
    # If no custom filtering, will use the class default.
    slice_filter_class = SLICE_FILTERS[method]
    slice_filter = slice_filter_class(
        slices=df_alignment,
        sample_name=sample_name,
        read_type=read_type,
        filter_stages=custom_filtering,
    )

    # Filter slices using the slice_filter
    logging.info(f"Filtering slices with method: {method}")
    slice_filter.filter_slices()

    if slice_stats:
        slice_filter.filter_stats.to_csv(f"{stats_prefix}.slice.stats.csv", index=False)

    if read_stats:
        slice_filter.read_stats.to_csv(f"{stats_prefix}.read.stats.csv", index=False)

    # Save reporter stats
    if cis_and_trans_stats:
        logging.info(f"Writing reporter statistics")
        slice_filter.cis_or_trans_stats.to_csv(
            f"{stats_prefix}.reporter.stats.csv", index=False
        )

    # Output slices filtered by viewpoint

    df_slices = slice_filter.slices
    df_slices_with_viewpoint = slice_filter.slices_with_viewpoint
    df_capture = slice_filter.captures

    if fragments:
        logging.info(f"Writing reporters at the fragment level")
        df_fragments = (
            slice_filter_class(df_slices)
            .fragments.join(
                df_capture["capture"], lsuffix="_slices", rsuffix="_capture"
            )
            .rename(
                columns={"capture_slices": "capture", "capture_capture": "viewpoint"}
            )
            .assign(id=lambda df: df["id"].astype("int64"))  # Enforce type
        )

        if output_format == "tsv":
            df_fragments.to_csv(f"{output_prefix}.fragments.tsv", sep="\t", index=False)
        elif output_format == "hdf5":
            df_fragments.to_hdf(
                f"{output_prefix}.slices.hdf5",
                key="fragments",
                data_columns=["viewpoint"],
                format="table",
                complib="blosc",
                complevel=2,
            )
        elif output_format == "parquet":
            if not df_fragments.empty:
                df_fragments.to_parquet(
                    f"{output_prefix}.fragments.parquet",
                    compression="snappy",
                    engine="pyarrow",
                )

    logging.info(f"Writing reporters slices")
    # Enforce dtype for parent_id
    df_slices_with_viewpoint = df_slices_with_viewpoint.assign(
        parent_id=lambda df: df["parent_id"].astype("int64")
    )

    if output_format == "tsv":
        df_slices_with_viewpoint.to_csv(
            f"{output_prefix}.slices.tsv", sep="\t", index=False
        )
    elif output_format == "hdf5":
        df_slices_with_viewpoint.to_hdf(
            f"{output_prefix}.slices.hdf5",
            key="slices",
            data_columns=["viewpoint"],
            format="table",
            complib="blosc",
            complevel=2,
        )
    elif output_format == "parquet":

        if not df_slices_with_viewpoint.empty:
            df_slices_with_viewpoint.to_parquet(
                f"{output_prefix}.slices.parquet",
                compression="snappy",
                engine="pyarrow",
            )

    logging.info(f"Completed analysis of bam file")
