"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComprehendGenericSyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_event_sources_1 = require("aws-cdk-lib/aws-lambda-event-sources");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
/**
 * Calls a Comprehend Classification endpoint and parses the result, filters on > 50 % confidence and sets the highest confidence score classification
 *
 * Input: "textract_result"."txt_output_location"
 * Output:  { "documentType": "AWS_PAYSTUBS" } (example will be at "classification"."documentType")
 *
 * Example (Python)
 * ```python
    comprehend_sync_task = tcdk.ComprehendGenericSyncSfnTask(
        self,
        "Classification",
        comprehend_classifier_arn=
        '<your comprehend classifier arn>',
        integration_pattern=sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        lambda_log_level="DEBUG",
        timeout=Duration.hours(24),
        input=sfn.TaskInput.from_object({
            "Token":
            sfn.JsonPath.task_token,
            "ExecutionId":
            sfn.JsonPath.string_at('$$.Execution.Id'),
            "Payload":
            sfn.JsonPath.entire_payload,
        }),
        result_path="$.classification")
    ```
 */
class ComprehendGenericSyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, ComprehendGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        var lambdaTimeout = props.lambdaTimeout === undefined ? 300 : props.lambdaTimeout;
        var lambdaMemory = props.lambdaMemory === undefined ? 256 : props.lambdaMemory;
        const duration = 20;
        this.comprehendSyncSQS = new sqs.Queue(this, 'ComprehendRequests', {
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(duration),
        });
        this.textractPutOnSQSFunction = new lambda.DockerImageFunction(this, 'PutOnSQS', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/put_on_sqs/')),
            architecture: lambda.Architecture.X86_64,
            memorySize: lambdaMemory,
            timeout: aws_cdk_lib_1.Duration.seconds(lambdaTimeout),
            environment: {
                LOG_LEVEL: lambdaLogLevel,
                COMPREHEND_CLASSIFIER_ARN: props.comprehendClassifierArn,
                SQS_QUEUE_URL: this.comprehendSyncSQS.queueUrl,
            },
        });
        this.textractPutOnSQSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sqs:SendMessage'], resources: [this.comprehendSyncSQS.queueArn] }));
        this.putOnSQSLambdaLogGroup = this.textractPutOnSQSFunction.logGroup;
        const putOnSQSFunctionInvoke = new tasks.LambdaInvoke(this, 'PutOnSQSFunctionInvoke', {
            lambdaFunction: this.textractPutOnSQSFunction,
        });
        this.comprehendSyncCallFunction = new lambda.DockerImageFunction(this, 'ComprehendSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/comprehend_sync/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(duration),
            environment: {
                LOG_LEVEL: lambdaLogLevel,
                COMPREHEND_CLASSIFIER_ARN: props.comprehendClassifierArn,
                SQS_QUEUE_URL: this.comprehendSyncSQS.queueName,
            },
        });
        this.comprehendSyncCallFunction.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.comprehendSyncSQS, { batchSize: 1 }));
        this.comprehendSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['comprehend:ClassifyDocument'], resources: ['*'] }));
        this.comprehendSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3:GetObject', 's3:ListBucket', 's3:PutObject'], resources: ['*'],
        }));
        this.comprehendSyncLambdaLogGroup = this.comprehendSyncCallFunction.logGroup;
        const workflow_chain = sfn.Chain.start(putOnSQSFunctionInvoke);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.comprehendSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure', 'states:SendTaskSuccess',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.ComprehendGenericSyncSfnTask = ComprehendGenericSyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
ComprehendGenericSyncSfnTask[_a] = { fqn: "amazon-textract-idp-cdk-constructs.ComprehendGenericSyncSfnTask", version: "0.0.11" };
ComprehendGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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