import { custom_resources, aws_iam as iam, CustomResource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The supported OrgObject types.
 *
 * @stability stable
 */
export declare enum OrgObjectTypes {
    /**
     * @stability stable
     */
    OU = "ou",
    /**
     * @stability stable
     */
    ACCOUNT = "account"
}
/**
 * The structure of an OrgObject
 */
export declare type OrgObject = {
    name: string;
    type: OrgObjectTypes;
    children: OrgObject[];
};
/**
 * @function processOrgObj
 * Function to process an OrgObject and create the corresponding AWS resources
 *
 * @param {Construct} this The construct resources will be added to.
 * @param {custom_resources.Provider} provider The custom resource provider the custom resources will utilized to create the resources.
 * @param {OrgObject} obj The OrgObject to process.
 * @param {string | OrganizationOU} parent The parent of the OrgObject. This is either a string, like for the org root, or an OrganizationOU object from the same stack.
 */
export declare function processOrgObj(this: Construct, provider: custom_resources.Provider, obj: OrgObject, parent: string | OrganizationOU): void;
/**
 * The properties for the OU custom resource provider.
 *
 * @stability stable
 */
export interface OrganizationOUProviderProps {
    /**
     * The role the custom resource should use for taking actions on OUs if one is not provided one will be created automatically.
     *
     * @stability stable
     */
    readonly role?: iam.IRole;
}
/**
 * The provider for OU custom resources.
 *
 * This creates a lambda function that handles custom resource requests for creating/updating/deleting OUs.
 *
 * @stability stable
 */
export declare class OrganizationOUProvider extends Construct {
    /**
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: OrganizationOUProviderProps);
}
/**
 * The properties of an OrganizationOU custom resource.
 *
 * @stability stable
 */
export interface OrganizationOUProps {
    /**
     * The name of the OU.
     *
     * @stability stable
     */
    readonly name: string;
    /**
     * The parent OU id.
     *
     * @stability stable
     */
    readonly parent: string | OrganizationOU;
    /**
     * The provider to use for the custom resource that will create the OU.
     *
     * You can create a provider with the OrganizationOuProvider class
     *
     * @stability stable
     */
    readonly provider: custom_resources.Provider;
    /**
     * Whether or not to import an existing OU if the new OU is a duplicate.
     *
     * If this is false and the OU already exists an error will be thrown.
     *
     * @default false
     * @stability stable
     */
    readonly importOnDuplicate?: boolean;
    /**
     * Whether or not to merge an OU with a duplicate when an OU is moved between parent OUs.
     *
     * If this is false and the OU already exists an error will be thrown.
     * If this is true and the OU already exists the accounts in the OU will be moved to the existing OU
     * and the duplicate, now empty, OU will be deleted.
     *
     * @default false
     * @stability stable
     */
    readonly allowMergeOnMove?: boolean;
    /**
     * Whether or not a missing OU should be recreated during an update.
     *
     * If this is false and the OU does not exist an error will be thrown when you try to update it.
     *
     * @stability stable
     */
    readonly allowRecreateOnUpdate?: boolean;
}
/**
 * The construct to create or update an Organization OU.
 *
 * This relies on the custom resource provider OrganizationOUProvider.
 *
 * @stability stable
 */
export declare class OrganizationOU extends Construct {
    /**
     * @stability stable
     */
    readonly resource: CustomResource;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: OrganizationOUProps);
}
