##
# UEFI Authenticated Variable Structure Support Library Unit Test
#
# Copyright (c) Microsoft Corporation
#
# SPDX-License-Identifier: BSD-2-Clause-Patent
##

# spell-checker: ignore deduplicated, deduplication, Dedpulication

import unittest
import uuid
import io
from edk2toollib.uefi.authenticated_variables_structure_support import \
    EfiSignatureDatabase, EfiSignatureList, EfiSignatureDataEfiCertSha256,\
    EfiSignatureDataEfiCertX509, EfiSignatureDataFactory
# import logging

VERBOSE = False

DEBIAN_CERT = '308202FC308201E4A003020102020500A7468DEF300D06092A864886F70D01010B05003020311E301C060355040313154465' \
              '6269616E2053656375726520426F6F74204341301E170D3136303831363138323235305A170D323630383136313832323530' \
              '5A3024312230200603550403131944656269616E2053656375726520426F6F74205369676E657230820122300D06092A8648' \
              '86F70D01010105000382010F003082010A0282010100D3D183900FDA65A22F075A6095EBF7C7867C2086DA65A3A612EB5B3B' \
              'CEC8FB3FA1724B9EDF50C50333A40C2B5FD641040DB6CF9548ED8AB2ADD6E501374E60CDB24A3804B3448094AF9F6E54DBA8' \
              '1F3CB74B30DE21816F09A366BA6A2B96D69A61770CD4ED3CD071BBAD8CF0225C3E25CC6D222E619795AF9B2E4D58B67E7802' \
              'C30EB9FAB25B27DE7DA2BE0C14AC73EC97B0155EEDEDE5A5753F78E071CE2FCE83ED533130984EE6F901A28888A623087C0D' \
              'B7543A1695ED5E795E904EFECDAADE82FCF696714E4949B9D3E9B0AB7FD72A47B75330277CDC6698096FD17EF57F3D3ED4A2' \
              '6A8859022F2F3DC8C628DE42FED9523D24C2FC409811F676BF8CBB650203010001A3393037301106096086480186F8420101' \
              '04040302041030150603551D25040E300C060A2B0601040182370A0301300B0603551D0F040403020780300D06092A864886' \
              'F70D01010B05000382010100571BA4604C29E9F27D6B5C93DBCC6C9F183F69489A75DE64F3834A09A92621EEE9565DE13ED9' \
              '75CBCC7FBF4DE4E8893D7E11428740C3D5E07179DC006CE17162C798C2CB270B2F9FCCECFA8BB2F30B9EF3F2C3C99FDB2593' \
              '90A4CDBB01E58EF4D755A8B4754131FD4E5D0318A0C2ACC5DE46E7DC1CCF12D59DE8479D938C32CD44D574C7309A57A556D0' \
              '7ECF0511B4F4F329F9DB9B53D2BD2FAD6A75264564BABA2896878EB7F07957FA7A0E3C4A3892BCF295F2E728D0F7D8981A5E' \
              '399EB56580BDF3DA123F507667299FD10B0A1E87975C72DBF301744ADD07BA76E96AFCDD22DB4602D7AF0AC5ED15BC0F2BA9' \
              'DB8DBF7F6FADA2B7C54D4A47B3C15690B617'

UBUNTU_CERT = '3082042030820308A003020102020101300D06092A864886F70D01010B0500308184310B3009060355040613024742311430' \
              '1206035504080C0B49736C65206F66204D616E3110300E06035504070C07446F75676C617331173015060355040A0C0E4361' \
              '6E6F6E6963616C204C74642E3134303206035504030C2B43616E6F6E6963616C204C74642E204D6173746572204365727469' \
              '66696361746520417574686F72697479301E170D3132303431323131333930385A170D3432303431313131333930385A307F' \
              '310B30090603550406130247423114301206035504080C0B49736C65206F66204D616E31173015060355040A0C0E43616E6F' \
              '6E6963616C204C74642E31143012060355040B0C0B53656375726520426F6F74312B302906035504030C2243616E6F6E6963' \
              '616C204C74642E2053656375726520426F6F74205369676E696E6730820122300D06092A864886F70D01010105000382010F' \
              '003082010A0282010100C95F9B628F0BB06482ACBEC9E262E34BD29F1E8AD5611A2B5D38F4B7CEB99AB843B8439777AB4F7F' \
              '0C70460BFC7F6DC66DEA805E01D2B7661E87DE0D6DD04197A8A5AF0C634FF77CC252CCA031A9BB895D991E466F5573B97669' \
              'ECD7C1FC21D6C607E74FBD22DEE4A85B2DDB95341997D6284B214CCABB1D79A6177F5AF967E65C78453D106DB017592611C5' \
              '57E37F4E82BAF62C4EC8374DFF85158447E0ED3B7C7FBCAFE90105A70C6FC3E98DA3CEBEA6E3CD3CB5582C9EC2031C602237' \
              '39FF4102C129A46551FF3334AA4215F99578FC2DF5DA8A857C829DFB372C6BA5A8DF7C550B802E3CB063E1CD384889E81406' \
              '0B82BCFDD407681B0F3ED915DD94111B0203010001A381A030819D300C0603551D130101FF04023000301F0603551D250418' \
              '301606082B06010505070303060A2B0601040182370A0306302C06096086480186F842010D041F161D4F70656E53534C2047' \
              '656E657261746564204365727469666963617465301D0603551D0E0416041461482AA2830D0AB2AD5AF10B7250DA9033DDCE' \
              'F0301F0603551D23041830168014AD91990BC22AB1F517048C23B6655A268E345A63300D06092A864886F70D01010B050003' \
              '820101008F8AA1061F29B70A4AD5C5FD81AB25EAC07DE2FC6A96A0799367EE050E251225E45AF6AA1AF112F3058D875EF15A' \
              '5CCB8D2373651D15B9DE226BD64967C9A3C6D7624E5CB5F903834081DC879C3C3F1C0D519F94650A844867E4A2F8A64AF0E7' \
              'CDCDBD94E309D25D2D161B05150BCB44B43E614222C42A5C4EC51DA3E2E052B2EBF48B2BDC38395DFB88A156655F2B4F26FF' \
              '06781012EB8C5D32E3C645AF259BA0FF8EEF4709A3E98B37929269767E343B9205674EB025EDBC5E5F8FB4D6CA40FFE4E231' \
              '230C8525AE0C5501ECE5475EDF5BBC1433E3C6F518B6D9F7DDB3B4A131D35A5C5D7D3EBF0AE4E4E8B4597D3BB48CA31BB520' \
              'A3B93E846F8C2100C339'

HASHSTR = [
    '0000000000000000000000000000000000000000000000000000000000000000',
    '1111111111111111111111111111111111111111111111111111111111111111',
    '2222222222222222222222222222222222222222222222222222222222222222',
    '3333333333333333333333333333333333333333333333333333333333333333',
    '4444444444444444444444444444444444444444444444444444444444444444',
    '5555555555555555555555555555555555555555555555555555555555555555',
    '6666666666666666666666666666666666666666666666666666666666666666',
    '7777777777777777777777777777777777777777777777777777777777777777',
    '8888888888888888888888888888888888888888888888888888888888888888',
    '9999999999999999999999999999999999999999999999999999999999999999'
]


def sha256_esl_builder(initList: []):
    """
    Creates an EfiSignatureList of type EFI_CERT_SHA256 from a list of (hash, sigowner) tuples

    initList is [ (hashString1, ownerGuidString1), ...]
    """

    Esl = EfiSignatureList(typeguid=EfiSignatureDataFactory.EFI_CERT_SHA256_GUID)
    SignatureSize = EfiSignatureDataEfiCertSha256.STATIC_STRUCT_SIZE if initList else 0
    Esl.AddSignatureHeader(SigHeader=None, SigSize=SignatureSize)
    for entry in initList:
        (hashStr, ownerGuidStr) = entry
        hashBytes = bytes.fromhex(hashStr)
        Esl.AddSignatureData(EfiSignatureDataEfiCertSha256(digest=hashBytes,
                                                           sigowner=uuid.UUID(ownerGuidStr)))
    return Esl


def x509_esl_builder(initList: []):
    """
    Creates an EfiSignatureList of type EFI_CERT_X509 from a tuple containing x509 and signatureowner hex strings

    initList is [ (hashString1, ownerGuidString1), ...]
    NOTE: x509 EfiSignatureLists only support 1 data element, so initList should either be empty or 1 element.
    The interface is a List for consistency with other esl_builders
    """

    listSize = len(initList)

    if (listSize > 1):
        raise Exception("initList length > 1, unsupported by type")

    Esl = EfiSignatureList(typeguid=EfiSignatureDataFactory.EFI_CERT_X509_GUID)

    if (listSize == 0):
        Esl.AddSignatureHeader(SigHeader=None, SigSize=0)
    else:
        (certHexStr, ownerGuidStr) = initList[0]
        certBytes = bytes.fromhex(certHexStr)
        Esl.AddSignatureHeader(SigHeader=None, SigSize=EfiSignatureDataEfiCertX509.STATIC_STRUCT_SIZE + len(certBytes))
        Esl.AddSignatureData(EfiSignatureDataEfiCertX509(cert=certBytes,
                                                         sigowner=uuid.UUID(ownerGuidStr)))
    return Esl


def basicEfiSignatureDatabase():
    owner = uuid.uuid4().hex
    certInput = [(DEBIAN_CERT, owner)]
    cert = x509_esl_builder(certInput)

    hashEslInput = [
        (HASHSTR[8], owner),
        (HASHSTR[0], owner),
        (HASHSTR[0], owner),
        (HASHSTR[5], owner),
        (HASHSTR[5], owner),
        (HASHSTR[5], owner),
        (HASHSTR[1], owner),
        (HASHSTR[2], owner),
        (HASHSTR[1], owner),
        (HASHSTR[2], owner)
    ]

    hashes = sha256_esl_builder(hashEslInput)

    EslList = [
        cert, hashes
    ]

    return EfiSignatureDatabase(EslList=EslList)


def BootHoleData():
    """ Returns a tuple of the (rawData, expectedDupes, expectedCanonical) for the BootHole dbx generated by EDK2 """

    owner = '77fa9abd-0359-4d32-bd60-28f4e78f784b'

    # First prepare EfiSignatureDatabase that is unsorted, unmerged, and contains duplicates
    originalHashes = [
        ('80B4D96931BF0D02FD91A61E19D14F1DA452E66DB2408CA8604D411F92659F0A', owner),
        ('F52F83A3FA9CFBD6920F722824DBE4034534D25B8507246B3B957DAC6E1BCE7A', owner),
        ('C5D9D8A186E2C82D09AFAA2A6F7F2E73870D3E64F72C4E08EF67796A840F0FBD', owner),
        ('363384D14D1F2E0B7815626484C459AD57A318EF4396266048D058C5A19BBF76', owner),
        ('1AEC84B84B6C65A51220A9BE7181965230210D62D6D33C48999C6B295A2B0A06', owner),
        ('E6CA68E94146629AF03F69C2F86E6BEF62F930B37C6FBCC878B78DF98C0334E5', owner),
        ('C3A99A460DA464A057C3586D83CEF5F4AE08B7103979ED8932742DF0ED530C66', owner),
        ('58FB941AEF95A25943B3FB5F2510A0DF3FE44C58C95E0AB80487297568AB9771', owner),
        ('5391C3A2FB112102A6AA1EDC25AE77E19F5D6F09CD09EEB2509922BFCD5992EA', owner),
        ('D626157E1D6A718BC124AB8DA27CBB65072CA03A7B6B257DBDCBBD60F65EF3D1', owner),
        ('D063EC28F67EBA53F1642DBF7DFF33C6A32ADD869F6013FE162E2C32F1CBE56D', owner),
        ('29C6EB52B43C3AA18B2CD8ED6EA8607CEF3CFAE1BAFE1165755CF2E614844A44', owner),
        ('90FBE70E69D633408D3E170C6832DBB2D209E0272527DFB63D49D29572A6F44C', owner),
        ('075EEA060589548BA060B2FEED10DA3C20C7FE9B17CD026B94E8A683B8115238', owner),
        ('07E6C6A858646FB1EFC67903FE28B116011F2367FE92E6BE2B36999EFF39D09E', owner),
        ('09DF5F4E511208EC78B96D12D08125FDB603868DE39F6F72927852599B659C26', owner),
        ('0BBB4392DAAC7AB89B30A4AC657531B97BFAAB04F90B0DAFE5F9B6EB90A06374', owner),
        ('0C189339762DF336AB3DD006A463DF715A39CFB0F492465C600E6C6BD7BD898C', owner),
        ('0D0DBECA6F29ECA06F331A7D72E4884B12097FB348983A2A14A0D73F4F10140F', owner),
        ('0DC9F3FB99962148C3CA833632758D3ED4FC8D0B0007B95B31E6528F2ACD5BFC', owner),
        ('106FACEACFECFD4E303B74F480A08098E2D0802B936F8EC774CE21F31686689C', owner),
        ('174E3A0B5B43C6A607BBD3404F05341E3DCF396267CE94F8B50E2E23A9DA920C', owner),
        ('18333429FF0562ED9F97033E1148DCEEE52DBE2E496D5410B5CFD6C864D2D10F', owner),
        ('2B99CF26422E92FE365FBF4BC30D27086C9EE14B7A6FFF44FB2F6B9001699939', owner),
        ('2BBF2CA7B8F1D91F27EE52B6FB2A5DD049B85A2B9B529C5D6662068104B055F8', owner),
        ('2C73D93325BA6DCBE589D4A4C63C5B935559EF92FBF050ED50C4E2085206F17D', owner),
        ('2E70916786A6F773511FA7181FAB0F1D70B557C6322EA923B2A8D3B92B51AF7D', owner),
        ('306628FA5477305728BA4A467DE7D0387A54F569D3769FCE5E75EC89D28D1593', owner),
        ('3608EDBAF5AD0F41A414A1777ABF2FAF5E670334675EC3995E6935829E0CAAD2', owner),
        ('3841D221368D1583D75C0A02E62160394D6C4E0A6760B6F607B90362BC855B02', owner),
        ('3FCE9B9FDF3EF09D5452B0F95EE481C2B7F06D743A737971558E70136ACE3E73', owner),
        ('4397DACA839E7F63077CB50C92DF43BC2D2FB2A8F59F26FC7A0E4BD4D9751692', owner),
        ('47CC086127E2069A86E03A6BEF2CD410F8C55A6D6BDB362168C31B2CE32A5ADF', owner),
        ('518831FE7382B514D03E15C621228B8AB65479BD0CBFA3C5C1D0F48D9C306135', owner),
        ('5AE949EA8855EB93E439DBC65BDA2E42852C2FDF6789FA146736E3C3410F2B5C', owner),
        ('6B1D138078E4418AA68DEB7BB35E066092CF479EEB8CE4CD12E7D072CCB42F66', owner),
        ('6C8854478DD559E29351B826C06CB8BFEF2B94AD3538358772D193F82ED1CA11', owner),
        ('6F1428FF71C9DB0ED5AF1F2E7BBFCBAB647CC265DDF5B293CDB626F50A3A785E', owner),
        ('71F2906FD222497E54A34662AB2497FCC81020770FF51368E9E3D9BFCBFD6375', owner),
        ('726B3EB654046A30F3F83D9B96CE03F670E9A806D1708A0371E62DC49D2C23C1', owner),
        ('72E0BD1867CF5D9D56AB158ADF3BDDBC82BF32A8D8AA1D8C5E2F6DF29428D6D8', owner),
        ('7827AF99362CFAF0717DADE4B1BFE0438AD171C15ADDC248B75BF8CAA44BB2C5', owner),
        ('81A8B965BB84D3876B9429A95481CC955318CFAA1412D808C8A33BFD33FFF0E4', owner),
        ('82DB3BCEB4F60843CE9D97C3D187CD9B5941CD3DE8100E586F2BDA5637575F67', owner),
        ('895A9785F617CA1D7ED44FC1A1470B71F3F1223862D9FF9DCC3AE2DF92163DAF', owner),
        ('8AD64859F195B5F58DAFAA940B6A6167ACD67A886E8F469364177221C55945B9', owner),
        ('8BF434B49E00CCF71502A2CD900865CB01EC3B3DA03C35BE505FDF7BD563F521', owner),
        ('8D8EA289CFE70A1C07AB7365CB28EE51EDD33CF2506DE888FBADD60EBF80481C', owner),
        ('9998D363C491BE16BD74BA10B94D9291001611736FDCA643A36664BC0F315A42', owner),
        ('9E4A69173161682E55FDE8FEF560EB88EC1FFEDCAF04001F66C0CAF707B2B734', owner),
        ('A6B5151F3655D3A2AF0D472759796BE4A4200E5495A7D869754C4848857408A7', owner),
        ('A7F32F508D4EB0FEAD9A087EF94ED1BA0AEC5DE6F7EF6FF0A62B93BEDF5D458D', owner),
        ('AD6826E1946D26D3EAF3685C88D97D85DE3B4DCB3D0EE2AE81C70560D13C5720', owner),
        ('AEEBAE3151271273ED95AA2E671139ED31A98567303A332298F83709A9D55AA1', owner),
        ('AFE2030AFB7D2CDA13F9FA333A02E34F6751AFEC11B010DBCD441FDF4C4002B3', owner),
        ('B54F1EE636631FAD68058D3B0937031AC1B90CCB17062A391CCA68AFDBE40D55', owner),
        ('B8F078D983A24AC433216393883514CD932C33AF18E7DD70884C8235F4275736', owner),
        ('B97A0889059C035FF1D54B6DB53B11B9766668D9F955247C028B2837D7A04CD9', owner),
        ('BC87A668E81966489CB508EE805183C19E6ACD24CF17799CA062D2E384DA0EA7', owner),
        ('C409BDAC4775ADD8DB92AA22B5B718FB8C94A1462C1FE9A416B95D8A3388C2FC', owner),
        ('C617C1A8B1EE2A811C28B5A81B4C83D7C98B5B0C27281D610207EBE692C2967F', owner),
        ('C90F336617B8E7F983975413C997F10B73EB267FD8A10CB9E3BDBFC667ABDB8B', owner),
        ('CB6B858B40D3A098765815B592C1514A49604FAFD60819DA88D7A76E9778FEF7', owner),
        ('CE3BFABE59D67CE8AC8DFD4A16F7C43EF9C224513FBC655957D735FA29F540CE', owner),
        ('D8CBEB9735F5672B367E4F96CDC74969615D17074AE96C724D42CE0216F8F3FA', owner),
        ('E92C22EB3B5642D65C1EC2CAF247D2594738EEBB7FB3841A44956F59E2B0D1FA', owner),
        ('FDDD6E3D29EA84C7743DAD4A1BDBC700B5FEC1B391F932409086ACC71DD6DBD8', owner),
        ('FE63A84F782CC9D3FCF2CCF9FC11FBD03760878758D26285ED12669BDC6E6D01', owner),
        ('FECFB232D12E994B6D485D2C7167728AA5525984AD5CA61E7516221F079A1436', owner),
        ('CA171D614A8D7E121C93948CD0FE55D39981F9D11AA96E03450A415227C2C65B', owner),
        ('55B99B0DE53DBCFE485AA9C737CF3FB616EF3D91FAB599AA7CAB19EDA763B5BA', owner),
        ('77DD190FA30D88FF5E3B011A0AE61E6209780C130B535ECB87E6F0888A0B6B2F', owner),
        ('C83CB13922AD99F560744675DD37CC94DCAD5A1FCBA6472FEE341171D939E884', owner),
        ('3B0287533E0CC3D0EC1AA823CBF0A941AAD8721579D1C499802DD1C3A636B8A9', owner),
        ('939AEEF4F5FA51E23340C3F2E49048CE8872526AFDF752C3A7F3A3F2BC9F6049', owner),
        ('64575BD912789A2E14AD56F6341F52AF6BF80CF94400785975E9F04E2D64D745', owner),
        ('45C7C8AE750ACFBB48FC37527D6412DD644DAED8913CCD8A24C94D856967DF8E', owner)
    ]
    originalEsl = sha256_esl_builder(originalHashes)

    cert1 = [(UBUNTU_CERT, owner)]
    cert1_esl = x509_esl_builder(cert1)

    cert2 = [(DEBIAN_CERT, owner)]
    cert2_esl = x509_esl_builder(cert2)

    newHashes = [
        ('81D8FB4C9E2E7A8225656B4B8273B7CBA4B03EF2E9EB20E0A0291624ECA1BA86', owner),
        ('B92AF298DC08049B78C77492D6551B710CD72AADA3D77BE54609E43278EF6E4D', owner),
        ('E19DAE83C02E6F281358D4EBD11D7723B4F5EA0E357907D5443DECC5F93C1E9D', owner),
        ('39DBC2288EF44B5F95332CB777E31103E840DBA680634AA806F5C9B100061802', owner),
        ('32F5940CA29DD812A2C145E6FC89646628FFCC7C7A42CAE512337D8D29C40BBD', owner),
        ('10D45FCBA396AEF3153EE8F6ECAE58AFE8476A280A2026FC71F6217DCF49BA2F', owner),
        ('4B8668A5D465BCDD9000AA8DFCFF42044FCBD0AECE32FC7011A83E9160E89F09', owner),
        ('89F3D1F6E485C334CD059D0995E3CDFDC00571B1849854847A44DC5548E2DCFB', owner),
        ('C9EC350406F26E559AFFB4030DE2EBDE5435054C35A998605B8FCF04972D8D55', owner),
        ('B3E506340FBF6B5786973393079F24B66BA46507E35E911DB0362A2ACDE97049', owner),
        ('9F1863ED5717C394B42EF10A6607B144A65BA11FB6579DF94B8EB2F0C4CD60C1', owner),
        ('DD59AF56084406E38C63FBE0850F30A0CD1277462A2192590FB05BC259E61273', owner),
        ('DBAF9E056D3D5B38B68553304ABC88827EBC00F80CB9C7E197CDBC5822CD316C', owner),
        ('65F3C0A01B8402D362B9722E98F75E5E991E6C186E934F7B2B2E6BE6DEC800EC', owner),
        ('5B248E913D71853D3DA5AEDD8D9A4BC57A917126573817FB5FCB2D86A2F1C886', owner),
        ('2679650FE341F2CF1EA883460B3556AAAF77A70D6B8DC484C9301D1B746CF7B5', owner),
        ('BB1DD16D530008636F232303A7A86F3DFF969F848815C0574B12C2D787FEC93F', owner),
        ('0CE02100F67C7EF85F4EED368F02BF7092380A3C23CA91FD7F19430D94B00C19', owner),
        ('95049F0E4137C790B0D2767195E56F73807D123ADCF8F6E7BF2D4D991D305F89', owner),
        ('02E6216ACAEF6401401FA555ECBED940B1A5F2569AED92956137AE58482EF1B7', owner),
        ('6EFEFE0B5B01478B7B944C10D3A8ACA2CCA4208888E2059F8A06CB5824D7BAB0', owner),
        ('9D00AE4CD47A41C783DC48F342C076C2C16F3413F4D2DF50D181CA3BB5AD859D', owner),
        ('D8D4E6DDF6E42D74A6A536EA62FD1217E4290B145C9E5C3695A31B42EFB5F5A4', owner),
        ('F277AF4F9BDC918AE89FA35CC1B34E34984C04AE9765322C3CB049574D36509C', owner),
        ('0DC24C75EB1AEF56B9F13AB9DE60E2ECA1C4510034E290BBB36CF60A549B234C', owner),
        ('835881F2A5572D7059B5C8635018552892E945626F115FC9CA07ACF7BDE857A4', owner),
        ('BADFF5E4F0FEA711701CA8FB22E4C43821E31E210CF52D1D4F74DD50F1D039BC', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('F1863EC8B7F43F94AD14FB0B8B4A69497A8C65ECBC2A55E0BB420E772B8CDC91', owner),
        ('7BC9CB5463CE0F011FB5085EB8BA77D1ACD283C43F4A57603CC113F22CEBC579', owner),
        ('E800395DBE0E045781E8005178B4BAF5A257F06E159121A67C595F6AE22506FD', owner),
        ('1CB4DCCAF2C812CFA7B4938E1371FE2B96910FE407216FD95428672D6C7E7316', owner),
        ('3ECE27CBB3EC4438CCE523B927C4F05FDC5C593A3766DB984C5E437A3FF6A16B', owner),
        ('68EE4632C7BE1C66C83E89DD93EAEE1294159ABF45B4C2C72D7DC7499AA2A043', owner),
        ('E24B315A551671483D8B9073B32DE11B4DE1EB2EAB211AFD2D9C319FF55E08D0', owner),
        ('E7C20B3AB481EC885501ECA5293781D84B5A1AC24F88266B5270E7ECB4AA2538', owner),
        ('7EAC80A915C84CD4AFEC638904D94EB168A8557951A4D539B0713028552B6B8C', owner),
        ('E7681F153121EA1E67F74BBCB0CDC5E502702C1B8CC55FB65D702DFBA948B5F4', owner),
        ('DCCC3CE1C00EE4B0B10487D372A0FA47F5C26F57A359BE7B27801E144EACBAC4', owner),
        ('0257FF710F2A16E489B37493C07604A7CDA96129D8A8FD68D2B6AF633904315D', owner),
        ('3A91F0F9E5287FA2994C7D930B2C1A5EE14CE8E1C8304AE495ADC58CC4453C0C', owner),
        ('495300790E6C9BF2510DABA59DB3D57E9D2B85D7D7640434EC75BAA3851C74E5', owner),
        ('81A8B2C9751AEB1FABA7DBDE5EE9691DC0EAEE2A31C38B1491A8146756A6B770', owner),
        ('8E53EFDC15F852CEE5A6E92931BC42E6163CD30FF649CCA7E87252C3A459960B', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('9FA4D5023FD43ECAFF4200BA7E8D4353259D2B7E5E72B5096EFF8027D66D1043', owner),
        ('D372C0D0F4FDC9F52E9E1F23FC56EE72414A17F350D0CEA6C26A35A6C3217A13', owner),
        ('5C5805196A85E93789457017D4F9EB6828B97C41CB9BA6D3DC1FCC115F527A55', owner),
        ('804E354C6368BB27A90FAE8E498A57052B293418259A019C4F53A2007254490F', owner),
        ('03F64A29948A88BEFFDB035E0B09A7370CCF0CD9CE6BCF8E640C2107318FAB87', owner),
        ('05D87E15713454616F5B0ED7849AB5C1712AB84F02349478EC2A38F970C01489', owner),
        ('06EB5BADD26E4FAE65F9A42358DEEF7C18E52CC05FBB7FC76776E69D1B982A14', owner),
        ('08BB2289E9E91B4D20FF3F1562516AB07E979B2C6CEFE2AB70C6DFC1199F8DA5', owner),
        ('0928F0408BF725E61D67D87138A8EEBC52962D2847F16E3587163B160E41B6AD', owner),
        ('09F98AA90F85198C0D73F89BA77E87EC6F596C491350FB8F8BBA80A62FBB914B', owner),
        ('0A75EA0B1D70EAA4D3F374246DB54FC7B43E7F596A353309B9C36B4FD975725E', owner),
        ('0C51D7906FC4931149765DA88682426B2CFE9E6AA4F27253EAB400111432E3A7', owner),
        ('0FA3A29AD05130D7FE5BF4D2596563CDED1D874096AACC181069932A2E49519A', owner),
        ('147730B42F11FE493FE902B6251E97CD2B6F34D36AF59330F11D02A42F940D07', owner),
        ('148FE18F715A9FCFE1A444CE0FFF7F85869EB422330DC04B314C0F295D6DA79E', owner),
        ('1B909115A8D473E51328A87823BD621CE655DFAE54FA2BFA72FDC0298611D6B8', owner),
        ('1D8B58C1FDB8DA8B33CCEE1E5F973AF734D90EF317E33F5DB1573C2BA088A80C', owner),
        ('1F179186EFDF5EF2DE018245BA0EAE8134868601BA0D35FF3D9865C1537CED93', owner),
        ('270C84B29D86F16312B06AAAE4EBB8DFF8DE7D080D825B8839FF1766274EFF47', owner),
        ('29CCA4544EA330D61591C784695C149C6B040022AC7B5B89CBD72800D10840EA', owner),
        ('2B2298EAA26B9DC4A4558AE92E7BB0E4F85CF34BF848FDF636C0C11FBEC49897', owner),
        ('2DCF8E8D817023D1E8E1451A3D68D6EC30D9BED94CBCB87F19DDC1CC0116AC1A', owner),
        ('311A2AC55B50C09B30B3CC93B994A119153EEEAC54EF892FC447BBBD96101AA1', owner),
        ('32AD3296829BC46DCFAC5EDDCB9DBF2C1EED5C11F83B2210CF9C6E60C798D4A7', owner),
        ('340DA32B58331C8E2B561BAF300CA9DFD6B91CD2270EE0E2A34958B1C6259E85', owner),
        ('362ED31D20B1E00392281231A96F0A0ACFDE02618953E695C9EF2EB0BAC37550', owner),
        ('367A31E5838831AD2C074647886A6CDFF217E6B1BA910BFF85DC7A87AE9B5E98', owner),
        ('3765D769C05BF98B427B3511903B2137E8A49B6F859D0AF159ED6A86786AA634', owner),
        ('386D695CDF2D4576E01BCACCF5E49E78DA51AF9955C0B8FA7606373B007994B3', owner),
        ('3A4F74BEAFAE2B9383AD8215D233A6CF3D057FB3C7E213E897BEEF4255FAEE9D', owner),
        ('3AE76C45CA70E9180C1559981F42622DD251BCA1FBE6B901C52EC11673B03514', owner),
        ('3BE8E7EB348D35C1928F19C769846788991641D1F6CF09514CA10269934F7359', owner),
        ('3E3926F0B8A15AD5A14167BB647A843C3D4321E35DBC44DCE8C837417F2D28B0', owner),
        ('400AC66D59B7B094A9E30B01A6BD013AFF1D30570F83E7592F421DBE5FF4BA8F', owner),
        ('4185821F6DAB5BA8347B78A22B5F9A0A7570CA5C93A74D478A793D83BAC49805', owner),
        ('41D1EEB177C0324E17DD6557F384E532DE0CF51A019A446B01EFB351BC259D77', owner),
        ('45876B4DD861D45B3A94800774027A5DB45A48B2A729410908B6412F8A87E95D', owner),
        ('4667BF250CD7C1A06B8474C613CDB1DF648A7F58736FBF57D05D6F755DAB67F4', owner),
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('57E6913AFACC5222BD76CDAF31F8ED88895464255374EF097A82D7F59AD39596', owner),
        ('5890FA227121C76D90ED9E63C87E3A6533EEA0F6F0A1A23F1FC445139BC6BCDF', owner),
        ('5D1E9ACBBB4A7D024B6852DF025970E2CED66FF622EE019CD0ED7FD841CCAD02', owner),
        ('61341E07697978220EA61E85DCD2421343F2C1BF35CC5B8D0AD2F0226F391479', owner),
        ('61CEC4A377BF5902C0FEAEE37034BF97D5BC6E0615E23A1CDFBAE6E3F5FB3CFD', owner),
        ('631F0857B41845362C90C6980B4B10C4B628E23DBE24B6E96C128AE3DCB0D5AC', owner),
        ('65B2E7CC18D903C331DF1152DF73CA0DC932D29F17997481C56F3087B2DD3147', owner),
        ('66AA13A0EDC219384D9C425D3927E6ED4A5D1940C5E7CD4DAC88F5770103F2F1', owner),
        ('6873D2F61C29BD52E954EEFF5977AA8367439997811A62FF212C948133C68D97', owner),
        ('6DBBEAD23E8C860CF8B47F74FBFCA5204DE3E28B881313BB1D1ECCDC4747934E', owner),
        ('6DEAD13257DFC3CCC6A4B37016BA91755FE9E0EC1F415030942E5ABC47F07C88', owner),
        ('70A1450AF2AD395569AD0AFEB1D9C125324EE90AEC39C258880134D4892D51AB', owner),
        ('72C26F827CEB92989798961BC6AE748D141E05D3EBCFB65D9041B266C920BE82', owner),
        ('781764102188A8B4B173D4A8F5EC94D828647156097F99357A581E624B377509', owner),
        ('788383A4C733BB87D2BF51673DC73E92DF15AB7D51DC715627AE77686D8D23BC', owner),
        ('78B4EDCAABC8D9093E20E217802CAEB4F09E23A3394C4ACC6E87E8F35395310F', owner),
        ('7F49CCB309323B1C7AB11C93C955B8C744F0A2B75C311F495E18906070500027', owner),
        ('82ACBA48D5236CCFF7659AFC14594DEE902BD6082EF1A30A0B9B508628CF34F4', owner),
        ('894D7839368F3298CC915AE8742EF330D7A26699F459478CF22C2B6BB2850166', owner),
        ('8C0349D708571AE5AA21C11363482332073297D868F29058916529EFC520EF70', owner),
        ('8D93D60C691959651476E5DC464BE12A85FA5280B6F524D4A1C3FCC9D048CFAD', owner),
        ('9063F5FBC5E57AB6DE6C9488146020E172B176D5AB57D4C89F0F600E17FE2DE2', owner),
        ('91656AA4EF493B3824A0B7263248E4E2D657A5C8488D880CB65B01730932FB53', owner),
        ('91971C1497BF8E5BC68439ACC48D63EBB8FAABFD764DCBE82F3BA977CAC8CF6A', owner),
        ('947078F97C6196968C3AE99C9A5D58667E86882CF6C8C9D58967A496BB7AF43C', owner),
        ('96E4509450D380DAC362FF8E295589128A1F1CE55885D20D89C27BA2A9D00909', owner),
        ('9783B5EE4492E9E891C655F1F48035959DAD453C0E623AF0FE7BF2C0A57885E3', owner),
        ('97A51A094444620DF38CD8C6512CAC909A75FD437AE1E4D22929807661238127', owner),
        ('97A8C5BA11D61FEFBB5D6A05DA4E15BA472DC4C6CD4972FC1A035DE321342FE4', owner),
        ('992820E6EC8C41DAAE4BD8AB48F58268E943A670D35CA5E2BDCD3E7C4C94A072', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('9954A1A99D55E8B189AB1BCA414B91F6A017191F6C40A86B6F3EF368DD860031', owner),
        ('9BAF4F76D76BF5D6A897BFBD5F429BA14D04E08B48C3EE8D76930A828FFF3891', owner),
        ('9C259FCB301D5FC7397ED5759963E0EF6B36E42057FD73046E6BD08B149F751C', owner),
        ('9DD2DCB72F5E741627F2E9E03AB18503A3403CF6A904A479A4DB05D97E2250A9', owner),
        ('9ED33F0FBC180BC032F8909CA2C4AB3418EDC33A45A50D2521A3B5876AA3EA2C', owner),
        ('A4D978B7C4BDA15435D508F8B9592EC2A5ADFB12EA7BAD146A35ECB53094642F', owner),
        ('A924D3CAD6DA42B7399B96A095A06F18F6B1ABA5B873B0D5F3A0EE2173B48B6C', owner),
        ('AD3BE589C0474E97DE5BB2BF33534948B76BB80376DFDC58B1FED767B5A15BFC', owner),
        ('B8D6B5E7857B45830E017C7BE3D856ADEB97C7290EB0665A3D473A4BEB51DCF3', owner),
        ('B93F0699598F8B20FA0DACC12CFCFC1F2568793F6E779E04795E6D7C22530F75', owner),
        ('BB01DA0333BB639C7E1C806DB0561DC98A5316F22FEF1090FB8D0BE46DAE499A', owner),
        ('BC75F910FF320F5CB5999E66BBD4034F4AE537A42FDFEF35161C5348E366E216', owner),
        ('BDD01126E9D85710D3FE75AF1CC1702A29F081B4F6FDF6A2B2135C0297A9CEC5', owner),
        ('BE435DF7CD28AA2A7C8DB4FC8173475B77E5ABF392F76B7C76FA3F698CB71A9A', owner),
        ('BEF7663BE5EA4DBFD8686E24701E036F4C03FB7FCD67A6C566ED94CE09C44470', owner),
        ('C2469759C1947E14F4B65F72A9F5B3AF8B6F6E727B68BB0D91385CBF42176A8A', owner),
        ('C3505BF3EC10A51DACE417C76B8BD10939A065D1F34E75B8A3065EE31CC69B96', owner),
        ('C42D11C70CCF5E8CF3FB91FDF21D884021AD836CA68ADF2CBB7995C10BF588D4', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('C69D64A5B839E41BA16742527E17056A18CE3C276FD26E34901A1BC7D0E32219', owner),
        ('CB340011AFEB0D74C4A588B36EBAA441961608E8D2FA80DCA8C13872C850796B', owner),
        ('CC8EEC6EB9212CBF897A5ACE7E8ABEECE1079F1A6DEF0A789591CB1547F1F084', owner),
        ('CF13A243C1CD2E3C8CEB7E70100387CECBFB830525BBF9D0B70C79ADF3E84128', owner),
        ('D89A11D16C488DD4FBBC541D4B07FAF8670D660994488FE54B1FBFF2704E4288', owner),
        ('D9668AB52785086786C134B5E4BDDBF72452813B6973229AB92AA1A54D201BF5', owner),
        ('DA3560FD0C32B54C83D4F2FF869003D2089369ACF2C89608F8AFA7436BFA4655', owner),
        ('DF02AAB48387A9E1D4C65228089CB6ABE196C8F4B396C7E4BBC395DE136977F6', owner),
        ('DF91AC85A94FCD0CFB8155BD7CBEFAAC14B8C5EE7397FE2CC85984459E2EA14E', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner),
        ('E36DFC719D2114C2E39AEA88849E2845AB326F6F7FE74E0E539B7E54D81F3631', owner),
        ('E39891F48BBCC593B8ED86CE82CE666FC1145B9FCBFD2B07BAD0A89BF4C7BFBF', owner),
        ('E6856F137F79992DC94FA2F43297EC32D2D9A76F7BE66114C6A13EFC3BCDF5C8', owner),
        ('EAFF8C85C208BA4D5B6B8046F5D6081747D779BADA7768E649D047FF9B1F660C', owner),
        ('EE83A566496109A74F6AC6E410DF00BB29A290E0021516AE3B8A23288E7E2E72', owner),
        ('EED7E0EFF2ED559E2A79EE361F9962AF3B1E999131E30BB7FD07546FAE0A7267', owner),
        ('F1B4F6513B0D544A688D13ADC291EFA8C59F420CA5DCB23E0B5A06FA7E0D083D', owner),
        ('F2A16D35B554694187A70D40CA682959F4F35C2CE0EAB8FD64F7AC2AB9F5C24A', owner),
        ('F31FD461C5E99510403FC97C1DA2D8A9CBE270597D32BADF8FD66B77495F8D94', owner),
        ('F48E6DD8718E953B60A24F2CBEA60A9521DEAE67DB25425B7D3ACE3C517DD9B7', owner),
        ('C805603C4FA038776E42F263C604B49D96840322E1922D5606A9B0BBB5BFFE6F', owner),
        ('1F16078CCE009DF62EDB9E7170E66CAAE670BCE71B8F92D38280C56AA372031D', owner),
        ('37A480374DAF6202CE790C318A2BB8AA3797311261160A8E30558B7DEA78C7A6', owner),
        ('408B8B3DF5ABB043521A493525023175AB1261B1DE21064D6BF247CE142153B9', owner),
        ('540801DD345DC1C33EF431B35BF4C0E68BD319B577B9ABE1A9CFF1CBC39F548F', owner)
    ]
    newHash_esl = sha256_esl_builder(newHashes)

    # EDK2 appended cert1, cert2, and the partially de-duped new hashes to the original dbx
    esd = EfiSignatureDatabase(EslList=[originalEsl, cert1_esl, cert2_esl, newHash_esl])

    # Now prepare the duplicate data, these were the 4 dupes in the new list that were not de-duped by EDK2
    expected_dupes = [
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner)
    ]
    expected_dupes_esl = sha256_esl_builder(expected_dupes)
    expected_dupes_esd = EfiSignatureDatabase(EslList=[expected_dupes_esl])

    # Now prepare the canonicalized data
    canonicalHashes = [
        ('0257FF710F2A16E489B37493C07604A7CDA96129D8A8FD68D2B6AF633904315D', owner),
        ('02E6216ACAEF6401401FA555ECBED940B1A5F2569AED92956137AE58482EF1B7', owner),
        ('03F64A29948A88BEFFDB035E0B09A7370CCF0CD9CE6BCF8E640C2107318FAB87', owner),
        ('05D87E15713454616F5B0ED7849AB5C1712AB84F02349478EC2A38F970C01489', owner),
        ('06EB5BADD26E4FAE65F9A42358DEEF7C18E52CC05FBB7FC76776E69D1B982A14', owner),
        ('075EEA060589548BA060B2FEED10DA3C20C7FE9B17CD026B94E8A683B8115238', owner),
        ('07E6C6A858646FB1EFC67903FE28B116011F2367FE92E6BE2B36999EFF39D09E', owner),
        ('08BB2289E9E91B4D20FF3F1562516AB07E979B2C6CEFE2AB70C6DFC1199F8DA5', owner),
        ('0928F0408BF725E61D67D87138A8EEBC52962D2847F16E3587163B160E41B6AD', owner),
        ('09DF5F4E511208EC78B96D12D08125FDB603868DE39F6F72927852599B659C26', owner),
        ('09F98AA90F85198C0D73F89BA77E87EC6F596C491350FB8F8BBA80A62FBB914B', owner),
        ('0A75EA0B1D70EAA4D3F374246DB54FC7B43E7F596A353309B9C36B4FD975725E', owner),
        ('0BBB4392DAAC7AB89B30A4AC657531B97BFAAB04F90B0DAFE5F9B6EB90A06374', owner),
        ('0C189339762DF336AB3DD006A463DF715A39CFB0F492465C600E6C6BD7BD898C', owner),
        ('0C51D7906FC4931149765DA88682426B2CFE9E6AA4F27253EAB400111432E3A7', owner),
        ('0CE02100F67C7EF85F4EED368F02BF7092380A3C23CA91FD7F19430D94B00C19', owner),
        ('0D0DBECA6F29ECA06F331A7D72E4884B12097FB348983A2A14A0D73F4F10140F', owner),
        ('0DC24C75EB1AEF56B9F13AB9DE60E2ECA1C4510034E290BBB36CF60A549B234C', owner),
        ('0DC9F3FB99962148C3CA833632758D3ED4FC8D0B0007B95B31E6528F2ACD5BFC', owner),
        ('0FA3A29AD05130D7FE5BF4D2596563CDED1D874096AACC181069932A2E49519A', owner),
        ('106FACEACFECFD4E303B74F480A08098E2D0802B936F8EC774CE21F31686689C', owner),
        ('10D45FCBA396AEF3153EE8F6ECAE58AFE8476A280A2026FC71F6217DCF49BA2F', owner),
        ('147730B42F11FE493FE902B6251E97CD2B6F34D36AF59330F11D02A42F940D07', owner),
        ('148FE18F715A9FCFE1A444CE0FFF7F85869EB422330DC04B314C0F295D6DA79E', owner),
        ('174E3A0B5B43C6A607BBD3404F05341E3DCF396267CE94F8B50E2E23A9DA920C', owner),
        ('18333429FF0562ED9F97033E1148DCEEE52DBE2E496D5410B5CFD6C864D2D10F', owner),
        ('1AEC84B84B6C65A51220A9BE7181965230210D62D6D33C48999C6B295A2B0A06', owner),
        ('1B909115A8D473E51328A87823BD621CE655DFAE54FA2BFA72FDC0298611D6B8', owner),
        ('1CB4DCCAF2C812CFA7B4938E1371FE2B96910FE407216FD95428672D6C7E7316', owner),
        ('1D8B58C1FDB8DA8B33CCEE1E5F973AF734D90EF317E33F5DB1573C2BA088A80C', owner),
        ('1F16078CCE009DF62EDB9E7170E66CAAE670BCE71B8F92D38280C56AA372031D', owner),
        ('1F179186EFDF5EF2DE018245BA0EAE8134868601BA0D35FF3D9865C1537CED93', owner),
        ('2679650FE341F2CF1EA883460B3556AAAF77A70D6B8DC484C9301D1B746CF7B5', owner),
        ('270C84B29D86F16312B06AAAE4EBB8DFF8DE7D080D825B8839FF1766274EFF47', owner),
        ('29C6EB52B43C3AA18B2CD8ED6EA8607CEF3CFAE1BAFE1165755CF2E614844A44', owner),
        ('29CCA4544EA330D61591C784695C149C6B040022AC7B5B89CBD72800D10840EA', owner),
        ('2B2298EAA26B9DC4A4558AE92E7BB0E4F85CF34BF848FDF636C0C11FBEC49897', owner),
        ('2B99CF26422E92FE365FBF4BC30D27086C9EE14B7A6FFF44FB2F6B9001699939', owner),
        ('2BBF2CA7B8F1D91F27EE52B6FB2A5DD049B85A2B9B529C5D6662068104B055F8', owner),
        ('2C73D93325BA6DCBE589D4A4C63C5B935559EF92FBF050ED50C4E2085206F17D', owner),
        ('2DCF8E8D817023D1E8E1451A3D68D6EC30D9BED94CBCB87F19DDC1CC0116AC1A', owner),
        ('2E70916786A6F773511FA7181FAB0F1D70B557C6322EA923B2A8D3B92B51AF7D', owner),
        ('306628FA5477305728BA4A467DE7D0387A54F569D3769FCE5E75EC89D28D1593', owner),
        ('311A2AC55B50C09B30B3CC93B994A119153EEEAC54EF892FC447BBBD96101AA1', owner),
        ('32AD3296829BC46DCFAC5EDDCB9DBF2C1EED5C11F83B2210CF9C6E60C798D4A7', owner),
        ('32F5940CA29DD812A2C145E6FC89646628FFCC7C7A42CAE512337D8D29C40BBD', owner),
        ('340DA32B58331C8E2B561BAF300CA9DFD6B91CD2270EE0E2A34958B1C6259E85', owner),
        ('3608EDBAF5AD0F41A414A1777ABF2FAF5E670334675EC3995E6935829E0CAAD2', owner),
        ('362ED31D20B1E00392281231A96F0A0ACFDE02618953E695C9EF2EB0BAC37550', owner),
        ('363384D14D1F2E0B7815626484C459AD57A318EF4396266048D058C5A19BBF76', owner),
        ('367A31E5838831AD2C074647886A6CDFF217E6B1BA910BFF85DC7A87AE9B5E98', owner),
        ('3765D769C05BF98B427B3511903B2137E8A49B6F859D0AF159ED6A86786AA634', owner),
        ('37A480374DAF6202CE790C318A2BB8AA3797311261160A8E30558B7DEA78C7A6', owner),
        ('3841D221368D1583D75C0A02E62160394D6C4E0A6760B6F607B90362BC855B02', owner),
        ('386D695CDF2D4576E01BCACCF5E49E78DA51AF9955C0B8FA7606373B007994B3', owner),
        ('39DBC2288EF44B5F95332CB777E31103E840DBA680634AA806F5C9B100061802', owner),
        ('3A4F74BEAFAE2B9383AD8215D233A6CF3D057FB3C7E213E897BEEF4255FAEE9D', owner),
        ('3A91F0F9E5287FA2994C7D930B2C1A5EE14CE8E1C8304AE495ADC58CC4453C0C', owner),
        ('3AE76C45CA70E9180C1559981F42622DD251BCA1FBE6B901C52EC11673B03514', owner),
        ('3B0287533E0CC3D0EC1AA823CBF0A941AAD8721579D1C499802DD1C3A636B8A9', owner),
        ('3BE8E7EB348D35C1928F19C769846788991641D1F6CF09514CA10269934F7359', owner),
        ('3E3926F0B8A15AD5A14167BB647A843C3D4321E35DBC44DCE8C837417F2D28B0', owner),
        ('3ECE27CBB3EC4438CCE523B927C4F05FDC5C593A3766DB984C5E437A3FF6A16B', owner),
        ('3FCE9B9FDF3EF09D5452B0F95EE481C2B7F06D743A737971558E70136ACE3E73', owner),
        ('400AC66D59B7B094A9E30B01A6BD013AFF1D30570F83E7592F421DBE5FF4BA8F', owner),
        ('408B8B3DF5ABB043521A493525023175AB1261B1DE21064D6BF247CE142153B9', owner),
        ('4185821F6DAB5BA8347B78A22B5F9A0A7570CA5C93A74D478A793D83BAC49805', owner),
        ('41D1EEB177C0324E17DD6557F384E532DE0CF51A019A446B01EFB351BC259D77', owner),
        ('4397DACA839E7F63077CB50C92DF43BC2D2FB2A8F59F26FC7A0E4BD4D9751692', owner),
        ('45876B4DD861D45B3A94800774027A5DB45A48B2A729410908B6412F8A87E95D', owner),
        ('45C7C8AE750ACFBB48FC37527D6412DD644DAED8913CCD8A24C94D856967DF8E', owner),
        ('4667BF250CD7C1A06B8474C613CDB1DF648A7F58736FBF57D05D6F755DAB67F4', owner),
        ('47CC086127E2069A86E03A6BEF2CD410F8C55A6D6BDB362168C31B2CE32A5ADF', owner),
        ('47FF1B63B140B6FC04ED79131331E651DA5B2E2F170F5DAEF4153DC2FBC532B1', owner),
        ('495300790E6C9BF2510DABA59DB3D57E9D2B85D7D7640434EC75BAA3851C74E5', owner),
        ('4B8668A5D465BCDD9000AA8DFCFF42044FCBD0AECE32FC7011A83E9160E89F09', owner),
        ('518831FE7382B514D03E15C621228B8AB65479BD0CBFA3C5C1D0F48D9C306135', owner),
        ('5391C3A2FB112102A6AA1EDC25AE77E19F5D6F09CD09EEB2509922BFCD5992EA', owner),
        ('540801DD345DC1C33EF431B35BF4C0E68BD319B577B9ABE1A9CFF1CBC39F548F', owner),
        ('55B99B0DE53DBCFE485AA9C737CF3FB616EF3D91FAB599AA7CAB19EDA763B5BA', owner),
        ('57E6913AFACC5222BD76CDAF31F8ED88895464255374EF097A82D7F59AD39596', owner),
        ('5890FA227121C76D90ED9E63C87E3A6533EEA0F6F0A1A23F1FC445139BC6BCDF', owner),
        ('58FB941AEF95A25943B3FB5F2510A0DF3FE44C58C95E0AB80487297568AB9771', owner),
        ('5AE949EA8855EB93E439DBC65BDA2E42852C2FDF6789FA146736E3C3410F2B5C', owner),
        ('5B248E913D71853D3DA5AEDD8D9A4BC57A917126573817FB5FCB2D86A2F1C886', owner),
        ('5C5805196A85E93789457017D4F9EB6828B97C41CB9BA6D3DC1FCC115F527A55', owner),
        ('5D1E9ACBBB4A7D024B6852DF025970E2CED66FF622EE019CD0ED7FD841CCAD02', owner),
        ('61341E07697978220EA61E85DCD2421343F2C1BF35CC5B8D0AD2F0226F391479', owner),
        ('61CEC4A377BF5902C0FEAEE37034BF97D5BC6E0615E23A1CDFBAE6E3F5FB3CFD', owner),
        ('631F0857B41845362C90C6980B4B10C4B628E23DBE24B6E96C128AE3DCB0D5AC', owner),
        ('64575BD912789A2E14AD56F6341F52AF6BF80CF94400785975E9F04E2D64D745', owner),
        ('65B2E7CC18D903C331DF1152DF73CA0DC932D29F17997481C56F3087B2DD3147', owner),
        ('65F3C0A01B8402D362B9722E98F75E5E991E6C186E934F7B2B2E6BE6DEC800EC', owner),
        ('66AA13A0EDC219384D9C425D3927E6ED4A5D1940C5E7CD4DAC88F5770103F2F1', owner),
        ('6873D2F61C29BD52E954EEFF5977AA8367439997811A62FF212C948133C68D97', owner),
        ('68EE4632C7BE1C66C83E89DD93EAEE1294159ABF45B4C2C72D7DC7499AA2A043', owner),
        ('6B1D138078E4418AA68DEB7BB35E066092CF479EEB8CE4CD12E7D072CCB42F66', owner),
        ('6C8854478DD559E29351B826C06CB8BFEF2B94AD3538358772D193F82ED1CA11', owner),
        ('6DBBEAD23E8C860CF8B47F74FBFCA5204DE3E28B881313BB1D1ECCDC4747934E', owner),
        ('6DEAD13257DFC3CCC6A4B37016BA91755FE9E0EC1F415030942E5ABC47F07C88', owner),
        ('6EFEFE0B5B01478B7B944C10D3A8ACA2CCA4208888E2059F8A06CB5824D7BAB0', owner),
        ('6F1428FF71C9DB0ED5AF1F2E7BBFCBAB647CC265DDF5B293CDB626F50A3A785E', owner),
        ('70A1450AF2AD395569AD0AFEB1D9C125324EE90AEC39C258880134D4892D51AB', owner),
        ('71F2906FD222497E54A34662AB2497FCC81020770FF51368E9E3D9BFCBFD6375', owner),
        ('726B3EB654046A30F3F83D9B96CE03F670E9A806D1708A0371E62DC49D2C23C1', owner),
        ('72C26F827CEB92989798961BC6AE748D141E05D3EBCFB65D9041B266C920BE82', owner),
        ('72E0BD1867CF5D9D56AB158ADF3BDDBC82BF32A8D8AA1D8C5E2F6DF29428D6D8', owner),
        ('77DD190FA30D88FF5E3B011A0AE61E6209780C130B535ECB87E6F0888A0B6B2F', owner),
        ('781764102188A8B4B173D4A8F5EC94D828647156097F99357A581E624B377509', owner),
        ('7827AF99362CFAF0717DADE4B1BFE0438AD171C15ADDC248B75BF8CAA44BB2C5', owner),
        ('788383A4C733BB87D2BF51673DC73E92DF15AB7D51DC715627AE77686D8D23BC', owner),
        ('78B4EDCAABC8D9093E20E217802CAEB4F09E23A3394C4ACC6E87E8F35395310F', owner),
        ('7BC9CB5463CE0F011FB5085EB8BA77D1ACD283C43F4A57603CC113F22CEBC579', owner),
        ('7EAC80A915C84CD4AFEC638904D94EB168A8557951A4D539B0713028552B6B8C', owner),
        ('7F49CCB309323B1C7AB11C93C955B8C744F0A2B75C311F495E18906070500027', owner),
        ('804E354C6368BB27A90FAE8E498A57052B293418259A019C4F53A2007254490F', owner),
        ('80B4D96931BF0D02FD91A61E19D14F1DA452E66DB2408CA8604D411F92659F0A', owner),
        ('81A8B2C9751AEB1FABA7DBDE5EE9691DC0EAEE2A31C38B1491A8146756A6B770', owner),
        ('81A8B965BB84D3876B9429A95481CC955318CFAA1412D808C8A33BFD33FFF0E4', owner),
        ('81D8FB4C9E2E7A8225656B4B8273B7CBA4B03EF2E9EB20E0A0291624ECA1BA86', owner),
        ('82ACBA48D5236CCFF7659AFC14594DEE902BD6082EF1A30A0B9B508628CF34F4', owner),
        ('82DB3BCEB4F60843CE9D97C3D187CD9B5941CD3DE8100E586F2BDA5637575F67', owner),
        ('835881F2A5572D7059B5C8635018552892E945626F115FC9CA07ACF7BDE857A4', owner),
        ('894D7839368F3298CC915AE8742EF330D7A26699F459478CF22C2B6BB2850166', owner),
        ('895A9785F617CA1D7ED44FC1A1470B71F3F1223862D9FF9DCC3AE2DF92163DAF', owner),
        ('89F3D1F6E485C334CD059D0995E3CDFDC00571B1849854847A44DC5548E2DCFB', owner),
        ('8AD64859F195B5F58DAFAA940B6A6167ACD67A886E8F469364177221C55945B9', owner),
        ('8BF434B49E00CCF71502A2CD900865CB01EC3B3DA03C35BE505FDF7BD563F521', owner),
        ('8C0349D708571AE5AA21C11363482332073297D868F29058916529EFC520EF70', owner),
        ('8D8EA289CFE70A1C07AB7365CB28EE51EDD33CF2506DE888FBADD60EBF80481C', owner),
        ('8D93D60C691959651476E5DC464BE12A85FA5280B6F524D4A1C3FCC9D048CFAD', owner),
        ('8E53EFDC15F852CEE5A6E92931BC42E6163CD30FF649CCA7E87252C3A459960B', owner),
        ('9063F5FBC5E57AB6DE6C9488146020E172B176D5AB57D4C89F0F600E17FE2DE2', owner),
        ('90FBE70E69D633408D3E170C6832DBB2D209E0272527DFB63D49D29572A6F44C', owner),
        ('91656AA4EF493B3824A0B7263248E4E2D657A5C8488D880CB65B01730932FB53', owner),
        ('91971C1497BF8E5BC68439ACC48D63EBB8FAABFD764DCBE82F3BA977CAC8CF6A', owner),
        ('939AEEF4F5FA51E23340C3F2E49048CE8872526AFDF752C3A7F3A3F2BC9F6049', owner),
        ('947078F97C6196968C3AE99C9A5D58667E86882CF6C8C9D58967A496BB7AF43C', owner),
        ('95049F0E4137C790B0D2767195E56F73807D123ADCF8F6E7BF2D4D991D305F89', owner),
        ('96E4509450D380DAC362FF8E295589128A1F1CE55885D20D89C27BA2A9D00909', owner),
        ('9783B5EE4492E9E891C655F1F48035959DAD453C0E623AF0FE7BF2C0A57885E3', owner),
        ('97A51A094444620DF38CD8C6512CAC909A75FD437AE1E4D22929807661238127', owner),
        ('97A8C5BA11D61FEFBB5D6A05DA4E15BA472DC4C6CD4972FC1A035DE321342FE4', owner),
        ('992820E6EC8C41DAAE4BD8AB48F58268E943A670D35CA5E2BDCD3E7C4C94A072', owner),
        ('992D359AA7A5F789D268B94C11B9485A6B1CE64362B0EDB4441CCC187C39647B', owner),
        ('9954A1A99D55E8B189AB1BCA414B91F6A017191F6C40A86B6F3EF368DD860031', owner),
        ('9998D363C491BE16BD74BA10B94D9291001611736FDCA643A36664BC0F315A42', owner),
        ('9BAF4F76D76BF5D6A897BFBD5F429BA14D04E08B48C3EE8D76930A828FFF3891', owner),
        ('9C259FCB301D5FC7397ED5759963E0EF6B36E42057FD73046E6BD08B149F751C', owner),
        ('9D00AE4CD47A41C783DC48F342C076C2C16F3413F4D2DF50D181CA3BB5AD859D', owner),
        ('9DD2DCB72F5E741627F2E9E03AB18503A3403CF6A904A479A4DB05D97E2250A9', owner),
        ('9E4A69173161682E55FDE8FEF560EB88EC1FFEDCAF04001F66C0CAF707B2B734', owner),
        ('9ED33F0FBC180BC032F8909CA2C4AB3418EDC33A45A50D2521A3B5876AA3EA2C', owner),
        ('9F1863ED5717C394B42EF10A6607B144A65BA11FB6579DF94B8EB2F0C4CD60C1', owner),
        ('9FA4D5023FD43ECAFF4200BA7E8D4353259D2B7E5E72B5096EFF8027D66D1043', owner),
        ('A4D978B7C4BDA15435D508F8B9592EC2A5ADFB12EA7BAD146A35ECB53094642F', owner),
        ('A6B5151F3655D3A2AF0D472759796BE4A4200E5495A7D869754C4848857408A7', owner),
        ('A7F32F508D4EB0FEAD9A087EF94ED1BA0AEC5DE6F7EF6FF0A62B93BEDF5D458D', owner),
        ('A924D3CAD6DA42B7399B96A095A06F18F6B1ABA5B873B0D5F3A0EE2173B48B6C', owner),
        ('AD3BE589C0474E97DE5BB2BF33534948B76BB80376DFDC58B1FED767B5A15BFC', owner),
        ('AD6826E1946D26D3EAF3685C88D97D85DE3B4DCB3D0EE2AE81C70560D13C5720', owner),
        ('AEEBAE3151271273ED95AA2E671139ED31A98567303A332298F83709A9D55AA1', owner),
        ('AFE2030AFB7D2CDA13F9FA333A02E34F6751AFEC11B010DBCD441FDF4C4002B3', owner),
        ('B3E506340FBF6B5786973393079F24B66BA46507E35E911DB0362A2ACDE97049', owner),
        ('B54F1EE636631FAD68058D3B0937031AC1B90CCB17062A391CCA68AFDBE40D55', owner),
        ('B8D6B5E7857B45830E017C7BE3D856ADEB97C7290EB0665A3D473A4BEB51DCF3', owner),
        ('B8F078D983A24AC433216393883514CD932C33AF18E7DD70884C8235F4275736', owner),
        ('B92AF298DC08049B78C77492D6551B710CD72AADA3D77BE54609E43278EF6E4D', owner),
        ('B93F0699598F8B20FA0DACC12CFCFC1F2568793F6E779E04795E6D7C22530F75', owner),
        ('B97A0889059C035FF1D54B6DB53B11B9766668D9F955247C028B2837D7A04CD9', owner),
        ('BADFF5E4F0FEA711701CA8FB22E4C43821E31E210CF52D1D4F74DD50F1D039BC', owner),
        ('BB01DA0333BB639C7E1C806DB0561DC98A5316F22FEF1090FB8D0BE46DAE499A', owner),
        ('BB1DD16D530008636F232303A7A86F3DFF969F848815C0574B12C2D787FEC93F', owner),
        ('BC75F910FF320F5CB5999E66BBD4034F4AE537A42FDFEF35161C5348E366E216', owner),
        ('BC87A668E81966489CB508EE805183C19E6ACD24CF17799CA062D2E384DA0EA7', owner),
        ('BDD01126E9D85710D3FE75AF1CC1702A29F081B4F6FDF6A2B2135C0297A9CEC5', owner),
        ('BE435DF7CD28AA2A7C8DB4FC8173475B77E5ABF392F76B7C76FA3F698CB71A9A', owner),
        ('BEF7663BE5EA4DBFD8686E24701E036F4C03FB7FCD67A6C566ED94CE09C44470', owner),
        ('C2469759C1947E14F4B65F72A9F5B3AF8B6F6E727B68BB0D91385CBF42176A8A', owner),
        ('C3505BF3EC10A51DACE417C76B8BD10939A065D1F34E75B8A3065EE31CC69B96', owner),
        ('C3A99A460DA464A057C3586D83CEF5F4AE08B7103979ED8932742DF0ED530C66', owner),
        ('C409BDAC4775ADD8DB92AA22B5B718FB8C94A1462C1FE9A416B95D8A3388C2FC', owner),
        ('C42D11C70CCF5E8CF3FB91FDF21D884021AD836CA68ADF2CBB7995C10BF588D4', owner),
        ('C452AB846073DF5ACE25CCA64D6B7A09D906308A1A65EB5240E3C4EBCAA9CC0C', owner),
        ('C5D9D8A186E2C82D09AFAA2A6F7F2E73870D3E64F72C4E08EF67796A840F0FBD', owner),
        ('C617C1A8B1EE2A811C28B5A81B4C83D7C98B5B0C27281D610207EBE692C2967F', owner),
        ('C69D64A5B839E41BA16742527E17056A18CE3C276FD26E34901A1BC7D0E32219', owner),
        ('C805603C4FA038776E42F263C604B49D96840322E1922D5606A9B0BBB5BFFE6F', owner),
        ('C83CB13922AD99F560744675DD37CC94DCAD5A1FCBA6472FEE341171D939E884', owner),
        ('C90F336617B8E7F983975413C997F10B73EB267FD8A10CB9E3BDBFC667ABDB8B', owner),
        ('C9EC350406F26E559AFFB4030DE2EBDE5435054C35A998605B8FCF04972D8D55', owner),
        ('CA171D614A8D7E121C93948CD0FE55D39981F9D11AA96E03450A415227C2C65B', owner),
        ('CB340011AFEB0D74C4A588B36EBAA441961608E8D2FA80DCA8C13872C850796B', owner),
        ('CB6B858B40D3A098765815B592C1514A49604FAFD60819DA88D7A76E9778FEF7', owner),
        ('CC8EEC6EB9212CBF897A5ACE7E8ABEECE1079F1A6DEF0A789591CB1547F1F084', owner),
        ('CE3BFABE59D67CE8AC8DFD4A16F7C43EF9C224513FBC655957D735FA29F540CE', owner),
        ('CF13A243C1CD2E3C8CEB7E70100387CECBFB830525BBF9D0B70C79ADF3E84128', owner),
        ('D063EC28F67EBA53F1642DBF7DFF33C6A32ADD869F6013FE162E2C32F1CBE56D', owner),
        ('D372C0D0F4FDC9F52E9E1F23FC56EE72414A17F350D0CEA6C26A35A6C3217A13', owner),
        ('D626157E1D6A718BC124AB8DA27CBB65072CA03A7B6B257DBDCBBD60F65EF3D1', owner),
        ('D89A11D16C488DD4FBBC541D4B07FAF8670D660994488FE54B1FBFF2704E4288', owner),
        ('D8CBEB9735F5672B367E4F96CDC74969615D17074AE96C724D42CE0216F8F3FA', owner),
        ('D8D4E6DDF6E42D74A6A536EA62FD1217E4290B145C9E5C3695A31B42EFB5F5A4', owner),
        ('D9668AB52785086786C134B5E4BDDBF72452813B6973229AB92AA1A54D201BF5', owner),
        ('DA3560FD0C32B54C83D4F2FF869003D2089369ACF2C89608F8AFA7436BFA4655', owner),
        ('DBAF9E056D3D5B38B68553304ABC88827EBC00F80CB9C7E197CDBC5822CD316C', owner),
        ('DCCC3CE1C00EE4B0B10487D372A0FA47F5C26F57A359BE7B27801E144EACBAC4', owner),
        ('DD59AF56084406E38C63FBE0850F30A0CD1277462A2192590FB05BC259E61273', owner),
        ('DF02AAB48387A9E1D4C65228089CB6ABE196C8F4B396C7E4BBC395DE136977F6', owner),
        ('DF91AC85A94FCD0CFB8155BD7CBEFAAC14B8C5EE7397FE2CC85984459E2EA14E', owner),
        ('E051B788ECBAEDA53046C70E6AF6058F95222C046157B8C4C1B9C2CFC65F46E5', owner),
        ('E19DAE83C02E6F281358D4EBD11D7723B4F5EA0E357907D5443DECC5F93C1E9D', owner),
        ('E24B315A551671483D8B9073B32DE11B4DE1EB2EAB211AFD2D9C319FF55E08D0', owner),
        ('E36DFC719D2114C2E39AEA88849E2845AB326F6F7FE74E0E539B7E54D81F3631', owner),
        ('E39891F48BBCC593B8ED86CE82CE666FC1145B9FCBFD2B07BAD0A89BF4C7BFBF', owner),
        ('E6856F137F79992DC94FA2F43297EC32D2D9A76F7BE66114C6A13EFC3BCDF5C8', owner),
        ('E6CA68E94146629AF03F69C2F86E6BEF62F930B37C6FBCC878B78DF98C0334E5', owner),
        ('E7681F153121EA1E67F74BBCB0CDC5E502702C1B8CC55FB65D702DFBA948B5F4', owner),
        ('E7C20B3AB481EC885501ECA5293781D84B5A1AC24F88266B5270E7ECB4AA2538', owner),
        ('E800395DBE0E045781E8005178B4BAF5A257F06E159121A67C595F6AE22506FD', owner),
        ('E92C22EB3B5642D65C1EC2CAF247D2594738EEBB7FB3841A44956F59E2B0D1FA', owner),
        ('EAFF8C85C208BA4D5B6B8046F5D6081747D779BADA7768E649D047FF9B1F660C', owner),
        ('EE83A566496109A74F6AC6E410DF00BB29A290E0021516AE3B8A23288E7E2E72', owner),
        ('EED7E0EFF2ED559E2A79EE361F9962AF3B1E999131E30BB7FD07546FAE0A7267', owner),
        ('F1863EC8B7F43F94AD14FB0B8B4A69497A8C65ECBC2A55E0BB420E772B8CDC91', owner),
        ('F1B4F6513B0D544A688D13ADC291EFA8C59F420CA5DCB23E0B5A06FA7E0D083D', owner),
        ('F277AF4F9BDC918AE89FA35CC1B34E34984C04AE9765322C3CB049574D36509C', owner),
        ('F2A16D35B554694187A70D40CA682959F4F35C2CE0EAB8FD64F7AC2AB9F5C24A', owner),
        ('F31FD461C5E99510403FC97C1DA2D8A9CBE270597D32BADF8FD66B77495F8D94', owner),
        ('F48E6DD8718E953B60A24F2CBEA60A9521DEAE67DB25425B7D3ACE3C517DD9B7', owner),
        ('F52F83A3FA9CFBD6920F722824DBE4034534D25B8507246B3B957DAC6E1BCE7A', owner),
        ('FDDD6E3D29EA84C7743DAD4A1BDBC700B5FEC1B391F932409086ACC71DD6DBD8', owner),
        ('FE63A84F782CC9D3FCF2CCF9FC11FBD03760878758D26285ED12669BDC6E6D01', owner),
        ('FECFB232D12E994B6D485D2C7167728AA5525984AD5CA61E7516221F079A1436', owner)
    ]
    canonical_hashes_esl = sha256_esl_builder(canonicalHashes)
    expected_canonical_esd = EfiSignatureDatabase(EslList=[canonical_hashes_esl, cert2_esl, cert1_esl])

    return (esd, expected_dupes_esd, expected_canonical_esd)


def MixedCertsData():
    """ Returns a tuple of (rawData, expectedDupes, expectedCanonical) for a mix of unsorted certs with dupes """

    owner = '77fa9abd-0359-4d32-bd60-28f4e78f784b'

    esls = [
        x509_esl_builder([(DEBIAN_CERT, owner)]),
        x509_esl_builder([(UBUNTU_CERT, owner)])
    ]

    input = [
        esls[1],
        esls[0],
        esls[1],
        esls[0],
        esls[1]
    ]
    esd = EfiSignatureDatabase(EslList=input)

    dupes = [
        esls[0],
        esls[1],
        esls[1]
    ]
    expected_dupes_esd = EfiSignatureDatabase(EslList=dupes)

    canonical = [
        esls[0],
        esls[1],
    ]
    expected_canonical_esd = EfiSignatureDatabase(EslList=canonical)

    return (esd, expected_dupes_esd, expected_canonical_esd)


class AuthVarSupportLibraryUnitTests(unittest.TestCase):

    def test_print(self):
        esd = basicEfiSignatureDatabase()
        print("\nPrint test:")
        esd.Print(compact=True)
        esd.Print(compact=False)

    def test_deserializeEqualsSerialize(self):
        # root = logging.getLogger()
        # root.setLevel(logging.INFO)

        original_esd = basicEfiSignatureDatabase()
        original_fs = io.BytesIO()
        original_esd.Write(original_fs)
        original_fs.seek(0)

        new_esd = EfiSignatureDatabase(filestream=original_fs)
        original_fs.close()

        self.assertEqual(original_esd.GetBytes(), new_esd.GetBytes())

    def test_EfiSignatureList_Sort_and_Deduplication_x509(self):
        owner = uuid.uuid4().hex

        subTestList = [
            (
                "SubTest_Empty",
                # input
                [],
                # expected after sort
                [],
                # expected duplicates
                [],
            ),
            (
                "SubTest_One_Entry",
                # input
                [(DEBIAN_CERT, owner)],
                # expected after sort
                [(DEBIAN_CERT, owner)],
                # expected duplicates
                [],
            )
        ]

        for subTest in subTestList:
            (testName, input, expected_sort, expected_dupes) = subTest
            with self.subTest(msg=testName):
                testEsl = x509_esl_builder(input)
                expected_sort_esl = x509_esl_builder(expected_sort)
                expected_dupes_esl = x509_esl_builder(expected_dupes)

                output_dupes = testEsl.SortBySignatureDataValue(deduplicate=True)

                self.assertEqual(output_dupes.GetBytes(), expected_dupes_esl.GetBytes())
                self.assertEqual(testEsl.GetBytes(), expected_sort_esl.GetBytes())

    def test_EfiSignatureList_Sort_and_Deduplication_sha256(self):

        owner1 = uuid.uuid4().hex
        owner2 = uuid.uuid4().hex

        subTestList = [
            (
                "SubTest_Empty",
                # input
                [],
                # expected after sort
                [],
                # expected duplicates
                [],
            ),
            (
                "SubTest_One_Entry",
                # input
                [
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                ],
                # expected duplicates
                [],
            ),
            (
                "SubTest_321",
                # input
                [
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected duplicates
                [],
            ),
            (
                "SubTest_32123",
                # input
                [
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected duplicates
                [
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
            ),
            (
                "SubTest_122333221",
                # input
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                ],
                # expected duplicates
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner1),
                    (HASHSTR[3], owner1),
                ],
            ),
            (
                "SubTest_8005551212",
                # input
                [
                    (HASHSTR[8], owner1),
                    (HASHSTR[0], owner1),
                    (HASHSTR[0], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1)
                ],
                # expected after sort
                [
                    (HASHSTR[0], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[8], owner1),
                ],
                # expected duplicates
                [
                    (HASHSTR[0], owner1),
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[5], owner1),
                    (HASHSTR[5], owner1),
                ],
            ),
            (
                "SubTest_122333221_mixed_sigowner",
                # input
                [
                    (HASHSTR[1], owner2),
                    (HASHSTR[2], owner2),
                    (HASHSTR[2], owner1),
                    (HASHSTR[3], owner2),
                    (HASHSTR[3], owner2),
                    (HASHSTR[3], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner2),
                    (HASHSTR[1], owner1),
                ],
                # expected after sort
                [
                    (HASHSTR[1], owner2),
                    (HASHSTR[2], owner2),
                    (HASHSTR[3], owner2),
                ],
                # expected duplicates
                [
                    (HASHSTR[1], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner1),
                    (HASHSTR[2], owner2),
                    (HASHSTR[3], owner2),
                    (HASHSTR[3], owner1),
                ],
            ),
        ]

        for subTest in subTestList:
            (testName, input, expected_sort, expected_dupes) = subTest
            with self.subTest(msg=testName):
                testEsl = sha256_esl_builder(input)
                expected_sort_esl = sha256_esl_builder(expected_sort)
                expected_dupes_esl = sha256_esl_builder(expected_dupes)

                output_dupes_esl = testEsl.SortBySignatureDataValue(deduplicate=True)

                self.assertEqual(testEsl.GetBytes(), expected_sort_esl.GetBytes())
                self.assertEqual(output_dupes_esl.GetBytes(), expected_dupes_esl.GetBytes())

    def test_EfiSignatureDatabase_Sort_and_Deduplication(self):
        subTestList = [
            ("MixedCerts", MixedCertsData()),
            ("BootHole", BootHoleData()),
        ]

        for subTest in subTestList:
            (testName, (Esd, ExpectedDupesEsd, ExpectedCanonicalEsd)) = subTest
            with self.subTest(msg=testName):

                (output_canonical_esd, output_dupes_esd) = Esd.GetCanonicalAndDupes()

                if VERBOSE:
                    print("Starting EfiSignatureDatabase: ")
                    Esd.Print(compact=True)

                    print("Dupes: ")
                    output_dupes_esd.Print(compact=True)

                    print("Canonical: ")
                    output_canonical_esd.Print(compact=True)

                self.assertEqual(output_dupes_esd.GetBytes(), ExpectedDupesEsd.GetBytes())
                self.assertEqual(output_canonical_esd.GetBytes(), ExpectedCanonicalEsd.GetBytes())
