"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdkPipeline = void 0;
const codepipeline = require("../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const actions_1 = require("./actions");
const construct_internals_1 = require("./private/construct-internals");
const stage_1 = require("./stage");
/**
 * A Pipeline to deploy CDK apps
 *
 * Defines an AWS CodePipeline-based Pipeline to deploy CDK applications.
 *
 * Automatically manages the following:
 *
 * - Stack dependency order.
 * - Asset publishing.
 * - Keeping the pipeline up-to-date as the CDK apps change.
 * - Using stack outputs later on in the pipeline.
 */
class CdkPipeline extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this._stages = [];
        this._outputArtifacts = {};
        if (!core_1.App.isApp(this.node.root)) {
            throw new Error('CdkPipeline must be created under an App');
        }
        this._cloudAssemblyArtifact = props.cloudAssemblyArtifact;
        const pipelineStack = core_1.Stack.of(this);
        this._pipeline = new codepipeline.Pipeline(this, 'Pipeline', {
            ...props,
            restartExecutionOnUpdate: true,
            stages: [
                {
                    stageName: 'Source',
                    actions: [props.sourceAction],
                },
                {
                    stageName: 'Build',
                    actions: [props.synthAction],
                },
                {
                    stageName: 'UpdatePipeline',
                    actions: [new actions_1.UpdatePipelineAction(this, 'UpdatePipeline', {
                            cloudAssemblyInput: this._cloudAssemblyArtifact,
                            pipelineStackName: pipelineStack.stackName,
                            cdkCliVersion: props.cdkCliVersion,
                            projectName: maybeSuffix(props.pipelineName, '-selfupdate'),
                        })],
                },
            ],
        });
        this._assets = new AssetPublishing(this, 'Assets', {
            cloudAssemblyInput: this._cloudAssemblyArtifact,
            cdkCliVersion: props.cdkCliVersion,
            pipeline: this._pipeline,
            projectName: maybeSuffix(props.pipelineName, '-publish'),
        });
    }
    /**
     * Add pipeline stage that will deploy the given application stage
     *
     * The application construct should subclass `Stage` and can contain any
     * number of `Stacks` inside it that may have dependency relationships
     * on one another.
     *
     * All stacks in the application will be deployed in the appropriate order,
     * and all assets found in the application will be added to the asset
     * publishing stage.
     */
    addApplicationStage(appStage, options = {}) {
        const stage = this.addStage(appStage.stageName);
        stage.addApplication(appStage, options);
        return stage;
    }
    /**
     * Add a new, empty stage to the pipeline
     *
     * Prefer to use `addApplicationStage` if you are intended to deploy a CDK
     * application, but you can use this method if you want to add other kinds of
     * Actions to a pipeline.
     */
    addStage(stageName) {
        const pipelineStage = this._pipeline.addStage({
            stageName,
        });
        const stage = new stage_1.CdkStage(this, stageName, {
            cloudAssemblyArtifact: this._cloudAssemblyArtifact,
            pipelineStage,
            stageName,
            host: {
                publishAsset: this._assets.addPublishAssetAction.bind(this._assets),
                stackOutputArtifact: (artifactId) => this._outputArtifacts[artifactId],
            },
        });
        this._stages.push(stage);
        return stage;
    }
    /**
     * Get the StackOutput object that holds this CfnOutput's value in this pipeline
     *
     * `StackOutput` can be used in validation actions later in the pipeline.
     */
    stackOutput(cfnOutput) {
        const stack = core_1.Stack.of(cfnOutput);
        if (!this._outputArtifacts[stack.artifactId]) {
            // We should have stored the ArtifactPath in the map, but its Artifact
            // property isn't publicly readable...
            this._outputArtifacts[stack.artifactId] = new codepipeline.Artifact(`Artifact_${stack.artifactId}_Outputs`);
        }
        return new stage_1.StackOutput(this._outputArtifacts[stack.artifactId].atPath('outputs.json'), cfnOutput.logicalId);
    }
    /**
     * Validate that we don't have any stacks violating dependency order in the pipeline
     *
     * Our own convenience methods will never generate a pipeline that does that (although
     * this is a nice verification), but a user can also add the stacks by hand.
     */
    validate() {
        const ret = new Array();
        ret.push(...this.validateDeployOrder());
        ret.push(...this.validateRequestedOutputs());
        return ret;
    }
    onPrepare() {
        super.onPrepare();
        // TODO: Support this in a proper way in the upstream library. For now, we
        // "un-add" the Assets stage if it turns out to be empty.
        this._assets.removeAssetsStageIfEmpty();
    }
    /**
     * Return all StackDeployActions in an ordered list
     */
    get stackActions() {
        return flatMap(this._pipeline.stages, s => s.actions.filter(isDeployAction));
    }
    *validateDeployOrder() {
        const stackActions = this.stackActions;
        for (const stackAction of stackActions) {
            // For every dependency, it must be executed in an action before this one is prepared.
            for (const depId of stackAction.dependencyStackArtifactIds) {
                const depAction = stackActions.find(s => s.stackArtifactId === depId);
                if (depAction === undefined) {
                    this.node.addWarning(`Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depId}', but that dependency is not deployed through the pipeline!`);
                }
                else if (!(depAction.executeRunOrder < stackAction.prepareRunOrder)) {
                    yield `Stack '${stackAction.stackName}' depends on stack ` +
                        `'${depAction.stackName}', but is deployed before it in the pipeline!`;
                }
            }
        }
    }
    *validateRequestedOutputs() {
        const artifactIds = this.stackActions.map(s => s.stackArtifactId);
        for (const artifactId of Object.keys(this._outputArtifacts)) {
            if (!artifactIds.includes(artifactId)) {
                yield `Trying to use outputs for Stack '${artifactId}', but Stack is not deployed in this pipeline. Add it to the pipeline.`;
            }
        }
    }
}
exports.CdkPipeline = CdkPipeline;
function isDeployAction(a) {
    return a instanceof actions_1.DeployCdkStackAction;
}
function flatMap(xs, f) {
    return Array.prototype.concat([], ...xs.map(f));
}
/**
 * Add appropriate publishing actions to the asset publishing stage
 */
class AssetPublishing extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.publishers = {};
        this._fileAssetCtr = 1;
        this._dockerAssetCtr = 1;
        this.myCxAsmRoot = path.resolve(construct_internals_1.assemblyBuilderOf(construct_internals_1.appOf(this)).outdir);
        // We MUST add the Stage immediately here, otherwise it will be in the wrong place
        // in the pipeline!
        this.stage = this.props.pipeline.addStage({ stageName: 'Assets' });
    }
    /**
     * Make sure there is an action in the stage to publish the given asset
     *
     * Assets are grouped by asset ID (which represent individual assets) so all assets
     * are published in parallel. For each assets, all destinations are published sequentially
     * so that we can reuse expensive operations between them (mostly: building a Docker image).
     */
    addPublishAssetAction(command) {
        // FIXME: this is silly, we need the relative path here but no easy way to get it
        const relativePath = path.relative(this.myCxAsmRoot, command.assetManifestPath);
        let action = this.publishers[command.assetId];
        if (!action) {
            // The asset ID would be a logical candidate for the construct path and project names, but if the asset
            // changes it leads to recreation of a number of Role/Policy/Project resources which is slower than
            // necessary. Number sequentially instead.
            //
            // FIXME: The ultimate best solution is probably to generate a single Project per asset type
            // and reuse that for all assets.
            const id = command.assetType === actions_1.AssetType.FILE ? `FileAsset${this._fileAssetCtr++}` : `DockerAsset${this._dockerAssetCtr++}`;
            action = this.publishers[command.assetId] = new actions_1.PublishAssetsAction(this, id, {
                actionName: command.assetId,
                cloudAssemblyInput: this.props.cloudAssemblyInput,
                cdkCliVersion: this.props.cdkCliVersion,
                assetType: command.assetType,
            });
            this.stage.addAction(action);
        }
        action.addPublishCommand(relativePath, command.assetSelector);
    }
    /**
     * Remove the Assets stage if it turns out we didn't add any Assets to publish
     */
    removeAssetsStageIfEmpty() {
        if (Object.keys(this.publishers).length === 0) {
            // Hacks to get access to innards of Pipeline
            // Modify 'stages' array in-place to remove Assets stage if empty
            const stages = this.props.pipeline._stages;
            const ix = stages.indexOf(this.stage);
            if (ix > -1) {
                stages.splice(ix, 1);
            }
        }
    }
}
function maybeSuffix(x, suffix) {
    if (x === undefined) {
        return undefined;
    }
    return `${x}${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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