"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/** Version "5.7.12". */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/** Version "5.7.mysql_aurora.2.03.2". */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/** Version "5.7.mysql_aurora.2.03.3". */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/** Version "5.7.mysql_aurora.2.03.4". */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/** Version "5.7.mysql_aurora.2.05.0". */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/** Version "5.7.mysql_aurora.2.06.0". */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/** Version "9.6.8". */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.11". */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.16". */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/** Version "10.4". */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10');
/** Version "10.11". */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10');
/** Version "10.12". */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10');
/** Version "11.4". */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11');
/** Version "11.6". */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11');
/** Version "11.7". */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11');
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
        });
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * The unversioned 'aurora' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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