"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (lambci)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                ...(_b = options.buildArgs) !== null && _b !== void 0 ? _b : {},
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_c = options.parcelVersion) !== null && _c !== void 0 ? _c : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_d = options.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_e = options.nodeModules) !== null && _e !== void 0 ? _e : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_f = options.sourceMaps) !== null && _f !== void 0 ? _f : false,
                    minify: (_g = options.minify) !== null && _g !== void 0 ? _g : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', containerEntryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', '/parcel-cache']
                : [],
        ].join(' ');
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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