"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // currently the custom resource requires a role to assume when interacting with the cluster
        // and we only have this role when kubectl is enabled.
        if (!props.cluster.kubectlEnabled) {
            throw new Error('adding Faregate Profiles to clusters without kubectl enabled is currently unsupported');
        }
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        this.podExecutionRole = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster._kubectlCreationRole);
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster._kubectlCreationRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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