"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.GeoRestriction = exports.ViewerCertificate = exports.LambdaEdgeEventType = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.HttpVersion = void 0;
const certificatemanager = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Controls the countries in which your content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                // eslint-disable-next-line max-len
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018,
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error('There must be at least one origin source - either an s3OriginSource or a customOriginSource');
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key],
                    };
                    originHeaders.push(oHeader);
                });
            }
            let s3OriginConfig;
            if (originConfig.s3OriginSource) {
                // first case for backwards compatibility
                if (originConfig.s3OriginSource.originAccessIdentity) {
                    // grant CloudFront OriginAccessIdentity read access to S3 bucket
                    originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                    s3OriginConfig = {
                        originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                    };
                }
                else {
                    s3OriginConfig = {};
                }
            }
            const connectionAttempts = (_a = originConfig.connectionAttempts) !== null && _a !== void 0 ? _a : 3;
            if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
                throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
            }
            const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
            if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
                throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeout
                            && originConfig.customOriginSource.originKeepaliveTimeout.toSeconds() || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeout
                            && originConfig.customOriginSource.originReadTimeout.toSeconds() || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                    }
                    : undefined,
                connectionAttempts,
                connectionTimeout,
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2ViX2Rpc3RyaWJ1dGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIndlYl9kaXN0cmlidXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsbUVBQW1FLENBQUMsa0VBQWtFO0FBQ3RJLHFDQUFxQyxDQUFDLG1EQUFtRDtBQUV6RixtQ0FBbUMsQ0FBQyxrREFBa0Q7QUFDdEYsa0NBQWtDLENBQUMsZ0RBQWdEO0FBQ25GLGlFQUF5RDtBQUN6RCxpREFBdUc7QUFFdkcsSUFBWSxXQUdYO0FBSEQsV0FBWSxXQUFXO0lBQ25CLGtDQUFtQixDQUFBO0lBQ25CLDhCQUFlLENBQUE7QUFDbkIsQ0FBQyxFQUhXLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBR3RCO0FBdUNEOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUNILElBQVksU0FHWDtBQUhELFdBQVksU0FBUztJQUNqQiw2QkFBZ0IsQ0FBQTtJQUNoQix3QkFBVyxDQUFBO0FBQ2YsQ0FBQyxFQUhXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBR3BCO0FBQ0Q7OztHQUdHO0FBQ0gsSUFBWSxzQkFNWDtBQU5ELFdBQVksc0JBQXNCO0lBQzlCLDBDQUFnQixDQUFBO0lBQ2hCLDBDQUFnQixDQUFBO0lBQ2hCLG9EQUEwQixDQUFBO0lBQzFCLHdEQUE4QixDQUFBO0lBQzlCLHdEQUE4QixDQUFBO0FBQ2xDLENBQUMsRUFOVyxzQkFBc0IsR0FBdEIsOEJBQXNCLEtBQXRCLDhCQUFzQixRQU1qQztBQXdIRCxJQUFZLGVBS1g7QUFMRCxXQUFZLGVBQWU7SUFDdkIsbUNBQWdCLENBQUE7SUFDaEIsbUNBQWdCLENBQUE7SUFDaEIsdUNBQW9CLENBQUE7SUFDcEIsdUNBQW9CLENBQUE7QUFDeEIsQ0FBQyxFQUxXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBSzFCO0FBZ0JEOztHQUVHO0FBQ0gsSUFBWSx3QkFJWDtBQUpELFdBQVksd0JBQXdCO0lBQ2hDLDJDQUFlLENBQUE7SUFDZixvREFBd0IsQ0FBQTtJQUN4Qix1Q0FBVyxDQUFBO0FBQ2YsQ0FBQyxFQUpXLHdCQUF3QixHQUF4QixnQ0FBd0IsS0FBeEIsZ0NBQXdCLFFBSW5DO0FBQ0Q7O0dBRUc7QUFDSCxJQUFZLDhCQUdYO0FBSEQsV0FBWSw4QkFBOEI7SUFDdEMsaURBQWUsQ0FBQTtJQUNmLDBEQUF3QixDQUFBO0FBQzVCLENBQUMsRUFIVyw4QkFBOEIsR0FBOUIsc0NBQThCLEtBQTlCLHNDQUE4QixRQUd6QztBQTBGRCxJQUFZLG1CQW1CWDtBQW5CRCxXQUFZLG1CQUFtQjtJQUMzQjs7O09BR0c7SUFDSCx3REFBaUMsQ0FBQTtJQUNqQzs7O09BR0c7SUFDSCwwREFBbUMsQ0FBQTtJQUNuQzs7T0FFRztJQUNILHdEQUFpQyxDQUFBO0lBQ2pDOztPQUVHO0lBQ0gsMERBQW1DLENBQUE7QUFDdkMsQ0FBQyxFQW5CVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQW1COUI7QUF5QkQ7O0dBRUc7QUFDSCxNQUFhLGlCQUFpQjtJQXFDMUIsWUFBb0MsS0FBZ0QsRUFBa0IsVUFBb0IsRUFBRTtRQUF4RixVQUFLLEdBQUwsS0FBSyxDQUEyQztRQUFrQixZQUFPLEdBQVAsT0FBTyxDQUFlO0lBQUksQ0FBQztJQXBDakk7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLGtCQUFrQixDQUFDLFdBQTRDLEVBQUUsVUFBb0MsRUFBRTtRQUNqSCxNQUFNLEVBQUUsU0FBUyxFQUFFLGdCQUFnQixHQUFHLFNBQVMsQ0FBQyxHQUFHLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLE9BQU8sR0FBRyxHQUFHLE9BQU8sQ0FBQztRQUNsSCxPQUFPLElBQUksaUJBQWlCLENBQUM7WUFDekIsaUJBQWlCLEVBQUUsV0FBVyxDQUFDLGNBQWMsRUFBRSxnQkFBZ0IsRUFBRSxzQkFBc0I7U0FDMUYsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNoQixDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsa0JBQWtCLENBQUMsZ0JBQXdCLEVBQUUsVUFBb0MsRUFBRTtRQUM3RixNQUFNLEVBQUUsU0FBUyxFQUFFLGdCQUFnQixHQUFHLFNBQVMsQ0FBQyxHQUFHLEVBQUUsY0FBYyxFQUFFLHNCQUFzQixFQUFFLE9BQU8sR0FBRyxHQUFHLE9BQU8sQ0FBQztRQUNsSCxPQUFPLElBQUksaUJBQWlCLENBQUM7WUFDekIsZ0JBQWdCLEVBQUUsZ0JBQWdCLEVBQUUsc0JBQXNCO1NBQzdELEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDaEIsQ0FBQztJQUNEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMsZ0NBQWdDLENBQUMsR0FBRyxPQUFpQjtRQUMvRCxPQUFPLElBQUksaUJBQWlCLENBQUMsRUFBRSw0QkFBNEIsRUFBRSxJQUFJLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRixDQUFDO0NBRUo7QUF0Q0QsOENBc0NDO0FBQ0Q7O0dBRUc7QUFDSCxNQUFhLGNBQWM7SUFrQ3ZCOzs7Ozs7O09BT0c7SUFDSCxZQUE2QixlQUEwQyxFQUFXLFNBQW1CO1FBQXhFLG9CQUFlLEdBQWYsZUFBZSxDQUEyQjtRQUFXLGNBQVMsR0FBVCxTQUFTLENBQVU7SUFBSSxDQUFDO0lBekMxRzs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBbUI7UUFDMUMsT0FBTyxJQUFJLGNBQWMsQ0FBQyxXQUFXLEVBQUUsY0FBYyxDQUFDLGlCQUFpQixDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7SUFDeEYsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFtQjtRQUMxQyxPQUFPLElBQUksY0FBYyxDQUFDLFdBQVcsRUFBRSxjQUFjLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztJQUN4RixDQUFDO0lBRU8sTUFBTSxDQUFDLGlCQUFpQixDQUFDLFNBQW1CO1FBQ2hELElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQ0FBb0MsQ0FBQyxDQUFDO1NBQ3pEO1FBQ0QsU0FBUyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUN6QixJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7Z0JBQy9DLG1DQUFtQztnQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsUUFBUSwrRUFBK0UsQ0FBQyxDQUFDO2FBQ3JKO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLFNBQVMsQ0FBQztJQUNyQixDQUFDOztBQWpDTCx3Q0EyQ0M7QUF0QmtCLDZCQUFjLEdBQUcsWUFBWSxDQUFDO0FBc0hqRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E2Qkc7QUFDSCxNQUFhLHlCQUEwQixTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBNEN2RCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQXFDOztRQUMvRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBckJyQjs7V0FFRztRQUNjLHNCQUFpQixHQUFHO1lBQ2pDLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUM7WUFDbkIsR0FBRyxFQUFFLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUM7WUFDL0IsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDO1NBQ3BFLENBQUM7UUFDRjs7V0FFRztRQUNjLHdCQUFtQixHQUVoQztZQUNBLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUNiLHNCQUFzQixDQUFDLE1BQU0sRUFBRSxzQkFBc0IsQ0FBQyxhQUFhO2dCQUNuRSxzQkFBc0IsQ0FBQyxXQUFXLEVBQUUsc0JBQXNCLENBQUMsYUFBYTthQUMzRTtZQUNELENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsTUFBTSxFQUFFLHNCQUFzQixDQUFDLE1BQU0sQ0FBQztTQUNsRixDQUFDO1FBR0UsSUFBSSxrQkFBa0IsR0FBK0M7WUFDakUsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLE9BQU8sRUFBRSxJQUFJO1lBQ2IsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxZQUFZO1lBQ2pHLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxJQUFJLFdBQVcsQ0FBQyxLQUFLO1lBQ25ELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxJQUFJLHlCQUFVLENBQUMsZUFBZTtZQUMxRCxXQUFXLEVBQUUsQ0FBQyxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJO1lBQ3ZFLG1DQUFtQztZQUNuQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsbUJBQW1CO1lBQy9DLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtTQUMzQixDQUFDO1FBQ0YsTUFBTSxTQUFTLEdBQXlCLEVBQUUsQ0FBQztRQUMzQyxNQUFNLE9BQU8sR0FBcUMsRUFBRSxDQUFDO1FBQ3JELElBQUksV0FBVyxHQUFHLENBQUMsQ0FBQztRQUNwQixLQUFLLE1BQU0sWUFBWSxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDNUMsTUFBTSxRQUFRLEdBQUcsU0FBUyxXQUFXLEVBQUUsQ0FBQztZQUN4QyxJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsSUFBSSxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbEUsTUFBTSxJQUFJLEtBQUssQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO2FBQ2xIO1lBQ0QsSUFBSSxZQUFZLENBQUMsa0JBQWtCLElBQUksWUFBWSxDQUFDLGNBQWMsRUFBRTtnQkFDaEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxrR0FBa0csQ0FBQyxDQUFDO2FBQ3ZIO1lBQ0QsTUFBTSxhQUFhLEdBQWlELEVBQUUsQ0FBQztZQUN2RSxJQUFJLFlBQVksQ0FBQyxhQUFhLEVBQUU7Z0JBQzVCLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsRUFBRTtvQkFDbEQsTUFBTSxPQUFPLEdBQStDO3dCQUN4RCxVQUFVLEVBQUUsR0FBRzt3QkFDZixXQUFXLEVBQUUsWUFBWSxDQUFDLGFBQWMsQ0FBQyxHQUFHLENBQUM7cUJBQ2hELENBQUM7b0JBQ0YsYUFBYSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDaEMsQ0FBQyxDQUFDLENBQUM7YUFDTjtZQUNELElBQUksY0FBa0UsQ0FBQztZQUN2RSxJQUFJLFlBQVksQ0FBQyxjQUFjLEVBQUU7Z0JBQzdCLHlDQUF5QztnQkFDekMsSUFBSSxZQUFZLENBQUMsY0FBYyxDQUFDLG9CQUFvQixFQUFFO29CQUNsRCxpRUFBaUU7b0JBQ2pFLFlBQVksQ0FBQyxjQUFjLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLENBQUM7b0JBQ3ZHLGNBQWMsR0FBRzt3QkFDYixvQkFBb0IsRUFBRSxxQ0FBcUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxvQkFBb0IsQ0FBQyx3QkFBd0IsRUFBRTtxQkFDekksQ0FBQztpQkFDTDtxQkFDSTtvQkFDRCxjQUFjLEdBQUcsRUFBRSxDQUFDO2lCQUN2QjthQUNKO1lBQ0QsTUFBTSxrQkFBa0IsU0FBRyxZQUFZLENBQUMsa0JBQWtCLG1DQUFJLENBQUMsQ0FBQztZQUNoRSxJQUFJLGtCQUFrQixHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsa0JBQWtCLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLEVBQUU7Z0JBQzNGLE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLENBQUMsQ0FBQzthQUNoRztZQUNELE1BQU0saUJBQWlCLEdBQUcsQ0FBQyxZQUFZLENBQUMsaUJBQWlCLElBQUksR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsQ0FBQztZQUNuRyxJQUFJLGlCQUFpQixHQUFHLENBQUMsSUFBSSxFQUFFLEdBQUcsaUJBQWlCLElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGlCQUFpQixDQUFDLEVBQUU7Z0JBQ3pGLE1BQU0sSUFBSSxLQUFLLENBQUMsc0ZBQXNGLENBQUMsQ0FBQzthQUMzRztZQUNELE1BQU0sY0FBYyxHQUFtQztnQkFDbkQsRUFBRSxFQUFFLFFBQVE7Z0JBQ1osVUFBVSxFQUFFLFlBQVksQ0FBQyxjQUFjO29CQUNuQyxDQUFDLENBQUMsWUFBWSxDQUFDLGNBQWMsQ0FBQyxjQUFjLENBQUMsd0JBQXdCO29CQUNyRSxDQUFDLENBQUMsWUFBWSxDQUFDLGtCQUFtQixDQUFDLFVBQVU7Z0JBQ2pELFVBQVUsRUFBRSxZQUFZLENBQUMsVUFBVTtnQkFDbkMsbUJBQW1CLEVBQUUsYUFBYSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDekUsY0FBYztnQkFDZCxrQkFBa0IsRUFBRSxZQUFZLENBQUMsa0JBQWtCO29CQUMvQyxDQUFDLENBQUM7d0JBQ0UsUUFBUSxFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLElBQUksRUFBRTt3QkFDeEQsU0FBUyxFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLElBQUksR0FBRzt3QkFDM0Qsc0JBQXNCLEVBQUUsWUFBWSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQjsrQkFDdkUsWUFBWSxDQUFDLGtCQUFrQixDQUFDLHNCQUFzQixDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUM7d0JBQzlFLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUI7K0JBQzdELFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsSUFBSSxFQUFFO3dCQUMxRSxvQkFBb0IsRUFBRSxZQUFZLENBQUMsa0JBQWtCLENBQUMsb0JBQW9CLElBQUksbUNBQW9CLENBQUMsVUFBVTt3QkFDN0csa0JBQWtCLEVBQUUsWUFBWSxDQUFDLGtCQUFrQixDQUFDLHdCQUF3QixJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQztxQkFDN0c7b0JBQ0QsQ0FBQyxDQUFDLFNBQVM7Z0JBQ2Ysa0JBQWtCO2dCQUNsQixpQkFBaUI7YUFDcEIsQ0FBQztZQUNGLEtBQUssTUFBTSxRQUFRLElBQUksWUFBWSxDQUFDLFNBQVMsRUFBRTtnQkFDM0MsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFLEdBQUcsUUFBUSxFQUFFLGNBQWMsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO2FBQzdEO1lBQ0QsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUM3QixXQUFXLEVBQUUsQ0FBQztTQUNqQjtRQUNELE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDckIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLElBQUksQ0FBQyxNQUFNLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ3RELE1BQU0sSUFBSSxLQUFLLENBQUMsVUFBVSxNQUFNLENBQUMsVUFBVSx3RkFBd0YsQ0FBQyxDQUFDO2FBQ3hJO1FBQ0wsQ0FBQyxDQUFDLENBQUM7UUFDSCxrQkFBa0IsR0FBRztZQUNqQixHQUFHLGtCQUFrQjtZQUNyQixPQUFPO1NBQ1YsQ0FBQztRQUNGLE1BQU0sZ0JBQWdCLEdBQUcsU0FBUyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQ2xGLElBQUksZ0JBQWdCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMvQixNQUFNLElBQUksS0FBSyxDQUFDLHVHQUF1RyxDQUFDLENBQUM7U0FDNUg7UUFDRCxrQkFBa0IsR0FBRyxFQUFFLEdBQUcsa0JBQWtCLEVBQUUsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDO1FBQ3ZJLE1BQU0sY0FBYyxHQUE0QyxFQUFFLENBQUM7UUFDbkUsS0FBSyxNQUFNLFFBQVEsSUFBSSxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUMsRUFBRTtZQUNoRSxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRTtnQkFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO2FBQzVFO1lBQ0QsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsb0JBQW9CLENBQTBDLENBQUMsQ0FBQztTQUN2SDtRQUNELGtCQUFrQixHQUFHLEVBQUUsR0FBRyxrQkFBa0IsRUFBRSxjQUFjLEVBQUUsY0FBYyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDdkgsSUFBSSxLQUFLLENBQUMsa0JBQWtCLElBQUksS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQ3JELE1BQU0sSUFBSSxLQUFLLENBQUM7Z0JBQ1osMEVBQTBFO2dCQUMxRSx5RUFBeUU7YUFDNUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztTQUNoQjtRQUNELElBQUksa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBQ2pELElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQzFCLE1BQU0sRUFBRSxVQUFVLEVBQUUsY0FBYyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDO1lBQzNGLGtCQUFrQixHQUFHLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUUsVUFBVSxDQUFDLEVBQUUsRUFBRSxjQUFjLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7U0FDcE07UUFDRCxJQUFJLGtCQUFrQixFQUFFO1lBQ3BCLE1BQU0sRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsT0FBTyxFQUFFLEdBQUcsa0JBQWtCLENBQUM7WUFDakUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxrQkFBa0IsRUFBRSxFQUFFLE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxDQUFDLENBQUM7WUFDbEUsTUFBTSxFQUFFLHNCQUFzQixFQUFFLGdCQUFnQixFQUFFLEdBQUcsaUJBQWlCLENBQUM7WUFDdkUsSUFBSSxzQkFBc0IsSUFBSSxJQUFJLElBQUksZ0JBQWdCLElBQUksSUFBSSxFQUFFO2dCQUM1RCxNQUFNLGNBQWMsR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsZ0JBQTZCLENBQUMsQ0FBQztnQkFDL0UsSUFBSSxjQUFjLENBQUMsT0FBTyxDQUFDLHNCQUFzQixDQUFDLFFBQVEsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUU7b0JBQ2xFLG1DQUFtQztvQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLHNCQUFzQixzQ0FBc0MsZ0JBQWdCLDZCQUE2QixjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztpQkFDNUo7YUFDSjtTQUNKO2FBQ0k7WUFDRCxrQkFBa0IsR0FBRyxFQUFFLEdBQUcsa0JBQWtCO2dCQUN4QyxpQkFBaUIsRUFBRSxFQUFFLDRCQUE0QixFQUFFLElBQUksRUFBRTthQUM1RCxDQUFDO1NBQ0w7UUFDRCxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDckIsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU0sSUFBSSxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1lBQ3hGLGtCQUFrQixHQUFHO2dCQUNqQixHQUFHLGtCQUFrQjtnQkFDckIsT0FBTyxFQUFFO29CQUNMLE1BQU0sRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLHdCQUF3QjtvQkFDbkQsY0FBYyxFQUFFLEtBQUssQ0FBQyxhQUFhLENBQUMsY0FBYyxJQUFJLEtBQUs7b0JBQzNELE1BQU0sRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLE1BQU07aUJBQ3JDO2FBQ0osQ0FBQztTQUNMO1FBQ0QsSUFBSSxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQ3RCLGtCQUFrQixHQUFHO2dCQUNqQixHQUFHLGtCQUFrQjtnQkFDckIsWUFBWSxFQUFFO29CQUNWLGNBQWMsRUFBRTt3QkFDWixlQUFlLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxlQUFlO3dCQUNyRCxTQUFTLEVBQUUsS0FBSyxDQUFDLGNBQWMsQ0FBQyxTQUFTO3FCQUM1QztpQkFDSjthQUNKLENBQUM7U0FDTDtRQUNELE1BQU0sWUFBWSxHQUFHLElBQUksc0NBQWUsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDekYsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO1FBQ3RDLElBQUksQ0FBQyxVQUFVLEdBQUcsWUFBWSxDQUFDLGNBQWMsQ0FBQztRQUM5QyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsWUFBWSxDQUFDLGNBQWMsQ0FBQztRQUMxRCxJQUFJLENBQUMsY0FBYyxHQUFHLFlBQVksQ0FBQyxHQUFHLENBQUM7SUFDM0MsQ0FBQztJQUNPLFVBQVUsQ0FBQyxLQUF5QixFQUFFLFdBQWtDO1FBQzVFLElBQUksUUFBUSxHQUFHO1lBQ1gsY0FBYyxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLHdCQUF3QixDQUFDLFFBQVEsQ0FBQztZQUNqRyxhQUFhLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxhQUFhLElBQUksOEJBQThCLENBQUMsUUFBUSxDQUFDO1lBQ3JHLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUSxLQUFLLEtBQUs7WUFDbEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxTQUFTLEVBQUU7WUFDNUQsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlLElBQUksRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsRUFBRTtZQUM5RixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRTtZQUNoRCxNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRTtZQUNoRCxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLG9CQUFvQixFQUFFLFdBQVcsSUFBSSxtQ0FBb0IsQ0FBQyxpQkFBaUI7U0FDOUUsQ0FBQztRQUNGLElBQUksQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUU7WUFDMUIsUUFBUSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFLEVBQUUsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQzFFO1FBQ0QsSUFBSSxLQUFLLENBQUMsMEJBQTBCLEVBQUU7WUFDbEMsUUFBUSxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxFQUFFO2dCQUMvQiwwQkFBMEIsRUFBRSxLQUFLLENBQUMsMEJBQTBCO3FCQUN2RCxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO29CQUNiLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUztvQkFDeEIsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLGNBQWMsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDLFdBQVc7aUJBQzFFLENBQUMsQ0FBQzthQUNOLENBQUMsQ0FBQztZQUNILDBFQUEwRTtZQUMxRSxLQUFLLE1BQU0sQ0FBQyxJQUFJLEtBQUssQ0FBQywwQkFBMEIsRUFBRTtnQkFDOUMsSUFBSSxDQUFDLENBQUMsY0FBYyxDQUFDLElBQUksSUFBSSxDQUFDLENBQUMsY0FBYyxDQUFDLElBQUksWUFBWSxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO29CQUM5RyxDQUFDLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO3dCQUN6RSxPQUFPLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQzt3QkFDM0IsVUFBVSxFQUFFLENBQUMsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsMEJBQTBCLENBQUMsQ0FBQztxQkFDckUsQ0FBQyxDQUFDLENBQUM7aUJBQ1A7YUFDSjtTQUNKO1FBQ0QsT0FBTyxRQUFRLENBQUM7SUFDcEIsQ0FBQztDQUNKO0FBblBELDhEQW1QQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNlcnRpZmljYXRlbWFuYWdlciBmcm9tIFwiLi4vLi4vYXdzLWNlcnRpZmljYXRlbWFuYWdlclwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNlcnRpZmljYXRlbWFuYWdlcidcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiLi4vLi4vYXdzLWxhbWJkYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSdcbmltcG9ydCAqIGFzIHMzIGZyb20gXCIuLi8uLi9hd3MtczNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMydcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENmbkRpc3RyaWJ1dGlvbiB9IGZyb20gJy4vY2xvdWRmcm9udC5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSURpc3RyaWJ1dGlvbiwgT3JpZ2luUHJvdG9jb2xQb2xpY3ksIFByaWNlQ2xhc3MsIFZpZXdlclByb3RvY29sUG9saWN5IH0gZnJvbSAnLi9kaXN0cmlidXRpb24nO1xuaW1wb3J0IHsgSU9yaWdpbkFjY2Vzc0lkZW50aXR5IH0gZnJvbSAnLi9vcmlnaW5fYWNjZXNzX2lkZW50aXR5JztcbmV4cG9ydCBlbnVtIEh0dHBWZXJzaW9uIHtcbiAgICBIVFRQMV8xID0gJ2h0dHAxLjEnLFxuICAgIEhUVFAyID0gJ2h0dHAyJ1xufVxuLyoqXG4gKiBDb25maWd1cmF0aW9uIGZvciBjdXN0b20gZG9tYWluIG5hbWVzXG4gKlxuICogQ2xvdWRGcm9udCBjYW4gdXNlIGEgY3VzdG9tIGRvbWFpbiB0aGF0IHlvdSBwcm92aWRlIGluc3RlYWQgb2YgYVxuICogXCJjbG91ZGZyb250Lm5ldFwiIGRvbWFpbi4gVG8gdXNlIHRoaXMgZmVhdHVyZSB5b3UgbXVzdCBwcm92aWRlIHRoZSBsaXN0IG9mXG4gKiBhZGRpdGlvbmFsIGRvbWFpbnMsIGFuZCB0aGUgQUNNIENlcnRpZmljYXRlIHRoYXQgQ2xvdWRGcm9udCBzaG91bGQgdXNlIGZvclxuICogdGhlc2UgYWRkaXRpb25hbCBkb21haW5zLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFsaWFzQ29uZmlndXJhdGlvbiB7XG4gICAgLyoqXG4gICAgICogQVJOIG9mIGFuIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIChBQ00pIGNlcnRpZmljYXRlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFjbUNlcnRSZWY6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBEb21haW4gbmFtZXMgb24gdGhlIGNlcnRpZmljYXRlXG4gICAgICpcbiAgICAgKiBCb3RoIG1haW4gZG9tYWluIG5hbWUgYW5kIFN1YmplY3QgQWx0ZXJuYXRpdmUgTmFtZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbmFtZXM6IHN0cmluZ1tdO1xuICAgIC8qKlxuICAgICAqIEhvdyBDbG91ZEZyb250IHNob3VsZCBzZXJ2ZSBIVFRQUyByZXF1ZXN0cy5cbiAgICAgKlxuICAgICAqIFNlZSB0aGUgbm90ZXMgb24gU1NMTWV0aG9kIGlmIHlvdSB3aXNoIHRvIHVzZSBvdGhlciBTU0wgdGVybWluYXRpb24gdHlwZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBTU0xNZXRob2QuU05JXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2xvdWRmcm9udC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9WaWV3ZXJDZXJ0aWZpY2F0ZS5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgc3NsTWV0aG9kPzogU1NMTWV0aG9kO1xuICAgIC8qKlxuICAgICAqIFRoZSBtaW5pbXVtIHZlcnNpb24gb2YgdGhlIFNTTCBwcm90b2NvbCB0aGF0IHlvdSB3YW50IENsb3VkRnJvbnQgdG8gdXNlIGZvciBIVFRQUyBjb25uZWN0aW9ucy5cbiAgICAgKlxuICAgICAqIENsb3VkRnJvbnQgc2VydmVzIHlvdXIgb2JqZWN0cyBvbmx5IHRvIGJyb3dzZXJzIG9yIGRldmljZXMgdGhhdCBzdXBwb3J0IGF0XG4gICAgICogbGVhc3QgdGhlIFNTTCB2ZXJzaW9uIHRoYXQgeW91IHNwZWNpZnkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFNTTHYzIGlmIHNzbE1ldGhvZCBWSVAsIFRMU3YxIGlmIHNzbE1ldGhvZCBTTklcbiAgICAgKi9cbiAgICByZWFkb25seSBzZWN1cml0eVBvbGljeT86IFNlY3VyaXR5UG9saWN5UHJvdG9jb2w7XG59XG4vKipcbiAqIFRoZSBTU0wgbWV0aG9kIENsb3VkRnJvbnQgd2lsbCB1c2UgZm9yIHlvdXIgZGlzdHJpYnV0aW9uLlxuICpcbiAqIFNlcnZlciBOYW1lIEluZGljYXRpb24gKFNOSSkgLSBpcyBhbiBleHRlbnNpb24gdG8gdGhlIFRMUyBjb21wdXRlciBuZXR3b3JraW5nIHByb3RvY29sIGJ5IHdoaWNoIGEgY2xpZW50IGluZGljYXRlc1xuICogIHdoaWNoIGhvc3RuYW1lIGl0IGlzIGF0dGVtcHRpbmcgdG8gY29ubmVjdCB0byBhdCB0aGUgc3RhcnQgb2YgdGhlIGhhbmRzaGFraW5nIHByb2Nlc3MuIFRoaXMgYWxsb3dzIGEgc2VydmVyIHRvIHByZXNlbnRcbiAqICBtdWx0aXBsZSBjZXJ0aWZpY2F0ZXMgb24gdGhlIHNhbWUgSVAgYWRkcmVzcyBhbmQgVENQIHBvcnQgbnVtYmVyIGFuZCBoZW5jZSBhbGxvd3MgbXVsdGlwbGUgc2VjdXJlIChIVFRQUykgd2Vic2l0ZXNcbiAqIChvciBhbnkgb3RoZXIgc2VydmljZSBvdmVyIFRMUykgdG8gYmUgc2VydmVkIGJ5IHRoZSBzYW1lIElQIGFkZHJlc3Mgd2l0aG91dCByZXF1aXJpbmcgYWxsIHRob3NlIHNpdGVzIHRvIHVzZSB0aGUgc2FtZSBjZXJ0aWZpY2F0ZS5cbiAqXG4gKiBDbG91ZEZyb250IGNhbiB1c2UgU05JIHRvIGhvc3QgbXVsdGlwbGUgZGlzdHJpYnV0aW9ucyBvbiB0aGUgc2FtZSBJUCAtIHdoaWNoIGEgbGFyZ2UgbWFqb3JpdHkgb2YgY2xpZW50cyB3aWxsIHN1cHBvcnQuXG4gKlxuICogSWYgeW91ciBjbGllbnRzIGNhbm5vdCBzdXBwb3J0IFNOSSBob3dldmVyIC0gQ2xvdWRGcm9udCBjYW4gdXNlIGRlZGljYXRlZCBJUHMgZm9yIHlvdXIgZGlzdHJpYnV0aW9uIC0gYnV0IHRoZXJlIGlzIGEgcHJvcmF0ZWQgbW9udGhseSBjaGFyZ2UgZm9yXG4gKiB1c2luZyB0aGlzIGZlYXR1cmUuIEJ5IGRlZmF1bHQsIHdlIHVzZSBTTkkgLSBidXQgeW91IGNhbiBvcHRpb25hbGx5IGVuYWJsZSBkZWRpY2F0ZWQgSVBzIChWSVApLlxuICpcbiAqIFNlZSB0aGUgQ2xvdWRGcm9udCBTU0wgZm9yIG1vcmUgZGV0YWlscyBhYm91dCBwcmljaW5nIDogaHR0cHM6Ly9hd3MuYW1hem9uLmNvbS9jbG91ZGZyb250L2N1c3RvbS1zc2wtZG9tYWlucy9cbiAqXG4gKi9cbmV4cG9ydCBlbnVtIFNTTE1ldGhvZCB7XG4gICAgU05JID0gJ3NuaS1vbmx5JyxcbiAgICBWSVAgPSAndmlwJ1xufVxuLyoqXG4gKiBUaGUgbWluaW11bSB2ZXJzaW9uIG9mIHRoZSBTU0wgcHJvdG9jb2wgdGhhdCB5b3Ugd2FudCBDbG91ZEZyb250IHRvIHVzZSBmb3IgSFRUUFMgY29ubmVjdGlvbnMuXG4gKiBDbG91ZEZyb250IHNlcnZlcyB5b3VyIG9iamVjdHMgb25seSB0byBicm93c2VycyBvciBkZXZpY2VzIHRoYXQgc3VwcG9ydCBhdCBsZWFzdCB0aGUgU1NMIHZlcnNpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAqL1xuZXhwb3J0IGVudW0gU2VjdXJpdHlQb2xpY3lQcm90b2NvbCB7XG4gICAgU1NMX1YzID0gJ1NTTHYzJyxcbiAgICBUTFNfVjEgPSAnVExTdjEnLFxuICAgIFRMU19WMV8yMDE2ID0gJ1RMU3YxXzIwMTYnLFxuICAgIFRMU19WMV8xXzIwMTYgPSAnVExTdjEuMV8yMDE2JyxcbiAgICBUTFNfVjFfMl8yMDE4ID0gJ1RMU3YxLjJfMjAxOCdcbn1cbi8qKlxuICogTG9nZ2luZyBjb25maWd1cmF0aW9uIGZvciBpbmNvbWluZyByZXF1ZXN0c1xuICovXG5leHBvcnQgaW50ZXJmYWNlIExvZ2dpbmdDb25maWd1cmF0aW9uIHtcbiAgICAvKipcbiAgICAgKiBCdWNrZXQgdG8gbG9nIHJlcXVlc3RzIHRvXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEEgbG9nZ2luZyBidWNrZXQgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldD86IHMzLklCdWNrZXQ7XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0byBpbmNsdWRlIHRoZSBjb29raWVzIGluIHRoZSBsb2dzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluY2x1ZGVDb29raWVzPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBXaGVyZSBpbiB0aGUgYnVja2V0IHRvIHN0b3JlIGxvZ3NcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gcHJlZml4LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZWZpeD86IHN0cmluZztcbn1cbi8qKlxuICogQSBzb3VyY2UgY29uZmlndXJhdGlvbiBpcyBhIHdyYXBwZXIgZm9yIENsb3VkRnJvbnQgb3JpZ2lucyBhbmQgYmVoYXZpb3JzLlxuICogQW4gb3JpZ2luIGlzIHdoYXQgQ2xvdWRGcm9udCB3aWxsIFwiYmUgaW4gZnJvbnQgb2ZcIiAtIHRoYXQgaXMsIENsb3VkRnJvbnQgd2lsbCBwdWxsIGl0J3MgYXNzZXRzIGZyb20gYW4gb3JpZ2luLlxuICpcbiAqIElmIHlvdSdyZSB1c2luZyBzMyBhcyBhIHNvdXJjZSAtIHBhc3MgdGhlIGBzM09yaWdpbmAgcHJvcGVydHksIG90aGVyd2lzZSwgcGFzcyB0aGUgYGN1c3RvbU9yaWdpblNvdXJjZWAgcHJvcGVydHkuXG4gKlxuICogT25lIG9yIHRoZSBvdGhlciBtdXN0IGJlIHBhc3NlZCwgYW5kIGl0IGlzIGludmFsaWQgdG8gcGFzcyBib3RoIGluIHRoZSBzYW1lIFNvdXJjZUNvbmZpZ3VyYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU291cmNlQ29uZmlndXJhdGlvbiB7XG4gICAgLyoqXG4gICAgICogVGhlIG51bWJlciBvZiB0aW1lcyB0aGF0IENsb3VkRnJvbnQgYXR0ZW1wdHMgdG8gY29ubmVjdCB0byB0aGUgb3JpZ2luLlxuICAgICAqIFlvdSBjYW4gc3BlY2lmeSAxLCAyLCBvciAzIGFzIHRoZSBudW1iZXIgb2YgYXR0ZW1wdHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAzXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29ubmVjdGlvbkF0dGVtcHRzPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2Ygc2Vjb25kcyB0aGF0IENsb3VkRnJvbnQgd2FpdHMgd2hlbiB0cnlpbmcgdG8gZXN0YWJsaXNoIGEgY29ubmVjdGlvbiB0byB0aGUgb3JpZ2luLlxuICAgICAqIFlvdSBjYW4gc3BlY2lmeSBhIG51bWJlciBvZiBzZWNvbmRzIGJldHdlZW4gMSBhbmQgMTAgKGluY2x1c2l2ZSkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBjZGsuRHVyYXRpb24uc2Vjb25kcygxMClcbiAgICAgKi9cbiAgICByZWFkb25seSBjb25uZWN0aW9uVGltZW91dD86IGNkay5EdXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBBbiBzMyBvcmlnaW4gc291cmNlIC0gaWYgeW91J3JlIHVzaW5nIHMzIGZvciB5b3VyIGFzc2V0c1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHMzT3JpZ2luU291cmNlPzogUzNPcmlnaW5Db25maWc7XG4gICAgLyoqXG4gICAgICogQSBjdXN0b20gb3JpZ2luIHNvdXJjZSAtIGZvciBhbGwgbm9uLXMzIHNvdXJjZXMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY3VzdG9tT3JpZ2luU291cmNlPzogQ3VzdG9tT3JpZ2luQ29uZmlnO1xuICAgIC8qKlxuICAgICAqIFRoZSBiZWhhdmlvcnMgYXNzb2NpYXRlZCB3aXRoIHRoaXMgc291cmNlLlxuICAgICAqIEF0IGxlYXN0IG9uZSAoZGVmYXVsdCkgYmVoYXZpb3IgbXVzdCBiZSBpbmNsdWRlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBiZWhhdmlvcnM6IEJlaGF2aW9yW107XG4gICAgLyoqXG4gICAgICogVGhlIHJlbGF0aXZlIHBhdGggdG8gdGhlIG9yaWdpbiByb290IHRvIHVzZSBmb3Igc291cmNlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC9cbiAgICAgKi9cbiAgICByZWFkb25seSBvcmlnaW5QYXRoPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEFueSBhZGRpdGlvbmFsIGhlYWRlcnMgdG8gcGFzcyB0byB0aGUgb3JpZ2luXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGFkZGl0aW9uYWwgaGVhZGVycyBhcmUgcGFzc2VkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9yaWdpbkhlYWRlcnM/OiB7XG4gICAgICAgIFtrZXk6IHN0cmluZ106IHN0cmluZztcbiAgICB9O1xufVxuLyoqXG4gKiBBIGN1c3RvbSBvcmlnaW4gY29uZmlndXJhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIEN1c3RvbU9yaWdpbkNvbmZpZyB7XG4gICAgLyoqXG4gICAgICogVGhlIGRvbWFpbiBuYW1lIG9mIHRoZSBjdXN0b20gb3JpZ2luLiBTaG91bGQgbm90IGluY2x1ZGUgdGhlIHBhdGggLSB0aGF0IHNob3VsZCBiZSBpbiB0aGUgcGFyZW50IFNvdXJjZUNvbmZpZ3VyYXRpb25cbiAgICAgKi9cbiAgICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIG9yaWdpbiBIVFRQIHBvcnRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IDgwXG4gICAgICovXG4gICAgcmVhZG9ubHkgaHR0cFBvcnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIG9yaWdpbiBIVFRQUyBwb3J0XG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCA0NDNcbiAgICAgKi9cbiAgICByZWFkb25seSBodHRwc1BvcnQ/OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGhlIGtlZXAgYWxpdmUgdGltZW91dCB3aGVuIG1ha2luZyBjYWxscyBpbiBzZWNvbmRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcyg1KVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9yaWdpbktlZXBhbGl2ZVRpbWVvdXQ/OiBjZGsuRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIHByb3RvY29sIChodHRwIG9yIGh0dHBzKSBwb2xpY3kgdG8gdXNlIHdoZW4gaW50ZXJhY3Rpbmcgd2l0aCB0aGUgb3JpZ2luLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgT3JpZ2luUHJvdG9jb2xQb2xpY3kuSHR0cHNPbmx5XG4gICAgICovXG4gICAgcmVhZG9ubHkgb3JpZ2luUHJvdG9jb2xQb2xpY3k/OiBPcmlnaW5Qcm90b2NvbFBvbGljeTtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVhZCB0aW1lb3V0IHdoZW4gY2FsbGluZyB0aGUgb3JpZ2luIGluIHNlY29uZHNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IER1cmF0aW9uLnNlY29uZHMoMzApXG4gICAgICovXG4gICAgcmVhZG9ubHkgb3JpZ2luUmVhZFRpbWVvdXQ/OiBjZGsuRHVyYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIFNTTCB2ZXJzaW9ucyB0byB1c2Ugd2hlbiBpbnRlcmFjdGluZyB3aXRoIHRoZSBvcmlnaW4uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBPcmlnaW5Tc2xQb2xpY3kuVExTdjFfMlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsbG93ZWRPcmlnaW5TU0xWZXJzaW9ucz86IE9yaWdpblNzbFBvbGljeVtdO1xufVxuZXhwb3J0IGVudW0gT3JpZ2luU3NsUG9saWN5IHtcbiAgICBTU0xfVjMgPSAnU1NMdjMnLFxuICAgIFRMU19WMSA9ICdUTFN2MScsXG4gICAgVExTX1YxXzEgPSAnVExTdjEuMScsXG4gICAgVExTX1YxXzIgPSAnVExTdjEuMidcbn1cbi8qKlxuICogUzMgb3JpZ2luIGNvbmZpZ3VyYXRpb24gZm9yIENsb3VkRnJvbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTM09yaWdpbkNvbmZpZyB7XG4gICAgLyoqXG4gICAgICogVGhlIHNvdXJjZSBidWNrZXQgdG8gc2VydmUgY29udGVudCBmcm9tXG4gICAgICovXG4gICAgcmVhZG9ubHkgczNCdWNrZXRTb3VyY2U6IHMzLklCdWNrZXQ7XG4gICAgLyoqXG4gICAgICogVGhlIG9wdGlvbmFsIE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgb2YgdGhlIG9yaWdpbiBpZGVudGl0eSBjbG91ZGZyb250IHdpbGwgdXNlIHdoZW4gY2FsbGluZyB5b3VyIHMzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIE9yaWdpbiBBY2Nlc3MgSWRlbnRpdHkgd2hpY2ggcmVxdWlyZXMgdGhlIFMzIGJ1Y2tldCB0byBiZSBwdWJsaWMgYWNjZXNzaWJsZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9yaWdpbkFjY2Vzc0lkZW50aXR5PzogSU9yaWdpbkFjY2Vzc0lkZW50aXR5O1xufVxuLyoqXG4gKiBBbiBlbnVtIGZvciB0aGUgc3VwcG9ydGVkIG1ldGhvZHMgdG8gYSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbi5cbiAqL1xuZXhwb3J0IGVudW0gQ2xvdWRGcm9udEFsbG93ZWRNZXRob2RzIHtcbiAgICBHRVRfSEVBRCA9ICdHSCcsXG4gICAgR0VUX0hFQURfT1BUSU9OUyA9ICdHSE8nLFxuICAgIEFMTCA9ICdBTEwnXG59XG4vKipcbiAqIEVudW1zIGZvciB0aGUgbWV0aG9kcyBDbG91ZEZyb250IGNhbiBjYWNoZS5cbiAqL1xuZXhwb3J0IGVudW0gQ2xvdWRGcm9udEFsbG93ZWRDYWNoZWRNZXRob2RzIHtcbiAgICBHRVRfSEVBRCA9ICdHSCcsXG4gICAgR0VUX0hFQURfT1BUSU9OUyA9ICdHSE8nXG59XG4vKipcbiAqIEEgQ2xvdWRGcm9udCBiZWhhdmlvciB3cmFwcGVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJlaGF2aW9yIHtcbiAgICAvKipcbiAgICAgKiBJZiBDbG91ZEZyb250IHNob3VsZCBhdXRvbWF0aWNhbGx5IGNvbXByZXNzIHNvbWUgY29udGVudCB0eXBlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRydWVcbiAgICAgKi9cbiAgICByZWFkb25seSBjb21wcmVzcz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogSWYgdGhpcyBiZWhhdmlvciBpcyB0aGUgZGVmYXVsdCBiZWhhdmlvciBmb3IgdGhlIGRpc3RyaWJ1dGlvbi5cbiAgICAgKlxuICAgICAqIFlvdSBtdXN0IHNwZWNpZnkgZXhhY3RseSBvbmUgZGVmYXVsdCBkaXN0cmlidXRpb24gcGVyIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uLlxuICAgICAqIFRoZSBkZWZhdWx0IGJlaGF2aW9yIGlzIGFsbG93ZWQgdG8gb21pdCB0aGUgXCJwYXRoXCIgcHJvcGVydHkuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaXNEZWZhdWx0QmVoYXZpb3I/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRydXN0ZWQgc2lnbmVycyBpcyBob3cgQ2xvdWRGcm9udCBhbGxvd3MgeW91IHRvIHNlcnZlIHByaXZhdGUgY29udGVudC5cbiAgICAgKiBUaGUgc2lnbmVycyBhcmUgdGhlIGFjY291bnQgSURzIHRoYXQgYXJlIGFsbG93ZWQgdG8gc2lnbiBjb29raWVzL3ByZXNpZ25lZCBVUkxzIGZvciB0aGlzIGRpc3RyaWJ1dGlvbi5cbiAgICAgKlxuICAgICAqIElmIHlvdSBwYXNzIGEgbm9uIGVtcHR5IHZhbHVlLCBhbGwgcmVxdWVzdHMgZm9yIHRoaXMgYmVoYXZpb3IgbXVzdCBiZSBzaWduZWQgKG5vIHB1YmxpYyBhY2Nlc3Mgd2lsbCBiZSBhbGxvd2VkKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRydXN0ZWRTaWduZXJzPzogc3RyaW5nW107XG4gICAgLyoqXG4gICAgICpcbiAgICAgKiBUaGUgZGVmYXVsdCBhbW91bnQgb2YgdGltZSBDbG91ZEZyb250IHdpbGwgY2FjaGUgYW4gb2JqZWN0LlxuICAgICAqXG4gICAgICogVGhpcyB2YWx1ZSBhcHBsaWVzIG9ubHkgd2hlbiB5b3VyIGN1c3RvbSBvcmlnaW4gZG9lcyBub3QgYWRkIEhUVFAgaGVhZGVycyxcbiAgICAgKiBzdWNoIGFzIENhY2hlLUNvbnRyb2wgbWF4LWFnZSwgQ2FjaGUtQ29udHJvbCBzLW1heGFnZSwgYW5kIEV4cGlyZXMgdG8gb2JqZWN0cy5cbiAgICAgKiBAZGVmYXVsdCA4NjQwMCAoMSBkYXkpXG4gICAgICpcbiAgICAgKi9cbiAgICByZWFkb25seSBkZWZhdWx0VHRsPzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBtZXRob2QgdGhpcyBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiByZXNwb25kcyBkby5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEdFVF9IRUFEXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWxsb3dlZE1ldGhvZHM/OiBDbG91ZEZyb250QWxsb3dlZE1ldGhvZHM7XG4gICAgLyoqXG4gICAgICogVGhlIHBhdGggdGhpcyBiZWhhdmlvciByZXNwb25kcyB0by5cbiAgICAgKiBSZXF1aXJlZCBmb3IgYWxsIG5vbi1kZWZhdWx0IGJlaGF2aW9ycy4gKFRoZSBkZWZhdWx0IGJlaGF2aW9yIGltcGxpY2l0bHkgaGFzIFwiKlwiIGFzIHRoZSBwYXRoIHBhdHRlcm4uIClcbiAgICAgKlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBhdGhQYXR0ZXJuPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFdoaWNoIG1ldGhvZHMgYXJlIGNhY2hlZCBieSBDbG91ZEZyb250IGJ5IGRlZmF1bHQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBHRVRfSEVBRFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNhY2hlZE1ldGhvZHM/OiBDbG91ZEZyb250QWxsb3dlZENhY2hlZE1ldGhvZHM7XG4gICAgLyoqXG4gICAgICogVGhlIHZhbHVlcyBDbG91ZEZyb250IHdpbGwgZm9yd2FyZCB0byB0aGUgb3JpZ2luIHdoZW4gbWFraW5nIGEgcmVxdWVzdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IG5vbmUgKG5vIGNvb2tpZXMgLSBubyBoZWFkZXJzKVxuICAgICAqXG4gICAgICovXG4gICAgcmVhZG9ubHkgZm9yd2FyZGVkVmFsdWVzPzogQ2ZuRGlzdHJpYnV0aW9uLkZvcndhcmRlZFZhbHVlc1Byb3BlcnR5O1xuICAgIC8qKlxuICAgICAqIFRoZSBtaW5pbXVtIGFtb3VudCBvZiB0aW1lIHRoYXQgeW91IHdhbnQgb2JqZWN0cyB0byBzdGF5IGluIHRoZSBjYWNoZVxuICAgICAqIGJlZm9yZSBDbG91ZEZyb250IHF1ZXJpZXMgeW91ciBvcmlnaW4uXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWluVHRsPzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXggYW1vdW50IG9mIHRpbWUgeW91IHdhbnQgb2JqZWN0cyB0byBzdGF5IGluIHRoZSBjYWNoZVxuICAgICAqIGJlZm9yZSBDbG91ZEZyb250IHF1ZXJpZXMgeW91ciBvcmlnaW4uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDMxNTM2MDAwKSAob25lIHllYXIpXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWF4VHRsPzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIERlY2xhcmVzIGFzc29jaWF0ZWQgbGFtYmRhQGVkZ2UgZnVuY3Rpb25zIGZvciB0aGlzIGRpc3RyaWJ1dGlvbiBiZWhhdmlvdXIuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBsYW1iZGEgZnVuY3Rpb24gYXNzb2NpYXRlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGxhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb25zPzogTGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbltdO1xufVxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbGFtYmRhIGV2ZW50IHR5cGUgZGVmaW5lcyBhdCB3aGljaCBldmVudCB0aGUgbGFtYmRhXG4gICAgICogaXMgY2FsbGVkIGR1cmluZyB0aGUgcmVxdWVzdCBsaWZlY3ljbGVcbiAgICAgKi9cbiAgICByZWFkb25seSBldmVudFR5cGU6IExhbWJkYUVkZ2VFdmVudFR5cGU7XG4gICAgLyoqXG4gICAgICogQSB2ZXJzaW9uIG9mIHRoZSBsYW1iZGEgdG8gYXNzb2NpYXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb246IGxhbWJkYS5JVmVyc2lvbjtcbn1cbmV4cG9ydCBlbnVtIExhbWJkYUVkZ2VFdmVudFR5cGUge1xuICAgIC8qKlxuICAgICAqIFRoZSBvcmlnaW4tcmVxdWVzdCBzcGVjaWZpZXMgdGhlIHJlcXVlc3QgdG8gdGhlXG4gICAgICogb3JpZ2luIGxvY2F0aW9uIChlLmcuIFMzKVxuICAgICAqL1xuICAgIE9SSUdJTl9SRVFVRVNUID0gJ29yaWdpbi1yZXF1ZXN0JyxcbiAgICAvKipcbiAgICAgKiBUaGUgb3JpZ2luLXJlc3BvbnNlIHNwZWNpZmllcyB0aGUgcmVzcG9uc2UgZnJvbSB0aGVcbiAgICAgKiBvcmlnaW4gbG9jYXRpb24gKGUuZy4gUzMpXG4gICAgICovXG4gICAgT1JJR0lOX1JFU1BPTlNFID0gJ29yaWdpbi1yZXNwb25zZScsXG4gICAgLyoqXG4gICAgICogVGhlIHZpZXdlci1yZXF1ZXN0IHNwZWNpZmllcyB0aGUgaW5jb21pbmcgcmVxdWVzdFxuICAgICAqL1xuICAgIFZJRVdFUl9SRVFVRVNUID0gJ3ZpZXdlci1yZXF1ZXN0JyxcbiAgICAvKipcbiAgICAgKiBUaGUgdmlld2VyLXJlc3BvbnNlIHNwZWNpZmllcyB0aGUgb3V0Z29pbmcgcmVwb25zZVxuICAgICAqL1xuICAgIFZJRVdFUl9SRVNQT05TRSA9ICd2aWV3ZXItcmVzcG9uc2UnXG59XG5leHBvcnQgaW50ZXJmYWNlIFZpZXdlckNlcnRpZmljYXRlT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogSG93IENsb3VkRnJvbnQgc2hvdWxkIHNlcnZlIEhUVFBTIHJlcXVlc3RzLlxuICAgICAqXG4gICAgICogU2VlIHRoZSBub3RlcyBvbiBTU0xNZXRob2QgaWYgeW91IHdpc2ggdG8gdXNlIG90aGVyIFNTTCB0ZXJtaW5hdGlvbiB0eXBlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFNTTE1ldGhvZC5TTklcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbG91ZGZyb250L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1ZpZXdlckNlcnRpZmljYXRlLmh0bWxcbiAgICAgKi9cbiAgICByZWFkb25seSBzc2xNZXRob2Q/OiBTU0xNZXRob2Q7XG4gICAgLyoqXG4gICAgICogVGhlIG1pbmltdW0gdmVyc2lvbiBvZiB0aGUgU1NMIHByb3RvY29sIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byB1c2UgZm9yIEhUVFBTIGNvbm5lY3Rpb25zLlxuICAgICAqXG4gICAgICogQ2xvdWRGcm9udCBzZXJ2ZXMgeW91ciBvYmplY3RzIG9ubHkgdG8gYnJvd3NlcnMgb3IgZGV2aWNlcyB0aGF0IHN1cHBvcnQgYXRcbiAgICAgKiBsZWFzdCB0aGUgU1NMIHZlcnNpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gU1NMdjMgaWYgc3NsTWV0aG9kIFZJUCwgVExTdjEgaWYgc3NsTWV0aG9kIFNOSVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlY3VyaXR5UG9saWN5PzogU2VjdXJpdHlQb2xpY3lQcm90b2NvbDtcbiAgICAvKipcbiAgICAgKiBEb21haW4gbmFtZXMgb24gdGhlIGNlcnRpZmljYXRlIChib3RoIG1haW4gZG9tYWluIG5hbWUgYW5kIFN1YmplY3QgQWx0ZXJuYXRpdmUgbmFtZXMpXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWxpYXNlcz86IHN0cmluZ1tdO1xufVxuLyoqXG4gKiBWaWV3ZXIgY2VydGlmaWNhdGUgY29uZmlndXJhdGlvbiBjbGFzc1xuICovXG5leHBvcnQgY2xhc3MgVmlld2VyQ2VydGlmaWNhdGUge1xuICAgIC8qKlxuICAgICAqIEdlbmVyYXRlIGFuIEFXUyBDZXJ0aWZpY2F0ZSBNYW5hZ2VyIChBQ00pIHZpZXdlciBjZXJ0aWZpY2F0ZSBjb25maWd1cmF0aW9uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gY2VydGlmaWNhdGUgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgKEFDTSkgY2VydGlmaWNhdGUuXG4gICAgICogICAgICAgICAgICAgICAgICAgIFlvdXIgY2VydGlmaWNhdGUgbXVzdCBiZSBsb2NhdGVkIGluIHRoZSB1cy1lYXN0LTEgKFVTIEVhc3QgKE4uIFZpcmdpbmlhKSkgcmVnaW9uIHRvIGJlIGFjY2Vzc2VkIGJ5IENsb3VkRnJvbnRcbiAgICAgKiBAcGFyYW0gb3B0aW9ucyBjZXJ0aWZpY2F0ZSBjb25maWd1cmF0aW9uIG9wdGlvbnNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21BY21DZXJ0aWZpY2F0ZShjZXJ0aWZpY2F0ZTogY2VydGlmaWNhdGVtYW5hZ2VyLklDZXJ0aWZpY2F0ZSwgb3B0aW9uczogVmlld2VyQ2VydGlmaWNhdGVPcHRpb25zID0ge30pIHtcbiAgICAgICAgY29uc3QgeyBzc2xNZXRob2Q6IHNzbFN1cHBvcnRNZXRob2QgPSBTU0xNZXRob2QuU05JLCBzZWN1cml0eVBvbGljeTogbWluaW11bVByb3RvY29sVmVyc2lvbiwgYWxpYXNlcywgfSA9IG9wdGlvbnM7XG4gICAgICAgIHJldHVybiBuZXcgVmlld2VyQ2VydGlmaWNhdGUoe1xuICAgICAgICAgICAgYWNtQ2VydGlmaWNhdGVBcm46IGNlcnRpZmljYXRlLmNlcnRpZmljYXRlQXJuLCBzc2xTdXBwb3J0TWV0aG9kLCBtaW5pbXVtUHJvdG9jb2xWZXJzaW9uLFxuICAgICAgICB9LCBhbGlhc2VzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogR2VuZXJhdGUgYW4gSUFNIHZpZXdlciBjZXJ0aWZpY2F0ZSBjb25maWd1cmF0aW9uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaWFtQ2VydGlmaWNhdGVJZCBJZGVudGlmaWVyIG9mIHRoZSBJQU0gY2VydGlmaWNhdGVcbiAgICAgKiBAcGFyYW0gb3B0aW9ucyBjZXJ0aWZpY2F0ZSBjb25maWd1cmF0aW9uIG9wdGlvbnNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21JYW1DZXJ0aWZpY2F0ZShpYW1DZXJ0aWZpY2F0ZUlkOiBzdHJpbmcsIG9wdGlvbnM6IFZpZXdlckNlcnRpZmljYXRlT3B0aW9ucyA9IHt9KSB7XG4gICAgICAgIGNvbnN0IHsgc3NsTWV0aG9kOiBzc2xTdXBwb3J0TWV0aG9kID0gU1NMTWV0aG9kLlNOSSwgc2VjdXJpdHlQb2xpY3k6IG1pbmltdW1Qcm90b2NvbFZlcnNpb24sIGFsaWFzZXMsIH0gPSBvcHRpb25zO1xuICAgICAgICByZXR1cm4gbmV3IFZpZXdlckNlcnRpZmljYXRlKHtcbiAgICAgICAgICAgIGlhbUNlcnRpZmljYXRlSWQsIHNzbFN1cHBvcnRNZXRob2QsIG1pbmltdW1Qcm90b2NvbFZlcnNpb24sXG4gICAgICAgIH0sIGFsaWFzZXMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBHZW5lcmF0ZSBhIHZpZXdlciBjZXJ0aWZjYXRlIGNvbmZpZ3VyYXRpb24gdXNpbmdcbiAgICAgKiB0aGUgQ2xvdWRGcm9udCBkZWZhdWx0IGNlcnRpZmljYXRlIChlLmcuIGQxMTExMTFhYmNkZWY4LmNsb3VkZnJvbnQubmV0KVxuICAgICAqIGFuZCBhIHtAbGluayBTZWN1cml0eVBvbGljeVByb3RvY29sLlRMU19WMX0gc2VjdXJpdHkgcG9saWN5LlxuICAgICAqXG4gICAgICogQHBhcmFtIGFsaWFzZXMgQWx0ZXJuYXRpdmUgQ05BTUUgYWxpYXNlc1xuICAgICAqICAgICAgICAgICAgICAgIFlvdSBhbHNvIG11c3QgY3JlYXRlIGEgQ05BTUUgcmVjb3JkIHdpdGggeW91ciBETlMgc2VydmljZSB0byByb3V0ZSBxdWVyaWVzXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBmcm9tQ2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZSguLi5hbGlhc2VzOiBzdHJpbmdbXSkge1xuICAgICAgICByZXR1cm4gbmV3IFZpZXdlckNlcnRpZmljYXRlKHsgY2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZTogdHJ1ZSB9LCBhbGlhc2VzKTtcbiAgICB9XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgcHJvcHM6IENmbkRpc3RyaWJ1dGlvbi5WaWV3ZXJDZXJ0aWZpY2F0ZVByb3BlcnR5LCBwdWJsaWMgcmVhZG9ubHkgYWxpYXNlczogc3RyaW5nW10gPSBbXSkgeyB9XG59XG4vKipcbiAqIENvbnRyb2xzIHRoZSBjb3VudHJpZXMgaW4gd2hpY2ggeW91ciBjb250ZW50IGlzIGRpc3RyaWJ1dGVkLlxuICovXG5leHBvcnQgY2xhc3MgR2VvUmVzdHJpY3Rpb24ge1xuICAgIC8qKlxuICAgICAqIFdoaXRlbGlzdCBzcGVjaWZpYyBjb3VudHJpZXMgd2hpY2ggeW91IHdhbnQgQ2xvdWRGcm9udCB0byBkaXN0cmlidXRlIHlvdXIgY29udGVudC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBsb2NhdGlvbnMgVHdvLWxldHRlciwgdXBwZXJjYXNlIGNvdW50cnkgY29kZSBmb3IgYSBjb3VudHJ5XG4gICAgICogdGhhdCB5b3Ugd2FudCB0byB3aGl0ZWxpc3QuIEluY2x1ZGUgb25lIGVsZW1lbnQgZm9yIGVhY2ggY291bnRyeS5cbiAgICAgKiBTZWUgSVNPIDMxNjYtMS1hbHBoYS0yIGNvZGUgb24gdGhlICpJbnRlcm5hdGlvbmFsIE9yZ2FuaXphdGlvbiBmb3IgU3RhbmRhcmRpemF0aW9uKiB3ZWJzaXRlXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyB3aGl0ZWxpc3QoLi4ubG9jYXRpb25zOiBzdHJpbmdbXSkge1xuICAgICAgICByZXR1cm4gbmV3IEdlb1Jlc3RyaWN0aW9uKCd3aGl0ZWxpc3QnLCBHZW9SZXN0cmljdGlvbi52YWxpZGF0ZUxvY2F0aW9ucyhsb2NhdGlvbnMpKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQmxhY2tsaXN0IHNwZWNpZmljIGNvdW50cmllcyB3aGljaCB5b3UgZG9uJ3Qgd2FudCBDbG91ZEZyb250IHRvIGRpc3RyaWJ1dGUgeW91ciBjb250ZW50LlxuICAgICAqXG4gICAgICogQHBhcmFtIGxvY2F0aW9ucyBUd28tbGV0dGVyLCB1cHBlcmNhc2UgY291bnRyeSBjb2RlIGZvciBhIGNvdW50cnlcbiAgICAgKiB0aGF0IHlvdSB3YW50IHRvIGJsYWNrbGlzdC4gSW5jbHVkZSBvbmUgZWxlbWVudCBmb3IgZWFjaCBjb3VudHJ5LlxuICAgICAqIFNlZSBJU08gMzE2Ni0xLWFscGhhLTIgY29kZSBvbiB0aGUgKkludGVybmF0aW9uYWwgT3JnYW5pemF0aW9uIGZvciBTdGFuZGFyZGl6YXRpb24qIHdlYnNpdGVcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGJsYWNrbGlzdCguLi5sb2NhdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgICAgIHJldHVybiBuZXcgR2VvUmVzdHJpY3Rpb24oJ2JsYWNrbGlzdCcsIEdlb1Jlc3RyaWN0aW9uLnZhbGlkYXRlTG9jYXRpb25zKGxvY2F0aW9ucykpO1xuICAgIH1cbiAgICBwcml2YXRlIHN0YXRpYyBMT0NBVElPTl9SRUdFWCA9IC9eW0EtWl17Mn0kLztcbiAgICBwcml2YXRlIHN0YXRpYyB2YWxpZGF0ZUxvY2F0aW9ucyhsb2NhdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgICAgIGlmIChsb2NhdGlvbnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Nob3VsZCBwcm92aWRlIGF0IGxlYXN0IDEgbG9jYXRpb24nKTtcbiAgICAgICAgfVxuICAgICAgICBsb2NhdGlvbnMuZm9yRWFjaChsb2NhdGlvbiA9PiB7XG4gICAgICAgICAgICBpZiAoIUdlb1Jlc3RyaWN0aW9uLkxPQ0FUSU9OX1JFR0VYLnRlc3QobG9jYXRpb24pKSB7XG4gICAgICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgbG9jYXRpb24gZm9ybWF0IGZvciBsb2NhdGlvbjogJHtsb2NhdGlvbn0sIGxvY2F0aW9uIHNob3VsZCBiZSB0d28tbGV0dGVyIGFuZCB1cHBlcmNhc2UgY291bnRyeSBJU08gMzE2Ni0xLWFscGhhLTIgY29kZWApO1xuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIGxvY2F0aW9ucztcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlcyBhbiBpbnN0YW5jZSBvZiBHZW9SZXN0cmljdGlvbiBmb3IgaW50ZXJuYWwgdXNlXG4gICAgICpcbiAgICAgKiBAcGFyYW0gcmVzdHJpY3Rpb25UeXBlIFNwZWNpZmllcyB0aGUgcmVzdHJpY3Rpb24gdHlwZSB0byBpbXBvc2UgKHdoaXRlbGlzdCBvciBibGFja2xpc3QpXG4gICAgICogQHBhcmFtIGxvY2F0aW9ucyBUd28tbGV0dGVyLCB1cHBlcmNhc2UgY291bnRyeSBjb2RlIGZvciBhIGNvdW50cnlcbiAgICAgKiB0aGF0IHlvdSB3YW50IHRvIHdoaXRlbGlzdC9ibGFja2xpc3QuIEluY2x1ZGUgb25lIGVsZW1lbnQgZm9yIGVhY2ggY291bnRyeS5cbiAgICAgKiBTZWUgSVNPIDMxNjYtMS1hbHBoYS0yIGNvZGUgb24gdGhlICpJbnRlcm5hdGlvbmFsIE9yZ2FuaXphdGlvbiBmb3IgU3RhbmRhcmRpemF0aW9uKiB3ZWJzaXRlXG4gICAgICovXG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihyZWFkb25seSByZXN0cmljdGlvblR5cGU6ICd3aGl0ZWxpc3QnIHwgJ2JsYWNrbGlzdCcsIHJlYWRvbmx5IGxvY2F0aW9uczogc3RyaW5nW10pIHsgfVxufVxuZXhwb3J0IGludGVyZmFjZSBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMge1xuICAgIC8qKlxuICAgICAqIEFsaWFzQ29uZmlndXJhdGlvbiBpcyB1c2VkIHRvIGNvbmZpZ3VyZWQgQ2xvdWRGcm9udCB0byByZXNwb25kIHRvIHJlcXVlc3RzIG9uIGN1c3RvbSBkb21haW4gbmFtZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vbmUuXG4gICAgICogQGRlcHJlY2F0ZWQgc2VlIHtAbGluayBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMjdmlld2VyQ2VydGlmaWNhdGV9IHdpdGgge0BsaW5rIFZpZXdlckNlcnRpZmljYXRlI2FjbUNlcnRpZmljYXRlfVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsaWFzQ29uZmlndXJhdGlvbj86IEFsaWFzQ29uZmlndXJhdGlvbjtcbiAgICAvKipcbiAgICAgKiBBIGNvbW1lbnQgZm9yIHRoaXMgZGlzdHJpYnV0aW9uIGluIHRoZSBDbG91ZEZyb250IGNvbnNvbGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNvbW1lbnQgaXMgYWRkZWQgdG8gZGlzdHJpYnV0aW9uLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGRlZmF1bHQgb2JqZWN0IHRvIHNlcnZlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBcImluZGV4Lmh0bWxcIiBpcyBzZXJ2ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVmYXVsdFJvb3RPYmplY3Q/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogSWYgeW91ciBkaXN0cmlidXRpb24gc2hvdWxkIGhhdmUgSVB2NiBlbmFibGVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVuYWJsZUlwVjY/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBtYXggc3VwcG9ydGVkIEhUVFAgVmVyc2lvbnMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBIdHRwVmVyc2lvbi5IVFRQMlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGh0dHBWZXJzaW9uPzogSHR0cFZlcnNpb247XG4gICAgLyoqXG4gICAgICogVGhlIHByaWNlIGNsYXNzIGZvciB0aGUgZGlzdHJpYnV0aW9uICh0aGlzIGltcGFjdHMgaG93IG1hbnkgbG9jYXRpb25zIENsb3VkRnJvbnQgdXNlcyBmb3IgeW91ciBkaXN0cmlidXRpb24sIGFuZCBiaWxsaW5nKVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgUHJpY2VDbGFzcy5QUklDRV9DTEFTU18xMDAgdGhlIGNoZWFwZXN0IG9wdGlvbiBmb3IgQ2xvdWRGcm9udCBpcyBwaWNrZWQgYnkgZGVmYXVsdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBwcmljZUNsYXNzPzogUHJpY2VDbGFzcztcbiAgICAvKipcbiAgICAgKiBUaGUgZGVmYXVsdCB2aWV3ZXIgcG9saWN5IGZvciBpbmNvbWluZyBjbGllbnRzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgUmVkaXJlY3RUb0hUVFBzXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmlld2VyUHJvdG9jb2xQb2xpY3k/OiBWaWV3ZXJQcm90b2NvbFBvbGljeTtcbiAgICAvKipcbiAgICAgKiBUaGUgb3JpZ2luIGNvbmZpZ3VyYXRpb25zIGZvciB0aGlzIGRpc3RyaWJ1dGlvbi4gQmVoYXZpb3JzIGFyZSBhIHBhcnQgb2YgdGhlIG9yaWdpbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBvcmlnaW5Db25maWdzOiBTb3VyY2VDb25maWd1cmF0aW9uW107XG4gICAgLyoqXG4gICAgICogT3B0aW9uYWwgLSBpZiB3ZSBzaG91bGQgZW5hYmxlIGxvZ2dpbmcuXG4gICAgICogWW91IGNhbiBwYXNzIGFuIGVtcHR5IG9iamVjdCAoe30pIHRvIGhhdmUgdXMgYXV0byBjcmVhdGUgYSBidWNrZXQgZm9yIGxvZ2dpbmcuXG4gICAgICogT21pc3Npb24gb2YgdGhpcyBwcm9wZXJ0eSBpbmRpY2F0ZXMgbm8gbG9nZ2luZyBpcyB0byBiZSBlbmFibGVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyBsb2dnaW5nIGlzIGVuYWJsZWQgYnkgZGVmYXVsdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBsb2dnaW5nQ29uZmlnPzogTG9nZ2luZ0NvbmZpZ3VyYXRpb247XG4gICAgLyoqXG4gICAgICogSG93IENsb3VkRnJvbnQgc2hvdWxkIGhhbmRsZSByZXF1ZXN0cyB0aGF0IGFyZSBub3Qgc3VjY2Vzc2Z1bCAoZWcgUGFnZU5vdEZvdW5kKVxuICAgICAqXG4gICAgICogQnkgZGVmYXVsdCwgQ2xvdWRGcm9udCBkb2VzIG5vdCByZXBsYWNlIEhUVFAgc3RhdHVzIGNvZGVzIGluIHRoZSA0eHggYW5kIDV4eCByYW5nZVxuICAgICAqIHdpdGggY3VzdG9tIGVycm9yIG1lc3NhZ2VzLiBDbG91ZEZyb250IGRvZXMgbm90IGNhY2hlIEhUVFAgc3RhdHVzIGNvZGVzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBjdXN0b20gZXJyb3IgY29uZmlndXJhdGlvbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBlcnJvckNvbmZpZ3VyYXRpb25zPzogQ2ZuRGlzdHJpYnV0aW9uLkN1c3RvbUVycm9yUmVzcG9uc2VQcm9wZXJ0eVtdO1xuICAgIC8qKlxuICAgICAqIFVuaXF1ZSBpZGVudGlmaWVyIHRoYXQgc3BlY2lmaWVzIHRoZSBBV1MgV0FGIHdlYiBBQ0wgdG8gYXNzb2NpYXRlIHdpdGggdGhpcyBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbi5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3doYXQtaXMtYXdzLXdhZi5odG1sXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIEFXUyBXZWIgQXBwbGljYXRpb24gRmlyZXdhbGwgd2ViIGFjY2VzcyBjb250cm9sIGxpc3QgKHdlYiBBQ0wpLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHdlYkFDTElkPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyB3aGV0aGVyIHlvdSB3YW50IHZpZXdlcnMgdG8gdXNlIEhUVFAgb3IgSFRUUFMgdG8gcmVxdWVzdCB5b3VyIG9iamVjdHMsXG4gICAgICogd2hldGhlciB5b3UncmUgdXNpbmcgYW4gYWx0ZXJuYXRlIGRvbWFpbiBuYW1lIHdpdGggSFRUUFMsIGFuZCBpZiBzbyxcbiAgICAgKiBpZiB5b3UncmUgdXNpbmcgQVdTIENlcnRpZmljYXRlIE1hbmFnZXIgKEFDTSkgb3IgYSB0aGlyZC1wYXJ0eSBjZXJ0aWZpY2F0ZSBhdXRob3JpdHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBWaWV3ZXJDZXJ0aWZpY2F0ZS5mcm9tQ2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZSgpXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vYXdzLmFtYXpvbi5jb20vcHJlbWl1bXN1cHBvcnQva25vd2xlZGdlLWNlbnRlci9jdXN0b20tc3NsLWNlcnRpZmljYXRlLWNsb3VkZnJvbnQvXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmlld2VyQ2VydGlmaWNhdGU/OiBWaWV3ZXJDZXJ0aWZpY2F0ZTtcbiAgICAvKipcbiAgICAgKiBDb250cm9scyB0aGUgY291bnRyaWVzIGluIHdoaWNoIHlvdXIgY29udGVudCBpcyBkaXN0cmlidXRlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGdlbyByZXN0cmljdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IGdlb1Jlc3RyaWN0aW9uPzogR2VvUmVzdHJpY3Rpb247XG59XG4vKipcbiAqIEludGVybmFsIG9ubHkgLSBqdXN0IGFkZHMgdGhlIG9yaWdpbklkIHN0cmluZyB0byB0aGUgQmVoYXZpb3JcbiAqL1xuaW50ZXJmYWNlIEJlaGF2aW9yV2l0aE9yaWdpbiBleHRlbmRzIEJlaGF2aW9yIHtcbiAgICByZWFkb25seSB0YXJnZXRPcmlnaW5JZDogc3RyaW5nO1xufVxuLyoqXG4gKiBBbWF6b24gQ2xvdWRGcm9udCBpcyBhIGdsb2JhbCBjb250ZW50IGRlbGl2ZXJ5IG5ldHdvcmsgKENETikgc2VydmljZSB0aGF0IHNlY3VyZWx5IGRlbGl2ZXJzIGRhdGEsIHZpZGVvcyxcbiAqIGFwcGxpY2F0aW9ucywgYW5kIEFQSXMgdG8geW91ciB2aWV3ZXJzIHdpdGggbG93IGxhdGVuY3kgYW5kIGhpZ2ggdHJhbnNmZXIgc3BlZWRzLlxuICogQ2xvdWRGcm9udCBmcm9udHMgdXNlciBwcm92aWRlZCBjb250ZW50IGFuZCBjYWNoZXMgaXQgYXQgZWRnZSBsb2NhdGlvbnMgYWNyb3NzIHRoZSB3b3JsZC5cbiAqXG4gKiBIZXJlJ3MgaG93IHlvdSBjYW4gdXNlIHRoaXMgY29uc3RydWN0OlxuICpcbiAqIGBgYHRzXG4gKiBpbXBvcnQgeyBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnXG4gKlxuICogY29uc3Qgc291cmNlQnVja2V0ID0gbmV3IEJ1Y2tldCh0aGlzLCAnQnVja2V0Jyk7XG4gKlxuICogY29uc3QgZGlzdHJpYnV0aW9uID0gbmV3IENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24odGhpcywgJ015RGlzdHJpYnV0aW9uJywge1xuICogIG9yaWdpbkNvbmZpZ3M6IFtcbiAqICAgIHtcbiAqICAgICAgczNPcmlnaW5Tb3VyY2U6IHtcbiAqICAgICAgczNCdWNrZXRTb3VyY2U6IHNvdXJjZUJ1Y2tldFxuICogICAgICB9LFxuICogICAgICBiZWhhdmlvcnMgOiBbIHtpc0RlZmF1bHRCZWhhdmlvcjogdHJ1ZX1dXG4gKiAgICB9XG4gKiAgXVxuICogfSk7XG4gKiBgYGBcbiAqXG4gKiBUaGlzIHdpbGwgY3JlYXRlIGEgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gdGhhdCB1c2VzIHlvdXIgUzNCdWNrZXQgYXMgaXQncyBvcmlnaW4uXG4gKlxuICogWW91IGNhbiBjdXN0b21pemUgdGhlIGRpc3RyaWJ1dGlvbiB1c2luZyBhZGRpdGlvbmFsIHByb3BlcnRpZXMgZnJvbSB0aGUgQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvblByb3BzIGludGVyZmFjZS5cbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb25cbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24gZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJRGlzdHJpYnV0aW9uIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbG9nZ2luZyBidWNrZXQgZm9yIHRoaXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gICAgICogSWYgbG9nZ2luZyBpcyBub3QgZW5hYmxlZCBmb3IgdGhpcyBkaXN0cmlidXRpb24gLSB0aGlzIHByb3BlcnR5IHdpbGwgYmUgdW5kZWZpbmVkLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBsb2dnaW5nQnVja2V0PzogczMuSUJ1Y2tldDtcbiAgICAvKipcbiAgICAgKiBUaGUgZG9tYWluIG5hbWUgY3JlYXRlZCBieSBDbG91ZEZyb250IGZvciB0aGlzIGRpc3RyaWJ1dGlvbi5cbiAgICAgKiBJZiB5b3UgYXJlIHVzaW5nIGFsaWFzZXMgZm9yIHlvdXIgZGlzdHJpYnV0aW9uLCB0aGlzIGlzIHRoZSBkb21haW5OYW1lIHlvdXIgRE5TIHJlY29yZHMgc2hvdWxkIHBvaW50IHRvLlxuICAgICAqIChJbiBSb3V0ZTUzLCB5b3UgY291bGQgY3JlYXRlIGFuIEFMSUFTIHJlY29yZCB0byB0aGlzIHZhbHVlLCBmb3IgZXhhbXBsZS4pXG4gICAgICpcbiAgICAgKiBAZGVwcmVjYXRlZCAtIFVzZSBgZGlzdHJpYnV0aW9uRG9tYWluTmFtZWAgaW5zdGVhZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBkb21haW4gbmFtZSBjcmVhdGVkIGJ5IENsb3VkRnJvbnQgZm9yIHRoaXMgZGlzdHJpYnV0aW9uLlxuICAgICAqIElmIHlvdSBhcmUgdXNpbmcgYWxpYXNlcyBmb3IgeW91ciBkaXN0cmlidXRpb24sIHRoaXMgaXMgdGhlIGRvbWFpbk5hbWUgeW91ciBETlMgcmVjb3JkcyBzaG91bGQgcG9pbnQgdG8uXG4gICAgICogKEluIFJvdXRlNTMsIHlvdSBjb3VsZCBjcmVhdGUgYW4gQUxJQVMgcmVjb3JkIHRvIHRoaXMgdmFsdWUsIGZvciBleGFtcGxlLilcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZGlzdHJpYnV0aW9uRG9tYWluTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBkaXN0cmlidXRpb24gSUQgZm9yIHRoaXMgZGlzdHJpYnV0aW9uLlxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSBkaXN0cmlidXRpb25JZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIE1hcHMgb3VyIG1ldGhvZHMgdG8gdGhlIHN0cmluZyBhcnJheXMgdGhleSBhcmVcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IE1FVEhPRF9MT09LVVBfTUFQID0ge1xuICAgICAgICBHSDogWydHRVQnLCAnSEVBRCddLFxuICAgICAgICBHSE86IFsnR0VUJywgJ0hFQUQnLCAnT1BUSU9OUyddLFxuICAgICAgICBBTEw6IFsnREVMRVRFJywgJ0dFVCcsICdIRUFEJywgJ09QVElPTlMnLCAnUEFUQ0gnLCAnUE9TVCcsICdQVVQnXSxcbiAgICB9O1xuICAgIC8qKlxuICAgICAqIE1hcHMgZm9yIHdoaWNoIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wgYXJlIGF2YWlsYWJsZSB0byB3aGljaCBTU0xNZXRob2RzXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSBWQUxJRF9TU0xfUFJPVE9DT0xTOiB7XG4gICAgICAgIFttZXRob2QgaW4gU1NMTWV0aG9kXTogc3RyaW5nW107XG4gICAgfSA9IHtcbiAgICAgICAgW1NTTE1ldGhvZC5TTkldOiBbXG4gICAgICAgICAgICBTZWN1cml0eVBvbGljeVByb3RvY29sLlRMU19WMSwgU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFNfVjFfMV8yMDE2LFxuICAgICAgICAgICAgU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFNfVjFfMjAxNiwgU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFNfVjFfMl8yMDE4LFxuICAgICAgICBdLFxuICAgICAgICBbU1NMTWV0aG9kLlZJUF06IFtTZWN1cml0eVBvbGljeVByb3RvY29sLlNTTF9WMywgU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFNfVjFdLFxuICAgIH07XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgbGV0IGRpc3RyaWJ1dGlvbkNvbmZpZzogQ2ZuRGlzdHJpYnV0aW9uLkRpc3RyaWJ1dGlvbkNvbmZpZ1Byb3BlcnR5ID0ge1xuICAgICAgICAgICAgY29tbWVudDogcHJvcHMuY29tbWVudCxcbiAgICAgICAgICAgIGVuYWJsZWQ6IHRydWUsXG4gICAgICAgICAgICBkZWZhdWx0Um9vdE9iamVjdDogcHJvcHMuZGVmYXVsdFJvb3RPYmplY3QgIT09IHVuZGVmaW5lZCA/IHByb3BzLmRlZmF1bHRSb290T2JqZWN0IDogJ2luZGV4Lmh0bWwnLFxuICAgICAgICAgICAgaHR0cFZlcnNpb246IHByb3BzLmh0dHBWZXJzaW9uIHx8IEh0dHBWZXJzaW9uLkhUVFAyLFxuICAgICAgICAgICAgcHJpY2VDbGFzczogcHJvcHMucHJpY2VDbGFzcyB8fCBQcmljZUNsYXNzLlBSSUNFX0NMQVNTXzEwMCxcbiAgICAgICAgICAgIGlwdjZFbmFibGVkOiAocHJvcHMuZW5hYmxlSXBWNiAhPT0gdW5kZWZpbmVkKSA/IHByb3BzLmVuYWJsZUlwVjYgOiB0cnVlLFxuICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgICAgIGN1c3RvbUVycm9yUmVzcG9uc2VzOiBwcm9wcy5lcnJvckNvbmZpZ3VyYXRpb25zLFxuICAgICAgICAgICAgd2ViQWNsSWQ6IHByb3BzLndlYkFDTElkLFxuICAgICAgICB9O1xuICAgICAgICBjb25zdCBiZWhhdmlvcnM6IEJlaGF2aW9yV2l0aE9yaWdpbltdID0gW107XG4gICAgICAgIGNvbnN0IG9yaWdpbnM6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5Qcm9wZXJ0eVtdID0gW107XG4gICAgICAgIGxldCBvcmlnaW5JbmRleCA9IDE7XG4gICAgICAgIGZvciAoY29uc3Qgb3JpZ2luQ29uZmlnIG9mIHByb3BzLm9yaWdpbkNvbmZpZ3MpIHtcbiAgICAgICAgICAgIGNvbnN0IG9yaWdpbklkID0gYG9yaWdpbiR7b3JpZ2luSW5kZXh9YDtcbiAgICAgICAgICAgIGlmICghb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlICYmICFvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGVyZSBtdXN0IGJlIGF0IGxlYXN0IG9uZSBvcmlnaW4gc291cmNlIC0gZWl0aGVyIGFuIHMzT3JpZ2luU291cmNlIG9yIGEgY3VzdG9tT3JpZ2luU291cmNlJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBpZiAob3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZSAmJiBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2UpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RoZXJlIGNhbm5vdCBiZSBib3RoIGFuIHMzT3JpZ2luU291cmNlIGFuZCBhIGN1c3RvbU9yaWdpblNvdXJjZSBpbiB0aGUgc2FtZSBTb3VyY2VDb25maWd1cmF0aW9uLicpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgY29uc3Qgb3JpZ2luSGVhZGVyczogQ2ZuRGlzdHJpYnV0aW9uLk9yaWdpbkN1c3RvbUhlYWRlclByb3BlcnR5W10gPSBbXTtcbiAgICAgICAgICAgIGlmIChvcmlnaW5Db25maWcub3JpZ2luSGVhZGVycykge1xuICAgICAgICAgICAgICAgIE9iamVjdC5rZXlzKG9yaWdpbkNvbmZpZy5vcmlnaW5IZWFkZXJzKS5mb3JFYWNoKGtleSA9PiB7XG4gICAgICAgICAgICAgICAgICAgIGNvbnN0IG9IZWFkZXI6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5DdXN0b21IZWFkZXJQcm9wZXJ0eSA9IHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGhlYWRlck5hbWU6IGtleSxcbiAgICAgICAgICAgICAgICAgICAgICAgIGhlYWRlclZhbHVlOiBvcmlnaW5Db25maWcub3JpZ2luSGVhZGVycyFba2V5XSxcbiAgICAgICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgICAgICAgb3JpZ2luSGVhZGVycy5wdXNoKG9IZWFkZXIpO1xuICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgbGV0IHMzT3JpZ2luQ29uZmlnOiBDZm5EaXN0cmlidXRpb24uUzNPcmlnaW5Db25maWdQcm9wZXJ0eSB8IHVuZGVmaW5lZDtcbiAgICAgICAgICAgIGlmIChvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2UpIHtcbiAgICAgICAgICAgICAgICAvLyBmaXJzdCBjYXNlIGZvciBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eVxuICAgICAgICAgICAgICAgIGlmIChvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2Uub3JpZ2luQWNjZXNzSWRlbnRpdHkpIHtcbiAgICAgICAgICAgICAgICAgICAgLy8gZ3JhbnQgQ2xvdWRGcm9udCBPcmlnaW5BY2Nlc3NJZGVudGl0eSByZWFkIGFjY2VzcyB0byBTMyBidWNrZXRcbiAgICAgICAgICAgICAgICAgICAgb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlLnMzQnVja2V0U291cmNlLmdyYW50UmVhZChvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2Uub3JpZ2luQWNjZXNzSWRlbnRpdHkpO1xuICAgICAgICAgICAgICAgICAgICBzM09yaWdpbkNvbmZpZyA9IHtcbiAgICAgICAgICAgICAgICAgICAgICAgIG9yaWdpbkFjY2Vzc0lkZW50aXR5OiBgb3JpZ2luLWFjY2Vzcy1pZGVudGl0eS9jbG91ZGZyb250LyR7b3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlLm9yaWdpbkFjY2Vzc0lkZW50aXR5Lm9yaWdpbkFjY2Vzc0lkZW50aXR5TmFtZX1gLFxuICAgICAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgczNPcmlnaW5Db25maWcgPSB7fTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCBjb25uZWN0aW9uQXR0ZW1wdHMgPSBvcmlnaW5Db25maWcuY29ubmVjdGlvbkF0dGVtcHRzID8/IDM7XG4gICAgICAgICAgICBpZiAoY29ubmVjdGlvbkF0dGVtcHRzIDwgMSB8fCAzIDwgY29ubmVjdGlvbkF0dGVtcHRzIHx8ICFOdW1iZXIuaXNJbnRlZ2VyKGNvbm5lY3Rpb25BdHRlbXB0cykpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2Nvbm5lY3Rpb25BdHRlbXB0czogWW91IGNhbiBzcGVjaWZ5IDEsIDIsIG9yIDMgYXMgdGhlIG51bWJlciBvZiBhdHRlbXB0cy4nKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGNvbnN0IGNvbm5lY3Rpb25UaW1lb3V0ID0gKG9yaWdpbkNvbmZpZy5jb25uZWN0aW9uVGltZW91dCB8fCBjZGsuRHVyYXRpb24uc2Vjb25kcygxMCkpLnRvU2Vjb25kcygpO1xuICAgICAgICAgICAgaWYgKGNvbm5lY3Rpb25UaW1lb3V0IDwgMSB8fCAxMCA8IGNvbm5lY3Rpb25UaW1lb3V0IHx8ICFOdW1iZXIuaXNJbnRlZ2VyKGNvbm5lY3Rpb25UaW1lb3V0KSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignY29ubmVjdGlvblRpbWVvdXQ6IFlvdSBjYW4gc3BlY2lmeSBhIG51bWJlciBvZiBzZWNvbmRzIGJldHdlZW4gMSBhbmQgMTAgKGluY2x1c2l2ZSkuJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCBvcmlnaW5Qcm9wZXJ0eTogQ2ZuRGlzdHJpYnV0aW9uLk9yaWdpblByb3BlcnR5ID0ge1xuICAgICAgICAgICAgICAgIGlkOiBvcmlnaW5JZCxcbiAgICAgICAgICAgICAgICBkb21haW5OYW1lOiBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2VcbiAgICAgICAgICAgICAgICAgICAgPyBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2UuczNCdWNrZXRTb3VyY2UuYnVja2V0UmVnaW9uYWxEb21haW5OYW1lXG4gICAgICAgICAgICAgICAgICAgIDogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZSEuZG9tYWluTmFtZSxcbiAgICAgICAgICAgICAgICBvcmlnaW5QYXRoOiBvcmlnaW5Db25maWcub3JpZ2luUGF0aCxcbiAgICAgICAgICAgICAgICBvcmlnaW5DdXN0b21IZWFkZXJzOiBvcmlnaW5IZWFkZXJzLmxlbmd0aCA+IDAgPyBvcmlnaW5IZWFkZXJzIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIHMzT3JpZ2luQ29uZmlnLFxuICAgICAgICAgICAgICAgIGN1c3RvbU9yaWdpbkNvbmZpZzogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZVxuICAgICAgICAgICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGh0dHBQb3J0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLmh0dHBQb3J0IHx8IDgwLFxuICAgICAgICAgICAgICAgICAgICAgICAgaHR0cHNQb3J0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLmh0dHBzUG9ydCB8fCA0NDMsXG4gICAgICAgICAgICAgICAgICAgICAgICBvcmlnaW5LZWVwYWxpdmVUaW1lb3V0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLm9yaWdpbktlZXBhbGl2ZVRpbWVvdXRcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAmJiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLm9yaWdpbktlZXBhbGl2ZVRpbWVvdXQudG9TZWNvbmRzKCkgfHwgNSxcbiAgICAgICAgICAgICAgICAgICAgICAgIG9yaWdpblJlYWRUaW1lb3V0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLm9yaWdpblJlYWRUaW1lb3V0XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJiYgb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5vcmlnaW5SZWFkVGltZW91dC50b1NlY29uZHMoKSB8fCAzMCxcbiAgICAgICAgICAgICAgICAgICAgICAgIG9yaWdpblByb3RvY29sUG9saWN5OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLm9yaWdpblByb3RvY29sUG9saWN5IHx8IE9yaWdpblByb3RvY29sUG9saWN5LkhUVFBTX09OTFksXG4gICAgICAgICAgICAgICAgICAgICAgICBvcmlnaW5Tc2xQcm90b2NvbHM6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UuYWxsb3dlZE9yaWdpblNTTFZlcnNpb25zIHx8IFtPcmlnaW5Tc2xQb2xpY3kuVExTX1YxXzJdLFxuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIGNvbm5lY3Rpb25BdHRlbXB0cyxcbiAgICAgICAgICAgICAgICBjb25uZWN0aW9uVGltZW91dCxcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgICBmb3IgKGNvbnN0IGJlaGF2aW9yIG9mIG9yaWdpbkNvbmZpZy5iZWhhdmlvcnMpIHtcbiAgICAgICAgICAgICAgICBiZWhhdmlvcnMucHVzaCh7IC4uLmJlaGF2aW9yLCB0YXJnZXRPcmlnaW5JZDogb3JpZ2luSWQgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBvcmlnaW5zLnB1c2gob3JpZ2luUHJvcGVydHkpO1xuICAgICAgICAgICAgb3JpZ2luSW5kZXgrKztcbiAgICAgICAgfVxuICAgICAgICBvcmlnaW5zLmZvckVhY2gob3JpZ2luID0+IHtcbiAgICAgICAgICAgIGlmICghb3JpZ2luLnMzT3JpZ2luQ29uZmlnICYmICFvcmlnaW4uY3VzdG9tT3JpZ2luQ29uZmlnKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBPcmlnaW4gJHtvcmlnaW4uZG9tYWluTmFtZX0gaXMgbWlzc2luZyBlaXRoZXIgUzNPcmlnaW5Db25maWcgb3IgQ3VzdG9tT3JpZ2luQ29uZmlnLiBBdCBsZWFzdCAxIG11c3QgYmUgc3BlY2lmaWVkLmApO1xuICAgICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgICAgZGlzdHJpYnV0aW9uQ29uZmlnID0ge1xuICAgICAgICAgICAgLi4uZGlzdHJpYnV0aW9uQ29uZmlnLFxuICAgICAgICAgICAgb3JpZ2lucyxcbiAgICAgICAgfTtcbiAgICAgICAgY29uc3QgZGVmYXVsdEJlaGF2aW9ycyA9IGJlaGF2aW9ycy5maWx0ZXIoYmVoYXZpb3IgPT4gYmVoYXZpb3IuaXNEZWZhdWx0QmVoYXZpb3IpO1xuICAgICAgICBpZiAoZGVmYXVsdEJlaGF2aW9ycy5sZW5ndGggIT09IDEpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVGhlcmUgY2FuIG9ubHkgYmUgb25lIGRlZmF1bHQgYmVoYXZpb3IgYWNyb3NzIGFsbCBzb3VyY2VzLiBbIE9uZSBkZWZhdWx0IGJlaGF2aW9yIHBlciBkaXN0cmlidXRpb24gXS4nKTtcbiAgICAgICAgfVxuICAgICAgICBkaXN0cmlidXRpb25Db25maWcgPSB7IC4uLmRpc3RyaWJ1dGlvbkNvbmZpZywgZGVmYXVsdENhY2hlQmVoYXZpb3I6IHRoaXMudG9CZWhhdmlvcihkZWZhdWx0QmVoYXZpb3JzWzBdLCBwcm9wcy52aWV3ZXJQcm90b2NvbFBvbGljeSkgfTtcbiAgICAgICAgY29uc3Qgb3RoZXJCZWhhdmlvcnM6IENmbkRpc3RyaWJ1dGlvbi5DYWNoZUJlaGF2aW9yUHJvcGVydHlbXSA9IFtdO1xuICAgICAgICBmb3IgKGNvbnN0IGJlaGF2aW9yIG9mIGJlaGF2aW9ycy5maWx0ZXIoYiA9PiAhYi5pc0RlZmF1bHRCZWhhdmlvcikpIHtcbiAgICAgICAgICAgIGlmICghYmVoYXZpb3IucGF0aFBhdHRlcm4pIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ3BhdGhQYXR0ZXJuIGlzIHJlcXVpcmVkIGZvciBhbGwgbm9uLWRlZmF1bHQgYmVoYXZpb3JzJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBvdGhlckJlaGF2aW9ycy5wdXNoKHRoaXMudG9CZWhhdmlvcihiZWhhdmlvciwgcHJvcHMudmlld2VyUHJvdG9jb2xQb2xpY3kpIGFzIENmbkRpc3RyaWJ1dGlvbi5DYWNoZUJlaGF2aW9yUHJvcGVydHkpO1xuICAgICAgICB9XG4gICAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHsgLi4uZGlzdHJpYnV0aW9uQ29uZmlnLCBjYWNoZUJlaGF2aW9yczogb3RoZXJCZWhhdmlvcnMubGVuZ3RoID4gMCA/IG90aGVyQmVoYXZpb3JzIDogdW5kZWZpbmVkIH07XG4gICAgICAgIGlmIChwcm9wcy5hbGlhc0NvbmZpZ3VyYXRpb24gJiYgcHJvcHMudmlld2VyQ2VydGlmaWNhdGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihbXG4gICAgICAgICAgICAgICAgJ1lvdSBjYW5ub3Qgc2V0IGJvdGggYWxpYXNDb25maWd1cmF0aW9uIGFuZCB2aWV3ZXJDZXJ0aWZpY2F0ZSBwcm9wZXJ0aWVzLicsXG4gICAgICAgICAgICAgICAgJ1BsZWFzZSBvbmx5IHVzZSB2aWV3ZXJDZXJ0aWZpY2F0ZSwgYXMgYWxpYXNDb25maWd1cmF0aW9uIGlzIGRlcHJlY2F0ZWQuJyxcbiAgICAgICAgICAgIF0uam9pbignICcpKTtcbiAgICAgICAgfVxuICAgICAgICBsZXQgX3ZpZXdlckNlcnRpZmljYXRlID0gcHJvcHMudmlld2VyQ2VydGlmaWNhdGU7XG4gICAgICAgIGlmIChwcm9wcy5hbGlhc0NvbmZpZ3VyYXRpb24pIHtcbiAgICAgICAgICAgIGNvbnN0IHsgYWNtQ2VydFJlZiwgc2VjdXJpdHlQb2xpY3ksIHNzbE1ldGhvZCwgbmFtZXM6IGFsaWFzZXMgfSA9IHByb3BzLmFsaWFzQ29uZmlndXJhdGlvbjtcbiAgICAgICAgICAgIF92aWV3ZXJDZXJ0aWZpY2F0ZSA9IFZpZXdlckNlcnRpZmljYXRlLmZyb21BY21DZXJ0aWZpY2F0ZShjZXJ0aWZpY2F0ZW1hbmFnZXIuQ2VydGlmaWNhdGUuZnJvbUNlcnRpZmljYXRlQXJuKHRoaXMsICdBbGlhc0NvbmZpZ3VyYXRpb25DZXJ0JywgYWNtQ2VydFJlZiksIHsgc2VjdXJpdHlQb2xpY3ksIHNzbE1ldGhvZCwgYWxpYXNlcyB9KTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoX3ZpZXdlckNlcnRpZmljYXRlKSB7XG4gICAgICAgICAgICBjb25zdCB7IHByb3BzOiB2aWV3ZXJDZXJ0aWZpY2F0ZSwgYWxpYXNlcyB9ID0gX3ZpZXdlckNlcnRpZmljYXRlO1xuICAgICAgICAgICAgT2JqZWN0LmFzc2lnbihkaXN0cmlidXRpb25Db25maWcsIHsgYWxpYXNlcywgdmlld2VyQ2VydGlmaWNhdGUgfSk7XG4gICAgICAgICAgICBjb25zdCB7IG1pbmltdW1Qcm90b2NvbFZlcnNpb24sIHNzbFN1cHBvcnRNZXRob2QgfSA9IHZpZXdlckNlcnRpZmljYXRlO1xuICAgICAgICAgICAgaWYgKG1pbmltdW1Qcm90b2NvbFZlcnNpb24gIT0gbnVsbCAmJiBzc2xTdXBwb3J0TWV0aG9kICE9IG51bGwpIHtcbiAgICAgICAgICAgICAgICBjb25zdCB2YWxpZFByb3RvY29scyA9IHRoaXMuVkFMSURfU1NMX1BST1RPQ09MU1tzc2xTdXBwb3J0TWV0aG9kIGFzIFNTTE1ldGhvZF07XG4gICAgICAgICAgICAgICAgaWYgKHZhbGlkUHJvdG9jb2xzLmluZGV4T2YobWluaW11bVByb3RvY29sVmVyc2lvbi50b1N0cmluZygpKSA9PT0gLTEpIHtcbiAgICAgICAgICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAke21pbmltdW1Qcm90b2NvbFZlcnNpb259IGlzIG5vdCBjb21wYWJ0aWJsZSB3aXRoIHNzbE1ldGhvZCAke3NzbFN1cHBvcnRNZXRob2R9LlxcblxcdFZhbGlkIFByb3RvY29scyBhcmU6ICR7dmFsaWRQcm90b2NvbHMuam9pbignLCAnKX1gKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICBkaXN0cmlidXRpb25Db25maWcgPSB7IC4uLmRpc3RyaWJ1dGlvbkNvbmZpZyxcbiAgICAgICAgICAgICAgICB2aWV3ZXJDZXJ0aWZpY2F0ZTogeyBjbG91ZEZyb250RGVmYXVsdENlcnRpZmljYXRlOiB0cnVlIH0sXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5sb2dnaW5nQ29uZmlnKSB7XG4gICAgICAgICAgICB0aGlzLmxvZ2dpbmdCdWNrZXQgPSBwcm9wcy5sb2dnaW5nQ29uZmlnLmJ1Y2tldCB8fCBuZXcgczMuQnVja2V0KHRoaXMsICdMb2dnaW5nQnVja2V0Jyk7XG4gICAgICAgICAgICBkaXN0cmlidXRpb25Db25maWcgPSB7XG4gICAgICAgICAgICAgICAgLi4uZGlzdHJpYnV0aW9uQ29uZmlnLFxuICAgICAgICAgICAgICAgIGxvZ2dpbmc6IHtcbiAgICAgICAgICAgICAgICAgICAgYnVja2V0OiB0aGlzLmxvZ2dpbmdCdWNrZXQuYnVja2V0UmVnaW9uYWxEb21haW5OYW1lLFxuICAgICAgICAgICAgICAgICAgICBpbmNsdWRlQ29va2llczogcHJvcHMubG9nZ2luZ0NvbmZpZy5pbmNsdWRlQ29va2llcyB8fCBmYWxzZSxcbiAgICAgICAgICAgICAgICAgICAgcHJlZml4OiBwcm9wcy5sb2dnaW5nQ29uZmlnLnByZWZpeCxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMuZ2VvUmVzdHJpY3Rpb24pIHtcbiAgICAgICAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHtcbiAgICAgICAgICAgICAgICAuLi5kaXN0cmlidXRpb25Db25maWcsXG4gICAgICAgICAgICAgICAgcmVzdHJpY3Rpb25zOiB7XG4gICAgICAgICAgICAgICAgICAgIGdlb1Jlc3RyaWN0aW9uOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICByZXN0cmljdGlvblR5cGU6IHByb3BzLmdlb1Jlc3RyaWN0aW9uLnJlc3RyaWN0aW9uVHlwZSxcbiAgICAgICAgICAgICAgICAgICAgICAgIGxvY2F0aW9uczogcHJvcHMuZ2VvUmVzdHJpY3Rpb24ubG9jYXRpb25zLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9O1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBDZm5EaXN0cmlidXRpb24odGhpcywgJ0NGRGlzdHJpYnV0aW9uJywgeyBkaXN0cmlidXRpb25Db25maWcgfSk7XG4gICAgICAgIHRoaXMubm9kZS5kZWZhdWx0Q2hpbGQgPSBkaXN0cmlidXRpb247XG4gICAgICAgIHRoaXMuZG9tYWluTmFtZSA9IGRpc3RyaWJ1dGlvbi5hdHRyRG9tYWluTmFtZTtcbiAgICAgICAgdGhpcy5kaXN0cmlidXRpb25Eb21haW5OYW1lID0gZGlzdHJpYnV0aW9uLmF0dHJEb21haW5OYW1lO1xuICAgICAgICB0aGlzLmRpc3RyaWJ1dGlvbklkID0gZGlzdHJpYnV0aW9uLnJlZjtcbiAgICB9XG4gICAgcHJpdmF0ZSB0b0JlaGF2aW9yKGlucHV0OiBCZWhhdmlvcldpdGhPcmlnaW4sIHByb3RvUG9saWN5PzogVmlld2VyUHJvdG9jb2xQb2xpY3kpIHtcbiAgICAgICAgbGV0IHRvUmV0dXJuID0ge1xuICAgICAgICAgICAgYWxsb3dlZE1ldGhvZHM6IHRoaXMuTUVUSE9EX0xPT0tVUF9NQVBbaW5wdXQuYWxsb3dlZE1ldGhvZHMgfHwgQ2xvdWRGcm9udEFsbG93ZWRNZXRob2RzLkdFVF9IRUFEXSxcbiAgICAgICAgICAgIGNhY2hlZE1ldGhvZHM6IHRoaXMuTUVUSE9EX0xPT0tVUF9NQVBbaW5wdXQuY2FjaGVkTWV0aG9kcyB8fCBDbG91ZEZyb250QWxsb3dlZENhY2hlZE1ldGhvZHMuR0VUX0hFQURdLFxuICAgICAgICAgICAgY29tcHJlc3M6IGlucHV0LmNvbXByZXNzICE9PSBmYWxzZSxcbiAgICAgICAgICAgIGRlZmF1bHRUdGw6IGlucHV0LmRlZmF1bHRUdGwgJiYgaW5wdXQuZGVmYXVsdFR0bC50b1NlY29uZHMoKSxcbiAgICAgICAgICAgIGZvcndhcmRlZFZhbHVlczogaW5wdXQuZm9yd2FyZGVkVmFsdWVzIHx8IHsgcXVlcnlTdHJpbmc6IGZhbHNlLCBjb29raWVzOiB7IGZvcndhcmQ6ICdub25lJyB9IH0sXG4gICAgICAgICAgICBtYXhUdGw6IGlucHV0Lm1heFR0bCAmJiBpbnB1dC5tYXhUdGwudG9TZWNvbmRzKCksXG4gICAgICAgICAgICBtaW5UdGw6IGlucHV0Lm1pblR0bCAmJiBpbnB1dC5taW5UdGwudG9TZWNvbmRzKCksXG4gICAgICAgICAgICB0cnVzdGVkU2lnbmVyczogaW5wdXQudHJ1c3RlZFNpZ25lcnMsXG4gICAgICAgICAgICB0YXJnZXRPcmlnaW5JZDogaW5wdXQudGFyZ2V0T3JpZ2luSWQsXG4gICAgICAgICAgICB2aWV3ZXJQcm90b2NvbFBvbGljeTogcHJvdG9Qb2xpY3kgfHwgVmlld2VyUHJvdG9jb2xQb2xpY3kuUkVESVJFQ1RfVE9fSFRUUFMsXG4gICAgICAgIH07XG4gICAgICAgIGlmICghaW5wdXQuaXNEZWZhdWx0QmVoYXZpb3IpIHtcbiAgICAgICAgICAgIHRvUmV0dXJuID0gT2JqZWN0LmFzc2lnbih0b1JldHVybiwgeyBwYXRoUGF0dGVybjogaW5wdXQucGF0aFBhdHRlcm4gfSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKGlucHV0LmxhbWJkYUZ1bmN0aW9uQXNzb2NpYXRpb25zKSB7XG4gICAgICAgICAgICB0b1JldHVybiA9IE9iamVjdC5hc3NpZ24odG9SZXR1cm4sIHtcbiAgICAgICAgICAgICAgICBsYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uczogaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnNcbiAgICAgICAgICAgICAgICAgICAgLm1hcChmbmEgPT4gKHtcbiAgICAgICAgICAgICAgICAgICAgZXZlbnRUeXBlOiBmbmEuZXZlbnRUeXBlLFxuICAgICAgICAgICAgICAgICAgICBsYW1iZGFGdW5jdGlvbkFybjogZm5hLmxhbWJkYUZ1bmN0aW9uICYmIGZuYS5sYW1iZGFGdW5jdGlvbi5mdW5jdGlvbkFybixcbiAgICAgICAgICAgICAgICB9KSksXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIC8vIGFsbG93IGVkZ2VsYW1iZGEuYW1hem9uYXdzLmNvbSB0byBhc3N1bWUgdGhlIGZ1bmN0aW9ucycgZXhlY3V0aW9uIHJvbGUuXG4gICAgICAgICAgICBmb3IgKGNvbnN0IGEgb2YgaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnMpIHtcbiAgICAgICAgICAgICAgICBpZiAoYS5sYW1iZGFGdW5jdGlvbi5yb2xlICYmIGEubGFtYmRhRnVuY3Rpb24ucm9sZSBpbnN0YW5jZW9mIGlhbS5Sb2xlICYmIGEubGFtYmRhRnVuY3Rpb24ucm9sZS5hc3N1bWVSb2xlUG9saWN5KSB7XG4gICAgICAgICAgICAgICAgICAgIGEubGFtYmRhRnVuY3Rpb24ucm9sZS5hc3N1bWVSb2xlUG9saWN5LmFkZFN0YXRlbWVudHMobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICAgICAgICAgICAgYWN0aW9uczogWydzdHM6QXNzdW1lUm9sZSddLFxuICAgICAgICAgICAgICAgICAgICAgICAgcHJpbmNpcGFsczogW25ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWRnZWxhbWJkYS5hbWF6b25hd3MuY29tJyldLFxuICAgICAgICAgICAgICAgICAgICB9KSk7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0b1JldHVybjtcbiAgICB9XG59XG4iXX0=