"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpOrigin = exports.S3Origin = exports.Origin = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const distribution_1 = require("./distribution");
const origin_access_identity_1 = require("./origin_access_identity");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class Origin {
    constructor(props) {
        this.domainName = props.domainName;
    }
    /**
     * Creates a pre-configured origin for a S3 bucket.
     * If this bucket has been configured for static website hosting, then `fromWebsiteBucket` should be used instead.
     *
     * An Origin Access Identity will be created and granted read access to the bucket.
     *
     * @param bucket the bucket to act as an origin.
     */
    static fromBucket(bucket) {
        if (bucket.isWebsite) {
            return new HttpOrigin({
                domainName: bucket.bucketWebsiteDomainName,
                protocolPolicy: distribution_1.OriginProtocolPolicy.HTTP_ONLY,
            });
        }
        else {
            return new S3Origin({ domainName: bucket.bucketRegionalDomainName, bucket });
        }
    }
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id() {
        if (!this.originId) {
            throw new Error('Cannot access originId until `bind` is called.');
        }
        return this.originId;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     *
     * @internal
     */
    _bind(scope, options) {
        this.originId = new core_1.Construct(scope, `Origin${options.originIndex}`).node.uniqueId;
    }
    /**
     * Creates and returns the CloudFormation representation of this origin.
     *
     * @internal
     */
    _renderOrigin() {
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            domainName: this.domainName,
            id: this.id,
            s3OriginConfig,
            customOriginConfig,
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
}
exports.Origin = Origin;
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
class S3Origin extends Origin {
    constructor(props) {
        super(props);
        this.bucket = props.bucket;
    }
    /** @internal */
    _bind(scope, options) {
        super._bind(scope, options);
        if (!this.originAccessIdentity) {
            this.originAccessIdentity = new origin_access_identity_1.OriginAccessIdentity(scope, `S3Origin${options.originIndex}`);
            this.bucket.grantRead(this.originAccessIdentity);
        }
    }
    renderS3OriginConfig() {
        return { originAccessIdentity: `origin-access-identity/cloudfront/${this.originAccessIdentity.originAccessIdentityName}` };
    }
}
exports.S3Origin = S3Origin;
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
class HttpOrigin extends Origin {
    constructor(props) {
        super(props);
        this.protocolPolicy = props.protocolPolicy;
    }
    renderCustomOriginConfig() {
        var _a;
        return {
            originProtocolPolicy: (_a = this.protocolPolicy) !== null && _a !== void 0 ? _a : distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
        };
    }
}
exports.HttpOrigin = HttpOrigin;
//# sourceMappingURL=data:application/json;base64,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