import { IBucket } from "../../aws-s3";
import { CfnDistribution } from './cloudfront.generated';
import { OriginProtocolPolicy } from './distribution';
/**
 * Properties to be used to create an Origin. Prefer to use one of the Origin.from* factory methods rather than
 * instantiating an Origin directly from these properties.
 *
 * @experimental
 */
export interface OriginProps {
    /**
     * The domain name of the Amazon S3 bucket or HTTP server origin.
     */
    readonly domainName: string;
}
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
export declare abstract class Origin {
    /**
     * Creates a pre-configured origin for a S3 bucket.
     * If this bucket has been configured for static website hosting, then `fromWebsiteBucket` should be used instead.
     *
     * An Origin Access Identity will be created and granted read access to the bucket.
     *
     * @param bucket the bucket to act as an origin.
     */
    static fromBucket(bucket: IBucket): Origin;
    /**
     * The domain name of the origin.
     */
    readonly domainName: string;
    private originId;
    constructor(props: OriginProps);
    /**
     * The unique id for this origin.
     *
     * Cannot be accesed until bind() is called.
     */
    get id(): string;
    protected renderS3OriginConfig(): CfnDistribution.S3OriginConfigProperty | undefined;
    protected renderCustomOriginConfig(): CfnDistribution.CustomOriginConfigProperty | undefined;
}
/**
 * Properties for an Origin backed by an S3 bucket
 *
 * @experimental
 */
export interface S3OriginProps extends OriginProps {
    /**
     * The bucket to use as an origin.
     */
    readonly bucket: IBucket;
}
/**
 * An Origin specific to a S3 bucket (not configured for website hosting).
 *
 * Contains additional logic around bucket permissions and origin access identities.
 *
 * @experimental
 */
export declare class S3Origin extends Origin {
    private readonly bucket;
    private originAccessIdentity;
    constructor(props: S3OriginProps);
    protected renderS3OriginConfig(): CfnDistribution.S3OriginConfigProperty | undefined;
}
/**
 * Properties for an Origin backed by an S3 website-configured bucket, load balancer, or custom HTTP server.
 *
 * @experimental
 */
export interface HttpOriginProps extends OriginProps {
    /**
     * Specifies the protocol (HTTP or HTTPS) that CloudFront uses to connect to the origin.
     *
     * @default OriginProtocolPolicy.HTTPS_ONLY
     */
    readonly protocolPolicy?: OriginProtocolPolicy;
}
/**
 * An Origin for an HTTP server or S3 bucket configured for website hosting.
 *
 * @experimental
 */
export declare class HttpOrigin extends Origin {
    private readonly protocolPolicy?;
    constructor(props: HttpOriginProps);
    protected renderCustomOriginConfig(): CfnDistribution.CustomOriginConfigProperty | undefined;
}
