"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AllowedMethods = exports.OriginProtocolPolicy = exports.ViewerProtocolPolicy = exports.PriceClass = exports.Distribution = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
const cache_behavior_1 = require("./private/cache-behavior");
/**
 * A CloudFront distribution with associated origin(s) and caching behavior(s).
 *
 * @experimental
 */
class Distribution extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.additionalBehaviors = [];
        this.origins = new Set();
        if (props.certificate) {
            const certificateRegion = core_1.Stack.of(this).parseArn(props.certificate.certificateArn).region;
            if (!core_1.Token.isUnresolved(certificateRegion) && certificateRegion !== 'us-east-1') {
                throw new Error('Distribution certificates must be in the us-east-1 region and the certificate you provided is in $Region.');
            }
        }
        this.defaultBehavior = new cache_behavior_1.CacheBehavior({ pathPattern: '*', ...props.defaultBehavior });
        this.addOrigin(this.defaultBehavior.origin);
        if (props.additionalBehaviors) {
            Object.entries(props.additionalBehaviors).forEach(([pathPattern, behaviorOptions]) => {
                this.addBehavior(pathPattern, behaviorOptions.origin, behaviorOptions);
            });
        }
        this.certificate = props.certificate;
        this.errorResponses = (_a = props.errorResponses) !== null && _a !== void 0 ? _a : [];
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig: {
                enabled: true,
                origins: core_1.Lazy.anyValue({ produce: () => this.renderOrigins() }),
                defaultCacheBehavior: this.defaultBehavior._renderBehavior(),
                cacheBehaviors: core_1.Lazy.anyValue({ produce: () => this.renderCacheBehaviors() }),
                viewerCertificate: this.certificate ? { acmCertificateArn: this.certificate.certificateArn } : undefined,
                customErrorResponses: this.renderErrorResponses(),
                priceClass: (_b = props.priceClass) !== null && _b !== void 0 ? _b : undefined,
            } });
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a Distribution construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        return new class extends core_1.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    /**
     * Adds a new behavior to this distribution for the given pathPattern.
     *
     * @param pathPattern the path pattern (e.g., 'images/*') that specifies which requests to apply the behavior to.
     * @param behaviorOptions the options for the behavior at this path.
     */
    addBehavior(pathPattern, origin, behaviorOptions = {}) {
        if (pathPattern === '*') {
            throw new Error('Only the default behavior can have a path pattern of \'*\'');
        }
        this.additionalBehaviors.push(new cache_behavior_1.CacheBehavior({ pathPattern, origin, ...behaviorOptions }));
        this.addOrigin(origin);
    }
    addOrigin(origin) {
        if (!this.origins.has(origin)) {
            this.origins.add(origin);
            origin._bind(this, { originIndex: this.origins.size });
        }
    }
    renderOrigins() {
        const renderedOrigins = [];
        this.origins.forEach(origin => renderedOrigins.push(origin._renderOrigin()));
        return renderedOrigins;
    }
    renderCacheBehaviors() {
        if (this.additionalBehaviors.length === 0) {
            return undefined;
        }
        return this.additionalBehaviors.map(behavior => behavior._renderBehavior());
    }
    renderErrorResponses() {
        if (this.errorResponses.length === 0) {
            return undefined;
        }
        function validateCustomErrorResponse(errorResponse) {
            if (errorResponse.responsePagePath && !errorResponse.responseHttpStatus) {
                throw new Error('\'responseCode\' must be provided if \'responsePagePath\' is defined');
            }
            if (!errorResponse.responseHttpStatus && !errorResponse.ttl) {
                throw new Error('A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid.');
            }
        }
        this.errorResponses.forEach(e => validateCustomErrorResponse(e));
        return this.errorResponses.map(errorConfig => {
            var _a;
            return {
                errorCachingMinTtl: (_a = errorConfig.ttl) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                errorCode: errorConfig.httpStatus,
                responseCode: errorConfig.responseHttpStatus,
                responsePagePath: errorConfig.responsePagePath,
            };
        });
    }
}
exports.Distribution = Distribution;
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 * See https://aws.amazon.com/cloudfront/pricing/ for full list of supported regions.
 */
var PriceClass;
(function (PriceClass) {
    /** USA, Canada, Europe, & Israel */
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    /** PRICE_CLASS_100 + South Africa, Kenya, Middle East, Japan, Singapore, South Korea, Taiwan, Hong Kong, & Philippines */
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    /** All locations */
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    /** HTTPS only */
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    /** Will redirect HTTP requests to HTTPS */
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    /** Both HTTP and HTTPS supported */
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * Defines what protocols CloudFront will use to connect to an origin.
 */
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    /** Connect on HTTP only */
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    /** Connect with the same protocol as the viewer */
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    /** Connect on HTTPS only */
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * The HTTP methods that the Behavior will accept requests on.
 */
class AllowedMethods {
    constructor(methods) { this.methods = methods; }
}
exports.AllowedMethods = AllowedMethods;
/** HEAD and GET */
AllowedMethods.ALLOW_GET_HEAD = new AllowedMethods(['GET', 'HEAD']);
/** HEAD, GET, and OPTIONS */
AllowedMethods.ALLOW_GET_HEAD_OPTIONS = new AllowedMethods(['GET', 'HEAD', 'OPTIONS']);
/** All supported HTTP methods */
AllowedMethods.ALLOW_ALL = new AllowedMethods(['GET', 'HEAD', 'OPTIONS', 'PUT', 'PATCH', 'POST', 'DELETE']);
//# sourceMappingURL=data:application/json;base64,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