"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand or SpotFleet compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole ? spotFleetRole.roleArn : undefined,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole
                ? props.serviceRole.roleArn
                : new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed === undefined ? true : props.managed;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,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